% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compare.R
\name{compare}
\alias{compare}
\title{Compare two objects}
\usage{
compare(
  x,
  y,
  ...,
  x_arg = "old",
  y_arg = "new",
  tolerance = NULL,
  max_diffs = if (in_ci()) Inf else 10,
  ignore_srcref = TRUE,
  ignore_attr = "waldo_opts",
  ignore_encoding = TRUE,
  ignore_function_env = FALSE,
  ignore_formula_env = FALSE,
  list_as_map = FALSE
)
}
\arguments{
\item{x, y}{Objects to compare. \code{y} is treated as the reference object
so messages describe how \code{x} is different to \code{y}}

\item{...}{A handful of other arguments are supported with a warning for
backward compatability. These include:
\itemize{
\item \code{all.equal()} arguments \code{checkNames} and \code{check.attributes}
\item \code{testthat::compare()} argument \code{tol}
}

All other arguments are ignored with a warning.}

\item{x_arg, y_arg}{Name of \code{x} and \code{y} arguments, used when generated paths
to internal components. These default to "old" and "new" since it's
most natural to supply the previous value then the new value.}

\item{tolerance}{If non-\code{NULL}, used as threshold for ignoring small
floating point difference when comparing numeric vectors. Setting to
any non-\code{NULL} value will cause integer and double vectors to be compared
based on their values, rather than their types.

It uses the same algorithm as \code{\link[=all.equal]{all.equal()}}, i.e., first we generate
\code{x_diff} and \code{y_diff} by subsetting \code{x} and \code{y} to look only locations
with differences. Then we check that
\code{mean(abs(x_diff - y_diff)) / mean(abs(y_diff))} (or just
\code{mean(abs(x_diff - y_diff))} if \code{y_diff} is small) is less than
\code{tolerance}.}

\item{max_diffs}{Control the maximum number of differences shown. The
default shows 10 differences when run interactively and all differences
when run in CI. Set \code{max_diffs = Inf} to see all differences.}

\item{ignore_srcref}{Ignore differences in function \code{srcref}s? \code{TRUE} by
default since the \code{srcref} does not change the behaviour of a function,
only its printed representation.}

\item{ignore_attr}{Ignore differences in specified attributes?
Supply a character vector to ignore differences in named attributes.
By default the \code{"waldo_opts"} attribute is listed in \code{ignore_attr} so
that changes to it are not reported; if you customize \code{ignore_attr}, you
will probably want to do this yourself.

For backward compatibility with \code{all.equal()}, you can also use \code{TRUE},
to all ignore differences in all attributes. This is not generally
recommended as it is a blunt tool that will ignore many important
functional differences.}

\item{ignore_encoding}{Ignore string encoding? \code{TRUE} by default, because
this is R's default behaviour. Use \code{FALSE} when specifically concerned
with the encoding, not just the value of the string.}

\item{ignore_function_env, ignore_formula_env}{Ignore the environments of
functions and formulas, respectively? These are provided primarily for
backward compatibility with \code{all.equal()} which always ignores these
environments.}

\item{list_as_map}{Compare lists as if they are mappings between names and
values. Concretely, this drops \code{NULLs} in both objects and sorts named
components.}
}
\value{
A character vector with class "waldo_compare". If there are no
differences it will have length 0; otherwise each element contains the
description of a single difference.
}
\description{
This compares two R objects, identifying the key differences. It:
\itemize{
\item Orders the differences from most important to least important.
\item Displays the values of atomic vectors that are actually different.
\item Carefully uses colour to emphasise changes (while still being readable
when colour isn't available).
\item Uses R code (not a text description) to show where differences arise.
\item Where possible, it compares elements by name, rather than by position.
\item Errs on the side of producing too much output, rather than too little.
}

\code{compare()} is an alternative to \code{\link[=all.equal]{all.equal()}}.
}
\section{Controlling comparisons}{


There are two ways for an object (rather than the person calling \code{compare()}
or \code{expect_equal()} to control how it is compared to other objects.
First, if the object has an S3 class, you can provide a \code{\link[=compare_proxy]{compare_proxy()}}
method that provides an alternative representation of the object; this is
particularly useful if important data is stored outside of R, e.g. in
an external pointer.

Alternatively, you can attach an attribute called \code{"waldo_opts"} to your
object. This should be a list of compare options, using the same names
and possible values as the arguments to this function. This option
is ignored by default (\code{ignore_attr}) so that you can set the options in
the object that you control. (If you don't want to see the attributes
interactively, you could attach them in a \code{\link[=compare_proxy]{compare_proxy()}} method.)

Options supplied in this way also affect all the children. This means
options are applied in the following order, from lowest to highest
precedence:
\enumerate{
\item Defaults from \code{compare()}.
\item The \code{waldo_opts} for the parents of \code{x}.
\item The \code{waldo_opts} for the parents of \code{y}.
\item The \code{waldo_opts} for \code{x}.
\item The \code{waldo_opts} for \code{y}.
\item User-specified arguments to \code{compare()}.
}

Use these techniques with care. If you accidentally cover up an important
difference you can create a confusing situation where \code{x} and \code{y} behave
differently but \code{compare()} reports no differences in the underlying objects.
}

\examples{
# Thanks to diffobj package comparison of atomic vectors shows differences
# with a little context
compare(letters, c("z", letters[-26]))
compare(c(1, 2, 3), c(1, 3))
compare(c(1, 2, 3), c(1, 3, 4, 5))
compare(c(1, 2, 3), c(1, 2, 5))

# More complex objects are traversed, stopping only when the types are
# different
compare(
  list(x = list(y = list(structure(1, z = 2)))),
  list(x = list(y = list(structure(1, z = "a"))))
)

# Where possible, recursive structures are compared by name
compare(iris, rev(iris))

compare(list(x = "x", y = "y"), list(y = "y", x = "x"))
# Otherwise they're compared by position
compare(list("x", "y"), list("x", "z"))
compare(list(x = "x", x = "y"), list(x = "x", y = "z"))

}
