# walkthrough for using package skeleton

require(rstantools)

# change directory to where this file is located
pkg_path <- getwd()

#--- step 0: create a package without Stan code --------------------------------

# if you want to create an entire package with Stan code from scratch,
# skip to step 5

pkg_name <- "RStanTest"
package.skeleton(name = pkg_name,
                 code_files = "postsamp.R")
# delete contents of "man" and NAMESPACE so roxygen2 can be used
file.remove(list.files(file.path(pkg_path, pkg_name, "man"),
                       full.names = TRUE))
file.remove(file.path(pkg_path, pkg_name, "NAMESPACE"))

# add testthat tests
usethis::use_testthat(pkg_name)
file.copy(from = "test-postsamp.R",
          to = file.path(pkg_name, "tests", "testthat", "test-postsamp.R"))

# setup documentation with roxygen2
roxygen2::roxygenize(pkg_name)

#--- Step 1: add Stan functionality to package ---------------------------------

# this sets up the package to do this
rstantools::use_rstan(pkg_name)

# follow the instructions to add to the NAMESPACE
# since we're using roxygen2, the right way to do this is to
# add the lines to an R file in the package.
# I like to make this the R/RStanTest-package.R file

file.copy(from = file.path(pkg_path, "RStanTest-package.R"),
          to = file.path(pkg_path, pkg_name, "R", "RStanTest-package.R"))

#--- Step 2: add Stan files to package -----------------------------------------

# copy stan_files to inst/stan
stan_files <- c("SimpleModel.stan", "SimpleModel2.stan")
file.copy(from = file.path(pkg_path, stan_files),
          to = file.path(pkg_path, pkg_name, "inst", "stan", stan_files))

# configure package to create C++ code associated with stan_files
# ***NOTE*** You MUST run this every time you
# modify/add/delete anything in inst/stan
rstantools::rstan_config(pkg_name)


# re-roxygen2, which will automatically compile package
roxygen2::roxygenize(pkg_name)

#--- step 4: install package and run tests -------------------------------------

utils::install.packages(pkg_name, repos = NULL, type = "source")

# quit and restart, then:
testthat::test_package("RStanTest", reporter = "progress")


#--- step 5: create an entire package with Stan code from scratch --------------

pkg_path <- getwd()
pkg_name <- "RStanTest"
code_files <- c("RStanTest-package.R", "postsamp.R")
stan_files <- c("SimpleModel.stan", "SimpleModel2.stan")

rstantools::rstan_package_skeleton_plus(name = pkg_name,
                                        path = pkg_path,
                                        code_files = code_files,
                                        stan_files = stan_files)
# add tests
usethis::use_testthat(pkg_name)
file.copy(from = "test-postsamp.R",
          to = file.path(pkg_name, "tests", "testthat", "test-postsamp.R"))

# roxygen2 can't overwrite NAMESPACE unless it created it,
# so trick it into thinking it did
# (and add the rstan stuff or else you get errors...)
cat("# Generated by roxygen2: do not edit by hand",
    "",
    "import(Rcpp,methods)",
    "useDynLib(RStanTest, .registration = TRUE)", sep = "\n",
    file = file.path(pkg_path, pkg_name, "NAMESPACE"))

roxygen2::roxygenize(pkg_name) # run roxygen2
utils::install.packages(pkg_name, repos = NULL, type = "source")

# quit and restart, then:
testthat::test_package("RStanTest", reporter = "progress")

