/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <errno.h>
#include <glib.h>
#include <libupower-glib/upower.h>

#include "powerd-internal.h"
#include "device-config.h"
#include "log.h"

static UpClient *up_client;
static double shutdown_temp = 68.0f;
static double battery_log_level = 4.0f;
static gboolean stats_logged;
static GHashTable *battery_state_hash;

static void up_device_changed_cb(UpClient *client,
#if UP_CHECK_VERSION(0,99,0)
                                 GParamSpec *pspec,
#else
                                 UpDevice *device,
#endif
                                 gpointer unused)
{
    gboolean on_battery;
    GPtrArray *devices;
    int i;
    gboolean log_stats = TRUE;
    gboolean low_batt_shutdown = TRUE;
    gboolean thermal_shutdown = FALSE;

    g_object_get(up_client,
                 "on_battery", &on_battery,
                 NULL);
    if (!on_battery)
        low_batt_shutdown = FALSE;
    devices = up_client_get_devices(up_client);
    for (i = 0; i < devices->len; i++) {
        UpDevice *device = g_ptr_array_index(devices, i);
        UpDeviceKind kind;
        gboolean is_present;
        g_object_get(device,
                     "kind", &kind,
                     "is-present", &is_present,
                     NULL);
        if (kind == UP_DEVICE_KIND_BATTERY && is_present) {
            double percentage, temp;
            guint state;
            guint prev_state;
            gchar *native_path;
            g_object_get(device,
                         "percentage", &percentage,
                         "temperature", &temp,
                         "state", &state,
                         "native-path", &native_path,
                         NULL);
            gpointer hvalue = g_hash_table_lookup(battery_state_hash,
                                                  native_path);
            if (hvalue == NULL)
                g_hash_table_insert(battery_state_hash,
                            g_strdup(native_path), GUINT_TO_POINTER(state));
            else {
                prev_state = GPOINTER_TO_UINT(hvalue);
                if (prev_state != state) {
                    if (state == UP_DEVICE_STATE_DISCHARGING ||
                                (prev_state == UP_DEVICE_STATE_DISCHARGING && (
                                state == UP_DEVICE_STATE_CHARGING ||
                                state == UP_DEVICE_STATE_FULLY_CHARGED))) {
                        powerd_debug("Turning screen on, battery state changes");
                        turn_display_on(TRUE, UNITY_SCREEN_REASON_NORMAL);
                    }
                    g_hash_table_replace(battery_state_hash,
                                         g_strdup(native_path),
                                         GUINT_TO_POINTER(state));
                }
            }
            g_free(native_path);

            /*
             * Log battery stats a little before emergency
             * shutdown is triggered to avoid any delays
             * when we actually need to shut down.
             */
            if (percentage > battery_log_level) {
                log_stats = FALSE;
                stats_logged = FALSE;
            }

            /*
             * Android shuts down when percentage reaches 0. We
             * use 1% to leave a little more headroom.
             */
            if (percentage > 1.0)
                low_batt_shutdown = FALSE;
            if (temp >= shutdown_temp) {
                powerd_warn("Critical battery temperature %f\n", temp);
                thermal_shutdown = TRUE;
                break;
            }
        }
    }
    if (log_stats && !stats_logged) {
        powerd_log_stats();
        stats_logged = TRUE;
    }
    if (low_batt_shutdown || thermal_shutdown) {
        powerd_warn("Initiating emergency shutdown");
        powerd_shutdown();
    }
}

int powerd_ps_init(void)
{
    GValue v = G_VALUE_INIT;

    /*
     * Override default shutdown temperature with device-specific
     * value, if avaialable
     */
    if (!device_config_get("shutdownBatteryTemperature", &v) &&
        G_VALUE_HOLDS_UINT(&v)) {
        shutdown_temp = (double)g_value_get_uint(&v) / 10.0f;
        g_value_unset(&v);
    }

    if (!device_config_get("criticalBatteryWarningLevel", &v) &&
        G_VALUE_HOLDS_UINT(&v)) {
        battery_log_level = (double)g_value_get_uint(&v);
        g_value_unset(&v);
    }

    up_client = up_client_new();
    if (!up_client) {
        powerd_warn("Could not allocate upower client");
        return -ENODEV;
    }

    /* Hash to track the state for the battery devices */
    battery_state_hash = g_hash_table_new(g_str_hash, g_str_equal);
    GPtrArray *devices;
    int i;
    devices = up_client_get_devices(up_client);
    for (i = 0; i < devices->len; i++) {
        UpDevice *device = g_ptr_array_index(devices, i);
        UpDeviceKind kind;
        gboolean is_present;
        guint state;
        gchar *native_path;
        g_object_get(device,
                     "kind", &kind,
                     "is-present", &is_present,
                     "state", &state,
                     "native-path", &native_path,
                     NULL);
        if (kind == UP_DEVICE_KIND_BATTERY && is_present) {
            g_hash_table_insert(battery_state_hash,
                    g_strdup(native_path), GUINT_TO_POINTER(state));
        }
        g_free(native_path);
    }

#if UP_CHECK_VERSION(0,99,0)
    g_signal_connect(up_client, "notify",
                     G_CALLBACK(up_device_changed_cb), NULL);
#else
    gboolean ret;
    ret = up_client_enumerate_devices_sync(up_client, NULL, NULL);
    if (!ret)
        powerd_warn("Could not enumerate upower devices");
    g_signal_connect(up_client, "device-changed",
                     G_CALLBACK(up_device_changed_cb), NULL);
#endif
    return 0;
}

void powerd_ps_deinit(void)
{
    if (up_client)
        g_object_unref(up_client);

    if (battery_state_hash) {
        g_hash_table_destroy(battery_state_hash);
        battery_state_hash = NULL;
    }
}
