/*	Polar_Stereographic_Elliptical_Projection

PIRL CVS ID: Polar_Stereographic_Elliptical_Projection.java,v 1.4 2012/09/25 23:06:57 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Image_Tools;

import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.PVL_Exception;

import	java.awt.Point;
import	java.awt.geom.Point2D;


public class Polar_Stereographic_Elliptical_Projection
	extends Projection
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Image_Tools.Polar_Stereographic_Elliptical_Projection (1.4 2012/09/25 23:06:57)";

private static final String
	PROJECTION_NAME					= "Polar Stereographic Elliptical";

private double
	E_over_2,
	Center_Latitude_Sign,
	Distance_Coefficient;


//  DEBUG control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_CONVERTERS		= 1 << 2,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Polar_Stereographic_Elliptical_Projection
	(
	Parameter	parameters
	)
	throws PVL_Exception
{
//	Initialize the base projection parameters.
super (parameters);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Polar_Stereographic_Elliptical_Projection: " + parameters);

if (Not_Identity)
	{
	//	Precalculated coefficients.

	E_over_2 = Eccentricity * 0.5;

	if (Center_Latitude < 0.0)
		Center_Latitude_Sign = -1.0;
	else
		Center_Latitude_Sign = 1.0;

	boolean
		at_pole = true;
	double
		coefficient_t = 0.0;
	if ((PI_OVER_2 - Math.abs (Center_Latitude)) > DBL_EPSILON)
		{
		coefficient_t = Coefficient_T (Center_Latitude);
		if (Math.abs (coefficient_t) >= DBL_EPSILON)
			at_pole = false;
		}
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("    at_pole = " + at_pole);

	//	For to_World and to_Image conversion.
	if (at_pole)
		Distance_Coefficient =
			StrictMath.sqrt
				(StrictMath.pow (1.0 + Eccentricity, 1.0 + Eccentricity) *
				 StrictMath.pow (1.0 - Eccentricity, 1.0 - Eccentricity))
			/ (2.0 * Equitorial_Radius);
	else
		{
		double
			phi = Center_Latitude_Sign * Center_Latitude,
			es = Eccentricity * StrictMath.sin (phi);
		Distance_Coefficient =
			coefficient_t /
				(Equitorial_Radius
					* StrictMath.cos (phi) / StrictMath.sqrt (1.0 - es * es));
		}
	}
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< Polar_Stereographic_Elliptical_Projection");
}

/*==============================================================================
	Accessors
*/
public String Name ()
{return PROJECTION_NAME;}

/*==============================================================================
	Converters
*/
/**	Get the world longitude,latitude coordinate for an image sample,line
	coordinate.
<p>
	The conversion algorithm is:
<p><blockquote>

</blockquote>
<p>
	@param	image_coordinate	The image sample,line coordinate.
	@return	The world longitude,latitude coordinate. The {@link
		Point2D#getX() x} value of the coordinate Point2D is the
		longitude; the {@link Point2D#getY() y} value is the latitude.
		Values are in degrees within the range 0-360.
*/
public Point2D.Double to_World
	(
	Point2D		image_coordinate
	)
	throws IllegalArgumentException, ArithmeticException
{
if ((DEBUG & DEBUG_CONVERTERS) != 0)
	System.out.println
		(">>> Polar_Stereographic_Elliptical_Projection.to_World: "
			+ image_coordinate);
if (image_coordinate == null)
	throw new IllegalArgumentException (ID + '\n'
		+ "Unable to project image coordinates ("
			+ (int)image_coordinate.getX () + ','
			+ (int)image_coordinate.getX () + ") to world coordinates\n"
		+ "because the image coordinates point is null.");
Point2D.Double
	world_coordinate = null;
if (Not_Identity)
	{
	double
		x = Center_Latitude_Sign
			* Sample_to_Projection_X (image_coordinate.getX ()),
		y = Center_Latitude_Sign
			* Line_to_Projection_Y   (image_coordinate.getY ()),
		distance = StrictMath.sqrt ((x * x) + (y * y)),
		coefficient_t = distance * Distance_Coefficient;

	//	Latitude.
	double
		latitude = Center_Latitude_Sign * Coefficient_Phi2 (coefficient_t);
	if ((DEBUG & DEBUG_CONVERTERS) != 0)
		System.out.println
			("    Center_Latitude_Sign = " + Center_Latitude_Sign + '\n'
			+"    x = " + x + '\n'
			+"    y = " + y + '\n'
			+"    distance = " + distance + '\n'
			+"    Distance_Coefficient = " + Distance_Coefficient + '\n'
			+"    coefficient_t = " + coefficient_t + '\n'
			+"    Coefficient_Phi2 = " + Coefficient_Phi2 (coefficient_t) + '\n'
			+"    latitude = " + latitude);
	if (Math.abs (latitude) > PI_OVER_2)
		throw new ArithmeticException (ID + '\n'
			+ "Projection of image coordinates ("
				+ (int)image_coordinate.getX () + ','
				+ (int)image_coordinate.getX () + ") to world coordinates\n"
			+ "resulted in an invalid latitude of "
				+ Math.toDegrees (latitude) + " degrees.");
	if (Planetocentric)
		latitude = Planetographic_to_Planetocentric (latitude);
	latitude = StrictMath.toDegrees (latitude);
	if ((DEBUG & DEBUG_CONVERTERS) != 0)
		System.out.println
			("    latitude = " + latitude);

	//	Longitude.
	double
		longitude = ((distance == 0.0) ?
			(Center_Latitude_Sign * Center_Longitude) :
			(Center_Latitude_Sign * StrictMath.atan2 (x, -y)
				+ Center_Longitude));
	if ((DEBUG & DEBUG_CONVERTERS) != 0)
		System.out.println ('\n'
			+"    Center_Longitude = " + Center_Longitude + '\n'
			+ ((distance == 0.0) ? ""
				: ("    StrictMath.atan2 (x, -y) = "
					+ StrictMath.atan2 (x, -y) + '\n'))
			+"    longitude = " + longitude);
	if (Positive_West)
		longitude *= -1.0;
	longitude = To_360 (StrictMath.toDegrees (longitude));
	if ((DEBUG & DEBUG_CONVERTERS) != 0)
		System.out.println
			("    longitude = " + longitude);

	world_coordinate = new Point2D.Double (longitude, latitude);
	}
else
	world_coordinate = new Point2D.Double
		(image_coordinate.getX (), image_coordinate.getY ());
if ((DEBUG & DEBUG_CONVERTERS) != 0)
	System.out.println
		("<<< Polar_Stereographic_Elliptical_Projection.to_World: "
			+ world_coordinate);
return world_coordinate;
}

/**	Get the image sample,line coordinate for a world longitude,latitude
	coordinate.
<p>
	The conversion algorithm is:
<p><blockquote>

</blockquote>
<p>
	@param	world_coordinate	The world longitude,latitude coordinate.
		The {@link Point2D#getX() x} value of the coordinate Point2D is
		the longitude; the {@link Point2D#getY() y} value is the
		latitude. Values are in degrees.
	@return	The image sample,line coordinate.
*/
public Point to_Image
	(
	Point2D		world_coordinate
	)
	throws IllegalArgumentException
{
if ((DEBUG & DEBUG_CONVERTERS) != 0)
	System.out.println
		(">>> Polar_Stereographic_Elliptical_Projection.to_Image: "
			+ world_coordinate);
if (world_coordinate == null)
	throw new IllegalArgumentException (ID + '\n'
		+ "Unable to project world coordinates to image coordinates\n"
		+ "because the world coordinates point is null.");
Point
	image_coordinate = null;
if (Not_Identity)
	{
	double
		longitude = StrictMath.toRadians (world_coordinate.getX ()),
		latitude  = StrictMath.toRadians (world_coordinate.getY ());
	if (Planetocentric)
		latitude = Planetocentric_to_Planetographic (latitude);
	if (Positive_West)
		longitude *= -1.0;

	double
		lambda = Center_Latitude_Sign * (longitude - Center_Longitude),
		distance = Coefficient_T (latitude) / Distance_Coefficient;

	image_coordinate = new Point
		(
		(int)(Projection_X_to_Sample (
			 Center_Latitude_Sign * distance * StrictMath.sin (lambda))
			+ 0.5),		//	Round to nearest pixel.
		(int)(Projection_Y_to_Line (
			-Center_Latitude_Sign * distance * StrictMath.cos (lambda))
			+ 0.5)		//	Round to nearest pixel.
		);
	}
else
	image_coordinate = new Point
		((int)image_coordinate.getX (), (int)image_coordinate.getY ());
if ((DEBUG & DEBUG_CONVERTERS) != 0)
	System.out.println
		("<<< Polar_Stereographic_Elliptical_Projection.to_Image: "
			+ image_coordinate);
return image_coordinate;
}

/*==============================================================================
	Derived values
*/
private double Coefficient_T
	(
	double	latitude
	)
{
if ((PI_OVER_2 - Math.abs (latitude)) < DBL_EPSILON)
	return 0.0;
latitude *= Center_Latitude_Sign;
double
	e_sin_lat = Eccentricity * StrictMath.sin (latitude);
return
	StrictMath.tan (0.5 * (PI_OVER_2 - latitude)) /
	StrictMath.pow ((1.0 - e_sin_lat) / (1.0 + e_sin_lat), E_over_2);
}


private double Coefficient_Phi2
	(
	double	coefficient_t
	)
	throws ArithmeticException
{
double
	phi = PI_OVER_2 - 2.0 * StrictMath.atan (coefficient_t),
	difference = Double.MAX_VALUE;
int
	iteration;
for (iteration = 0;
	 iteration < 15 &&
		difference > 0.0000000001;
	 iteration++)
	{
	double
		eccentricity_sine_phi = Eccentricity * StrictMath.sin (phi),
		new_phi = PI_OVER_2 - 2.0
			* StrictMath.atan (coefficient_t
				* StrictMath.pow
					(
					(1.0 - eccentricity_sine_phi) /
					(1.0 + eccentricity_sine_phi),
					E_over_2
					));
	difference = Math.abs (new_phi - phi);
	phi = new_phi;
	}
if (iteration == 15)
	throw new ArithmeticException (ID + '\n'
		+ "Computation of coefficient phi2 failed to converge\n"
		+ "using coefficient t of " + coefficient_t + '.');
return phi;
}

}
