/*	Stream_Logger

PIRL CVS ID: Stream_Logger.java,v 1.17 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor;

import	java.io.InputStream;
import	java.io.InputStreamReader;
import	java.io.BufferedReader;
import	java.io.IOException;
import	java.io.Writer;
import	java.util.Vector;
import	java.util.ListIterator;

/** A <I>Stream_Logger</I> is a Thread that is used to forward the
	lines read from an InputStream to one or more Writers.
<P>
	Once started a Stream_Logger reads a line of input from its
	InputStream and then writes the line to each Writer in its list.
	Writers may be added or removed from the list at any time, before or
	after the start of the Stream_Logger. If no Writer has been bound
	to the Stream_Logger the input stream is still read and buffered.
<P>
	Lines read are stored in a buffer up to an amount that may be
	changed at any time.
<P>
	A Stream_Logger stops when an end of file or IOException has been
	encountered on its InputStream. However, it may be notified to end
	before reading the next line. It may also be notified to close if
	no input has arrived, but continue to the normal end of file (or
	IOException) if data has already arrived.
<P>
	@author		Bradford Castalia - UA/PIRL
	@version	1.17
	@see	Thread
*/
public class Stream_Logger
	extends Thread
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Stream_Logger (1.17 2012/04/16 06:04:10)";

/**	The default {@link #Polling_Interval(long) polling interval}.
*/
public static final int
	DEFAULT_POLLING_INTERVAL	= 300;
//	Initial read polling interval (msec).
private volatile long
	Polling_Interval	= DEFAULT_POLLING_INTERVAL;

//	Input stream.
private String
	Stream_Name;
private static int
	READER_BUFFER_SIZE	= 1024;
private	BufferedReader
	Data_Reader;

//	Output log writers.
private Vector<Writer>
	Writers				= new Vector<Writer> ();

/**	The minimum {@link #Buffer_Size(int) buffer size}.
*/
public static final int
	MINIMUM_BUFFER_SIZE	= READER_BUFFER_SIZE;
/**	The default {@link #Buffer_Size(int) buffer size}.
*/
public static final int
	DEFAULT_BUFFER_SIZE	= 8 * READER_BUFFER_SIZE;
private int
	Line_Buffer_Size	= DEFAULT_BUFFER_SIZE;
//	Last lines buffer.
private StringBuffer
	Line_Buffer			= new StringBuffer (Line_Buffer_Size);
private static String
	NL;
static
	{
	if ((NL = System.getProperty ("line.separator")) == null)
		 NL = "\n";
	}
private static int
	NL_LENGTH			= NL.length ();

//	Stream_Logger state.
private static final int
	IDLE				= 0,
	ABORT				= 1 << 0,
	CLOSE				= 1 << 1,
	READING				= 1 << 2;
private volatile int
	State				= IDLE;


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTOR	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_LOGGING		= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Stream_Logger connecting an InputStream with a log
	Writer.
<P>
	The InputStream is wrapped in an InputStreamReader and a
	BufferedReader. The Writer, if not null, is the first entry in the
	Vector of log Writers.
<P>
	<B>Note</B>: This Thread is marked as a {@link Thread#setDaemon
	daemon} so it does not need to finish before the user application
	can exit.
<P>
	@param	stream_name		The name of the stream to prefix to each line
		logged along with a ": " separator. If null no line prefix is
		added.
	@param	input_stream	The InputStream to be read.
	@param	writer		A Writer to use for logging lines. If
		null, the Stream_Logger will not have any initial Writer.
	@throws	IllegalArgumentException	If the input_stream is null.
*/
public Stream_Logger
	(
	String		stream_name,
	InputStream	input_stream,
	Writer		writer
	)
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">-< Stream_Logger: " + stream_name +
		((writer == null) ? "" : " with Writer"));
if (stream_name == null)
	stream_name = "";
else
	stream_name += ": ";
Stream_Name = stream_name;

if (writer != null)
	Writers.add (writer);

if (input_stream == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't construct a Stream_Logger on a null InputStream.");
Data_Reader = new BufferedReader
	(new InputStreamReader (input_stream), READER_BUFFER_SIZE);
setDaemon (true);
}

/**	Constructs a Stream_Logger for an InputStream without an initial
	log Writer.
<P>
	@param	stream_name		The name of the stream to prepend to
		each line logged. This may be null.
	@param	input_stream	The InputStream to be read.
	@throws	IllegalArgumentException	If the input_stream is null.
	@see	#Stream_Logger(String, InputStream, Writer)
*/
public Stream_Logger
	(
	String		stream_name,
	InputStream	input_stream
	)
{this (stream_name, input_stream, null);}

private Stream_Logger ()
{}

/*==============================================================================
	Accessors
*/
/**	Adds a Writer to the list of writers where read lines will sent.
<P>
	If the Writer is already in the list it is not added.
<P>
	@param	writer	The Writer to add.
	@return	true if the Writer was added to the list; false if it
		was already in the list or is null.
	@see	#Remove(Writer)
*/
public boolean Add
	(
	Writer		writer
	)
{
if (writer == null)
	return false;
boolean
	added;
synchronized (Writers)
	{
	added = ! Writers.contains (writer);
	if (added)
		Writers.add (writer);
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">-< Stream_Logger.Add: " + Stream_Name + ' ' + added);
return added;
}

/**	Removes a Writer from the Vector of log Writers.
<P>
	@param	writer	The Writer to remove.
	@return	true if the Writer was removed to the list; false if it
		was not in the list or is null.
	@see	#Add(Writer)
*/
public boolean Remove
	(
	Writer		writer
	)
{
if (writer == null)
	return false;
synchronized (Writers) {return Writers.remove (writer);}
}

/**	Gets the last-lines-read buffer.
<P>
	If the Stream_Logger is {@link #run() running} the contents of the
	buffer will change if a new line arrives. Obviously, the buffer
	returned should not be modified while the Stream_Logger is running.
	<B>N.B.</B>: Synchronize on the buffer if necessary, though this will
	block reading of further lines until the lock is released.
<P>
	@return	A StringBuffer containing the last lines read from the
		InputStream.
*/
public StringBuffer Buffer ()
{return Line_Buffer;}

/**	Sets the maximum size of the last-lines-read buffer.
<P>
	<B>Note</B>: The buffer size is never allowed to be less than the
	{@link #MINIMUM_BUFFER_SIZE}.
<P>
	<B>Warning</b>: Do not call this method while sychronized on the
	buffer as this will result in a deadlock.
<P>
	@param	size	The maximum number of characters allowed in the
		buffer. If the buffer is currently larger than this amount
		whole lines will be removed from the front of the buffer to
		bring the content amount below the size.
	@return	This Stream_Logger.
*/
public Stream_Logger Buffer_Size
	(
	int		size
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Stream_Logger.Buffer_Size: " + Stream_Name + ' ' + size);
if (size < MINIMUM_BUFFER_SIZE)
	size = MINIMUM_BUFFER_SIZE;
synchronized (Line_Buffer)
	{
	if (size < Line_Buffer.length ())
		{
		//	Buffer overflow; drop leading line(s).
		int
			index = Line_Buffer.indexOf (NL, size);
		if (index < 0)
			//	The current content won't fit; drop it all.
			index = Line_Buffer.length ();
		else
			index += NL_LENGTH;
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Length " + Line_Buffer.length ()
				+ "; drop first " + index + " characters.");
		Line_Buffer.delete (0, index);
		}
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Stream_Logger.Buffer_Size: " + size);
Line_Buffer_Size = size;
return this;
}

/**	Gets the maximum size of the last-lines-read buffer.
<P>
	@return	The maximum number of characters allowed in the buffer.
	@see	#Buffer_Size(int)
*/
public int Buffer_Size ()
{return Line_Buffer_Size;}

/**	Clears the last-lines-read buffer.
<P>
	<B>Warning</b>: Do not call this method while sychronized on the
	buffer as this will result in a deadlock.
*/
public void Clear_Buffer ()
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">-< Stream_Logger.Clear_Buffer: " + Stream_Name);
synchronized (Line_Buffer)
	{Line_Buffer.delete (0, Line_Buffer.length ());}
}

/**	Sets the polling interval to check for the arrival of data.
<P>
	The polling interval is only used while waiting for data to arrive on
	the input stream. After that point blocking reads are used.
<P>
	A short polling interval has the disadvantage of consuming more
	CPU cycles while waiting for data to arrive. This can have an
	adverse affect on system performance, especially if no data
	arrives for a long time.
<P>
	A long polling interval has the disadvantage of introducing
	a delay up to the interval time between when data first arrives
	and input actually begins. This is generally not a problem if
	the delay is not so long as to interfere with data delivery
	downstream or cause data loss due to buffer overflow upstream.
<P>
	@param	interval	The amount of time to wait, in milliseconds,
		between tests to see if data has arrived. The minimum time is one
		millisecond. This initial default is {@link
		#DEFAULT_POLLING_INTERVAL} milliseconds.
	@return	This Stream_Logger.
	@see	#Close()
*/
public Stream_Logger Polling_Interval
	(
	long	interval
	)
{
if (interval < 1)
	interval = 1;
Polling_Interval = interval;
return this;
}

/**	Gets the polling interval to check for the initial arrival of data.
<P>
	@return	The polling interval in milliseconds.
	@see	#Polling_Interval(long)
*/
public long Polling_Interval ()
{return Polling_Interval;}

/*==============================================================================
	Runnable
*/
/**	Runs the Stream_Logger.
<P>
	Each line of input {@link BufferedReader#readLine() read} from the
	InputStream is written to each Writer in the Vector of Writers. Each
	line written is prepended with the stream name followed by ": ",
	unless no stream name was provided. The system's line ending
	sequence, from the "line.separator" property, is appended to each
	line written.
<P>
	Each line read is also appended to the last-lines-read buffer. Only
	entire lines are buffered; lines that are too big to fit in the
	buffer are dropped. Lines, and only entire lines, are removed from
	the front of the buffer as needed to make room for new lines to be
	added at the end.
<P>
	Logging continues until an end-of-file or IOException is encountered
	on the InputStream, or the logger is told to <CODE>{@link #End()
	End}</CODE>.
<P>
	<B>N.B.</B>: If a Writer throws an IOException it is removed from
	the list of Writers.
*/
public void run ()
{log_lines ();}

/**	Ends the Stream_Logger.
<P>
	Logging ends before the next line is read, but after the
	current line has finished logging.
<P>
	<B>N.B.</B>: Logging stops even if there may be more input
	available from the InputStream after the current line.
*/
public void End ()
{State |= ABORT;}

/**	Effectively closes the input stream.
<P>
	The input stream is initially polled for a ready condition
	(characters are available) before begining to read lines. This
	prevents the Stream_Logger from becoming blocked on input which
	will never arrive, thus allowing it to be externally halted.
	However, it may not be known if input has arrived or not.
<P>
	If input has not arrived then the Stream_Logger is to stop polling
	and <CODE>{@link #End() End}</CODE>. If input has arrived then the
	Stream_Logger is to continue reading until EOF or an IOException
	occurs.
*/
public void Close ()
{State |= CLOSE;}

/*------------------------------------------------------------------------------
	Logging
*/
private void log_lines ()
{
State |= READING;
if ((DEBUG & DEBUG_LOGGING) != 0)
	System.out.println
		(">>> Stream_Logger.log_lines: " + Stream_Name
			+ " State = " + State + " - " + Logging_State ());
String
	line = null;
try
	{
	//	Wait for data to arrive.
	while (State == READING &&
			! Data_Reader.ready ())
		{
		try {sleep (Polling_Interval);}
		catch (InterruptedException e) {}
		}
	if ((DEBUG & DEBUG_LOGGING) != 0)
		System.out.println
			("    Stream_Logger.log_lines: " + Stream_Name
				+ " State = " + State + " - " + Logging_State ());
	if (Data_Reader.ready ())
		{
		while ((State & ABORT) == 0)
			{
			//	Read until EOF or IOException.
			line = Data_Reader.readLine ();
			if (line == null)
				{
				//	EOF.
				if ((DEBUG & DEBUG_LOGGING) != 0)
					System.out.println
						("    Stream_Logger.log_lines: "
							+ Stream_Name + " EOF");
				break;
				}
			log (line);
			yield ();
			if ((DEBUG & DEBUG_LOGGING) != 0)
				System.out.println
					("    Stream_Logger.log_lines: " + Stream_Name
						+ " State = " + State + " - " + Logging_State ());
			}
		}
	}
catch (IOException exception)
	{
	if ((DEBUG & DEBUG_LOGGING) != 0)
		System.out.println
			("    Stream_Logger.log_lines: " + Stream_Name
				+ " IOException - " + exception.getMessage ());
	}
if ((DEBUG & DEBUG_LOGGING) != 0)
	System.out.println
		("<<< Stream_Logger.log_lines: " + Stream_Name
			+ " State = " + State + " - " + Logging_State ());
State = IDLE;
}


private String Logging_State ()
{
String
	state = "";
if (State == IDLE)
	state = "IDLE";
else
	{
	if ((State & READING) != 0)
		state = "READING";
	if ((State & CLOSE) != 0)
		{
		if (state.length () != 0)
			state += " ";
		state += "CLOSE";
		}
	if ((State & ABORT) != 0)
		{
		if (state.length () != 0)
			state += " ";
		state += "ABORT";
		}
	}
return state;
}


private void log
	(
	String	line
	)
{
if ((DEBUG & DEBUG_LOGGING) != 0)
	System.out.println
		(">-< Stream_Logger.log: " + Stream_Name + " " + line);
//	Send the line to each writer.
Writer
	writer;
synchronized (Writers)
	{
	ListIterator<Writer>
		writers = Writers.listIterator ();
	while (writers.hasNext ())
		{
		writer = writers.next ();
		try
			{
			synchronized (writer)
				{
				writer.write (Stream_Name + line + NL);
				writer.flush ();
				}
			}
		catch (IOException exception)
			{writers.remove ();}
		}
	}

//	Put the line in the buffer.
int
	index = Line_Buffer.length () + line.length ();
if (index > Line_Buffer_Size)
	{
	//	Buffer overflow; drop leading line(s).
	index = Line_Buffer.indexOf (NL, index -= Line_Buffer_Size);
	if (index < 0)
		//	The line won't fit; drop it.
		return;
	index += NL_LENGTH;
	Line_Buffer.delete (0, index);
	}
Line_Buffer.append (line);
}

}	//	End of Stream_Logger class.

