/*	Remote_Theater

PIRL CVS ID: Remote_Theater.java,v 1.36 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Conductor.Management;
import	PIRL.Conductor.Processing_Listener;
import	PIRL.Conductor.Processing_Changes;
import	PIRL.Conductor.Processing_Event;
import	PIRL.Messenger.*;
import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Strings.String_Buffer;
import	PIRL.Utilities.Styled_Multiwriter;

import	java.io.Writer;
import	java.io.IOException;
import	java.io.EOFException;
import	java.util.Vector;
import	java.util.Iterator;


/**	A <i>Remote_Theater</i> implements the remote Conductor management
	side of a {@link Theater}.
<p>
	A Remote_Theater is used by network distributed clients that will be
	using the {@link Management} interface of Conductors via a
	Stage_Manager proxy server.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.36
	@see		Message_Delivered_Listener
*/
public class Remote_Theater
	extends
		Theater
	implements
		Management,
		Message_Delivered_Listener
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Remote_Theater (1.36 2012/04/16 06:04:11)";


/**	The name to be used in describing objects of this class.
*/
public static final String
	REMOTE_THEATER_NAME			= "Remote_Theater";

private static final Message
	REMOTE_THEATER_IDENTITY;
static
	{
	REMOTE_THEATER_IDENTITY = Message.Identity ()
	.Set (NAME_PARAMETER_NAME, REMOTE_THEATER_NAME)
	.Set (CLASS_ID_PARAMETER_NAME, ID);
	}

//	Also used as the wait-response lock.
private Message
	Conductor_Identity			= new Message ();

private String
	Authentication_Key			= null;

/**	The default amount of time (seconds) to wait for a response to a
	protocol message.
*/
public static final int
	DEFAULT_RESPONSE_TIMEOUT	= 10;
private volatile int
	Response_Timeout			= DEFAULT_RESPONSE_TIMEOUT;

/**	The default number of protocol transmission retries before a
	Remote_Management_Exception will be thrown.
*/
public static final int
	DEFAULT_MESSAGE_RETRIES		= 3;
private int
	Message_Retries				= DEFAULT_MESSAGE_RETRIES;

private Thread
	Waiting						= null;
private Message
	Response					= null;


private Vector<Processing_Listener>
	Processing_Listeners		= new Vector<Processing_Listener> ();

/**	Object to which non-protocol Messages will be delivered.
*/
protected Message_Delivered_Listener
	Non_Protocol_Message_Listener	= null;

private Styled_Multiwriter
	Log_Writer					= new Styled_Multiwriter ();


/**	Problem case message accounting.
*/
protected int
	Undeliverable_Messages		= 0,
	NACK_Messages				= 0,
	Unknown_Messages			= 0;


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONNECTION			= 1 << 1,
	DEBUG_MESSAGES				= 1 << 2,
	DEBUG_PROCESSING_LISTENER	= 1 << 3,
	DEBUG_LOG_WRITER			= 1 << 4,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Remote_Theater
	(
	String	host,
	int		port,
	Message	identity,
	String	connection_address
	)
	throws IOException
{Open (host, port, identity, connection_address);}


public Remote_Theater
	(
	String	host,
	Message	identity,
	String	connection_address
	)
	throws IOException
{Open (host, 0, identity, connection_address);}


public Remote_Theater ()
{}

/*==============================================================================
	Accessors
*/
/**	Get the amount of time to wait for a response to a protocol message.
<p>
	@return	The maximum amount of time (seconds) to wait for a response
		to a protocol message.
*/
public int Response_Timeout ()
{return Response_Timeout;}

/**	Set the amount of time to wait for a response to a protocol message.
<p>
	@param	seconds	The maximum amount of time (seconds) to wait for a response
		to a protocol message. If less than or equal to zero the
		{@link #DEFAULT_RESPONSE_TIMEOUT} will be used.
	@return	This Remote_Theater object.
*/
public Remote_Theater Response_Timeout
	(
	int	seconds
	)
{
if (seconds <= 0)
	seconds = DEFAULT_RESPONSE_TIMEOUT;
Response_Timeout = seconds;
return this;
}

/**	Get the identity of the Conductor with which this Remote_Theater is
	associated.
<p>
	<b>N.B.</b>: The availability of a Conductor identity does not imply
	that this Remote_Theater is {@link Theater#Opened() opened} for
	that Conductor.
<p>
	@return	A {@link PIRL.Conductor.Conductor#Identity() Conductor
		identity Message}. This will be an empty Message if no identity
		has yet been obtained. It will be null if the identity has become
		corrupted.
*/
public Message Conductor_Identity ()
{
Message
	identity = null;
if (Conductor_Identity != null)
	{
	try {identity = new Message (Conductor_Identity);}
	catch (PVL_Exception exception) {}
	}
return identity;
}

/**	Get the maximum number of protocol message retries before a
	Remote_Management_Exception will be thrown.
<p>
	@return	The maximum number of protocol message retries.
*/
public int Message_Retries ()
{return Message_Retries;}

/**	Set the number of protocol message retries before a
	Remote_Management_Exception will be thrown.
<p>
	@param	retries	The maximum number of protocol message retries.
		If zero the {@link #DEFAULT_MESSAGE_RETRIES} will be used. If
		negative retries will continue until successful (this could
		result in an indefintate protocol hang).
	@return	This Remote_Theater object.
*/
public Remote_Theater Message_Retries
	(
	int	retries
	)
{
if (retries == 0)
	retries = DEFAULT_MESSAGE_RETRIES;
Message_Retries = retries;
return this;
}

/**	Get the number of undeliverable messages that have occurred.
<p>
	An underliverable Message contains the {@link #UNDELIVERABLE_ACTION}
	value for the {@link #ACTION_PARAMETER_NAME}.
<p>
	@return	The number of undeliverable messages that have occurred.
*/
public int Undeliverable_Messages ()
{return Undeliverable_Messages;}

/**	Get the number of NACK (negative acknowledge) messages that have occurred.
<p>
	A NACK Message contains the {@link #NACK_ACTION} value for the
	{@link #ACTION_PARAMETER_NAME}.
<P>
	@return	The number of NACK messages that have occurred.
*/
public int NACK_Messages ()
{return NACK_Messages;}

/**	Get the number of unknown messages that have occurred.
<p>
	An unknown message has no {@link #ACTION_PARAMETER_NAME}, this
	parameter has no value, or the value is not known as a protocol
	action. A {@link Message_Delivered_Listener listener} may be
	set to receive non-protocol messages.
<p>
	@return	The number of undeliverable messages that have occurred.
	@see	#Non_Protocol_Message_Listener(Message_Delivered_Listener)
*/
public int Unknown_Messages ()
{return Unknown_Messages;}

/**	Get the listener for non-protocol messages.
<p>
	@return	A Message_Delivered_Listener to which non-protocol messages
		will be delivered. This may be null.
	@see	#Non_Protocol_Message_Listener(Message_Delivered_Listener)
*/
public Message_Delivered_Listener Non_Protocol_Message_Listener ()
{
if (Non_Protocol_Message_Listener == null)
	return null;
else
	{
	synchronized (Non_Protocol_Message_Listener)
		{return Non_Protocol_Message_Listener;}
	}
}

/**	Set the listenter for non-protocol messages.
<p>
	A non-protocol message does not participate in the implementation of
	the remote Conductor Management interface. This includes Messages
	with the {@link #DONE_ACTION}, {@link #UNDELIVERABLE_ACTION}, {@link
	#NACK_ACTION} or {@link #MESSENGERS_REPORT_ACTION} value for the
	{@link #ACTION_PARAMETER_NAME}, or any message that does not have a
	recognized protocol action.
<p>
	@param	listener	A Message_Delivered_Listener to which non-protocol
		messages will be delivered.
	@return	This Remote_Theater object.
	@see	#Non_Protocol_Message_Listener()
*/
public Remote_Theater Non_Protocol_Message_Listener
	(
	Message_Delivered_Listener	listener
	)
{
if (Non_Protocol_Message_Listener == null)
	Non_Protocol_Message_Listener = listener;
else
	{
	synchronized (Non_Protocol_Message_Listener)
		{Non_Protocol_Message_Listener = listener;}
	}
return this;
}


public String toString ()
{
return ID + NL
	+ super.toString ();
}

/*==============================================================================
	Stage_Manager
*/
/**	Open this Theater.
<p>
	The current communication channel is {@link #Close() closed}. A new
	{@link Theater#Open(String, int, Message) connection} with the
	Stage_Manager is established using the identity provided.
<p>
	Then a {@link #CONDUCTOR_CONNECT_ACTION} Message is sent to the
	Stage_Manager with the connection ID. The response is checked for an
	{@link #EXPLANATION_PARAMETER_NAME} indicating that the request could
	not be satisfied (unknown Messenger ID). If the response is good then
	the {@link Message#Route_To() route-to list} of the Conductor
	Identity Message is set to the value provided by its {@link
	#ROUTE_TO_PARAMETER_NAME} parameter.
<p>
	@param	host	The name or IP address of the host system where a
		Stage_Manager is running. If null, "localhost" will be used.
	@param	port		The port number to use to establish the
		communication connection. If less than or equal to zero the
		{@link Theater#Default_Port(int) default port} will be used.
	@param	identity	The Message used to provide the Stage_Manager
		with the required identity information.
	@param	connection_address	The Messenger {@link Messenger#Address()
		address} used by the Stage_Manager to identify the Messenger used
		to establish a connection with the Local_Theater object
		bound to a Conductor.
	@throws	IOException	If a connection could not be established to
		the Stage_Manager. This will be a Theater_Protocol_Exception
		if there were any problems in the content of the Messages
		used to establish the connection.
	@throws	IllegalArgumentException	If the identity or connection_address
		is null.
	@see	Theater#Open(String, int, Message)
*/
public void Open
	(
	String	host,
	int		port,
	Message	identity,
	String	connection_address
	)
	throws IOException
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">>> Remote_Theater.Open:" + NL
		+"    Host - " + host + NL
		+"    Port - " + port + NL
		+"    Identity - " + NL
			+ identity + NL
		+"    Connection address - " + connection_address);

if (identity == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't open without an identity.");
if (connection_address == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't open without a Conductor connection address.");

//	Close if opened.
Close ();

//	Connect to the Stage_Manager.
Open (host, port, identity);

//	Request a Conductor connection.
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Sending Conductor connection Message");
Send_Message
	(Message
	.Action (CONDUCTOR_CONNECT_ACTION)
	.Set (ADDRESS_PARAMETER_NAME, connection_address));

Message
	response = Receive_Message ();
if (response == null ||
	response.Get (EXPLANATION_PARAMETER_NAME) != null)
	{
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "Could not open because the Conductor connection failed." + NL
		+ ((response == null) ?
			("No response to the " + CONDUCTOR_CONNECT_ACTION + " request" + NL
			+"after " + Receive_Timeout () + " seconds.") :
			response.Get (EXPLANATION_PARAMETER_NAME)),
		((response == null) ?
			Theater_Protocol_Exception.TIMEOUT :
			Theater_Protocol_Exception.INVALID_MESSAGE));
	}

//	Validate the response message.
Parameter
	parameter = response.Find (IDENTITY_ACTION);
if (parameter == null)
	{
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "Could not open because the Conductor connection failed:" + NL
		+ "No Conductor identity in the connection response -" + NL
		+ response.Description (),
		Theater_Protocol_Exception.INVALID_MESSAGE);
	}
try {Conductor_Identity = new Message (parameter);}
catch (PVL_Exception exception)
	{
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "Could not open because the Conductor connection failed:" + NL
		+ "Invalid Conductor identity -" + NL
		+ parameter.Description (),
		Theater_Protocol_Exception.INVALID_MESSAGE);
	}
parameter = Conductor_Identity.Find (ROUTE_TO_PARAMETER_NAME);
if (parameter == null)
	{
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "Could not open because the Conductor connection failed:" + NL
		+ "No " + ROUTE_TO_PARAMETER_NAME
			+ " parameter in the Conductor identity -" + NL
		+ Conductor_Identity,
		Theater_Protocol_Exception.INVALID_MESSAGE);
	}
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Conductor route-to: " + parameter.Description ());

//	Set the route-to list of the identity message.
try {Conductor_Identity.Route_To (parameter.Value ());}
catch (PVL_Exception exception)
	{
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "Could not open because the Conductor connection failed:" + NL
		+ "Invalid " + ROUTE_TO_PARAMETER_NAME
			+ " parameter value in the Conductor identity -" + NL
		+ Conductor_Identity,
		Theater_Protocol_Exception.INVALID_MESSAGE,
		exception);
	}
String
	connected_address = Conductor_Identity.Get (ADDRESS_PARAMETER_NAME);
if (connected_address == null)
	Conductor_Identity.Set (ADDRESS_PARAMETER_NAME, connection_address);
else
if (! connected_address.equals (connection_address))
	{
	Conductor_Identity.Set (ADDRESS_PARAMETER_NAME, connection_address);
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "The opened theater's connection address - " + connected_address + NL
		+ "is not the requested connection address - " + connection_address,
		Theater_Protocol_Exception.INVALID_MESSAGE);
	}

//	Start asychronous message listening.
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Starting asychronous message listening");
Employer (this);
Listen_for_Messages ();
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("<<< Remote_Theater.Open");
}

/**	Open this Theater.
<p>
	The {@link Theater#Default_Port(int) default port} will be used.
<p>
	@param	host	The name or IP address of the host system where a
		Stage_Manager is running. If null, "localhost" will be used.
	@param	identity	The Message used to provide the Stage_Manager
		with the required identity information.
	@param	connection_address	The Messenger {@link Messenger#Address()
		address} used by the Stage_Manager to identify the Messenger used
		to establish a connection with the Local_Theater object
		bound to a Conductor.
	@throws	IOException	If a connection could not be established to
		the Stage_Manager. This will be a Theater_Protocol_Exception
		if there were any problems in the content of the Messages
		used to establish the connection.
	@throws	IllegalArgumentException	If the identity or connection_address
		is null.
	@see	#Open(String, int, Message, String)
*/
public void Open
	(
	String	host,
	Message	identity,
	String	connection_address
	)
	throws IOException
{Open (host, 0, identity, connection_address);}

/**	Close this Theater.
<p>
	If currently {@link Theater#Opened() opened} a {@link
	Theater#CONDUCTOR_DISCONNECT_ACTION} Message is {@link
	Theater#Send_Message(Message) sent} with the {@link
	Theater#ADDRESS_PARAMETER_NAME} set from the same parameter of the
	{@link #Conductor_Identity() Conductor identity}. Then the
	communication channel is {@link Theater#Close() closed}.
<p>
	@return	true if this Remote_Theater was opened when this method was
		invoked; false if it was already closed.
*/
public synchronized boolean Close ()
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">>> Remote_Theater.Close");
boolean
	closed = false;
if (Opened ())
	{
	String
		connection_address = Conductor_Identity.Get (ADDRESS_PARAMETER_NAME);
	if (connection_address != null)
		{
		try {Send_Message
				(Message
				.Action (CONDUCTOR_DISCONNECT_ACTION)
				.Set (ADDRESS_PARAMETER_NAME, connection_address));}
		catch (IOException exception)
			{/* Nothing for it */}
		}
	super.Close ();
	closed = true;
	}
Response (null);
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("<<< Remote_Theater.Close: " + closed);
return closed;
}

/*==============================================================================
	Messages
*/
//	Message_Delivered_Listener
/**	Get the identity description of this Conductor Management interface
	implementation.
<p>
	@return	A Message describing this Remote_Theater object.
*/
public Message Listener_Identity ()
{return REMOTE_THEATER_IDENTITY;}


//	Message_Delivered_Listener
/**	Take delivery of a Message.
<p>
	The Action parameter value of the event's
	Message is used to determine an {@link #Action_Code(String) action
	code} that selects a handler method if the Action is known.

	The delivered Message is handled before returning. Most Messages
	will just be posted as a Response which will wake up any thread that
	is {@link #Wait_for_Response() waiting for a response}. Typically
	this happens as a result of a {@link #Send_and_Wait(Message)
	send-and_wait message}.
<p>
	Messages that are asynchronously conveying {@link
	#Add_Processing_Listener(Processing_Listener) Conductor processing
	events} or {@link #Add_Log_Writer(Writer) Conductor log reports} are
	handled immediately.
<p>
	Non-protocol messages are delivered to the {@link
	#Non_Protocol_Message_Listener(Message_Delivered_Listener) non-protocol
	listener}, if there is one.
*/
public void Message_Delivered
	(
	Message_Delivered_Event	event
	)
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Remote_Theater.Message_Delivered:" + NL
		+"    Message -" + NL
		+ event.Message.Routing () + NL
		+ event.Message);

switch (Action_Code (event.Message.Action ()))
	{
	//	Asynchronous protocol messages:

	case PROCESSING_CHANGES_CODE:
		Processing_Event_Occurred (event.Message);
		break;

	case LOG_WRITER_CODE:
		Write_Log (event.Message);
		break;

	//	Protocol response messages:

	case IDENTIFY_CODE:
	case CONDUCTOR_CONNECT_CODE:
	case CONDUCTOR_DISCONNECT_CODE:
	case IDENTITY_CODE:
	//	 START_CODE
	case PROCESSING_STATE_CODE:
	case STOP_CODE:
	//	 QUIT_CODE
	case CONFIGURATION_CODE:
	case SOURCES_CODE:
	case PROCEDURES_CODE:
	case GET_POLL_INTERVAL_CODE:
	//	 SET_POLL_INTERVAL_CODE
	case GET_RESOLVER_DEFAULT_VALUE_CODE:
	//	 SET_RESOLVER_DEFAULT_VALUE_CODE
	case GET_STOP_ON_FAILURE_CODE:
	//	 SET_STOP_ON_FAILURE_CODE
	case SEQUENTIAL_FAILURES_CODE:
	//	 RESET_SEQUENTIAL_FAILURES_CODE
	case PROCESSING_EXCEPTION_CODE:
	case CONDUCTOR_STATE_CODE:
	//	 ADD_PROCESSING_LISTENER_CODE
	//	 REMOVE_PROCESSING_LISTENER_CODE
	//	 ADD_LOG_WRITER_CODE
	//	 REMOVE_LOG_WRITER_CODE
	//	 ENABLE_LOG_WRITER_CODE
		Response (event.Message);
		break;

	//	Basic Messages:

	case DONE_CODE:
		Close ();
		break;

	case UNDELIVERABLE_CODE:
		//!!!	Error handling or reporting?
		Undeliverable_Messages++;
		Response (event.Message);
		Non_Protocol_Message (event);
		break;

	case NACK_CODE:
		//!!!	Error handling or reporting?
		NACK_Messages++;
		Response (event.Message);

	//	 MESSENGERS_REPORT_CODE
	default:
		Non_Protocol_Message (event);
		break;
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Remote_Theater.Message_Delivered");
}


private class Waiting_for_Response
	extends Thread
{
public void run ()
{
try {sleep (Response_Timeout * 1000);}
catch (InterruptedException exception) {}
}
}	//	End of Waiting_for_Response class.

/**	Send a Message and wait for a response.
<p>
	The previous response is cleared, the Message is sent and then the
	current thread is blocked {@link #Wait_for_Response() waiting for a
	response}. When the response arrives it is returned.
<p>
	@param	message	The Message to be sent.
	@return	The response Message. This will be null if the connection to
		the Stage_Manager was lost or the response did not arrive before
		the {@link #Response_Timeout(int) response timeout} expired.
		Check if the Stage_Manager is {@link #Opened() opened} to
		determine the exact cause of a null response.
	@throws	Remote_Management_Exception	If Message could not be sent
		because it contains invalid PVL.
*/
protected Message Send_and_Wait
	(
	Message	message
	)
{
Response = null;
int
	tries = 0;
while (true)
	{
	try {Send_Message (message); break;}
	catch (IOException exception)
		{
		if (exception instanceof EOFException)
			//	Connection lost.
			return null;
		else if (++tries == Message_Retries)
			throw new Remote_Management_Exception (ID + NL
				+ "Unable to send message -" + NL
				+ message + NL
				+ exception);
		}
	}
return Wait_for_Response ();
}

/**	Wait for a response Message.
<p>
	If a non-null response is currently posted that will be immediately
	returned without any waiting. <b>N.B.</b>: The previously posted
	response must be cleared when it is no longer relevant or it will be
	incorrectly returned as the new response; {@link
	#Send_and_Wait(Message)}, for example, clears the response before
	sending a message that is expected to result in a new response.
<p>
	After confirming that there is no response available a thread is
	started that will sleep until the {@link #Response_Timeout() response
	timeout} expires or it is woken when a response arrives. The current
	thread joins the waiting thread which prevents the method from
	returning until a response arrives or the timeout occurs.
<p>
	@return	The response Message. This will be null if a response timeout
		occurred before a response arrived.
	@throws	IllegalThreadStateException	If a waiting thread is active.
*/
protected Message Wait_for_Response ()
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Remote_Theater.Wait_for_Response");
if (Waiting != null &&
	Waiting.isAlive ())
	throw new IllegalThreadStateException (ID + NL
		+ "Already waiting for a response.");
synchronized (Conductor_Identity)
	{
	if (Response != null)
		return Response;
	Waiting = new Waiting_for_Response (); 
	Waiting.start ();
	}
while (Waiting.isAlive ())
	{
	try {Waiting.join ();}
	catch (InterruptedException exception) 
		{
		if ((DEBUG & DEBUG_MESSAGES) != 0)
			System.out.println
				("--- Remote_Theater.Wait_for_Response: interrupted");
		}
	}
Waiting = null;
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Remote_Theater.Wait_for_Response");
return Response;
}

/**	Set the Stage_Manager response Message.
<p>
	The Conductor_Identity is used as a lock object to synchronize the
	update of the Response Message and get the response Waiting thread.
	If the Waiting thread is not null and is alive then it is interrupted
	to release the Wait_for_Response block.
<p>
	@param	message	The Message to be set as the Response. This may be
		null to clear the Response and stop the Waiting thread.
*/
private void Response
	(
	Message	message
	)
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Remote_Theater.Response:" + NL
		+ message);
Thread
	waiting;
synchronized (Conductor_Identity)
	{
	Response = message;
	waiting = Waiting;
	}
if (waiting != null &&
	waiting.isAlive ())
	{
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			("    Interrupting the Waiting Thread");
	try {waiting.interrupt ();}
	catch (SecurityException exception) {}
	catch (IllegalThreadStateException exception) {}
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Remote_Theater.Response");
}

/*==============================================================================
	Management
*/
public Message Identity ()
{
Protocol_Action_Response (IDENTITY_ACTION, null);
return Response;
}


public void Start ()
{Protocol_Action (START_ACTION, null);}


public void Stop ()
{Protocol_Action (STOP_ACTION, null);}


public void Quit ()
{Protocol_Action (QUIT_ACTION, null);}


public int Processing_State ()
{
int
	tries = 0;
while (true)
	{
	try {return Integer.parseInt
			(Protocol_Action_Response
				(PROCESSING_STATE_ACTION, VALUE_PARAMETER_NAME));}
	catch (NumberFormatException exception)
		{
		if (++tries == Message_Retries)
			throw new Remote_Management_Exception (ID + NL
				+ "Invalid processing state value received.");
		}
	}
}


public Configuration Configuration ()
{
Protocol_Action_Response (CONFIGURATION_ACTION, null);
if (Response.Get (EXPLANATION_PARAMETER_NAME) != null)
	//	No configuration.
	return null;
Parameter
	group = Response.Find (CONFIGURATION_PARAMETER_NAME, Parameter.AGGREGATE);
if (group == null)
	return null;
Configuration
	configuration = null;
Configuration.Include_Default = false;
int
	tries = 0;
while (true)
	{
	try
		{
		configuration = new Configuration (group);
		String
			name = Response.Get (NAME_PARAMETER_NAME);
		if (name != null)
			configuration.Name (name);
		break;
		}
	catch (Configuration_Exception exception)
		{
		if (++tries == Message_Retries)
			throw new Remote_Management_Exception (ID + NL
				+ "Invalid Configuration received.");
		}
	}
return configuration;
}


public Management Poll_Interval
	(
	int		seconds
	)
{
Protocol_Action (SET_POLL_INTERVAL_ACTION, String.valueOf (seconds));
return this;
}


public int Poll_Interval ()
{
int
	tries = 0;
while (true)
	{
	try {return Integer.parseInt
			(Protocol_Action_Response
				(GET_POLL_INTERVAL_ACTION, VALUE_PARAMETER_NAME));}
	catch (NumberFormatException exception)
		{
		if (++tries == Message_Retries)
			throw new Remote_Management_Exception (ID + NL
				+ "Invalid poll interval value received.");
		}
	}
}


public Management Resolver_Default_Value
	(
	String	value
	)
{
Protocol_Action (SET_RESOLVER_DEFAULT_VALUE_ACTION,
	(value == null) ? "null" : value);
return this;
}


public String Resolver_Default_Value ()
{
String
	value = Protocol_Action_Response
		(GET_RESOLVER_DEFAULT_VALUE_ACTION, VALUE_PARAMETER_NAME);
return (value.equals ("null") ? null : value);
}


public Management Stop_on_Failure
	(
	int		failure_count
	)
{
Protocol_Action
	(SET_STOP_ON_FAILURE_ACTION, String.valueOf (failure_count));
return this;
}


public int Stop_on_Failure ()
{
int
	tries = 0;
while (true)
	{
	try {return Integer.parseInt
			(Protocol_Action_Response
				(GET_STOP_ON_FAILURE_ACTION, VALUE_PARAMETER_NAME));}
	catch (NumberFormatException exception)
		{
		if (++tries == Message_Retries)
			throw new Remote_Management_Exception (ID + NL
				+ "Invalid stop on failure value received.");
		}
	}
}


public int Sequential_Failures ()
{
int
	tries = 0;
while (true)
	{
	try {return Integer.parseInt
			(Protocol_Action_Response
				(SEQUENTIAL_FAILURES_ACTION, VALUE_PARAMETER_NAME));}
	catch (NumberFormatException exception)
		{
		if (++tries == Message_Retries)
			throw new Remote_Management_Exception (ID + NL
				+ "Invalid sequential failures value received.");
		}
	}
}


public Management Reset_Sequential_Failures ()
{
Protocol_Action (RESET_SEQUENTIAL_FAILURES_ACTION, null);
return this;
}

/**	Get the most recent Exception from the Conductor.
<p>
	<b.N.B.<b>: A generic Exception will be returned since the specific
	type was not conveyed through the messaging protocol (an enhancement
	could provide this...).
<p>
	@return	An Exception with the Conductor Exception message. This will
		be null if no Exception is obtained from the Conductor.
*/
public Exception Processing_Exception ()
{
String
	value = Protocol_Action_Response
		(PROCESSING_STATE_ACTION, VALUE_PARAMETER_NAME);
return (value.equals ("null") ? null : new Exception (value));
}


public boolean Connected_to_Stage_Manager ()
{return Opened ();}

/*------------------------------------------------------------------------------
	Table Records
*/
public Vector<Vector<String>> Sources ()
{
Protocol_Action_Response (SOURCES_ACTION, null);
return Table (Response.Value_of (TABLE_PARAMETER_NAME));
}


public Vector<Vector<String>> Procedures ()
{
Protocol_Action_Response (PROCEDURES_ACTION, null);
return Table (Response.Value_of (TABLE_PARAMETER_NAME));
}

/*------------------------------------------------------------------------------
	Protocol Message
*/
/**	Send a protocol Message without waiting for a response.
<p>
	A Message specifying the protocol action and, optionally, a protocol
	value, is generated and sent with routing to the Messenger having the
	Identity with which this object was constructed. The Message is expected
	to be delivered to a Local_Theater bound to a Conductor.
<p>
	@param	action	The {@link Message#Action(String) Action} value of
		the Message.
	@param	value	The {@link #VALUE_PARAMETER_NAME} value to be added to
		the Message. If null, this parameter is not included.
	@throws	Remote_Management_Exception	If a PVL_Exception occured
		indicating an invalid Message.
*/
private void Protocol_Action
	(
	String	action,
	String	value
	)
{
try {Send_Message
	(Message
	.Action (action)
	.Set (VALUE_PARAMETER_NAME, value)
	.Route_To (Conductor_Identity));}
catch (IOException exception)
	{/* Connection lost. */}
}

/**	Send a protocol Message and wait for a response.
<p>
	A Message specifying the protocol action is generated and sent with
	routing to the Messenger having the Identity with which this object
	was constructed. The Message is expected to be delivered to a
	Local_Theater bound to a Conductor. The current thread waits for a
	response. Then the validity of the response is checked. If a value
	was expected it is returned.
<p>
	@param	action	The {@link Message#Action(String) Action} value of
		the Message.
	@param	value_parameter_name	The name of the response parameter
		that is expected to contain a response value.
		If null, no value is expected.
	@return	The value of the response value_parameter_name parameter.
		This will be null if no value is expected.
	@throws	Remote_Management_Exception	If a protocol error occurred.
		This could be due to a PVL_Exception indicating an invalid
		Message, timeout while waiting for a response, a response that
		does not have a matching action parameter, or the response
		message not containing the required value parameter.
*/
private String Protocol_Action_Response
	(
	String	action,
	String	value_parameter_name
	)
{
String
	value = null;
Message
	send =
		Message
		.Action (action)
		.Route_To (Conductor_Identity),
	response;
int
	tries = 0;
while (true)
	{
	response = Send_and_Wait (send);
	++tries;
	if (response == null)
		{
		if (tries != Message_Retries)
			continue;
		throw new Remote_Management_Exception (ID + NL
			+ "No " + action + " action response after "
				+ Response_Timeout
				+ " second" + ((Response_Timeout > 1) ? "s." : "."));
		}
	if (! action.equals (response.Action ()))
		{
		if (tries != Message_Retries)
			continue;
		throw new Remote_Management_Exception (ID + NL
			+ "Invalid " + action + " action response -" + NL
			+ response);
		}
	if (value_parameter_name != null)
		{
		value = response.Get (value_parameter_name);
		if (value == null)
			{
			if (tries != Message_Retries)
				continue;
			throw new Remote_Management_Exception (ID + NL
				+ "Invalid " + action + " action response." + NL
				+ "Missing " + value_parameter_name + " parameter -" + NL
				+ response);
			}
		}
	break;
	}
return value;
}


private void Non_Protocol_Message
	(
	Message_Delivered_Event	event
	)
{
Unknown_Messages++;
if (Non_Protocol_Message_Listener != null)
	Non_Protocol_Message_Listener.Message_Delivered (event);
}

/*------------------------------------------------------------------------------
	Processing_Changes
*/
public Processing_Changes State ()
{
Protocol_Action_Response (CONDUCTOR_STATE_ACTION, null);
return Processing_Changes (Response);
}


public Management Add_Processing_Listener
	(
	Processing_Listener	listener
	)
{
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		(">>> Remote_Theater.Add_Processing_Listener:" + NL
		+ listener);
if (listener != null)
	{
	boolean
		report = false;
	synchronized (Processing_Listeners)
		{
		if (Processing_Listeners.isEmpty ())
			{
			if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
				System.out.println
					("    Processing_Listeners is empty");
			Processing_Listeners.add (listener);
			Protocol_Action (ADD_PROCESSING_LISTENER_ACTION, null);
			}
		else
		if (! Processing_Listeners.contains (listener))
			{
			if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
				System.out.println
					("    Added to Processing_Listeners");
			Processing_Listeners.add (listener);
			report = true;
			}
		if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
			System.out.println
				("    " + Processing_Listeners.size ()
					+ " Processing_Listeners");
		}

	if (report)
		{
		//	Report the current Conductor state.
		Processing_Changes
			changes = State ();
		if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
			System.out.println
				("    Reporting Processing_Changes -" + NL
				+ changes);
		if (changes != null)
			listener.Processing_Event_Occurred
				(new Processing_Event (this, changes));
		}
	}
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("<<< Remote_Theater.Add_Processing_Listener");
return this;
}


public boolean Remove_Processing_Listener
	(
	Processing_Listener	listener
	)
{
boolean
	removed = false;
if (listener != null)
	{
	synchronized (Processing_Listeners)
		{
		removed = Processing_Listeners.remove (listener);
		if (Processing_Listeners.isEmpty ())
			Protocol_Action (REMOVE_PROCESSING_LISTENER_ACTION, null);
		}
	}
return removed;
}


private void Processing_Event_Occurred
	(
	Message	message
	)
{
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		(">>> Remote_Theater.Processing_Event_Occurred:" + NL
		+ message);
synchronized (Processing_Listeners)
	{
	if (Processing_Listeners.isEmpty ())
		{
		if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
			System.out.println
				("    Processing_Listeners is empty" + NL
				+"<<< Remote_Theater.Processing_Event_Occurred");
		return;
		}
	}

Processing_Changes
	changes = Processing_Changes (message);
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("    Processing_Changes -" + NL
		+ changes);
if (changes == null)
	return;
Processing_Event
	event = new Processing_Event (this, changes);

synchronized (Processing_Listeners)
	{
	int
		index = -1,
		size = Processing_Listeners.size ();
	if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
		System.out.println
			("    " + size + " Processing_Listeners");
	while (++index < size)
		Processing_Listeners.get (index)
			.Processing_Event_Occurred (event);
	}
if ((DEBUG & DEBUG_PROCESSING_LISTENER) != 0)
	System.out.println
		("<<< Remote_Theater.Processing_Event_Occurred");
}

/*------------------------------------------------------------------------------
	Log Writer
*/
public Management Add_Log_Writer
	(
	Writer	writer
	)
{
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Remote_Theater.Add_Log_Writer_Route: " + writer);
boolean
	add = Log_Writer.Is_Empty ();
if (Log_Writer.Add (writer) &&
	add)
	Protocol_Action (ADD_LOG_WRITER_ACTION, null);
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("<<< Remote_Theater.Add_Log_Writer_Route");
return this;
}


public boolean Remove_Log_Writer
	(
	Writer	writer
	)
{
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Remote_Theater.Remove_Log_Writer_Route: " + writer);
boolean
	removed = Log_Writer.Remove (writer);
if (removed &&
	Log_Writer.Is_Empty ())
	Protocol_Action (REMOVE_LOG_WRITER_ACTION, null);
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("<<< Remote_Theater.Remove_Log_Writer_Route: " + removed);
return removed;
}


public Management Enable_Log_Writer
	(
	Writer	writer,
	boolean	enable
	)
{
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Remote_Theater.Enable_Log_Writer_Route: "
			+ writer + ' ' + enable);
Protocol_Action (ENABLE_LOG_WRITER_ACTION, String.valueOf (enable));
Log_Writer.Suspend (writer, enable);
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("<<< Remote_Theater.Enable_Log_Writer_Route");
return this;
}


private void Write_Log
	(
	Message	message
	)
{
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		(">>> Remote_Theater.Write_Log: " + NL
		+ message);
String
	string;
if (message == null ||
	(string = message.Get (LOG_WRITTEN_PARAMETER_NAME)) == null ||
	string.length () == 0)
	{
	if ((DEBUG & DEBUG_LOG_WRITER) != 0)
		System.out.println
			("    No " + LOG_WRITTEN_PARAMETER_NAME + " parameter value" + NL
			+"<<< Remote_Theater.Write_Log");
	return;
	}

try {Log_Writer.Write
		(new String_Buffer (string).escape_to_special ().toString (),
		Messenger_Styled_Writer.Style_Parameter
			(message.Find (LOG_STYLE_PARAMETER_GROUP)));}
catch (IOException exception)
	{
	//!!!	Remove the offending writer.
	if ((DEBUG & DEBUG_LOG_WRITER) != 0)
		System.out.println
			("!!! Log_Writer failed -" + NL
			+ exception);
	}
if ((DEBUG & DEBUG_LOG_WRITER) != 0)
	System.out.println
		("<<< Remote_Theater.Write_Log");
}


}
