///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file BoxObject.h 
 * \brief Contains the definition of the StdObjects::BoxObject class. 
 */

#ifndef __STDOBJ_BOX_OBJECT_H
#define __STDOBJ_BOX_OBJECT_H

#include <core/Core.h>
#include <core/scene/objects/geometry/SimpleGeometryObject.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/gui/properties/PropertiesEditor.h>

#include <stdobjects/StdObjects.h>

namespace StdObjects {

/**
 * \brief The procedural box geometry object.
 * 
 * \author Alexander Stukowski
 */
class STDOBJECTS_DLLEXPORT BoxObject : public SimpleGeometryObject
{
public:
	/// \brief Default constructors.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the 
	///                  data stored in a scene file after the instance has been created.
	BoxObject(bool isLoading = false);

	/// \brief Returns the controller that controls the width of the box object.
	/// \return The internal controller object.
	/// \sa setWidthController()
	FloatController* widthController() const { return _width; }
	
	/// \brief Returns the controller that controls the length of the box object.
	/// \return The internal controller object.
	/// \sa setLengthController()
	FloatController* lengthController() const { return _length; }

	/// \brief Returns the controller that controls the height of the box object.
	/// \return The internal controller object.
	/// \sa setHeightController()
	FloatController* heightController() const { return _height; }

	/// \brief Replaces the controller that controls the width of the box object with a new controller.
	/// \param ctrl The new controller for the width parameter.
	/// \undoable
	/// \sa widthController()
	void setWidthController(FloatController* ctrl) { _width = ctrl; }
	
	/// \brief Replaces the controller that controls the length of the box object with a new controller.
	/// \param ctrl The new controller for the length parameter.
	/// \undoable
	/// \sa lengthController()
	void setLengthController(FloatController* ctrl) { _length = ctrl; }
	
	/// \brief Replaces the controller that controls the height of the box object with a new controller.
	/// \param ctrl The new controller for the height parameter.
	/// \undoable
	/// \sa heightController()
	void setHeightController(FloatController* ctrl) { _height = ctrl; }
	
	// From SceneObject:

	/// \brief Asks the object for its validity interval at the given time.
	/// \param time The animation time at which the validity interval should be computed.
	/// \return The maximum time interval that contains \a time and during which the object does not change.
	virtual TimeInterval objectValidity(TimeTicks time);

protected:

	// From SimpleGeometryObject:

	/// \brief Builds the mesh representation of this geometry object.
	/// \param[in] time The animation time at which to generate the mesh.
	/// \param[out] mesh The mesh object that is be filled by the method.
	/// \param[out] meshValidity The method stores the validty interval of the generated mesh in this output variable.
	virtual void buildMesh(TimeTicks time, TriMesh& mesh, TimeInterval& meshValidity);

protected:

	/// This controller stores the width of the box.
	ReferenceField<FloatController> _width;

	/// This controller stores the length of the box.
	ReferenceField<FloatController> _length;

	/// This controller stores the height of the box.
	ReferenceField<FloatController> _height;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(BoxObject)
	DECLARE_REFERENCE_FIELD(_width)
	DECLARE_REFERENCE_FIELD(_length)
	DECLARE_REFERENCE_FIELD(_height)
};


/**
 * \brief A properties editor for the BoxObject class.
 * 
 * This editor class creates and manages the user interface through which the
 * user can alter the object's parameters.
 * 
 * \author Alexander Stukowski
 */
class BoxObjectEditor : public PropertiesEditor
{
protected:
	
	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);
	
private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(BoxObjectEditor)
};

};

#endif // __STDOBJ_BOX_OBJECT_H
