/*
 * Copyright 2015-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;

import org.junit.Test;

/**
 * Unit tests for {@link AssertionFailedError}.
 *
 * @author Marc Philipp
 * @author Sam Brannen
 * @since 1.0
 */
public class AssertionFailedErrorTests {

	private static final String EMPTY = "";
	private static final String BLANK = "   ";

	@Test
	public void defaultConstructorUsesEmptyMessage() {
		assertEquals(EMPTY, new AssertionFailedError().getMessage());
	}

	@Test
	public void nullMessageIsConvertedToEmptyMessage() {
		assertEquals(EMPTY, new AssertionFailedError(null).getMessage());
		assertEquals(EMPTY, new AssertionFailedError(null, null).getMessage());
		assertEquals(EMPTY, new AssertionFailedError(null, "foo", "bar").getMessage());
		assertEquals(EMPTY, new AssertionFailedError(null, "foo", "bar", null).getMessage());
	}

	@Test
	public void blankMessageIsConvertedToEmptyMessage() {
		assertEquals(EMPTY, new AssertionFailedError(BLANK).getMessage());
		assertEquals(EMPTY, new AssertionFailedError(BLANK, null).getMessage());
		assertEquals(EMPTY, new AssertionFailedError(BLANK, "foo", "bar").getMessage());
		assertEquals(EMPTY, new AssertionFailedError(BLANK, "foo", "bar", null).getMessage());
	}

	@Test
	public void toStringChecks() {
		String className = AssertionFailedError.class.getName();
		assertEquals(className, new AssertionFailedError().toString());
		assertEquals(className, new AssertionFailedError(null).toString());
		assertEquals(className, new AssertionFailedError(null, "foo", "bar").toString());
		assertEquals(className, new AssertionFailedError(null, new RuntimeException()).toString());
		assertEquals(className, new AssertionFailedError(null, "foo", "bar", new RuntimeException()).toString());

		assertEquals(className, new AssertionFailedError(EMPTY).toString());
		assertEquals(className, new AssertionFailedError(EMPTY, "foo", "bar").toString());
		assertEquals(className, new AssertionFailedError(EMPTY, new RuntimeException()).toString());
		assertEquals(className, new AssertionFailedError(EMPTY, "foo", "bar", new RuntimeException()).toString());

		assertEquals(className, new AssertionFailedError(BLANK).toString());
		assertEquals(className, new AssertionFailedError(BLANK, "foo", "bar").toString());
		assertEquals(className, new AssertionFailedError(BLANK, new RuntimeException()).toString());
		assertEquals(className, new AssertionFailedError(BLANK, "foo", "bar", new RuntimeException()).toString());

		assertEquals(className + ": message", new AssertionFailedError("message").toString());
		assertEquals(className + ": message", new AssertionFailedError("message", "foo", "bar").toString());
		assertEquals(className + ": message", new AssertionFailedError("message", new RuntimeException()).toString());
		assertEquals(className + ": message",
			new AssertionFailedError("message", "foo", "bar", new RuntimeException()).toString());
	}

	@Test
	public void messageAndCauseAreStored() {
		RuntimeException cause = new RuntimeException("cause");

		AssertionFailedError error = new AssertionFailedError("my message", cause);

		assertEquals("my message", error.getMessage());
		assertEquals(cause, error.getCause());
		assertFalse(error.isExpectedDefined());
		assertFalse(error.isActualDefined());
	}

	@Test
	public void expectedAndActualValuesAreStored() {
		AssertionFailedError errorWithExpectedAndActual = new AssertionFailedError(null, "foo", "bar");
		assertTrue(errorWithExpectedAndActual.isExpectedDefined());
		assertEquals("foo", errorWithExpectedAndActual.getExpected().getValue());
		assertTrue(errorWithExpectedAndActual.isActualDefined());
		assertEquals("bar", errorWithExpectedAndActual.getActual().getValue());
	}

	@Test
	public void returnsNullForExpectedAndActualWhenNotPassedToConstructor() {
		AssertionFailedError errorWithoutExpectedAndActual = new AssertionFailedError();
		assertFalse(errorWithoutExpectedAndActual.isExpectedDefined());
		assertNull(errorWithoutExpectedAndActual.getExpected());
		assertFalse(errorWithoutExpectedAndActual.isActualDefined());
		assertNull(errorWithoutExpectedAndActual.getActual());
	}

	@Test
	public void serializationWorksForAssertionFailedErrorWithMessageAndExpectedAndActualValues() throws Exception {
		AssertionFailedError error = serializeAndDeserialize(new AssertionFailedError("a message", "foo", "bar"));

		assertEquals("a message", error.getMessage());
		assertTrue(error.isExpectedDefined());
		assertEquals("foo", error.getExpected().getValue());
		assertTrue(error.isActualDefined());
		assertEquals("bar", error.getActual().getValue());
	}

	@Test
	public void serializationWorksForAssertionFailedErrorWithoutAnyValues() throws Exception {
		AssertionFailedError error = serializeAndDeserialize(new AssertionFailedError());

		assertEquals(EMPTY, error.getMessage());
		assertFalse(error.isExpectedDefined());
		assertNull(error.getExpected());
		assertFalse(error.isActualDefined());
		assertNull(error.getActual());
	}

	@Test
	public void deserializationOfAssertionFailedErrorWorksForVersion_1_0_0() throws Exception {
		// Generated using:
		// FileOutputStream outputStream = new FileOutputStream("src/test/resources/AssertionFailedError_serializedVersion_1_0_0.out");
		// serialize(new AssertionFailedError("message", "foo", "bar"), outputStream);
		// outputStream.close();

		AssertionFailedError error = (AssertionFailedError) deserializeClasspathResource(
			"/AssertionFailedError_serializedVersion_1_0_0.out");

		assertEquals("message", error.getMessage());
		assertTrue(error.isExpectedDefined());
		assertEquals("foo", error.getExpected().getValue());
		assertTrue(error.isActualDefined());
		assertEquals("bar", error.getActual().getValue());
	}

	@Test
	public void ephemeralValueIsOmittedFromSerialization() throws Exception {
		class NonSerializable {
			public final String guid = "8675309";
		}

		AssertionFailedError error = serializeAndDeserialize(
			new AssertionFailedError("a message", new NonSerializable(), new NonSerializable()));
		assertEquals("a message", error.getMessage());
		assertTrue(error.isExpectedDefined());
		assertNull(error.getExpected().getValue());
		assertNull(error.getExpected().getEphemeralValue());
		assertTrue(error.isActualDefined());
		assertNull(error.getActual().getValue());
		assertNull(error.getActual().getEphemeralValue());
	}

	private Object deserializeClasspathResource(String name) throws Exception {
		InputStream inputStream = getClass().getResourceAsStream(name);
		try {
			return deserialize(inputStream);
		}
		finally {
			inputStream.close();
		}
	}

	private AssertionFailedError serializeAndDeserialize(AssertionFailedError originalError) throws Exception {
		byte[] bytes = serialize(originalError);
		Object deserializedObject = deserialize(new ByteArrayInputStream(bytes));
		assertEquals(AssertionFailedError.class, deserializedObject.getClass());
		return (AssertionFailedError) deserializedObject;
	}

	private Object deserialize(InputStream inputStream) throws Exception {
		ObjectInputStream in = new ObjectInputStream(inputStream);
		return in.readObject();
	}

	private byte[] serialize(Object object) throws Exception {
		ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
		serialize(object, byteArrayOutputStream);
		return byteArrayOutputStream.toByteArray();
	}

	private void serialize(Object object, OutputStream outputStream) throws Exception {
		ObjectOutputStream out = new ObjectOutputStream(outputStream);
		out.writeObject(object);
		out.flush();
	}

}
