/*
 *  Copyright (c) 2012 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <cstdlib>

// GTLCore Headers
#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Type.h>
#include <GTLCore/Value.h>

// GTLFragment Headers
#include <GTLFragment/LibrariesManager.h>

// OpenShiva Headers
#include <OpenShiva/Version.h>
#include <OpenShiva/Kernel.h>
#include <OpenShiva/Source.h>

#include "CheckImage.h"

#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

void printVersion()
{
  std::cout << OpenShiva::LibraryShortName() << " - " << OpenShiva::LibraryName() << " - " << OpenShiva::LibraryVersionString() << std::endl;
  std::cout << OpenShiva::LibraryCopyright() << std::endl;
  std::cout << "Shiva Version : " << OpenShiva::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : shivacheck [option] fileName.shiva" << std::endl;
  std::cout << std::endl;
  std::cout << "This program run various test to check if a kernel is properly written. Currently check:" << std::endl;
  std::cout << " * the kernel compile" << std::endl;
  std::cout << " * the neededRect function is properly implemented" << std::endl;
  std::cout << std::endl;
  std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << std::endl;
  std::cout << "  -w --width [w]          define the width of the output" << std::endl;
  std::cout << "  -h --height [h]         define the height of the output" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}


int main(int argc, char** argv)
{
  GTLCore::String fileName = "";

  int width = 800;
  int height = 600;
  int channelsCount = 4;
  
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        GTLFragment::LibrariesManager::instance()->addDirectory(argv[ai]);
      }
    }else if(ARG_IS("-w", "--width")) {
      if( ai == argc )
      {
        std::cerr << "Expected width after -w --width." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        width = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-h", "--height")) {
      if( ai == argc )
      {
        std::cerr << "Expected height after -h --height." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        height = GTLCore::String( argv[ai] ).toInt();
      }
    } else {
      fileName = argv[ai];
    }
  }
  
  bool success = true;
  
  if( fileName == "")
  {
    std::cerr << "Invalid command line parameters." << std::endl;
    printHelp();
    return EXIT_FAILURE;
  } else {
    // Check that the kernel compile
    OpenShiva::Source s;
    s.loadFromFile(fileName);
    OpenShiva::Kernel p(channelsCount);
    p.setSource(s);
    p.setParameter(OpenShiva::Kernel::IMAGE_HEIGHT, float(width));
    p.setParameter(OpenShiva::Kernel::IMAGE_WIDTH, float(height));
    p.compile();
    if(not p.isCompiled())
    {
      std::cerr << "Error: " << std::endl << p.compilationMessages().toString() << std::endl;
      return EXIT_FAILURE;
    }
    std::cout << "Check: compilation ok." << std::endl;
    
    // Check that the neededRect function works correctly
    if(s.countInputImages() > 0)
    {
      if(not p.hasNeededFunction())
      {
        std::cerr << "Error: missing needed function." << std::endl;
        return EXIT_FAILURE;
      } else {
        GTLCore::PixelDescription pixel( GTLCore::Type::UnsignedInteger8, channelsCount);
        std::list<const GTLCore::AbstractImage*> images;
        std::list<CheckImage*> cis;
        std::list<GTLCore::RegionI> dods;
        for(int i = 0; i < s.countInputImages(); ++i)
        {
          CheckImage* ci = new CheckImage(width, height, pixel);
          cis.push_back(ci);
          images.push_back(ci);
          dods.push_back(GTLCore::RegionI(0, 0, width, height));
        }
        
        CheckImage image(width, height, pixel);
        
        for(int y = 0; y < height; ++y)
        {
          for(int x = 0; x < width; ++x)
          {
            GTLCore::RegionF pixel_region(x,y, 1, 1);
            GTLCore::RegionI pixel_region_i(x,y, 1, 1);
            
            foreach(CheckImage* ci, cis )
            {
              ci->reset();
            }
            
            p.evaluatePixels( pixel_region_i, images, &image );
            
            int idx = 0;
            foreach(CheckImage* ci, cis )
            {
              GTLCore::RegionI accessed = ci->accessed();
              GTLCore::RegionI needed = p.needed(pixel_region_i, idx, dods).toRegionI();
              if(needed.united(accessed) != needed)
              {
                std::cerr << "Invalid needed for pixel (" << x << ", " << y << ") got " << needed << " expected at least " << accessed << std::endl;
                success = false;
              }
                
              ++idx;
            }
            
          }
        }
      }
      
    }
  }
  
  if(success)
  {
    return EXIT_SUCCESS;
  } else {
    return EXIT_FAILURE;
  }
}
