/*###############################################################################
# Linux Management Providers (LMP), SSH provider package
# Copyright (C) 2010 Feng Zhanlei, REDFLAG <fengzhanlei@redflag-linux.com>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by Guillaume BOTTEX <guillaumebottex@uxsystem.net>, UXSystem, 2010
###############################################################################*/

#include "cmpiOpenDRIM_SSHSettingData.h"
#include "OpenDRIM_SSHSettingDataAccess.h"

static const CMPIBroker * _broker;

#ifdef CMPI_VER_100
#define SSH_OpenDRIM_SSHSettingDataProviderSetInstance SSH_OpenDRIM_SSHSettingDataProviderModifyInstance
#endif

int SSH_OpenDRIM_SSHSettingData_init(const CMPIBroker* broker);

int SSH_OpenDRIM_SSHSettingData_finalize();

/* ---------------------------------------------------------------------------*/
/*                      Instance Provider Interface                           */
/* ---------------------------------------------------------------------------*/

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderCleanup(CMPIInstanceMI * mi, const CMPIContext* ctx, CMPIBoolean terminate) {
	_E_;
	int errorCode = SSH_OpenDRIM_SSHSettingData_finalize();
	CMPIStatus rc;
	CMSetStatus(&rc, (CMPIrc) errorCode);
	_L_;
	return rc;
}

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderEnumInstanceNames(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* ref) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	vector<OpenDRIM_SSHSettingData> instances;
	string errorMessage;
	
	int errorCode = SSH_OpenDRIM_SSHSettingData_retrieve(_broker, ctx, instances, NULL, errorMessage, "ein");
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	for (size_t i=0; i < instances.size(); i++) {
		CMPIObjectPath* op = SSH_OpenDRIM_SSHSettingData_toCMPIObjectPath(_broker, instances[i]);
		CMReturnObjectPath(rslt, op);
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderEnumInstances(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* ref, const char** properties) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	vector<OpenDRIM_SSHSettingData> instances;
	string errorMessage;
	
	int errorCode = SSH_OpenDRIM_SSHSettingData_retrieve(_broker, ctx, instances, properties, errorMessage, "ei");
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	for (size_t i=0; i < instances.size(); i++) {
		CMPIInstance* ci = SSH_OpenDRIM_SSHSettingData_toCMPIInstance(_broker, instances[i]);
		CMReturnInstance(rslt, ci);
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderGetInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop, const char** properties) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_SSHSettingData instance;
	SSH_OpenDRIM_SSHSettingData_toCPP(_broker, cop, instance);
	string errorMessage;
	
	int errorCode = SSH_OpenDRIM_SSHSettingData_getInstance(_broker, ctx, instance, properties, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMPIInstance* ci = SSH_OpenDRIM_SSHSettingData_toCMPIInstance(_broker, instance);
	CMReturnInstance(rslt, ci);
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderSetInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop, const CMPIInstance* ci, const char** properties) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_SSHSettingData newInstance, oldInstance;
	SSH_OpenDRIM_SSHSettingData_toCPP(_broker, ci, newInstance);
	SSH_OpenDRIM_SSHSettingData_toCPP(_broker, cop, oldInstance);
	string errorMessage;
	
	int errorCode = SSH_OpenDRIM_SSHSettingData_getInstance(_broker, ctx, oldInstance, properties, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = SSH_OpenDRIM_SSHSettingData_setInstance(_broker, ctx, newInstance, oldInstance, properties, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderCreateInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop, const CMPIInstance* ci) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_SSHSettingData instance;
	SSH_OpenDRIM_SSHSettingData_toCPP(_broker, ci, instance);
	string errorMessage;
	
	int errorCode = SSH_OpenDRIM_SSHSettingData_getInstance(_broker, ctx, instance, NULL, errorMessage);
	if (errorCode != NOT_FOUND) {
		if (errorCode == OK)
			errorCode = ALREADY_EXISTS;
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = SSH_OpenDRIM_SSHSettingData_createInstance(_broker, ctx, instance, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = SSH_OpenDRIM_SSHSettingData_getInstance(_broker, ctx, instance, NULL, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMPIObjectPath* _cop = SSH_OpenDRIM_SSHSettingData_toCMPIObjectPath(_broker, instance);
	CMReturnObjectPath(rslt, _cop);
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderDeleteInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_SSHSettingData instance;
	SSH_OpenDRIM_SSHSettingData_toCPP(_broker, cop, instance);
	string errorMessage;
	
	int errorCode = SSH_OpenDRIM_SSHSettingData_getInstance(_broker, ctx, instance, NULL, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = SSH_OpenDRIM_SSHSettingData_deleteInstance(_broker, ctx, instance, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus SSH_OpenDRIM_SSHSettingDataProviderExecQuery(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* ref, const char* lang, const char* query) {
	_E_;
	CMReturn(CMPI_RC_ERR_NOT_SUPPORTED);
}

/* ---------------------------------------------------------------------------*/
/*                   End of Instance Provider Interface                       */
/* ---------------------------------------------------------------------------*/

/* ---------------------------------------------------------------------------*/
/*                              Provider Factory                              */
/* ---------------------------------------------------------------------------*/

int SSH_OpenDRIM_SSHSettingData_init(const CMPIBroker* broker) {
	_E_;
	_broker = broker;
	static bool initialized = false;
	if (!initialized) {
		string errorMessage;
		int errorCode = SSH_OpenDRIM_SSHSettingData_load(broker, errorMessage);
		if (errorCode != OK) {
			errorMessage += "SSH_OpenDRIM_SSHSettingData_init FAILED: " + (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
			string cmd = "/bin/echo \"" + errorMessage + "\" >> cmpi_prov_debug.txt";
			system(cmd.c_str());
			return -1;
		}
		initialized = true;
	}
	_L_;
	return OK;
}

int SSH_OpenDRIM_SSHSettingData_finalize() {
	_E_;
	static bool finalized = false;
	if (!finalized) {
		string errorMessage;
		int errorCode = SSH_OpenDRIM_SSHSettingData_unload(errorMessage);
		if (errorCode != OK) {
			errorMessage += "SSH_OpenDRIM_SSHSettingData_unload FAILED: " + (string) OpenDRIM_SSHSettingData_classnames[0] + ": " + errorMessage;
			string cmd = "/bin/echo \"" + errorMessage + "\" >> cmpi_prov_debug.txt";
			system(cmd.c_str());
			return errorCode;
		}
		finalized = true;
	}
	_L_;
	return OK;
}

#define SSH_OpenDRIM_SSHSettingData_INIT if (SSH_OpenDRIM_SSHSettingData_init(_broker) < 0) return NULL

CMInstanceMIStub(SSH_OpenDRIM_SSHSettingDataProvider, SSH_OpenDRIM_SSHSettingDataProvider, _broker, SSH_OpenDRIM_SSHSettingData_INIT);

/* ---------------------------------------------------------------------------*/
/*                          End of Provider Factory                           */
/* ---------------------------------------------------------------------------*/
