"""Rescheduling of failed cleanup tasks."""

import logging
import random
import sqlite3
import time
from arcnagios.utils import nth

busy_timeout = 10

_default_log = logging.getLogger(__name__)

_create_sql = """\
CREATE TABLE %s (
    n_attempts integer NOT NULL,
    t_sched integer NOT NULL,
    task_type varchar(16) NOT NULL,
    arg text NOT NULL
)"""

def format_time(t):
    return time.strftime('%Y-%m-%d %H:%M', time.localtime(t))

class TaskType(object):
    def __init__(self, h, min_delay = 3600, max_attempts = 12, delay_dev = 0.1):
        self.handler = h
        self.min_delay = min_delay
        self.max_attempts = max_attempts
        self.delay_dev = delay_dev

    def next_delay(self, n_attempts):
        return (self.min_delay << n_attempts) * random.gauss(1.0, self.delay_dev)

class Rescheduler(object):
    def __init__(self, db_path, table_name, log = _default_log):
        self._db = sqlite3.connect(db_path, busy_timeout)
        self._table = table_name
        self._task_types = {}
        self._log = log
        try:
            self._db.execute(_create_sql % self._table)
        except sqlite3.OperationalError:
            pass

    def close(self):
        self._db.close()

    def _update(self, stmt, *args):
        r = self._db.execute(stmt, args)
        self._db.commit()
        return r

    def _query(self, stmt, *args):
        return self._db.execute(stmt, args)

    def register(self, task_type_name, h, min_delay = 3600, max_attempts = 12,
                 delay_dev = 0.1):
        self._task_types[task_type_name] \
            = TaskType(h, min_delay, max_attempts, delay_dev)

    def schedule(self, task_type_name, arg, n_attempts = 0):
        handler = self._task_types[task_type_name]
        t_sched = time.time() + handler.next_delay(n_attempts)
        self._update('INSERT INTO %s (n_attempts, t_sched, task_type, arg) '
                     'VALUES (?, ?, ?, ?)' % self._table,
                     n_attempts, t_sched, task_type_name, arg)

    def _unschedule_rowid(self, rowid):
        self._update('DELETE FROM %s WHERE ROWID = ?' % self._table, rowid)

    def _reschedule_rowid(self, rowid, n_attempts, t_sched):
        self._update('UPDATE %s SET n_attempts = ?, t_sched = ? '
                     'WHERE ROWID = ?' % self._table,
                     n_attempts, t_sched, rowid)

    def call(self, task_type_name, arg):
        if self._task_types[task_type_name].handler(arg, 0):
            return True
        else:
            self.schedule(task_type_name, arg, n_attempts = 1)
            return False

    def run(self, timeout = 100):
        """Run pending jobs. Currently timeout is the deadline for starting
           jobs, so the maximum full running time will be timeout plus the
           maximum time of an individual job."""
        t_now = time.time()
        t_deadline = t_now + timeout
        r = self._query('SELECT ROWID, n_attempts, t_sched, task_type, arg '
                        'FROM %s WHERE t_sched <= ?' % self._table,
                        t_now)
        success_count = 0
        failed_count = 0
        resched_count = 0
        postponed_count = 0
        for rowid, n_attempts, t_sched, task_type_name, arg in r:
            if not task_type_name in self._task_types:
                self._log.warning('No task type %s.', task_type_name)
                continue

            if time.time() >= t_deadline:
                postponed_count += 1
                continue

            task_type = self._task_types[task_type_name]
            try:
                ok = task_type.handler(arg, n_attempts)
            except Exception as exn: # pylint: disable=broad-except
                self._log.error('Task %s(%r) raised exception: %s',
                                task_type_name, arg, exn)
                ok = False

            if ok:
                self._log.info('Finished %s(%r)', task_type_name, arg)
                self._unschedule_rowid(rowid)
                success_count += 1
            elif n_attempts >= task_type.max_attempts:
                self._log.info('Giving up on %s(%r)', task_type_name, arg)
                self._unschedule_rowid(rowid)
                failed_count += 1
            else:
                t_sched = t_now + task_type.next_delay(n_attempts)
                n_attempts += 1
                self._log.info('Scheduling %s attempt at %s to %s(%r)',
                               nth(n_attempts), format_time(t_sched),
                               task_type_name, arg)
                self._reschedule_rowid(rowid, n_attempts, t_sched)
                resched_count += 1
        return (success_count, resched_count, failed_count, postponed_count)
