/* xmalloc.c -- memory management with out of memory reporting
   Copyright (C) 2000 Gary V. Vaughan
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <string.h>

#include "util.h"

/*
 * A wrapper for malloc() that aborts program execution (via a call to our
 * own error() function) if malloc() cannot allocate the requested memory.
 */
void *
xmalloc(size_t num)
{
    void *new;
    
    new = malloc(num);
    if (!new)
	error("Memory exhausted");

    return new;
}

/*
 * A wrapper for realloc() that aborts program execution (via a call to our
 * own error() function) if realloc() cannot allocate the requested memory.
 */
void *
xrealloc(void *p, size_t num)
{
    void *new;

    /*
     * realloc(3) on Linux says "If ptr is NULL, then the call is equivalent to
     * malloc(size), for all values of size" so there really is no need to
     * check for 'p' being NULL and call malloc() if it is. However, just to be
     * on the safe side we perform the check (perhaps on other platforms a
     * check is not made).
     */
    if (p == NULL)
	return xmalloc(num);

    /*
     * realloc(3) says "if size is equal to zero, and ptr is not NULL, then the
     * call is equivalent to free(ptr)". We should not consider a failure a
     * call to realloc() when "num" is zero, which is why we only call error()
     * when realloc() returns NULL and "num" is not zero.
     */
    new = realloc(p, num);
    if (!new && num != 0)
	error("Memory exhausted");

    return new;
}

/*
 * A wrapper for calloc() that uses our xmalloc() wrapper for malloc().
 */
void *
xcalloc(size_t num, size_t size)
{
    void *new;
    
    new = xmalloc(num * size);
    memset(new, 0, num * size);

    return new;
}
