/* pdu-802dot1q.c
  
   PDU builder for 802.1q tags

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-802dot1q.h"

static void
builder(const GNode *pdu, void *dest)
{
    uint16_t tci; /* Tag Control Information (TCI)*/
    uint16_t eth_type, *proto;
    const GNode *next_pdu;
    void *field;

    tci  = (num_next(_pb_pdata(pdu, "prio") ) << PRIO_SHIFT) & T_PRIO;

    if (_pb_pdata(pdu, "cfi") )
	tci |= T_CFI;

    /* Default to VLAN 1 */
    tci |= ( (field = _pb_pdata(pdu, "vlan") ) ? num_next(field) : 1) & T_VID;

    tci = htons(tci);

    SSVAL(dest, 0, tci);

    /* Put in place Ethernet type */
    if (  !(field = _pb_pdata(pdu, "proto") )
	&& (next_pdu = pb_nextpdu(pdu) ) ) {
	/*
	 * User didn't specify the ethertype so we try to guess the right
	 * value based on the next PDU.
	 */

	proto = g_hash_table_lookup(ethertypes, pb_getname(next_pdu) );
	eth_type = proto ? *proto : 0;
    } else
	/*
	 * Use for IP protocol whatever the user specified. Note that if
	 * there is no user-specified IP protocol *and* there is no
	 * next PDU then we end up here, but that's okay because
	 * num_next(NULL) is 0.
	 */
	eth_type = num_next(field);

    eth_type = htons(eth_type);
    SSVAL(dest, sizeof(tci), eth_type);
}

#if 0
static void
pdu_hdr_dumper(pdu_t *p, const char *prefix)
{
    struct vlanhdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    User priority: %s\n", prefix,
	   num_info(hdr_data->user_priority) );
    printf("%s    CFI: %d\n", prefix, hdr_data->cfi);
    printf("%s    VLAN ID: %s\n", prefix, num_info(hdr_data->vid) );
    printf("%s    Ethernet protocol: %s\n", prefix,
	   num_info(hdr_data->ether_proto) );
}
#endif

static pdu_t pdu_8021q = {
    .name = "dot1q",
    .description = "802.1q tag",
    .documented_in = "IEEE Std 802.1Q",
    .len = sizeof(uint32_t),
    .fields = (field_t []) {
	{.name = "vlan", .type = PDU_FTYPE_UINT16},
	{.name = "cfi", .type = PDU_FTYPE_BIT},
	{.name = "prio", .type = PDU_FTYPE_UINT16},
	{.name = "proto", .type = PDU_FTYPE_UINT16},
	{.name = NULL}
    },
    .build = &builder
};

void
_pb_register_dot1q(void)
{
    _pb_register_protocol(&pdu_8021q);
}
