/* isanswer.c

   Matching of received responses to injected stimuli

   The base algorithm of the "send_expect" Network Expect command is
   based on the algorithm of the "sr" command of Scapy, the packet
   manipulation program written in Python by Philippe Biondi. There are
   some differences, specially that Network Expect's implementation
   stores sent and received packets in raw (binary) form, but the basics
   are the same.

   Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "includes.h"
#include <string.h>

#include "tcl_ws-ftypes.h"
#include "packets-priv.h"

/*
 * "isanswer" function.
 *
 * Input parameters:
 *
 * struct dissection_results *rcvd: pointer to structure that describes
 *   dissection results for the received packet.
 *
 * guint rcvd_curr_layer: layer of the received packet we're currently
 *   looking at.
 *
 * struct dissection_results *sent: pointer to structure that describes
 *   dissection results for the sent packet.
 *
 * guint sent_curr_layer: layer of the sent packet we're currently
 *   looking at. It's a pointer because the "isanswer" function may choose
 *   to modify it.
 *
 * Return:
 *
 * 1 if received packet is an answer to sent packet. 0 otherwise.
 */
typedef int (*isanswer_func_t)(Tcl_Interp *, int, int *);

static int isanswer_eth(Tcl_Interp *, int, int *);
static int isanswer_arp(Tcl_Interp *, int, int *);
static int isanswer_ip(Tcl_Interp *, int, int *);
static int isanswer_icmp(Tcl_Interp *, int, int *);
static int isanswer_tcp(Tcl_Interp *, int, int *);
static int isanswer_udp(Tcl_Interp *, int, int *);
static int isanswer_dns(Tcl_Interp *, int, int *);

struct {
    char *proto_name;
    isanswer_func_t isanswer_func;
} isanswer_funcs[] = {
    {"eth", &isanswer_eth},
    {"arp", &isanswer_arp},
    {"ip", &isanswer_ip},
    {"icmp", &isanswer_icmp},
    {"tcp", &isanswer_tcp},
    {"udp", &isanswer_udp},
    {"dns", &isanswer_dns},
    {NULL, NULL}
};

/*
 * Determines whether received packet "rcvd" is an answer to sent packet
 * "sent". "rcvd" has been received on the wire and "sent" has been built
 * by us.
 */
int
pkt_isanswer(Tcl_Interp *interp,
	     const u_char *rcvd, struct wtap_pkthdr *rcvd_wphdr, frame_data *rcvd_fdata,
	     const u_char *sent, struct wtap_pkthdr *sent_wphdr, frame_data *sent_fdata)
{
    int layer_r, layer_s, i;
    int isanswer;
    char *proto_name_r, *proto_name_s;
    int llength_r, llength_s;
    Tcl_Obj *listObj;
    Tcl_Namespace *nsPtr;

    if ( (nsPtr = Tcl_FindNamespace(interp, "r", NULL, TCL_GLOBAL_ONLY) ) )
	Tcl_DeleteNamespace(nsPtr);

    if ( (nsPtr = Tcl_FindNamespace(interp, "s", NULL, TCL_GLOBAL_ONLY) ) )
	Tcl_DeleteNamespace(nsPtr);

    /*
     * Dissect received packet.
     */
#ifdef DEBUG
    printf("===== Begin dissection received packet =====\n");
#endif
    pkt_dissect_tcl(interp, rcvd, rcvd_wphdr, rcvd_fdata, NULL /* no display filter */,
		    "r", 1 /* create non-ws vars */);
#ifdef DEBUG
    printf("===== End dissection received packet =====\n");
#endif

    /*
     * Dissect sent packet.
     */
#ifdef DEBUG
    printf("===== Begin dissection sent packet =====\n");
#endif
    pkt_dissect_tcl(interp, sent, sent_wphdr, sent_fdata, NULL /* no display filter */,
		    "s", 1 /* create non-ws vars */);
#ifdef DEBUG
    printf("===== End dissection sent packet =====\n");
#endif

    listObj = Tcl_GetVar2Ex(interp, "s::pdu(list)", NULL, 0);
    Tcl_ListObjLength(interp, listObj, &llength_s);

    listObj = Tcl_GetVar2Ex(interp, "r::pdu(list)", NULL, 0);
    Tcl_ListObjLength(interp, listObj, &llength_r);

#ifdef DEBUG
    if (llength_s > llength_r)
	printf("layers sent packet (%d) > layers rcvd packet (%d)\n",
	       llength_s, llength_r);
#endif

    /*
     * Figure out at which layers in the received and sent packet we start to
     * compare.  We do this because a sent packet can be injected at OSI layer
     * 2 or at layer 3, and a received packet always has a OSI layer 2
     * protocol. So, the idea with all this is to skip the OSI layer 2 protocol
     * of the received packet in the comparison if the sent packet does not
     * have one.
     */
    proto_name_s = _pkt_layername(interp, "s", 1);
    if (!strcmp("raw", proto_name_s) ) {
	layer_r = layer_s = 2; /* Past the "raw" layer */
#ifdef DEBUG
	printf("Stimulus injected at layer 3; skipping layer 2 of received "
	       "packet.\n");
#endif
    } else
	layer_r = layer_s = 1;

    /* Go over each layer of the received packet */
    for (isanswer = 1 /* layer_r and layer_s initialized above */;
	 isanswer && layer_r < llength_r;
	 layer_r++, layer_s++) {
	proto_name_r = _pkt_layername(interp, "r", layer_r);

	/*
	 * Lookup pointer to insanswer function and call hash function if
	 * found
	 */
	for (i = 0; isanswer_funcs[i].proto_name; i++) {
	    if (!strcmp(isanswer_funcs[i].proto_name, proto_name_r) ) {
		isanswer = isanswer_funcs[i].isanswer_func(interp, layer_r, &layer_s);
#ifdef DEBUG
		printf("isanswer function for proto \"%s\" returned %d.\n",
		       proto_name_r, isanswer);
#endif
		if (!isanswer)
		    /* Not an answer; bail out. */
		    break;
	    }
	}
    }

#ifdef DEBUG
    if (isanswer)
	fprintf(stderr, "%s is answer for %s\n",
		Tcl_GetVar(interp, "r::frame.protocols", 0),
		Tcl_GetVar(interp, "s::frame.protocols", 0) );
#endif

    return isanswer;
}

static int
isanswer_eth(Tcl_Interp *interp, int curr_layer_r _U_, int *curr_layer_s)
{
    uint16_t ether_type_s, ether_type_r;

    if (!strcmp("eth", _pkt_layername(interp, "s", *curr_layer_s) ) ) {
	_pkt_get_uint16(interp, "r::eth.type", &ether_type_r);
	_pkt_get_uint16(interp, "s::eth.type", &ether_type_s);

	if (ether_type_s == ether_type_r)
	    return 1;
    }

    return 0;
}

static int
isanswer_arp(Tcl_Interp *interp, int curr_layer_r _U_, int *curr_layer_s)
{
    uint16_t arp_op_r, arp_op_s;
    ip_addr_t arp_spa, arp_tpa;

    if (!strcmp("arp", _pkt_layername(interp, "s", *curr_layer_s) ) ) {
	_pkt_get_uint16(interp, "r::arp.opcode", &arp_op_r);
	_pkt_get_uint16(interp, "s::arp.opcode", &arp_op_s);
	_pkt_get_ipaddr(interp, "r::arp.src.proto_ipv4", &arp_spa);
	_pkt_get_ipaddr(interp, "s::arp.dst.proto_ipv4", &arp_tpa);

	if (   ntohs(arp_op_r) == ARP_OP_REPLY
	    && ntohs(arp_op_s) == ARP_OP_REQUEST
	    && !memcmp(&arp_spa, &arp_tpa, IP_ADDR_LEN) )
	    return 1;
    }

    return 0;
}

static int
isanswer_ip(Tcl_Interp *interp, int curr_layer_r, int *curr_layer_s)
{
    uint8_t ip_proto_r, ip_proto_s, icmp_type_r;
    uint16_t ip_id_r, ip_id_s;
    ip_addr_t ip_src_r, ip_src_s, ip_dst_r, ip_dst_s;
    int ip_in_icmp;

    if (strcmp("ip", _pkt_layername(interp, "s", *curr_layer_s) ) )
	return 0;

    _pkt_get_uint8(interp, "s::ip.proto", &ip_proto_s);
    _pkt_get_uint16(interp, "s::ip.id", &ip_id_s);
    _pkt_get_ipaddr(interp, "s::ip.src", &ip_src_s);
    _pkt_get_ipaddr(interp, "s::ip.dst", &ip_dst_s);

    /* XXX - add check for IPinIP inside an ICMP error */
    ip_in_icmp = curr_layer_r - 1 >= 0
		 && !strcmp("icmp",
			    _pkt_layername(interp, "r", curr_layer_r - 1) );

    if (ip_in_icmp) {
	/*
	 * The received packet is an ICMP error and we are currently looking
	 * at the IP header of the *original* packet that caused the error,
	 * i.e. the IP header embedded in the ICMP error message.
	 */

	_pkt_get_uint8(interp, "r::icmp.ip.proto", &ip_proto_r);
	_pkt_get_uint16(interp, "r::icmp.ip.id", &ip_id_r);
	_pkt_get_ipaddr(interp, "r::icmp.ip.src", &ip_src_r);
	_pkt_get_ipaddr(interp, "r::icmp.ip.dst", &ip_dst_r);

	/*
	 * For the received packet to be an answer basically the IP protocol,
	 * IP source and destination addresses, and IP ID of the IP header
	 * embedded in the ICMP error message must be the same as in the
	 * IP packet that we sent.
	 */
	return     ip_proto_r == ip_proto_s
		&& ip_src_r == ip_src_s
		&& ip_dst_r == ip_dst_s
		&& ip_id_r == ip_id_s;
    }

    _pkt_get_uint8(interp, "r::ip.proto", &ip_proto_r);

    if (ip_proto_r == IP_PROTO_ICMP
	&& !strcmp("icmp", _pkt_layername(interp, "r", curr_layer_r + 1) ) ) {
	/*
	 * The IP payload is an ICMP message.
	 */

	_pkt_get_uint8(interp, "r::icmp.type", &icmp_type_r);
	if (   icmp_type_r == ICMP_UNREACH
	    || icmp_type_r == ICMP_SRCQUENCH
	    || icmp_type_r == ICMP_REDIRECT
	    || icmp_type_r == ICMP_TIMEXCEED
	    || icmp_type_r == ICMP_PARAMPROB) {
	    /*
	     * The received packet contains an ICMP error. We are interested
	     * in the IP header embedded in the ICMP message, so there's
	     * nothing to do here. We return "true" so upper layers decide
	     * whether this packet is an answer.
	     */
	    (*curr_layer_s)--; /* Don't advance location inside the sent
				  packet that we're currently looking at. */
	    return 1;
	}
    }

    _pkt_get_ipaddr(interp, "r::ip.src", &ip_src_r);

    return ip_proto_r == ip_proto_s && ip_src_r == ip_dst_s;
}

static int
isanswer_icmp(Tcl_Interp *interp, int curr_layer_r, int *curr_layer_s)
{
    uint8_t icmp_type_r, icmp_type_s;
    uint8_t icmp_code_r, icmp_code_s;
    uint16_t icmp_id_r, icmp_id_s;
    uint16_t icmp_seq_r, icmp_seq_s;
    int icmp_in_icmp;

    _pkt_get_uint8(interp, "s::icmp.type", &icmp_type_s);
    _pkt_get_uint8(interp, "s::icmp.code", &icmp_code_s);
    _pkt_get_uint16(interp, "s::icmp.ident", &icmp_id_s);
    _pkt_get_uint16(interp, "s::icmp.seq", &icmp_seq_s);

    icmp_in_icmp = curr_layer_r - 2 >= 0
		   && !strcmp("icmp",
			      _pkt_layername(interp, "r", curr_layer_r - 2) );

    if (icmp_in_icmp) {
	/*
	 * The received packet is an ICMP error and we are currently looking
	 * at the ICMP header of the *original* packet that caused the error,
	 * i.e. the ICMP header embedded in the ICMP error message.
	 */

	if (strcmp("icmp", _pkt_layername(interp, "s", *curr_layer_s) ) )
	    return 0;

	_pkt_get_uint8(interp, "r::icmp.icmp.type", &icmp_type_r);
	_pkt_get_uint8(interp, "r::icmp.icmp.code", &icmp_code_r);

	if (icmp_type_r != icmp_type_s || icmp_code_r != icmp_code_s)
	    return 0;

	/*
	 * Scapy checks for ICMP types (actually codes - need to report that
	 * bug) 0 (echo reply), 8 (echo request), 13 (timestamp request), 14
	 * (timestamp reply), 17 (address mask request), and 18 (address mask
	 * reply). We actually check for the requests only since I don't think
	 * we should respond positively when a response we get is for example
	 * an echo reply and the injected stimulus was an echo request. We also
	 * check for information reply.
	 */
	if (   icmp_type_r == ICMP_ECHO
	    || icmp_type_r == ICMP_TSTAMP
	    || icmp_type_r == ICMP_INFO
	    || icmp_type_r == ICMP_MASK) {
	    _pkt_get_uint16(interp, "r::icmp.icmp.ident", &icmp_id_r);
	    _pkt_get_uint16(interp, "r::icmp.icmp.seq", &icmp_seq_r);

	    return icmp_id_r == icmp_id_r && icmp_seq_r == icmp_seq_r;
	} else
	    return 0;
    }

    /*
     * The ICMP message we are currently looking at is *not* the payload of an
     * IP packet embedded in an ICMP error message. This means we are looking
     * at the ICMP message that is the payload of the outer IP packet. We now
     * check to see if this ICMP message is an error message (in which case we
     * skip) and determine whether it is an answer.
     */

    _pkt_get_uint8(interp, "r::icmp.type", &icmp_type_r);
    if (   icmp_type_r == ICMP_UNREACH
	|| icmp_type_r == ICMP_SRCQUENCH
	|| icmp_type_r == ICMP_REDIRECT
	|| icmp_type_r == ICMP_TIMEXCEED
	|| icmp_type_r == ICMP_PARAMPROB) {
	/*
	 * The received packet contains an ICMP error. We are interested
	 * in the ICMP header embedded in the ICMP message, so there's
	 * nothing to do here. We return "true" so upper layers decide
	 * whether this packet is an answer.
	 */
	(*curr_layer_s)--; /* Don't advance location inside the sent
			      packet that we're currently looking at. */
	return 1;
    }

    if (strcmp("icmp", _pkt_layername(interp, "s", *curr_layer_s) ) )
	return 0;

    _pkt_get_uint16(interp, "r::icmp.ident", &icmp_id_r);
    _pkt_get_uint16(interp, "r::icmp.seq", &icmp_seq_r);

    return ( (icmp_type_r == ICMP_ECHOREPLY && icmp_type_s == ICMP_ECHO)
	|| (icmp_type_r == ICMP_TSTAMPREPLY && icmp_type_s == ICMP_TSTAMP)
	|| (icmp_type_r == ICMP_INFOREPLY && icmp_type_s == ICMP_INFO)
	|| (icmp_type_r == ICMP_MASKREPLY && icmp_type_s == ICMP_MASKREPLY) )
        && (icmp_id_r == icmp_id_s && icmp_seq_r == icmp_seq_s );
}

static int
isanswer_tcp(Tcl_Interp *interp, int curr_layer_r, int *curr_layer_s)
{
    int tcp_in_icmp, retval;
    uint16_t tcp_sport_r, tcp_sport_s, tcp_dport_r, tcp_dport_s;

    if (strcmp("tcp", _pkt_layername(interp, "s", *curr_layer_s) ) )
	return 0;

    tcp_in_icmp = curr_layer_r - 2 >= 0
		  && !strcmp("icmp",
			     _pkt_layername(interp, "r", curr_layer_r - 2) );

    _pkt_get_uint16(interp, "s::tcp.srcport", &tcp_sport_s);
    _pkt_get_uint16(interp, "s::tcp.dstport", &tcp_dport_s);

    if (tcp_in_icmp) {
	_pkt_get_uint16(interp, "r::icmp.tcp.srcport", &tcp_sport_r);
	_pkt_get_uint16(interp, "r::icmp.tcp.dstport", &tcp_dport_r);

	retval = tcp_sport_r == tcp_sport_s && tcp_dport_r == tcp_dport_s;
    } else {
	_pkt_get_uint16(interp, "r::tcp.srcport", &tcp_sport_r);
	_pkt_get_uint16(interp, "r::tcp.dstport", &tcp_dport_r);

	retval = tcp_sport_r == tcp_dport_s && tcp_dport_r == tcp_sport_s;
    }

    return retval;
}

static int
isanswer_udp(Tcl_Interp *interp, int curr_layer_r, int *curr_layer_s)
{
    int udp_in_icmp, retval;
    uint16_t udp_sport_r, udp_sport_s, udp_dport_r, udp_dport_s;

    if (strcmp("udp", _pkt_layername(interp, "s", *curr_layer_s) ) )
	return 0;

    udp_in_icmp = curr_layer_r - 2 >= 0
		  && !strcmp("icmp",
			     _pkt_layername(interp, "r", curr_layer_r - 2) );

    _pkt_get_uint16(interp, "s::udp.srcport", &udp_sport_s);
    _pkt_get_uint16(interp, "s::udp.dstport", &udp_dport_s);

    if (udp_in_icmp) {
	_pkt_get_uint16(interp, "r::icmp.udp.srcport", &udp_sport_r);
	_pkt_get_uint16(interp, "r::icmp.udp.dstport", &udp_dport_r);

	retval = udp_sport_r == udp_sport_s && udp_dport_r == udp_dport_s;
    } else {
	_pkt_get_uint16(interp, "r::udp.srcport", &udp_sport_r);
	_pkt_get_uint16(interp, "r::udp.dstport", &udp_dport_r);

	retval = udp_sport_r == udp_dport_s && udp_dport_r == udp_sport_s;
    }

    return retval;
}

static int
isanswer_dns(Tcl_Interp *interp, int curr_layer_r _U_, int *curr_layer_s)
{
    uint16_t id_r, id_s;
    int response_r, response_s;

    if (strcmp("dns", _pkt_layername(interp, "s", *curr_layer_s) ) )
	return 0;

    _pkt_get_uint16(interp, "r::dns.id", &id_r);
    _pkt_get_boolean(interp, "r::dns.flags.response", &response_r);
    _pkt_get_uint16(interp, "s::dns.id", &id_s);
    _pkt_get_boolean(interp, "s::dns.flags.response", &response_s);

    return id_r == id_s && response_r == 1 && response_s == 0;
}
