/*
   nexp_pbuild.c
  
   Network Expect interface to libpbuild.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#include "nexp_speakers.h"

/*
 * Dumps a PDU Control Block (PCB).
 */
void
pcb_dump(pcb_t *pcb, int verbose _U_)
{
    printf("PDU Control Block (PCB) at %p:\n", pcb);
    printf("  Size: %zu\n", pb_len(pcb->pdu) );
    printf("  Sending delay: %lu seconds %lu microseconds\n",
	   pcb->delay.tv_sec, pcb->delay.tv_usec);
}

/*
 * Releases all the structures allocated by a PDU Control Block.
 */
void
pcb_destroy(pcb_t *pcb)
{
    if (pcb->pdu)
	/* The PCB has a linked PDU; it needs to be freed. */
	pb_destroy( (GNode *) pcb->pdu);

    free( (void *) pcb->def);
}

/*
 * Outputs a PDU based on the information in the PDU Control Block (PCB)
 * passed as a parameter. Different output methods are available, depending
 * on the speaker type associated with the PCB.
 */
void
nexp_pdu_output(pcb_t *pcb, uint8_t *pdu, size_t pdu_size)
{
    size_t bytes_written;

    switch (pcb->speaker->type) {
    case SPEAKER_PCAP:
	dump_pcap(pcb, pdu, pdu_size);
	break;
    case SPEAKER_HEX:
	printf("Packet size: %zu bytes\n\n", pdu_size);
	dump(pdu, pdu_size);
	break;
    case SPEAKER_STDOUT:
	fwrite(pdu, pdu_size, 1, stdout);
	break;
    case SPEAKER_IPV4USR:
    case SPEAKER_IPV4:
#ifdef __linux__
    case SPEAKER_IPV6:
#endif
    case SPEAKER_ETHER:
	bytes_written = wire_dump(pcb, pdu, pdu_size);
	break;
    case SPEAKER_NULL:
	/* Do nothing; we're sending packets to a black hole */
	break;
    case SPEAKER_STREAM:
    case SPEAKER_DGRAM:
	bytes_written = write(pcb->speaker->_socket.fd, pdu, pdu_size);
	break;
    }

    /*
     * Update the time the PDU was written to its associated speaker.
     */
    gettimeofday(&pcb->speaker->ts, NULL);
}
