/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "transport.h"
#include "libsyncml/sml_error_internals.h"
#include "libsyncml/sml_support.h"

SmlBool smlDataSyncTransportObexClientInit(
			SmlDataSyncObject *dsObject,
			SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, dsObject, error);
	CHECK_ERROR_REF

	if (dsObject->url &&
	    !smlTransportSetConfigOption(
			dsObject->tsp,
			SML_TRANSPORT_CONFIG_URL, dsObject->url,
			error))
		goto error;

	smlTrace(TRACE_EXIT, "%s - TRUE", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlDataSyncTransportObexClientConnect(SmlDataSyncObject *dsObject, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, dsObject, error);
	CHECK_ERROR_REF

	/* Connect transport layer */

	if (!smlTransportConnect(dsObject->tsp, error))
		goto error;

	/* Create a SAN */

	SmlNotificationVersion sanVersion = SML_SAN_VERSION_UNKNOWN;
	switch(dsObject->version)
	{
		case SML_VERSION_10:
			sanVersion = SML_SAN_VERSION_10;
			break;
		case SML_VERSION_11:
			sanVersion = SML_SAN_VERSION_11;
			break;
		case SML_VERSION_12:
			sanVersion = SML_SAN_VERSION_12;
			break;
		case SML_VERSION_UNKNOWN:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown SyncML SAN Version.");
			goto error;
			break;
	}

	SmlNotification *san = smlNotificationNew(
				sanVersion,
				SML_SAN_UIMODE_UNSPECIFIED,
				SML_SAN_INITIATOR_USER, 1,
				dsObject->identifier, "/",
				dsObject->useWbxml ? SML_MIMETYPE_WBXML : SML_MIMETYPE_XML,
				error);
	if (!san)
		goto error;

	smlNotificationSetManager(san, dsObject->manager);

	/* init digest - SAN uses always MD5 */
	if (dsObject->username != NULL && strlen(dsObject->username))
	{
		SmlCred *cred = smlCredNewAuth(
					SML_AUTH_TYPE_MD5,
					dsObject->username,
					dsObject->password,
					error);
		if (!cred)
			goto error;
		smlNotificationSetCred(san, cred);
		smlCredUnref(cred);
		cred = NULL;
	}

	GList *o = dsObject->datastores;
	for (; o; o = o->next) { 
		SmlDataSyncDatastore *datastore = o->data;
		if (!smlDsServerAddSan(datastore->server, san, error))
			goto error;
	}

	if (!smlNotificationSend(san, dsObject->tsp, error))
		goto error;

	smlNotificationFree(san);
	san = NULL;

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, smlErrorPrint(error));
	return FALSE;
}

