/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     trav_xy.c                                                      */
/*                                                                          */
/*                                                                          */
/* description: Find element at position (xy) and local coordinates         */
/*              return 0 if xy \not\in domain.                              */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Univesitaet Bremen                                           */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Oliver Kriessl                                               */
/*             Institut fuer Mathematik                                     */
/*             Albert-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg                                             */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

static const REAL *g_xy0, *g_xy;
static REAL       *g_sp;
static EL_INFO final_el_info[1];
static REAL   final_lambda[N_LAMBDA] = {0};

static int find_el_at_pt_recursive(EL_INFO *el_info, REAL lambda[N_LAMBDA],
				   int outside)
{
  FUNCNAME("find_el_at_pt_recursive");
  EL      *el = el_info->el;
  EL_INFO c_el_info[1];
  REAL    c_lambda[N_LAMBDA];
  int     i, inside, dim = el_info->mesh->dim;

  if (IS_LEAF_EL(el)) {
    *final_el_info = *el_info;
    if (outside < 0) {
      for (i=0; i<=dim; i++)  final_lambda[i] = lambda[i];
      return(true);
    }
    else 
    {  /* outside */
      if (g_xy0) 
      { /* find boundary point of [xy0, xy] */
	REAL s;
	world_to_coord(el_info, g_xy0, c_lambda);

	s = lambda[outside] / (lambda[outside] - c_lambda[outside]);
	for (i=0; i<=dim; i++) 
	{
	  final_lambda[i] = s * c_lambda[i] + (1.0-s) * lambda[i];
	}
	if (g_sp) *g_sp = s;

	if(dim == 3)
	  MSG("outside finest level on el %d: s=%.3le\n", INDEX(el), s);

	return(false);  /* ??? */
      }
      else return(false);
    }
  }

  switch(dim) {
  case 1:
    if (lambda[0] >= lambda[1]) {
      fill_elinfo(0, el_info, c_el_info);
      if (outside >= 0) 
	{
	  outside = world_to_coord_1d(c_el_info, g_xy, c_lambda);
	  if (outside >= 0) ERROR("point outside domain\n");
	}
      else {
	c_lambda[0] = lambda[0] - lambda[1];
	c_lambda[1] = 2.0 * lambda[1];
      }
    }
    else {
      fill_elinfo(1, el_info, c_el_info);
      if (outside >= 0) 
	{
	  outside = world_to_coord_1d(c_el_info, g_xy, c_lambda);
	  if (outside >= 0) ERROR("point outside domain\n");
	}
      else {
	c_lambda[1] = lambda[1] - lambda[0];
	c_lambda[0] = 2.0 * lambda[0];
      }
    }
    break;
#if DIM_OF_WORLD > 1
  case 2:
    if (lambda[0] >= lambda[1]) {
      fill_elinfo(0, el_info, c_el_info);
      if (el->new_coord) {
	outside = world_to_coord_2d(c_el_info, g_xy, c_lambda);
	if (outside >= 0) {
	  ERROR("outside curved boundary child 0\n");
	}
      }
      else {
	c_lambda[0] = lambda[2];
	c_lambda[1] = lambda[0] - lambda[1];
	c_lambda[2] = 2.0 * lambda[1];
      }
    }
    else {
      fill_elinfo(1, el_info, c_el_info);
      if (el->new_coord) {
	outside = world_to_coord_2d(c_el_info, g_xy, c_lambda);
	if (outside >= 0) {
	  ERROR("outside curved boundary child 1\n");
	}
      }
      else {
	c_lambda[0] = lambda[1] - lambda[0];
	c_lambda[1] = lambda[2];
	c_lambda[2] = 2.0 * lambda[0];
      }
    }
    break;
#if DIM_OF_WORLD > 2
  case 3:
    if (el->new_coord) {
      int ichild, c_outside;


      if (lambda[0] >= lambda[1])
	ichild = 0;
      else
	ichild = 1;
      fill_elinfo(ichild, el_info, c_el_info);
      c_outside = world_to_coord_3d(c_el_info, g_xy, c_lambda);
      
      if (c_outside>=0) {  /* test is other child is better... */
	REAL c_lambda2[N_LAMBDA];
	int  c_outside2;
	EL_INFO c_el_info2[1];
	
	fill_elinfo(1-ichild, el_info, c_el_info2);
	c_outside2 = world_to_coord_3d(c_el_info2, g_xy, c_lambda2);
	
	MSG("new_coord CHILD %d: outside=%d, lambda=(%.2lf %.2lf %.2lf %.2lf)\n", ichild, c_outside, c_lambda[0],c_lambda[1],c_lambda[2],c_lambda[3]);
	MSG("new_coord CHILD %d: outside=%d, lambda=(%.2lf %.2lf %.2lf %.2lf)\n", 1-ichild, c_outside2, c_lambda2[0],c_lambda2[1],c_lambda2[2],c_lambda2[3]);
	
	/* if (c_lambda[0] < c_lambda2[0]) { */
	if ((c_outside2 < 0) ||
	    (c_lambda2[c_outside2] > c_lambda[c_outside])) {

	  for (i=0; i <= dim; i++) c_lambda[i] = c_lambda2[i];
	  c_outside = c_outside2;
	  *c_el_info = *c_el_info2;
	  ichild = 1 - ichild;
	}
      }
      outside = c_outside;
      
      /* if (outside >= 0) ERROR("outside curved boundary child 0\n"); */
    }
    else {  /* no new_coord */
      if (lambda[0] >= lambda[1]) {
	fill_elinfo(0, el_info, c_el_info);
	MSG("transform lambda child 0\n");
	c_lambda[0] = lambda[0] - lambda[1];
	c_lambda[1] = lambda[child_vertex_3d[el_info->el_type][0][1]];
	c_lambda[2] = lambda[child_vertex_3d[el_info->el_type][0][2]];
	c_lambda[3] = 2.0 * lambda[1];
      }
      else {
	fill_elinfo(1, el_info, c_el_info);
	MSG("transform lambda child 1\n");
	c_lambda[0] = lambda[1] - lambda[0];
	c_lambda[1] = lambda[child_vertex_3d[el_info->el_type][1][1]];
	c_lambda[2] = lambda[child_vertex_3d[el_info->el_type][1][2]];
	c_lambda[3] = 2.0 * lambda[0];
      }
    }
#endif
#endif
  }
    
  inside = find_el_at_pt_recursive(c_el_info, c_lambda, outside);
  return(inside);
}

/*--------------------------------------------------------------------------*/

int find_el_at_pt(MESH *mesh, const REAL_D xy, EL_INFO **el_info_p, FLAGS flag,
		  REAL bary[N_LAMBDA], const MACRO_EL *start_mel,
		  const REAL_D xy0, REAL *sp)
{
  const MACRO_EL  *mel;
  REAL      lambda[N_LAMBDA];
  EL_INFO   mel_info[1];
  int       i, k, inside, dim = mesh->dim;

  TEST_EXIT(el_info_p,"need pointer to pointer to an el_info structure\n");

  if (start_mel)
    mel = start_mel;
  else
    mel = mesh->macro_els;

  mel_info->fill_flag = flag|FILL_COORDS;
  g_xy  = xy;
  g_xy0 = xy0;
  g_sp  = sp;

  fill_macro_info(mesh, mel, mel_info);

  while ((k = world_to_coord(mel_info, xy, lambda)) >= 0) {
    if (mel->neigh[k]) {
      mel = mel->neigh[k];
      fill_macro_info(mesh, mel, mel_info);
      continue;
    }

    break;
  }

  /* now, descend in tree to find leaf element at point */
  inside = find_el_at_pt_recursive(mel_info, lambda, k);
  for (i = 0; i <= dim; i++) bary[i] = final_lambda[i];
  *el_info_p = final_el_info;
  
  /* if ((k>=0) || (!inside)) WAIT; */

  return(inside);
}
