/*=========================================================================

  Module:    $RCSfile: vtkKWFileListTable.cxx,v $

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkKWFileListTable.h"

#include "vtkKWApplication.h"
#include "vtkKWOptions.h"
#include "vtkKWEntry.h"
#include "vtkKWEntryWithLabel.h"
#include "vtkKWIcon.h"
#include "vtkKWFileBrowserUtilities.h"
#include "vtkKWInternationalization.h"
#include "vtkKWLabel.h"
#include "vtkMath.h"
#include "vtkKWMenu.h"
#include "vtkKWMessageDialog.h"
#include "vtkKWMultiColumnList.h"
#include "vtkKWMultiColumnListWithScrollbars.h"
#include "vtkKWPushButton.h"
#include "vtkKWScrollbar.h"
#include "vtkKWSimpleEntryDialog.h"
#include "vtkKWTkUtilities.h"

#include "vtkDirectory.h"
#include "vtkObjectFactory.h"
#include <vtksys/SystemTools.hxx>
#include <vtksys/ios/sstream>
#include <vtksys/stl/string>
#include <vtksys/Glob.hxx>
#include <sys/stat.h>
#include <time.h>
#include <locale.h>

// #define _MY_DEBUG

//----------------------------------------------------------------------------
vtkStandardNewMacro( vtkKWFileListTable );
vtkCxxRevisionMacro(vtkKWFileListTable, "$Revision: 1.29 $");

/* 
 * This part was generated by ImageConvert from image:
 *    folderxp16x13.png (zlib, base64) (image file)
 */
#define image_folder_width          16
#define image_folder_height         13
#define image_folder_pixel_size     4
#define image_folder_length         456

static unsigned char image_folder[] = 
  "eNpjYICAMzON/5+eYfT/5DSD/8en6P0HCikCMRcDEQCkFx0c6tcEmWEBNYMRC0bTOxMN//"
  "+/t0vl/652xf/bW+T+b22S+b+5Qer/hlpxmNs4QfpBbv7/feL//1/6MDEWsK5a9D8zM3Mp"
  "UKsSyB0g//5/3fr//7OG/0cn6WDFB3rV/+/pVP6/o1X+/5ZGabA7QBjkDlBY/X9QBVZHFA"
  "DaAwKLC7jA7gDru5L3HxIOMwniH0fjwfSCXHaQ/imgcP5/Jg0cd2A1oLDAg9/vCQOrm5fN"
  "CtI/FeQ3kH6I+2dCwgIPfrbZF6xuTiYzWD8ojkD69/eoQfQDwwIfvrPSGaxuVjojWD8ofk"
  "H6QWEL1g8MC3z48gIrsLqZaQxg/4PiADltgOJ3dYXQ/xWl/P+XFvGAwxkUViD/gtwMshek"
  "l4mJaStQfx00fVqA2CD3kIBBei0AqWHo/Q==";

//----------------------------------------------------------------------------
class vtkKWFileListTableInternals
{
public:
  
  vtkKWFileListTableInternals()
  {
    this->SortedOrder = false;
    this->NumberThousandsSeparator = '\0';
  }
  
  vtksys_stl::string FolderImage;
  bool SortedOrder; //true, increasing; false, decreasing
  char NumberThousandsSeparator;
};

//----------------------------------------------------------------------------
vtkKWFileListTable::vtkKWFileListTable()
{  
  this->Internals = new vtkKWFileListTableInternals;
  this->FileList = vtkKWMultiColumnListWithScrollbars::New();
  this->ContextMenu = NULL;

  this->FileSelectedCommand      = NULL;
  this->FileDoubleClickedCommand = NULL;
  this->FileDeletedCommand       = NULL;
  this->FileRenamedCommand       = NULL;
  this->FolderCreatedCommand     = NULL;

  this->ParentDirectory          = NULL;
  this->FilePattern              = NULL;
  this->FileExtensions           = NULL;
}

//----------------------------------------------------------------------------
vtkKWFileListTable::~vtkKWFileListTable()
{
  this->FileList->GetWidget()->RemoveBinding("<FocusIn>", this, "FocusInCallback");

  if (this->FileSelectedCommand)
    { 
    delete [] this->FileSelectedCommand;
    this->FileSelectedCommand = NULL;
    }

  if (this->FileDoubleClickedCommand)
    {
    delete [] this->FileDoubleClickedCommand;
    this->FileDoubleClickedCommand = NULL;
    }    

  if (this->FileDeletedCommand)
    {
    delete [] this->FileDeletedCommand;
    this->FileDeletedCommand = NULL;
    }

  if (this->FileRenamedCommand)
    {
    delete [] this->FileRenamedCommand;
    this->FileRenamedCommand = NULL;
    }

  if (this->FolderCreatedCommand)
    {
    delete [] this->FolderCreatedCommand;
    this->FolderCreatedCommand = NULL;
    }

  this->FileList->Delete();
  if (this->ContextMenu)
    {
    this->ContextMenu->Delete();
    this->ContextMenu = NULL;
    }

  if (this->ParentDirectory)
    {
    delete [] this->ParentDirectory;
    this->ParentDirectory = NULL;
    }

  if (this->FilePattern)
    {
    delete [] this->FilePattern;
    this->FilePattern = NULL;
    }

  if (this->FileExtensions)
    {
    delete [] this->FileExtensions;
    this->FileExtensions = NULL;
    }

  if (this->Internals)
    {      
    delete this->Internals;
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::CreateWidget()
{
  // Check if already created

  if (this->IsCreated())
    {
    vtkErrorMacro(<< this->GetClassName() << " already created");
    return;
    }

  // Call the superclass to create the whole widget

  this->Superclass::CreateWidget();
  
  this->FileList->SetParent(this);
  this->FileList->Create();
  this->FileList->GetVerticalScrollbar()->SetConfigurationOptionAsInt(
    "-takefocus", 0);
  this->FileList->GetHorizontalScrollbar()->SetConfigurationOptionAsInt(
    "-takefocus", 0);

  vtkKWMultiColumnList *filelist = this->FileList->GetWidget();

  // Setup the image for the tree node

  this->Internals->FolderImage = filelist->GetWidgetName();
  this->Internals->FolderImage.append("_0");
  if (!vtkKWTkUtilities::UpdatePhoto(
        this->GetApplication(),
        this->Internals->FolderImage.c_str(),
        image_folder, 
        image_folder_width, 
        image_folder_height, 
        image_folder_pixel_size,
        image_folder_length))
    {
    vtkWarningMacro(
      << "Error updating Tk photo " 
      << this->Internals->FolderImage.c_str());
    }
    
  // Setup the file list aspects

  filelist->MovableColumnsOn();
  filelist->SetWidth(60);
  filelist->SetHeight(18);
  //filelist->ExportSelectionOn();

  int col_index = filelist->AddColumn("Name");
  filelist->SetColumnFormatCommand(
    col_index, this, "GetRealNameStringCallback");
  filelist->SetColumnSortMode(
    col_index, vtkKWMultiColumnList::SortModeDictionary);

  col_index = filelist->AddColumn("Size");
  filelist->SetColumnAlignmentToRight(col_index);
  filelist->SetColumnFormatCommand(
    col_index, this, "GetFormatSizeStringCallback");
  filelist->SetColumnSortModeToInteger(col_index);

  col_index = filelist->AddColumn("Modified time");
  filelist->SetColumnAlignmentToRight(col_index);
  filelist->SetColumnFormatCommand(
    col_index, this, "GetFormatTimeStringCallback");
  filelist->SetColumnSortMode(
    col_index, vtkKWMultiColumnList::SortModeCommand);
  filelist->SetColumnSortCommand(
    col_index, this, "SortTimeCallback");

  col_index = filelist->AddColumn("HiddenSort");
  filelist->ColumnVisibilityOff(col_index);
  filelist->SetColumnSortModeToInteger(col_index);

  for(int i = 0; i < filelist->GetNumberOfColumns(); i++)
    {
    filelist->SetColumnEditable(i, 0);
    }

  filelist->SetSortedColumnBackgroundColor(0.96, 0.96, 0.96);
  filelist->SetBackgroundColor(1.0, 1.0, 1.0);
  filelist->SetColumnSeparatorsVisibility(0);
  filelist->SetStripeHeight(0);
  filelist->ColorSortedColumnOn();
  filelist->ClearStripeBackgroundColor();
  filelist->SetRowSpacing(0);
  
  filelist->SetSelectionCommand(this, "SelectedFileChangedCallback");

  filelist->SetBinding("<Delete>", this, "RemoveSelectedFileCallback");
  filelist->SetUneditableCellDoubleClickCommand(
    this, "FileDoubleClickCallback");
  filelist->SetRightClickCommand(
    this, "ContextMenuCallback");

  // Change the Home/End/Prior/Next keypress behavior to be like win explorer

  filelist->SetBinding("<Home>", this, "KeyHomeEndNavigationCallback Home");
  filelist->SetBinding("<End>", this, "KeyHomeEndNavigationCallback End");
  filelist->SetBinding("<Prior>", this, 
                       "KeyPriorNextNavigationCallback  %W %x %y %X %Y Prior");
  filelist->SetBinding("<Next>", this, 
                       "KeyPriorNextNavigationCallback  %W %x %y %X %Y Next");
  
  filelist->SetBinding("<Return>", this, "FileDoubleClickCallback");
  filelist->SetBinding("<F2>", this, "RenameFileCallback");
  filelist->SetBinding("<FocusIn>", this, "FocusInCallback");

  this->Script(
    "pack %s -side top -fill both -expand true -padx 1 -pady 1",
    this->FileList->GetWidgetName());
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::AddBindingToInternalWidget(const char* kwevent,
    vtkObject *obj, const char* method)
{
  this->FileList->GetWidget()->AddBinding(kwevent, obj, method);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::RemoveBindingFromInternalWidget(const char* kwevent,
    vtkObject *obj, const char* method)
{
  this->FileList->GetWidget()->RemoveBinding(kwevent, obj, method);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetSelectionMode(int arg)
{
  if (arg != this->FileList->GetWidget()->GetSelectionMode())
    {
    this->FileList->GetWidget()->SetSelectionMode(arg);
    }
}

void vtkKWFileListTable::SetSelectionModeToSingle() 
{ 
  this->SetSelectionMode(vtkKWOptions::SelectionModeSingle); 
};
void vtkKWFileListTable::SetSelectionModeToBrowse() 
{ 
  this->SetSelectionMode(vtkKWOptions::SelectionModeBrowse); 
};
void vtkKWFileListTable::SetSelectionModeToMultiple() 
{ 
  this->SetSelectionMode(vtkKWOptions::SelectionModeMultiple); 
};
void vtkKWFileListTable::SetSelectionModeToExtended() 
{ 
  this->SetSelectionMode(vtkKWOptions::SelectionModeExtended); 
};

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetParentDirectory(const char* _arg)
{
  this->SetParentDirectoryInternal(_arg);
  this->ShowFileList(_arg, this->GetFilePattern(), 
    this->GetFileExtensions());
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetParentDirectoryInternal(const char *_arg)
{
  if (this->ParentDirectory == NULL && _arg == NULL) 
    { 
    return;
    }

  if (this->ParentDirectory && _arg && 
    (!strcmp(this->ParentDirectory, _arg))) 
    {
    return;
    }

  if (this->ParentDirectory) 
    { 
    delete [] this->ParentDirectory; 
    }

  if (_arg)
    {
    this->ParentDirectory = new char[strlen(_arg) + 1];
    strcpy(this->ParentDirectory, _arg);
    }
  else
    {
    this->ParentDirectory = NULL;
    }

  this->Modified();
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFilePattern(const char* _arg)
{
  this->SetFilePatternInternal(_arg);
  this->ShowFileList(this->GetParentDirectory(), _arg, 
    this->GetFileExtensions());
}
//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFilePatternInternal(const char *_arg)
{
  if (this->FilePattern == NULL && _arg == NULL) 
    { 
    return;
    }

  if (this->FilePattern && _arg && 
    (!strcmp(this->FilePattern, _arg))) 
    {
    return;
    }

  if (this->FilePattern) 
    { 
    delete [] this->FilePattern; 
    }

  if (_arg)
    {
    this->FilePattern = new char[strlen(_arg) + 1];
    strcpy(this->FilePattern, _arg);
    }
  else
    {
    this->FilePattern = NULL;
    }

  this->Modified();
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFileExtensions(const char* _arg)
{
  this->SetFileExtensionsInternal(_arg);
  this->ShowFileList(this->GetParentDirectory(), 
    this->GetFilePattern(), _arg);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFileExtensionsInternal(const char *_arg)
{
  if (this->FileExtensions == NULL && _arg == NULL) 
    { 
    return;
    }

  if (this->FileExtensions && _arg && 
    (!strcmp(this->FileExtensions, _arg))) 
    {
    return;
    }

  if (this->FileExtensions) 
    { 
    delete [] this->FileExtensions; 
    }

  if (_arg)
    {
    this->FileExtensions = new char[strlen(_arg) + 1];
    strcpy(this->FileExtensions, _arg);
    }
  else
    {
    this->FileExtensions = NULL;
    }

  this->Modified();
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::Focus()
{
  if (this->IsCreated())
    {
    // force the window focus to this table
    this->FileList->GetWidget()->Focus();
    }
}

//----------------------------------------------------------------------------
int vtkKWFileListTable::HasFocus()
{
  if (this->IsCreated())
    {
    // force the window focus to this table
    return this->FileList->GetWidget()->HasFocus();
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::GetSelectionBackgroundColor(
  double *r, double *g, double *b)
{
  this->FileList->GetWidget()->GetSelectionBackgroundColor(r, g, b);
}

//----------------------------------------------------------------------------
double* vtkKWFileListTable::GetSelectionBackgroundColor()
{
  return this->FileList->GetWidget()->GetSelectionBackgroundColor();
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetSelectionBackgroundColor(
  double r, double g, double b)
{
  this->FileList->GetWidget()->SetSelectionBackgroundColor(r, g, b);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::GetSelectionForegroundColor(
  double *r, double *g, double *b)
{
  this->FileList->GetWidget()->GetSelectionForegroundColor(r, g, b);
}

//----------------------------------------------------------------------------
double* vtkKWFileListTable::GetSelectionForegroundColor()
{
  return this->FileList->GetWidget()->GetSelectionForegroundColor();
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetSelectionForegroundColor(double r, double g, double b)
{
  this->FileList->GetWidget()->SetSelectionForegroundColor(r, g, b);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::GetTableBackgroundColor(
  double *r, double *g, double *b)
{
  this->FileList->GetWidget()->GetBackgroundColor(r, g, b);
}

//----------------------------------------------------------------------------
double* vtkKWFileListTable::GetTableBackgroundColor()
{
  return this->FileList->GetWidget()->GetBackgroundColor();
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetTableBackgroundColor(
  double r, double g, double b)
{
  this->FileList->GetWidget()->SetBackgroundColor(r, g, b);
}

//----------------------------------------------------------------------------
const char* vtkKWFileListTable::GetCellText(int row, int col)
{
  if (this->FileList->IsCreated())
    {
    if (col == 0)
      {
      return this->GetRealNameStringCallback(
        this->FileList->GetWidget()->GetCellText(row, col));
      }
    else if (col == 1)
      {
      return this->GetFormatSizeStringCallback(
        this->FileList->GetWidget()->GetCellText(row, col));
      }
    else if (col == 2)
      {
      return this->GetFormatTimeStringCallback(
        this->FileList->GetWidget()->GetCellText(row, col));
      }
    }
    
  return NULL;
}

//----------------------------------------------------------------------------
int vtkKWFileListTable::GetNumberOfSelectedFileNames()
{
  return this->FileList->GetWidget()->GetNumberOfSelectedRows();
}

//----------------------------------------------------------------------------
const char *vtkKWFileListTable::GetNthSelectedFileName(int i)
{
  int selrows = 
    this->FileList->GetWidget()->GetNumberOfSelectedRows();
  if (i < 0 || i >= selrows)
    {
    return NULL;
    }
  if (selrows > 0)
    {
    int *indices = new int [selrows];
    this->FileList->GetWidget()->GetSelectedRows(indices);
    int row = indices[i];
    delete [] indices;
    return this->GetRowFileName(row);
    }
  else
    {
    return NULL;
    }
}

//----------------------------------------------------------------------------
char* vtkKWFileListTable::GetRowFileName(int row)
{
  vtksys_stl::string fullname = this->GetParentDirectory();
  if (!fullname.empty())
    {
    vtksys_stl::string filename = this->GetCellText(row, 0);
    if (!KWFileBrowser_HasTrailingSlash(fullname.c_str()))
      {
      fullname += KWFileBrowser_PATH_SEPARATOR;
      }
    fullname.append(filename.c_str());
    return KWFileBrowser_GetUnixPath(fullname.c_str());
    }
  else
    {
    return NULL;
    }
}

//----------------------------------------------------------------------------
int vtkKWFileListTable::ShowFileList(
  const char* inpath, 
  const char* filepattern,
  const char* fileextensions)
{
  if (!this->IsCreated() || !inpath || !(*inpath))
    {
    return 0;
    }

  vtksys_stl::string parentpath = inpath;
  vtkKWTkUtilities::SetTopLevelMouseCursor(this, "watch");

  // Check if we need use pattern filters

  bool bUsePattern = false;
  vtkstd::vector<vtkstd::string> filenames;
  if (filepattern && *filepattern)
    {
    vtksys_stl::string fullPattern = filepattern;
    if (fullPattern.find("*") != vtksys_stl::string::npos ||
        fullPattern.find("?") != vtksys_stl::string::npos)
      {
      bUsePattern = true;
      vtksys::Glob glob;
      glob.RecurseOff();

      vtkstd::vector<vtkstd::string> components;
      vtksys::SystemTools::SplitPath(fullPattern.c_str(), components);

      // If Pattern is a relative path, prepend with Directory
      
      if (components[0] == "")
        {
        components.insert(components.begin(), this->GetParentDirectory());
        fullPattern = vtksys::SystemTools::JoinPath(components);
        }
      
      if (!glob.FindFiles(fullPattern.c_str()))
        {
        vtkErrorMacro(<< "FindFileNames: Glob action failed for \"" <<
                      fullPattern.c_str() << "\"");
        return 0;
        }

      // copy the filenames from glob

      filenames = glob.GetFiles();
      this->SetFilePatternInternal(fullPattern.c_str());
      }
    }
     
  // Check if we need use extension filters

  bool bUseExt = false;  
  vtksys_stl::vector<vtksys_stl::string> fileexts;
  vtksys_stl::vector<vtksys_stl::string>::iterator it;
  
  // The pattern filter should overwrite extension filter.

  if (!bUsePattern)
    {
    if (fileextensions && *fileextensions)
      {
      vtksys::SystemTools::Split(fileextensions, fileexts, ' ');
      }
    if (fileexts.size()>0)
      {
      bUseExt = true;
      for(it = fileexts.begin(); it != fileexts.end(); it++)
        {
        if (strcmp((*it).c_str(), ".*") == 0)
          {
          bUseExt = false;
          break;
          }
        }
      if(bUseExt)
        {
        this->SetFileExtensionsInternal(fileextensions);
        }
      }
    }
    
  vtkKWMultiColumnList *filelist = this->FileList->GetWidget();
  
#if defined (_MY_DEBUG)  
  cout << "-----------------UpdateFileList: " << parentpath << endl;
  clock_t start = clock();
#endif

  vtkDirectory *dir = vtkDirectory::New();
  if (!dir->Open(parentpath.c_str()))
    {
    dir->Delete();
    this->FileList->GetWidget()->ClearSelection();
    if (this->FileList->GetWidget()->GetNumberOfRows())
      {
      this->FileList->GetWidget()->DeleteAllRows();
      }
    this->SetParentDirectoryInternal(NULL);
    this->SetFilePatternInternal(NULL);
    this->SetFileExtensionsInternal(NULL);
    vtkKWTkUtilities::SetTopLevelMouseCursor(this, NULL);
    return 0;
    }
      
  int num_files = dir->GetNumberOfFiles();
  
#if defined (_MY_DEBUG)  
  double durationopen = (double)(clock() - start) / CLOCKS_PER_SEC;
  cout << "Dir open time: " << durationopen << endl;
  start = clock();
#endif

  // Have these two flags so that we do not need to do strcmp
  // for every file in the directory

  bool dotfound = false, dotdotfound = false;

  vtksys_ios::ostringstream tk_filecmd, tk_foldercmd, tk_configcelllist;
  const char *listname = filelist->GetWidgetName();

  int folder_index = 0; 
  vtksys_stl::string filename = "",fullname = "";
  const char* image_name = this->Internals->FolderImage.c_str();
   
  struct stat fs;
  this->SetParentDirectoryInternal(parentpath.c_str());
  if (!KWFileBrowser_HasTrailingSlash(parentpath.c_str()))
    {
    parentpath += KWFileBrowser_PATH_SEPARATOR;
    }
      
  vtksys_stl::string::size_type dot_pos;

  for (int i = 0; i < num_files; i++)
    {
    filename = dir->GetFile(i);
    // skip . and ..
    if (!dotfound || !dotdotfound)
      { 
      if (strcmp(filename.c_str(), ".") == 0)
        {
        dotfound=true;
        continue;
        }
      else if (strcmp(filename.c_str(), "..") == 0)
        {
        dotdotfound = true;
        continue;
        }
      }
      
    fullname = parentpath;
    fullname += filename;
    if (stat(fullname.c_str(), &fs) != 0)
      {
      continue;
      }

  // for files, prepend 1 to name; add 1 to size; prepend + to time;
  // for folders, prepend 0 to name; size alway 0; prepend - to time;
  // so that during sorting, we can always keep folder and files separated.
    
#if defined( _WIN32 )
    if (!(fs.st_mode & _S_IFDIR)) 
#else
    if (!S_ISDIR(fs.st_mode)) 
#endif
      {
      if (bUsePattern)
        {
        for (it=filenames.begin(); it != filenames.end(); it++)
          {
          if (strcmp(vtksys::SystemTools::GetFilenameName(
                       (*it).c_str()).c_str(),
                     filename.c_str()) == 0)
            {
            filenames.erase(it);
            tk_filecmd << listname << " insert end {\"b" 
                       << filename.c_str() << "\"" << " " << ++fs.st_size 
                       << " +" << fs.st_mtime << " 1}" << endl;
            break;
            }
          }
        }
      else if (bUseExt)
        {
        for(it = fileexts.begin(); it != fileexts.end(); it++)
          {
          dot_pos = filename.rfind(".");
          if (dot_pos != vtksys_stl::string::npos &&
              strcmp((*it).c_str(), filename.substr(dot_pos).c_str()) == 0)
            {
            // The ++size is for sorting 0-byte files (become 1 byte) 
            // with folders (size 0)
            tk_filecmd << listname << " insert end {\"b" 
                       << filename.c_str() << "\"" << " " << ++fs.st_size 
                       << " +" << fs.st_mtime << " 1}" << endl;
            break;
            }
          }
        }
      else
        {
        // The ++size is for sorting 0-byte files (become 1 byte) 
        // with folders (size 0)
        tk_filecmd << listname << " insert end {\"b" 
                   << filename.c_str() << "\"" << " " << ++fs.st_size 
                   << " +" << fs.st_mtime << " 1}" << endl;
        }
      }
    else
      {
      tk_foldercmd << listname << " insert end {\"a" 
                   << filename.c_str() << "\" 0 -" 
                   << fs.st_mtime << " 0}" << endl;
      // Add folder image to this cell
      tk_configcelllist << folder_index++ << "," << 0 
        << " -image " << "{" << image_name << "} ";
      }//end if (isfolder)
    }//end for
  
#if defined (_MY_DEBUG)  
  durationopen = (double)(clock() - start) / CLOCKS_PER_SEC;
  cout << "Creat Script time: " << durationopen << endl;
  start = clock();
#endif
  
  dir->Delete();
  filelist->ClearSelection();
  if (filelist->GetNumberOfRows())
    {
    filelist->DeleteAllRows();
    }

  //Add the folder first 

  if (tk_foldercmd.str() != "")
    {
    vtkKWTkUtilities::EvaluateSimpleString(
      this->GetApplication(), tk_foldercmd.str().c_str());
    }

  //Add the folder icons

  if (tk_configcelllist.str() != "")
    {
    vtksys_ios::ostringstream tk_configcellcmd;
    tk_configcellcmd << listname << " configcells "
      << tk_configcelllist.str() << endl;
    vtkKWTkUtilities::EvaluateSimpleString(
      this->GetApplication(), tk_configcellcmd.str().c_str());
    }
  
  //Now add the file entries

  if (tk_filecmd.str() != "")
    {
    vtkKWTkUtilities::EvaluateSimpleString(
      this->GetApplication(), tk_filecmd.str().c_str());
    }

#if defined (_MY_DEBUG)  
  durationopen = (double)(clock() - start) / CLOCKS_PER_SEC;
  cout << "Run script time: " << durationopen << endl;
#endif

  int sortcol = filelist->GetLastSortedColumn();
  if (sortcol >= 0)
    {
    filelist->SortByColumn(sortcol, filelist->GetLastSortedOrder());
    }
  else
    {
    filelist->SortByColumnIncreasingOrder(0);
    }

  vtkKWTkUtilities::SetTopLevelMouseCursor(this, NULL);
  
  return 1;
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SelectFileName(const char* filename)
{
  if (!filename || !(*filename) ||
      !vtksys::SystemTools::FileExists(filename))
    {
    return;
    }
  
  vtkKWMultiColumnList *filelist = this->FileList->GetWidget();
  int numRows = filelist->GetNumberOfRows();
  vtksys_stl::string shortname = 
    vtksys::SystemTools::GetFilenameName(filename);
  for(int i = 0; i < numRows; i++)
    {
    if (!strcmp(this->GetCellText(i, 0), shortname.c_str()))
      {
      if(filelist->GetSelectionMode() == vtkKWOptions::SelectionModeSingle)
        {
        filelist->ClearSelection();
        }
      filelist->SelectRow(i);
      break;
      }
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::DeselectFileName(const char* filename)
{
  if (!filename || !(*filename) ||
      !vtksys::SystemTools::FileExists(filename))
    {
    return;
    }
  
  vtkKWMultiColumnList *filelist = this->FileList->GetWidget();
  int selrows = filelist->GetNumberOfSelectedRows();
  if (selrows>0)
    {
    int *indices = new int [selrows];
    filelist->GetSelectedRows(indices);
    vtksys_stl::string shortname = 
      vtksys::SystemTools::GetFilenameName(filename);
    for(int i = 0; i < selrows; i++)
      {
      if (!strcmp(this->GetCellText(indices[i], 0), shortname.c_str()))
        {
        filelist->DeselectRow(indices[i]);
        break;
        }
      }
    delete [] indices;
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::ClearSelection()
{
  this->FileList->GetWidget()->ClearSelection();
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::ScrollToFile(const char* prefix)
{
  if(prefix && *prefix)
    {
    vtkKWMultiColumnList *filelist = this->FileList->GetWidget();
    int numRows = filelist->GetNumberOfRows();
    for(int i = 0; i < numRows; i++)
      {
      if (!strncmp(this->GetCellText(i, 0), prefix, strlen(prefix)))
        {
        filelist->SeeRow(i);
        break;
        }
      }
    }
}

//----------------------------------------------------------------------------
char* vtkKWFileListTable::GetRealNameStringCallback(
  const char* celltext)
{
  if (celltext && *celltext)
    {
    static char newstr[256];
    strcpy(newstr, ++celltext);
    return newstr;
    }
  return NULL;
}

//----------------------------------------------------------------------------
char * vtkKWFileListTable::GetFormatSizeStringCallback(
  const char* celltext)
{
  if (celltext && *celltext)
    {
    long unsigned int u;
    sscanf(celltext, "%lu", &u);
    if (u > 0)
      {
      // determine thousands separator
      if (this->Internals->NumberThousandsSeparator == '\0') 
        {
        struct lconv *current_locale = localeconv();
        // if the locale defines thousands_sep 
        if (current_locale->thousands_sep && *current_locale->thousands_sep) 
          {
          // copy the thousands separator. 
          this->Internals->NumberThousandsSeparator = 
            *(current_locale->thousands_sep);
          }
        else 
          {
          // default is the comma
          this->Internals->NumberThousandsSeparator = ',';
          }
        }

      char buffer[50];
      sprintf(buffer, "%d", 
              vtkMath::Round(ceil((double)(u-1)/1024)));
      // put in the thousand sep if necessary  
      vtksys_stl::string outStr = buffer;
      vtksys_stl::string tmpStr = buffer;
      while (tmpStr.size() > 3)
        {
        outStr.insert(tmpStr.size() - 3, 1, 
                      this->Internals->NumberThousandsSeparator);
        tmpStr = tmpStr.substr(0,tmpStr.size()-3);
        }
      static char newStr[64];
      sprintf(newStr, "%s KB", outStr.c_str());
      return newStr;
      }
    }
  return NULL;
}
//----------------------------------------------------------------------------
int vtkKWFileListTable::SortTimeCallback(
  const char* celltext1,
  const char* celltext2)
{
  int num1 = atoi(celltext1);
  int num2 = atoi(celltext2);
  
  if (num1 == num2)
    {
    return 0;
    }
    
  if (num1 < 0 && num2 < 0)
    {
    return (num1 > num2 ? -1 : 1);
    }
  
  return (num1 > num2 ? 1 : -1);
}

//----------------------------------------------------------------------------
char * vtkKWFileListTable::GetFormatTimeStringCallback(
  const char* celltext)
{
  if (celltext && *celltext)
    {
    time_t t=0;
    sscanf(++celltext, "%lu", &t);
    struct tm *new_time = localtime(&t);
    static char buffer[100];
    strftime(buffer, 100, "%c", new_time); 
    return buffer;
    }
  return NULL;
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::FileDoubleClickCallback()
{
  if (this->FileList->GetWidget()->GetNumberOfSelectedRows() > 0)
    {
    int selrows = this->FileList->GetWidget()->GetNumberOfSelectedRows();
    int *indices = new int [selrows];
    this->FileList->GetWidget()->GetSelectedRows(indices);
    vtksys_stl::string fullname = this->GetParentDirectory();

    if (!fullname.empty())
      {
      vtksys_stl::string filename = this->GetCellText(indices[0], 0);
      if (!KWFileBrowser_HasTrailingSlash(fullname.c_str()))
        {
        fullname += KWFileBrowser_PATH_SEPARATOR;
        }
      fullname.append(filename.c_str());
      this->InvokeFileDoubleClickedCommand(fullname.c_str());
      }
    delete [] indices;
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SelectedFileChangedCallback()
{
  if (this->FileList->GetWidget()->GetNumberOfSelectedRows() > 0)
    {
    int selrows = this->FileList->GetWidget()->GetNumberOfSelectedRows();
    int *indices = new int [selrows];
    this->FileList->GetWidget()->GetSelectedRows(indices);
    this->InvokeFileSelectedCommand(this->GetRowFileName(indices[0]));
    delete [] indices;
    }
  else
    {
    this->InvokeFileSelectedCommand(NULL);
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::KeyHomeEndNavigationCallback(
  const char *key)
{
  if (this->FileList->GetWidget()->GetNumberOfSelectedRows() > 0 &&
      key && *key)
    {
    vtksys_ios::ostringstream tk_cmd;
    const char* name = this->FileList->GetWidget()->GetWidgetName();
    if (strcmp(key, "Home") == 0)
      {
      tk_cmd << "tablelist::changeSelection " << name 
             << " [tablelist::firstVisibleRow "
             << name << "] -1" << endl;
      }
    else if (strcmp(key, "End") == 0)
      {
      tk_cmd << "tablelist::changeSelection " << name 
             << " [tablelist::lastVisibleRow "
             << name << "] -1" << endl;
      }
    vtkKWTkUtilities::EvaluateSimpleString(
      this->GetApplication(), tk_cmd.str().c_str());
    }
}

//----------------------------------------------------------------------------
//Reference: proc tablelist::yviewSubCmd {win argList} -- in tablelistBind.tcl
//           proc tablelist::autoScan win 
void vtkKWFileListTable::KeyPriorNextNavigationCallback(
  const char *w, int x, int y, 
  int root_x, int root_y, const char* key)
{
  if (this->IsCreated() && key && *key)
    {
    vtksys_ios::ostringstream tk_cmd;
    const char* widgetname = this->FileList->GetWidget()->GetWidgetName();

    tk_cmd << "upvar #0 ::tablelist::ns" 
           << widgetname<< "::data data" << endl;
    
    tk_cmd << "set w $data(body)" << endl;

    //Figure out how many units in each page

    tk_cmd << "set btmY [expr {[winfo height $w] - 1}]" << endl;
    tk_cmd << "set topRow [expr {int([$w index @0,0]) - 1}]" << endl;
    tk_cmd << "set btmRow [expr {int([$w index @0,$btmY]) - 1}]" << endl;
    
    tk_cmd << "set cur_selrow [lindex [" 
           << widgetname << " curselection] 0]" << endl;
    tk_cmd << "set toScroll 0" << endl;
    
    if (strcmp(key, "Prior") == 0)
      {
      tk_cmd << "if { $cur_selrow > $topRow } {" << endl;
        tk_cmd << " set row_index $topRow" << endl;
      tk_cmd << "} else {" << endl;   
        tk_cmd << " set toScroll 1" << endl;
      tk_cmd << " }" << endl;
      }
    else // <Next>
      {
      tk_cmd << "if { $cur_selrow < [expr {$btmRow -1}] } {" << endl;
        tk_cmd << " set row_index [expr {$btmRow}]" << endl;
      tk_cmd << "} else {" << endl;   
        tk_cmd << " set toScroll 1" << endl;
        //tk_cmd << "$w yview scroll 1 pages" << endl;
      tk_cmd << " }" << endl;
      }
    
    tk_cmd << "if { $toScroll == 1 } {" << endl;   
      tk_cmd << "set curNonHiddenCount [tablelist::getNonHiddenRowCount "
             << widgetname 
             << " 0 $cur_selrow]" << endl;
             
      tk_cmd << "set winNonHiddenCount [tablelist::getNonHiddenRowCount "
             << widgetname << " $topRow $btmRow]" << endl;
      tk_cmd << "set delta [expr {$winNonHiddenCount - 1}]" << endl;
      
      if (strcmp(key, "Prior") == 0)
        {
        tk_cmd << "set delta [expr {(-1)*$delta}]" << endl;
        }
      else
        {
        tk_cmd << "set delta 0" << endl;
        }

      // Scroll the window  
      tk_cmd << "set offset [expr {$curNonHiddenCount + $delta}]" << endl;
      
      tk_cmd << "set row_index [tablelist::nonHiddenRowOffsetToRowIndex "
             << widgetname << " $offset]" << endl;

      if (strcmp(key, "Next") == 0)
        {
        tk_cmd << "set row_index [expr {$row_index - 2}]" << endl;
        }
      
      tk_cmd << "$w yview $row_index" << endl;
      
      tk_cmd << "set btmY [expr {[winfo height $w] - 1}]" << endl;
      tk_cmd << "set topRow [expr {int([$w index @0,0]) - 1}]" << endl;
      tk_cmd << "set btmRow [expr {int([$w index @0,$btmY]) - 1}]" << endl;
      if (strcmp(key, "Prior") == 0)
        {
        tk_cmd << "set row_index $topRow" << endl;
        }
      else 
        {
        tk_cmd << "set row_index $btmRow" << endl;
        }      
    tk_cmd << " }" << endl;
    
    // Set selection   
    tk_cmd << "tablelist::changeSelection " << widgetname
           << " $row_index -1" << endl;
     
    vtkKWTkUtilities::EvaluateSimpleString(
      this->GetApplication(), tk_cmd.str().c_str());
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::ExploreFileCallback()
{
#ifdef _WIN32
  if (this->FileList->GetWidget()->GetNumberOfSelectedRows() > 0)
    {
      if (vtksys::SystemTools::FileIsDirectory(
            this->GetSelectedFileName()))
        {
        this->GetApplication()->OpenLink(
          this->GetSelectedFileName());
        }
    }
#endif
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::FocusInCallback()
{
  if (this->IsAlive())
    {
    if (this->FileList->GetWidget()->GetNumberOfRows() > 0 &&
      this->FileList->GetWidget()->GetNumberOfSelectedRows()<=0)
      {
      this->FileList->GetWidget()->SelectRow(0);
      }
    }
}

//----------------------------------------------------------------------------
int vtkKWFileListTable::RenameFileCallback()
{
  if (this->FileList->GetWidget()->GetNumberOfSelectedRows() <= 0)
    {
    return 0;
    }

  int selrows = this->FileList->GetWidget()->GetNumberOfSelectedRows();
  vtksys_stl::string parentdir = this->GetParentDirectory();
    
  // Prompt the user for the name of the folder  

  vtkKWSimpleEntryDialog *dlg = vtkKWSimpleEntryDialog::New();
  dlg->SetParent(this);
  dlg->SetMasterWindow(this->GetParentTopLevel());
  dlg->SetDisplayPositionToPointer();
  dlg->SetTitle(ks_("File Browser|Dialog|Title|Rename file"));
  dlg->SetStyleToOkCancel();
  dlg->Create();
  dlg->GetEntry()->GetLabel()->SetText(
    ks_("File Browser|Dialog|File name:"));
  dlg->GetEntry()->GetWidget()->SetValue(
    vtksys::SystemTools::GetFilenameName(
      this->GetSelectedFileName()).c_str());
  dlg->SetText(ks_("File Browser|Dialog|Enter a new name for this file"));
    
  int ok = dlg->Invoke();
  vtksys_stl::string newname = dlg->GetEntry()->GetWidget()->GetValue();
  dlg->Delete();
  if (ok)
    {
    if (newname.empty() || 
        strcmp(newname.c_str(), "") == 0 ||
        strcmp(newname.c_str(), ".") == 0 ||
        strcmp(newname.c_str(), "..") == 0)
      {
      vtkKWMessageDialog::PopupMessage(
        this->GetApplication(), this, 
        ks_("File Browser|Title|Error!"),
        "Please enter a valid file name!", 
        vtkKWMessageDialog::ErrorIcon | 
        vtkKWMessageDialog::InvokeAtPointer);
      return 0;
      }
    }
  else
    {
    return 0;
    }    
    
  // Prompt the user for confirmation

  vtkDirectory *dir = vtkDirectory::New();
  if (!dir->Open(parentdir.c_str()))
    {
    dir->Delete();
    vtkKWMessageDialog::PopupMessage(
      this->GetApplication(), this, 
      ks_("File Browser|Title|Error!"),
      k_("The parent directory can't be opened."), 
      vtkKWMessageDialog::ErrorIcon | 
      vtkKWMessageDialog::InvokeAtPointer);
    return 0;
    }

  vtksys_stl::string filename, fullname;

  // Check if the file is already created

  for (int i = 0; i < dir->GetNumberOfFiles(); i++)
    {
    filename = dir->GetFile(i);
    if (strcmp(filename.c_str(), newname.c_str()) == 0)
      {
      vtkKWMessageDialog::PopupMessage(
        this->GetApplication(), this, 
        ks_("File Browser|Title|Error!"),
        k_("The file already exists."), 
        vtkKWMessageDialog::ErrorIcon | 
        vtkKWMessageDialog::InvokeAtPointer);
      dir->Delete();
      return 0;
      }
    }
      
  dir->Delete();

  // Rename the selected file

  if (!KWFileBrowser_HasTrailingSlash(parentdir.c_str()))
    {
    parentdir += KWFileBrowser_PATH_SEPARATOR;
    }
  fullname = parentdir + newname;
  vtksys_stl::string oldfile = this->GetSelectedFileName();

  if (rename(this->GetSelectedFileName(), fullname.c_str()) == 0)
    {
    int *indices = new int [selrows];
    this->FileList->GetWidget()->GetSelectedRows(indices);

    this->InvokeFileSelectedCommand(fullname.c_str());

    // Setup prefix for display

    vtksys_stl::string text;
    if (!vtksys::SystemTools::FileIsDirectory(fullname.c_str()))
      {
      text = "b";
      }
    else
      {
      text = "a";
      }
    this->FileList->GetWidget()->SetCellText(
      indices[0], 0, text.append(newname.c_str()).c_str());
    this->InvokeFileRenamedCommand(oldfile.c_str(), fullname.c_str());
    delete [] indices;
      
    return 1;
    }
  else
    {
    vtkKWMessageDialog::PopupMessage(
      this->GetApplication(), this, 
      ks_("File Browser|Title|Error!"),
      "The file name can not be changed!", 
      vtkKWMessageDialog::ErrorIcon | 
      vtkKWMessageDialog::InvokeAtPointer);
    }

  return 0;
}

//----------------------------------------------------------------------------
int vtkKWFileListTable::RemoveSelectedFileCallback()
{
  if (this->FileList->GetWidget()->GetNumberOfSelectedRows() <= 0)
    {
    return 0;
    }

  int selrows = this->FileList->GetWidget()->GetNumberOfSelectedRows();

  // Prompt the user for confirmation

  if (!vtkKWMessageDialog::PopupYesNo( 
        this->GetApplication(), 
        this, 
        ks_("File Browser|Title|Delete file"),
        k_("Are you sure you want to delete the selected file?"), 
        vtkKWMessageDialog::WarningIcon | 
        vtkKWMessageDialog::InvokeAtPointer))
    {
    return 0;
    }

  int *indices = new int [selrows];
  this->FileList->GetWidget()->GetSelectedRows(indices);
  vtksys_stl::string fullname = this->GetParentDirectory();

  if (!fullname.empty())
    {
    vtksys_stl::string filename = this->GetCellText(indices[0], 0);
    if (!KWFileBrowser_HasTrailingSlash(fullname.c_str()))
      {
      fullname += KWFileBrowser_PATH_SEPARATOR;
      }
    fullname.append(filename.c_str());
    int isDir = 0;
    if (vtksys::SystemTools::FileIsDirectory(fullname.c_str()))
      {
      if (!vtksys::SystemTools::RemoveADirectory(fullname.c_str()))
        {
        vtkKWMessageDialog::PopupMessage(
          this->GetApplication(), this, 
          ks_("File Browser|Title|Error!"),
          "The directory can not be removed!", 
          vtkKWMessageDialog::ErrorIcon | 
          vtkKWMessageDialog::InvokeAtPointer);
        delete[] indices;
        return 0;
        }
      isDir = 1;
      }
    else if (vtksys::SystemTools::FileExists(fullname.c_str()))
      {
      if (!vtksys::SystemTools::RemoveFile(fullname.c_str()))
        {
        vtkKWMessageDialog::PopupMessage(
          this->GetApplication(), this, 
          ks_("File Browser|Title|Error!"),
          "The file can not be removed!", 
          vtkKWMessageDialog::ErrorIcon | 
          vtkKWMessageDialog::InvokeAtPointer);
        delete[] indices;
        return 0;
        }
      isDir = 0;
      }
    this->FileList->GetWidget()->DeleteRow(indices[0]);
    int numrows = this->FileList->GetWidget()->GetNumberOfRows();
    if (numrows > 0)
      {
      this->FileList->GetWidget()->ClearSelection();
      if (indices[0] >= numrows)
        {
        this->FileList->GetWidget()->SelectRow(numrows - 1);
        }
      else
        {
        this->FileList->GetWidget()->SelectRow(indices[0]);
        }
      }
    this->InvokeFileDeletedCommand(fullname.c_str(), isDir);
    }
  else
    {
    delete [] indices;
    return 0;
    }
  delete [] indices;
  return 1;
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFileSelectedCommand(
  vtkObject *object, const char *method)
{
  this->SetObjectMethodCommand(&this->FileSelectedCommand, object, method);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::InvokeFileSelectedCommand(const char* path)
{
  if (this->FileSelectedCommand && *this->FileSelectedCommand)
    {
    this->Script("%s \"%s\"", this->FileSelectedCommand, 
                 vtksys::SystemTools::EscapeChars(
                   KWFileBrowser_GetUnixPath(path), 
                   KWFileBrowser_ESCAPE_CHARS).c_str());
    }

  this->InvokeEvent(
    vtkKWFileListTable::FileSelectionChangedEvent, (void*)path);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFileDeletedCommand(
  vtkObject *object, const char *method)
{
  this->SetObjectMethodCommand(&this->FileDeletedCommand, object, method);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::InvokeFileDeletedCommand(
  const char* path, int isDir)
{
  if (this->FileDeletedCommand && *this->FileDeletedCommand)
    {
    this->Script("%s \"%s\" %d", this->FileDeletedCommand, 
                 vtksys::SystemTools::EscapeChars(
                   KWFileBrowser_GetUnixPath(path), 
                   KWFileBrowser_ESCAPE_CHARS).c_str(), isDir);
    }
  
  this->InvokeEvent(
    vtkKWFileListTable::FileDeletedEvent, (void*)path);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFileDoubleClickedCommand(
  vtkObject *object, const char *method)
{
  this->SetObjectMethodCommand(
    &this->FileDoubleClickedCommand, object, method);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::InvokeFileDoubleClickedCommand(const char* path)
{
  if (this->FileDoubleClickedCommand && *this->FileDoubleClickedCommand)
    {
    this->Script("%s \"%s\"", this->FileDoubleClickedCommand, 
                 vtksys::SystemTools::EscapeChars(
                   KWFileBrowser_GetUnixPath(path), 
                   KWFileBrowser_ESCAPE_CHARS).c_str());
    }
  this->InvokeEvent(
    vtkKWFileListTable::FileDoubleClickedEvent, (void*)path);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFileRenamedCommand(
  vtkObject *object, const char *method)
{
  this->SetObjectMethodCommand(&this->FileRenamedCommand, object, method);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::InvokeFileRenamedCommand(const char* oldname,
                                                  const char* newname)
{
  if (this->FileRenamedCommand && *this->FileRenamedCommand)
    {
    this->Script("%s \"%s\" \"%s\"", this->FileRenamedCommand, 
                 vtksys::SystemTools::EscapeChars(
                   KWFileBrowser_GetUnixPath(oldname), 
                   KWFileBrowser_ESCAPE_CHARS).c_str(), 
                 vtksys::SystemTools::EscapeChars(
                   KWFileBrowser_GetUnixPath(newname), 
                   KWFileBrowser_ESCAPE_CHARS).c_str());
    }

  this->InvokeEvent(
    vtkKWFileListTable::FileRenamedEvent, (void*)oldname);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetFolderCreatedCommand(
  vtkObject *object, const char *method)
{
  this->SetObjectMethodCommand(&this->FolderCreatedCommand, object, method);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::InvokeFolderCreatedCommand(const char* filename)
{
  if (this->FolderCreatedCommand && *this->FolderCreatedCommand)
    {
    this->Script("%s \"%s\"", this->FolderCreatedCommand, 
                 vtksys::SystemTools::EscapeChars(
                   KWFileBrowser_GetUnixPath(filename), 
                   KWFileBrowser_ESCAPE_CHARS).c_str());
    }

  this->InvokeEvent(
    vtkKWFileListTable::FolderCreatedCommand, (void*)filename);
}

//---------------------------------------------------------------------------
void vtkKWFileListTable::ContextMenuCallback(
  int row, int vtkNotUsed(col), int x, int y)
{
  if (!this->IsCreated())
    {
    return;
    }

  if (!this->ContextMenu)
    {
    this->ContextMenu = vtkKWMenu::New();
    }
  if (!this->ContextMenu->IsCreated())
    {
    this->ContextMenu->SetParent(this->FileList->GetWidget());
    this->ContextMenu->Create();
    }
  this->ContextMenu->DeleteAllItems();

  // if there is no row clicked, popup context menu with new folder

  if (row < 0 || this->FileList->GetWidget()->GetNumberOfRows() <= 0)
    {
    this->PopulateContextMenu(0);
    }
  else
    {
    int num_sel = this->FileList->GetWidget()->GetNumberOfSelectedRows();
    if (num_sel<=0)
      {
      this->FileList->GetWidget()->SelectRow(row);
      }
    else
      {
      int* indices = new int[num_sel];
      this->FileList->GetWidget()->GetSelectedRows(indices);
      if (indices[0] != row)
        {
        this->FileList->GetWidget()->ClearSelection();
        this->FileList->GetWidget()->SelectRow(row);
        }
      delete [] indices;
      }
    this->PopulateContextMenu(1);
    }

  if (this->ContextMenu->GetNumberOfItems())
    {
    this->ContextMenu->PopUp(x, y);
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::CreateNewFolderCallback(
  const char* parentdir)
{
  if (!parentdir || !(*parentdir))
    {
    return;
    }
 
  vtksys_stl::string selfile = parentdir;
  vtkKWMultiColumnList *filelist = this->FileList->GetWidget();
  if (!vtksys::SystemTools::FileIsDirectory(selfile.c_str()))
    {
    vtkKWMessageDialog::PopupMessage(
      this->GetApplication(), this, 
      ks_("File Browser|Title|Error!"),
      k_("Please select a directory first."), 
      vtkKWMessageDialog::ErrorIcon | 
      vtkKWMessageDialog::InvokeAtPointer);
    return;
    }
  
  // Prompt the user for the name of the folder  
  vtkKWSimpleEntryDialog *dlg = vtkKWSimpleEntryDialog::New();
  dlg->SetParent(this);
  dlg->SetMasterWindow(this->GetParentTopLevel());
  dlg->SetDisplayPositionToPointer();
  dlg->SetTitle(
    ks_("File Browser|Dialog|Title|Create new folder"));
  dlg->SetStyleToOkCancel();
  dlg->Create();
  dlg->GetEntry()->GetLabel()->SetText(
    ks_("File Browser|Dialog|Folder name:"));
  dlg->SetText(
    ks_("File Browser|Dialog|Enter a name for this new folder"));
  
  int ok = dlg->Invoke();
  vtksys_stl::string foldername = dlg->GetEntry()->GetWidget()->GetValue();
  dlg->Delete();
  if (ok)
    {
    if (foldername.empty() || strcmp(foldername.c_str(), ".") == 0 ||
      strcmp(foldername.c_str(), "..") == 0)
      {
      vtkKWMessageDialog::PopupMessage(
        this->GetApplication(), this, 
        ks_("File Browser|Title|Error!"),
        "Please enter a valid folder name!", 
        vtkKWMessageDialog::ErrorIcon | 
        vtkKWMessageDialog::InvokeAtPointer);
      return;
      }
    }
  else
    {
    return;
    }    

  vtkDirectory *dir = vtkDirectory::New();
  if (!dir->Open(selfile.c_str()))
    {
    dir->Delete();
    vtkKWMessageDialog::PopupMessage(
      this->GetApplication(), this, 
      ks_("File Browser|Title|Error!"),
      k_("The selected directory can't be opened."), 
      vtkKWMessageDialog::ErrorIcon | 
      vtkKWMessageDialog::InvokeAtPointer);
    return;
    }
  
  vtksys_stl::string filename, fullname;

  // Check if the folder is already created

  for (int i = 0; i < dir->GetNumberOfFiles(); i++)
    {
    filename = dir->GetFile(i);
    if (strcmp(filename.c_str(), foldername.c_str()) == 0)
      {
      vtkKWMessageDialog::PopupMessage(
        this->GetApplication(), this, 
        ks_("File Browser|Title|Error!"),
        k_("The folder name already exists."), 
        vtkKWMessageDialog::ErrorIcon | 
        vtkKWMessageDialog::InvokeAtPointer);
      dir->Delete();
      return;
      }
    }
    
  // dd the new folder
  
  if (!KWFileBrowser_HasTrailingSlash(selfile.c_str()))
    {
    selfile += KWFileBrowser_PATH_SEPARATOR;
    }
  fullname = selfile + foldername;
  
  if (!dir->MakeDirectory(fullname.c_str()))
    {
    vtkKWMessageDialog::PopupMessage(
      this->GetApplication(), this, 
      ks_("File Browser|Title|Error!"),
      k_("The new directory can not be created."), 
      vtkKWMessageDialog::ErrorIcon | 
      vtkKWMessageDialog::InvokeAtPointer);
    dir->Delete();
    return;
    }
  
  dir->Delete();

  if (KWFileBrowser_ComparePath(selfile.c_str(),
                                       this->GetParentDirectory()))
    {
    this->InvokeFolderCreatedCommand(selfile.c_str());
    }
  else
    {
    this->InvokeFileDoubleClickedCommand(selfile.c_str());
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::PopulateContextMenu(
  int rowselected)
{
  char command[256];
  const char* callback = "CreateNewFolderCallback";
  const char* text = "Create new folder";

  if (!rowselected)
    {
    sprintf(command, "%s \"%s\"", callback,
            vtksys::SystemTools::EscapeChars(
              this->GetParentDirectory(), 
              KWFileBrowser_ESCAPE_CHARS).c_str());
    this->ContextMenu->AddCommand(text, this, command);
    }
  else if (vtksys::SystemTools::FileIsDirectory(this->GetSelectedFileName()))
    {
    sprintf(command, "%s \"%s\"", callback,
            vtksys::SystemTools::EscapeChars(
              this->GetSelectedFileName(), 
              KWFileBrowser_ESCAPE_CHARS).c_str());
    this->ContextMenu->AddCommand(text, this, command);
    }
  
  if (rowselected)
    {
#ifdef _WIN32
    if (vtksys::SystemTools::FileIsDirectory(this->GetSelectedFileName()))
      {
      this->ContextMenu->AddCommand(
        "Explore", this, "ExploreFileCallback");
      }
#endif
    // Rename file

    int index = this->ContextMenu->AddCommand(
      "Rename", this, "RenameFileCallback");
    this->ContextMenu->SetItemAccelerator(index, "F2");
    this->ContextMenu->SetBindingForItemAccelerator(
      index, this->ContextMenu->GetParent());
    
    // Delete file

    this->ContextMenu->AddCommand(
      "Delete", this, "RemoveSelectedFileCallback");
    }
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::UpdateEnableState()
{
  this->Superclass::UpdateEnableState();

  this->PropagateEnableState(this->FileList);
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetTableWidth(int width)
{
  if (this->FileList)
    {
    this->FileList->GetWidget()->SetWidth(width);
    }
}

//----------------------------------------------------------------------------
int vtkKWFileListTable::GetTableWidth()
{
  if (this->FileList)
    {
    return this->FileList->GetWidget()->GetWidth();
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::SetTableHeight(int height)
{
  if (this->FileList)
    {
    this->FileList->GetWidget()->SetHeight(height);
    }
}

//----------------------------------------------------------------------------
int vtkKWFileListTable::GetTableHeight()
{
  if (this->FileList)
    {
    return this->FileList->GetWidget()->GetHeight();
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkKWFileListTable::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "ParentDirectory: " 
     << (this->ParentDirectory?this->ParentDirectory:"none") 
     << endl;
  os << indent << "FilePattern: " 
     << (this->FilePattern?this->FilePattern:"none") 
     << endl;
  os << indent << "FileExtensions: " 
     << (this->FileExtensions?this->FileExtensions:"none") 
     << endl;
  os << indent << "ParentDirectory: " 
     << (this->ParentDirectory?this->ParentDirectory:"none") 
     << endl;
  os << indent << "FileSelectedCommand: " 
     << (this->FileSelectedCommand?this->FileSelectedCommand:"none") 
     << endl;
  os << indent << "FileDoubleClickedCommand: " 
     << (this->FileDoubleClickedCommand?this->FileDoubleClickedCommand:"none") 
     << endl;
  os << indent << "FileDeletedCommand: " 
     << (this->FileDeletedCommand?this->FileDeletedCommand:"none") 
     << endl;
  os << indent << "FileRenamedCommand: " 
     << (this->FileRenamedCommand?this->FileRenamedCommand:"none") 
     << endl;
}
