// -*- c++ -*-
// Generated by assa-genesis
//------------------------------------------------------------------------------
// $Id: Granule.cpp,v 1.59 2008/08/17 04:18:57 vlg Exp $
//------------------------------------------------------------------------------
//                            Granule.cpp
//------------------------------------------------------------------------------
//  Copyright (c) Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Wed Dec 31 23:18:34 2003
//
//------------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#   include "config.h"
#endif

#include <iostream>
#include <fstream>
#include <unistd.h>				// access(2), getcwd(3)
#include <stdlib.h>				// system(3)

#include <gtk/gtklabel.h>
#include <glibmm/ustring.h>
#include <gtkmm/main.h>
#include <gtkmm/rc.h>

#include <assa/CommonUtils.h>
#include <assa/Logger.h>
#include <assa/SigAction.h>
#include <assa/IniFile.h>

#include "Granule-main.h"
#include "Granule.h"
#include "GrappConf.h"
#include "MainWindow.h"

#ifdef IS_HILDON
#  include <hildonmm.h>
#endif

#if !defined(IS_WIN32)
#  include <libgen.h>			// basename(3), dirname(3)
#  include <pwd.h>				// getpwnam(2), getlogin(2)
#endif

// Static declarations

ASSA_DECL_SINGLETON(Granule);

static const int TIMEOUT = 2000;  // 2 seconds (1,000 mls = 1 sec).

template <> xmlExternalEntityLoader Granule::m_default_entity_loader = 0;

/*******************************************************************************
                          Member Functions
*******************************************************************************/
Granule::
Granule () :
	m_timeout        (0.5),
	m_secs_in_day    (86400),
	m_secs_in_week   (604800),
	m_dump_cardboxes ("no"),
	m_main_window    (NULL),
	m_kit            (NULL),
	m_async_adapter_enabled (false),
	m_async_timer    (0)
{
    add_opt ('g', "gtk-options",    &m_gtk_options );
	add_opt (0,   "secs-in-day",    &m_secs_in_day );
	add_opt (0,   "secs-in-week",   &m_secs_in_week);
	add_opt (0,   "dump-cardboxes", &m_dump_cardboxes);

    // ---Configuration---
    rm_opt ('n', "instance"     );
    rm_opt ('p', "port"         );

    // ---Process bookkeeping---
    rm_opt ('b', "daemon"       );
    rm_opt ('l', "pidfile"      );
    rm_opt ('L', "ommit-pidfile");

	m_ommit_pidfile = "yes";	// don't lock up on PID 

    /* By defauil disable all debugging
     */
    /* m_mask = GRAPP | DECK | GUITRACE | ASSA::ASSAERR; */
	m_mask = 0;
    m_log_file = "granule.log";
}

/**
 * Description: 
 *
 *  Read configuration from [options] secion of the configuration
 *  file, ~/.granule, and populate command-line options. 
 *  They might be overwritten by command-line arguments supplied with argv_.
 *  Call GenServer::init() to start initialization.
 *
 *  If the application configuration file, ~/.granule,
 *  is missing, copy the default from $GRAPPDATDIR.
 *  Gtkmm hasn't been initialized yet.
 *  
 *  WARNING: It is [options], not [Options] that GenServer parses!
 */
void 
Granule::
init (int* argc_, char* argv_[], const char* help_info_)
{
	string rcfile;
	std::string v;
	bool parse_default_rc = true;

#if !defined (IS_WIN32)            /* POSIX systems */
	m_config_file = "none";		/* But note, GrappConf relies on this value! */
	parse_args ((const char**) argv_);

	if (m_config_file != "none") {
		rcfile = ASSA::Utils::strenv (m_config_file.c_str ());
	}
	else {
		string defaultrc (GRAPPDATDIR);	     /* /usr/share/granule/granule */
#ifdef IS_DESKTOP
		defaultrc += "/granule-linux.conf";
#elif IS_HILDON
		defaultrc += "/granule-hildon.conf";
#elif IS_PDA
		defaultrc += "/granule-pda.conf";
#else
		defaultrc += "/granule-linux.conf";
#endif
		rcfile = ASSA::Utils::strenv ("~/.granule");

		if (::access (rcfile.c_str (), R_OK) < 0) {
			if (::access (defaultrc.c_str (), R_OK) == 0) {
				string cmd ("cp -p ");
				cmd += defaultrc + " " + rcfile;
				if (::system (cmd.c_str ()) < 0) {
					parse_default_rc = false;
				}
			}
			else {
				parse_default_rc = false;
			}
		}
	}
#else /* IS_WIN32 */
	rcfile = "granule.ini"; // same place where the binary is.
#endif

	/**
	 * Load [options] section of the configuration file.
	 */
	if (parse_default_rc) 
	{
		m_default_config_file = rcfile;
		ASSA::IniFile cfg (rcfile);
		if (cfg.load () == 0) {
			if (parse_config_file (cfg) < 0) {
				std::cerr << "Failed to parse rcfile : "
						  << get_opt_error () << std::endl;
				return;
			}
		}
	}

	ASSA::GenServer::init (argc_, argv_, help_info_);

	/** Fix alternative config file name if specified on command-line.
	 *  On a command line it may look like --config-file=~/.granule.whatever.
	 *  This is libassa bug!
	 */
	if (m_config_file != "none") {
		m_config_file = rcfile;
	}
	
	/*
	  TRACE    = 0x00001
	  APP      = 0x00002 (GRAPP)
	  USR1     = 0x00004 (GUITRACE)
	  USR2     = 0x00008 (DECK)
	  USR3     = 0x00010
	  ALL_APPS = 0x0001F
	  ERROR    = 0x00020
	  FORK     = 0x40000
	*/

	if (m_log_level >= 0) 
	{
		switch (m_log_level) 
		{
		case 6:	 m_mask = ALL;     break;    // Log message flood!
		case 5:	 m_mask = 0x3F;    break;    // ALL_APPS+ERRORS
		case 4:  m_mask = 0x40026; break;    // +FORK
		case 3:  m_mask = 0x26;    break;    // GRAPP+GUITRACE+ERRORS
		case 2:  m_mask = 0x25;    break;    // TRACE+GUITRACE+ERRORS
		case 1:	 m_mask = 0x22;    break;    // GRAPP+ERRORS
		case 0:	 m_mask = 0x0;     break;
		default: m_mask = ALL;	// >6
		}
	}
	LOGGER->enable_groups (m_mask);

	/** Install my own entity loader. 
	 *  This will let me redefine DTD URLs and support backward compatability.
	 */
	m_default_entity_loader = xmlGetExternalEntityLoader();
	xmlSetExternalEntityLoader (my_xml_entity_loader);

	/** Initialize XML parser
	 */
	xmlInitParser ();
	xmlIndentTreeOutput = 1;
}

Granule::
~Granule ()
{
    trace_with_mask("Granule::~Granule",GUITRACE);

	/** Cleanup XML parser
	 */
	xmlCleanupParser ();

	/** Remove log file.
	 */
	if (m_mask == 0) {
		::unlink (m_log_file.c_str ());
	}
	
	/** Disconnect from async timer.
	 *  Failing to do so causes core dumps on exit from libsigc-2.0's
	 *  sigc::trackable::remove_destroy_notify_callback ().
	 */
	if (async_adapter_enabled ()) {
		set_async_events_adapter (false);
	}
}

void
Granule::
dump_package_envs ()
{
	DL((APP,"\n= Environment variables set by automake: =\n"
		"\n"
		"\tGRAPPDATDIR..: \"%s\"\n"
		"\tGRAPPXMLDIR...: \"%s\"\n"
		"\tDATDIR.......: \"%s\"\n"
		"\tLOCALEDIR.....: \"%s\"\n"
		"\tPACKAGE.......: \"%s\"\n"
		"\tVERSION.......: \"%s\"\n"
#ifdef IS_HILDON
		"\tIS_HILDON.....: \"yes\"\n"
#else
		"\tIS_HILDON.....: \"no\"\n"
#endif
#ifdef IS_PDA
		"\tIS_PDA........: \"yes\"\n"
#else
		"\tIS_PDA........: \"no\"\n"
#endif
		"\n",
		GRAPPDATDIR, GRAPPXMLDIR, DATDIR, LOCALEDIR, PACKAGE, VERSION));
}

/* Rewrite the url of the granule DTDs to local files so we can
 * use the local DTDs instead of going out to the web.
 * 
 * Look for /etc/xml for backwards compatibility with old files for POSIX
 * systems and in the binary installation directories for Win32.
 */
xmlParserInputPtr
Granule::
my_xml_entity_loader (const char* url_, 
					  const char* id_,
					  xmlParserCtxtPtr ctxt_)
{
    trace_with_mask("Granule::my_xml_entity_loader",GUITRACE);

	if (url_) DL ((GRAPP,"URL requested: \"%s\"\n",   url_));
	if (id_ ) DL ((GRAPP,"ID  requested:= \"%s\"\n",  id_));

    if (strcmp (url_, "/etc/xml/granule/granule.dtd"              ) == 0 ||
		strcmp (url_, "file:///etc/xml/granule/granule.dtd"       ) == 0 ||
		strcmp (url_, "http://granule.sourceforge.net/granule.dtd") == 0) 
	{
		url_ = GRAPPXMLDIR "/granule.dtd";
    } 
	else 
		if (strcmp (url_, "/etc/xml/granule/cardfile.dtd"              ) == 0 ||
			strcmp (url_, "file:///etc/xml/granule/cardfile.dtd"       ) == 0 ||
			strcmp (url_, "http://granule.sourceforge.net/cardfile.dtd") == 0)
	{
        url_ = GRAPPXMLDIR "/cardfile.dtd";
    }

	DL ((GRAPP,"URL replaced: \"%s\"\n", url_));
    
    return m_default_entity_loader (url_, id_, ctxt_);
}

void
Granule::
init_service ()
{
    trace_with_mask("Granule::init_service",GUITRACE);

    // ASSA::Log::disable_timestamp ();
    int gtk_argc = 0;
    char** gtk_argv = NULL;

	DL((ALL,"gtk_options = \"%s\"\n",     m_gtk_options.c_str ()));
	DL((ALL,"log_stdout  = %s\n",         m_log_stdout.c_str ()));
	DL((ALL,"log_file    = \"%s\"\n",     m_log_file.c_str ()));
	DL((ALL,"log_size    = %d\n",         m_log_size));
	DL((ALL,"log_level   = \"%d\"\n",     m_log_level));
	DL((ALL,"mask        = 0x%X\n",       m_mask));
	DL((ALL,"dump_cardboxes = %s\n",      m_dump_cardboxes.c_str ()));

	/**
	 * Block SIGCHLD for a moment
	 */
#if !defined(IS_WIN32)
	ASSA::SigAction old_chld_act;
	struct sigaction* sa;
	old_chld_act.retrieve_action (SIGCHLD);
	sa = old_chld_act;
#endif
	
	/**
	 * ATTENTION:
	 *    Passing --g-fatal-warnings would kill Granule BEFORE
	 *    the first Gtk+ critical warning is written to stdout. 
     *    If you want to examine warnings, don't use this option.
	 */
    m_gtk_options = "granule " + m_gtk_options;
    CmdLineOpts::str_to_argv (m_gtk_options, gtk_argc, gtk_argv);

	DL ((ALL,"Cmd line args (%d):\n", gtk_argc));
	DL ((ALL,"-----------------------------\n"));
	for (int i=0; i<gtk_argc; i++) {
		DL ((ALL,"[%2d] = \"%s\"\n", i, gtk_argv[i]));
	}
	DL ((ALL,"-----------------------------\n"));

    m_kit = new Gtk::Main (&gtk_argc, &gtk_argv);

    CmdLineOpts::free_argv (gtk_argv);

	/** I suspect one of the Gnome libraries changes action of
	 *  SIGCHLD signal. Reset it to SIG_IGN.
	 */
#if !defined(IS_WIN32)
	ASSA::SigAction ignore (SIG_IGN);
	ignore.register_action (SIGCHLD, &old_chld_act);
#endif

    /** Load history from ~/.granule configuration file
	 */
    CONFIG->load_config (m_secs_in_day, m_secs_in_week);
	dump_package_envs ();

    m_main_window = new MainWindow ();
	m_main_window->init ();

	/** Set theme font
	 */
	Gtk::RC::parse_string(
			"style \"resource\" { font_name = \""
			+ CONFIG->fonts_db ().get_app_font ().to_string () +
			"\" } "
			"widget \"*\" style \"resource\"");

#ifdef IS_HILDON
	/**
	 * Initialize hildon-libsmm library.
	 * You may call this more than once.
	 */
	Hildon::init ();

	/** 
	 * "The Hildon::Program object is created by calling 
	 *  the static get_instance() method, which takes no parameters 
	 *  and returns a new Hildon::Program."
	 *
	 * "Call to get_instance() returns the Hildon::Program for the 
	 *  current process. The object is created on the first call."
	 */
	m_hildon_program = Hildon::Program::get_instance ();
	m_hildon_program->add_window (*m_main_window);

#endif /* IS_HILDON */

    DL((ASSA::APP,"Service has been initialized\n"));
}

/** Having a constant timer to run might place unnecessary
 *  strain on the battery life of N8x0. We use TIMEOUT=2 secs.
 *  here, but event that might be an overkill.
 *
 *  @param enable_ If true, enable adapter; disable if false.
 */
void
Granule::
set_async_events_adapter (bool enable_)
{
    trace_with_mask("Granule::enable_async_events_adapter",GUITRACE);

	if (!m_async_adapter_enabled) {
		if (enable_) {
			sigc::slot0<bool> tslot = sigc::mem_fun (*this, &Granule::timer_cb);
			m_tconn = Glib::signal_timeout().connect (tslot, TIMEOUT);
			m_async_timer = REACTOR->registerTimerHandler (this, 
														   TimeVal (1800.0),
														   "async-adapter");
			m_async_adapter_enabled = true;
		}
	}
	else {
		if (!enable_) {
			m_tconn.disconnect ();
			m_async_adapter_enabled = false;
		}
	}

	DL ((GRAPP,"Async events adapter is %s\n", 
		 (m_async_adapter_enabled ? "enabled" : "disabled")));
}

/**
 * Run Reactor's event loop.
 */
bool
Granule::
timer_cb ()
{
	ASSA::TimeVal tv (m_timeout);
    REACTOR->waitForEvents (&tv);
    return true;
}

/**
 * When Async Adapter Timer expires in 30 minutes,
 * we disable it. This way we save on the battery life of 
 * handhelds. Note that this comes only to be an issue when
 * the user iconizes Granule and leaves it running in a background.
 */
int
Granule::
handle_timeout (ASSA::TimerId tid_)
{
    trace_with_mask("Granule::handle_timeout",GUITRACE);

	if (m_async_timer == 0) {
		DL ((GRAPP, "Hmm, a runaway Async Adapter Timer?\n"));
		return 0;
	}

	if (m_async_timer != tid_) {
		DL ((GRAPP, "Hmm, unassigned Async Adapter Timer?\n"));
	}

	DL ((GRAPP,"Disable Async Adapter!\n"));

	m_tconn.disconnect ();
	m_async_adapter_enabled = false;

	REACTOR->removeTimerHandler (m_async_timer);
	m_async_timer = 0;

	return 0;
}


void
Granule::
process_events ()
{
    trace_with_mask("Granule::process_events",GUITRACE);

	m_main_window->show ();
	gtk_main ();

    DL((ASSA::APP,"MainWindow run() returns.\n"));

	m_main_window->close_cb ();
	REACTOR->stopReactor ();

    DL((ASSA::APP,"Service stopped!\n"));
}

void
Granule::
fatal_signal_hook ()
{
    trace_with_mask("Granule::fatal_signal_hook",GUITRACE);

	DL ((ASSA::ALL,"Application terminated with 'kill'\n"));
	gtk_exit (0);
}

/**===========================================================================**
 * Common utilities that are used everywhere                                  **
 **===========================================================================**
 */

/** 
 * Stip Pango markup. The caller owns the memory chunk returned.
 * @return result returned by pango_parse_markup or NULL on error.
 */
gchar* 
Granule::
strip_pango_markup (const char* src_)
{
	GError* error = NULL;
	gchar* result = NULL;

	if (src_ == NULL || ::strlen (src_) == 0) {
		result = NULL;
	}
	else {
		if (!pango_parse_markup (src_, -1, 0, NULL, &result, NULL, &error)) 
		{
			if (error != NULL) {
				DL ((GRAPP,"Error: pango_parse_markup() = %s\n", 
					 error->message));
				g_error_free (error);
			}
			result = NULL;
		}
	}

	return result;
}

bool
Granule::
check_markup (const Glib::ustring& str_)
{
	return (pango_parse_markup (str_.c_str (), -1, 0, NULL, NULL, NULL, NULL));
}

/**
 * For multiline entries (starting with v 1.1.5), 
 * remove part of the string following first newline character.
 * But first, strip possible Pango markup. Markup can span across
 * multiple lines and interpreting only first line might result
 * in invalid syntax.
 */
Glib::ustring
Granule::
trim_multiline (const Glib::ustring& orig_text_)
{	
	Glib::ustring text_;
	gchar* pc = Granule::strip_pango_markup (orig_text_.c_str ());

	if (pc == NULL) {
		return text_;
	}

	text_ = pc;
	g_free (pc);

	Glib::ustring::size_type idx  = text_.find_first_of ("\n");
	Glib::ustring result (text_);

	if (idx != std::string::npos) {
		result.replace (idx, text_.size (), "");
	}

	return result;
}

void
Granule::
remove_common_prefixes (gchar*& src_)
{
	static const gchar* prefixes [] = { "to ", "a ", "an ", NULL };

	if (src_ == NULL) {
		return;
	}

	for (int i = 0; prefixes [i]; i++) 
	{
		if (g_str_has_prefix (src_, prefixes [i])) 
		{
			gchar* chp = g_strdup (src_ + strlen (prefixes [i]));
			g_free (src_);
			src_ = chp;
			break;
		}
	}
}

xmlChar*
Granule::
get_node_by_xpath (xmlDocPtr parser_, const char* xpath_)
{
	xmlXPathContextPtr context = NULL;
	xmlXPathObjectPtr  result = NULL;
	xmlNodeSet* nodeset = NULL;
	xmlChar* keyword = NULL;

	context = xmlXPathNewContext (parser_);
	if (context == NULL) {
		DL ((ASSA::ASSAERR,"Failed to create context.\n"));
		goto done;
	}
	
	result = xmlXPathEval (BAD_CAST xpath_, context);

	if (xmlXPathNodeSetIsEmpty (result->nodesetval)) {

//		DL ((ASSA::ASSAERR,"Empty nodeset for xpath \"%s\"\n", xpath_));
		goto done;
	}
	nodeset = result->nodesetval;

	/** We expect only one node
	 */
	keyword = xmlNodeListGetString (parser_, 
									nodeset->nodeTab[0]->xmlChildrenNode, 
									1);
  done:
	if (result) {
		xmlXPathFreeObject (result);
	}
	if (context) {
		xmlXPathFreeContext (context);
	}

	return keyword;
}

/**
 * Method create_from_file () throws an exception if filename_ is invalid. 
 * One way to avoid the crash is to check for file existence before 
 * calling this method:
 *
 *   if (Glib::file_test (file_path, Glib::FILE_TEST_EXISTS)) {
 *       ixbuf_ref = Granule::create_from_file (file_path);
 *   }
 */
Glib::RefPtr<Gdk::Pixbuf>
Granule::
create_from_file (const std::string& filename_, int width_, int height_)
{
	Glib::RefPtr<Gdk::Pixbuf> pixbuf;

#ifdef GLIBMM_EXCEPTIONS_ENABLED

	if (width_ > 0 && height_ > 0) {
		pixbuf = Gdk::Pixbuf::create_from_file (filename_, 
												width_, height_, false);
	}
	else {
		pixbuf = Gdk::Pixbuf::create_from_file (filename_);
	}

#else
	std::auto_ptr<Glib::Error> err;

	if (width_ > 0 && height_ > 0) {
		pixbuf = Gdk::Pixbuf::create_from_file (filename_, width_, height_, 
												false, err);
	}
	else {
		pixbuf = Gdk::Pixbuf::create_from_file (filename_, err);
	}

	if (!pixbuf) {
		DL ((GRAPP,"Gdk::Pixbuf::create_from_file (%s) failed.\n", 
			 filename_.c_str ()));
		if (err.get ()) {
			DL ((GRAPP,"Reason: %s\n", err->what ().c_str ()));
		}
	}
#endif
	return (pixbuf);
}


std::string
Granule::
locale_from_utf8 (const Glib::ustring& utf8_string_)
{
	std::string result;
#ifdef GLIBMM_EXCEPTIONS_ENABLED
	try {
		result = Glib::locale_from_utf8 (utf8_string_);
	}
	catch (Glib::ConvertError& e) {
		/* cannot convert */
	}
#else
	std::auto_ptr<Glib::Error> err;
	result = Glib::locale_from_utf8 (utf8_string_, err);
#endif
	return result;
}

Glib::ustring
Granule::
locale_to_utf8 (const std::string& opsys_string_)
{
#ifdef GLIBMM_EXCEPTIONS_ENABLED
	return Glib::locale_to_utf8 (opsys_string_);
#else
	std::auto_ptr<Glib::Error> err;
	return (Glib::locale_to_utf8 (opsys_string_, err));
#endif
}

Glib::ustring
Granule::
filename_to_utf8 (const std::string& opsys_string_)
{
#ifdef GLIBMM_EXCEPTIONS_ENABLED

	try {
		return Glib::filename_to_utf8 (opsys_string_);
	}
	catch (const Glib::ConvertError& error)	{
		if (error.code() != Glib::ConvertError::ILLEGAL_SEQUENCE) {
			throw;
		}
		return Glib::ustring ();
	}

#else

	std::auto_ptr<Glib::Error> err;
	return Glib::filename_to_utf8 (opsys_string_, err);

#endif
}
	
std::string
Granule::
filename_from_utf8 (const Glib::ustring& utf8_string_)
{
#ifdef GLIBMM_EXCEPTIONS_ENABLED

	try {
		return Glib::filename_from_utf8 (utf8_string_);
	}
	catch (const Glib::ConvertError&) {
		return std::string();
	}

#else

	std::auto_ptr<Glib::Error> err;
	return Glib::filename_from_utf8 (utf8_string_, err);

#endif
}

void
Granule::
hide_fcd_gtk_labels (GtkContainer* container_, int& counter_)
{
#ifdef IS_PDA 
	const gchar* blabel;
	GList* child;
	GList* child2;

	if (counter_ > 2) {
		return;
	}
	child = gtk_container_get_children(container_);
	while (child != NULL) 
	{
		if (GTK_IS_LABEL (child->data)) {
			blabel = gtk_label_get_text (GTK_LABEL (child->data));
			if (blabel != NULL && (!strcmp (blabel, "Add") || 
								   !strcmp (blabel, "Remove")))
				{
					gtk_widget_hide (GTK_WIDGET (child->data));
					++counter_;
					return;
				}
		}
		if (GTK_IS_CONTAINER (child->data)) {
			hide_fcd_gtk_labels (GTK_CONTAINER (child->data), counter_);
		}
		child = g_list_next(child);
	}
#endif
}

/** 
 * In some older versions of Gtk+, set_angle() is not available.
 */
void
Granule::
rotate_label (Gtk::Label* label_, double angle_, const char* alt_text_)
{
#ifdef IS_PDA
#if (GTKMM_MINOR_VERSION < 6)
	label_->set_text (alt_text_);
#else
	label_->set_angle (angle_);
#endif
#else
	label_->set_angle (angle_);
#endif
}

/** 
 * Save parent window position. Move child to that new position.
 **/
void
Granule::
move_child_to_parent_position (Gtk::Window& parent_, Gtk::Window& child_) 
{
    trace_with_mask("Granule::move_child_to_parent_position",GUITRACE);

#if !defined (IS_HILDON)
	int x, y;

	parent_.get_position (x, y);
	child_.move (x, y);
	CONFIG->set_win_position (x, y);
#endif

	parent_.hide ();
	child_.show_all ();
}

/** 
 * If child moved, save new position, and move parent's window there.
 **/
void
Granule::
move_parent_to_child_position (Gtk::Window& parent_, Gtk::Window& child_)
{
    trace_with_mask("Granule::move_parent_to_child_position",GUITRACE);

#if !defined (IS_HILDON)
	int x, y;

	child_.get_position (x, y);
	parent_.move (x, y);
	CONFIG->set_win_position (x, y);
#endif

	child_.hide ();
	parent_.show_all ();
}

std::string
Granule::
calculate_relative_path (const std::string& abs_pname_,
						 const std::string& abs_fname_)
{
	trace_with_mask("Granule::calculate_relative_path", GUITRACE);

	if (abs_pname_ == abs_fname_) {
		return (".");
	}

	/** If filename path is empty, we return $cwd.
	 */
	if (abs_fname_.length () == 0) {
		return (".");
	}

	/** We might be given relative pathname - this is the
	 *  caller problem, but at least we return something relative.
	 */
	if (! g_path_is_absolute (abs_pname_.c_str ())) {
		string result (".");
		result += ASSA_DIR_SEPARATOR_S;
		return (result + Granule::get_basename (abs_fname_.c_str ()));
	}

	std::string::iterator ap_iter;
	std::string::iterator af_iter;
	std::string::iterator slash;

	std::string ap (abs_pname_);
	std::string af (abs_fname_);

	DL ((GRAPP,"abs_pname = \"%s\"\n", ap.c_str ()));
	DL ((GRAPP,"abs_fname = \"%s\"\n", af.c_str ()));

	if (ap [ap.size ()-1] != ASSA_DIR_SEPARATOR) {
		ap += ASSA_DIR_SEPARATOR_S;
	}

	/* Remove prefix similar in both string (if any)
	 */
	ap_iter = ap.begin ();
	af_iter = af.begin ();

	while (ap_iter != ap.end () || af_iter != af.end ()) {
		if (*ap_iter != *af_iter) {
			break;
		}
		ap_iter++, af_iter++;
	}
	ap.erase (ap.begin (), ap_iter);
	af.erase (af.begin (), af_iter);

	/**
	 * In AP string, replace directory names between consequtive '/' with
	 * '..'. Then, remove leading '/'.
	 */
	ap_iter = ap.begin ();
	slash = ap_iter;
	do {
		while (*slash != ASSA_DIR_SEPARATOR && slash != ap.end ()) { 
			slash++;
		}
		if (slash == ap.end ()) {
			break;
		}
		ap.replace (ap_iter, slash, "..");
		ap_iter += 3;
		slash = ap_iter;
	} while (ap_iter != ap.end ());

	/* Add AF to AP and return result 
	 */
	std::string result = ap + af;
	DL ((GRAPP, "result = \"%s\"\n", result.c_str ()));
	return (result);
}

std::string
Granule::
calculate_absolute_path (const std::string& abs_pname_,
						 const std::string& rel_fname_)
{
	std::string::size_type idx;

	std::string ap (abs_pname_);
	std::string rf (rel_fname_);

	/** For each '..' found, remove it and its preceding directory name
		(i.e. /Words/Travel/../.. => /Words/..).
	*/
	while (rf.find (".." ASSA_DIR_SEPARATOR_S) != std::string::npos) 
	{
		rf.erase (rf.begin (), rf.begin ()+3);
		idx = ap.rfind (ASSA_DIR_SEPARATOR);
		if (idx != std::string::npos) {
			ap.erase (idx);
		}
	}

	/** If the first character is not '/', then you don't have an absolute
		path as you migh have expected.
	*/
	return (ap + ASSA_DIR_SEPARATOR_S + rf);
}

/**
 * WARNING:
 *
 * From man dirname(3):
 *
 *   "Both dirname and basename may modify the contents of path,
 *    so if you need to  preserve  the  pathname string,
 *    copies should be passed to these functions.
 *
 *    Furthermore, dirname and basename may return  pointers  to
 *    statically  allocated  memory which may overwritten by subsequent
 *     calls."
 */

/**
 * Get the name of the file without any leading directory components. 
 */
string
Granule::
get_basename (const char* path_)
{
	char buf [PATH_MAX];
	gchar* p;
	string result;

	if (path_ == NULL) {
		return result;
	}
	
	::strncpy (buf, path_, PATH_MAX-1);
	p = g_path_get_basename (buf);
	result = p;
	g_free (p);
	return result;
}

/** Get the directory components of a file name. 
 *  If the file name has no directory components "." is returned.
 */
string
Granule::
get_dirname (const char* path_)
{
	char buf [PATH_MAX];
	gchar* p;
	string result;

	if (path_ == NULL) {
		return result;
	}
	
	::strncpy (buf, path_, PATH_MAX-1);
	p = g_path_get_dirname (buf);
	result = p;
	g_free (p);

	return result;
}

string
Granule::
get_current_working_dir ()
{
	string result;

#ifdef GRAPP_DEPRICATED
    char* cptr = new char [PATH_MAX+1];
    Assure_exit (getcwd (cptr, PATH_MAX) != NULL);
    result = cptr;
    delete [] cptr;
	return result;
#endif
	result = Glib::get_current_dir ();
	return result;
}

string
Granule::
find_pixmap_path (const char* pixmap_)
{
	std::string result;

#if defined (IS_WIN32)
	result  = "../share/granule/pixmaps/";
	result += pixmap_;
#else
	result  = GRAPPDATDIR;
	result += "pixmaps/";
	result += pixmap_;
#endif

	DL ((GRAPP,"pixmap path: \"%s\"\n", result.c_str ()));

	return result;
}

const char*
Granule::
quality_answer_str (AnswerQuality qa_)
{
	static char* result;

	switch (qa_) {
	case ANSWER_BAD_BLACKOUT: result = "BAD (Complete blackout)";       break;
	case ANSWER_BAD_PARTIAL:  result = "BAD (Can recall correct)";      break;
	case ANSWER_BAD_RECALL:   result = "BAD ( Can easealy recalled correct)"; 
		break;
	case ANSWER_OK_DIFFICULT: result = "OK (Recalled with difficulty)"; break;
	case ANSWER_OK_HESITANT:  result = "OK (Recalled with hesitation)"; break;
	case ANSWER_OK:           result = "OK (Perfect recall)";           break;
	default:
		result = "ERR (Unexpected answer!!!)";
	}
	return result;
}

bool
Granule::
test_file_exists (const Glib::ustring& fname_, 
				  const Glib::ustring& basepath_,
				  Glib::ustring& file_path_)
{
	file_path_ = "";

	if (fname_.size () > 0) {
		if (Glib::path_is_absolute (fname_.c_str ())) {
			file_path_ = fname_;
		}
		else {
			file_path_ = basepath_;
			file_path_ += G_DIR_SEPARATOR_S + fname_;
		}
		return (Glib::file_test (file_path_, Glib::FILE_TEST_EXISTS));
	}

	DL ((GRAPP,"Would not test an empty filename!\n"));
	return false;
}
