@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-association-protocol: List the supported ECtrans Protocols
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;

my %opt = ( gateway => undef, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  gateway=s
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $protocolName = $ARGV[0];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# If no Gateway is specified then use the default Gateway
$opt{gateway} = $controlChannel->getGatewayName()->result if ( not $opt{gateway} );

if ( not($protocolName) ) {

	# Get the list of protocols
	my $protocols = $controlChannel->getAssociationProtocolList( $token, $opt{gateway} );

	# Display the information for each protocol
	foreach $protocol ( $protocols->valueof('//getAssociationProtocolListResponse/return') ) {
		printf "%-20s %-20s %-10s %s\n", $protocol->{name}, $protocol->{module}, ( $protocol->{active} eq 'true' ? 'active' : 'not-active' ),
		  $protocol->{comment};
	}
}
else {

	# Get the detail for the specified protocolName
	my $protocol = $controlChannel->getAssociationProtocol( $token, $opt{gateway}, $protocolName )->valueof('//getAssociationProtocolResponse/return');
	print "     Name: " . $protocol->{name} . "\n";
	print "Shortname: " . $protocol->{module} . ( $protocol->{activeModule} eq 'true' ? '' : ' (disabled)' ) . "\n";
	print "   Active: " . ( $protocol->{active} eq 'true' ? 'yes' : 'no' ) . "\n";
	print "  Comment: " . $protocol->{comment} . "\n";
	print "  Archive: " . $protocol->{archive} . "\n";
	print "    Value: " . $protocol->{value} . "\n";
}

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-association-protocol - List the supported ECtrans Protocols

=head1 SYNOPSIS

B<ecaccess-association-protocol -version|-help|-manual>

B<ecaccess-association-protocol [-debug] [-gateway> I<name>B<] [>I<protocol-name>B<]>

=head1 DESCRIPTION

List all the ECtrans Protocols supported. When a I<protocol-name> is specified then
the details for this Protocol are displayed.

=head1 ARGUMENTS

=over 8

=item I<protocol-name> (optional)

The name of the ECtrans Protocol to retrieve the details.

=back

=head1 OPTIONS

=over 8

=item B<-gateway> I<name>

This is the name of the ECaccess Gateway where to list the ECtrans Protocols supported.
It is by default the Gateway you are connected to. In order to get the name
of your current Gateway you can use the B<ecaccess-gateway-name> command. When
using the commands at ECMWF the default Gateway is always "ecaccess.ecmwf.int".

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-association-protocol>

List all the ECtrans Protocols supported on your default Gateway.

B<ecaccess-association-protocol -gateway> I<ecaccess.ecmwf.int> I<genericFtp>

List the details/options of the Protocol I<genericFtp> on the I<ecaccess.ecmwf.int> Gateway.

=head1 SEE ALSO

B<ecaccess-association-delete>, B<ecaccess-association-get>, B<ecaccess-association-list>,
B<ecaccess-association-put> and B<ecaccess>.

=cut

__END__
:endofperl
