/*******************************************************************************

  Intel Data Center Bridging (DCB) Software
  Copyright(c) 2007-2009 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  e1000-eedc Mailing List <e1000-eedc@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "dcb_osdep.h"
#include "ports.h"
#include "tlv.h"
#include "l2_packet.h"
#include "states.h"
#include "dcb_types.h"
#include "dcb_protocol.h"
#include "messages.h"

extern struct unpacked_tlv *Type0; /* Declared in agent */

boolean_t mibConstrInfoLLDPDU(struct port *port)
{
	struct l2_ethhdr eth;
	u8  own_addr[ETH_ALEN];
	u32 fb_offset = 0;
	u32 datasize = 0;
	struct packed_tlv *ptlv =  NULL;

	memcpy(eth.h_dest, multi_cast_source, ETH_ALEN);
	l2_packet_get_own_src_addr(port->l2,(u8 *)&own_addr);
	memcpy(eth.h_source, &own_addr, ETH_ALEN);
	eth.h_proto = htons(ETH_P_LLDP);
	port->tx.frameout =  (u8 *)malloc(ETH_FRAME_LEN);
	if (port->tx.frameout == NULL) {
		printf("InfoLLDPDU: Failed to malloc frame buffer \n");
		return FALSE;
	}
	memset(port->tx.frameout,0,ETH_FRAME_LEN);
	memcpy(port->tx.frameout, (void *)&eth, sizeof(struct l2_ethhdr));
	fb_offset += sizeof(struct l2_ethhdr);

	/* Load Type1 */
	ptlv =  pack_tlv(port->tlvs.chassis);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv =  free_pkd_tlv(ptlv);

	/* Load Type2 */
	ptlv =  pack_tlv(port->tlvs.portid);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv =  free_pkd_tlv(ptlv);

	/* Load Type3 */
	ptlv =  pack_tlv(port->tlvs.ttl);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv =  free_pkd_tlv(ptlv);

	if (port->dcbx_st == dcbx_subtype2) {
		/* Load Type127 - dcbx subtype 2*/
		if (tlv_ok(port->tlvs.dcbx2)) {
			ptlv =  pack_tlv(port->tlvs.dcbx2);
			if (!ptlv || ((ptlv->size+fb_offset) >
				ETH_MAX_DATA_LEN))
				goto error;
			memcpy(port->tx.frameout+fb_offset,
				ptlv->tlv, ptlv->size);
			datasize += ptlv->size;
			fb_offset += ptlv->size;
			ptlv =  free_pkd_tlv(ptlv);
		}
	} else {
		/* Load Type127 - dcbx subtype1 */
		if (tlv_ok(port->tlvs.dcbx1)) {
			ptlv =  pack_tlv(port->tlvs.dcbx1);
			if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
				goto error;
			memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
			datasize += ptlv->size;
			fb_offset += ptlv->size;
			ptlv =  free_pkd_tlv(ptlv);
		}
	}

	/* Load Type0 */
	ptlv =  pack_tlv(Type0);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv =  free_pkd_tlv(ptlv);

	if (datasize < ETH_MIN_DATA_LEN)
		port->tx.sizeout = ETH_MIN_PKT_LEN;
	else
		port->tx.sizeout = fb_offset;

	return TRUE;

error:
	ptlv = free_pkd_tlv(ptlv);
	if (port->tx.frameout)
		free(port->tx.frameout);
	port->tx.frameout = NULL;
	printf("InfoLLDPDU: packed TLV too large for tx frame\n");
	return FALSE;
}

void txInitializeLLDP(struct port *port)
{
	port->tx.localChange = 0;
	port->stats.statsFramesOutTotal = 0;
	port->timers.reinitDelay   = REINIT_DELAY;
	port->timers.msgTxHold     = DEFAULT_TX_HOLD;
	port->timers.msgTxInterval = FASTSTART_TX_INTERVAL;
	port->timers.txDelay       = FASTSTART_TX_DELAY;
	port->timers.txShutdownWhile = 0;
	port->timers.txDelayWhile = 0;
	l2_packet_get_port_state(port->l2, (u8 *)&(port->portEnabled));
	return;
}

boolean_t mibConstrShutdownLLDPDU(struct port *port)
{
	struct l2_ethhdr eth;
	u8  own_addr[ETH_ALEN];
	u32 fb_offset = 0;
	u32 datasize = 0;
	struct packed_tlv *ptlv =  NULL;

	memcpy(eth.h_dest, multi_cast_source, ETH_ALEN);
	l2_packet_get_own_src_addr(port->l2,(u8 *)&own_addr);
	memcpy(eth.h_source, &own_addr, ETH_ALEN);
	eth.h_proto = htons(ETH_P_LLDP);
	port->tx.frameout =  (u8 *)malloc(ETH_FRAME_LEN);
	if (port->tx.frameout == NULL) {
		printf("ShutdownLLDPDU: Failed to malloc frame buffer \n");
		return FALSE;
	}
	memset(port->tx.frameout,0,ETH_FRAME_LEN);
	memcpy(port->tx.frameout, (void *)&eth, sizeof(struct l2_ethhdr));
	fb_offset += sizeof(struct l2_ethhdr);

	/* Load Type1 */
	ptlv =  pack_tlv(port->tlvs.chassis);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv =  free_pkd_tlv(ptlv);

	/* Load Type2 */
	ptlv =  pack_tlv(port->tlvs.portid);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv =  free_pkd_tlv(ptlv);

	/* Load Type3 */
	u16 tmpTTL = 0;
	memcpy(port->tlvs.ttl->info, &tmpTTL, port->tlvs.ttl->length);
	ptlv =  pack_tlv(port->tlvs.ttl);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv = free_pkd_tlv(ptlv);

	/* Load Type0 */
	ptlv = pack_tlv(Type0);
	if (!ptlv || ((ptlv->size+fb_offset) > ETH_MAX_DATA_LEN))
		goto error;
	memcpy(port->tx.frameout+fb_offset, ptlv->tlv, ptlv->size);
	datasize += ptlv->size;
	fb_offset += ptlv->size;
	ptlv = free_pkd_tlv(ptlv);

	if (datasize < ETH_MIN_DATA_LEN) {
		port->tx.sizeout = ETH_MIN_PKT_LEN;
	} else {
		port->tx.sizeout = fb_offset;
	}
	return TRUE;

error:
	ptlv = free_pkd_tlv(ptlv);
	if (port->tx.frameout)
		free(port->tx.frameout);
	port->tx.frameout = NULL;
	printf("InfoLLDPDU: packed TLV too large for tx frame\n");
	return FALSE;
}

u8 txFrame(struct port *port)
{

	int status = 0;

	status = l2_packet_send(port->l2, (u8 *)&multi_cast_source,
		htons(ETH_P_LLDP),port->tx.frameout,port->tx.sizeout);
	free(port->tx.frameout);
	port->tx.frameout = NULL;
	port->stats.statsFramesOutTotal++;
	if (port->stats.statsFramesOutTotal == FASTSTART_TX_COUNT) {
		/* We sent the fast start transmissions */
		port->timers.msgTxInterval = DEFAULT_TX_INTERVAL;
		port->timers.txDelay       = DEFAULT_TX_DELAY;
	}

	return 0;
}


void run_tx_sm(struct port *port, int update_timers)
{
	if (update_timers == TRUE)
		update_tx_timers(port);

	set_tx_state(port);
	do {
		switch(port->tx.state) {
		case TX_LLDP_INITIALIZE:
			txInitializeLLDP(port);
			break;
		case TX_IDLE:
			process_tx_idle(port);
			break;
		case TX_SHUTDOWN_FRAME:
			process_tx_shutdown_frame(port);
			break;
		case TX_INFO_FRAME:
			process_tx_info_frame(port);
			break;
		default:
			printf("ERROR The TX State Machine is broken!\n");
			log_message(MSG_ERR_TX_SM_INVALID, "%s", port->ifname);
		}
	} while (set_tx_state(port) == TRUE);

	return;
}

boolean_t set_tx_state(struct port *port)
{
	if ((port->portEnabled == FALSE) && (port->prevPortEnabled == TRUE)) {
		printf("set_tx_state: port was disabled\n");
		tx_change_state(port, TX_LLDP_INITIALIZE);
	}
	port->prevPortEnabled = port->portEnabled;

	switch (port->tx.state) {
	case TX_LLDP_INITIALIZE:
		if ((port->adminStatus == enabledRxTx) ||
			(port->adminStatus == enabledTxOnly)) {
			tx_change_state(port, TX_IDLE);
			return TRUE;
		}
		return FALSE;
	case TX_IDLE:
		if ((port->adminStatus == disabled) ||
			(port->adminStatus == enabledRxOnly)) {
			tx_change_state(port, TX_SHUTDOWN_FRAME);
			return TRUE;
		}
		if ((port->timers.txDelayWhile == 0) &&
			((port->timers.txTTR == 0) ||
			(port->tx.localChange))) {
			tx_change_state(port, TX_INFO_FRAME);
			return TRUE;
		}
		return FALSE;
	case TX_SHUTDOWN_FRAME:
		if (port->timers.txShutdownWhile == 0) {
			printf("set_tx_state; TX_SHUTDOWN_FRAME!\n");
			tx_change_state(port, TX_LLDP_INITIALIZE);
			return TRUE;
		}
		return FALSE;
	case TX_INFO_FRAME:
		tx_change_state(port, TX_IDLE);
		return TRUE;
	default:
		printf("ERROR: The TX State Machine is broken!\n");
		log_message(MSG_ERR_TX_SM_INVALID, "%s", port->ifname);
		return FALSE;
	}
}

void process_tx_idle(struct port *port)
{
	return;
}

void process_tx_shutdown_frame(struct port *port)
{
	if (mibConstrShutdownLLDPDU(port)) {
		txFrame(port);
	}
	port->timers.txShutdownWhile = port->timers.reinitDelay;
	return;
}

void process_tx_info_frame(struct port *port) {
	if (mibConstrInfoLLDPDU(port)) {
		txFrame(port);
	}
	return;
}

void	update_tx_timers(struct port *port)
{
	if (port->timers.txShutdownWhile)
		port->timers.txShutdownWhile--;

	if (port->timers.txDelayWhile)
		port->timers.txDelayWhile--;

	if (port->timers.txTTR)
		port->timers.txTTR--;
	return;
}

void tx_change_state(struct port *port, u8 newstate)
{
	u32 tmpTTL = 0;

	switch(newstate) {
	case TX_LLDP_INITIALIZE:
		if ((port->tx.state != TX_SHUTDOWN_FRAME) &&
			port->portEnabled) {
			printf("tx_change_state: TX_LLDP_INITIALIZE \n");
			assert(port->portEnabled);
			assert(port->tx.state == TX_SHUTDOWN_FRAME);
		}
		break;
	case TX_IDLE:
		if (!(port->tx.state == TX_LLDP_INITIALIZE ||
			port->tx.state == TX_INFO_FRAME)) {
			assert(port->tx.state == TX_LLDP_INITIALIZE);
			assert(port->tx.state == TX_INFO_FRAME);
		}
		tmpTTL = DEFAULT_TX_INTERVAL * port->timers.msgTxHold;
		if (tmpTTL > 65535) {
			port->tx.txTTL = htons(65535);
		} else {
			port->tx.txTTL = htons((u16)tmpTTL);
		}
		if (port->tlvs.ttl) {
			memcpy(port->tlvs.ttl->info,
				&(port->tx.txTTL), port->tlvs.ttl->length);
		}
		tmpTTL = 0;
		port->timers.txTTR = port->timers.msgTxInterval;
		port->tx.localChange = FALSE;
		port->timers.txDelayWhile = port->timers.txDelay;
		break;
	case TX_SHUTDOWN_FRAME:
	case TX_INFO_FRAME:
		assert(port->tx.state == TX_IDLE);
		break;
	default:
		printf("ERROR: tx_change_state:  default\n");
		log_message(MSG_ERR_TX_SM_INVALID, "%s", port->ifname);
	}

	port->tx.state = newstate;
	return;
}

void process_somethingChangedLocal(char *ifname, boolean_t NewFeatureTLVs)
{
	struct port *port = porthead;
	boolean_t   success;

	while (port != NULL) {
		if (strncmp(ifname, port->ifname,
			MAX_DEVICE_NAME_LEN)) {
			port = port->next;
			continue;
		}
		/* Need to improve this. Perhaps use tlv masks
		 * to add/delete tlvs from TX.
		 * Curently pg, pfc and fcoe tlvs creation
		 * based on Advertise flag in store. */

		if (port->tlvs.control != NULL)
			port->tlvs.control = free_unpkd_tlv(
				port->tlvs.control);
		port->tlvs.control = bld_dcbx_ctrl_tlv(port);

		if (port->tlvs.control == NULL) {
			printf("process_somethingChangedLocal: "
				"bld_dcbx_ctrl_tlv failed\n");
			assert(port->tlvs.control);
			break;
		}

		if (NewFeatureTLVs) {
			if (port->dcbx_st == dcbx_subtype2) {
				if (port->tlvs.pg2 != NULL)
					port->tlvs.pg2 =free_unpkd_tlv(
						port->tlvs.pg2);
				port->tlvs.pg2 = bld_dcbx2_pg_tlv(port,
					&success);
				if (!success) {
					printf("bld_dcbx2_pg_tlv: failed\n");
					break;
				}
			} else {
				if (port->tlvs.pg1 != NULL)
					port->tlvs.pg1 = free_unpkd_tlv(
						port->tlvs.pg1);
				port->tlvs.pg1 =
					bld_dcbx1_pg_tlv(port, &success);
				if (!success) {
					printf("bld_dcbx1_pg_tlv: failed\n");
					break;
				}
			}

			if (port->dcbx_st == dcbx_subtype2) {
				if (port->tlvs.pfc2 != NULL)
					port->tlvs.pfc2 = 
						free_unpkd_tlv(
						port->tlvs.pfc2);
				port->tlvs.pfc2 =
					bld_dcbx2_pfc_tlv(port,
						&success);
				if (!success) {
					printf("bld_dcbx2_pfc_tlv: failed\n");
					break;
				}
			} else {
				if (port->tlvs.pfc1 != NULL)
					port->tlvs.pfc1 = free_unpkd_tlv(
						port->tlvs.pfc1);
				port->tlvs.pfc1 = 
					bld_dcbx1_pfc_tlv(port, &success);
				if (!success) {
					printf("bld_dcbx1_pfc_tlv: failed\n");
					break;
				}
			}

			if (port->dcbx_st == dcbx_subtype2) {
				if (port->tlvs.app2 != NULL)
					port->tlvs.app2 = 
						free_unpkd_tlv(
						port->tlvs.app2);
				port->tlvs.app2 =
					bld_dcbx2_app_tlv(port,
						APP_FCOE_STYPE, &success);
				if (!success) {
					printf("bld_dcbx2_app_tlv: failed\n");
					break;
				}
			} else {
				if (port->tlvs.app1 != NULL)
					port->tlvs.app1 = free_unpkd_tlv(
						port->tlvs.app1);
				port->tlvs.app1 = bld_dcbx1_app_tlv(port,
					APP_FCOE_STYPE, &success);
				if (!success) {
					printf("bld_dcbx1_app_tlv: failed\n");
					break;
				}
			}

			if (port->tlvs.llink != NULL)
				port->tlvs.llink = free_unpkd_tlv(
					port->tlvs.llink);

			port->tlvs.llink = bld_dcbx_llink_tlv(port,
				LLINK_FCOE_STYPE, &success);
			if (!success) {
				printf("bld_dcbx_llink_tlv: failed\n");
				break;
			}
		}

		if (port->dcbx_st == dcbx_subtype2) {
			if (port->tlvs.dcbx2 != NULL)
				port->tlvs.dcbx2 = free_unpkd_tlv(
					port->tlvs.dcbx2);
			port->tlvs.dcbx2 = bld_dcbx2_tlv(port);
			if (port->tlvs.dcbx2 == NULL) {
				printf("bld_dcbx2_tlv: failed\n");
				break;
			}
		} else {
			if (port->tlvs.dcbx1 != NULL)
				port->tlvs.dcbx1 = free_unpkd_tlv(
					port->tlvs.dcbx1);
			port->tlvs.dcbx1 = bld_dcbx1_tlv(port);
			if (port->tlvs.dcbx1 == NULL) {
				printf("bld_dcbx1_tlv: failed\n");
				break;
			}
		}

		port->tx.localChange = 1;
		run_tx_sm(port, FALSE);
		break;
	}

	return;
}
