/*  BaitFisher (version 1.2.7) a program for designing DNA target enrichment baits
 *  Copyright 2013-2016 by Christoph Mayer
 *
 *  This source file is part of the BaitFisher-package.
 * 
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with BaitFisher.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  For any enquiries send an Email to Christoph Mayer
 *  c.mayer.zfmk@uni-bonn.de
 *
 *  When publishing work that is based on the results please cite:
 *  Mayer et al. 2016: BaitFisher: A software package for multi-species target DNA enrichment probe design
 *  
 */
#include "global-types-and-parameters.h"
#include "tclap/CmdLine.h"
#include <string>
#include "faststring2.h"

using namespace TCLAP;
using namespace std;

faststring                       global_bait_filename;
char                             global_mode;
faststring                       global_blast_exe;
faststring                       global_output_filename;

double                           global_blast_min_hit_coverage_of_bait;
double                           global_blast_max_first_hit_evalue;
double                           global_blast_max_second_hit_evalue;
faststring                       global_blast_db;

faststring                       global_blast_extra_commandline;
faststring                       global_blast_evalue_commandline;

unsigned                         global_thinning_step_width;

bool                             global_use_GUI;
bool                             global_stats_mode;

unsigned                         global_conversion_mode;

faststring                       global_ProbeID_prefix;

unsigned                         global_verbosity;


bool file_exists(const char* fn)
{
  ifstream is;
  is.open(fn);
  if (is.fail())
  {
    return false;
  }
  is.close();
  return true;
}


bool blast_exe_exists()
{
  faststring cmd = global_blast_exe + " -version 1> /dev/null 2> /dev/null";
  int err = system(cmd.c_str());

  if (err == 0)
    return true;
  else
    return false;
}

bool blast_db_files_exist(faststring &blast_db_base_name)
{
  faststring tmp;

  tmp = blast_db_base_name + ".nhr";
  if (!file_exists(tmp.c_str()))
    return false;
  tmp = blast_db_base_name + ".nin";
  if (!file_exists(tmp.c_str()))
    return false;
  tmp = blast_db_base_name + ".nsq";
  if (!file_exists(tmp.c_str()))
    return false;
  return true;
}


void good_bye_and_exit(FILE *of, int error)
{
  if (error == 0)
    fprintf(of, "#\n#\n## Finished successfully. ##\n");
  fprintf(stderr, "\n" PROGNAME " says goodbye.\n\n");
  exit(error);
}



void init_param()
{
  global_blast_exe                      = "blastn";
  global_blast_min_hit_coverage_of_bait = 0;
  global_blast_max_first_hit_evalue     = 0.000001;
  global_blast_max_second_hit_evalue    = 0.000001;
  global_thinning_step_width            = 0;

  global_use_GUI         = false;
  global_stats_mode      = false;

  global_mode            = 0; // 0: Undefined, 'c': conversion, ....
  global_conversion_mode = 0; // 0: No conversion, 1: Agilent-Germany

  global_verbosity       = 2;
}



void read_and_init_parameters(int argc, char** argv)
{
  init_param();

  try
  {
    CmdLine cmd("This program can be used to produce the final output file for creating baits, or it can be used to filter bait loci obtained from the Bait-Fisher program according "
                "to different criteria. The bait file produced by BaitFisher computes a tiling desing for each possible starting position. The purpuse of BaitFilter is to determine for each target alignment/gene/feature the optimal bait region. As input it requires the bait file generated by the BaitFisher program or a BaitFile generated by a previous filtering run of BaitFilter. "
		"This bait file is specified with the -i command line parameter (see below). Furthermore, the user has to specifiy an output file name with the -o parameter and a filter mode with "
		"the -m parameter.\n"
		"To convert a file to a customn output format, see the -c option below.\n"
		"To compute stats of an input file, see the -S option below.\n" 
		"The different filter modes provided by BaitFilter are the following:\n" 
		"1a) Retain only the best bait locus per alignment file. Criterion: Minimize number of required baits.\n"
		"1b) Retain only the best bait locus per alignment file. Criterion: Maximize number of sequenences.\n"
		"2a) Retain only best bait locus per feature (requires that features were selected in Bait-Fisher). Criterion: Minimize number of required baits.\n"
		"2b) Retain only best bait locus per feature (requires that features were selected in Bait-Fisher). Criterion: Maximize number of sequenences.\n"
		"3) Use a blast search of the bait sequences against a reference genome to detect putative non-unique target loci. Non unique target sites will have multiple good hits against the reference genome. "
		"  Furthermore, a minimum coverage of the best blast hit of bait sequence against the genome can be specified. Note that all blast modes require additional command line parameters!"
		"  These modes remove bait regions for which multiple blast hits where found. Different versions of this mode are available:\n" 
		" 3a) If a single bait is not unique, remove all bait regions from the current gene.\n" 
		" 3b) If a single bait is not unique, remove all bait regions from the current feature (if applicable).\n"
		" 3c) If a single bait is not unique, remove only the bait region that contains this bait.\n"
		" 4) Thin out the given bait file: Retain only every Nth bait region, where N has to be specified by the user. Two submodes are available:\n" 
		" 4a) Thin out bait regions by retaining only every Nth bait region in a bait file. The starting offset will by chosen such that the number of required baits is minimized.\n"
		" 4b) Thin out bait regions by retaining only every Nth bait region in a bait file. The starting offset will by chosen such that the number of sequences the result is baised on is maximized.\n",
		' ', VERSION);

    SwitchArg global_stats_Arg("S", "stats", "Compute stats for the input file and report these. This mode is automatically used for all modes specified with -m or the conversion mode specified with -c."
				   "The purpose of the -S option is to compute stats without having to filter or convert the input file. In particular, the -S mode does not require specifying an output file.\n"
				   "This option has no effect if combined with the -m or -c modes.\n", false);
    cmd.add( global_stats_Arg );


    ValueArg<unsigned> verbostiy_Arg("", "verbosity",
				     "The verbosity option controls the amount of information " PROGNAME 
				     " writes to the console while running. 0: report only error messages that lead to exiting the program. "
				     "1: report also wanrings, 2: report also progress, 3: report more detailed progress, >10: debug output. 10000: all possible dignostic output.",
				     false, global_verbosity, "unsigned integer");
    cmd.add( verbostiy_Arg );


    ValueArg<string> conversion_IDprefix_Arg("", "ID-prefix",
       "In the conversion mode Agilent-Germany each converted file should get a unique ProdeID prefix, since even among multiple files, "
					     "ProbeIDs are not allowed to be identical. This this option the user is able to specifiy a prefix string to all probe IDs in this file.",
	false, "", "string");
    cmd.add( conversion_IDprefix_Arg );

   
    ValueArg<unsigned> thinning_step_width_Arg("t", "thinning-step-width",
					      "Thin out the bait file by retaining only every Nth bait region. This option specified the step width N. If one of the moded thin-b," 
					      " thin-s is active, this option is required, otherwise it is not allowed to set this parameter.",
					      false, global_thinning_step_width, "positive integer");
    cmd.add( thinning_step_width_Arg );

    ValueArg<string> blast_exe_Arg("B", "blast-executable",
       "Name of or path+name to the blast executable. Default: nblast. Minimum version number: Blast+ 2.2.x",
       false, global_blast_exe.c_str(), "string");
    cmd.add( blast_exe_Arg );

    ValueArg<string> blast_evalue_extra_Arg("", "blast-evalue-cutoff",
       "When invoking the blast command, a default value of twice the --<blast-first-hit-evalue is used. This should guarantee that all hits necessary for the blast filter are found. However, for the coverage filtering a smaller threshold might be necessary. This can be specified here.",
       false, "", "floating point number");
    cmd.add( blast_evalue_extra_Arg );

    ValueArg<string> blast_extra_commandline_Arg("", "blast-extra-commandline",
       "When invoking the blast command, extra commandline parameters can be passed to the blast command. As an example with this option it is possible to specifiy the number of threads the blast command should use.",
       false, "", "string");
    cmd.add( blast_extra_commandline_Arg );

    ValueArg<string> blast_db_Arg("", "ref-blast-db",
       "Base name to a blast data base file. This name is passed to the blast command. This is the name of the fasta file of your reference genome."
       " IMPORTANT: The makeblastdb program has to be called before starting the " PROGNAME " program. makeblastdb takes the fasta file and "
       " creates data base files out of it.",
	false, "", "string");
    cmd.add( blast_db_Arg );

    ValueArg<double> blast_hit_coverage_Arg("", "blast-min-hit-coverage-of-baits-in-tiling-stack",
       "Specifies a minimum hit coverage for the primary hit which at least one bait has to have in each tiling stack. Otherwise the bait region is discarded. "
       "If not specified, no hit coverage is checked. This parameter can only be used in conjunction with other filters. Since the order in which the coverage "
       "filter and the other filters are applied matters, the order is defined as follows: For the mode options: ab, as, fb, fs the coverage is checked before "
       "determining the optimal bait region. For the mode options: blast-a, blast-f, blast-l the hit coverage is checked after filtering for baits with multiple "
       "good hits to the reference genome.", false, global_blast_min_hit_coverage_of_bait, "floating point number");
    cmd.add( blast_hit_coverage_Arg );

    ValueArg<double> blast_primary_hit_evalue_Arg("", "blast-first-hit-evalue",
						  "Maximum evalue for the first hit. A bait is characterized to bind ambiguously, if we have two good hits. This option is the evalue threshold for the first hit.",
	false, global_blast_max_first_hit_evalue, "floating point number");
    cmd.add( blast_primary_hit_evalue_Arg );

    ValueArg<double> blast_secondary_hit_evalue_Arg("", "blast-second-hit-evalue",
						    "Maximum evalue for the second hit. A bait is characterized to bind ambiguously, if we have two good hits. This option is the evalue threshold for the second hit.",
	false, global_blast_max_second_hit_evalue, "floating point number");
    cmd.add( blast_secondary_hit_evalue_Arg );

 
    // The most important options must be mentioned first. Not they are specified in (reverse) order:

    // (#4) in option list 
    ValueArg<string> bait_filter_mode_Arg("m","mode",
				      "Appart form the input file this is the most essential option. This option specifies which filter mode Bait-Filter uses. (See the user manual for more details):\n"
				      "\"ab\":      Retain only the best bait locus for each alignemntfile (e.g. gene) when using the optimality criterion"
				      "             to minimize the total number of required baits.\n"
				      "\"as\":      Retain only the best bait locus for each alignemntfile (e.g. gene) when using the optimality criterion"
				      "             to maximize the number of sequences the result is based on.\n"
				      "\"fb\":      Retain only the best bait locus for each feature (e.g. CDS) when using the optimality criterion"
				      "             to minimize the total number of required baits. Only applicable if alignment cutting has been used in Bait-Fisher.\n"
				      "\"fs\":      Retain only the best bait locus for each feature (e.g. CDS) when using the optimality criterion"
				      "             to maximize the number of sequences the result is based on. Only applicable if alignment cutting has been used in Bait-Fisher. \n"
				      "\"blast-a\": Remove all bait loci of ALIGNMENTs for which one or more baits have multiple good hits to a reference genome.\n"
				      "\"blast-f\": Remove all bait loci of FEATUREs for which one or more baits have multiple good hits to a reference genome.\n"
				      "\"blast-l\": Remove bait LOCI that contain a bait that hos multiple good hits to a reference genome.\n"
                                      "\"thin-b\":  Thin out a bait file to every Nth bait region, by finding the start position that minimizes the number of baits.\n"
                                      "\"thin-s\":  Thin out a bait file to every Nth bait region, by finding the start position that maximizes the number of sequences.\n",
					  false, "","string");
    cmd.add(bait_filter_mode_Arg);    

    // (#3) in option list 
    ValueArg<string> conversion_mode_Arg("c", "convert",
       "Allows the user to produce the final output file for the bait producing company. "
       "In this mode, BaitFilter reads the input bait file and instead of doing a filtering step, it produces a "
       "costumn bait file that can be uploaded to the baits producing company. In order to avoid confuction "
       "a filtering step cannot be done in the same run as the conversion. If you want to filter a bait file and "
       "convert the output, you will need to call this program twice, first to do the filtering and second to do "
       " the conversion. Allowed conversion parameters currently are: \"Agilent-Germany\".\n"
       " New output formats can be added upon request. Please contact the author: Christoph Mayer, Email: Mayer Christoph <c.mayer.zfmk@uni-bonn.de>",
	false, "", "string");
    cmd.add( conversion_mode_Arg );

    // (#2) in option list 
    ValueArg<string> out_filename_Arg("o", "output-bait-file-name",
       "Name of the output bait file. This is the file the contains the filtered bait loci and the baits.",
	false, "", "string");
    cmd.add( out_filename_Arg );

    // (#1) in option list 
    ValueArg<string> bait_filename_Arg("i", "input-bait-file-name",
       "Name of the input bait locus file. This is the bait file obtained from the Bait-Fisher program.",
	true, "", "string");
    cmd.add( bait_filename_Arg );
    



//************************************************
    cmd.parse( argc, argv );
//************************************************

    faststring tmp, tmp1;

    // Assigning parameters to variables:
    global_bait_filename                  = bait_filename_Arg.getValue().c_str();
    tmp                                   = bait_filter_mode_Arg.getValue().c_str();
    global_output_filename                = out_filename_Arg.getValue().c_str();
    global_blast_exe                      = blast_exe_Arg.getValue().c_str();
    global_blast_db                       = blast_db_Arg.getValue().c_str();
    global_blast_min_hit_coverage_of_bait = blast_hit_coverage_Arg.getValue();
    global_blast_max_first_hit_evalue     = blast_primary_hit_evalue_Arg.getValue();
    global_blast_max_second_hit_evalue    = blast_secondary_hit_evalue_Arg.getValue();

    global_blast_evalue_commandline       = blast_evalue_extra_Arg.getValue().c_str();
    global_blast_extra_commandline        = blast_extra_commandline_Arg.getValue().c_str();
    global_thinning_step_width            = thinning_step_width_Arg.getValue();
    tmp1                                  = conversion_mode_Arg.getValue().c_str();
    global_ProbeID_prefix                 = conversion_IDprefix_Arg.getValue().c_str();
    global_verbosity                      = verbostiy_Arg.getValue();

    global_stats_mode                     = global_stats_Arg.getValue();

    if (global_output_filename.empty() && (!tmp.empty() || !tmp1.empty()) )
    {
      cerr << "The -o parameter, used to specify an output file, is a required parameter for all modes specified with the -m and -c parameter." << endl; 
      good_bye_and_exit(stdout, -2);
    }

    if (tmp.empty() && tmp1.empty() && !global_stats_mode )
    {
      cerr << "When using the BaitFisher program you have to specify either the -m (--mode) option, the -c (--convert) option or the -S option." << endl;
      cerr << "These are the three main running modes. " << endl;
      cerr << "See the BaitFilter manual for help. ";
      cerr << "Type " << PROGNAME << " -h for a quick online help." << endl;
      good_bye_and_exit(stdout, -1);
    }

    if (!tmp.empty() && !tmp1.empty() )
    {
      cerr << "When using the BaitFisher program you have to specify either the -m (--mode) option or the -c (--convert) option." << endl;
      cerr << "Filtering runs and conversion runs cannot be combined in one run, but have to be done consequtively, by invoking " << PROGNAME << " twice." << endl;
      cerr << "See the BaitFilter manual for help. ";
      cerr << "Type " << PROGNAME << " -h for a quick online help." << endl;
      good_bye_and_exit(stdout, -1);
    }

    if (global_stats_mode)
    {
      global_mode = 'S';
    }
    else if (!tmp.empty())
    {
      global_conversion_mode = 0;

      if (tmp == "ab")      // Best per alignment, number of baits
      {
	global_mode = 'a';
      }
      else if (tmp == "as")
      {
	global_mode = 'A';
      }
      else if (tmp == "fb")
      {
	global_mode = 'f';
      }
      else if (tmp == "fs")
      {
	global_mode = 'F';
      }
      else if (tmp == "blast-a")
      {
	global_mode = 'b';
      }
      else if (tmp == "blast-f")
      {
	global_mode = 'B';
      }
      else if (tmp == "blast-l")
      {
	global_mode = 'x';
      }
      else if (tmp == "thin-b")
      {
	global_mode = 't';
	if (global_thinning_step_width == 0)
	{
	  cerr << "Error: In \"thin-b\" mode, the --thinning-step-width is a required parameter and the specified value must be greater than 0,\n" << endl;
	  exit(-11);
	}
      }
      else if (tmp == "thin-s")
      {
	global_mode = 'T';
	if (global_thinning_step_width == 0)
	{
	  cerr << "Error: In \"thin-s\" mode, the --thinning-step-width is a required parameter and the specified value must be greater than 0,\n" << endl;
	  exit(-11);
	}
      }
      else
      {
	cerr << "Unknown option for parameter -m <string> : " << tmp << " encountered." << endl;
	cerr << "Allowed modes can be viewed with: " << PROGNAME << " -h " << endl;
	exit(-3);
      }
    } // END if (!tmp.empty())
    else // if ( !tmp1.empty() )  // The if is not necessary, since either tmp or tmp1 are ensured to be non-empty.
    {
      global_mode = 'c';

      if (tmp1 == "Agilent-Germany")
      {
	global_conversion_mode = 1;
      }
      else 
      {
	cerr << "Unknown option for parameter -c <string> : " << tmp1 << " encountered." << endl;
	cerr << "Allowed modes can be viewed with: " << PROGNAME << " -h " << endl;
	exit(-3);  	
      }
    }
  }
  catch (ArgException &e)
  {
    cerr << "Error: " << e.error() << " for arg " << e.argId() << endl;
    good_bye_and_exit(stdout, -1);
  }

  // Do some checks here:

  // Check whether the blast executable can be found and can be executed.
  // Check whether the data base file exists.

  if (global_thinning_step_width > 0 && tolower(global_mode) != 't' ) 
  {
    cerr << "Error: The --thinning-step-width option only has an effect in the \"-m thin-b\" or \"-m thin-s\" modes. Specifying this value in other modes is not allowed." << endl;
    exit(-12);
  }

  if (global_mode == 'b' || global_mode == 'B' || global_mode == 'x' )
  {
    bool good = blast_exe_exists();

    if (!good)
    {
      cerr << "Error: You try to run a blast filter, but the blast executable does not seem to work." << endl;
      cerr << "       I was not able to run the blast program with the following command line option:" << endl;
      cerr << "       " << global_blast_exe << " -version" << endl;
      cerr << "       This command should not give an error message, but display the version number of blast." << endl;
      cerr << "       Exiting." << endl;
      exit(-1);
    }

    if (global_blast_db.empty())
    {
      cerr << "Error: You try to run a blast filter, but the index files of your reference genome is not specified." << endl;
      cerr << "       Use the --ref-blast_db option to specifiy a blast data base index file." << endl;
      exit(-1);
    }

    good = blast_db_files_exist(global_blast_db);

    if (!good)
    {
      cerr << "Error: You try to run a blast filter, but the index files of your reference genome does not exist." << endl;
      cerr << "       If you only have a fasta file at the moment, you have to use the makeblastdb program which comes with your blast installation." << endl;
      cerr << "       Creation of the blast index was successful if you created file with the extension: nin, nsq and nhr." << endl;
      cerr << "       Exiting." << endl;
      exit(-1);
    }
  }
  else
  {
    if (!global_blast_db.empty())
    {
      cerr << "Warning: You specified a blast database file but you run " << PROGNAME
	   << " in a non blast mode. Thus, the specified data base file will not be used." << endl;
    }
    
    
  }
}



void print_parameters(FILE *of, const char *s)
{
  fprintf(of,   "%sBaitFilter has been started with the following command line parameters:\n",                      s);
  fprintf(of,   "%sInput bait file name:   %s\n",   s, global_bait_filename.c_str());
  fprintf(of,   "%sOutput bait file name:  %s\n",   s, global_output_filename.c_str());
  if (global_mode == 'a')
  {
    fprintf(of, "%sFilter mode:            Retain only the best locus per alignment file; criterion: minimize number of baits.\n", s);
  }
  else if (global_mode == 'A')
  {
    fprintf(of, "%sFilter mode:            Retain only the best locus per alignment file; criterion: maximize number of sequences in locus.\n", s);
  }
  else if (global_mode == 'f')
  {
    fprintf(of, "%sFilter mode:            Retain only the best locus per feature; criterion: minimize number of baits.\n", s);
  }
  else if (global_mode == 'F')
  {
    fprintf(of, "%sFilter mode:            Retain only the best locus per feature; criterion: maximize number of sequences in locus.\n", s);
  }
  else if (global_mode == 'b')
  {
    fprintf(of, "%sFilter mode:            Blast filter: Remove all bait loci from output that belong to the ALIGNMENT in which one bait had multiple good hits to the reference genome.\n", s);
  }
  else if (global_mode == 'B')
  {
    fprintf(of, "%sFilter mode:            Blast filter: Remove all bait loci from output that belong to the FEATUE in which one bait had multiple good hits to the reference genome.\n", s);
  }
  else if (global_mode == 'x')
  {
    fprintf(of, "%sFilter mode:            Blast filter: Remove all bait loci from output that belong to the LOCI in which one bait had multiple good hits to the reference genome.\n", s);
  }
  else if (global_mode == 't')
  {
    fprintf(of, "%sFilter mode:            Thinning mode: Retain only every %u th bait region. Choose the starting offset such that the number of baits is minimized.\n", s, global_thinning_step_width);    
  }
  else if (global_mode == 'T')
  {
    fprintf(of, "%sFilter mode:            Thinning mode: Retain only every %u th bait region. Choose the starting offset such that the number of sequences the result is based on is maximized.\n", s, global_thinning_step_width);    
  }
  else if (global_mode == 'c')
  {
    if (global_conversion_mode == 1)
    {
      fprintf(of, "%sConversion mode:        Convert bait file to Agilent-Germany format.\n", s);
    }
    else
    {
      fprintf(of, "%sERROR: Invalid internal conversion mode: %c.\n", s, global_mode);
      exit(-99);
    }
  }
  else if (global_mode == 'S')
  {
    fprintf(of, "%sMode:                   Prints stats mode.\n", s); 
  }
  else
  {
    fprintf(of, "%sERROR: Invalid internal mode: %c.\n", s, global_mode);
    exit(-99);
  }



  if (global_mode == 'b' || global_mode == 'B' || global_mode == 'x' )
  {
    fprintf(of, "%sBlast search settings and parameters:\n",   s);
    fprintf(of, "%sBlast executable:                                 %s\n",   s, global_blast_exe.c_str());
    fprintf(of, "%sBlast data base file name:                        %s\n",  s, global_blast_db.c_str());
    fprintf(of, "%sBlast minimum hit coverage of bait in first hit:  %lf\n", s, global_blast_min_hit_coverage_of_bait);
    fprintf(of, "%sBlast maximum evalue of best hit:                 %lf\n", s, global_blast_max_first_hit_evalue);
    fprintf(of, "%sBlast maximum evalue of second best hit:          %lf\n", s, global_blast_max_second_hit_evalue);
  }

  if (!global_blast_extra_commandline.empty())
  {
    fprintf(of, "%sBlast extra commandline argument:                 %s\n", s, global_blast_extra_commandline.c_str());
  }

  if (!global_blast_evalue_commandline.empty())
  {
    fprintf(of, "%sBlast evalue cut-off:                             %s\n", s, global_blast_evalue_commandline.c_str());
  }


//   fprintf(of, "%s   Sequence range to process:       %d\t-\t",  s, global_sequence_from);
//   if (global_sequence_to == -1u)
//     fprintf(of, "last sequence\n");
//   else
//     fprintf(of, "%d\n", global_sequence_to);
}


void print_output_creation_message(FILE *of, char *s)
{
  fprintf(of, "%sOutput created by " PROGNAME ", version " VERSION "\n", s);
  fprintf(of, "%s\n", s);
  print_parameters(of, s);
  fprintf(of, "%s\n", s);
}
