#define _GNU_SOURCE
#include <errno.h>
#include <string.h>

#include <sched.h>
#include <stdio.h>

#include <fcntl.h>
#include <unistd.h>
#include "sysdep.h"
#include <asm/types.h>
#include <sys/ioctl.h>
#include "../martian.h"

#define PREFIX 	"smp"
#include "log.h"

static int cpu = -1;

static int set_irq_affinity (int irq) 
{
	int fd, res; 
	int counter, num, _cpu;
	char path[256];
	char buf[256];
	char *ptr;
	
	sprintf (path, "/proc/irq/%d/smp_affinity", irq);

	fd = open (path, O_RDWR);
	if (fd < 0) {
		if (errno == ENOENT) {
			LOGDEBUG (2, "%s not found\n", path);
		}
		else {
			LOGSYSERR (Debug, "open");
			LOGDEBUG (2, "failed to open %s\n", path);
		}
		return -1;
	}
	
	num = read (fd, buf, sizeof buf);
	if (num < 0) {
		LOGSYSERR (Debug, "read");
		close (fd);
		return -1;
	}

	buf[sizeof buf - 1] = '\0';
	LOGDEBUG (Note, "initial irq affinity in proc %s\n", buf);

	
	/* parsing to find the first available processor */
	ptr = buf + num - 2; 	/* check '\n' at the end */
	_cpu = counter = 0;
	while (ptr > buf) {
		switch (*ptr) {
		case '0':
		/* skip */
			_cpu += 4;
			break;

		case '3':
		case '5':
		case '7':
		case '9':
		case 'b':
		case 'd':
		case 'f':
			*ptr = '1';
		case '1':
			goto l_escape;
			
		case '6':
		case 'a':
		case 'e':
			*ptr = '2';
		case '2':
			_cpu++;
			goto l_escape;
			
		case 'c':
			*ptr = '4';
		case '4':
			_cpu += 2;
			goto l_escape;

		case '8':
			_cpu += 3;
			goto l_escape;
			
		case ',':
			if (counter != 8) {
				LOGWARN ("wrong separator placement\n");
				goto l_cleanup;
			}
			counter = 0; ptr--;
			continue;
		
		default:
			LOGWARN ("unexpected symbol %c\n", *ptr);
			goto l_cleanup;
		}
		counter++;
		ptr--;
		if (counter > 8) {
			LOGWARN ("too many digits in a chunk\n");
			goto l_cleanup;
		}
	}
	LOGWARN ("no processors available");
l_cleanup:	
	close (fd);
	return -1;

l_escape:
	if (counter == 8) {
		LOGWARN ("too many digits in a chunk\n");
		goto l_cleanup;
	}
	res = write (fd, ptr, num - (ptr - buf) - 1);
	if (res <= 0)
		LOGSYSERR (Debug, "write");
	else {
		if (res != num - (ptr - buf) - 1) 
			LOGWARN ("write inconsistency\n");
		cpu = _cpu;
	}

	close (fd);

	LOGDEBUG (2, "using cpu %d\n", cpu);
	if (Debugged (Note)) {
		fd = open (path, O_RDWR);
		num = read (fd, buf, sizeof buf);
		buf[sizeof buf - 1] = '\0';
		LOGDEBUG (2, "irq affinity set %s\n", buf);
	}

	return cpu;
}

const char keyword[] = "processor\t:";
static int get_cpu_number (void) {
	int counter;
	char buf[] = "processor\t:";
	FILE *fp = fopen ("/proc/cpuinfo", "r");
	if (fp == 0) {
		LOGDEBUG (2, "Cannot open /proc/cpuinfo (%s)\n", strerror(errno));
		return -1;
	}

	counter = 0;
	/* inv: no '\n' in buf &&
	        fp -> line start  */
	do {
		char *ptr;

		if (fgets (buf, sizeof buf, fp) == NULL) 
			break;

		if (strcmp (buf, keyword) == 0) {
			counter++;
		}
		else if ((ptr = (char *) memchr (buf, '\n', sizeof buf)) != NULL) {
			*ptr = '\0';
			/* we are on the next line start && no '\n' in buf */
			continue;
		}
		
		/* move to next line */
		int c;
		do {
			c = fgetc (fp);
			if (c == EOF) 
				goto l_quit;
			
		} while (c != '\n');
		
	} while (1);
l_quit:
	fclose (fp);
	
	if (counter == 0) 
		LOGDEBUG (2, "get_cpu_num: unexpected end of file\n");
		
	return (counter == 0) ? -1 : counter;
}

void test_affinity_dump (void );

int smp_setup (int irq, int dev_fd)
{
	int num = get_cpu_number();

	config.affinity_cpu = -1;

	/*test_affinity_dump ();*/
	/* if (num == -1) num = 1;*/
	if (num <= 1) {
		config.smp = 0; 
		if (config.true_smp) {
			LOGWARN ("--smp ignored: UP box.\n");
			config.true_smp = 0;
		}
		return 0;
	}

	config.smp = 1;

	if (config.true_smp) {
		if (ioctl (dev_fd, MARTIAN_TRUE_SMP) == 0) {
			LOGINFO ("running in true smp mode -- EXPERIMENTAL.\n");
			return 1;
		}
		LOGERR ("Kernel module denied running in true smp mode\n");
		config.true_smp = 0;
	}
	
	LOGINFO ("configuring\n");
	/* */
	config.affinity_cpu = set_irq_affinity (irq);
	ioctl (dev_fd, MARTIAN_CHECK_SMP, config.affinity_cpu);

	return 1;
}

static inline int 
bits_range_zero (unsigned long val, int start, int end) 
{
	if (start > end) 
		return 1;
	return ((val >> start) & ((1 << (1 + end - start)) - 1)) == 0;
}

static inline int 
bits_range_set (unsigned long val, int start, int end) 
{
	if (start > end) 
		return 1;
	else {
		unsigned mask = ((1 << (1 + end - start)) - 1);
		return ((val >> start) & mask) == mask;
	}
}

#undef PREFIX
#define PREFIX 	"smp: affinity"

static void dump_affinity (cpu_set_t *cpus) {
	int _cpu, _cpul, _cpur, next_to_check;
	__cpu_mask cpus_unit = cpus->__bits[0], cpus_unit_copy;

	cpus_unit_copy = cpus_unit;
	_cpu = 0;  next_to_check = 0;
	if (cpus_unit) while (1) {
		_cpul = __builtin_ctzl (cpus_unit);
		logdebugadd (Note, "%d", _cpu + _cpul);
		ASSERT (bits_range_zero (cpus_unit_copy, next_to_check, (_cpu + _cpul) - 1),
			break);

		_cpul++;
		cpus_unit >>= _cpul;
		/* assert (~cpus_unit != 0) */
		/* cpus_unit & 0x1 - unknown */
		_cpu += _cpul;
		next_to_check = _cpu - 1;

		_cpur = __builtin_ctzl (~cpus_unit);
		switch (_cpur) {
		case 0:
			break;

		case 1:
			logdebugadd (Note, ", %d", _cpu);
			break;

		default:
			logdebugadd (Note, "..%d", _cpu + _cpur - 1);
		}

		ASSERT (bits_range_set (cpus_unit_copy, next_to_check, (_cpu + _cpur) - 1),
			break);
		next_to_check = _cpu + _cpur;

		_cpur++;
		_cpu += _cpur;
		cpus_unit >>= _cpur;
		/* cpus_unit & 0x1 - unknown */

		if (! cpus_unit) {
			logdebugadd (Note, ".\n");
			break;
		}
		logdebugadd (Note, ", ");
	}
	else {
		logdebugadd (Note, "none\n");
	}
}

#include <stdlib.h>
static unsigned my_rand (unsigned start, unsigned end) {
	unsigned long long range = 1 + (unsigned long long ) end - start;
	return start + (unsigned ) ((rand() * range) / (RAND_MAX + 1UL));
}

void test_affinity_dump (void ) {
	int cpur, cpul, _cpu;
	cpu_set_t cpus;
	__cpu_mask *cpus_unit = &cpus.__bits[0];

	CPU_ZERO (&cpus);
	/* generate sequence */
	LOGDEBUG (Note, "Testing affinity dump.\n");
	int testc;
	for (testc = 1; testc < 10; testc++) {
		srand (testc);
		_cpu = 0; 
		*cpus_unit = 0;
		LOGDEBUG (Note, "generated: ");
		while (_cpu < 8 * sizeof *cpus_unit) {
			int i;
			cpul = my_rand (_cpu, 8 * sizeof *cpus_unit - 1);
			cpur = my_rand (cpul, 8 * sizeof *cpus_unit - 1);
			for (i = cpul; i <= cpur; i++)
				*cpus_unit |= (1 << i);
			_cpu = cpur + 2;
			logdebugadd (Note, "%d:%d, ", cpul, cpur);
		}
		logdebugadd (Note, "got: ");
		{
			dump_affinity (&cpus);
		}
	}
}


int pin_thread (void)
{
	if (!config.smp || config.true_smp || config.affinity_cpu == -1) 
		return 0;

	else {
		int res;
		cpu_set_t cpus;

		res = sched_getaffinity (mgettid(), sizeof cpus, & cpus);
		if (res != 0) {
			LOGSYSERR (Debug, "sched_getaffinity");
			LOGDEBUG (Note, "failed to get affinity for thread %d\n", mgettid());
		}
		else {
			LOGDEBUG (Note, "hard initial affinity cpus: ");
			dump_affinity (&cpus);
		}

		CPU_ZERO (&cpus);
		CPU_SET (config.affinity_cpu, &cpus);

		res = sched_setaffinity (mgettid(), sizeof cpus, &cpus);
		if (res < 0) {
			LOGSYSERR (Debug, "sched_setaffinity");
			LOGWARN ("failed to pin thread %d\n", mgettid());
		}

		res = sched_getaffinity (mgettid(), sizeof cpus, & cpus);
		if (res != 0) {
			LOGSYSERR (Debug, "sched_getaffinity");
			LOGDEBUG (Note, "failed to get affinity for thread %d\n", mgettid());
		}
		else {
			LOGDEBUG (Note, "hard set affinity cpu: ");
			dump_affinity (&cpus);
		}

		return res;
	}
}

#ifdef MARTIAN_HELPER_TESTS
int pin_thread_test (void) __attribute__ (( alias ("pin_thread") ));
int set_irq_affinity_test (int irq) __attribute__ (( alias ("set_irq_affinity") ));
int get_cpu_number_test (void) __attribute__ (( alias ("get_cpu_number") ));
#endif
