#include <glib.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib-object.h>

#include "apt-cache-generator.h"
#include "apt-cache-parser.h"
#include "webapp-info.h"

typedef struct _AptCacheTestFixture {
  AptCacheGenerator *generator;
  AptCacheParser *parser;
} AptCacheTestFixture;

static gchar *
print_parser_results (AptCacheTestFixture *fixture)
{
  GVariant *serialized_variant;
  gchar *ret;

  serialized_variant = apt_cache_parser_serialize_results (fixture->parser);
  ret = g_variant_print (serialized_variant, FALSE);
  g_variant_unref (serialized_variant);
  return ret;
}

static void
setup_fixture_one_package (AptCacheTestFixture *fixture,
			   gconstpointer user_data)
{
  AptCacheGenerator *generator;
  
  generator = apt_cache_generator_new ();
  apt_cache_generator_add_application (generator, "foo");
  
  fixture->generator = generator;
  fixture->parser = apt_cache_parser_new ();
}

static void
setup_fixture_multiple_packages (AptCacheTestFixture *fixture,
				 gconstpointer user_data)
{
  AptCacheGenerator *generator;
  
  generator = apt_cache_generator_new ();
  apt_cache_generator_add_application (generator, "foo");
  apt_cache_generator_add_application (generator, "bar");
  apt_cache_generator_add_application (generator, "baz");
  apt_cache_generator_add_application (generator, "bam");
  
  fixture->generator = generator;
  fixture->parser = apt_cache_parser_new ();
}

static void
setup_fixture_one_webapp_package (AptCacheTestFixture *fixture,
				  gconstpointer user_data)
{
  AptCacheGenerator *generator;
  
  generator = apt_cache_generator_new ();
  apt_cache_generator_add_webapp (generator, "unity-webapps-angrybirds",
				  "AngryBirds",
				  "test.ts",
				  "http://test.ts/foo/*",
				  "http://test.ts/bar/*",
				  NULL);
  
  fixture->generator = generator;
  fixture->parser = apt_cache_parser_new ();
}

static void
teardown_fixture (AptCacheTestFixture *fixture,
		  gconstpointer user_data)
{
  apt_cache_generator_free (fixture->generator);
  apt_cache_parser_free (fixture->parser);
}

static void
test_generator_single_app (AptCacheTestFixture *fixture,
			   gconstpointer user_data)
{
  const gchar *const expected_results = "Package: foo\n"
    "Version: 1.0\n"
    "Description: Lorum ipsum dolor sit amet\n\n";
  gchar *available;
  
  available = apt_cache_generator_get_available (fixture->generator);
  g_assert_cmpstr (available, ==, expected_results);
  g_free (available);
}

static void
test_generator_single_webapp (AptCacheTestFixture *fixture,
			      gconstpointer user_data)
{
  const gchar *const expected_results = "Package: unity-webapps-angrybirds\n"
    "Version: 1.0\n"
    "Description: Lorum ipsum dolor sit amet\n"
    "Ubuntu-Webapps-Includes: http://test.ts/foo/*;http://test.ts/bar/*\n"
    "Ubuntu-Webapps-Name: AngryBirds\n"
    "Ubuntu-Webapps-Domain: test.ts\n\n";
  gchar *available;
  
  available = apt_cache_generator_get_available (fixture->generator);
  g_assert_cmpstr (available, ==, expected_results);
  g_free (available);
}

static void
test_parser_no_webapps (AptCacheTestFixture *fixture,
			gconstpointer user_data)
{
  gchar *available;
  gboolean parsed;

  available = apt_cache_generator_get_available (fixture->generator);
  parsed = apt_cache_parser_parse_available (fixture->parser, available);
  g_assert (parsed);
  g_assert (g_list_length (apt_cache_parser_get_webapp_infos (fixture->parser)) == 0);
}

static void
test_parser_one_webapp (AptCacheTestFixture *fixture, 
			gconstpointer user_data)
{
  gchar *available;
  gboolean parsed;
  const gchar *const expected_parser_results = 
    "([('unity-webapps-angrybirds', 'AngryBirds', [('http://test.ts/foo/*',), ('http://test.ts/bar/*',)])],)";

  available = apt_cache_generator_get_available (fixture->generator);
  parsed = apt_cache_parser_parse_available (fixture->parser, available);
  g_assert (parsed);
  // Leaks
  g_assert_cmpstr(print_parser_results (fixture), ==, expected_parser_results);
}

gint
main (gint argc, gchar **argv)
{
  g_type_init ();
  g_test_init (&argc, &argv, NULL);
  
  g_test_add("/Generator/SinglePackage", AptCacheTestFixture,
	     NULL,
	     setup_fixture_one_package, test_generator_single_app,
	     teardown_fixture);
  g_test_add("/Generator/SingleWebappPackage", AptCacheTestFixture,
	     NULL,
	     setup_fixture_one_webapp_package, test_generator_single_webapp,
	     teardown_fixture);
  g_test_add("/Parser/SinglePackageNoWebapps", AptCacheTestFixture,
	     NULL,
	     setup_fixture_one_package, test_parser_no_webapps,
	     teardown_fixture);
  g_test_add("/Parser/MultiplePackagesNoWebapps", AptCacheTestFixture,
	     NULL,
	     setup_fixture_multiple_packages, test_parser_no_webapps,
	     teardown_fixture);
  g_test_add("/Parser/SingleWebapp", AptCacheTestFixture,
	     NULL,
	     setup_fixture_one_webapp_package, test_parser_one_webapp,
	     teardown_fixture);

  return g_test_run ();
}
