/* AbstractLibrary.cpp */

/* Copyright (C) 2011-2020 Michael Lugmair (Lucio Carreras)
 *
 * This file is part of sayonara player
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "AbstractLibrary.h"

#include "Components/Tagging/ChangeNotifier.h"

#include "Interfaces/LibraryPlaylistInteractor.h"

#include "Utils/Algorithm.h"
#include "Utils/MetaData/MetaDataList.h"
#include "Utils/MetaData/Album.h"
#include "Utils/MetaData/Artist.h"
#include "Utils/MetaData/MetaDataSorting.h"
#include "Utils/Settings/Settings.h"
#include "Utils/Logger/Logger.h"
#include "Utils/Language/Language.h"
#include "Utils/FileUtils.h"
#include "Utils/ExtensionSet.h"
#include "Utils/Set.h"

#include <QHash>

struct AbstractLibrary::Private
{
	LibraryPlaylistInteractor* playlistInteractor;
	Util::Set<ArtistId> selectedArtists;
	Util::Set<AlbumId> selectedAlbums;
	Util::Set<TrackID> selectedTracks;

	ArtistList artists;
	AlbumList albums;
	MetaDataList tracks;
	MetaDataList currentTracks;
	MetaDataList filteredTracks;

	Gui::ExtensionSet extensions;

	int trackCount;

	Library::Sortings sortorder;
	Library::Filter filter;
	bool loaded;

	Private(LibraryPlaylistInteractor* playlistInteractor) :
		playlistInteractor {playlistInteractor},
		trackCount(0),
		sortorder(GetSetting(Set::Lib_Sorting)),
		loaded(false)
	{
		filter.setMode(Library::Filter::Fulltext);
		filter.setFiltertext("", GetSetting(Set::Lib_SearchMode));
	}
};

AbstractLibrary::AbstractLibrary(LibraryPlaylistInteractor* playlistInteractor, QObject* parent) :
	QObject(parent)
{
	m = Pimpl::make<Private>(playlistInteractor);

	auto* mdcn = Tagging::ChangeNotifier::instance();
	connect(mdcn, &Tagging::ChangeNotifier::sigMetadataChanged,
	        this, &AbstractLibrary::metadataChanged);

	connect(mdcn, &Tagging::ChangeNotifier::sigMetadataDeleted,
	        this, &AbstractLibrary::metadataChanged);

	connect(mdcn, &Tagging::ChangeNotifier::sigAlbumsChanged,
	        this, &AbstractLibrary::albumsChanged);
}

AbstractLibrary::~AbstractLibrary() = default;

void AbstractLibrary::load()
{
	{ // init artist sorting mode
		ListenSettingNoCall(Set::Lib_SortIgnoreArtistArticle, AbstractLibrary::ignoreArtistArticleChanged);

		const auto b = GetSetting(Set::Lib_SortIgnoreArtistArticle);
		MetaDataSorting::setIgnoreArticle(b);
	}

	m->filter.clear();

	refetch();

	m->trackCount = getTrackCount();
	m->loaded = true;
}

bool AbstractLibrary::isLoaded() const
{
	return m->loaded;
}

void AbstractLibrary::emitAll()
{
	prepareArtists();
	prepareAlbums();
	prepareTracks();

	emit sigAllArtistsLoaded();
	emit sigAllAlbumsLoaded();
	emit sigAllTracksLoaded();
}

void AbstractLibrary::refetch()
{
	m->selectedArtists.clear();
	m->selectedAlbums.clear();
	m->selectedTracks.clear();
	m->filter.clear();

	m->artists.clear();
	m->albums.clear();
	m->tracks.clear();

	getAllArtists(m->artists);
	getAllAlbums(m->albums);
	getAllTracks(m->tracks);

	emitAll();
}

void AbstractLibrary::refreshCurrentView()
{
	/* Waring! Sorting after each fetch is important here! */
	/* Do not call emit_stuff() in order to avoid double sorting */
	IndexSet selectedArtistIndexes, selectedAlbumIndexes, selectedTrackIndexes;

	const auto selectedArtists = std::move(m->selectedArtists);
	const auto selectedAlbums = std::move(m->selectedAlbums);
	const auto selectedTracks = std::move(m->selectedTracks);

	fetchByFilter(m->filter, true);

	prepareArtists();
	for(int i = 0; i < m->artists.count(); i++)
	{
		if(selectedArtists.contains(m->artists[i].id()))
		{
			selectedArtistIndexes.insert(i);
		}
	}

	changeArtistSelection(selectedArtistIndexes);

	prepareAlbums();
	for(auto i = 0; i < m->albums.count(); i++)
	{
		if(selectedAlbums.contains(m->albums[i].id()))
		{
			selectedAlbumIndexes.insert(i);
		}
	}

	changeAlbumSelection(selectedAlbumIndexes);

	prepareTracks();

	const auto& tracks = this->tracks();
	for(auto i = 0; i < tracks.count(); i++)
	{
		if(selectedTracks.contains(tracks[i].id()))
		{
			selectedTrackIndexes.insert(i);
		}
	}

	emit sigAllAlbumsLoaded();
	emit sigAllArtistsLoaded();
	emit sigAllTracksLoaded();

	if(!selectedTrackIndexes.isEmpty())
	{
		changeTrackSelection(selectedTrackIndexes);
	}
}

void AbstractLibrary::metadataChanged()
{
	auto* mdcn = static_cast<Tagging::ChangeNotifier*>(sender());
	const auto& changedTracks = mdcn->changedMetadata();

	QHash<TrackID, int> idRowMap;
	{ // build lookup tree
		int i = 0;
		for(auto it = m->tracks.begin(); it != m->tracks.end(); it++, i++)
		{
			idRowMap[it->id()] = i;
		}
	}

	auto needsRefresh = false;
	for(auto it = changedTracks.begin(); it != changedTracks.end(); it++)
	{
		const auto& oldTrack = it->first;
		const auto& newTrack = it->second;

		needsRefresh =
			(oldTrack.artist() != newTrack.artist()) ||
			(oldTrack.albumArtist() != newTrack.albumArtist()) ||
			(oldTrack.album() != newTrack.album());

		if(idRowMap.contains(oldTrack.id()))
		{
			const auto row = idRowMap[oldTrack.id()];
			m->tracks[row] = newTrack;

			emit sigCurrentTrackChanged(row);
		}
	}

	if(needsRefresh)
	{
		refreshCurrentView();
	}
}

void AbstractLibrary::albumsChanged()
{
	auto* mdcn = static_cast<Tagging::ChangeNotifier*>(sender());

	QHash<AlbumId, int> idRowMap;
	{ // build lookup tree
		auto i = 0;
		for(auto it = m->albums.begin(); it != m->albums.end(); it++, i++)
		{
			idRowMap[it->id()] = i;
		}
	}

	const auto changedAlbums = mdcn->changedAlbums();
	for(const auto& albumPair : changedAlbums)
	{
		const auto& oldAlbum = albumPair.first;
		const auto& newAlbum = albumPair.second;

		if(idRowMap.contains(oldAlbum.id()))
		{
			const auto row = idRowMap[oldAlbum.id()];
			m->albums[row] = newAlbum;

			emit sigCurrentAlbumChanged(row);
		}
	}
}

void AbstractLibrary::findTrack(TrackID id)
{
	MetaData track;
	getTrackById(id, track);

	if(track.id() < 0)
	{
		return;
	}

	{ // clear old selections/filters
		if(!m->selectedArtists.isEmpty())
		{
			selectedArtistsChanged(IndexSet());
		}

		if(!m->selectedAlbums.isEmpty())
		{
			selectedAlbumsChanged(IndexSet());
		}

		// make sure, that no artist_selection_changed or album_selection_changed
		// messes things up
		emitAll();
	}

	{ // clear old fetched artists/albums/tracks
		m->tracks.clear();
		m->artists.clear();
		m->albums.clear();

		m->selectedTracks.clear();
		m->filteredTracks.clear();
		m->selectedArtists.clear();
		m->selectedAlbums.clear();
	}

	m->tracks.emplace_back(std::move(track));

	{ // artist
		Artist artist;
		getArtistById(track.artistId(), artist);
		m->artists.emplace_back(std::move(artist));
	}

	{ // album
		Album album;
		getAlbumById(track.albumId(), album);
		m->albums.emplace_back(album);
	}

	getAllTracksByAlbum({track.albumId()}, m->tracks, Library::Filter());
	m->selectedTracks << track.id();

	emitAll();
}

void AbstractLibrary::prepareFetchedTracksForPlaylist(bool createNewPlaylist)
{
	m->playlistInteractor->createPlaylist(tracks(), createNewPlaylist);
}

void AbstractLibrary::prepareCurrentTracksForPlaylist(bool createNewPlaylist)
{
	m->playlistInteractor->createPlaylist(currentTracks(), createNewPlaylist);
}

void AbstractLibrary::prepareTracksForPlaylist(const QStringList& paths, bool createNewPlaylist)
{
	m->playlistInteractor->createPlaylist(paths, createNewPlaylist);
}

void AbstractLibrary::playNextFetchedTracks()
{

	m->playlistInteractor->insertAfterCurrentTrack(tracks());
}

void AbstractLibrary::playNextCurrentTracks()
{
	m->playlistInteractor->insertAfterCurrentTrack(currentTracks());
}

void AbstractLibrary::appendFetchedTracks()
{
	m->playlistInteractor->append(tracks());
}

void AbstractLibrary::appendCurrentTracks()
{
	m->playlistInteractor->append(currentTracks());
}

void AbstractLibrary::changeArtistSelection(const IndexSet& indexes)
{
	Util::Set<ArtistId> selectedArtists;
	for(auto idx : indexes)
	{
		const auto& artist = m->artists[static_cast<ArtistList::Size>(idx)];
		selectedArtists.insert(artist.id());
	}

	if(selectedArtists == m->selectedArtists)
	{
		return;
	}

	m->selectedArtists = std::move(selectedArtists);
	m->albums.clear();
	m->tracks.clear();

	if(!m->selectedArtists.isEmpty())
	{
		getAllTracksByArtist(m->selectedArtists.toList(), m->tracks, m->filter);
		getAllAlbumsByArtist(m->selectedArtists.toList(), m->albums, m->filter);
	}

	else if(!m->filter.cleared())
	{
		getAllTracksBySearchstring(m->filter, m->tracks);
		getAllAlbumsBySearchstring(m->filter, m->albums);
		getAllArtistsBySearchstring(m->filter, m->artists);
	}

	else
	{
		getAllTracks(m->tracks);
		getAllAlbums(m->albums);
	}

	prepareArtists();
	prepareAlbums();
	prepareTracks();
}

const MetaDataList& AbstractLibrary::tracks() const
{
	return (m->filteredTracks.isEmpty())
	       ? m->tracks
	       : m->filteredTracks;
}

const AlbumList& AbstractLibrary::albums() const
{
	return m->albums;
}

const ArtistList& AbstractLibrary::artists() const
{
	return m->artists;
}

const MetaDataList& AbstractLibrary::currentTracks() const
{
	return (m->selectedTracks.isEmpty())
	       ? tracks()
	       : m->currentTracks;
}

void AbstractLibrary::changeCurrentDisc(Disc disc)
{
	if(m->selectedAlbums.size() != 1)
	{
		return;
	}

	getAllTracksByAlbum(m->selectedAlbums.toList(), m->tracks, m->filter);

	if(disc != std::numeric_limits<Disc>::max())
	{
		m->tracks.removeTracks([disc](const MetaData& track) {
			return (track.discnumber() != disc);
		});
	}

	prepareTracks();
	emit sigAllTracksLoaded();
}

const IdSet& AbstractLibrary::selectedTracks() const
{
	return m->selectedTracks;
}

const IdSet& AbstractLibrary::selectedAlbums() const
{
	return m->selectedAlbums;
}

const IdSet& AbstractLibrary::selectedArtists() const
{
	return m->selectedArtists;
}

Library::Filter AbstractLibrary::filter() const
{
	return m->filter;
}

void AbstractLibrary::changeFilter(Library::Filter filter, bool force)
{
	const auto filtertext = filter.filtertext(false);

	if(filter.mode() != Library::Filter::InvalidGenre)
	{
		if(filtertext.join("").size() < 3)
		{
			filter.clear();
		}

		else
		{
			const auto searchModeMask = GetSetting(Set::Lib_SearchMode);
			filter.setFiltertext(filtertext.join(","), searchModeMask);
		}
	}

	if(filter == m->filter)
	{
		return;
	}

	fetchByFilter(filter, force);
	emitAll();
}

void AbstractLibrary::selectedArtistsChanged(const IndexSet& indexes)
{
	// happens, when the model is set at initialization of table views
	if(m->selectedArtists.isEmpty() && indexes.isEmpty())
	{
		return;
	}

	changeArtistSelection(indexes);

	emit sigAllAlbumsLoaded();
	emit sigAllTracksLoaded();
}

void AbstractLibrary::changeAlbumSelection(const IndexSet& indexes, bool ignoreArtists)
{
	Util::Set<AlbumId> selectedAlbums;

	for(const auto& index : indexes)
	{
		if(index < m->albums.count())
		{
			const auto& album = m->albums[index];
			selectedAlbums.insert(album.id());
		}
	}

	m->tracks.clear();
	m->selectedAlbums = selectedAlbums;

	// only show tracks of selected album / artist
	if(!m->selectedArtists.isEmpty() && !ignoreArtists)
	{
		if(!m->selectedAlbums.isEmpty())
		{
			MetaDataList tracks;
			getAllTracksByAlbum(m->selectedAlbums.toList(), tracks, m->filter);

			Util::Algorithm::moveIf(tracks, m->tracks, [&](const auto& track) {
				const auto artistId = (GetSetting(Set::Lib_ShowAlbumArtists))
				                      ? track.albumArtistId()
				                      : track.artistId();

				return m->selectedArtists.contains(artistId);
			});
		}

		else
		{
			getAllTracksByArtist(m->selectedArtists.toList(), m->tracks, m->filter);
		}
	}

		// only album is selected
	else if(!m->selectedAlbums.isEmpty())
	{
		getAllTracksByAlbum(m->selectedAlbums.toList(), m->tracks, m->filter);
	}

		// neither album nor artist, but searchstring
	else if(!m->filter.cleared())
	{
		getAllTracksBySearchstring(m->filter, m->tracks);
	}

		// no album, no artist, no searchstring
	else
	{
		getAllTracks(m->tracks);
	}

	prepareTracks();
}

void AbstractLibrary::selectedAlbumsChanged(const IndexSet& indexes, bool ignoreArtists)
{
	// happens, when the model is set at initialization of table views
	if(m->selectedAlbums.isEmpty() && indexes.isEmpty())
	{
		return;
	}

	changeAlbumSelection(indexes, ignoreArtists);
	emit sigAllTracksLoaded();
}

void AbstractLibrary::changeTrackSelection(const IndexSet& indexes)
{
	m->selectedTracks.clear();
	m->currentTracks.clear();

	for(const auto& index : indexes)
	{
		if(index < 0 || index >= tracks().count())
		{
			continue;
		}

		const auto& track = tracks().at(index);

		m->currentTracks << track;
		m->selectedTracks.insert(track.id());
	}
}

void AbstractLibrary::selectedTracksChanged(const IndexSet& indexes)
{
	changeTrackSelection(indexes);
}

void AbstractLibrary::fetchByFilter(Library::Filter filter, bool force)
{
	if((m->filter == filter) &&
	   (m->selectedArtists.empty()) &&
	   (m->selectedAlbums.empty()) &&
	   !force)
	{
		return;
	}

	m->filter = filter;

	m->artists.clear();
	m->albums.clear();
	m->tracks.clear();

	m->selectedArtists.clear();
	m->selectedAlbums.clear();

	if(m->filter.cleared())
	{
		getAllArtists(m->artists);
		getAllAlbums(m->albums);
		getAllTracks(m->tracks);
	}

	else
	{
		getAllArtistsBySearchstring(m->filter, m->artists);
		getAllAlbumsBySearchstring(m->filter, m->albums);
		getAllTracksBySearchstring(m->filter, m->tracks);
	}
}

void AbstractLibrary::fetchTracksByPath(const QStringList& paths)
{
	m->tracks.clear();

	if(!paths.isEmpty())
	{
		getAllTracksByPath(paths, m->tracks);
	}

	emitAll();
}

void AbstractLibrary::changeTrackSortorder(Library::SortOrder sortOrder)
{
	if(sortOrder == m->sortorder.so_tracks)
	{
		return;
	}

	auto sorting = GetSetting(Set::Lib_Sorting);
	sorting.so_tracks = sortOrder;
	SetSetting(Set::Lib_Sorting, sorting);
	m->sortorder = sorting;

	prepareTracks();
	emit sigAllTracksLoaded();
}

void AbstractLibrary::changeAlbumSortorder(Library::SortOrder sortOrder)
{
	if(sortOrder == m->sortorder.so_albums)
	{
		return;
	}

	auto sorting = GetSetting(Set::Lib_Sorting);
	sorting.so_albums = sortOrder;
	SetSetting(Set::Lib_Sorting, sorting);

	m->sortorder = sorting;

	prepareAlbums();
	emit sigAllAlbumsLoaded();
}

void AbstractLibrary::changeArtistSortorder(Library::SortOrder sortOrder)
{
	if(sortOrder == m->sortorder.so_artists)
	{
		return;
	}

	auto sorting = GetSetting(Set::Lib_Sorting);
	sorting.so_artists = sortOrder;
	SetSetting(Set::Lib_Sorting, sorting);

	m->sortorder = sorting;

	prepareArtists();
	emit sigAllArtistsLoaded();
}

Library::Sortings AbstractLibrary::sortorder() const
{
	return m->sortorder;
}

void AbstractLibrary::importFiles(const QStringList& files)
{
	Q_UNUSED(files)
}

void AbstractLibrary::deleteCurrentTracks(Library::TrackDeletionMode mode)
{
	if(mode != Library::TrackDeletionMode::None)
	{
		deleteTracks(currentTracks(), mode);
	}
}

void AbstractLibrary::deleteFetchedTracks(Library::TrackDeletionMode mode)
{
	if(mode != Library::TrackDeletionMode::None)
	{
		deleteTracks(tracks(), mode);
	}
}

void AbstractLibrary::deleteAllTracks()
{
	MetaDataList tracks;
	getAllTracks(tracks);
	deleteTracks(tracks, Library::TrackDeletionMode::OnlyLibrary);
}

void AbstractLibrary::deleteTracks(const MetaDataList& tracks, Library::TrackDeletionMode mode)
{
	if(mode == Library::TrackDeletionMode::None)
	{
		return;
	}

	const auto fileEntry = (mode == Library::TrackDeletionMode::AlsoFiles)
	                       ? Lang::get(Lang::Files)
	                       : Lang::get(Lang::Entries);
	QString answerString;

	auto failCount = 0;
	if(mode == Library::TrackDeletionMode::AlsoFiles)
	{
		failCount = std::count_if(tracks.begin(), tracks.end(), [](const auto& track) {
			return (QFile(track.filepath()).remove() == false);
		});
	}

	answerString = (failCount == 0)
	               ? tr("All %1 could be removed").arg(fileEntry)
	               : tr("%1 of %2 %3 could not be removed")
		               .arg(failCount)
		               .arg(tracks.size())
		               .arg(fileEntry);

	emit sigDeleteAnswer(answerString);
	Tagging::ChangeNotifier::instance()->deleteMetadata(tracks);

	refreshCurrentView();
}

void AbstractLibrary::deleteTracksByIndex(const IndexSet& indexes, Library::TrackDeletionMode mode)
{
	if(mode == Library::TrackDeletionMode::None || indexes.isEmpty())
	{
		return;
	}

	MetaDataList tracksToDelete;
	const auto& tracks = this->tracks();

	for(const auto& index : indexes)
	{
		tracksToDelete.push_back(tracks[index]);
	}

	deleteTracks(tracksToDelete, mode);
}

void AbstractLibrary::prepareTracks()
{
	m->extensions.clear();
	m->filteredTracks.clear();

	for(const auto& track : tracks())
	{
		m->extensions.addExtension(Util::File::getFileExtension(track.filepath()), false);
	}

	m->tracks.sort(m->sortorder.so_tracks);
}

void AbstractLibrary::prepareAlbums()
{
	m->albums.sort(m->sortorder.so_albums);
}

void AbstractLibrary::prepareArtists()
{
	m->artists.sort(m->sortorder.so_artists);
}

void AbstractLibrary::ignoreArtistArticleChanged()
{
	const auto ignoreArticle = GetSetting(Set::Lib_SortIgnoreArtistArticle);
	MetaDataSorting::setIgnoreArticle(ignoreArticle);

	refreshCurrentView();
}

Gui::ExtensionSet AbstractLibrary::extensions() const
{
	return m->extensions;
}

bool AbstractLibrary::isReloading() const
{
	return false;
}

bool AbstractLibrary::isEmpty() const
{
	return m->tracks.isEmpty() && (m->trackCount == 0);
}

void AbstractLibrary::setExtensions(const Gui::ExtensionSet& extensions)
{
	m->extensions = extensions;
	m->filteredTracks.clear();

	if(m->extensions.hasEnabledExtensions())
	{
		Util::Algorithm::copyIf(m->tracks, m->filteredTracks, [&](const auto& track) {
			const auto extension = ::Util::File::getFileExtension(track.filepath());
			return (m->extensions.isEnabled(extension));
		});
	}

	emit sigAllTracksLoaded();
}

