/***************************************************************
 * Name:      phdialog.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2020
 * License:   GPL3
 **************************************************************/

#include <wx/gbsizer.h>
#include "dxdefs.h"
#include "phdialog.h"
#include "dxutils.h"
#include "dxicons.h"
#include "dxsettings.h"

bool sortHolidays(Holiday i, Holiday j)
{
    return i.month()*100+i.day()<j.month()*100+j.day();
}

IMPLEMENT_DYNAMIC_CLASS(PublicholidayDialog, wxDialog)

BEGIN_EVENT_TABLE(PublicholidayDialog, wxDialog)
    EVT_BUTTON(wxID_OK, PublicholidayDialog::OnOk)
    EVT_MENU(ID_PHNEW, PublicholidayDialog::OnAdd)
    EVT_MENU(ID_PHEDIT, PublicholidayDialog::OnEdit)
    EVT_MENU(ID_PHREMOVE, PublicholidayDialog::OnRemove)
    EVT_LIST_ITEM_RIGHT_CLICK(ID_PHLIST, PublicholidayDialog::OnListRightClick)
    EVT_LIST_ITEM_ACTIVATED(ID_PHLIST, PublicholidayDialog::OnListActivated)
    EVT_LIST_ITEM_SELECTED(ID_PHLIST, PublicholidayDialog::OnListSelected)
    EVT_LIST_ITEM_DESELECTED(ID_PHLIST, PublicholidayDialog::OnListDeselected)
    EVT_BUTTON(ID_PHLOAD, PublicholidayDialog::OnLoad)
    EVT_CHECKBOX(ID_PHREMIND, PublicholidayDialog::OnRemindHoliday)
END_EVENT_TABLE()

PublicholidayDialog::PublicholidayDialog(wxWindow *parent)
: wxDialog(parent, wxID_ANY, _("Public holidays"))
{
    m_holidays = iengine->holidays();
    std::sort(m_holidays.begin(),m_holidays.end(),sortHolidays);

    m_index = -1;

    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);
#if defined (__WXGTK__) || defined (__WXMAC__)
    m_panel = new wxPanel(this, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxBORDER_SUNKEN);
#else
    m_panel = new wxPanel(this);
#endif
    wxBoxSizer *sizer1 = new wxBoxSizer(wxVERTICAL);
    m_list = new wxListCtrl(m_panel, ID_PHLIST, wxDefaultPosition, wxDefaultSize, wxLC_REPORT|wxLC_HRULES|wxLC_VRULES|wxLC_SINGLE_SEL);
    m_list->InsertColumn(0, _("Day"));
    m_list->InsertColumn(1, _("Month"));
    m_list->InsertColumn(2, _("Name"));
    m_list->SetColumnWidth(2, 300);
    sizer1->Add(m_list, 1, wxALL|wxEXPAND, 0);
    m_panel->SetSizer(sizer1);
    sizer1->Fit(m_panel);
    sizer1->SetSizeHints(m_panel);
    mainSizer->Add(m_panel, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    m_bar = new dxToolBar(this);
    m_bar->AddTool(ID_PHNEW, wxEmptyString, ICO_ADD, ICO_GADD, wxITEM_NORMAL, _("Add holiday"));
    m_bar->AddTool(ID_PHEDIT, wxEmptyString, ICO_EDIT, ICO_GEDIT, wxITEM_NORMAL, _("Edit holiday"));
    m_bar->AddTool(ID_PHREMOVE, wxEmptyString,ICO_DELETE, ICO_GDELETE, wxITEM_NORMAL, _("Remove holiday"));
    m_bar->Realize();
    mainSizer->Add(m_bar, 0, wxALL, 5);

    m_easternMonday = new wxCheckBox(this, wxID_ANY, _("Easter Monday"));
    m_easternMonday->SetValue(dxsettings.easternMonday());
    mainSizer->Add(m_easternMonday, 0, wxALL, 5);
    m_easternFriday = new wxCheckBox(this, wxID_ANY, _("Good Friday"));
    m_easternFriday->SetValue(dxsettings.easternFriday());
    mainSizer->Add(m_easternFriday, 0, wxALL, 5);

    if(dxsettings.locale().StartsWith("cs"))
    {
        wxButton *btn = new wxButton(this, ID_PHLOAD, wxString::FromUTF8("Načíst státní svátky"));
        mainSizer->Add(btn, 0, wxALL, 5);
    }

    mainSizer->AddSpacer(5);

    wxGridBagSizer *gridSizer = new wxGridBagSizer(5, 5);
    m_remindHoliday = new wxCheckBox(this, ID_PHREMIND, _("Remind holidays"));
    m_remindHoliday->SetValue(dxsettings.remindHoliday());
    gridSizer->Add(m_remindHoliday, wxGBPosition(0,0), wxGBSpan(1,2), wxEXPAND|wxALL, 5);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Days before:")), wxGBPosition(1,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_remindHolidayDays = new wxSpinCtrl(this);
    m_remindHolidayDays->SetRange(0,31);
    m_remindHolidayDays->SetValue(dxsettings.remindHolidayDays());
    m_remindHolidayDays->Enable(dxsettings.remindHoliday());
    gridSizer->Add(m_remindHolidayDays, wxGBPosition(1,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Time:")), wxGBPosition(2,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_remindHolidayTime = new dxTimePickerCtrl(this, dxsettings.remindHolidayTime());
    m_remindHolidayTime->Enable(dxsettings.remindHoliday());
    gridSizer->Add(m_remindHolidayTime, wxGBPosition(2,1), wxDefaultSpan, wxEXPAND|wxALL, 5);

    mainSizer->Add(gridSizer, 1, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);

    fillList();
    enableTools();
}

void PublicholidayDialog::OnOk(wxCommandEvent& /*event*/)
{
    iengine->setHolidays(m_holidays);
    dxsettings.setEasternMonday(m_easternMonday->GetValue());
    dxsettings.setEasternFriday(m_easternFriday->GetValue());
    dxsettings.setRemindHoliday(m_remindHoliday->GetValue());
    dxsettings.setRemindHolidayDays(m_remindHolidayDays->GetValue());
    dxsettings.setRemindHolidayTime(m_remindHolidayTime->time());
    EndModal(wxID_OK);
    return;
}

void PublicholidayDialog::OnAdd(wxCommandEvent &/*event*/)
{
    HolidayDialog dialog(this);
    if(dialog.ShowModal()==wxID_OK)
    {
        m_holidays.push_back(iengine->addHoliday(dialog.name(), dialog.day(), dialog.month()));
        std::sort(m_holidays.begin(),m_holidays.end(),sortHolidays);
        fillList();
        m_list->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
        m_index = 0;
        enableTools();
    }

}

void PublicholidayDialog::OnEdit(wxCommandEvent &/*event*/)
{
    HolidayDialog dialog(this, m_holidays[m_index]);
    if(dialog.ShowModal()==wxID_OK)
    {
        wxInt64 id =  m_holidays[m_index].id();
        updateHoliday(id, dialog.name(), dialog.day(), dialog.month());
        std::sort(m_holidays.begin(),m_holidays.end(),sortHolidays);
        m_index = 0;
        std::vector<Holiday>::iterator i;
        for(i = m_holidays.begin(); i != m_holidays.end(); ++i, ++m_index)
        {
            if((*i).id() == id)
            {
                break;
            }
        }
        m_list->SetItemState(m_index, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
    }
}

void PublicholidayDialog::OnRemove(wxCommandEvent &/*event*/)
{
    if(m_index < 0) return;
    wxMessageDialog dialog(this, _("Do you really want to delete the Holiday?")+"\n"+m_holidays[m_index].name(), _("Confirm Holiday Deletion"), wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION);
    if(dialog.ShowModal() == wxID_YES)
    {
        std::vector<Holiday>::iterator i;
        for(i = m_holidays.begin(); i != m_holidays.end(); ++i)
        {
            if((*i).id() == m_holidays[m_index].id())
            {
                m_holidays.erase(i);
                std::sort(m_holidays.begin(),m_holidays.end(),sortHolidays);
                fillList();
                if(m_holidays.size())
                {
                    m_list->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
                    m_index = 0;
                }
                else
                {
                    m_index = -1;
                }
                enableTools();
                break;
            }
        }
    }
}

void PublicholidayDialog::OnListRightClick(wxListEvent &event)
{
    m_index = event.GetIndex();
    wxMenu popup;
    popup.Append(ID_PHNEW, _("Add holiday"));
    if(m_index != -1)
    {
        popup.AppendSeparator();
        popup.Append(ID_PHEDIT, _("Edit holiday"));
        popup.Append(ID_PHREMOVE, _("Remove holiday"));

    }
    PopupMenu(&popup, event.GetPoint()+m_panel->GetPosition());
}

void PublicholidayDialog::OnListActivated(wxListEvent &event)
{
    m_index = event.GetIndex();
    HolidayDialog dialog(this, m_holidays[m_index]);
    if(dialog.ShowModal()==wxID_OK)
    {
        wxInt64 id =  m_holidays[m_index].id();
        updateHoliday(id, dialog.name(), dialog.day(), dialog.month());
        std::sort(m_holidays.begin(),m_holidays.end(),sortHolidays);
        m_index = 0;
        std::vector<Holiday>::iterator i;
        for(i = m_holidays.begin(); i != m_holidays.end(); ++i, ++m_index)
        {
            if((*i).id() == id)
            {
                break;
            }
        }
        m_list->SetItemState(m_index, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
    }
}

void PublicholidayDialog::OnListSelected(wxListEvent &event)
{
    m_index = event.GetIndex();
    enableTools();
}

void PublicholidayDialog::OnListDeselected(wxListEvent &/*event*/)
{
    m_index = -1;
    enableTools();
}

void PublicholidayDialog::fillList()
{
    m_list->DeleteAllItems();
    m_list->Hide();
    long row = 0;
    std::vector<Holiday>::iterator i;
    for(i = m_holidays.begin(); i != m_holidays.end(); ++i, row++)
    {
        wxListItem item;
        item.SetId(row);
        item.SetText(wxString::Format("%02d",(*i).day()));
        m_list->InsertItem(item);
        m_list->SetItem(row, 1, wxDateTime::GetMonthName((*i).month()));
        m_list->SetItem(row, 2, (*i).name());
    }
    m_list->Show();
}

void PublicholidayDialog::enableTools()
{
    if(m_index == -1)
    {
        m_bar->EnableTool(ID_PHNEW, true);
        m_bar->EnableTool(ID_PHEDIT, false);
        m_bar->EnableTool(ID_PHREMOVE, false);
    }
    else
    {
        m_bar->EnableTool(ID_PHNEW, true);
        m_bar->EnableTool(ID_PHEDIT, true);
        m_bar->EnableTool(ID_PHREMOVE, true);
    }
}

void PublicholidayDialog::updateHoliday(wxInt64 id, const wxString &name, int day, wxDateTime::Month month)
{
    std::vector<Holiday>::iterator i;
    for(i = m_holidays.begin(); i != m_holidays.end(); ++i)
    {
        if((*i).id() == id)
        {
            m_holidays.erase(i);
            m_holidays.push_back(Holiday(id,name,day,month,wxDateTime::GetCurrentYear()-1));
            std::sort(m_holidays.begin(),m_holidays.end(),sortHolidays);
            fillList();
            break;
        }
    }
}

void PublicholidayDialog::OnLoad(wxCommandEvent &/*event*/)
{
    if(dxsettings.locale().StartsWith("cs"))
    {
        m_easternFriday->SetValue(true);
        m_easternMonday->SetValue(true);
        m_holidays.clear();
        m_holidays.push_back(Holiday(1,wxString::FromUTF8("Nový rok"),1,wxDateTime::Month::Jan,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(2,wxString::FromUTF8("Den upálení mistra Jana Husa"),6,wxDateTime::Month::Jul,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(3,wxString::FromUTF8("Svátek práce"),1,wxDateTime::Month::May,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(4,wxString::FromUTF8("Den vítězství"),8,wxDateTime::Month::May,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(5,wxString::FromUTF8("Den slovanských věrozvěstů Cyrila a Metoděje"),5,wxDateTime::Month::Jul,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(6,wxString::FromUTF8("Den české státnosti"),28,wxDateTime::Month::Sep,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(7,wxString::FromUTF8("Den vzniku samostatného Československa"),28,wxDateTime::Month::Oct,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(8,wxString::FromUTF8("Den boje za svobodu a demokracii"),17,wxDateTime::Month::Nov,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(9,wxString::FromUTF8("Štědrý den"),24,wxDateTime::Month::Dec,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(10,wxString::FromUTF8("1. svátek vánoční"),25,wxDateTime::Month::Dec,wxDateTime::GetCurrentYear()-1));
        m_holidays.push_back(Holiday(11,wxString::FromUTF8("2. svátek vánoční"),26,wxDateTime::Month::Dec,wxDateTime::GetCurrentYear()-1));
        std::sort(m_holidays.begin(),m_holidays.end(),sortHolidays);
        fillList();
        m_list->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
        m_index = 0;
        enableTools();
    }
}

void PublicholidayDialog::OnRemindHoliday(wxCommandEvent &/*event*/)
{
    if(m_remindHoliday->GetValue())
    {
        m_remindHolidayDays->Enable();
        m_remindHolidayTime->Enable();
    }
    else
    {
        m_remindHolidayDays->Disable();
        m_remindHolidayTime->Enable(false);
    }
}

IMPLEMENT_DYNAMIC_CLASS(HolidayDialog, wxDialog)

BEGIN_EVENT_TABLE(HolidayDialog, wxDialog)
    EVT_CHOICE(ID_PHCOMBO, HolidayDialog::OnCombo)
    EVT_BUTTON(wxID_OK, HolidayDialog::OnOk)
END_EVENT_TABLE()

HolidayDialog::HolidayDialog(wxWindow *parent)
    : wxDialog(parent, wxID_ANY, _("New Holiday"))
{
    createControls();
    m_month->SetSelection(0);
}

HolidayDialog::HolidayDialog(wxWindow *parent, const Holiday &holiday)
    : wxDialog(parent, wxID_ANY, _("Edit Holiday"))
{
    createControls();
    m_day->SetValue(holiday.day());
    m_day->SetRange(1, dxutils::lastdayofmonth(holiday.month(),wxDateTime::GetCurrentYear()));
    m_month->SetSelection(holiday.month());
    m_name->SetValue(holiday.name());
}

int HolidayDialog::day() const
{
    return m_day->GetValue();
}

wxDateTime::Month HolidayDialog::month() const
{
    return static_cast<wxDateTime::Month>(m_month->GetSelection());
}

wxString HolidayDialog::name() const
{
    return m_name->GetValue();
}

void HolidayDialog::createControls()
{
    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);

    wxGridBagSizer *gridSizer = new wxGridBagSizer(5, 5);
    int i=0;
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Day:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_day = new wxSpinCtrl(this, wxID_ANY);
    m_day->SetRange(1,31);
    gridSizer->Add(m_day, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    i++;
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Month:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_month = new wxChoice(this, ID_PHCOMBO);
    m_month->Append(_("January"));
    m_month->Append(_("February"));
    m_month->Append(_("March"));
    m_month->Append(_("April"));
    m_month->Append(_("May"));
    m_month->Append(_("June"));
    m_month->Append(_("July"));
    m_month->Append(_("August"));
    m_month->Append(_("September"));
    m_month->Append(_("October"));
    m_month->Append(_("November"));
    m_month->Append(_("December"));
    gridSizer->Add(m_month, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    i++;
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Name:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_name = new wxTextCtrl(this, wxID_ANY, wxEmptyString, wxDefaultPosition, wxSize(300,-1));
    gridSizer->Add(m_name, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    mainSizer->Add(gridSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);
}

void HolidayDialog::OnCombo(wxCommandEvent &/*event*/)
{
    m_day->SetRange(1,dxutils::lastdayofmonth(static_cast<wxDateTime::Month>(m_month->GetSelection()), wxDateTime::GetCurrentYear()));
}

void HolidayDialog::OnOk(wxCommandEvent &/*event*/)
{
    if(m_name->GetValue().IsEmpty())
    {
        wxMessageDialog dialog(this, _("Holiday has to have some name"), _("Warning"), wxOK|wxICON_WARNING);
        dialog.ShowModal();
        m_name->SetFocus();
        return;
    }
    EndModal(wxID_OK);
    return;
}
