/***************************************************************
 * Name:      eventfilter.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2021
 * License:   GPL3
 **************************************************************/

#include <wx/gbsizer.h>
#include "eventfilter.h"
#include "dxsettings.h"
#include "dxutils.h"

IMPLEMENT_DYNAMIC_CLASS(EventFilter, wxDialog)

BEGIN_EVENT_TABLE(EventFilter, wxDialog)
    EVT_BUTTON(wxID_OK, EventFilter::OnOk)
    EVT_BUTTON(wxID_CANCEL, EventFilter::OnCancel)
    EVT_BUTTON(wxID_APPLY, EventFilter::OnApply)
END_EVENT_TABLE()

EventFilter::EventFilter(wxWindow *parent)
    : wxDialog(parent, wxID_ANY, _("Events filter")), m_reminders(static_cast<dxremindersFrame*>(parent))
{
    m_filters.Add(Filter(FA_CONTAINS, false, wxEmptyString));
    m_filters.Add(Filter(FA_GREATEREQUAL, false, wxDateTime::Now()));
    m_filters.Add(Filter(FA_EQUAL, false, Reminder(0)));
    m_filters.Add(Filter(FA_EQUAL, false, 0));
    m_filters.Add(Filter(FA_EQUAL, false, wxDateSpan(0), wxTimeSpan(0)));
    m_filters.Add(Filter(FA_EQUAL, false, 1));
    createControls();
    fillControls();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
}

EventFilter::EventFilter(wxWindow *parent, dxFilterArray filters)
    : wxDialog(parent, wxID_ANY, _("Events filter")), m_reminders(static_cast<dxremindersFrame*>(parent)), m_filters(filters)
{
    createControls();
    fillControls();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
}

dxFilterArray EventFilter::filters() const
{
    return m_filters;
}

void EventFilter::createControls()
{
    m_mainSizer = new wxBoxSizer(wxVERTICAL);

    m_book = new wxListbook(this, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxLB_LEFT);

    wxPanel *event = new wxPanel(m_book);
    wxBoxSizer *box1 = new wxBoxSizer(wxHORIZONTAL);
    wxGridBagSizer *gridSizer1 = new wxGridBagSizer(5, 5);
    m_eventAction = new wxChoice(event, wxID_ANY);
    m_eventAction->Append(_("Equal"), new FilterAction(FA_EQUAL));
    m_eventAction->Append(_("Contains"), new FilterAction(FA_CONTAINS));
    gridSizer1->Add(m_eventAction, wxGBPosition(0,0), wxDefaultSpan, wxEXPAND);
    m_eventValue = new wxTextCtrl(event, wxID_ANY);
    m_eventValue->SetMinSize(wxSize(250,-1));
    gridSizer1->Add(m_eventValue, wxGBPosition(0,1), wxDefaultSpan, wxEXPAND|wxALL);
    m_eventUsage = new wxCheckBox(event, wxID_ANY, _("Use"));
    gridSizer1->Add(m_eventUsage, wxGBPosition(1,0), wxGBSpan(1,2), wxEXPAND);
    box1->Add(gridSizer1, 1, wxEXPAND|wxALL, 5);
    event->SetSizer(box1);
    box1->Fit(event);
    box1->SetSizeHints(event);
    m_book->AddPage(event, _("Event"));


    if(dxsettings.colDateShow())
    {
        wxPanel *date = new wxPanel(m_book);
        wxBoxSizer *vbox2 = new wxBoxSizer(wxVERTICAL);
        wxBoxSizer *box2 = new wxBoxSizer(wxHORIZONTAL);
        m_dateAction = new wxChoice(date, wxID_ANY);
        m_dateAction->Append(_("Earlier than"), new FilterAction(FA_LESS));
        m_dateAction->Append(_("Earlier than or Equal"), new FilterAction(FA_LESSEQUAL));
        m_dateAction->Append(_("Equal"), new FilterAction(FA_EQUAL));
        m_dateAction->Append(_("Later than or Equal"), new FilterAction(FA_GREATEREQUAL));
        m_dateAction->Append(_("Later than"), new FilterAction(FA_GREATER));
        m_dateValue = new dxCalendarPickerCtrl(date, wxDateTime::Now());
        m_dateUsage = new wxCheckBox(date, wxID_ANY, _("Use"));
        box2->Add(m_dateAction, 1, wxALL, 5);
        box2->Add(m_dateValue, 1, wxALL, 5);
        vbox2->Add(box2);
        vbox2->Add(m_dateUsage, 0, wxALL);
        date->SetSizer(vbox2);
        vbox2->Fit(date);
        vbox2->SetSizeHints(date);
        m_book->AddPage(date, _("Date"));

        wxPanel *now = new wxPanel(m_book);
        wxBoxSizer *box3 = new wxBoxSizer(wxHORIZONTAL);
        wxGridBagSizer *gridSizer3 = new wxGridBagSizer(5, 5);
        gridSizer3->Add(new wxStaticText(now, wxID_ANY, _("Years:")), wxGBPosition(0,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_timefilterYears = new wxSpinCtrl(now);
        gridSizer3->Add(m_timefilterYears, wxGBPosition(0,1), wxDefaultSpan, wxEXPAND);
        gridSizer3->Add(new wxStaticText(now, wxID_ANY, _("Months:")), wxGBPosition(1,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_timefilterMonths = new wxSpinCtrl(now);
        m_timefilterMonths->SetRange(0,11);
        gridSizer3->Add(m_timefilterMonths, wxGBPosition(1,1), wxDefaultSpan, wxEXPAND);
        gridSizer3->Add(new wxStaticText(now, wxID_ANY, _("Weeks:")), wxGBPosition(2,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_timefilterWeeks = new wxSpinCtrl(now);
        m_timefilterWeeks->SetRange(0,4);
        gridSizer3->Add(m_timefilterWeeks, wxGBPosition(2,1), wxDefaultSpan, wxEXPAND);
        gridSizer3->Add(new wxStaticText(now, wxID_ANY, _("Days:")), wxGBPosition(3,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_timefilterDays = new wxSpinCtrl(now);
        m_timefilterDays->SetRange(0,31);
        gridSizer3->Add(m_timefilterDays, wxGBPosition(3,1), wxDefaultSpan, wxEXPAND);
        gridSizer3->Add(new wxStaticText(now, wxID_ANY, _("Hours:")), wxGBPosition(4,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_timefilterHours = new wxSpinCtrl(now);
        m_timefilterHours->SetRange(0,23);
        gridSizer3->Add(m_timefilterHours, wxGBPosition(4,1), wxDefaultSpan, wxEXPAND);
        gridSizer3->Add(new wxStaticText(now, wxID_ANY, _("Minutes:")), wxGBPosition(5,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_timefilterMinutes = new wxSpinCtrl(now);
        m_timefilterMinutes->SetRange(0,59);
        gridSizer3->Add(m_timefilterMinutes, wxGBPosition(5,1), wxDefaultSpan, wxEXPAND);
        m_timefilterUsage = new wxCheckBox(now, wxID_ANY, _("Use"));
        gridSizer3->Add(m_timefilterUsage, wxGBPosition(6,0), wxGBSpan(1,2), wxEXPAND);
        box3->Add(gridSizer3, 1, wxEXPAND|wxALL, 5);
        now->SetSizer(box3);
        box3->Fit(now);
        box3->SetSizeHints(now);
        m_book->AddPage(now, _("From now"));
    }

    if(dxsettings.colReminderShow())
    {
        wxPanel *reminder = new wxPanel(m_book);
        wxBoxSizer *box4 = new wxBoxSizer(wxHORIZONTAL);
        wxGridBagSizer *gridSizer4 = new wxGridBagSizer(5, 5);
        m_reminderAction = new wxChoice(reminder, wxID_ANY);
        m_reminderAction->Append(_("Less than"), new FilterAction(FA_LESS));
        m_reminderAction->Append(_("Less than or Equal"), new FilterAction(FA_LESSEQUAL));
        m_reminderAction->Append(_("Equal"), new FilterAction(FA_EQUAL));
        m_reminderAction->Append(_("Greater than or Equal"), new FilterAction(FA_GREATEREQUAL));
        m_reminderAction->Append(_("Greater than"), new FilterAction(FA_GREATER));
        gridSizer4->Add(m_reminderAction, wxGBPosition(0,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        wxGridBagSizer *rmdSizer = new wxGridBagSizer(5, 5);
        rmdSizer->Add(new wxStaticText(reminder, wxID_ANY, _("Days:")), wxGBPosition(0,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_reminderDays = new wxSpinCtrl(reminder);
        rmdSizer->Add(m_reminderDays, wxGBPosition(0,1), wxDefaultSpan, wxEXPAND);
        rmdSizer->Add(new wxStaticText(reminder, wxID_ANY, _("Hours:")), wxGBPosition(1,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_reminderHours = new wxSpinCtrl(reminder);
        m_reminderHours->SetRange(0,23);
        rmdSizer->Add(m_reminderHours, wxGBPosition(1,1), wxDefaultSpan, wxEXPAND);
        rmdSizer->Add(new wxStaticText(reminder, wxID_ANY, _("Minutes:")), wxGBPosition(2,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_reminderMins = new wxSpinCtrl(reminder);
        m_reminderMins->SetRange(0,59);
        rmdSizer->Add(m_reminderMins, wxGBPosition(2,1), wxDefaultSpan, wxEXPAND);
        gridSizer4->Add(rmdSizer, wxGBPosition(0,1), wxDefaultSpan);
        m_reminderUsage = new wxCheckBox(reminder, wxID_ANY, _("Use"));
        gridSizer4->Add(m_reminderUsage, wxGBPosition(1,0), wxGBSpan(1,2), wxEXPAND);
        box4->Add(gridSizer4, 1, wxEXPAND|wxALL, 5);
        reminder->SetSizer(box4);
        box4->Fit(reminder);
        box4->SetSizeHints(reminder);
        m_book->AddPage(reminder, _("Reminder"));
    }

    if(dxsettings.colRecurrenceShow())
    {
        wxPanel *recurrence = new wxPanel(m_book);
        wxBoxSizer *box5 = new wxBoxSizer(wxHORIZONTAL);
        wxGridBagSizer *gridSizer5 = new wxGridBagSizer(5, 5);
        gridSizer5->Add(new wxStaticText(recurrence, wxID_ANY, _("Equal")), wxGBPosition(0,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_recurrenceValue = new wxChoice(recurrence, wxID_ANY, wxDefaultPosition, wxDefaultSize, dxutils::recurrences());
        m_recurrenceValue->SetSelection(dxutils::recurrenceTypeToSelection(dxsettings.defaultRecurrence()));
        gridSizer5->Add(m_recurrenceValue, wxGBPosition(0,1), wxDefaultSpan);
        m_recurrenceUsage = new wxCheckBox(recurrence, wxID_ANY, _("Use"));
        gridSizer5->Add(m_recurrenceUsage, wxGBPosition(1,0), wxGBSpan(1,2), wxEXPAND);
        box5->Add(gridSizer5, 1, wxEXPAND|wxALL, 5);
        recurrence->SetSizer(box5);
        box5->Fit(recurrence);
        box5->SetSizeHints(recurrence);
        m_book->AddPage(recurrence, _("Recurrence"));
    }

    wxPanel *hidden = new wxPanel(m_book);
    wxBoxSizer *box6 = new wxBoxSizer(wxHORIZONTAL);
    m_hiddenfilterUsage = new wxCheckBox(hidden, wxID_ANY, _("Use"));
    box6->Add(m_hiddenfilterUsage, 0, wxALL);
    hidden->SetSizer(box6);
    box6->Fit(hidden);
    box6->SetSizeHints(hidden);
    m_book->AddPage(hidden, _("Show hidden event(s)"));

    m_mainSizer->Add(m_book, 1, wxEXPAND|wxALL, 5);
    m_mainSizer->AddSpacer(SIZER_SPACE);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    wxButton* applyButton = new wxButton(this, wxID_APPLY, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(applyButton);
    btnSizer->Realize();
    m_mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(m_mainSizer);
}

void EventFilter::fillControls()
{
    if(m_filters.GetCount() == 6)
    {
        if(m_filters[0].type() == FT_TEXT)
        {
            m_eventAction->SetSelection(m_filters[0].action()==FA_EQUAL?0:1);
            m_eventValue->ChangeValue(m_filters[0].text());
            m_eventUsage->SetValue(m_filters[0].active());
        }
        if(m_filters[1].type() == FT_DATUM && dxsettings.colDateShow())
        {
            m_dateAction->SetSelection(m_filters[1].action());
            m_dateValue->setDate(m_filters[1].datum());
            m_dateUsage->SetValue(m_filters[1].active());
        }
        if(m_filters[2].type() == FT_REMINDER && dxsettings.colReminderShow())
        {
            m_reminderAction->SetSelection(m_filters[2].action());
            m_reminderDays->SetValue(m_filters[2].reminder().days());
            m_reminderHours->SetValue(m_filters[2].reminder().hours());
            m_reminderMins->SetValue(m_filters[2].reminder().mins());
            m_reminderUsage->SetValue(m_filters[2].active());
        }
        if(m_filters[3].type() == FT_INTEGER && dxsettings.colRecurrenceShow())
        {
            m_recurrenceValue->SetSelection(dxutils::recurrenceTypeToSelection(m_filters[3].intvalue()));
            m_recurrenceUsage->SetValue(m_filters[3].active());
        }
        if(m_filters[4].type() == FT_TIME && dxsettings.colDateShow())
        {
            m_timefilterYears->SetValue(m_filters[4].datespan().GetYears());
            m_timefilterMonths->SetValue(m_filters[4].datespan().GetMonths());
            m_timefilterWeeks->SetValue(m_filters[4].datespan().GetWeeks());
            m_timefilterDays->SetValue(m_filters[4].datespan().GetDays());
            m_timefilterHours->SetValue(m_filters[4].timespan().GetHours());
            m_timefilterMinutes->SetValue(m_filters[4].timespan().GetMinutes()-m_filters[4].timespan().GetHours()*60);
            m_timefilterUsage->SetValue(m_filters[4].active());
        }
        if(m_filters[5].type() == FT_INTEGER)
        {
            m_hiddenfilterUsage->SetValue(m_filters[5].active());
        }
    }
    m_book->SetSelection(wxMin(m_book->GetPageCount()-1,dxsettings.lastFilter()));
}

void EventFilter::OnOk(wxCommandEvent& /*event*/)
{
    if(m_filters[0].type() == FT_TEXT)
    {
        m_filters[0].setAction(*static_cast<FilterAction*>(m_eventAction->GetClientData(static_cast<unsigned int>(m_eventAction->GetSelection()))));
        m_filters[0].setText(m_eventValue->GetValue());
        m_filters[0].setActive(m_eventUsage->GetValue());
    }

    if(m_filters[1].type() == FT_DATUM && dxsettings.colDateShow())
    {
        m_filters[1].setAction(*static_cast<FilterAction*>(m_dateAction->GetClientData(static_cast<unsigned int>(m_dateAction->GetSelection()))));
        m_filters[1].setDatum(m_dateValue->date());
        m_filters[1].setActive(m_dateUsage->GetValue());
    }

    if(m_filters[2].type() == FT_REMINDER && dxsettings.colReminderShow())
    {
        m_filters[2].setAction(*static_cast<FilterAction*>(m_reminderAction->GetClientData(static_cast<unsigned int>(m_reminderAction->GetSelection()))));
        m_filters[2].setReminder(Reminder(m_reminderDays->GetValue(),m_reminderHours->GetValue(),m_reminderMins->GetValue()));
        m_filters[2].setActive(m_reminderUsage->GetValue());
    }

    if(m_filters[3].type() == FT_INTEGER && dxsettings.colRecurrenceShow())
    {
        m_filters[3].setAction(FA_EQUAL);
        m_filters[3].setIntvalue(dxutils::recurrenceSelectionToType(m_recurrenceValue->GetSelection()));
        m_filters[3].setActive(m_recurrenceUsage->GetValue());
    }

    if(m_filters[4].type() == FT_TIME && dxsettings.colDateShow())
    {
        m_filters[4].setAction(FA_EQUAL);
        m_filters[4].setDatespan(wxDateSpan(m_timefilterYears->GetValue(),m_timefilterMonths->GetValue(),m_timefilterWeeks->GetValue(),m_timefilterDays->GetValue()));
        m_filters[4].setTimespan(wxTimeSpan(m_timefilterHours->GetValue(),m_timefilterMinutes->GetValue()));
        m_filters[4].setActive(m_timefilterUsage->GetValue());
    }
    if(m_filters[5].type() == FT_INTEGER)
    {
        m_filters[5].setActive(m_hiddenfilterUsage->GetValue());
    }
    dxsettings.setLastFilter(m_book->GetSelection());
    EndModal(wxID_OK);
}

void EventFilter::OnCancel(wxCommandEvent& /*event*/)
{
    EndModal(wxID_CANCEL);
}

void EventFilter::OnApply(wxCommandEvent &/*event*/)
{
    if(m_filters[0].type() == FT_TEXT)
    {
        m_filters[0].setAction(*static_cast<FilterAction*>(m_eventAction->GetClientData(static_cast<unsigned int>(m_eventAction->GetSelection()))));
        m_filters[0].setText(m_eventValue->GetValue());
        m_filters[0].setActive(m_eventUsage->GetValue());
    }

    if(m_filters[1].type() == FT_DATUM && dxsettings.colDateShow())
    {
        m_filters[1].setAction(*static_cast<FilterAction*>(m_dateAction->GetClientData(static_cast<unsigned int>(m_dateAction->GetSelection()))));
        m_filters[1].setDatum(m_dateValue->date());
        m_filters[1].setActive(m_dateUsage->GetValue());
    }

    if(m_filters[2].type() == FT_REMINDER && dxsettings.colReminderShow())
    {
        m_filters[2].setAction(*static_cast<FilterAction*>(m_reminderAction->GetClientData(static_cast<unsigned int>(m_reminderAction->GetSelection()))));
        m_filters[2].setReminder(Reminder(m_reminderDays->GetValue(),m_reminderHours->GetValue(),m_reminderMins->GetValue()));
        m_filters[2].setActive(m_reminderUsage->GetValue());
    }

    if(m_filters[3].type() == FT_INTEGER && dxsettings.colRecurrenceShow())
    {
        m_filters[3].setAction(FA_EQUAL);
        m_filters[3].setIntvalue(dxutils::recurrenceSelectionToType(m_recurrenceValue->GetSelection()));
        m_filters[3].setActive(m_recurrenceUsage->GetValue());
    }

    if(m_filters[4].type() == FT_TIME && dxsettings.colDateShow())
    {
        m_filters[4].setAction(FA_EQUAL);
        m_filters[4].setDatespan(wxDateSpan(m_timefilterYears->GetValue(),m_timefilterMonths->GetValue(),m_timefilterWeeks->GetValue(),m_timefilterDays->GetValue()));
        m_filters[4].setTimespan(wxTimeSpan(m_timefilterHours->GetValue(),m_timefilterMinutes->GetValue()));
        m_filters[4].setActive(m_timefilterUsage->GetValue());
    }
    if(m_filters[5].type() == FT_INTEGER)
    {
        m_filters[5].setActive(m_hiddenfilterUsage->GetValue());
    }

    m_reminders->applyFilter(m_filters);
}

