/***************************************************************
 * Name:      eventdialog.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2013
 * License:   GPL3
 **************************************************************/

#include <wx/gbsizer.h>
#include <wx/numdlg.h>
#include "dxdefs.h"
#include "eventdialog.h"
#include "dxicons.h"
#include "dxsettings.h"
#include "dxutils.h"

IMPLEMENT_DYNAMIC_CLASS(ReminderDialog, wxDialog)

ReminderDialog::ReminderDialog(wxWindow* parent, wxInt64 reminder)
: wxDialog(parent, wxID_ANY, _("Reminder time"))
{
    wxInt64 mins = reminder;
    wxInt64 days = mins/(60*24);
    mins -= days*60*24;
    wxInt64 hours = mins/60;
    mins -= hours*60;

    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);

    wxBoxSizer *rmdSizer = new wxBoxSizer(wxHORIZONTAL);
    rmdSizer->Add(new wxStaticText(this, wxID_ANY, _("Days:")), 0, wxALIGN_CENTER_VERTICAL, 1);
    m_days = new wxSpinCtrl(this);
    m_days->SetValue(days);
    rmdSizer->Add(m_days);
    rmdSizer->Add(new wxStaticText(this, wxID_ANY, _("Hours:")), 0, wxALIGN_CENTER_VERTICAL, 1);
    m_hours = new wxSpinCtrl(this);
    m_hours->SetRange(0,23);
    m_hours->SetValue(hours);
    rmdSizer->Add(m_hours);
    rmdSizer->Add(new wxStaticText(this, wxID_ANY, _("Minutes:")), 0, wxALIGN_CENTER_VERTICAL, 1);
    m_mins = new wxSpinCtrl(this);
    m_mins->SetRange(0,59);
    m_mins->SetValue(mins);
    rmdSizer->Add(m_mins);
    mainSizer->Add(rmdSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);
}

wxInt64 ReminderDialog::reminder()
{
    return m_days->GetValue()*60*24+m_hours->GetValue()*60+m_mins->GetValue();
}

IMPLEMENT_DYNAMIC_CLASS(RecurrenceDialog, wxDialog)

RecurrenceDialog::RecurrenceDialog(wxWindow* parent, int days, int months, int hours, int minutes)
: wxDialog(parent, wxID_ANY, _("Recurrence"))
{
    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);

    wxBoxSizer *rmdSizer = new wxBoxSizer(wxHORIZONTAL);
    rmdSizer->Add(new wxStaticText(this, wxID_ANY, _("Months:")), 0, wxALIGN_CENTER_VERTICAL, 1);
    m_moths = new wxSpinCtrl(this);
    m_moths->SetValue(months);
    rmdSizer->Add(m_moths);
    rmdSizer->Add(new wxStaticText(this, wxID_ANY, _("Days:")), 0, wxALIGN_CENTER_VERTICAL, 1);
    m_days = new wxSpinCtrl(this);
    m_days->SetRange(0,31);
    m_days->SetValue(days);
    rmdSizer->Add(m_days);
    rmdSizer->Add(new wxStaticText(this, wxID_ANY, _("Hours:")), 0, wxALIGN_CENTER_VERTICAL, 1);
    m_hours = new wxSpinCtrl(this);
    m_hours->SetRange(0,23);
    m_hours->SetValue(hours);
    rmdSizer->Add(m_hours);
    rmdSizer->Add(new wxStaticText(this, wxID_ANY, _("Minutes:")), 0, wxALIGN_CENTER_VERTICAL, 1);
    m_minutes = new wxSpinCtrl(this);
    m_minutes->SetRange(0,59);
    m_minutes->SetValue(minutes);
    rmdSizer->Add(m_minutes);
    mainSizer->Add(rmdSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);
}

int RecurrenceDialog::days()
{
    return m_days->GetValue();
}

int RecurrenceDialog::months()
{
    return m_moths->GetValue();
}

int RecurrenceDialog::hours()
{
    return m_hours->GetValue();
}

int RecurrenceDialog::minutes()
{
    return m_minutes->GetValue();
}

IMPLEMENT_DYNAMIC_CLASS(EventDialog, wxDialog)

BEGIN_EVENT_TABLE(EventDialog, wxDialog)
    EVT_BUTTON(wxID_OK, EventDialog::OnOk)
    EVT_BUTTON(wxID_CANCEL, EventDialog::OnCancel)
    EVT_CHOICE(ID_REMINDERCHOICE, EventDialog::OnReminder)
    EVT_CHOICE(ID_RECURRENCECHOICE, EventDialog::OnRecurrence)
    EVT_BUTTON(ID_REMINDEREDIT, EventDialog::OnReminderEdit)
    EVT_BUTTON(ID_RECURRENCEEDIT, EventDialog::OnRecurrenceEdit)
    EVT_CHOICE(ID_UNTILCHOICE, EventDialog::OnUntil)
    EVT_BUTTON(ID_UNTILEDIT, EventDialog::OnUntilEdit)
    EVT_TEXT(ID_DESCRIPTION, EventDialog::OnDescriptionChanged)
END_EVENT_TABLE()

EventDialog::EventDialog(wxWindow *parent)
: wxDialog(parent, wxID_ANY, _("New event")), m_editing(false), m_id(-1), m_reminder(0), m_reminderOwn(0), m_type(0), m_days(0), m_months(0), m_daysOwn(0), m_monthsOwn(0),
  m_mday(0), m_mweek(1), m_minutes(0), m_hours(0), m_minutesOwn(0), m_hoursOwn(0), m_until(INFINITY_DATE)
{
    createControls();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
    Bind(wxEVT_CHAR_HOOK, &EventDialog::OnKeyUp, this);
}

EventDialog::EventDialog(wxWindow *parent, const wxDateTime &date)
: wxDialog(parent, wxID_ANY, _("New event")), m_editing(false), m_id(-1), m_reminder(0), m_reminderOwn(0), m_type(0), m_days(0), m_months(0), m_daysOwn(0), m_monthsOwn(0),
  m_mday(0), m_mweek(1), m_minutes(0), m_hours(0), m_minutesOwn(0), m_hoursOwn(0), m_until(INFINITY_DATE)
{
    createControls();
    m_mainSizer->Fit(this);
    m_dateCtrl->setDate(date);
    m_mainSizer->SetSizeHints(this);
    Bind(wxEVT_CHAR_HOOK, &EventDialog::OnKeyUp, this);
}

EventDialog::EventDialog(wxWindow *parent, Event event)
: wxDialog(parent, wxID_ANY, _("Edit event")), m_editing(true), m_id(event.id()), m_reminder(0), m_reminderOwn(0), m_type(0), m_days(0), m_months(0), m_daysOwn(0), m_monthsOwn(0),
  m_mday(0), m_mweek(1), m_minutes(0), m_hours(0), m_minutesOwn(0), m_hoursOwn(0), m_until(INFINITY_DATE)
{
    createControls();
    fillControls(event);
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
    Bind(wxEVT_CHAR_HOOK, &EventDialog::OnKeyUp, this);
}

void EventDialog::createControls()
{
    m_mainSizer = new wxBoxSizer(wxVERTICAL);

    wxGridBagSizer *gridSizer = new wxGridBagSizer(5, 5);

    int i=0;

    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Event:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_eventCtrl = new wxTextCtrl(this, wxID_ANY);
    gridSizer->Add(m_eventCtrl, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    i++;

    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Description:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_descriptionCtrl = new wxTextCtrl(this, ID_DESCRIPTION, wxEmptyString, wxDefaultPosition, wxSize(250, 50), wxTE_MULTILINE | wxHSCROLL);
    gridSizer->Add(m_descriptionCtrl, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    i++;

    m_dateText = new wxStaticText(this, wxID_ANY, _("Date:"));
    gridSizer->Add(m_dateText, wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_dateCtrl = new dxCalendarPickerCtrl(this, roundTime());
    gridSizer->Add(m_dateCtrl, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    i++;

    m_timeText = new wxStaticText(this, wxID_ANY, _("Time:"));
    gridSizer->Add(m_timeText, wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_timeCtrl = new dxTimePickerCtrl(this, roundTime());
    gridSizer->Add(m_timeCtrl, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    i++;

    m_reminderText = new wxStaticText(this, wxID_ANY, _("Reminder:"));
    gridSizer->Add(m_reminderText, wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_reminderCtrl = new wxChoice(this, ID_REMINDERCHOICE, wxDefaultPosition, wxDefaultSize, dxutils::reminders());
    m_reminderCtrl->SetSelection(dxsettings.defaultReminder());
    gridSizer->Add(m_reminderCtrl, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    m_reminderEdit = new wxBitmapButton(this, ID_REMINDEREDIT, ICO_EDIT);
    gridSizer->Add(m_reminderEdit, wxGBPosition(i,2), wxDefaultSpan, wxEXPAND|wxALL, 5);
    m_reminderEdit->Hide();
    i++;

    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Recurrence:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_recurrenceCtrl = new wxChoice(this, ID_RECURRENCECHOICE, wxDefaultPosition, wxDefaultSize, dxutils::recurrences());
    m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(dxsettings.defaultRecurrence()));
    gridSizer->Add(m_recurrenceCtrl, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    m_recurrenceEdit = new wxBitmapButton(this, ID_RECURRENCEEDIT, ICO_EDIT);
    gridSizer->Add(m_recurrenceEdit, wxGBPosition(i,2), wxDefaultSpan, wxEXPAND|wxALL, 5);
    m_recurrenceEdit->Hide();
    i++;

    m_untilText = new wxStaticText(this, wxID_ANY, _("Until:"));
    gridSizer->Add(m_untilText, wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_untilChoice = new wxChoice(this, ID_UNTILCHOICE);
    m_untilChoice->Append(_("Forever"));
    m_untilChoice->Append(dxutils::formatDate(m_until, dxsettings.dateFormat()));
    gridSizer->Add(m_untilChoice, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    m_untilChoice->SetSelection(0);
    m_untilEdit = new wxBitmapButton(this, ID_UNTILEDIT, ICO_DATE);
    gridSizer->Add(m_untilEdit, wxGBPosition(i,2), wxDefaultSpan, wxEXPAND|wxALL, 5);
    m_untilText->Hide();
    m_untilChoice->Hide();
    m_untilEdit->Hide();
    i++;

    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Hide event:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_hideEvent = new wxCheckBox(this, wxID_ANY, "");
    gridSizer->Add(m_hideEvent, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    i++;

    m_alwaysText = new wxStaticText(this, wxID_ANY, _("Always show:"));
    gridSizer->Add(m_alwaysText, wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_always = new wxCheckBox(this, wxID_ANY, "");
    gridSizer->Add(m_always, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    m_alwaysText->Hide();
    m_always->Hide();

    if(m_id == -1 && dxsettings.defaultRecurrence() == R_NONE)
    {
        m_timeText->Hide();
        m_timeCtrl->Hide();
        m_dateText->Hide();
        m_dateCtrl->Hide();
        m_reminderText->Hide();
        m_reminderEdit->Hide();
        m_reminderCtrl->Hide();
    }

    gridSizer->SetEmptyCellSize(wxSize(0,0));

    m_mainSizer->Add(gridSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    m_mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(m_mainSizer);
}

void EventDialog::fillControls(Event event)
{
    m_eventCtrl->SetValue(event.event());
    m_descriptionCtrl->SetValue(event.descr());
    m_dateCtrl->setDate(event.date());
    m_timeCtrl->setTime(event.date());
    m_reminder = event.reminder();
    m_reminderOwn = event.reminder();
    if(m_reminder < 0)
    {
        m_reminderText->Hide();
        m_reminderCtrl->Hide();
    }
    else
    {
        int id;
        switch(m_reminder)
        {
            case 0: id=0; break; //0 mins
            case 5: id=1; break; //5 mins
            case 15: id=2; break; //15 mins
            case 60: id=3; break; //1 hours
            case 120: id=4; break; //2 hours
            case 1440: id=5; break; //1 day
            case 2880: id=6; break; //2 days
            default:
            {
                id=7;
                wxInt64 mins = m_reminder;
                wxInt64 days = mins/(60*24);
                mins -= days*60*24;
                wxInt64 hours = mins/60;
                mins -= hours*60;
                wxString text = _("Custom");
                if(mins || hours || days)
                {
                    text << " (";
                }
                if(days)
                {
                    text << wxString::Format(wxPLURAL("%lld day","%lld days",days),days);
                    if(mins || hours) text << ", ";
                }
                if(hours)
                {
                    text << wxString::Format(wxPLURAL("%lld hour","%lld hours",hours),hours);
                    if(mins) text << ", ";
                }
                if(mins)
                {
                    text << wxString::Format(wxPLURAL("%lld minute","%lld minutes",mins),mins);
                }
                if(mins || hours || days)
                {
                    text << ")";
                }
                m_reminderCtrl->SetString(7, text);
                m_reminderEdit->Show();
                break;
            }
        }
        m_reminderCtrl->SetSelection(id);
    }
    m_type = event.recurrence();
    m_minutes = event.minutes();
    m_hours = event.hours();
    m_minutesOwn = event.minutes();
    m_hoursOwn = event.hours();
    m_days = event.days();
    m_months = event.months();
    m_daysOwn = event.days();
    m_monthsOwn = event.months();
    m_mday = event.monthlyday();
    m_mweek = event.monthlyweek();
    if(m_type == R_MINUTES)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_HOURLY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_WEEKLYPLUSBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_WEEKLYGIVENBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_2WEEKLYPLUSBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_2WEEKLYGIVENBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_MONTHLYATDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_MONTHLYPLUSBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_MONTHLYGIVENBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_QUARTERLYPLUSBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_QUARTERLYGIVENBUSINESSDAY)
    {
        m_recurrenceEdit->Show();
    }
    if(m_type == R_OWN)
    {
        m_recurrenceEdit->Show();
    }
    m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
    recurrenceText();
    m_until = event.until();
    m_untilChoice->SetString(1, dxutils::formatDate(m_until.IsSameDate(INFINITY_DATE)?untilDate():m_until, dxsettings.dateFormat()));
    if(m_until.IsSameDate(INFINITY_DATE)) m_untilChoice->SetSelection(0);
    else m_untilChoice->SetSelection(1);
    if(m_type != R_ONCE && m_type != R_ONCENOTDELETE && m_type != R_NONE)
    {
        m_untilText->Show();
        m_untilChoice->Show();
        if(!m_until.IsSameDate(INFINITY_DATE)) m_untilEdit->Show();
    }
    if(m_type == R_NONE)
    {
        m_timeText->Hide();
        m_timeCtrl->Hide();
        m_dateText->Hide();
        m_dateCtrl->Hide();
        m_reminderText->Hide();
        m_reminderEdit->Hide();
        m_reminderCtrl->Hide();
        m_alwaysText->Show();
        m_always->Show();
        m_always->SetValue(event.always());
    }
    if(m_type == R_ONCENOTDELETE)
    {
        m_alwaysText->Show();
        m_always->Show();
        m_always->SetValue(event.always());
    }
    if(iengine->isEventOnHiddenList(event.id()))
    {
        m_hideEvent->SetValue(true);
    }
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
}

void EventDialog::OnOk(wxCommandEvent& /*event*/)
{
    if(!m_eventCtrl->GetValue().IsEmpty())
    {
        m_type = dxutils::recurrenceSelectionToType(m_recurrenceCtrl->GetSelection());
        wxDateTime date = createDateTime();
        if(m_editing)
        {
            if(date > wxDateTime::Now() && m_reminder < 0)
                m_reminder = 0;
            int day = date.GetDay();
            if(m_type == R_WEEKLYBUSINESS || m_type == R_WEEKLYPLUSBUSINESSDAY || m_type == R_WEEKLYGIVENBUSINESSDAY
                    || m_type == R_2WEEKLYBUSINESS || m_type == R_2WEEKLYPLUSBUSINESSDAY || m_type == R_2WEEKLYGIVENBUSINESSDAY)
            {
                day = date.GetWeekDay();
            }
            Event ev = iengine->getEvent(m_id);
            bool reminded = ev.reminded();
            if(m_type == R_NONE)
            {
                reminded = true;
            }
            else
            {
                if(ev.recurrence() == R_NONE) reminded = false;
            }
            if((m_type==R_NONE||m_type==R_ONCENOTDELETE) && m_always->GetValue())
            {
                m_hideEvent->SetValue(false);
            }
            iengine->editEvent(Event(m_id,m_eventCtrl->GetValue(),m_descriptionCtrl->GetValue(),date,m_reminder,reminded,day,
                                     static_cast<int>(date.GetMonth())+1,m_hideEvent->GetValue(),
                                     m_type,m_minutes,m_hours,m_days,m_months,m_mday,m_mweek,m_reminder,m_until,
                                     m_type==R_NONE||m_type==R_ONCENOTDELETE?m_always->GetValue():false,ev.color()));
        }
        else
        {
            int day = date.GetDay();
            if(m_type == R_WEEKLYBUSINESS || m_type == R_WEEKLYPLUSBUSINESSDAY || m_type == R_WEEKLYGIVENBUSINESSDAY
                    || m_type == R_2WEEKLYBUSINESS || m_type == R_2WEEKLYPLUSBUSINESSDAY || m_type == R_2WEEKLYGIVENBUSINESSDAY)
            {
                day = date.GetWeekDay();
            }
            if((m_type==R_NONE||m_type==R_ONCENOTDELETE) && m_always->GetValue())
            {
                m_hideEvent->SetValue(false);
            }
            iengine->addEvent(Event(-1,m_eventCtrl->GetValue(),m_descriptionCtrl->GetValue(),date,m_reminder,
                                    m_type == R_NONE?true:false,day,static_cast<int>(date.GetMonth())+1,m_hideEvent->GetValue(),
                                    m_type,m_minutes,m_hours,m_days,m_months,m_mday,m_mweek,m_reminder,m_until,
                                    m_type==R_NONE||m_type==R_ONCENOTDELETE?m_always->GetValue():false,0));
        }
        EndModal(wxID_OK);
        return;
    }
    wxMessageDialog dialog(this, _("Event has to have some name"), _("Warning"), wxOK|wxICON_WARNING);
    dialog.ShowModal();
    m_eventCtrl->SetFocus();
}

void EventDialog::OnCancel(wxCommandEvent& /*event*/)
{
    EndModal(wxID_CANCEL);
}

void EventDialog::OnReminder(wxCommandEvent& event)
{
    switch(event.GetInt())
    {
        case 0: m_reminder = 0; m_reminderEdit->Hide(); break;
        case 1: m_reminder = 5; m_reminderEdit->Hide(); break;
        case 2: m_reminder = 15; m_reminderEdit->Hide(); break;
        case 3: m_reminder = 60; m_reminderEdit->Hide(); break;
        case 4: m_reminder = 120; m_reminderEdit->Hide(); break;
        case 5: m_reminder = 1440; m_reminderEdit->Hide(); break;
        case 6: m_reminder = 2880; m_reminderEdit->Hide(); break;
        default:
        {
            m_reminder = m_reminderOwn;
            m_reminderEdit->Show();
            break;
        }
    }
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
}

void EventDialog::OnRecurrence(wxCommandEvent& event)
{
    m_type = dxutils::recurrenceSelectionToType(event.GetInt());
    switch(m_type)
    {
        case R_ONCE: m_days = 0; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_ONCENOTDELETE: m_days = 0; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_MINUTES: m_minutes = 1; m_recurrenceEdit->Show(); break;
        case R_HOURLY: m_hours = 1; m_recurrenceEdit->Show(); break;
        case R_DAILY: m_days = 1; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_DAILYBUSINESS: m_days = 1; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_WEEKLY: m_days = 7; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_WEEKLYBUSINESS: m_days = 7; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_WEEKLYPLUSBUSINESSDAY: m_days = 7; m_months = 0; m_recurrenceEdit->Show(); break;
        case R_WEEKLYGIVENBUSINESSDAY: m_days = 7; m_months = 0; m_recurrenceEdit->Show(); break;
        case R_2WEEKLY: m_days = 14; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_2WEEKLYBUSINESS: m_days = 14; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_2WEEKLYPLUSBUSINESSDAY: m_days = 14; m_months = 0; m_recurrenceEdit->Show(); break;
        case R_2WEEKLYGIVENBUSINESSDAY: m_days = 14; m_months = 0; m_recurrenceEdit->Show(); break;
        case R_MONTHLY: m_days = 0; m_months = 1; m_recurrenceEdit->Hide(); break;
        case R_MONTHLYBUSINESS: m_days = 0; m_months = 1; m_recurrenceEdit->Hide(); break;
        case R_MONTHLYATDAY: m_days = 0; m_months = 1; m_recurrenceEdit->Show(); break;
        case R_MONTHLYPLUSBUSINESSDAY: m_days = 0; m_months = 1; m_recurrenceEdit->Show(); break;
        case R_MONTHLYGIVENBUSINESSDAY: m_days = 0; m_months = 1; m_recurrenceEdit->Show(); break;
        case R_QUARTERLY: m_days = 0; m_months = 3; m_recurrenceEdit->Hide(); break;
        case R_QUARTERLYBUSINESS: m_days = 0; m_months = 3; m_recurrenceEdit->Hide(); break;
        case R_QUARTERLYPLUSBUSINESSDAY: m_days = 0; m_months = 3; m_recurrenceEdit->Show(); break;
        case R_QUARTERLYGIVENBUSINESSDAY: m_days = 0; m_months = 3; m_recurrenceEdit->Show(); break;
        case R_YEARLY: m_days = 0; m_months = 0; m_recurrenceEdit->Hide(); break;
        case R_NONE: m_days = 0; m_months = 0; m_recurrenceEdit->Hide(); break;
        default: m_minutes = m_minutesOwn; m_hours = m_hoursOwn; m_days = m_daysOwn; m_months = m_monthsOwn; m_recurrenceEdit->Show(); break;
    }
    if((m_type == R_WEEKLYPLUSBUSINESSDAY || m_type == R_WEEKLYGIVENBUSINESSDAY || m_type == R_2WEEKLYPLUSBUSINESSDAY || m_type == R_2WEEKLYGIVENBUSINESSDAY
        || m_type == R_MONTHLYPLUSBUSINESSDAY || m_type == R_MONTHLYGIVENBUSINESSDAY || m_type == R_QUARTERLYPLUSBUSINESSDAY || m_type == R_QUARTERLYGIVENBUSINESSDAY) && m_mday == 0)
    {
        m_mday = 1;
    }
    recurrenceText();
    if(m_type != R_ONCE && m_type != R_ONCENOTDELETE && m_type != R_NONE)
    {
        m_untilText->Show();
        m_untilChoice->Show();
        if(m_untilChoice->GetSelection() == 1) m_untilEdit->Show();
        else m_untilEdit->Hide();
    }
    else
    {
        m_untilText->Hide();
        m_untilChoice->Hide();
        m_untilEdit->Hide();
    }
    if(m_type == R_NONE)
    {
        m_timeText->Hide();
        m_timeCtrl->Hide();
        m_dateText->Hide();
        m_dateCtrl->Hide();
        m_reminderText->Hide();
        m_reminderCtrl->Hide();
        m_alwaysText->Show();
        m_always->Show();
    }
    else if(m_type == R_ONCENOTDELETE)
    {
        m_alwaysText->Show();
        m_always->Show();
    }
    else
    {
        m_timeText->Show();
        m_timeCtrl->Show();
        m_dateText->Show();
        m_dateCtrl->Show();
        m_reminderText->Show();
        m_reminderCtrl->Show();
        m_alwaysText->Hide();
        m_always->Hide();
    }
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
}

void EventDialog::OnKeyUp(wxKeyEvent &event)
{
    if(event.GetKeyCode() == WXK_NUMPAD_ENTER || event.GetKeyCode() == WXK_RETURN)
    {
        wxCommandEvent ev;
        OnOk(ev);
    }
#if !(defined (__WXMSW__))
    if(event.GetKeyCode() == WXK_TAB && m_descriptionCtrl->HasFocus())
    {
        if(dxutils::recurrenceSelectionToType(m_recurrenceCtrl->GetSelection()) == R_NONE)
            m_recurrenceCtrl->SetFocusFromKbd();
        else
            m_dateCtrl->SetFocusFromKbd();
    }
#endif
    event.Skip();
}

wxDateTime EventDialog::createDateTime()
{
    wxDateTime date = m_dateCtrl->date();
    date.SetHour(m_timeCtrl->time().GetHour());
    date.SetMinute(m_timeCtrl->time().GetMinute());
    date.SetSecond(0);
    return date;
}

wxDateTime EventDialog::untilDate()
{
    wxDateTime date = createDateTime();
    wxDateSpan span;
    switch(m_type) {
    case R_DAILY: span.SetDays(2); break;
    case R_DAILYBUSINESS: span.SetDays(6); break;
    case R_WEEKLY: span.SetDays(1); span.SetWeeks(1); break;
    case R_2WEEKLY: span.SetDays(1); span.SetWeeks(2); break;
    case R_MONTHLY: span.SetDays(1); span.SetMonths(1); break;
    case R_OWN: span.SetDays(1+m_daysOwn); span.SetMonths(m_monthsOwn); break;
    case R_MONTHLYATDAY: span.SetDays(1); span.SetMonths(1); break;
    case R_MONTHLYBUSINESS: span.SetDays(1); span.SetMonths(1); break;
    case R_MONTHLYPLUSBUSINESSDAY: span.SetDays(1); span.SetMonths(1); break;
    case R_QUARTERLYPLUSBUSINESSDAY: span.SetDays(1); span.SetMonths(3); break;
    case R_MONTHLYGIVENBUSINESSDAY: span.SetDays(1); span.SetMonths(1); break;
    case R_QUARTERLYGIVENBUSINESSDAY: span.SetDays(1); span.SetMonths(3); break;
    case R_YEARLY: span.SetDays(1); span.SetYears(1); break;
    case R_QUARTERLYBUSINESS: span.SetDays(1); span.SetMonths(3); break;
    case R_QUARTERLY: span.SetDays(1); span.SetMonths(3); break;
    case R_HOURLY: span.SetDays(1); break;
    case R_MINUTES: span.SetDays(1); break;
    case R_WEEKLYBUSINESS:
    case R_WEEKLYPLUSBUSINESSDAY:
    case R_WEEKLYGIVENBUSINESSDAY: span.SetDays(1); span.SetWeeks(1); break;
    case R_2WEEKLYBUSINESS:
    case R_2WEEKLYPLUSBUSINESSDAY:
    case R_2WEEKLYGIVENBUSINESSDAY: span.SetDays(1); span.SetWeeks(2); break;
    default: break;
    }
    date += span;
    return date;
}

wxDateTime EventDialog::roundTime()
{
    if(dxsettings.reminderTime() == 0)
    {
        wxDateTime date = wxDateTime::Now()+wxTimeSpan::Minute();
        date.SetSecond(0);
        return date;
    }
    if(dxsettings.reminderTime() == 1)
    {
        wxDateTime date = wxDateTime::Now();
        date.SetHour(wxMin(static_cast<wxDateTime::wxDateTime_t>(23),dxsettings.reminderPresetTime().GetHour()));
        date.SetMinute(wxMin(static_cast<wxDateTime::wxDateTime_t>(59),dxsettings.reminderPresetTime().GetMinute()));
        date.SetSecond(wxMin(static_cast<wxDateTime::wxDateTime_t>(59),dxsettings.reminderPresetTime().GetSecond()));
        return date;
    }
    wxDateTime date = wxDateTime::Now()+wxTimeSpan::Minutes(dxsettings.reminderMinutes()+1);
    date.SetSecond(0);
    return date;
}

void EventDialog::OnReminderEdit(wxCommandEvent &/*event*/)
{
    ReminderDialog dialog(this, m_reminderOwn);
    if(dialog.ShowModal() == wxID_OK)
    {
        m_reminder = dialog.reminder();
        m_reminderOwn = m_reminder;
        wxInt64 mins = m_reminder;
        wxInt64 days = mins/(60*24);
        mins -= days*60*24;
        wxInt64 hours = mins/60;
        mins -= hours*60;
        wxString text = _("Custom");
        if(mins || hours || days)
        {
            text << " (";
        }
        if(days)
        {
            text << wxString::Format(wxPLURAL("%lld day","%lld days",days),days);
            if(mins || hours) text << ", ";
        }
        if(hours)
        {
            text << wxString::Format(wxPLURAL("%lld hour","%lld hours",hours),hours);
            if(mins) text << ", ";
        }
        if(mins)
        {
            text << wxString::Format(wxPLURAL("%lld minute","%lld minutes",mins),mins);
        }
        if(mins || hours || days)
        {
            text << ")";
        }
        m_reminderCtrl->SetString(7, text);
        m_reminderCtrl->SetSelection(7);
    }
}

void EventDialog::OnRecurrenceEdit(wxCommandEvent &/*event*/)
{
    if(m_type == R_OWN)
    {
        RecurrenceDialog dialog(this, m_daysOwn, m_monthsOwn, m_hoursOwn, m_minutesOwn);
        if(dialog.ShowModal() == wxID_OK)
        {
            m_days = dialog.days();
            m_months = dialog.months();
            m_daysOwn = m_days;
            m_monthsOwn = m_months;
            m_minutes = dialog.minutes();
            m_hours = dialog.hours();
            m_minutesOwn = m_minutes;
            m_hoursOwn = m_hours;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_MONTHLYATDAY)
    {
        RecurrenceMontlyDayDialog dialog(this, m_mday, m_mweek);
        if(dialog.ShowModal() == wxID_OK)
        {
            m_mday = dialog.day();
            m_mweek = dialog.week();
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_WEEKLYPLUSBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("Business days:"), _("Question"), wxMax(m_mday,1), 1, 4);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_WEEKLYGIVENBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("At business day:"), _("Enter business day"), wxMax(m_mday,1), 1, 5);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_2WEEKLYPLUSBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("Business days:"), _("Question"), wxMax(m_mday,1), 1, 4);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_2WEEKLYGIVENBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("At business day:"), _("Enter business day"), wxMax(m_mday,1), 1, 5);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_MONTHLYPLUSBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("Business days:"), _("Question"), wxMax(m_mday,1), 1, 23);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_MONTHLYGIVENBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("At business day:"), _("Enter business day"), wxMax(m_mday,1), 1, 23);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_QUARTERLYPLUSBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("Business days:"), _("Question"), wxMax(m_mday,1), 1, 23);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_QUARTERLYGIVENBUSINESSDAY)
    {
        long day = wxGetNumberFromUser(wxEmptyString, _("At business day:"), _("Enter business day"), wxMax(m_mday,1), 1, 23);
        if(day != -1)
        {
            m_mday = day;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_HOURLY)
    {
        long hours = wxGetNumberFromUser(wxEmptyString, _("Every hours:"), _("Enter hourly"), wxMax(m_hours,1), 1, 23);
        if(hours != -1)
        {
            m_hours = hours;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    if(m_type == R_MINUTES)
    {
        long minutes = wxGetNumberFromUser(wxEmptyString, _("Every minutes:"), _("Enter minutes"), wxMax(m_minutes,1), 1, 59);
        if(minutes != -1)
        {
            m_minutes = minutes;
            m_recurrenceCtrl->SetSelection(dxutils::recurrenceTypeToSelection(m_type));
        }
    }
    recurrenceText();
}

void EventDialog::recurrenceText()
{
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_OWN),  ownRecurrenceText());
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_MONTHLYATDAY), wxString::Format(_("Monthly every %d. %s of month"), m_mweek, wxDateTime::GetWeekDayName(static_cast<wxDateTime::WeekDay>(wxMin(6,m_mday)))));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_WEEKLYPLUSBUSINESSDAY), wxString::Format(wxPLURAL("Weekly plus %d business day","Weekly plus %d business days",m_mday), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_WEEKLYGIVENBUSINESSDAY), wxString::Format(_("Weekly at %d business day"), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_2WEEKLYPLUSBUSINESSDAY), wxString::Format(wxPLURAL("Fortnightly plus %d business day","Fortnightly plus %d business days",m_mday), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_2WEEKLYGIVENBUSINESSDAY), wxString::Format(_("Fortnightly at %d business day"), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_MONTHLYPLUSBUSINESSDAY), wxString::Format(wxPLURAL("Monthly plus %d business day","Monthly plus %d business days",m_mday), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_MONTHLYGIVENBUSINESSDAY), wxString::Format(_("Monthly at %d business day"), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_QUARTERLYPLUSBUSINESSDAY), wxString::Format(wxPLURAL("Quarterly plus %d business day","Quarterly plus %d business days",m_mday), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_QUARTERLYGIVENBUSINESSDAY), wxString::Format(_("Quarterly at %d business day"), m_mday));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_HOURLY), wxString::Format(wxPLURAL("Hourly every %d hour","Hourly every %d hours",m_hours), m_hours));
    m_recurrenceCtrl->SetString(dxutils::recurrenceTypeToSelection(R_MINUTES), wxString::Format(wxPLURAL("Every %d minute","Every %d minutes",m_minutes), m_minutes));
}

wxString EventDialog::ownRecurrenceText()
{
    wxString text = _("Custom");
    if(m_minutesOwn || m_hoursOwn || m_daysOwn || m_monthsOwn)
    {
        text << " (";
    }
    if(m_monthsOwn)
    {
        text << wxString::Format(wxPLURAL("%d month","%d months",m_monthsOwn),m_monthsOwn);
        if(m_minutesOwn || m_hoursOwn || m_daysOwn) text << ", ";
    }
    if(m_daysOwn)
    {
        text << wxString::Format(wxPLURAL("%d day","%d days",m_daysOwn),m_daysOwn);
        if(m_minutesOwn || m_hoursOwn) text << ", ";
    }
    if(m_hoursOwn)
    {
        text << wxString::Format(wxPLURAL("%d hour","%d hours",m_hoursOwn),m_hoursOwn);
        if(m_minutesOwn) text << ", ";
    }
    if(m_minutesOwn)
    {
        text << wxString::Format(wxPLURAL("%d minute","%d minutes",m_minutesOwn),m_minutesOwn);
    }
    if(m_minutesOwn || m_hoursOwn || m_daysOwn || m_monthsOwn)
    {
        text << ")";
    }
    return text;
}

void EventDialog::OnUntil(wxCommandEvent &/*event*/)
{
    if(m_untilChoice->GetSelection() == 0)
    {
        m_untilEdit->Hide();
        m_until = INFINITY_DATE;
        m_untilChoice->SetString(1, dxutils::formatDate(untilDate(), dxsettings.dateFormat()));
    }
    else
    {
        m_untilEdit->Show();
    }
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
}

void EventDialog::OnUntilEdit(wxCommandEvent &/*event*/)
{
    DateDialog dialog(this, m_until.IsSameDate(INFINITY_DATE)?untilDate():m_until);
    if(dialog.ShowModal() == wxID_OK)
    {
        m_until = dialog.date();
        m_untilChoice->SetString(1, dxutils::formatDate(m_until, dxsettings.dateFormat()));
    }
}

void EventDialog::OnDescriptionChanged(wxCommandEvent &/*event*/)
{
    if(!m_descriptionCtrl->GetValue().IsEmpty() && m_descriptionCtrl->GetValue().Last() == '\t')
    {
        m_descriptionCtrl->ChangeValue(m_descriptionCtrl->GetValue().Left(m_descriptionCtrl->GetValue().Len()-1));
    }
}

IMPLEMENT_DYNAMIC_CLASS(RecurrenceMontlyDayDialog, wxDialog)

RecurrenceMontlyDayDialog::RecurrenceMontlyDayDialog(wxWindow *parent, int mday, int mweek)
: wxDialog(parent, wxID_ANY, _("Recurrence"))
{
    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);

    wxGridBagSizer *gridSizer = new wxGridBagSizer(5,5);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Week day")), wxGBPosition(0,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_mday = new wxChoice(this, wxID_ANY);
    for(wxDateTime::WeekDay wd = wxDateTime::Sun; wd < wxDateTime::Inv_WeekDay; wxNextWDay(wd))
    {
        m_mday->Append(wxDateTime::GetWeekDayName(wd));
    }
    m_mday->SetSelection(wxMin(6,mday));
    gridSizer->Add(m_mday, wxGBPosition(0,1), wxDefaultSpan, wxEXPAND);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Week of month")), wxGBPosition(1,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_mweek = new wxSpinCtrl(this);
    m_mweek->SetRange(1,5);
    m_mweek->SetValue(mweek);
    gridSizer->Add(m_mweek, wxGBPosition(1,1), wxDefaultSpan, wxEXPAND);
    mainSizer->Add(gridSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);
}

int RecurrenceMontlyDayDialog::day()
{
    return m_mday->GetSelection();
}

int RecurrenceMontlyDayDialog::week()
{
    return m_mweek->GetValue();
}
