/*
 * ViSP, open source Visual Servoing Platform software.
 * Copyright (C) 2005 - 2023 by Inria. All rights reserved.
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * See the file LICENSE.txt at the root directory of this source
 * distribution for additional information about the GNU GPL.
 *
 * For using ViSP with software that can not be combined with the GNU
 * GPL, please contact Inria about acquiring a ViSP Professional
 * Edition License.
 *
 * See https://visp.inria.fr for more information.
 *
 * This software was developed at:
 * Inria Rennes - Bretagne Atlantique
 * Campus Universitaire de Beaulieu
 * 35042 Rennes Cedex
 * France
 *
 * If you have questions regarding the use of this file, please contact
 * Inria at visp@inria.fr
 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Description:
 * Provide some simple operation on column vectors.
 */

#ifndef _vpColVector_h_
#define _vpColVector_h_

#include <visp3/core/vpArray2D.h>
#include <visp3/core/vpMath.h>
#include <visp3/core/vpPoseVector.h>
#include <visp3/core/vpRotationVector.h>
#include <visp3/core/vpRowVector.h>

class vpMatrix;
class vpRowVector;
class vpRotationVector;
class vpTranslationVector;
class vpPoseVector;

/*!
  \file vpColVector.h
  \brief definition of column vector class as well
  as a set of operations on these vector
*/

/*!
  \class vpColVector
  \ingroup group_core_matrices

  \brief Implementation of column vector and the associated operations.

  This class provides a data structure for a column vector that contains
  values of double. It contains also some functions to achieve a set of
  operations on these vectors.

  The vpColVector class is derived from vpArray2D<double>.

  The code below shows how to create a 3-element column vector of doubles, set the element values and access them:
  \code
  #include <visp3/code/vpColVector.h

  int main()
  {
    vpColVector v(3);
    v[0] = -1; v[1] = -2.1; v[2] = -3;

    std::cout << "v:" << std::endl;
    for (unsigned int i = 0; i < v.size(); i++) {
      std::cout << v[i] << std::endl;
    }
  }
  \endcode
  Once build, this previous code produces the following output:
  \code{.unparsed}
  v:
  -1
  -2.1
  -3
  \endcode
  You can also use operator<< to initialize a column vector as previously:
  \code
  #include <visp3/code/vpColVector.h

  int main()
  {
    vpColVector v;
    v << -1, -2.1, -3;
    std::cout << "v:" << v << std::endl;
  }
  \endcode

  If ViSP is build with c++11 enabled, you can do the same using:
  \code
  #include <visp3/code/vpColVector.h

  int main()
  {
  #if (VISP_CXX_STANDARD >= VISP_CXX_STANDARD_11)
    vpColVector v({-1, -2.1, -3});
    std::cout << "v:\n" << v << std::endl;
  #endif
  }
  \endcode
  The vector could also be initialized using operator=(const std::initializer_list< double > &)
  \code
  int main()
  {
  #if (VISP_CXX_STANDARD >= VISP_CXX_STANDARD_11)
    vpColVector v;
    v = {-1, -2.1, -3};
  #endif
  }
  \endcode

  <b>JSON serialization</b>

  Since ViSP 3.6.0, if ViSP is build with \ref soft_tool_json 3rd-party we introduce JSON serialization capabilities for vpColVector.
  The following sample code shows how to save a pose vector in a file named `col-vector.json`
  and reload the values from this JSON file.
  \code
  #include <visp3/core/vpColVector.h>

  int main()
  {
  #if defined(VISP_HAVE_NLOHMANN_JSON)
    std::string filename = "col-vector.json";
    {
      vpColVector v({ 1, 2, 3, 4 });
      std::ofstream file(filename);
      const nlohmann::json j = v;
      file << j;
      file.close();
    }
    {
      std::ifstream file(filename);
      const nlohmann::json j = nlohmann::json::parse(file);
      vpColVector v;
      v = j.get<vpColVector>();
      file.close();
      std::cout << "Read homogeneous matrix from " << filename << ":\n" << v.t() << std::endl;
    }
  #endif
  }
  \endcode
  If you build and execute the sample code, it will produce the following output:
  \code{.unparsed}
  Read homogeneous matrix from col-vector.json:
  1  2  3  4
  \endcode

  The content of the `pose-vector.json` file is the following:
  \code{.unparsed}
  $ cat col-vector.json
  {"cols":1,"data":[1.0,2.0,3.0,4.0],"rows":4,"type":"vpColVector"}
  \endcode
*/
class VISP_EXPORT vpColVector : public vpArray2D<double>
{
  friend class vpMatrix;

public:
  //! Basic constructor that creates an empty 0-size column vector.
  vpColVector() : vpArray2D<double>() { }
  //! Construct a column vector of size n. \warning Elements are not
  //! initialized. If you want to set an initial value use
  //! vpColVector(unsigned int, double).
  explicit vpColVector(unsigned int n) : vpArray2D<double>(n, 1) { }
  //! Construct a column vector of size n. Each element is set to \e val.
  vpColVector(unsigned int n, double val) : vpArray2D<double>(n, 1, val) { }
  //! Copy constructor that allows to construct a column vector from an other
  //! one.
  vpColVector(const vpColVector &v) : vpArray2D<double>(v) { }
  vpColVector(const vpColVector &v, unsigned int r, unsigned int nrows);
  //! Constructor that initialize a column vector from a 3-dim (Euler or
  //! \f$\theta {\bf u}\f$) or 4-dim (quaternion) rotation vector.
  vpColVector(const vpRotationVector &v);
  //! Constructor that initialize a column vector from a 6-dim pose vector.
  vpColVector(const vpPoseVector &p);
  //! Constructor that initialize a column vector from a 3-dim translation
  //! vector.
  vpColVector(const vpTranslationVector &t);
  vpColVector(const vpMatrix &M);
  vpColVector(const vpMatrix &M, unsigned int j);
  vpColVector(const std::vector<double> &v);
  vpColVector(const std::vector<float> &v);
#if (VISP_CXX_STANDARD >= VISP_CXX_STANDARD_11)
  vpColVector(vpColVector &&v);
  vpColVector(const std::initializer_list<double> &list) : vpArray2D<double>(static_cast<unsigned int>(list.size()), 1)
  {
    std::copy(list.begin(), list.end(), data);
  }
#endif
  /*!
    Destructor.
  */
  virtual ~vpColVector() { }

  /*!
    Removes all elements from the vector (which are destroyed),
    leaving the container with a size of 0.
  */
  void clear()
  {
    if (data != NULL) {
      free(data);
      data = NULL;
    }

    if (rowPtrs != NULL) {
      free(rowPtrs);
      rowPtrs = NULL;
    }
    rowNum = colNum = dsize = 0;
  }

  std::ostream &cppPrint(std::ostream &os, const std::string &matrixName = "A", bool octet = false) const;
  std::ostream &csvPrint(std::ostream &os) const;

  /*!
    Converts a column vector containing angles in degrees into radians and returns a reference to the vector.
    \return A reference to the vector with values expressed in [rad].
    \sa rad2deg()
  */
  inline vpColVector &deg2rad()
  {
    double d2r = M_PI / 180.0;

    (*this) *= d2r;
    return (*this);
  }

  /*!
     Extract a sub-column vector from a column vector.
     \param r : Index of the row corresponding to the first element of the
     vector to extract. \param colsize : Size of the vector to extract.
     \exception vpException::fatalError If the vector to extract is not
     contained in the original one.

     \code
     vpColVector v1;
     for (unsigned int i=0; i<4; i++)
       v1.stack(i);
     // v1 is equal to [0 1 2 3]^T
     vpColVector v2 = v1.extract(1, 3);
     // v2 is equal to [1 2 3]^T
     \endcode
   */
  vpColVector extract(unsigned int r, unsigned int colsize) const
  {
    if (r >= rowNum || r + colsize > rowNum) {
      throw(vpException(vpException::fatalError,
                        "Cannot extract a (%dx1) column vector from a (%dx1) "
                        "column vector starting at index %d",
                        colsize, rowNum, r));
    }

    return vpColVector(*this, r, colsize);
  }

  double frobeniusNorm() const;
  vpColVector hadamard(const vpColVector &v) const;

  double infinityNorm() const;
  void init(const vpColVector &v, unsigned int r, unsigned int nrows);
  void insert(unsigned int i, const vpColVector &v);
  void insert(const vpColVector &v, unsigned int i);

  std::ostream &maplePrint(std::ostream &os) const;
  std::ostream &matlabPrint(std::ostream &os) const;

  vpColVector &normalize();
  vpColVector &normalize(vpColVector &x) const;

  //! Operator that allows to set a value of an element \f$v_i\f$: v[i] = x
  inline double &operator[](unsigned int n) { return *(data + n); }
  //! Operator that allows to get the value of an element \f$v_i\f$: x = v[i]
  inline const double &operator[](unsigned int n) const { return *(data + n); }
  //! Copy operator.   Allow operation such as A = v
  vpColVector &operator=(const vpColVector &v);
  vpColVector &operator=(const vpPoseVector &p);
  vpColVector &operator=(const vpRotationVector &rv);
  vpColVector &operator=(const vpTranslationVector &tv);
  vpColVector &operator=(const vpMatrix &M);
  vpColVector &operator=(const std::vector<double> &v);
  vpColVector &operator=(const std::vector<float> &v);
  vpColVector &operator=(double x);
#if (VISP_CXX_STANDARD >= VISP_CXX_STANDARD_11)
  vpColVector &operator=(vpColVector &&v);
  vpColVector &operator=(const std::initializer_list<double> &list);
#endif
  //! Comparison operator.
  bool operator==(const vpColVector &v) const;
  bool operator==(double v) const;
  bool operator!=(const vpColVector &v) const;
  bool operator!=(double v) const;

  double operator*(const vpColVector &x) const;
  vpMatrix operator*(const vpRowVector &v) const;
  vpColVector operator*(double x) const;
  vpColVector &operator*=(double x);

  vpColVector operator/(double x) const;
  vpColVector &operator/=(double x);

  vpColVector operator+(const vpColVector &v) const;
  vpTranslationVector operator+(const vpTranslationVector &t) const;
  vpColVector &operator+=(vpColVector v);

  vpColVector operator-(const vpColVector &v) const;
  vpColVector &operator-=(vpColVector v);
  vpColVector operator-() const;

  vpColVector &operator<<(const vpColVector &v);
  vpColVector &operator<<(double *);
  vpColVector &operator<<(double val);
  vpColVector &operator,(double val);

  int print(std::ostream &s, unsigned int length, char const *intro = 0) const;

  /*!
    Converts a column vector containing angles in radians into degrees and returns a reference to the vector.
    \return A reference to the vector with values expressed in [deg].
    \sa deg2rad()
  */
  inline vpColVector &rad2deg()
  {
    double r2d = 180.0 / M_PI;

    (*this) *= r2d;
    return (*this);
  }

  void reshape(vpMatrix &M, const unsigned int &nrows, const unsigned int &ncols);
  vpMatrix reshape(unsigned int nrows, unsigned int ncols);

  /*! Modify the size of the column vector.
    \param i : Size of the vector. This value corresponds to the vector number
    of rows.
    \param flagNullify : If true, set the data to zero.
    \exception vpException::fatalError When \e ncols is not equal to 1.
   */
  void resize(unsigned int i, bool flagNullify = true) { vpArray2D<double>::resize(i, 1, flagNullify); }
  /*!
    Resize the column vector to a \e nrows-dimension vector.
    This function can only be used with \e ncols = 1.
    \param nrows : Vector number of rows. This value corresponds
    to the size of the vector.
    \param ncols : Vector number of columns. This value should be set to 1.
    \param flagNullify : If true, set the data to zero.
    \exception vpException::fatalError When \e ncols is not equal to 1.

    */
  void resize(unsigned int nrows, unsigned int ncols, bool flagNullify)
  {
    if (ncols != 1) {
      throw(vpException(vpException::fatalError,
                        "Cannot resize a column vector to a (%dx%d) "
                        "dimension vector that has more than one column",
                        nrows, ncols));
    }
    vpArray2D<double>::resize(nrows, ncols, flagNullify);
  }

  void stack(double d);
  void stack(const vpColVector &v);

  double sum() const;
  double sumSquare() const;
  vpRowVector t() const;
  std::vector<double> toStdVector() const;
  vpRowVector transpose() const;
  void transpose(vpRowVector &v) const;

  /*!
     Compute and return the cross product of two 3-dimension vectors: \f$a
     \times b\f$. \param a : 3-dimension column vector. \param b : 3-dimension
     column vector. \return The cross product \f$a \times b\f$.

     \exception vpException::dimensionError If the vectors dimension is not
     equal to 3.
   */
  inline static vpColVector cross(const vpColVector &a, const vpColVector &b) { return crossProd(a, b); }
  static vpColVector crossProd(const vpColVector &a, const vpColVector &b);

  static double dotProd(const vpColVector &a, const vpColVector &b);
  static vpColVector invSort(const vpColVector &v);
  static double median(const vpColVector &v);
  static double mean(const vpColVector &v);
  // Compute the skew matrix [v]x
  static vpMatrix skew(const vpColVector &v);

  static vpColVector sort(const vpColVector &v);

  static vpColVector stack(const vpColVector &A, const vpColVector &B);
  static void stack(const vpColVector &A, const vpColVector &B, vpColVector &C);

  static double stdev(const vpColVector &v, bool useBesselCorrection = false);

#ifdef VISP_HAVE_NLOHMANN_JSON
  friend void to_json(nlohmann::json &j, const vpColVector &cam);
  friend void from_json(const nlohmann::json &j, vpColVector &cam);
#endif

#if defined(VISP_BUILD_DEPRECATED_FUNCTIONS)
  /*!
    @name Deprecated functions
  */
  //@{
  /*!
     \deprecated Provided only for compat with previous releases.
     This function does nothing.
   */
  vp_deprecated void init() { }
  /*!
     \deprecated You should rather use extract().
   */
  vp_deprecated vpColVector rows(unsigned int first_row, unsigned int last_row) const
  {
    return vpColVector(*this, first_row - 1, last_row - first_row + 1);
  }
  /*!
     \deprecated You should rather use eye()
   */
  vp_deprecated void setIdentity(const double &val = 1.0);
  /*!
     \deprecated You should rather use stack(const vpColVector &)
   */
  vp_deprecated void stackMatrices(const vpColVector &r) { stack(r); }
  /*!
     \deprecated You should rather use stack(const vpColVector &A, const
     vpColVector &B)
   */
  vp_deprecated static vpColVector stackMatrices(const vpColVector &A, const vpColVector &B) { return stack(A, B); }
  /*!
     \deprecated You should rather use stack(const vpColVector &A, const
     vpColVector &B, vpColVector &C)
   */
  vp_deprecated static void stackMatrices(const vpColVector &A, const vpColVector &B, vpColVector &C)
  {
    stack(A, B, C);
  }

  vp_deprecated void insert(const vpColVector &v, unsigned int r, unsigned int c = 0);
  vp_deprecated double euclideanNorm() const;
  //@}
#endif
};

#ifndef DOXYGEN_SHOULD_SKIP_THIS
VISP_EXPORT
#endif
vpColVector operator*(const double &x, const vpColVector &v);

#ifdef VISP_HAVE_NLOHMANN_JSON
inline void to_json(nlohmann::json &j, const vpColVector &v)
{
  const vpArray2D<double> *asArray = (vpArray2D<double>*) & v;
  to_json(j, *asArray);
  j["type"] = "vpColVector";
}
inline void from_json(const nlohmann::json &j, vpColVector &v)
{
  vpArray2D<double> *asArray = (vpArray2D<double>*) & v;
  from_json(j, *asArray);
  if (v.getCols() != 1) {
    throw vpException(vpException::badValue, "From JSON, tried to read a 2D array into a vpColVector");
  }
}


#endif

#endif
