use core:lang;
use lang:bnf;
use lang:bs;
use lang:bs:macro;

/**
 * A declared parser.
 */
class Parser extends lang:bs:NamedDecl {
	// Override to create function body.
	FnBody createBody(BSRawFn fn, Scope scope, Rule start, NameSet[] include) : abstract;

	// Create, specify the start rule (as a name).
	init(Scope scope, SStr name) {
		init {
			scope = scope;
			name = name;
		}
	}

	// Options for this function.
	private SFreeOptions? options;

	// Scope.
	private Scope scope;

	// Name.
	SStr name;

	// Syntax entered in the parser block.
	ParserSyntax? syntax;

	// Parse a binary buffer rather than a string.
	Bool binary;

	// Called to make the parser into a binary parser.
	void makeBinary() {
		binary = true;
	}

	// Set syntax. Called by the grammar.
	void setSyntax(ParserSyntax syntax) {
		this.syntax = syntax;
	}

	// Set options. Called by the grammar.
	void setOptions(SFreeOptions options) {
		this.options = options;
	}

	// Function class so that we can do lazy-loading to get load order right.
	class ParserFn extends BSRawFn {
		init(Parser parser, ParserSyntax syntax, SrcName startName, NameLookup scopeTop) {
			Package syntaxPkg = syntax.createPackage(parser.scope, scopeTop);

			// Add includes. Most importantly so that we may access both 'syntaxPkg' and the current package.
			NameSet[] include;
			include << syntaxPkg;
			for (name in syntax.use) {
				unless (inc = parser.scope.find(name) as NameSet)
					throw SyntaxError(name.pos, "Unknown package: ${name}");
				include << inc;
			}

			Scope s = addIncludes(parser.scope, include);
			Rule start = findRule(s, startName);

			// We also need to include the current "top" package in the include directory. It is OK to
			// do this after adding includes to the scope since the scope has its top set anyway.
			include << ScopeLookup:firstPkg(scopeTop);

			// Now we can start creating the actual parser functions!
			SimpleName resultName;
			resultName.add("parser");

			ValParam[] params = start.params.clone;
			if (parser.binary) {
				params.insert(0, ValParam(named{core:io:Buffer}, " input"));
				params.insert(1, ValParam(named{Nat}, " start"));
				resultName.add("BinaryResult", [start.result]);
			} else {
				params.insert(0, ValParam(named{Str}, " input"));
				params.insert(1, ValParam(named{Str:Iter}, " start"));
				resultName.add("Result", [start.result]);
			}

			var resultType = s.value(resultName, parser.name.pos);

			init(resultType, parser.name, params, null) {
				parser = parser;
				start = start;
				scope = s;
				include = include;
			}
		}

		// Data we need when creating the body.
		Parser? parser;
		Rule? start;
		Scope scope;
		NameSet[] include;

		FnBody createBody() : override {
			unless (parser)
				throw InternalError("Should not need to create the body more than once!");
			unless (start)
				throw InternalError("Should not need to create the body more than once!");

			parser.createBody(this, scope, start, include);
		}

		void clearBody() : override {
			// Shed some weight.
			parser = null;
			start = null;
			scope = Scope();
			include.clear();
		}
	}

	// Create the parser. The generated function takes (Str, Str:Iter, ...) as parameters, where
	// ... is the parameters to the root rule.
	protected Named doCreate() : override {
		unless (syntax)
			throw InternalError("The syntax for the parser was not set properly.");
		unless (startName = syntax.start)
			throw SyntaxError(name.pos, "You need to specify the start rule with 'start = <name>;'.");
		unless (scopeTop = scope.top)
			throw InternalError("Expected to find a package in the scope.");

		ParserFn fn(this, syntax, startName, scopeTop);

		// Apply options.
		if (thread)
			fn.thread(scope, thread);
		if (options)
			options.transform(scope, fn);

		fn;
	}

}

// Find a rule. Throws on error.
private Rule findRule(Scope scope, SrcName name) on Compiler {
	if (rule = scope.find(name) as Rule)
		rule;
	else
		throw SyntaxError(name.pos, "Unknown rule: ${name}");
}

// Add a set of NameSet:s to a Scope. Making sure not to modify the original scope.
private Scope addIncludes(Scope scope, NameSet[] add) on Compiler {
	if (lookup = scope.lookup) {
		lookup = lookup.clone;
		if (lookup as BSLookup) {
			for (x in add)
				if (x as Package)
					lookup.includes << x;
		} else if (lookup as ScopeExtra) {
			for (x in add)
				lookup.extra << x;
		}
		scope.lookup = lookup;
	}
	scope;
}


// Create the overloads needed for the parser.
MultiDecl parserOverloads(Parser parser) {
	MultiDecl([parser, ParserOverload(parser)]);
}

/**
 * Overload of a parser. Wraps the X(Str, Iter) with X(Str).
 */
class ParserOverload extends lang:bs:NamedDecl {
	init(Parser parser) {
		init {
			wrap = parser;
		}
	}

	Parser wrap;

	protected Named doCreate() : override {
		unless (wrapFn = wrap.create() as BSRawFn)
			throw InternalError("Expected a function.");

		ValParam[] params;
		for (i, x in wrapFn.docParams) {
			if (i != 1) // Remove the position parameter
				params << ValParam(x.type, x.name);
		}

		BSTreeFn fn(wrapFn.result, SStr(wrapFn.name, wrapFn.pos), params, wrapFn.declaredThread);

		// Apply thread options.
		if (thread = wrapFn.declaredThread)
			fn.runOn(thread);

		fn;
	}

	protected void doResolve(Named fn) : override {
		unless (wrapFn = wrap.create() as BSRawFn)
			throw InternalError("Expected a function.");

		unless (fn as BSTreeFn)
			throw InternalError("Expected a tree function.");

		Scope scope(wrapFn);

		FnBody body(fn, scope);
		fn.body = body;

		Actuals actuals;
		LocalVarAccess str(SrcPos(), body.parameters[0]); // Source string.
		actuals.add(str);
		if (wrap.binary)
			actuals.add(NumLiteral(SrcPos(), 0));
		else
			actuals.add(FnCall(SrcPos(), scope, named{Str:begin<Str>}, Actuals(str)));

		for (Nat i = 2; i < body.parameters.count; i++)
			actuals.add(LocalVarAccess(SrcPos(), body.parameters[i]));
		body.add(FnCall(SrcPos(), scope, wrapFn, actuals));
	}
}

class ParserSyntax extends lang:bnf:FileContents {
	// Name of the start rule.
	SrcName? start;

	void push(FileItem item) : override {
		if (item as StartDecl) {
			if (start)
				throw SyntaxError(item.name.pos, "You can only specify the start rule once.");
			start = item.name;
		} else {
			super:push(item);
		}
	}
}

// Create a package that contains the productions in here.
Package createPackage(lang:bnf:FileContents this, Scope scope, NameLookup parent) on Compiler {
	Package p("<anonymous>");
	p.parentLookup = parent;

	SyntaxLookup lookup;
	if (old = scope.lookup as ScopeExtra) {
		lookup.extra << parent;
		for (x in old.extra)
			lookup.extra << x;
	} else if (old = scope.lookup as BSLookup) {
		lookup.extra << parent;
		for (x in old.includes)
			lookup.extra << x;
	}

	// New includes.
	for (x in use) {
		unless (found = rootScope.find(x))
			throw SyntaxError(x.pos, "The package ${x} does not exist!");
		lookup.extra << found;
	}

	scope.lookup = lookup;
	scope.top = p;

	for (x in rules) {
		p.add(Rule(x, scope));
	}

	var delims = delimiters(scope);

	for (x in productions) {
		Str name = if (n = x.name) {
			n;
		} else {
			p.anonName();
		};
		p.add(ProductionType(name, x, delims, scope));
	}

	p;
}

// Declare what the start rule is.
class StartDecl extends lang:bnf:FileItem {
	init(SrcName name) {
		init { name = name; }
	}

	SrcName name;
}
