% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hypothesis_test.R
\name{hypothesis_test}
\alias{hypothesis_test}
\alias{hypothesis_test.default}
\alias{hypothesis_test.ggeffects}
\title{(Pairwise) comparisons between predictions}
\usage{
hypothesis_test(model, ...)

\method{hypothesis_test}{default}(
  model,
  terms = NULL,
  test = "pairwise",
  p_adjust = NULL,
  df = NULL,
  verbose = TRUE,
  ...
)

\method{hypothesis_test}{ggeffects}(
  model,
  test = "pairwise",
  p_adjust = NULL,
  df = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{A fitted model object, or an object of class \code{ggeffects}.}

\item{...}{Arguments passed down to \code{\link[=data_grid]{data_grid()}} when creating the reference
grid.}

\item{terms}{Character vector with the names of the focal terms from \code{model},
for which contrasts or comparisons should be displayed. At least one term
is required, maximum length is three terms. If the first focal term is numeric,
contrasts or comparisons for the \emph{slopes} of this numeric predictor are
computed (possibly grouped by the levels of further categorical focal
predictors).}

\item{test}{Hypothesis to test. By default, pairwise-comparisons are
conducted. See section \emph{Introduction into contrasts and pairwise comparisons}.}

\item{p_adjust}{Character vector, if not \code{NULL}, indicates the method to
adjust p-values. See \code{\link[stats:p.adjust]{stats::p.adjust()}} for details. Further possible
adjustment methods are \code{"tukey"} or \code{"sidak"}. Some caution is necessary
when adjusting p-value for multiple comparisons. See also section
\emph{P-value adjustment} below.}

\item{df}{Degrees of freedom that will be used to compute the p-values and
confidence intervals. If \code{NULL}, degrees of freedom will be extracted from
the model using \code{\link[insight:get_df]{insight::get_df()}} with \code{type = "wald"}.}

\item{verbose}{Toggle messages and warnings.}
}
\value{
A data frame containing predictions (e.g. for \code{test = NULL}),
contrasts or pairwise comparisons of adjusted predictions or estimated
marginal means.
}
\description{
Function to test differences of adjusted predictions for
statistical significance. This is usually called contrasts or (pairwise)
comparisons.
}
\section{Introduction into contrasts and pairwise comparisons}{


There are many ways to test contrasts or pairwise comparisons. A
detailed introduction with many (visual) examples is shown in
\href{https://strengejacke.github.io/ggeffects/articles/introduction_comparisons.html}{this vignette}.
}

\section{P-value adjustment for multiple comparisons}{


Note that p-value adjustment for methods supported by \code{p.adjust()} (see also
\code{p.adjust.methods}), each row is considered as one set of comparisons, no
matter which \code{test} was specified. That is, for instance, when \code{hypothesis_test()}
returns eight rows of predictions (when \code{test = NULL}), and \code{p_adjust = "bonferroni"},
the p-values are adjusted in the same way as if we had a test of pairwise
comparisons (\code{test = "pairwise"}) where eight rows of comparisons are
returned. For methods \code{"tukey"} or \code{"sidak"}, a rank adjustment is done
based on the number of combinations of levels from the focal predictors
in \code{terms}. Thus, the latter two methods may be useful for certain tests
only, in particular pairwise comparisons.
}

\examples{
\dontrun{
if (requireNamespace("marginaleffects") && interactive()) {
  data(efc)
  efc$c172code <- as.factor(efc$c172code)
  efc$c161sex <- as.factor(efc$c161sex)
  levels(efc$c161sex) <- c("male", "female")
  m <- lm(barthtot ~ c12hour + neg_c_7 + c161sex + c172code, data = efc)

  # direct computation of comparisons
  hypothesis_test(m, "c172code")

  # passing a `ggeffects` object
  pred <- ggpredict(m, "c172code")
  hypothesis_test(pred)

  # test for slope
  hypothesis_test(m, "c12hour")

  # interaction - contrasts by groups
  m <- lm(barthtot ~ c12hour + c161sex * c172code + neg_c_7, data = efc)
  hypothesis_test(m, c("c161sex", "c172code"), test = NULL)

  # interaction - pairwise comparisons by groups
  hypothesis_test(m, c("c161sex", "c172code"))

  # p-value adjustment
  hypothesis_test(m, c("c161sex", "c172code"), p_adjust = "tukey")

  # specific comparisons
  hypothesis_test(m, c("c161sex", "c172code"), test = "b2 = b1")

  # interaction - slope by groups
  m <- lm(barthtot ~ c12hour + neg_c_7 * c172code + c161sex, data = efc)
  hypothesis_test(m, c("neg_c_7", "c172code"))
}
}
}
