/*
 * This file is part of libelemental, a periodic table library with detailed
 * information on elements.
 *
 * Copyright (C) 2006-2007 Kevin Daughtridge <kevin@kdau.com>
 * Copyright (C) 2003 Jonas Frantz <jonas.frantz@welho.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include "private.hh"
#include <libelemental/data.hh>

namespace Elemental {


#define F(value, qual) Float (value, qual)
#define FL(count, ...) FloatList ((double[]) { __VA_ARGS__ }, count)
#define IL(count, ...) IntList ((long[]) { __VA_ARGS__ }, count)

	
const unsigned int table_length = 118;


//******************************************************************************


const ElementData table_data[] =
{

{	N_("Hydrogen"),	Q_NA,	"H",	1,	Series::NONMETAL,	1,	1,	Block::S,	Q_NA,	Q_UNK,	0.08988,	N_("Colorless, odorless, tasteless gas"),	Event (1766, N_("England")),	"Henry Cavendish",	N_("Greek: hydor (water) + gennan (to form)"),	1.00794,	14.1,	25,	37,	120,	"154 (-1e)",	LatticeType::HEX,	194,	FL (3, 470.0, 470.0, 340.0),	FL (3, 90.0, 90.0, 120.0),	"1s<sup>1</sup>",	IL (2, 1, -1),	2.20,	72.77,	1312.0,	14.01,	20.28,	0.117,	0.904,	14.304,	0.1805,	110,	color (1.00, 1.00, 1.00),	N_("Where relevant, values given are for H<sub>2</sub>."),	},
{	N_("Helium"),	Q_NA,	"He",	2,	Series::NOBLE_GAS,	18,	1,	Block::S,	Q_NA,	Q_UNK,	0.1786,	N_("Inert, colorless, odorless, tasteless gas"),	Event (1895, N_("Scotland/Sweden")),	"William Ramsay / Per Teodor Cleve, Abraham Langlet",	N_("Greek: helios (sun)"),	4.002602,	31.8,	F (31, Q_EST),	32,	140,	"93",	LatticeType::FCC,	225,	FL (3, 424.2, 424.2, 424.2),	FL (3, 90.0, 90.0, 90.0),	"1s<sup>2</sup>",	Q_NA,	Q_UNK,	F (0, Q_EST),	2372.3,	0.95,	4.22,	0.0138,	0.0829,	5.193,	0.1513,	Q_UNK,	color (0.85, 1.00, 1.00),	N_("Melting point value given is for 2.5 MPa."),	},

{	N_("Lithium"),	Q_NA,	"Li",	3,	Series::ALKALI_METAL,	1,	2,	Block::S,	0.534,	0.512,	Q_NA,	N_("Soft, silvery-white metal"),	Event (1817, N_("Sweden")),	"Johan August Arfwedson",	N_("Greek: lithos (stone)"),	F (6.941, Q_CA),	13.1,	145,	134,	220,	"68 (+1e)",	LatticeType::BCC,	229,	FL (3, 351.0, 351.0, 351.0),	FL (3, 90.0, 90.0, 90.0),	"[He] 2s<sup>1</sup>",	IL (1, 1),	0.98,	59.62,	520.2,	453.69,	1615,	3.00,	147.1,	3.582,	84.8,	400,	color (0.80, 0.50, 1.00),	Q_NA,	},
{	N_("Beryllium"),	Q_NA,	"Be",	4,	Series::ALKALINE_EARTH_METAL,	2,	2,	Block::S,	1.85,	1.690,	Q_NA,	N_("Hard, brittle, steel-gray metal"),	Event (1798, N_("France")),	"Louis Nicolas Vauquelin",	N_("Greek: beryllos (beryl, a mineral)"),	9.012182,	5.0,	105,	90,	190,	"35 (+2e)",	LatticeType::HEX,	194,	FL (3, 228.58, 228.58, 358.43),	FL (3, 90.0, 90.0, 90.0),	"[He] 2s<sup>2</sup>",	IL (1, 2),	1.57,	F (0, Q_EST),	899.5,	1560,	2742,	7.895,	297,	1.825,	200,	1000,	color (0.76, 1.00, 0.00),	Q_NA,	},
{	N_("Boron"),	Q_NA,	"B",	5,	Series::SEMIMETAL,	13,	2,	Block::P,	2.34,	2.08,	Q_NA,	N_("Hard, brittle, lustrous black semimetal"),	Event (1808, N_("England/France")),	N_("Humphry Davy / Joseph Louis Gay-Lussac, Louis Jacques Th&#233;nard"),	N_("Arabic: buraq, Persian: burah (borax)"),	10.811,	4.6,	85,	82,	180,	"23 (+3e)",	LatticeType::RHO,	166,	FL (3, 506.0, 506.0, 506.0),	FL (3, 58.060, 58.060, 58.060),	"[He] 2s<sup>2</sup> 2p<sup>1</sup>",	IL (1, 3),	2.04,	26.99,	800.6,	2349,	4200,	50.2,	480,	1.026,	27.4,	1250,	color (1.00, 0.71, 0.71),	Q_NA,	},
{	N_("Carbon"),	Q_NA,	"C",	6,	Series::NONMETAL,	14,	2,	Block::P,	2.267,	Q_NA,	Q_NA,	N_("Dense, black"),	Q_NA,	Q_UNK,	N_("Latin: carbo (charcoal)"),	12.0107,	5.3,	70,	77,	170,	"16 (+4e), 260 (-4e)",	LatticeType::HEX,	194,	FL (3, 246.4, 246.4, 671.1),	FL (3, 90.0, 90.0, 120.0),	"[He] 2s<sup>2</sup> 2p<sup>2</sup>",	IL (3, 4, 2, -4),	2.55,	121.78,	1086.5,	Q_NA,	F (4000, Q_CA),	117,	Q_NA,	0.709,	F (142, Q_CA),	1860,	color (0.50, 0.50, 0.50),	N_("Where variable, values given are for graphite. At standard pressure, carbon sublimates at the listed boiling point. Thermal conductivity ranges from 119 to 165 W/(m&#183;K)."),	},
{	N_("Nitrogen"),	Q_NA,	"N",	7,	Series::NONMETAL,	15,	2,	Block::P,	Q_NA,	Q_UNK,	1.251,	N_("Colorless, odorless, tasteless, and generally inert gas"),	Event (1772, N_("Scotland")),	"Daniel Rutherford",	N_("Greek: nitron (soda) + gennan (to form)"),	14.0067,	17.3,	65,	75,	160,	"13 (+5e), 171 (-3e)",	LatticeType::HEX,	194,	FL (3, 386.1, 386.1, 626.5),	FL (3, 90.0, 90.0, 120.0),	"[He] 2s<sup>2</sup> 2p<sup>3</sup>",	IL (5, 5, 4, 3, 2, -3),	3.04,	F (0, Q_EST),	1402.3,	63.15,	77.36,	0.720,	5.57,	1.040,	0.02583,	Q_UNK,	color (0.05, 0.05, 1.00),	N_("Where relevant, values given are for N<sub>2</sub>."),	},
{	N_("Oxygen"),	Q_NA,	"O",	8,	Series::NONMETAL,	16,	2,	Block::P,	Q_NA,	Q_UNK,	1.429,	N_("Colorless, odorless, tasteless gas; pale blue liquid"),	Event (1774, N_("England/Sweden")),	"Joseph Priestley / Carl Wilhelm Scheele",	N_("Greek: oxys (acid) + gennan (to form)"),	15.9994,	14.0,	60,	73,	155,	"132 (-2e)",	LatticeType::MONO,	12,	FL (3, 540.3, 342.9, 508.6),	FL (3, 90.0, 132.530, 90.0),	"[He] 2s<sup>2</sup> 2p<sup>4</sup>",	IL (2, -2, -1),	3.44,	141.004,	1313.9,	54.36,	90.20,	0.444,	6.82,	0.918,	0.02658,	Q_UNK,	color (1.00, 0.05, 0.05),	N_("Where relevant, values given are for O<sub>2</sub>."),	},
{	N_("Fluorine"),	Q_NA,	"F",	9,	Series::HALOGEN,	17,	2,	Block::P,	Q_NA,	Q_UNK,	1.7,	N_("Greenish-yellow, pungent, corrosive gas"),	Event (1886, N_("France")),	"Joseph Henri Moissan",	N_("Latin: fluere (to flow)"),	18.9984032,	17.1,	50,	71,	150,	"133 (-1e)",	LatticeType::MONO,	15,	FL (3, 550.0, 328.0, 728.0),	FL (3, 90.0, 90.0, 90.0),	"[He] 2s<sup>2</sup> 2p<sup>5</sup>",	IL (1, -1),	3.98,	328.165,	1681.0,	53.53,	85.03,	0.510,	6.62,	0.824,	0.0277,	Q_UNK,	color (0.70, 1.00, 1.00),	N_("Where relevant, values given are for F<sub>2</sub>."),	},
{	N_("Neon"),	Q_NA,	"Ne",	10,	Series::NOBLE_GAS,	18,	2,	Block::P,	Q_NA,	Q_UNK,	0.9002,	N_("Colorless, odorless, tasteless gas"),	Event (1898, N_("England")),	"William Ramsay, Morris Travers",	N_("Greek: neos (new)"),	20.1797,	16.8,	F (38, Q_EST),	69,	154,	Q_NA,	LatticeType::FCC,	225,	FL (3, 442.9, 442.9, 442.9),	FL (3, 90.0, 90.0, 90.0),	"[He] 2s<sup>2</sup> 2p<sup>6</sup>",	Q_NA,	Q_UNK,	F (0, Q_EST),	2080.7,	24.56,	27.07,	0.335,	1.71,	1.030,	0.0491,	63,	color (0.70, 0.89, 0.96),	Q_NA,	},

{	N_("Sodium"),	"Natrium",	"Na",	11,	Series::ALKALI_METAL,	1,	3,	Block::S,	0.968,	0.927,	Q_NA,	N_("Soft, silvery-white metal"),	Event (1807, N_("England")),	"Humphry Davy",	N_("English: soda; Latin: natrium (sodium carbonate)"),	22.98976928,	23.7,	180,	154,	240,	"97 (+1e)",	LatticeType::BCC,	229,	FL (3, 429.06, 429.06, 429.06),	FL (3, 90.0, 90.0, 90.0),	"[Ne] 3s<sup>1</sup>",	IL (1, 1),	0.93,	52.87,	495.8,	370.87,	1156,	2.60,	97.42,	1.228,	142,	150,	color (0.67, 0.36, 0.95),	Q_NA,	},
{	N_("Magnesium"),	Q_NA,	"Mg",	12,	Series::ALKALINE_EARTH_METAL,	2,	3,	Block::S,	1.738,	1.584,	Q_NA,	N_("Lightweight, malleable, silvery-white metal"),	Event (1755, N_("England")),	"Joseph Black",	N_("Named after Magnesia, a district of Thessaly in Greece."),	24.3050,	14.0,	150,	130,	220,	"66 (+2e)",	LatticeType::HEX,	194,	FL (3, 320.94, 320.94, 524.08),	FL (3, 90.0, 90.0, 120.0),	"[Ne] 3s<sup>2</sup>",	IL (1, 2),	1.31,	F (0, Q_EST),	737.7,	923,	1363,	8.48,	128,	1.023,	156,	318,	color (0.54, 1.00, 0.00),	Q_NA,	},
{	N_("Aluminium"),	"Aluminum",	"Al",	13,	Series::POST_TRANSITION_METAL,	13,	3,	Block::P,	2.70,	2.375,	Q_NA,	N_("Soft, lightweight, silvery-white metal"),	Event (1825, N_("Denmark")),	N_("Hans Christian &#216;rsted"),	N_("Latin: alumen (alum)"),	26.9815386,	10.0,	125,	118,	210,	"51 (+3e)",	LatticeType::FCC,	225,	FL (3, 404.95, 404.95, 404.95),	FL (3, 90.0, 90.0, 90.0),	"[Ne] 3s<sup>2</sup> 3p<sup>1</sup>",	IL (1, 3),	1.61,	41.86,	577.5,	933.47,	2792,	10.71,	294.0,	0.897,	237,	394,	color (0.75, 0.65, 0.65),	Q_NA,	},
{	N_("Silicon"),	Q_NA,	"Si",	14,	Series::SEMIMETAL,	14,	3,	Block::P,	2.33,	2.57,	Q_NA,	N_("Amorphous form is a brown powder; crystalline form is gray and lustrous."),	Event (1823, N_("Sweden")),	N_("J&#246;ns Jakob Berzelius"),	N_("Latin: silex, silicis (flint)"),	28.0855,	12.1,	110,	111,	210,	"42 (+4e), 271 (-4e)",	LatticeType::FCC,	227,	FL (3, 543.09, 543.09, 543.09),	FL (3, 90.0, 90.0, 90.0),	"[Ne] 3s<sup>2</sup> 3p<sup>2</sup>",	IL (2, 4, -4),	1.90,	134.07,	786.5,	1687,	3538,	50.21,	359,	0.705,	149,	625,	color (0.50, 0.60, 0.60),	Q_NA,	},
{	N_("Phosphorus"),	Q_NA,	"P",	15,	Series::NONMETAL,	15,	3,	Block::P,	1.823,	Q_UNK,	Q_NA,	N_("The most common white form is a waxy, phosphorescent solid."),	Event (1669, N_("Germany")),	"Henning Brand",	N_("Greek: phos (light) + phoros (bearer)"),	30.973762,	17.0,	100,	106,	195,	"35 (+5e), 212 (-3e)",	LatticeType::TRI,	2,	FL (3, 1145.0, 505.3, 1126.1),	FL (3, 71.840, 90.370, 71.560),	"[Ne] 3s<sup>2</sup> 3p<sup>3</sup>",	IL (3, 5, 3, -3),	2.19,	72.03,	1011.8,	317.30,	550,	0.66,	12.4,	0.769,	0.236,	Q_UNK,	color (1.00, 0.50, 0.00),	N_("Where variable, values given are for white phosphorus."),	},
{	N_("Sulfur"),	"Sulphur",	"S",	16,	Series::NONMETAL,	16,	3,	Block::P,	1.96,	1.819,	Q_NA,	N_("Tasteless, odorless, light-yellow, brittle solid"),	Q_NA,	Q_UNK,	N_("Latin: sulpur (brimstone), Arabic: sufra (yellow)"),	32.065,	15.5,	100,	102,	180,	"30 (+6e), 184 (-2e)",	LatticeType::ORTH,	70,	FL (3, 1043.7, 1284.5, 2436.9),	FL (3, 90.0, 90.0, 90.0),	"[Ne] 3s<sup>2</sup> 3p<sup>4</sup>",	IL (4, 6, 4, 2, -2),	2.58,	200.410,	999.6,	388.36,	717.8,	1.727,	45,	0.710,	0.205,	Q_UNK,	color (1.00, 1.00, 0.19),	N_("Where variable, values given are for beta form."),	},
{	N_("Chlorine"),	Q_NA,	"Cl",	17,	Series::HALOGEN,	17,	3,	Block::P,	Q_NA,	1.5649,	3.2,	N_("Greenish-yellow, disagreeable gas"),	Event (1774, N_("Sweden")),	"Carl Wilhelm Scheele",	N_("Greek: chloros (greenish yellow)"),	35.453,	18.7,	100,	99,	180,	"27 (+7e), 181 (-1e)",	LatticeType::ORTH,	64,	FL (3, 622.35, 445.61, 817.85),	FL (3, 90.0, 90.0, 90.0),	"[Ne] 3s<sup>2</sup> 3p<sup>5</sup>",	IL (5, 7, 5, 3, 1, -1),	3.16,	349,	1251.2,	171.6,	239.11,	6.406,	20.41,	0.479,	0.0089,	Q_UNK,	color (0.12, 0.94, 0.12),	N_("Where relevant, values given are for Cl<sub>2</sub>. Liquid density value is for -35&#176;C."),	},
{	N_("Argon"),	Q_NA,	"Ar",	18,	Series::NOBLE_GAS,	18,	3,	Block::P,	Q_NA,	Q_UNK,	1.784,	N_("Colorless, tasteless, odorless noble gas"),	Event (1894, N_("Scotland")),	"Baron Rayleigh, William Ramsay",	N_("Greek: argos (inactive)"),	39.948,	24.2,	71,	97,	188,	Q_NA,	LatticeType::FCC,	225,	FL (3, 525.6, 525.6, 525.6),	FL (3, 90.0, 90.0, 90.0),	"[Ne] 3s<sup>2</sup> 3p<sup>6</sup>",	Q_NA,	Q_UNK,	F (0, Q_EST),	1520.6,	83.80,	87.30,	1.18,	6.43,	0.520,	0.01772,	85,	color (0.50, 0.82, 0.89),	Q_NA,	},

{	N_("Potassium"),	"Kalium",	"K",	19,	Series::ALKALI_METAL,	1,	4,	Block::S,	0.89,	0.828,	Q_NA,	N_("Soft, waxy, silvery-white metal"),	Event (1807, N_("England")),	"Humphry Davy",	N_("English: pot + ash; Arabic: al qaliy"),	39.0983,	45.3,	220,	196,	280,	"133 (+1e)",	LatticeType::BCC,	229,	FL (3, 532.8, 532.8, 532.8),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 4s<sup>1</sup>",	IL (1, 1),	0.82,	48.38,	418.8,	336.53,	1032,	2.231,	76.90,	0.757,	102.5,	100,	color (0.56, 0.25, 0.83),	Q_NA,	},
{	N_("Calcium"),	Q_NA,	"Ca",	20,	Series::ALKALINE_EARTH_METAL,	2,	4,	Block::S,	1.55,	1.378,	Q_NA,	N_("Fairly hard, silvery-white metal"),	Event (1808, N_("England")),	"Humphry Davy",	N_("Latin: calx (lime)"),	40.078,	29.9,	180,	174,	240,	"99 (+2e)",	LatticeType::FCC,	225,	FL (3, 558.84, 558.84, 558.84),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 4s<sup>2</sup>",	IL (1, 2),	1.00,	2.37,	589.8,	1115,	1757,	8.54,	154.7,	0.647,	201,	230,	color (0.24, 1.00, 0.00),	Q_NA,	},
{	N_("Scandium"),	Q_NA,	"Sc",	21,	Series::TRANSITION_METAL,	3,	4,	Block::D,	2.985,	2.80,	Q_NA,	N_("Fairly soft, silvery-white metal"),	Event (1879, N_("Sweden")),	"Lars Fredrik Nilson",	N_("Named after Scandinavia (Latin: Scandia)."),	44.955912,	15.0,	160,	144,	230,	"72.3 (+3e)",	LatticeType::HEX,	194,	FL (3, 330.9, 330.9, 527.33),	FL (3, 90.0, 90.0, 120.0),	"[Ar] 3d<sup>1</sup> 4s<sup>2</sup>",	IL (1, 3),	1.36,	18,	633.1,	1814,	3109,	14.1,	332.7,	0.568,	15.8,	Q_UNK,	color (0.90, 0.90, 0.90),	Q_NA,	},
{	N_("Titanium"),	Q_NA,	"Ti",	22,	Series::TRANSITION_METAL,	4,	4,	Block::D,	4.506,	4.11,	Q_NA,	N_("Shiny, dark-gray metal"),	Event (1791, N_("England")),	"William Gregor",	N_("Greek: Titanes (early deities)"),	47.867,	10.6,	140,	136,	215,	"68 (+4e), 94 (+2e)",	LatticeType::HEX,	194,	FL (3, 295.8, 295.8, 468.55),	FL (3, 90.0, 90.0, 120.0),	"[Ar] 3d<sup>2</sup> 4s<sup>2</sup>",	IL (2, 4, 3),	1.54,	8.4,	658.8,	1941,	3560,	14.15,	425,	0.523,	21.9,	380,	color (0.75, 0.76, 0.78),	Q_NA,	},
{	N_("Vanadium"),	Q_NA,	"V",	23,	Series::TRANSITION_METAL,	5,	4,	Block::D,	6.0,	5.5,	Q_NA,	N_("Soft, ductile, silvery-white metal"),	Event (1801, N_("Mexico")),	N_("Andr&#233;s Manuel del R&#237;o"),	N_("Old Norse: Vanadis (Lady of the Vanir, i.e. Freya, a fertility deity)"),	50.9415,	8.35,	135,	125,	205,	"59 (+5e), 74 (+3e)",	LatticeType::BCC,	229,	FL (3, 303.0, 303.0, 303.0),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>3</sup> 4s<sup>2</sup>",	IL (5, 5, 4, 3, 2, 0),	1.63,	51,	650.9,	2183,	3680,	21.5,	459,	0.489,	30.7,	390,	color (0.65, 0.65, 0.67),	Q_NA,	},
{	N_("Chromium"),	Q_NA,	"Cr",	24,	Series::TRANSITION_METAL,	6,	4,	Block::D,	7.15,	6.3,	Q_NA,	N_("Very hard, crystalline, steel-gray metal"),	Event (1797, N_("France")),	"Louis Nicolas Vauquelin",	N_("Greek: chroma (color)"),	51.9961,	7.23,	140,	127,	205,	"52 (+6e), 63 (+3e)",	LatticeType::BCC,	229,	FL (3, 291.0, 291.0, 291.0),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>5</sup> 4s<sup>1</sup>",	IL (4, 6, 3, 2, 0),	1.66,	65.2,	652.9,	2180,	2944,	21.0,	339.5,	0.449,	93.9,	460,	color (0.54, 0.60, 0.78),	Q_NA,	},
{	N_("Manganese"),	Q_NA,	"Mn",	25,	Series::TRANSITION_METAL,	7,	4,	Block::D,	7.21,	5.95,	Q_NA,	N_("Hard, brittle, gray-white metal"),	Event (1774, N_("Sweden")),	"Johann Gottlieb Gahn",	N_("Italian: manganese, Latin: magnes (magnet)"),	54.938045,	7.39,	140,	139,	205,	"46 (+7e), 80 (+2e)",	LatticeType::SC,	217,	FL (3, 891.25, 891.25, 891.25),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>5</sup> 4s<sup>2</sup>",	IL (7, 7, 6, 4, 3, 2, 0, -1),	1.55,	F (0, Q_EST),	717.3,	1519,	2334,	12.91,	221,	0.479,	7.81,	400,	color (0.61, 0.48, 0.78),	Q_NA,	},
{	N_("Iron"),	"Ferrum",	"Fe",	26,	Series::TRANSITION_METAL,	8,	4,	Block::D,	7.86,	6.98,	Q_NA,	N_("Malleable, ductile, silvery-white metal"),	Q_NA,	Q_UNK,	Message (N_("Etruscan: aisar (deities); Latin: ferrum"), Q_EST),	55.845,	7.1,	140,	125,	205,	"64 (+3e), 74 (+2e)",	LatticeType::BCC,	229,	FL (3, 286.65, 286.65, 286.65),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>6</sup> 4s<sup>2</sup>",	IL (5, 6, 3, 2, 0, -2),	1.83,	14.6,	762.5,	1811,	3134,	13.81,	340,	0.449,	80.4,	460,	color (0.50, 0.48, 0.78),	Q_NA,	},
{	N_("Cobalt"),	Q_NA,	"Co",	27,	Series::TRANSITION_METAL,	9,	4,	Block::D,	8.90,	7.75,	Q_NA,	N_("Hard, ductile, lustrous bluish-gray metal"),	Event (1737, N_("Sweden")),	"George Brandt",	N_("German: kobold (evil spirit)"),	58.933195,	6.7,	135,	126,	Q_UNK,	"63 (+3e), 72 (+2e)",	LatticeType::HEX,	194,	FL (3, 250.71, 250.71, 406.95),	FL (3, 90.0, 90.0, 120.0),	"[Ar] 3d<sup>7</sup> 4s<sup>2</sup>",	IL (4, 3, 2, 0, -1),	1.88,	64.0,	760.4,	1768,	3200,	16.06,	377,	0.421,	100,	385,	color (0.44, 0.48, 0.78),	Q_NA,	},
{	N_("Nickel"),	Q_NA,	"Ni",	28,	Series::TRANSITION_METAL,	10,	4,	Block::D,	8.908,	7.81,	Q_NA,	N_("Hard, malleable, silvery-white metal"),	Event (1751, N_("Sweden")),	"Axel Fredrik Cronstedt",	N_("German: kupfernickel (false copper)"),	58.6934,	6.6,	135,	121,	Q_UNK,	"69 (+2e)",	LatticeType::FCC,	225,	FL (3, 352.4, 352.4, 352.4),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>8</sup> 4s<sup>2</sup>",	IL (3, 3, 2, 0),	1.91,	111.6,	737.1,	1728,	3186,	17.48,	377.5,	0.444,	90.9,	375,	color (0.36, 0.48, 0.76),	Q_NA,	},
{	N_("Copper"),	"Cuprum",	"Cu",	29,	Series::TRANSITION_METAL,	11,	4,	Block::D,	8.96,	8.02,	Q_NA,	N_("Malleable, ductile, reddish-brown metal"),	Q_NA,	Q_UNK,	N_("Late Latin: cuprum, Latin: [aes] Cyprium (metal of Cyprus)"),	63.546,	7.1,	135,	138,	Q_UNK,	"72 (+2e), 96 (+1e)",	LatticeType::FCC,	225,	FL (3, 361.49, 361.49, 361.49),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>10</sup> 4s<sup>1</sup>",	IL (2, 2, 1),	1.90,	119.24,	745.5,	1357.77,	2835,	13.26,	300.4,	0.385,	401,	315,	color (1.00, 0.48, 0.38),	Q_NA,	},
{	N_("Zinc"),	Q_NA,	"Zn",	30,	Series::TRANSITION_METAL,	12,	4,	Block::D,	7.14,	6.57,	Q_NA,	N_("Bluish-silver, ductile metal"),	Event (1526, N_("Switzerland")),	"Paracelsus",	N_("German: zink (sharp point, prong, tooth)"),	65.409,	9.2,	135,	131,	210,	"74 (+2e)",	LatticeType::HEX,	194,	FL (3, 266.49, 266.49, 494.68),	FL (3, 90.0, 90.0, 120.0),	"[Ar] 3d<sup>10</sup> 4s<sup>2</sup>",	IL (1, 2),	1.65,	F (0, Q_EST),	906.4,	692.68,	1180,	7.32,	123.6,	0.388,	116,	234,	color (0.49, 0.50, 0.69),	Q_NA,	},
{	N_("Gallium"),	Q_NA,	"Ga",	31,	Series::POST_TRANSITION_METAL,	13,	4,	Block::P,	5.91,	6.095,	Q_NA,	N_("Soft, blue-white metal"),	Event (1875, N_("France")),	N_("Paul &#201;mile Lecoq de Boisbaudran"),	N_("Latin: Gallia (Gaul, an ancient area including modern France)"),	69.723,	11.8,	130,	126,	210,	"62 (+3e), 81 (+1e)",	LatticeType::ORTH,	64,	FL (3, 451.97, 766.33, 452.6),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>10</sup> 4s<sup>2</sup> 4p<sup>1</sup>",	IL (1, 3),	1.81,	41,	578.8,	302.9146,	2477,	5.59,	254,	0.371,	40.6,	240,	color (0.76, 0.56, 0.56),	Q_NA,	},
{	N_("Germanium"),	Q_NA,	"Ge",	32,	Series::SEMIMETAL,	14,	4,	Block::P,	5.323,	5.60,	Q_NA,	N_("Grayish-white metal"),	Event (1886, N_("Germany")),	"Clemens Winkler",	N_("Named after Germany (Latin: Germania), where it was discovered."),	72.64,	13.6,	125,	122,	210,	"53 (+4e), 73 (+2e)",	LatticeType::FCC,	225,	FL (3, 565.75, 565.75, 565.75),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>10</sup> 4s<sup>2</sup> 4p<sup>2</sup>",	IL (1, 4),	2.01,	118.94,	762,	1211.40,	3106,	36.94,	334,	0.320,	60.2,	360,	color (0.40, 0.56, 0.56),	Q_NA,	},
{	N_("Arsenic"),	Q_NA,	"As",	33,	Series::SEMIMETAL,	15,	4,	Block::P,	5.727,	5.22,	Q_NA,	N_("Steel gray, brittle semimetal"),	Event (1250, N_("Germany")),	"Albertus Magnus",	N_("Persian: zarnik (yellow pigment)"),	74.92160,	13.1,	115,	119,	205,	"46 (+5e), 222 (-3e)",	LatticeType::RHO,	166,	FL (3, 375.98, 375.98, 1054.75),	FL (3, 90.0, 90.0, 120.0),	"[Ar] 3d<sup>10</sup> 4s<sup>2</sup> 4p<sup>3</sup>",	IL (3, 5, 3, -2),	2.18,	78.5,	947.0,	1090,	887,	24.44,	Q_NA,	0.329,	50.2,	285,	color (0.74, 0.50, 0.89),	Q_NA,	},
{	N_("Selenium"),	Q_NA,	"Se",	34,	Series::NONMETAL,	16,	4,	Block::P,	4.81,	3.99,	Q_NA,	N_("A soft gray solid, similar to sulfur"),	Event (1817, N_("Sweden")),	N_("J&#246;ns Jakob Berzelius"),	N_("Greek: selene (the moon)"),	78.96,	16.5,	115,	116,	190,	"42 (+6e), 191 (-2e)",	LatticeType::MONO,	14,	FL (3, 905.4, 908.3, 1160.1),	FL (3, 90.0, 90.810, 90.0),	"[Ar] 3d<sup>10</sup> 4s<sup>2</sup> 4p<sup>4</sup>",	IL (3, 6, 4, -2),	2.55,	194.97,	941.0,	494,	958,	6.69,	95.48,	0.321,	0.519,	Q_UNK,	color (1.00, 0.63, 0.00),	N_("Thermal conductivity value is for amorphous selenium. Where variable, other values are for gray selenium."),	},
{	N_("Bromine"),	Q_NA,	"Br",	35,	Series::HALOGEN,	17,	4,	Block::P,	Q_NA,	3.1028,	Q_NA,	N_("Reddish-brown liquid"),	Event (1826, N_("France")),	N_("Antoine J&#233;r&#244;me Balard"),	N_("Greek: bromos (stench)"),	79.904,	23.5,	115,	114,	190,	"47 (+5e), 196 (-1e)",	LatticeType::ORTH,	64,	FL (3, 672.65, 464.51, 870.23),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>10</sup> 4s<sup>2</sup> 4p<sup>5</sup>",	IL (5, 7, 5, 3, 1, -1),	2.96,	342.54,	1139.9,	265.8,	332.0,	10.57,	29.96,	0.474,	0.122,	Q_UNK,	color (0.65, 0.16, 0.16),	N_("Where relevant, values given are for Br<sub>2</sub>. Liquid density value is for ~20&#176;C."),	},
{	N_("Krypton"),	Q_NA,	"Kr",	36,	Series::NOBLE_GAS,	18,	4,	Block::P,	Q_NA,	Q_UNK,	3.749,	N_("Dense, colorless, odorless, and tasteless gas"),	Event (1898, N_("Great Britain")),	"William Ramsay, Morris Travers",	N_("Greek: kryptos (hidden)"),	83.798,	32.2,	F (88, Q_EST),	110,	202,	Q_NA,	LatticeType::FCC,	225,	FL (3, 570.6, 570.6, 570.6),	FL (3, 90.0, 90.0, 90.0),	"[Ar] 3d<sup>10</sup> 4s<sup>2</sup> 4p<sup>6</sup>",	IL (1, 2),	3.00,	F (0, Q_EST),	1350.8,	115.79,	119.93,	1.64,	9.08,	0.248,	0.00943,	Q_UNK,	color (0.36, 0.72, 0.82),	Q_NA,	},

{	N_("Rubidium"),	Q_NA,	"Rb",	37,	Series::ALKALI_METAL,	1,	5,	Block::S,	1.532,	1.46,	Q_NA,	N_("Soft, silvery-white, highly reactive metal"),	Event (1860, N_("Germany")),	"Robert Bunsen, Gustav Kirchhoff",	N_("Latin: rubidus (deep red, for the flames from its salts)"),	85.4678,	55.9,	235,	211,	290,	"147 (+1e)",	LatticeType::BCC,	229,	FL (3, 558.5, 558.5, 558.5),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 5s<sup>1</sup>",	IL (1, 1),	0.82,	46.89,	403.0,	312.46,	961,	2.19,	75.77,	0.363,	58.2,	Q_UNK,	color (0.44, 0.18, 0.69),	Q_NA,	},
{	N_("Strontium"),	Q_NA,	"Sr",	38,	Series::ALKALINE_EARTH_METAL,	2,	5,	Block::S,	2.64,	6.980,	Q_NA,	N_("Silvery, malleable metal"),	Event (1790, N_("Scotland")),	"Adair Crawford",	N_("Named after Strontian, Scotland."),	87.62,	33.7,	200,	192,	255,	"112 (+2e)",	LatticeType::FCC,	225,	FL (3, 608.49, 608.49, 608.49),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 5s<sup>2</sup>",	IL (1, 2),	0.95,	5.02,	549.5,	1050,	1655,	7.43,	136.9,	0.301,	35.4,	Q_UNK,	color (0.00, 1.00, 0.00),	Q_NA,	},
{	N_("Yttrium"),	Q_NA,	"Y",	39,	Series::TRANSITION_METAL,	3,	5,	Block::D,	4.472,	4.24,	Q_NA,	N_("Silvery, ductile, fairly reactive metal"),	Event (1794, N_("Finland")),	"Johan Gadolin",	N_("Named after Ytterby, Sweden, where one of its minerals was first found."),	88.90585,	19.8,	180,	162,	240,	"89.3 (+3e)",	LatticeType::HEX,	194,	FL (3, 364.74, 364.74, 573.06),	FL (3, 90.0, 90.0, 120.0),	"[Kr] 4d<sup>1</sup> 5s<sup>2</sup>",	IL (1, 3),	1.22,	30,	600,	1799,	3609,	11.42,	365,	0.298,	17.2,	Q_UNK,	color (0.58, 1.00, 1.00),	Q_NA,	},
{	N_("Zirconium"),	Q_NA,	"Zr",	40,	Series::TRANSITION_METAL,	4,	5,	Block::D,	6.52,	5.8,	Q_NA,	N_("Gray-white, lustrous, corrosion-resistant metal"),	Event (1789, N_("Germany")),	"Martin Heinrich Klaproth",	N_("Arabic: zarkun, Persian: zargun (gold-like)"),	91.224,	14.1,	155,	148,	230,	"79 (+4e)",	LatticeType::HEX,	194,	FL (3, 323.2, 323.2, 514.7),	FL (3, 90.0, 90.0, 120.0),	"[Kr] 4d<sup>2</sup> 5s<sup>2</sup>",	IL (1, 4),	1.33,	41,	640.1,	2128,	4682,	14,	573,	0.278,	22.6,	250,	color (0.58, 0.88, 0.88),	Q_NA,	},
{	N_("Niobium"),	Q_NA,	"Nb",	41,	Series::TRANSITION_METAL,	5,	5,	Block::D,	8.57,	Q_UNK,	Q_NA,	N_("Shiny white, soft, ductile metal"),	Event (1801, N_("England")),	"Charles Hatchett",	N_("Greek: Niobe (mythological daughter of Tantalus)"),	92.90638,	10.8,	145,	137,	215,	"69 (+5e)",	LatticeType::BCC,	229,	FL (3, 330.04, 330.04, 330.04),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>4</sup> 5s<sup>1</sup>",	IL (2, 5, 3),	1.6,	86,	652.1,	2750,	5017,	30,	689.9,	0.265,	53.7,	275,	color (0.45, 0.76, 0.79),	Q_NA,	},
{	N_("Molybdenum"),	Q_NA,	"Mo",	42,	Series::TRANSITION_METAL,	6,	5,	Block::D,	10.28,	9.33,	Q_NA,	N_("Silvery white, hard metal"),	Event (1778, N_("Sweden")),	"Carl Wilhelm Scheele",	N_("Greek: molybdos (lead-like)"),	95.94,	9.4,	145,	145,	210,	"62 (+6e), 70 (+4e)",	LatticeType::BCC,	229,	FL (3, 314.7, 314.7, 314.7),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>5</sup> 5s<sup>1</sup>",	IL (5, 6, 5, 4, 3, 2, 0),	2.16,	72.3,	684.3,	2896,	4912,	37.48,	617,	0.251,	138,	380,	color (0.33, 0.71, 0.71),	Q_NA,	},
{	N_("Technetium"),	Q_NA,	"Tc",	43,	Series::TRANSITION_METAL,	7,	5,	Block::D,	F (11, Q_ISO),	Q_UNK,	Q_NA,	N_("Silvery-gray metal"),	Event (1937, N_("Italy")),	N_("Carlo Perrier, Emilio Segr&#232;"),	N_("Greek: technetos (artificial)"),	F (98, Q_ISO),	8.5,	135,	156,	205,	"56 (+7e)",	LatticeType::HEX,	194,	FL (3, 273.5, 273.5, 438.8),	FL (3, 90.0, 90.0, 120.0),	"[Kr] 4d<sup>5</sup> 5s<sup>2</sup>",	IL (1, 7),	1.9,	F (0, Q_EST),	702,	F (2430, Q_ISO),	F (4538, Q_ISO),	33.29,	585.2,	F (0.25, Q_ISO),	50.6,	Q_UNK,	color (0.23, 0.62, 0.62),	Q_NA,	},
{	N_("Ruthenium"),	Q_NA,	"Ru",	44,	Series::TRANSITION_METAL,	8,	5,	Block::D,	12.45,	10.65,	Q_NA,	N_("Rare, silver-gray, extremely brittle metal"),	Event (1844, N_("Russia")),	"Karl Klaus",	N_("Named after Russia (Latin: Ruthenia), where it was discovered."),	101.07,	8.3,	130,	126,	205,	"67 (+4e)",	LatticeType::HEX,	194,	FL (3, 270.59, 270.59, 428.15),	FL (3, 90.0, 90.0, 120.0),	"[Kr] 4d<sup>7</sup> 5s<sup>1</sup>",	IL (7, 8, 6, 4, 3, 2, 0, -2),	2.2,	101.0,	710.2,	2607,	4423,	38.59,	591.6,	0.238,	117,	Q_UNK,	color (0.14, 0.56, 0.56),	Q_NA,	},
{	N_("Rhodium"),	Q_NA,	"Rh",	45,	Series::TRANSITION_METAL,	9,	5,	Block::D,	12.41,	10.7,	Q_NA,	N_("Silvery white, hard metal"),	Event (1803, N_("England")),	"William Hyde Wollaston",	N_("Greek: rhodon (rose, for solutions of its salts)"),	102.90550,	8.3,	135,	135,	Q_UNK,	"68 (+3e)",	LatticeType::FCC,	225,	FL (3, 380.34, 380.34, 380.34),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>8</sup> 5s<sup>1</sup>",	IL (6, 5, 4, 3, 2, 1, 0),	2.28,	110.3,	719.7,	2237,	3968,	26.59,	494,	0.243,	150,	Q_UNK,	color (0.04, 0.49, 0.55),	Q_NA,	},
{	N_("Palladium"),	Q_NA,	"Pd",	46,	Series::TRANSITION_METAL,	10,	5,	Block::D,	12.023,	10.38,	Q_NA,	N_("Silvery-white, soft, malleable and ductile metal"),	Event (1803, N_("England")),	"William Hyde Wollaston",	N_("Named after the asteroid Pallas, then recently discovered."),	106.42,	8.9,	140,	131,	205,	"65 (+4e), 80 (+2e)",	LatticeType::FCC,	225,	FL (3, 389.07, 389.07, 389.07),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>10</sup>",	IL (3, 4, 2, 0),	2.20,	54.24,	804.4,	1828.05,	3236,	16.74,	362,	0.244,	71.8,	275,	color (0.00, 0.41, 0.52),	Q_NA,	},
{	N_("Silver"),	"Argentum",	"Ag",	47,	Series::TRANSITION_METAL,	11,	5,	Block::D,	10.49,	9.320,	Q_NA,	N_("Silvery, ductile, and malleable metal"),	Q_NA,	Q_UNK,	N_("Old English: seolfor; Latin: argentum"),	107.8682,	10.3,	160,	153,	210,	"89 (+2e), 126 (+1e)",	LatticeType::FCC,	225,	FL (3, 408.53, 408.53, 408.53),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>10</sup> 5s<sup>1</sup>",	IL (2, 2, 1),	1.93,	125.86,	731.0,	1234.93,	2435,	11.28,	258,	0.235,	429,	215,	color (0.88, 0.88, 1.00),	Q_NA,	},
{	N_("Cadmium"),	Q_NA,	"Cd",	48,	Series::TRANSITION_METAL,	12,	5,	Block::D,	8.65,	7.996,	Q_NA,	N_("Soft, malleable, blue-white metal"),	Event (1817, N_("Germany")),	"Friedrich Strohmeyer",	N_("Greek: kadmeia (calamine)"),	112.411,	13.1,	155,	148,	220,	"97 (+2e)",	LatticeType::HEX,	194,	FL (3, 297.94, 297.94, 561.86),	FL (3, 90.0, 90.0, 120.0),	"[Kr] 4d<sup>10</sup> 5s<sup>2</sup>",	IL (1, 2),	1.69,	F (0, Q_EST),	867.8,	594.22,	1040,	6.21,	99.87,	0.232,	96.6,	120,	color (1.00, 0.85, 0.56),	Q_NA,	},
{	N_("Indium"),	Q_NA,	"In",	49,	Series::POST_TRANSITION_METAL,	13,	5,	Block::P,	7.31,	7.02,	Q_NA,	N_("Very soft, silvery-white metal"),	Event (1863, N_("Germany")),	"Ferdinand Reich, Theodor Richter",	N_("Latin: indicum (indigo, after the line of that color in its spectrum)"),	114.818,	15.7,	155,	144,	220,	"81 (+3e)",	LatticeType::TET,	139,	FL (3, 325.23, 325.23, 494.61),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>10</sup> 5s<sup>2</sup> 5p<sup>1</sup>",	IL (1, 3),	1.78,	39,	558.3,	429.75,	2345,	3.281,	231.8,	0.233,	81.8,	129,	color (0.65, 0.46, 0.45),	Q_NA,	},
{	N_("Tin"),	"Stannum",	"Sn",	50,	Series::POST_TRANSITION_METAL,	14,	5,	Block::P,	7.265,	6.99,	Q_NA,	N_("Silvery-white, soft, malleable and ductile metal"),	Q_NA,	Q_UNK,	Message (N_("pre-Indo-European; Latin: stannum"), Q_EST),	118.710,	16.3,	145,	141,	225,	"71 (+4e), 93 (+2)",	LatticeType::TET,	141,	FL (3, 583.18, 583.18, 318.19),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>10</sup> 5s<sup>2</sup> 5p<sup>2</sup>",	IL (2, 4, 2),	1.96,	107.30,	708.6,	505.08,	2875,	7.03,	296.1,	0.228,	66.8,	170,	color (0.40, 0.50, 0.50),	N_("Where variable, values given are for white form."),	},
{	N_("Antimony"),	"Stibium",	"Sb",	51,	Series::SEMIMETAL,	15,	5,	Block::P,	6.697,	6.53,	Q_NA,	N_("Hard, silvery-white, brittle semimetal"),	Q_NA,	Q_NA,	Message (N_("Greek: anti (against) + monos (alone); Latin: stibium (one of its minerals)"), Q_EST),	121.760,	18.4,	145,	138,	220,	"62 (+6e), 245 (-3)",	LatticeType::RHO,	166,	FL (3, 430.7, 430.7, 1127.3),	FL (3, 90.0, 90.0, 120.0),	"[Kr] 4d<sup>10</sup> 5s<sup>2</sup> 5p<sup>3</sup>",	IL (3, 5, 3, -2),	2.05,	101.06,	834,	903.78,	1860,	19.79,	193.43,	0.207,	24.4,	200,	color (0.62, 0.39, 0.71),	Q_NA,	},
{	N_("Tellurium"),	Q_NA,	"Te",	52,	Series::SEMIMETAL,	16,	5,	Block::P,	6.24,	5.70,	Q_NA,	N_("Silvery-white, brittle semimetal"),	Event (1782, N_("Romania")),	N_("Franz-Joseph M&#252;ller von Reichenstein"),	N_("Latin: tellus (earth)"),	127.60,	20.5,	140,	135,	210,	"56 (+6e), 211 (-2e)",	LatticeType::RHO,	152,	FL (3, 445.72, 445.72, 592.9),	FL (3, 90.0, 90.0, 120.0),	"[Kr] 4d<sup>10</sup> 5s<sup>2</sup> 5p<sup>4</sup>",	IL (3, 6, 4, 2),	2.1,	190.16,	869.3,	722.66,	1261,	17.49,	114.1,	0.202,	F (2.6, Q_CA),	Q_UNK,	color (0.83, 0.48, 0.00),	N_("Thermal conductivity ranges from 1.97 to 3.38 W/(m&#183;K)."),	},
{	N_("Iodine"),	Q_NA,	"I",	53,	Series::HALOGEN,	17,	5,	Block::P,	4.933,	Q_UNK,	Q_NA,	N_("Shiny, black nonmetallic solid"),	Event (1811, N_("France")),	"Bernard Courtois",	N_("Greek: iodes (violet-colored)"),	126.90447,	25.7,	140,	133,	210,	"50 (+7e), 220 (-1e)",	LatticeType::ORTH,	64,	FL (3, 718.02, 471.02, 981.03),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>10</sup> 5s<sup>2</sup> 5p<sup>5</sup>",	IL (4, 7, 5, 1, -1),	2.66,	295,	1008.4,	386.85,	457.4,	15.52,	41.57,	0.214,	0.449,	Q_UNK,	color (0.58, 0.00, 0.58),	N_("Where relevant, values given are for I<sub>2</sub>."),	},
{	N_("Xenon"),	Q_NA,	"Xe",	54,	Series::NOBLE_GAS,	18,	5,	Block::P,	Q_NA,	Q_UNK,	5.894,	N_("Heavy, colorless, and odorless noble gas"),	Event (1898, N_("England")),	"William Ramsay, Morris Travers",	N_("Greek: xenon (stranger)"),	131.293,	42.9,	F (108, Q_EST),	130,	216,	Q_NA,	LatticeType::FCC,	225,	FL (3, 620.23, 620.23, 620.23),	FL (3, 90.0, 90.0, 90.0),	"[Kr] 4d<sup>10</sup> 5s<sup>2</sup> 5p<sup>6</sup>",	IL (1, 7),	2.6,	F (0, Q_EST),	1170.4,	161.4,	165.03,	2.27,	12.64,	0.158,	0.00565,	Q_UNK,	color (0.26, 0.62, 0.69),	Q_NA,	},

{	N_("Caesium"),	"Cesium",	"Cs",	55,	Series::ALKALI_METAL,	1,	6,	Block::S,	1.93,	1.843,	Q_NA,	N_("Very soft, ductile, light gray metal"),	Event (1860, N_("Germany")),	"Robert Bunsen, Gustav Kirchhoff",	N_("Latin: caesius (sky blue, for the blue lines of its spectrum)"),	132.9054519,	70.0,	260,	225,	300,	"167 (+1e)",	LatticeType::BCC,	229,	FL (3, 614.1, 614.1, 614.1),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 6s<sup>1</sup>",	IL (1, 1),	0.79,	45.51,	375.7,	301.59,	944,	2.09,	63.9,	0.242,	35.9,	Q_UNK,	color (0.34, 0.09, 0.56),	Q_NA,	},
{	N_("Barium"),	Q_NA,	"Ba",	56,	Series::ALKALINE_EARTH_METAL,	2,	6,	Block::S,	3.51,	3.338,	Q_NA,	N_("Soft, slightly malleable, silver-white metal"),	Event (1808, N_("England")),	"Humphry Davy",	N_("Greek: barys (heavy or dense)"),	137.327,	39.0,	215,	198,	270,	"134 (+2e)",	LatticeType::BCC,	229,	FL (3, 502.8, 502.8, 502.8),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 6s<sup>2</sup>",	IL (1, 2),	0.89,	13.95,	502.9,	1000,	2170,	7.12,	140.3,	0.204,	18.4,	Q_UNK,	color (0.00, 0.79, 0.00),	Q_NA,	},

{	N_("Lanthanum"),	Q_NA,	"La",	57,	Series::LANTHANIDE,	3,	6,	Block::F,	6.162,	5.94,	Q_NA,	N_("Silvery-white, soft, malleable, and ductile metal"),	Event (1839, N_("Sweden")),	"Carl Gustaf Mosander",	N_("Greek: lanthanein (to lie hidden)"),	138.90547,	22.5,	195,	169,	250,	"101.6 (+3e)",	LatticeType::HEX,	194,	FL (3, 377.2, 377.2, 1214.4),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 5d<sup>1</sup> 6s<sup>2</sup>",	IL (1, 3),	1.10,	45,	538.1,	1193,	3737,	6.20,	402.1,	0.195,	13.4,	132,	color (0.44, 0.83, 1.00),	Q_NA,	},
{	N_("Cerium"),	Q_NA,	"Ce",	58,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	6.770,	6.55,	Q_NA,	N_("Malleable, ductile, iron-gray metal"),	Event (1803, N_("Germany/Sweden")),	N_("Martin Heinrich Klaproth / J&#246;ns Jakob Berzelius, Wilhelm von Hisinger"),	N_("Named after the asteroid Ceres, then recently discovered."),	140.116,	21.0,	185,	Q_UNK,	248,	"92 (+4e), 103.4 (+3e)",	LatticeType::HEX,	194,	FL (3, 362.0, 362.0, 599.0),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>1</sup> 5d<sup>1</sup> 6s<sup>2</sup>",	IL (2, 4, 3),	1.12,	92,	534.4,	1068,	3716,	5.46,	398,	0.192,	11.3,	Q_UNK,	color (1.00, 1.00, 0.78),	Q_NA,	},
{	N_("Praseodymium"),	Q_NA,	"Pr",	59,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	6.77,	6.50,	Q_NA,	N_("Silvery white, moderately soft, malleable, and ductile metal"),	Event (1885, N_("Austria")),	"Carl Auer von Welsbach",	N_("Greek: prasios (green, for its salts) + didymos (twin)"),	140.90765,	20.8,	185,	Q_UNK,	247,	"90 (+4e), 101.3 (+3e)",	LatticeType::HEX,	194,	FL (3, 367.25, 367.25, 1183.54),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>3</sup> 6s<sup>2</sup>",	IL (2, 4, 3),	1.13,	Q_UNK,	527,	1208,	3793,	6.89,	331,	0.193,	12.5,	Q_UNK,	color (0.85, 1.00, 0.78),	Q_NA,	},
{	N_("Neodymium"),	Q_NA,	"Nd",	60,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	7.01,	6.89,	Q_NA,	N_("Silvery-white, rare-earth metal that oxidizes easily in air"),	Event (1885, N_("Austria")),	"Carl Auer von Welsbach",	N_("Greek: neos (new) + didymos (twin)"),	144.242,	20.6,	185,	Q_UNK,	245,	"99.5 (+3e)",	LatticeType::HEX,	194,	FL (3, 365.8, 365.8, 1179.9),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>4</sup> 6s<sup>2</sup>",	IL (1, 3),	1.14,	Q_UNK,	533.1,	1297,	3347,	7.14,	289,	0.190,	16.5,	Q_UNK,	color (0.78, 1.00, 0.78),	Q_NA,	},
{	N_("Promethium"),	Q_NA,	"Pm",	61,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	F (7.26, Q_ISO),	Q_UNK,	Q_NA,	Q_UNK,	Event (1945, N_("United States")),	"Jacob A. Marinsky, Lawrence E. Glendenin, Charles D. Coryell",	N_("Greek: Prometheus (a mythological Titan)"),	F (145, Q_ISO),	Q_UNK,	185,	Q_UNK,	243,	"97.9 (+3e)",	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Xe] 4f<sup>5</sup> 6s<sup>2</sup>",	IL (1, 3),	Q_UNK,	Q_UNK,	540,	F (1315, Q_ISO),	F (3273, Q_ISO),	7.13,	289,	Q_UNK,	17.9,	Q_UNK,	color (0.64, 1.00, 0.78),	Q_NA,	},
{	N_("Samarium"),	Q_NA,	"Sm",	62,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	7.52,	7.16,	Q_NA,	N_("Silvery rare-earth metal"),	Event (1853, N_("Switzerland")),	"Jean Charles Galissard de Marignac",	N_("Named after the mineral samarskite."),	150.36,	19.9,	185,	Q_UNK,	242,	"96.4 (+3e)",	LatticeType::RHO,	166,	FL (3, 362.1, 362.1, 2625.0),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>6</sup> 6s<sup>2</sup>",	IL (2, 3, 2),	1.17,	Q_UNK,	544.5,	1345,	2067,	8.62,	165,	0.197,	13.3,	166,	color (0.56, 1.00, 0.78),	Q_NA,	},
{	N_("Europium"),	Q_NA,	"Eu",	63,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	5.244,	5.13,	Q_NA,	N_("Soft, silvery-white metal"),	Event (1901, N_("France")),	N_("Eug&#232;ne-Antole Demar&#231;ay"),	N_("Named after the European continent."),	151.964,	28.9,	185,	Q_UNK,	240,	"95 (+3e), 109 (+2e)",	LatticeType::BCC,	229,	FL (3, 458.1, 458.1, 458.1),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>7</sup> 6s<sup>2</sup>",	IL (2, 3, 2),	Q_UNK,	Q_UNK,	547.1,	1099,	1802,	9.21,	176,	0.182,	F (13.9, Q_EST),	Q_UNK,	color (0.38, 1.00, 0.78),	Q_NA,	},
{	N_("Gadolinium"),	Q_NA,	"Gd",	64,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	7.90,	7.4,	Q_NA,	N_("Soft, ductile, silvery-white metal"),	Event (1880, N_("Switzerland")),	"Jean Charles Galissard de Marignac",	N_("Named, via the mineral gadolinite, in honor of Finnish chemist Johan Gadolin."),	157.25,	19.9,	180,	Q_UNK,	238,	"93.8 (+3e)",	LatticeType::HEX,	194,	FL (3, 363.6, 363.6, 578.26),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>7</sup> 5d<sup>1</sup> 6s<sup>2</sup>",	IL (1, 3),	1.20,	Q_UNK,	593.4,	1585,	3546,	10.05,	301.3,	0.236,	10.6,	Q_UNK,	color (0.27, 1.00, 0.78),	Q_NA,	},
{	N_("Terbium"),	Q_NA,	"Tb",	65,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	8.23,	7.65,	Q_NA,	N_("Soft, ductile, silvery-gray, rare-earth metal"),	Event (1843, N_("Sweden")),	"Carl Gustaf Mosander",	N_("Named after Ytterby, Sweden, where it was discovered."),	158.92535,	19.2,	175,	Q_UNK,	237,	"84 (+4e), 92.3 (+3e)",	LatticeType::HEX,	194,	FL (3, 360.1, 360.1, 569.36),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>9</sup> 6s<sup>2</sup>",	IL (2, 4, 3),	Q_UNK,	Q_UNK,	565.8,	1629,	3503,	10.15,	293,	0.182,	11.1,	Q_UNK,	color (0.19, 1.00, 0.78),	Q_NA,	},
{	N_("Dysprosium"),	Q_NA,	"Dy",	66,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	8.540,	8.37,	Q_NA,	N_("Soft, lustrous, silvery metal"),	Event (1886, N_("France")),	N_("Paul &#201;mile Lecoq de Boisbaudran"),	N_("Greek: dys- (un-) + prositos (approachable)"),	162.500,	19.0,	175,	Q_UNK,	235,	"90.8 (+3e)",	LatticeType::HEX,	194,	FL (3, 359.3, 359.3, 565.37),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>10</sup> 6s<sup>2</sup>",	IL (1, 3),	1.22,	Q_UNK,	573.0,	1680,	2840,	11.06,	280,	0.170,	10.7,	Q_UNK,	color (0.12, 1.00, 0.78),	Q_NA,	},
{	N_("Holmium"),	Q_NA,	"Ho",	67,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	8.79,	8.34,	Q_NA,	N_("Fairly soft, malleable, lustrous, silvery metal"),	Event (1878, N_("Switzerland/Sweden")),	"Marc Delafontaine, Jacques-Louis Soret / Per Teodor Cleve",	N_("Named after Stockholm (Latin: Holmia), Sweden."),	164.93032,	18.7,	175,	Q_UNK,	233,	"89.4 (+3e)",	LatticeType::HEX,	194,	FL (3, 357.73, 357.73, 561.58),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>11</sup> 6s<sup>2</sup>",	IL (1, 3),	1.23,	Q_UNK,	581.0,	1734,	2993,	17.0,	265,	0.165,	16.2,	Q_UNK,	color (0.00, 1.00, 0.61),	Q_NA,	},
{	N_("Erbium"),	Q_NA,	"Er",	68,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	9.066,	8.86,	Q_NA,	N_("Soft, malleable, silvery metal"),	Event (1843, N_("Sweden")),	"Carl Gustaf Mosander",	N_("Named after Ytterby, Sweden, where it was discovered."),	167.259,	18.4,	175,	Q_UNK,	232,	"88.1 (+3e)",	LatticeType::HEX,	194,	FL (3, 355.88, 355.88, 558.74),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>12</sup> 6s<sup>2</sup>",	IL (1, 3),	1.24,	Q_UNK,	589.3,	1802,	3141,	19.90,	280,	0.168,	14.5,	Q_UNK,	color (0.00, 0.90, 0.46),	Q_NA,	},
{	N_("Thulium"),	Q_NA,	"Tm",	69,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	9.32,	8.56,	Q_NA,	N_("Soft, malleable, ductile, silvery metal"),	Event (1879, N_("Sweden")),	"Per Teodor Cleve",	N_("Named after Scandinavia (Latin: Thule, a mythical country of the north)."),	168.93421,	18.1,	175,	Q_UNK,	230,	"87 (+3e)",	LatticeType::HEX,	194,	FL (3, 353.75, 353.75, 555.46),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>13</sup> 6s<sup>2</sup>",	IL (2, 3, 2),	1.25,	99,	596.7,	1818,	2223,	16.84,	247,	0.160,	16.9,	Q_UNK,	color (0.00, 0.83, 0.32),	Q_NA,	},
{	N_("Ytterbium"),	Q_NA,	"Yb",	70,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	6.90,	6.21,	Q_NA,	N_("Silvery, lustrous, malleable, and ductile metal"),	Event (1878, N_("Switzerland")),	"Jean Charles Galissard de Marignac",	N_("Named after Ytterby, Sweden, where it was discovered."),	173.04,	24.8,	175,	Q_UNK,	228,	"85.8 (+3e), 93 (+2e)",	LatticeType::FCC,	225,	FL (3, 548.47, 548.47, 548.47),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 6s<sup>2</sup>",	IL (2, 3, 2),	Q_UNK,	Q_UNK,	603.4,	1097,	1469,	7.66,	159,	0.155,	38.5,	Q_UNK,	color (0.00, 0.75, 0.22),	Q_NA,	},
{	N_("Lutetium"),	Q_NA,	"Lu",	71,	Series::LANTHANIDE,	Q_NA,	6,	Block::F,	9.841,	9.3,	Q_NA,	N_("Silvery-white, hard, dense, rare-earth metal"),	Event (1907, N_("France")),	"Georges Urbain",	N_("Named after Paris (Latin: Lutetia), where it was discovered."),	174.967,	17.8,	175,	160,	227,	"85 (+3e)",	LatticeType::HEX,	194,	FL (3, 350.31, 350.31, 555.09),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>14</sup> 5d<sup>1</sup> 6s<sup>2</sup>",	IL (1, 3),	1.27,	33,	523.5,	1925,	3675,	F (22, Q_CA),	414,	0.154,	16.4,	Q_UNK,	color (0.00, 0.67, 0.14),	Q_NA,	},

{	N_("Hafnium"),	Q_NA,	"Hf",	72,	Series::TRANSITION_METAL,	4,	6,	Block::D,	13.31,	12,	Q_NA,	N_("Silvery, ductile metal"),	Event (1923, N_("Denmark")),	"Dirk Coster, Georg von Hevesy",	N_("Named after Copenhagen (Latin: Hafnia), Denmark, where it was discovered."),	178.49,	13.6,	155,	150,	225,	"78 (+4e)",	LatticeType::HEX,	194,	FL (3, 319.64, 319.64, 505.11),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>14</sup> 5d<sup>2</sup> 6s<sup>2</sup>",	IL (1, 4),	1.3,	Q_UNK,	658.5,	2506,	4876,	27.2,	571,	0.144,	23.0,	Q_UNK,	color (0.30, 0.76, 1.00),	Q_NA,	},
{	N_("Tantalum"),	Q_NA,	"Ta",	73,	Series::TRANSITION_METAL,	5,	6,	Block::D,	16.69,	15,	Q_NA,	N_("Gray, heavy, hard metal"),	Event (1802, N_("Sweden")),	"Anders Gustaf Ekeberg",	N_("Greek: Tantalus (a mythological king)"),	180.94788,	10.9,	145,	138,	220,	"68 (+5e)",	LatticeType::BCC,	229,	FL (3, 330.13, 330.13, 330.13),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>3</sup> 6s<sup>2</sup>",	IL (1, 5),	1.5,	31,	761,	3290,	5731,	36.57,	732.8,	0.140,	57.5,	225,	color (0.30, 0.65, 1.00),	Q_NA,	},
{	N_("Tungsten"),	"Wolfram",	"W",	74,	Series::TRANSITION_METAL,	6,	6,	Block::D,	19.25,	17.6,	Q_NA,	N_("Tough, steel-gray to white metal"),	Event (1783, N_("Spain")),	N_("Juan Jos&#233; Elhuyar, Fausto Elhuyar"),	N_("Swedish: tung (heavy) + sten (stone); German: wolfram"),	183.84,	9.53,	135,	146,	210,	"62 (+6e), 70 (+4e)",	LatticeType::BCC,	229,	FL (3, 316.52, 316.52, 316.52),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>4</sup> 6s<sup>2</sup>",	IL (6, 6, 5, 4, 3, 2, 0),	2.36,	79,	770,	3695,	5828,	52.31,	806.7,	0.132,	173,	310,	color (0.13, 0.58, 0.84),	Q_NA,	},
{	N_("Rhenium"),	Q_NA,	"Re",	75,	Series::TRANSITION_METAL,	7,	6,	Block::D,	21.02,	18.9,	Q_NA,	N_("Dense, silvery-white metal"),	Event (1925, N_("Germany")),	"Walter Noddack, Ida Tacke, Otto Berg",	N_("Named after the Rhine (Latin: Rhenus) river."),	186.207,	8.85,	135,	159,	205,	"53 (+7e), 72 (+4e)",	LatticeType::HEX,	194,	FL (3, 276.1, 276.1, 445.6),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>14</sup> 5d<sup>5</sup> 6s<sup>2</sup>",	IL (5, 5, 4, 3, 2, -1),	1.9,	F (0, Q_EST),	760,	3459,	5869,	60.43,	704,	0.137,	48.0,	416,	color (0.15, 0.49, 0.67),	Q_NA,	},
{	N_("Osmium"),	Q_NA,	"Os",	76,	Series::TRANSITION_METAL,	8,	6,	Block::D,	22.61,	20,	Q_NA,	N_("Blue-white, lustrous, hard metal"),	Event (1804, N_("England")),	"Smithson Tennant",	N_("Greek: osme (odor)"),	190.23,	8.43,	130,	128,	Q_UNK,	"69 (+6e), 88 (+4e)",	LatticeType::HEX,	194,	FL (3, 273.44, 273.44, 431.73),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>14</sup> 5d<sup>6</sup> 6s<sup>2</sup>",	IL (7, 8, 6, 4, 3, 2, 0, -2),	2.2,	104.0,	840,	3306,	5285,	57.85,	738,	0.130,	87.6,	Q_UNK,	color (0.15, 0.40, 0.59),	Q_NA,	},
{	N_("Iridium"),	Q_NA,	"Ir",	77,	Series::TRANSITION_METAL,	9,	6,	Block::D,	22.65,	19,	Q_NA,	N_("White, brittle metal"),	Event (1803, N_("England")),	"Smithson Tennant",	N_("Latin: iris (rainbow)"),	192.217,	8.54,	135,	137,	Q_UNK,	"68 (+4e)",	LatticeType::FCC,	225,	FL (3, 383.9, 383.9, 383.9),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>7</sup> 6s<sup>2</sup>",	IL (7, 6, 4, 3, 2, 1, 0, -1),	2.20,	150.9,	880,	2719,	4701,	41.12,	231.8,	0.131,	147,	430,	color (0.09, 0.33, 0.53),	Q_NA,	},
{	N_("Platinum"),	Q_NA,	"Pt",	78,	Series::TRANSITION_METAL,	10,	6,	Block::D,	21.45,	19.77,	Q_NA,	N_("Very heavy, soft, silvery-white metal"),	Event (1741, N_("Peru")),	"Antonio de Ulloa",	N_("Spanish: platina (little silver)"),	195.084,	9.10,	135,	128,	205,	"65 (+4e), 80 (+2e)",	LatticeType::FCC,	225,	FL (3, 392.42, 392.42, 392.42),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>9</sup> 6s<sup>1</sup>",	IL (3, 4, 2, 0),	2.28,	205.04,	870,	2041.4,	4098,	22.17,	469,	0.133,	71.6,	230,	color (0.96, 0.93, 0.82),	Q_NA,	},
{	N_("Gold"),	"Aurum",	"Au",	79,	Series::TRANSITION_METAL,	11,	6,	Block::D,	19.3,	17.31,	Q_NA,	N_("Soft, malleable, yellow metal"),	Q_NA,	Q_UNK,	N_("Anglo-Saxon: geolo (yellow); Latin: aurum"),	196.966569,	10.2,	135,	144,	210,	"85 (+3e), 137 (+1e)",	LatticeType::FCC,	225,	FL (3, 407.82, 407.82, 407.82),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>1</sup>",	IL (2, 3, 1),	2.54,	222.75,	890.1,	1337.33,	3129,	12.55,	324,	0.129,	318,	170,	color (0.80, 0.82, 0.12),	Q_NA,	},
{	N_("Mercury"),	"Hydrargum",	"Hg",	80,	Series::TRANSITION_METAL,	12,	6,	Block::D,	Q_NA,	13.534,	Q_NA,	N_("Heavy, silver-white metal that is in its liquid state at room temperature"),	Q_NA,	Q_UNK,	N_("Latin: Mercurius (a deity associated with speed); Greek/Latin: hydrargum (liquid silver)"),	200.59,	14.8,	150,	149,	205,	"110 (+2e), 127 (+1e)",	LatticeType::RHO,	166,	FL (3, 300.5, 300.5, 300.5),	FL (3, 70.520, 70.520, 70.520),	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>2</sup>",	IL (2, 2, 1),	2.00,	F (0, Q_EST),	1007.1,	234.32,	629.88,	2.29,	59.11,	0.140,	8.30,	100,	color (0.71, 0.71, 0.76),	N_("Liquid density value is for 20&#176;C."),	},
{	N_("Thallium"),	Q_NA,	"Tl",	81,	Series::POST_TRANSITION_METAL,	13,	6,	Block::P,	11.85,	11.22,	Q_NA,	N_("Soft, gray metal"),	Event (1861, N_("England")),	"William Crookes",	N_("Greek: thallos (green twig, for a bright green line in its spectrum)"),	204.3833,	17.2,	190,	148,	220,	"95 (+3e), 147 (+1e)",	LatticeType::HEX,	194,	FL (3, 345.66, 345.66, 552.48),	FL (3, 90.0, 90.0, 120.0),	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>2</sup> 6p<sup>1</sup>",	IL (2, 3, 1),	1.62,	36,	589.4,	577,	1746,	4.14,	165,	0.129,	46.1,	96,	color (0.65, 0.33, 0.30),	Q_NA,	},
{	N_("Lead"),	"Plumbum",	"Pb",	82,	Series::POST_TRANSITION_METAL,	14,	6,	Block::P,	11.34,	10.66,	Q_NA,	N_("Very soft, highly malleable and ductile, blue-white shiny metal"),	Q_NA,	Q_UNK,	N_("Anglo-Saxon: lead; Latin: plumbum"),	207.2,	18.3,	180,	147,	230,	"84 (+4e), 120 (+2e)",	LatticeType::FCC,	225,	FL (3, 495.08, 495.08, 495.08),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>2</sup> 6p<sup>2</sup>",	IL (2, 4, 2),	2.33,	35,	715.6,	600.61,	2022,	4.77,	179.5,	0.129,	35.3,	88,	color (0.34, 0.35, 0.38),	Q_NA,	},
{	N_("Bismuth"),	Q_NA,	"Bi",	83,	Series::POST_TRANSITION_METAL,	15,	6,	Block::P,	9.78,	10.05,	Q_NA,	N_("Hard, brittle, steel-gray metal with a pinkish tinge"),	Event (1753, N_("France")),	N_("Claude Fran&#231;ois Geoffroy"),	N_("German: wismuth"),	208.98040,	21.3,	160,	146,	230,	"74 (+5e), 96 (+3e)",	LatticeType::MONO,	12,	FL (3, 667.4, 611.7, 330.4),	FL (3, 90.0, 110.330, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>2</sup> 6p<sup>3</sup>",	IL (2, 5, 3),	2.02,	90.92,	703,	544.7,	1837,	11.30,	151,	0.122,	7.97,	120,	color (0.62, 0.31, 0.71),	Q_NA,	},
{	N_("Polonium"),	Q_NA,	"Po",	84,	Series::SEMIMETAL,	16,	6,	Block::P,	9.196,	Q_UNK,	Q_NA,	N_("Silvery-gray metal"),	Event (1898, N_("France")),	N_("Marie Curie-Sk&#322;odowska, Pierre Curie"),	N_("Named after Poland, the native country of Marie Curie-Sk&#322;odowska."),	F (209, Q_ISO),	22.7,	190,	Q_UNK,	Q_UNK,	"67 (+6e)",	LatticeType::BCC,	221,	FL (3, 335.9, 335.9, 335.9),	FL (3, 90.0, 90.0, 90.0),	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>2</sup> 6p<sup>4</sup>",	IL (3, 6, 4, 2),	2.0,	Q_UNK,	812.1,	527,	1235,	F (13, Q_CA),	102.91,	F (0.13, Q_ISO),	Q_UNK,	Q_UNK,	color (0.67, 0.36, 0.00),	N_("Where variable, values given are for alpha form."),	},
{	N_("Astatine"),	Q_NA,	"At",	85,	Series::HALOGEN,	17,	6,	Block::P,	Q_UNK,	Q_UNK,	Q_NA,	N_("Unstable, radioactive halogen"),	Event (1940, N_("United States")),	N_("Dale R. Corson, K.R. MacKenzie, Emilio Segr&#232;"),	N_("Greek: astatos (unstable)"),	F (210, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"62 (+7e)",	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>2</sup> 6p<sup>5</sup>",	IL (5, 7, 5, 3, 1, -1),	2.2,	Q_UNK,	F (920, Q_EST),	575,	F (610, Q_EST),	Q_UNK,	F (40, Q_CA),	Q_UNK,	1.7,	Q_UNK,	color (0.46, 0.31, 0.27),	Q_NA,	},
{	N_("Radon"),	Q_NA,	"Rn",	86,	Series::NOBLE_GAS,	18,	6,	Block::P,	Q_NA,	Q_UNK,	9.73,	N_("Heavy radioactive gas"),	Event (1898, N_("Germany")),	"Friedrich Ernst Dorn",	N_("Named after the element radium, from which it is produced by radioactive decay."),	F (222, Q_ISO),	Q_UNK,	F (120, Q_EST),	145,	Q_UNK,	Q_NA,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Xe] 4f<sup>14</sup> 5d<sup>10</sup> 6s<sup>2</sup> 6p<sup>6</sup>",	Q_NA,	Q_UNK,	F (0, Q_EST),	1037,	202,	211.3,	3.247,	18.10,	0.094,	0.00361,	Q_UNK,	color (0.26, 0.51, 0.59),	Q_NA,	},

{	N_("Francium"),	Q_NA,	"Fr",	87,	Series::ALKALI_METAL,	1,	7,	Block::S,	Q_UNK,	Q_UNK,	Q_NA,	Q_UNK,	Event (1939, N_("France")),	"Marguerite Perey",	N_("Named after France, where it was discovered."),	F (223, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"180 (+1e)",	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Rn] 7s<sup>1</sup>",	IL (1, 2),	0.7,	Q_UNK,	380,	300,	F (950, Q_EST),	F (2, Q_CA),	F (65, Q_CA),	F (0.143, Q_ISO),	Q_UNK,	Q_UNK,	color (0.26, 0.00, 0.40),	Q_NA,	},
{	N_("Radium"),	Q_NA,	"Ra",	88,	Series::ALKALINE_EARTH_METAL,	2,	7,	Block::S,	5.5,	Q_UNK,	Q_NA,	N_("Silvery white, radioactive element"),	Event (1898, N_("France")),	N_("Marie Curie-Sk&#322;odowska, Pierre Curie"),	N_("Latin: radius (ray)"),	F (226, Q_ISO),	45.0,	215,	Q_UNK,	Q_UNK,	"143 (+2e)",	LatticeType::BCC,	229,	FL (3, 514.8, 514.8, 514.8),	FL (3, 90.0, 90.0, 90.0),	"[Rn] 7s<sup>2</sup>",	IL (1, 2),	0.9,	Q_UNK,	509.3,	973,	2010,	8.5,	113,	Q_UNK,	18.6,	Q_UNK,	color (0.00, 0.49, 0.00),	Q_NA,	},

{	N_("Actinium"),	Q_NA,	"Ac",	89,	Series::ACTINIDE,	3,	7,	Block::F,	F (10, Q_ISO),	Q_UNK,	Q_NA,	N_("Heavy, silvery-white metal that is very radioactive"),	Event (1899, N_("France")),	N_("Andr&#233;-Louis Debierne"),	N_("Greek: aktis, aktinos (ray)"),	F (227, Q_ISO),	22.54,	195,	Q_UNK,	Q_UNK,	"118 (+3e)",	LatticeType::FCC,	225,	FL (3, 567.0, 567.0, 567.0),	FL (3, 90.0, 90.0, 90.0),	"[Rn] 6d<sup>1</sup> 7s<sup>2</sup>",	IL (1, 3),	1.1,	Q_UNK,	499,	F (1323, Q_CA),	F (3471, Q_ISO),	14,	400,	0.120,	12,	Q_UNK,	color (0.44, 0.67, 0.98),	Q_NA,	},
{	N_("Thorium"),	Q_NA,	"Th",	90,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	11.7,	Q_UNK,	Q_NA,	N_("Gray, soft, malleable, ductile, radioactive metal"),	Event (1828, N_("Sweden")),	N_("J&#246;ns Jakob Berzelius"),	N_("Old Norse: Thor (thunder deity)"),	232.03806,	19.8,	180,	Q_UNK,	240,	"102 (+4e)",	LatticeType::FCC,	225,	FL (3, 508.42, 508.42, 508.42),	FL (3, 90.0, 90.0, 90.0),	"[Rn] 6d<sup>2</sup> 7s<sup>2</sup>",	IL (1, 4),	1.3,	Q_UNK,	587,	2115,	5061,	13.81,	514,	0.113,	54.0,	100,	color (0.00, 0.73, 1.00),	Q_NA,	},
{	N_("Protactinium"),	Q_NA,	"Pa",	91,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	15.37,	Q_UNK,	Q_NA,	N_("Silvery-white, radioactive metal"),	Event (1918, N_("England/Germany")),	"Frederick Soddy, John Cranston / Otto Hahn, Lise Meitner",	N_("Greek: protos (before) + actinium (which it forms by decay)"),	231.03588,	15.0,	180,	Q_UNK,	Q_UNK,	"89 (+5e), 113 (+3e)",	LatticeType::TET,	139,	FL (3, 392.5, 392.5, 323.8),	FL (3, 90.0, 90.0, 90.0),	"[Rn] 5f<sup>2</sup> 6d<sup>1</sup> 7s<sup>2</sup>",	IL (2, 5, 4),	1.5,	Q_UNK,	568,	1841,	F (4300, Q_EST),	12.34,	481,	Q_UNK,	47,	Q_UNK,	color (0.00, 0.63, 1.00),	Q_NA,	},
{	N_("Uranium"),	Q_NA,	"U",	92,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	19.1,	17.3,	Q_NA,	N_("Silvery-white, dense, ductile and malleable, radioactive metal"),	Event (1789, N_("Germany")),	"Martin Heinrich Klaproth",	N_("Named after the planet Uranus."),	238.02891,	12.5,	175,	Q_UNK,	230,	"80 (+6e), 97 (+4e)",	LatticeType::ORTH,	63,	FL (3, 285.37, 586.95, 495.48),	FL (3, 90.0, 90.0, 90.0),	"[Rn] 5f<sup>3</sup> 6d<sup>1</sup> 7s<sup>2</sup>",	IL (4, 6, 5, 4, 3),	1.38,	Q_UNK,	597.6,	1405.3,	4404,	9.14,	417.1,	0.116,	27.5,	Q_UNK,	color (0.00, 0.56, 1.00),	Q_NA,	},
{	N_("Neptunium"),	Q_NA,	"Np",	93,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	20.2,	Q_UNK,	Q_NA,	N_("Silvery metal"),	Event (1940, N_("United States")),	"Edwin McMillan, Philip H. Abelson",	N_("Named after the planet Neptune."),	F (237, Q_ISO),	21.1,	175,	Q_UNK,	Q_UNK,	"95 (+4e), 110 (+3e)",	LatticeType::ORTH,	62,	FL (3, 666.3, 472.3, 488.7),	FL (3, 90.0, 90.0, 90.0),	"[Rn] 5f<sup>4</sup> 6d<sup>1</sup> 7s<sup>2</sup>",	IL (4, 6, 5, 4, 3),	1.36,	Q_UNK,	604.5,	917,	4273,	3.20,	336,	F (0.124, Q_ISO),	6.3,	Q_UNK,	color (0.00, 0.50, 1.00),	Q_NA,	},
{	N_("Plutonium"),	Q_NA,	"Pu",	94,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	19.816,	16.63,	Q_NA,	N_("Silvery-white, radioactive metal"),	Event (1940, N_("United States")),	"Glenn T. Seaborg, Edwin McMillan, Arthur C. Wahl, Joseph W. Kennedy",	N_("Named after the planet Pluto."),	F (244, Q_ISO),	Q_UNK,	175,	Q_UNK,	Q_UNK,	"93 (+4e), 108 (+3e)",	LatticeType::MONO,	11,	FL (3, 618.3, 482.2, 1096.3),	FL (3, 90.0, 101.790, 90.0),	"[Rn] 5f<sup>6</sup> 7s<sup>2</sup>",	IL (4, 6, 5, 4, 3),	1.28,	Q_UNK,	584.7,	912.5,	3501,	2.82,	333.5,	F (0.145, Q_ISO),	6.74,	Q_UNK,	color (0.00, 0.42, 1.00),	Q_NA,	},
{	N_("Americium"),	Q_NA,	"Am",	95,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	12,	Q_UNK,	Q_NA,	N_("Silvery-white, radioactive metal"),	Event (1944, N_("United States")),	"Glenn T. Seaborg, Leon O. Morgan, Ralph A. James, Albert Ghiorso",	N_("Named after the American continents, by analogy with europium."),	F (243, Q_ISO),	20.8,	175,	Q_UNK,	Q_UNK,	"92 (+4e), 107 (+3e)",	LatticeType::HEX,	194,	FL (3, 346.81, 346.81, 1124.1),	FL (3, 90.0, 90.0, 120.0),	"[Rn] 5f<sup>7</sup> 7s<sup>2</sup>",	IL (4, 6, 5, 4, 3),	1.3,	Q_UNK,	578,	1449,	2880,	14.39,	Q_UNK,	F (0.258, Q_ISO),	10,	Q_UNK,	color (0.33, 0.36, 0.95),	Q_NA,	},
{	N_("Curium"),	Q_NA,	"Cm",	96,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	F (13.51, Q_ISO),	Q_UNK,	Q_NA,	N_("Silvery, malleable, synthetic radioactive metal"),	Event (1944, N_("United States")),	"Glenn T. Seaborg, Ralph A. James, Albert Ghiorso",	N_("Named in honor of Pierre Curie and Marie Curie-Sk&#322;odowska."),	F (247, Q_ISO),	18.28,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	LatticeType::HEX,	194,	FL (3, 349.6, 349.6, 1133.1),	FL (3, 90.0, 90.0, 120.0),	"[Rn] 5f<sup>7</sup> 6d<sup>1</sup> 7s<sup>2</sup>",	IL (2, 4, 3),	1.3,	Q_UNK,	581,	F (1613, Q_ISO),	F (3383, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.47, 0.36, 0.89),	Q_NA,	},
{	N_("Berkelium"),	Q_NA,	"Bk",	97,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	13.25,	Q_UNK,	Q_NA,	N_("Radioactive, synthetic metal"),	Event (1949, N_("United States")),	"Albert Ghiorso, Glenn T. Seaborg, Stanley G. Tompson, Kenneth Street Jr.",	N_("Named after the University of California, Berkeley."),	F (247, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	LatticeType::HEX,	194,	FL (3, 341.6, 341.6, 1106.9),	FL (3, 90.0, 90.0, 120.0),	"[Rn] 5f<sup>9</sup> 7s<sup>2</sup>",	IL (2, 4, 3),	1.3,	Q_UNK,	601,	1259,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	10,	Q_UNK,	color (0.54, 0.31, 0.89),	N_("Where variable, values given are for beta form."),	},
{	N_("Californium"),	Q_NA,	"Cf",	98,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	15.1,	Q_UNK,	Q_NA,	N_("Powerful neutron emitter"),	Event (1950, N_("United States")),	"Albert Ghiorso, Glenn T. Seaborg, Stanley G. Tompson, Kenneth Street Jr.",	N_("Named after the state and the University of California."),	F (251, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	LatticeType::HEX,	194,	FL (3, 338.0, 338.0, 1102.5),	FL (3, 90.0, 90.0, 120.0),	"[Rn] 5f<sup>10</sup> 7s<sup>2</sup>",	IL (2, 4, 3),	1.3,	Q_UNK,	608,	F (1173, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.63, 0.21, 0.83),	Q_NA,	},
{	N_("Einsteinium"),	Q_NA,	"Es",	99,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	8.84,	Q_UNK,	Q_NA,	N_("Radioactive, synthetic metal"),	Event (1952, N_("United States")),	N_("Argonne Lab., Los Alamos Lab., Univ. of California"),	N_("Named in honor of German-American physicist Albert Einstein."),	F (252, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Rn] 5f<sup>11</sup> 7s<sup>2</sup>",	IL (1, 3),	1.3,	Q_UNK,	619,	1133,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.70, 0.12, 0.83),	Q_NA,	},
{	N_("Fermium"),	Q_NA,	"Fm",	100,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	Q_UNK,	Q_UNK,	Q_NA,	N_("Radioactive, synthetic metal"),	Event (1953, N_("United States")),	N_("Argonne Lab., Los Alamos Lab., Univ. of California"),	N_("Named in honor of Italian physicist Enrico Fermi."),	F (257, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Rn] 5f<sup>12</sup> 7s<sup>2</sup>",	IL (1, 3),	1.3,	Q_UNK,	627,	F (1800, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.70, 0.12, 0.73),	Q_NA,	},
{	N_("Mendelevium"),	Q_NA,	"Md",	101,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	Q_UNK,	Q_UNK,	Q_NA,	N_("Radioactive, synthetic metal"),	Event (1955, N_("United States")),	"Albert Ghiorso, Glenn T. Seaborg, Bernard Harvey, Gregory Choppin, Stanley G. Thompson",	N_("Named in honor of Russian chemist Dmitri Mendeleev, who devised the periodic table."),	F (258, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Rn] 5f<sup>13</sup> 7s<sup>2</sup>",	IL (1, 3),	1.3,	Q_UNK,	635,	1100,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.70, 0.05, 0.65),	Q_NA,	},
{	N_("Nobelium"),	Q_NA,	"No",	102,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	Q_UNK,	Q_UNK,	Q_NA,	N_("Radioactive, synthetic metal"),	Event (1957, N_("Sweden")),	N_("Nobel Institute for Physics"),	N_("Named in honor of Swedish chemist Alfred Nobel, inventor of dynamite and founder of the Nobel Prizes."),	F (259, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	"[Rn] 5f<sup>14</sup> 7s<sup>2</sup>",	IL (2, 3, 2),	1.3,	Q_UNK,	642,	1100,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.74, 0.05, 0.53),	Q_NA,	},
{	N_("Lawrencium"),	Q_NA,	"Lr",	103,	Series::ACTINIDE,	Q_NA,	7,	Block::F,	Q_UNK,	Q_UNK,	Q_NA,	N_("Radioactive, synthetic metal"),	Event (1961, N_("United States")),	N_("Albert Ghiorso, Torbj&#248;rn Sikkeland, Almon Larsh, Robert M. Latimer"),	N_("Named in honor of American physicist Ernest O. Lawrence, inventor of the cyclotron."),	F (262, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	String ("[Rn] 5f<sup>14</sup> 7s<sup>2</sup> 7p<sup>1</sup>", Q_EST),	IL (1, 3),	Q_UNK,	Q_UNK,	470,	1900,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.78, 0.00, 0.40),	Q_NA,	},

{	N_("Rutherfordium"),	Q_NA,	"Rf",	104,	Series::TRANSITION_METAL,	4,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	N_("Radioactive, synthetic metal"),	Event (1964, N_("USSR")),	N_("Joint Inst. for Nuclear Research"),	N_("Named in honor of New Zealand physicist Ernest Rutherford, who pioneered nuclear physics."),	F (261, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	String ("[Rn] 5f<sup>14</sup> 6d<sup>2</sup> 7s<sup>2</sup>", Q_EST),	Q_UNK,	Q_UNK,	Q_UNK,	580,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.80, 0.00, 0.35),	Q_NA,	},
{	N_("Dubnium"),	Q_NA,	"Db",	105,	Series::TRANSITION_METAL,	5,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	N_("Radioactive, synthetic metal"),	Event (1967, N_("USSR/United States")),	N_("Joint Inst. for Nuclear Research / Albert Ghiorso et al."),	N_("Named after Dubna, Russia, where it was discovered."),	F (262, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.82, 0.00, 0.31),	Q_NA,	},
{	N_("Seaborgium"),	Q_NA,	"Sg",	106,	Series::TRANSITION_METAL,	6,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	N_("Radioactive, synthetic metal"),	Event (1974, N_("USSR/United States")),	N_("Joint Inst. for Nuclear Research / Univ. of California, Berkeley"),	N_("Named in honor of American physical chemist Glenn Seaborg, known for research on transuranium elements."),	F (266, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.85, 0.00, 0.27),	Q_NA,	},
{	N_("Bohrium"),	Q_NA,	"Bh",	107,	Series::TRANSITION_METAL,	7,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	N_("Radioactive, synthetic metal"),	Event (1976, N_("USSR/Germany")),	N_("Joint Inst. for Nuclear Research / Inst. for Heavy Ion Research"),	N_("Named in honor of Danish physicist Niels Bohr, a pioneer in quantum mechanics."),	F (264, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.88, 0.00, 0.22),	Q_NA,	},
{	N_("Hassium"),	Q_NA,	"Hs",	108,	Series::TRANSITION_METAL,	8,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (1984, N_("Germany")),	N_("Peter Armbruster, Gottfried M&#252;nzenberg, et al. (Inst. for Heavy Ion Research)"),	N_("Named after Hessen (Latin: Hassia), Germany, where it was discovered."),	F (277, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.90, 0.00, 0.18),	Q_NA,	},
{	N_("Meitnerium"),	Q_NA,	"Mt",	109,	Series::TRANSITION_METAL,	9,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (1982, N_("Germany")),	N_("Peter Armbruster, Gottfried M&#252;nzenberg, et al. (Inst. for Heavy Ion Research)"),	N_("Named in honor of Austrian-Swedish physicist Lise Meitner."),	F (268, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.92, 0.00, 0.15),	Q_NA,	},
{	N_("Darmstadtium"),	Q_NA,	"Ds",	110,	Series::TRANSITION_METAL,	10,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (1994, N_("Germany")),	N_("S. Hofmann, V. Ninov, et al. (Inst. for Heavy Ion Research)"),	N_("Named after Darmstadt, Germany, where it was discovered."),	F (281, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.93, 0.00, 0.14),	Q_NA,	},
{	N_("Roentgenium"),	Q_NA,	"Rg",	111,	Series::TRANSITION_METAL,	11,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (1994, N_("Germany")),	N_("S. Hofmann, V. Ninov, et al. (Inst. for Heavy Ion Research)"),	N_("Named in honor of German physicist Wilhelm Roentgen, who discovered x-rays."),	F (272, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.94, 0.00, 0.13),	Q_NA,	},
{	N_("Copernicium"),	Q_NA,	"Cn",	112,	Series::TRANSITION_METAL,	12,	7,	Block::D,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (1996, N_("Germany")),	N_("Inst. for Heavy Ion Research"),	N_("Named in honor of the astronomer Nicolaus Copernicus."),	F (285, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.95, 0.00, 0.12),	Q_NA,	},
{	N_("Nihonium"),		Q_NA,	"Nh",	113,	Series::POST_TRANSITION_METAL,	13,	7,	Block::P,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (2004, N_("Japan")),	N_("RIKEN Nishina Center for Accelerator-Based Science, Japan"),	N_("Japanese: Nihon (the Land of Rising Sun)"),	F (284, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.96, 0.00, 0.11),	Q_NA,	},
{	N_("Flerovium"),	Q_NA,	"Fl",	114,	Series::POST_TRANSITION_METAL,	14,	7,	Block::P,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (1999, N_("Russia")),	N_("Joint Inst. for Nuclear Research"),	N_("Named after the Flerov Laboratory of Nuclear Reactions of the Joint Inst. for Nuclear Research in Dubna, Russia, where it was discovered."),	F (289, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.97, 0.00, 0.10),	Q_NA,	},
{	N_("Moscovium"),	Q_NA,	"Mc",	115,	Series::POST_TRANSITION_METAL,	15,	7,	Block::P,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (2003, N_("Russia/United States")),	N_("Joint Inst. for Nuclear Research / Lawrence Livermore Lab."),	N_("Named after Moscow, Russia"),	F (288, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.98, 0.00, 0.09),	Q_NA,	},
{	N_("Livermorium"),	Q_NA,	"Lv",	116,	Series::POST_TRANSITION_METAL,	16,	7,	Block::P,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (2001, N_("Russia")),	N_("Joint Inst. for Nuclear Research"),	N_("The name recognizes the Lawrence Livermore National Laboratory, California, which collaborated with the Joint Inst. for Nuclear Research (JINR), Russia on the discovery."),	F (292, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.99, 0.00, 0.08),	Q_NA,	},
{	N_("Tennessine"),	Q_NA,	"Ts",	117,	Series::POST_TRANSITION_METAL,	17,	7,	Block::P,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (2009, N_("Russia/United States")),	N_("Joint Inst. for Nuclear Research / Lawrence Livermore National Lab. / Vanderbilt University / Oak Ridge National Lab."),	N_("Named after Tennessee, United States"),	F (294, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (1.00, 0.00, 0.07),	Q_NA,	},
{	N_("Oganesson"),	Q_NA,	"Og",	118,	Series::NOBLE_GAS,	18,	7,	Block::P,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Event (2002, N_("Russia/United States")),	N_("Joint Inst. for Nuclear Research / Lawrence Livermore National Lab."),	N_("Named after nuclear physicist Yuri Oganessian"),	F (294, Q_ISO),	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	Q_UNK,	color (0.16, 0.40, 0.49),	Q_NA,	},

}; // ElementData table_data[]

} // namespace Elemental
