#!/usr/bin/env python3

import importlib.util
import os

from fenrirscreenreader.core.i18n import _

# Load base configuration class
_base_path = os.path.join(os.path.dirname(__file__), "..", "config_base.py")
_spec = importlib.util.spec_from_file_location("config_base", _base_path)
_module = importlib.util.module_from_spec(_spec)
_spec.loader.exec_module(_module)
config_command = _module.config_command


class command(config_command):
    def __init__(self):
        super().__init__()

    def get_description(self):
        return "Set screen text encoding"

    def run(self):
        current_encoding = self.get_setting("screen", "encoding", "auto")

        # Present current encoding
        self.present_text(f"Current screen encoding: {current_encoding}")

        # Cycle through available encodings
        encodings = ["auto", "utf-8", "cp1252", "iso-8859-1"]
        try:
            current_index = encodings.index(current_encoding)
            next_index = (current_index + 1) % len(encodings)
            new_encoding = encodings[next_index]
        except ValueError:
            new_encoding = "auto"  # Default to auto

        success = self.set_setting("screen", "encoding", new_encoding)

        if success:
            self.present_text(f"Screen encoding set to: {new_encoding}")
            if new_encoding == "auto":
                self.present_text(
                    "Fenrir will automatically detect text encoding"
                )
            else:
                self.present_text(f"Fenrir will use {new_encoding} encoding")
            self.play_sound("Accept")
        else:
            self.present_text("Failed to change screen encoding")
            self.play_sound("Error")
