/*
  This file is part of TALER
  Copyright (C) 2020 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_delete_reserve.c
 * @brief command to test DELETE /reserves/$RESERVE_PUB
 * @author Jonathan Buchanan
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State of a "DELETE /reserves/$RESERVE_PUB" CMD.
 */
struct DeleteReserveState
{

  /**
   * Handle for a "DELETE reserve" request.
   */
  struct TALER_MERCHANT_ReserveDeleteHandle *rdh;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Base URL of the merchant serving the request.
   */
  const char *merchant_url;

  /**
   * Reference to a command that provides a reserve.
   */
  const char *reserve_reference;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

  /**
   * Use purge, not delete.
   */
  bool purge;

};


/**
 * Callback for a DELETE /reserves/$RESERVE_PUB operation.
 *
 * @param cls closure for this function
 * @param hr response being processed
 */
static void
delete_reserve_cb (void *cls,
                   const struct TALER_MERCHANT_HttpResponse *hr)
{
  struct DeleteReserveState *drs = cls;

  drs->rdh = NULL;
  if (drs->http_status != hr->http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                hr->http_status,
                (int) hr->ec,
                TALER_TESTING_interpreter_get_current_label (drs->is));
    TALER_TESTING_interpreter_fail (drs->is);
    return;
  }
  switch (hr->http_status)
  {
  case MHD_HTTP_NO_CONTENT:
    break;
  case MHD_HTTP_NOT_FOUND:
    break;
  case MHD_HTTP_CONFLICT:
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unhandled HTTP status.\n");
  }
  TALER_TESTING_interpreter_next (drs->is);
}


/**
 * Run the "DELETE reserve" CMD.
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
delete_reserve_run (void *cls,
                    const struct TALER_TESTING_Command *cmd,
                    struct TALER_TESTING_Interpreter *is)
{
  struct DeleteReserveState *drs = cls;
  const struct TALER_TESTING_Command *reserve_cmd;
  const struct TALER_ReservePublicKeyP *reserve_pub;

  reserve_cmd = TALER_TESTING_interpreter_lookup_command (
    is,
    drs->reserve_reference);
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_reserve_pub (reserve_cmd,
                                           0,
                                           &reserve_pub))
    TALER_TESTING_FAIL (is);

  drs->is = is;
  if (drs->purge)
    drs->rdh = TALER_MERCHANT_reserve_purge (is->ctx,
                                             drs->merchant_url,
                                             reserve_pub,
                                             &delete_reserve_cb,
                                             drs);
  else
    drs->rdh = TALER_MERCHANT_reserve_delete (is->ctx,
                                              drs->merchant_url,
                                              reserve_pub,
                                              &delete_reserve_cb,
                                              drs);

  GNUNET_assert (NULL != drs->rdh);
}


/**
 * Free the state of a "DELETE reserve" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
delete_reserve_cleanup (void *cls,
                        const struct TALER_TESTING_Command *cmd)
{
  struct DeleteReserveState *drs = cls;

  if (NULL != drs->rdh)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "DELETE /reserves/$RESERVE_PUB operation did not complete\n");
    TALER_MERCHANT_reserve_delete_cancel (drs->rdh);
  }
  GNUNET_free (drs);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_delete_reserve (const char *label,
                                           const char *merchant_url,
                                           const char *reserve_reference,
                                           unsigned int http_status)
{
  struct DeleteReserveState *drs;

  drs = GNUNET_new (struct DeleteReserveState);
  drs->merchant_url = merchant_url;
  drs->reserve_reference = reserve_reference;
  drs->http_status = http_status;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = drs,
      .label = label,
      .run = &delete_reserve_run,
      .cleanup = &delete_reserve_cleanup
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_purge_reserve (const char *label,
                                          const char *merchant_url,
                                          const char *reserve_reference,
                                          unsigned int http_status)
{
  struct DeleteReserveState *drs;

  drs = GNUNET_new (struct DeleteReserveState);
  drs->merchant_url = merchant_url;
  drs->reserve_reference = reserve_reference;
  drs->http_status = http_status;
  drs->purge = true;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = drs,
      .label = label,
      .run = &delete_reserve_run,
      .cleanup = &delete_reserve_cleanup
    };

    return cmd;
  }
}


/* end of testing_api_cmd_delete_reserve.c */
