package Venus::Role::Stashable;

use 5.018;

use strict;
use warnings;

use Venus::Role 'with';

# BUILDERS

sub BUILD {
  my ($self, $data) = @_;

  $self->{'$stash'} = delete $data->{'$stash'} || {} if !$self->{'$stash'};

  return $self;
};

# METHODS

sub stash {
  my ($self, $key, $value) = @_;

  return $self->{'$stash'} if !exists $_[1];

  return $self->{'$stash'}->{$key} if !exists $_[2];

  $self->{'$stash'}->{$key} = $value;

  return $value;
}

# EXPORTS

sub EXPORT {
  ['stash']
}

1;



=head1 NAME

Venus::Role::Stashable - Stashable Role

=cut

=head1 ABSTRACT

Stashable Role for Perl 5

=cut

=head1 SYNOPSIS

  package Example;

  use Venus::Class;

  with 'Venus::Role::Stashable';

  attr 'test';

  package main;

  my $example = Example->new(test => time);

  # $example->stash;

=cut

=head1 DESCRIPTION

This package modifies the consuming package and provides methods for stashing
data within the object.

=cut

=head1 METHODS

This package provides the following methods:

=cut

=head2 stash

  stash(Any $key, Any $value) (Any)

The stash method is used to fetch and stash named values associated with the
object. Calling this method without arguments returns all values.

I<Since C<0.01>>

=over 4

=item stash example 1

  package main;

  my $example = Example->new(test => time);

  my $stash = $example->stash;

  # {}

=back

=over 4

=item stash example 2

  package main;

  my $example = Example->new(test => time);

  my $stash = $example->stash('test', {1..4});

  # { 1 => 2, 3 => 4 }

=back

=over 4

=item stash example 3

  package main;

  my $example = Example->new(test => time);

  my $stash = $example->stash('test');

  # undef

=back

=cut

=head1 AUTHORS

Awncorp, C<awncorp@cpan.org>

=cut

=head1 LICENSE

Copyright (C) 2000, Al Newkirk.

This program is free software, you can redistribute it and/or modify it under
the terms of the Apache license version 2.0.

=cut