package Lingua::TR::ASCII::Data;
$Lingua::TR::ASCII::Data::VERSION = '0.16';
use strict;
use warnings;
use base qw(Exporter);
use utf8;
use constant CONTEXT_SIZE => 10;

our @EXPORT = qw(
    $PATTERN
    $ASCIIFY
    $TOGGLE_ACCENT
    $DOWNCASE_ASCIIFY
    $UPCASE_ACCENTS
    CONTEXT_SIZE
);

our $ASCIIFY = {qw(
    ç    c
    Ç    C
    ğ    g
    Ğ    G
    ö    o
    Ö    O
    ü    u
    Ü    U
    ı    i
    İ    I
    ş    s
    Ş    S
)};

our $DOWNCASE_ASCIIFY = {qw(
    ç    c
    Ç    c
    ğ    g
    Ğ    g
    ö    o
    Ö    o
    ı    i
    İ    i
    ş    s
    Ş    s
    ü    u
    Ü    u
)};

our $UPCASE_ACCENTS = {qw(
    ç    C
    Ç    C
    ğ    G
    Ğ    G
    ö    O
    Ö    O
    ı    I
    İ    i
    ş    S
    Ş    S
    ü    U
    Ü    U
)};

our $TOGGLE_ACCENT = {qw(
    c    ç
    C    Ç
    g    ğ
    G    Ğ
    o    ö
    O    Ö
    u    ü
    U    Ü
    i    ı
    I    İ
    s    ş
    S    Ş
    ç    c
    Ç    C
    ğ    g
    Ğ    G
    ö    o
    Ö    O
    ü    u
    Ü    U
    ı    i
    İ    I
    ş    s
    Ş    S
)};

foreach my $ch ( 'A'..'Z' ) {
    my $lc = lc $ch;
    $DOWNCASE_ASCIIFY->{ $ch } =
    $DOWNCASE_ASCIIFY->{ $lc } =
    $UPCASE_ACCENTS->{   $ch } =
    $UPCASE_ACCENTS->{   $lc } = $lc;
}

# Data hash imported from
# https://github.com/berkerpeksag/ruby-turkish-deasciifier/
#     blob/master/lib/deasciifier/table.rb
#

our $PATTERN = {'c' => {
    'bu aXa' => -1, 'Cki Xi' => -2, 'na Xog' => 3,
    'ram Xo' => 4, 'gol aX' => 5, 'huyu X' => -6,
    'm Xars' => 7, 'Ik Xip' => -8, 'ncu X ' => 9,
    'zay Xo' => 10, 'Xincik' => 11, 'vre aX' => 12,
    'mUS aX' => -13, 'ie Xol' => -14, ' eva X' => -15,
    'im teX' => -16, 'kriz X' => 17, 'k Xipl' => 18,
    'Ir 1 X' => -19, 'Xamdal' => 20, 'mem Xi' => -21,
    'tI Xat' => 22, 'I Xalt' => -23, 'elinde Xop' => -24,
    'e Xopu' => 25, 'zi Xat' => 26, 'ube aX' => 27,
    'sis Xa' => 28, 'n selX' => 29, 'rgIXiy' => 30,
    'n mUXt' => 31, 'rIS uX' => 32, 'kireXe' => 33,
    'Xiklid' => -34, 'lave X' => 35, 'i minX' => 36,
    'es Xin' => -37, 're Xat' => 38, 'yhin X' => -39,
    'az haX' => -40, 'mi Xar' => 41, 'orge X' => -42,
    'kovaX ' => -43, 'ak Xat' => 44, 'Or aXa' => -45,
    'rpar X' => -46, 'da teX' => -47, 'de Xeb' => -48,
    'yarInX' => -49, 'eXeres' => -50, 'fon aX' => 51,
    'ri Xem' => -52, 'tino X' => -53, '5 i aX' => 54,
    '1 00 X' => -55, 'i dolX' => 56, 'rarInX' => -57,
    'kIS aX' => 58, 'I penX' => -59, 'n aXim' => -60,
    'va aXi' => 61, 'iz Xik' => 62, 'olma X' => 63,
    'g un X' => 64, ' reXet' => 65, 'Xalisk' => 66,
    'u seXi' => 67, 'uXular' => -68, 'birinX' => -69,
    'n Xocu' => 70, 'aXlari' => 71, 's uX ' => -72,
    'n OXe' => -73, 'USXus' => -74, 'boraX' => -75,
    'laXo ' => 76, 'Xookt' => 77, 'Xamca' => 78,
    'ikXik' => -79, 'penXo' => 80, 'elaXo' => 81,
    'Xubak' => 82, 'ikelX' => -83, 'Xolly' => -84,
    'I c X' => 85, 'Xivid' => 86, 'eXvir' => 87,
    'teviX' => -88, 'konXi' => 89, 'tiniX' => 90,
    'boriX' => -91, 'refaX' => 92, 'a gaX' => 93,
    'as tX' => 94, 'clisX' => -95, '0 Xul' => -96,
    'riraX' => 97, 'Con X' => 98, ' bulX' => 99,
    'Xuyoz' => -100, 'Xuyo ' => -101, 'r d X' => 102,
    'oXano' => 103, 'eXon ' => 104, 'Xitra' => 105,
    'he uX' => -106, '4 akX' => 107, '269 X' => 108,
    'a uXe' => 109, ' Cu X' => 110, 'laniX' => 111,
    'iCaXa' => -112, 'nXars' => 113, '  dX ' => 114,
    'Xambe' => 115, 'dad X' => -116, '51 Xb' => -117,
    'romaX' => -118, 'zXak ' => -119, 'kuXar' => 120,
    'Ximbl' => -121, 'fdarX' => -122, 'Xerag' => 123,
    'Xoret' => -124, 'sXiml' => 125, 'krosX' => 126,
    'Xagay' => -127, '5 Xi ' => 128, 'Xikig' => 129,
    'Xabac' => -130, ' leXe' => 131, 'Xokk ' => -132,
    'zI X ' => 133, 'heriX' => -134, 'CiviX' => -135,
    ' kunX' => -136, 'raviX' => -137, 'yo X ' => 138,
    ' Xies' => 139, 'zeliX' => 140, 't Xa ' => 141,
    'u daX' => 142, 'Xopea' => -143, 'U ruX' => -144,
    'GdaSX' => -145, 'Xont ' => 146, 'n h X' => 147,
    'I h X' => 148, 'Xilea' => -149, 'juiXi' => 150,
    'IXiyo' => 151,  'iyteX' => 152, 'eXemd' => -153,
    'GdI X' => 154,  'Xinyu' => 155, 'Xoruw' => -156,
    'yanXe' => 157,  'Xogit' => -158, 'Xarip' => -159,
    'rXal ' => 160,  'e Xeg' => 161, 'kapuX' => 162,
    's aXk' => -163, '  Xi ' => 164, ' Ci X' => 165,
    'kuSXa' => -166, 'az X ' => 167, 'kanXe' => 168,
    'oXanl' => 169,  'ily X' => -170, 'lpenX' => 171,
    'Xinn ' => 172,  'Xarpo' => -173, 'Xails' => 174,
    'rSi X' => -175, 'rdziX' => -176, 'gariX' => -177,
    'roXel' => 178,  'ay X ' => 179, 'UzgUX' => 180,
    'aXise' => -181, ' ka X' => 182, 'liXic' => 183,
    'c Xit' => -184, 'hraXe' => 185, 'tmuSX' => -186,
    'menkX' => -187, 'od Xu' => 188, 'ylk X' => -189,
    'm y X' => 190, 'Xiroz' => 191, 'utXun' => -192,
    'oniXl' => -193, 'l aXd' => 194, 'uXins' => 195,
    'borXi' => 196, 'Xormo' => -197, 'Xokan' => -198,
    'hotiX' => 199, 'rXent' => -200, ' liX ' => 201,
    'Xopol' => 202,  'GaXay' => 203, ' Xova' => -204,
    'mka X' => 205,  'rIlIX' => 206, 'mihaX' => 207,
    'voriX' => -208, 'faXas' => 209, 'cur X' => -210,
    'm aSX' => -211, 'umiXi' => 212, '1 3 X' => 213,
    'lfo X' => -214, 'nXalo' => 215, 'usiXi' => 216,
    'ta Xk' => 217, 'UkXil' => -218, 't U X' => 219,
    'katiX' => 220,  'ikaXt' => -221, 'm Xoo' => 222,
    'gur X' => 223,  'Xorai' => -224, 'Xamga' => 225,
    '0 uX ' => -226, 'Xapo ' => -227, 'mamiX' => -228,
    'un OX' => -229, 'viXka' => -230, 'Xolen' => 231,
    'bow X' => -232, 'ebiXe' => -233, 'visiX' => -234,
    'GretX' => -235, '33 uX' => -236, 'Xizek' => -237,
    'bereX' => -238, 'ep X ' => 239, 'In X ' => 240,
    'lk X ' => 241,  'Xike ' => 242, 'parXe' => -243,
    'sUrXe' => -244, 'olXel' => 245, 'Xolat' => -246,
    'zunX ' => -247, 'daXay' => 248, 'maXhu' => 249,
    'u Xa ' => 250,  'Xano' => -251,  ' iXan' => 252,
    'grgIX' => -253, 'Xapuc' => -254, 'ub Xa' => 255,
    ' Xite' => -256, 'Xayet' => -257, ' Xikk' => -258,
    ' nunX' => 259,  'a Xau' => 260,  'UmUSX' => 261,
    'Xamye' => -262, ' Xamy' => 263, 'U O X' => 264,
    ' aaXi' => 265, 'gUniX' => 266, 'Xhir ' => 267,
    'Xaliz' => -268, ' os X' => -269, 'bu X ' => 270,
    'SI X ' => 271, 'Xonac' => 272, 'aht X' => 273,
    ' s Xr' => 274, ' aXkl' => 275, 'aXarb' => -276,
    '31 Xa' => 277, 'gb Xi' => -278, 'OkXis' => -279,
    'urtIX' => 280, 'edeXa' => 281, 'ure X' => -282,
    'p Xul' => 283, 'ni X ' => 284, 'IlkIX' => 285,
    'riciX' => 286, 'Xolad' => -287, 'b Xal' => -288,
    'u Xe ' => 289,  'Xulli' => -290, 'y Xip' => -291,
    'lasaX' => 292,  'C Xav' => -293, ' kliX' => 294,
    'emoX ' => -295, 'Xasit' => 296, 'fle X' => 297,
    'biXek' => -298, 'Xopel' => -299,'2 Xiy' => -300,
    'C Xo ' => 301,  'enseX' => -302, 'UleXo' => 303,
    ' iXre' => 304, 'Xecel' => -305, 'filiX' => -306,
    'dre X' => -307, 'ulenX' => -308, 'Xides' => -309,
    'I e X' => 310, 'ulaXi' => -311, 'IrmiX' => -312,
    'p paX' => 313, 'i Xod' => -314, 'Xukad' => 315,
    'piriX' => 316, 'dy Xa' => -317, 'nXuh' => -318,
    'manXu' => 319, 'mra X' => -320, 'S Xi ' => 321,
    'h Xi ' => 322, 'Xifle' => -323, 'Xigan' => 324,
    'erkeX' => 325, 'otXul' => -326, 'p erX' => -327,
    'uXarm' => -328, 'vcanX' => 329, ' aXih' => 330,
    'Xapsh' => -331, 'Xabul' => -332, 'b aX ' => -333,
    'rasiX' => -334, 'Xetla' => -335, ' Xays' => -336,
    'nXiko' => 337,  'deXel' => 338, ' aXa ' => 339,
    '  Xrp' => 340, ' hp X' => -341, 'li OX' => 342,
    'rXas ' => -343, 'S Xa ' => 344, 'gio X' => -345,
    'Xobur' => -346, 'zIk X' => 347, 'ep Xo' => 348,
    'y ahX' => -349, 'em X ' => 350, 'guk X' => -351,
    'I Xo ' => 352, '  SuX' => 353, 'm Xut' => 354,
    'haXla' => 355, '  e X' => -356, 'Xinis' => -357,
    'Xinta' => 358, 'nlatX' => -359, 'Sok X' => -360,
    'k Xe ' => 361, ' kiXh' => 362, 'pinXi' => 363,
    'fi aX' => 364, 'skeXi' => -365, 'liXek' => -366,
    'uro X' => -367, 'tCa X' => 368, 'p liX' => 369,
    'nakiX' => -370, 'ty Xe' => -371, ' feXe' => 372,
    'biXir' => -373, 'asalX' => -374, 'erkaX' => 375,
    'n poX' => 376, '5 Xur' => -377, 'aXare' => -378,
    'saXid' => -379, 'nXan ' => -380, 'dinXa' => 381,
    'nXag ' => 382, 'seXev' => -383, 'S Xeb' => -384,
    ' okaX' => -385, 'kit X' => 386, '  orX' => 387,
    'ekkeX' => -388, 'n puX' => -389, 'erenX' => -390,
    'aCkIX' => 391, 'o uXa' => 392, 'pe Xo' => -393,
    'Xalgo' => -394, 'laaaX' => 395, 'eniXi' => -396,
    'z Xuc' => 397, 'f Xul' => -398, 'kukiX' => -399,
    ' ngoX' => -400, 'koiXi' => 401, 'Xitak' => 402,
    'Xiste' => 403, 'maniX' => 404, 'e boX' => -405,
    'Xino ' => -406, 'hXan ' => -407, 'eXiog' => -408,
    'fa X ' => 409, 'panXo' => 410, 'saXim' => 411,
    ' daXa' => 412, 'lXugi' => 413, 'ed Xa' => -414,
    'saXan' => 415, 'Guk X' => 416, 'garXi' => -417,
    ' apaX' => 418, 'bataX' => -419, 'b eyX' => 420,
    'i z X' => 421, 'i s X' => 422, 'Xoptu' => -423,
    'Xiban' => 424, 'Xogam' => -425, 'anXic' => 426,
    'ikaCX' => -427, 'S Xoc' => 428, 'of aX' => -429,
    'Xilav' => 430, 'nXerl' => 431, 'e Xib' => -432,
    'Xileg' => 433, 'Xayiv' => -434, 'eji X' => 435,
    'Xopli' => -436, 'llaXi' => -437, 'u erX' => -438,
    'I Xeb' => -439, 'iXiye' => -440, '45 Xa' => 441,
    'm Xep' => -442, 'isXi ' => 443, 'smuSX' => -444,
    'f Xil' => 445, 'Xolga' => -446, 'aXada' => -447,
    'GmaXi' => -448, 'mes X' => -449, 'a baX' => -450,
    'Xinga' => 451, 'rpenX' => 452, 'C Xe ' => 453,
    'Xilab' => 454, 'ar X ' => 455, 'Xayam' => -456,
    'jevaX' => -457, 'omanX' => -458, 'eberX' => -459,
    'iskoX' => 460, 'oy Xo' => 461, 'kaniX' => 462,
    'arXak' => -463, 'amaXa' => -464, 'karXe' => 465,
    'C Xoc' => 466, 'stanX' => -467, 'ranXe' => -468,
    'orveX' => 469, 'leriX' => -470, 'maXis' => -471,
    'Xamag' => 472, 'aS iX' => 473, 'a uXa' => 474,
    'anXin' => -475, 'Xerel' => -476, 'rt aX' => 477,
    'Xalig' => -478, 'ediXi' => -479, 't ilX' => 480,
    'myonX' => -481, 'Xetes' => 482, 'retiX' => -483,
    'Xanli' => -484, ' Xesa' => -485, 'bovaX' => -486,
    'pis X' => -487, 'Xecen' => 488, 'UreXe' => -489,
    'U miX' => -490, 'urmaX' => -491, 'Xceg' => 492,
    'vereX' => -493, 'rdUnX' => -494, 'acatX' => 495,
    'ndy X' => -496, 'lerXe' => -497, 'layaX' => -498,
    'laXag' => -499, 'mumXi' => 500, ' aCaX' => -501,
    'n Xik' => 502,  'Xesit' => 503, 'Xerek' => 504,
    'a Xik' => 505,  'SXeg' => -506, 'tavX' => 507,
    'amXl' => 508,   'dstX' => -509, 'Xolw' => -510,
    'zkUX' => 511, 'erIX' => -512, 'Xrec' => 513,
    'Xsit' => 514, 'yodX' => 515, 'tXeg' => -516,
    'bseX' => -517, 'goXn' => 518, 'Xlah' => -519,
    'Xolb' => -520, 'hXur' => -521, 'edoX' => -522,
    'umXe' => 523, 'geXu' => 524, 'usuX' => -525,
    'rXee' => 526, 'etcX' => 527, 'Xiok' => 528,
    'Xogh' => -529, 'iXma' => -530, 'koXr' => 531,
    'hefX' => -532, 'fXad' => -533, 'moXl' => -534,
    'gliX' => -535, 'skUX' => 536, 'tiyX' => 537,
    'vyaX' => 538, 'C X ' => 539, 'koXc' => 540,
    'vniX' => 541, 'Xaeb' => 542, 'IlXu' => 543,
    'CelX' => -544, 'SXig' => -545, 'piXl' => 546,
    ' Xoi' => -547, 'uicX' => 548, ' UXc' => 549,
    'girX' => -550, 'Xoct' => -551, 'keXc' => 552,
    'ruXi' => -553, 'pUrX' => 554, 'pIrX' => -555,
    'nkXl' => -556, 'CiaX' => 557, 'yaIX' => -558,
    ' UlX' => 559, 'iy X' => 560, 'eetX' => -561,
    'blIX' => -562, 'aveX' => -563, 'rnuX' => -564,
    'Xtif' => -565, 'feXl' => 566, 'oXev' => -567,
    'ljuX' => 568, 'Xorf' => -569, 'tabX' => -570,
    'geaX' => 571, 'kUCX' => 572, 'kUXc' => 573,
    'OnXi' => 574, 'iXet' => -575, 'tXas' => -576,
    'mlaX' => -577, 'wo X' => -578, 'apiX' => 579,
    'hiXd' => 580, 'Xagr' => 581, 'uy X' => -582,
    'Xebl' => 583, 'rilX' => -584, 'Xelv' => -585,
    'y Xk' => 586, 'Xsu ' => -587, 'peXk' => -588,
    'ukuX' => 589, ' uzX' => 590, 'x Xo' => -591,
    'yaXm' => -592, 'IXaa' => -593, 'OzaX' => 594,
    'Xech' => -595, 'OrtX' => -596, 'oXko' => -597,
    'kXah' => 598, 'uliX' => -599, 'lilX' => -600,
    'CliX' => -601, 'Xipt' => -602, 'Xal1' => 603,
    'oyXa' => 604, 'CiXc' => 605, 'rokX' => -606,
    'lasX' => 607, 'iXoc' => 608, 'morX' => -609,
    'boXo' => -610, 'ouaX' => -611, 'Xamt' => 612,
    'dIrX' => -613, 'buCX' => 614, 'nXov' => 615,
    'nXs ' => 616, 'Xirn' => -617, 'onX ' => -618,
    'e Xg' => -619, 'Xatr' => -620, 'ondX' => 621,
    'nXez' => 622, 'odiX' => -623, 'Xork' => -624,
    'Xoum' => 625, 'inkX' => -626, 'utuX' => -627,
    'uXcu' => 628, 'yuhX' => -629, 'Xnae' => 630,
    'Xao ' => -631, 'umcX' => 632, 'Xve ' => 633,
    'uXea' => -634, 'Xayy' => 635, 'kiXd' => 636,
    'udoX' => -637, ' gXu' => 638, 'gneX' => 639,
    'Xesz' => -640, ' Xcs' => 641, 'ruXr' => 642,
    'Xozy' => -643, 'lviX' => -644, 'Xibr' => -645,
    'Xlur' => -646, 'ulXi' => 647, ' nIX' => -648,
    'sezX' => 649, ' isX' => -650, 'raXd' => 651,
    'isaX' => -652, 'srtX' => -653, 'leXa' => 654,
    'pelX' => -655, 'tXem' => 656, 'I Xg' => -657,
    'zilX' => -658, 'vasX' => -659, '4 yX' => -660,
    'uzUX' => 661, 'gelX' => -662, 'Xuan' => 663,
    'nviX' => -664, 'lguX' => -665, 'Xcag' => 666,
    'ureX' => -667, 'agaX' => -668, 'imXa' => 669,
    'muXa' => -670, 'guXu' => -671, 'tXce' => 672,
    'Xucg' => 673, 'nXvi' => 674, 'veXk' => -675,
    'Xavk' => -676, 'liXl' => 677, 'Xadu' => 678,
    'my X' => -679, 'Xdis' => 680, 'pmeX' => -681,
    'lseX' => -682, 'oduX' => -683, 'uSuX' => -684,
    'iXev' => 685, 'Xulh' => 686, 'Xiol' => -687,
    'paSX' => 688, 'UziX' => 689, 'Xot ' => -690,
    'uXit' => -691, 'ikiX' => -692, 'kiXa' => 693,
    ' zUX' => -694, 'omeX' => -695, 'kkeX' => 696,
    'aXif' => -697, 'I dX' => 698, 'fuXk' => -699,
    'Xirv' => -703, 'eXig' => -704, 'punX' => -705,
    'msiX' => -706, 'syaX' => -707, 'aIrX' => -708,
    'Xinh' => 709, 'oXea' => -710, 'ierX' => -711,
    'usaX' => -712, ' gIX' => -713, 'dluX' => -714,
    ' aXc' => -715, 'fliX' => -716, 'suXo' => 717,
    'ssoX' => -718, 'puXu' => -719, 'fXan' => -720,
    'fInX' => -721, 'heXe' => -722, 'nCIX' => -723,
    'Xin1' => 724, 'a Xb' => -725, 'mXak' => 726,
    'halX' => -727, 'Xesc' => -728, 'gIXa' => -729,
    'umaX' => -730, 'breX' => -731, 'Xatf' => -732,
    'Xevt' => -733, 'iXig' => -734, 'Xuls' => 735,
    'vaXt' => 736, 'Xlaf' => -737, ' yeX' => 738,
    'elaX' => -739, 'reXo' => -740, 'rlaX' => -741,
    'danX' => -742, 'nXur' => -743, 'ibiX' => -744,
    ' UXk' => 745, 'USXe' => -746, 'uziX' => -747,
    'nduX' => -748, 'zXes' => -749, 'Xaz ' => -750,
    ' aSX' => 751, 'Xoco' => -752, 'proX' => -753,
    'plIX' => -754, 'Xenn' => -755, 'lXam' => 756,
    'betX' => 757, 'Xagn' => -758, 'Xigd' => 759,
    'Xesm' => 760, 'IkIX' => -761, ' bIX' => 762,
    'asaX' => -763, 'hiiX' => 764, 'Xabu' => 765,
    'ekeX' => -766, 'rliX' => -767, 'rkIX' => -768,
    'Xmil' => -769, 'UnUX' => 770, 'SUnX' => -771,
    'ImaX' => -772, 'Xade' => -773, 'Xatc' => -774,
    'Xilm' => 775, 'marX' => -776, 'rXus' => -777,
    'porX' => -778, 'Xtik' => 779, 'kaXt' => 780,
    'enXi' => -781, 'eteX' => -782, 'eXeg' => -783,
    'silX' => -784, 'yrIX' => -785, 'Xozu' => 786,
    ' iXi' => 787, 'UXn' => 788, 'yXp' => 789,
    'Xbc' => -790, 'aXj' => 791, 'Xgw' => -792,
    'tXp' => 793, 'IeX' => -794, 'Xcg' => 795,
    'Xzo' => 796, 'hXp' => -797, 'Xsc' => -798,
    'Xo2' => -799, 'kXp' => -800, ' X1' => -801,
    'aXf' => -802, 'weX' => -803, 'kcX' => 804,
    'dXg' => 805, 'wrX' => -806, 'znX' => -807,
    'jnX' => -808, 'ueX' => -809, 'Xth' => -810,
    'iXp' => -811, 'Xzy' => -812, 'Xej' => -813,
    'tXb' => -814, 'SCX' => -815, 'SXc' => 816,
    'Xeu' => -817, 'uXv' => 818, 'UbX' => -819,
    'sOX' => 820, 'chX' => -821, 'Xix' => -822,
    'voX' => -823, 'lcX' => 824, ' SX' => -825,
    ' lX' => -826, 'tCX' => -827, 'UXv' => 828,
    ' hX' => -829, ' fX' => -830, 'euX' => -831,
    'jeX' => -832, '4aX' => -833, 'eXg' => -834,
    'llX' => -835, 'ylX' => -836, 'rzX' => 837,
    'GuX' => -838, 'thX' => -839, 'yXb' => 840,
    'rXh' => -841, 'jlX' => 842, 'iXm' => 843,
    'IXt' => 844, 'Xd ' => -845, 'iuX' => -846,
    'Xlo' => -847, 'Xah' => -848, 'yIX' => -849,
    'Xks' => -850, 'Xok' => 851, 'qX' => -852,
    'Xx' => -853, 'X2' => 854, 'X5' => 855,
    'X7' => -856, 'abilme gUXune' => 857,
    'imin sonuXud' => 858,
    '0 stankoviX' => -859, 'i smeonoviX' => -860,
    'n mijatoviX' => 861, 'I silah gUX' => 862,
    'layan saraX' => 863, 'k halilagiX' => 864,
    'zlar polisX' => -865, 'i halilagiX' => 866,
    ' elindeki X' => 867, 'nlarInIn Xo' => 868,
    'n de aXikt' => -869, 'a ivanoviX' => -870,
    'Iyla baliX' => 871, 'ar ve Xam ' => 872,
    'i raftingX' => 873, 'k markoviX' => -874,
    'rIlmayan X' => 875, 'asInda Xop' => -876,
    'i bir Xam ' => 877, 'nin Xanlar' => 878,
    'i sonuXudu' => 879, 'r dolara X' => 880,
    'a aXisini' => -881, 'I Xanlar ' => 882,
    'kan a aXi' => -883, 'u da Xaba' => -884,
    'a aXiyord' => -885, 'i de Xaba' => -886,
    'e vlaoviX' => 887, 'i adI geX' => 888,
    'ne ve Xog' => -889, 'an aXilar' => -890,
    'na geXeli' => 891, 'e keynesX' => -892,
    'ka bir aX' => 893, 'lara aXiy' => -894,
    'zeki Xab' => -895, 'e armutX' => -896,
    'a aXisiy' => -897, 'unda Xos' => 898,
    'r aXisid' => -899, ' aXisin ' => -900,
    'men Xop ' => -901, 'okanoviX' => -902,
    'n genXel' => 903, 'Cek kauX' => -904,
    'son haXi' => 905, 'bi dostX' => -906,
    'an mahXu' => -907, '  bu Xos' => 908,
    'ulu Xam ' => 909, 'im aXim ' => -910,
    'bin Xam ' => 911, 'tU saraX' => 912, 'I kuruXa' => 913,
    'nce Xamd' => 914, 'vaS aXis' => -915, 'e pen X' => -916,
    'n Xiler ' => 917, 'ek aXili' => 918, 'emiS Xin' => -919,
    'isini Xo' => 920, 'eki Xim ' => 921, ' de Xam ' => 922,
    'zasIna X' => 923, ' Xandik' => 924, 'joroviX' => -925,
    'n kukoX' => 926, 'Xandarl' => 927, 'nobil X' => 928,
    'nuna iX' => -929, 'in Xad ' => 930, '  barXa' => 931,
    'a avdiX' => -932, 'kada X ' => 933, 'ecep Xi' => -934,
    'ir haX ' => 935, 'aXiklis' => -936, 'panoviX' => -937,
    ' tarXin' => 938, 'oXanind' => 939, 't Xilar' => 940,
    'er Xayi' => 941, 'lI saX ' => -942, ' ta Xam' => 943,
    'lit Xam' => 944, 'ri Xaml' => 945, 'batIXim' => 946,
    'lsa Xam' => 947, 'an nahX' => -948, 'ok aXiy' => -949,
    ' paXin ' => 950, ' Xinlik' => -951, 'ri Xat ' => 952,
    'm saraX' => 953, ' Xurumd' => -954, 'a arXa ' => 955,
    'lan akX' => 956, 'un akXa' => 957, 'ol Xin ' => -958,
    'usam aX' => 959, 'ma Xaml' => 960, 'da faXa' => 961,
    'aS aXi ' => -962, 'trafI X' => 963, 'bu Xin ' => -964,
    'ze aXiy' => -965, ' un aX ' => -966, 't Xarsi' => 967,
    'i o gUX' => 968, 'niGde X' => 969, 'tekli X' => 970,
    'hce maX' => -971, 't urumX' => 972, 'dada Xa' => 973,
    'in aXiy' => -974, 'Um aXis' => -975, 'lis Xe ' => 976,
    've aXim' => -977, 'ce Xop ' => -978, '  Xamla' => 979,
    'le Xosu' => 980, 'et genX' => 981, 'S aXil ' => -982,
    'ni aXis' => 983, ' Sen X ' => 984, 'rlIk Xa' => 985,
    'Xarcal' => -986, 'kut Xu' => 987, 'd maX ' => -988,
    'urem X' => 989, '8 Xarm' => -990, 'afa aX' => -991,
    'kentiX' => 992, 'Xollez' => -993, 'I Xor ' => -994,
    'si Xop' => -995, 'ner X ' => 996, 'aloviX' => -997,
    'z aXiz' => 998, 'Xatanz' => -999, 'cop Xo' => -1000,
    'haraXo' => -1001, 'Xemred' => 1002, ' maXak' => 1003,
    'ru Xam' => 1004, 'Xallin' => -1005, 'In Xi ' => 1006,
    'Xilesu' => -1007, 'Xagiva' => -1008, 'Xarine' => -1009,
    'eXeme ' => -1010, 'zon X ' => 1011, 'yin X ' => 1012,
    'l Xirc' => 1013, 'aolo X' => -1014, 'Xogani' => -1015,
    'uzoviX' => -1016, 'uroviX' => -1017, 'Xivit ' => 1018,
    'fiye X' => 1019, 'borXul' => 1020, 'n Ximm' => -1021,
    'zangoX' => 1022, 'ste aX' => 1023, 'ri uXl' => 1024,
    'Xentik' => 1025, 'Xopluy' => -1026, 'arakaX' => 1027,
    'Xapule' => -1028, 'Xucukl' => 1029, ' Xerme' => -1030,
    'eXemin' => -1031, ' geXi ' => 1032, 'Xorvan' => 1033,
    'Xelebr' => -1034, ' Xelli' => 1035, 'atoriX' => 1036,
    'rk Xu ' => 1037, ' Xello' => 1038, 'Xevri ' => -1039,
    ' meXim' => 1040, 'Xiple ' => -1041, 'rm Xan' => 1042,
    'yaratX' => -1043, 'aya X ' => 1044, 'iteliX' => 1045,
    'li teX' => -1046, 'g Xole' => -1047, 'enXayd' => 1048,
    'ol Xul' => -1049, 'on Xim' => 1050, 'gin X ' => 1051,
    'vaXev ' => -1052, 'Xevriy' => -1053, 'a aXim' => -1054,
    'Xisimi' => 1055, 'Xelil ' => -1056, 'as Xil' => -1057,
    't Xinc' => 1058, ' aXinm' => -1059, 'eramiX' => -1060,
    ' Can X' => 1061, '  Xop ' => -1062, 'm aXiy' => -1063,
    'Xorduk' => 1064, ' Xinim' => -1065, 'l Xars' => 1066,
    'k Xars' => 1067, 'i momX' => 1068, 'un Xat' => 1069,
    'unXell' => 1070, 'p Xars' => 1071, 'Xalis ' => 1072,
    ' iC aX' => 1073, 'hayalX' => -1074, 'Xanakk' => 1075,
    'Xolas ' => -1076, 'da taX' => -1077, 'OGrenX' => -1078,
    'aXiss' => -1079, 'Xorel' => -1080, 'poreX' => -1081,
    'Xars ' => -1082, '6 akX' => 1083, '5 akX' => 1084,
    's Xip' => -1085, 'Xopta' => -1086, 'SilXe' => -1087,
    'uto X' => -1088, 'Xapou' => -1089, 'lIXag' => -1090,
    ' OlUX' => 1091, 'Xalib' => -1093, 'C Xop' => -1094,
    'iXolu' => -1095, ' eXci' => 1096, 'iadiX' => -1097,
    'eXhil' => 1098, 'vehiX' => -1099, 'giniX' => 1100,
    'Xidam' => 1101, ' iXha' => 1102, 'uk X ' => 1103,
    'SarkX' => -1104, 'UnXer' => 1105, 'Xayil' => -1106,
    'kXel ' => -1107, 'nn Xi' => 1108, 'aXipa' => -1109,
    'mrsIX' => 1110, 's Xi ' => 1111, 'Xuzum' => 1112,
    'Xallu' => -1113, 'saXu ' => 1114, 'mleX ' => 1115,
    'Xesim' => -1116, 'ehiXe' => -1117, 'za X ' => 1118,
    'izaX ' => 1119, 'k Xo ' => 1120, ' Xibu' => -1121,
    'uXal ' => -1122, ' baXe' => 1123, 'Xarei' => -1124,
    'Xatic' => -1125, 'beliX' => 1126, 'v Xam' => 1127,
    'GraXi' => 1128, 'Xipol' => -1129, 'dahiX' => -1130,
    ' Xabo' => -1131, ' Xyp ' => 1132, 'Xadas' => 1133,
    'dIXal' => 1134, 'hXenl' => 1135, 'odziX' => -1136,
    ' Xaaa' => 1137, ' peXi' => 1138, 'roliX' => -1139,
    'g Xii' => 1140, 'pamiX' => -1141, 'n gaX' => 1142,
    'aselX' => 1143, 'lingX' => 1144, 'o Xi ' => 1145,
    'Xorey' => -1146, 'perXe' => 1147, 'i iXk' => 1148,
    'Xitat' => -1149, 'Xoraz' => -1150, 'Xelia' => -1151,
    'kuXuk' => -1152, 'adXek' => 1153, 'CalIX' => 1154,
    'k Xul' => 1155, 'banXe' => 1156, 'OlgeX' => -1157,
    'liXeg' => -1158, 'erhaX' => 1159, 'foXal' => 1160,
    'dalXa' => 1161, 'kmiSX' => -1162, ' Xicc' => -1163,
    ' Xic ' => -1164, 'Ximbo' => -1165, 'i Xu ' => 1166,
    'Xold ' => -1167, 'teXko' => 1168, 'ymISX' => -1169,
    'daliX' => -1170, 'Xall ' => -1171, 'nXika' => 1172,
    'k kuX' => -1173, 'i daX' => 1174, 'juriX' => -1175,
    'rie X' => -1176, ' aXme' => -1177, 'Xeter' => -1178,
    'U Xul' => -1179, 'hIrXi' => 1180, 'eXme ' => 1181,
    'ilXi ' => -1182, 'IlXi ' => -1183, 'Xeber' => -1184,
    'anXos' => 1185, 'oiniX' => 1186, 'Xelal' => -1187,
    'UXunc' => 1188, 'nsu X' => 1189, 'raliX' => 1190,
    'mayaX' => -1191, 'Xitie' => -1192, 'Xipp' => -1193,
    'a Xg' => -1194, 'suXh' => -1195, 'OXel' => -1196,
    'ekX ' => 1197, 'etuX' => -1198, ' bXe' => -1199,
    'Xcok' => 1200, 'oseX' => -1201, 'mXat' => 1202,
    'abXi' => 1203, 'tUlX' => -1204, 'seXn' => 1205,
    'olIX' => -1206, 'iCX ' => -1207, 'iXc ' => 1208,
    'Xegs' => -1209, 'delX' => -1210, 'Xec ' => -1211,
    'nXai' => 1212, 'eguX' => -1213, 'Xizv' => -1214,
    'filX' => -1215, 'Xlag' => -1216, 'aiXi' => 1217,
    'Xave' => -1218, 'rreX' => -1219, 'auXk' => -1220,
    'oyiX' => 1221, 'lkIX' => -1222, 'Xalh' => -1223,
    ' imX' => 1224, 'paXu' => 1225, 'akUX' => 1226,
    'ameX' => -1227, ' ihX' => -1228, 'hilX' => -1229,
    '  OX' => -1230, ' oXi' => 1231, 'gasX' => -1232,
    'cXis' => 1233, 'oaXi' => 1234, 'Xatu' => -1235,
    'praX' => -1236, 'aCXa' => -1237, 'Xipr' => -1238,
    'braX' => -1239, 'liXk' => -1240, 'naXt' => 1241,
    'zalX' => -1242, 'cilX' => -1243, 'teXl' => 1244,
    'Xlea' => -1245, 'lriX' => -1246, ' vuX' => -1247,
    'yriX' => -1248, 'ImIX' => -1249, 'Xab ' => -1250,
    'juiX' => -1251, 'Xii ' => -1252, 'farX' => -1253,
    'dviX' => 1254, 'heXk' => -1255, ' aXu' => -1256,
    'CXeg' => -1257, 'Xso ' => -1259, 'Xopo' => -1260,
    ' raX' => -1261, ' InX' => -1262, 'anIX' => -1263,
    'Xaru' => -1264, 'llIX' => -1265, ' keX' => 1266,
    'truX' => -1267, 'tirX' => -1268, 'derX' => -1269,
    'denX' => -1270, 'aXta' => 1271, 'nmaX' => -1272,
    'Xats' => -1273, ' muX' => -1274, 'Xagl' => 1275,
    'Xobo' => -1276, 'piXi' => 1277, 'UrUX' => -1278,
    'nriX' => -1279, ' CiX' => 1280, 'chiX' => -1281,
    'Xisl' => 1282, ' CeX' => 1283, 'riXk' => -1284,
    'sveX' => 1285, 'kinX' => -1286, 'uraX' => -1287,
    'yunX' => -1288, 'bahX' => 1289, 'yiXe' => -1290,
    'leXe' => -1291, 'uXp' => -1292, '0X ' => -1293,
    'Xgt' => -1294, 'Xgp' => -1295, 'Xsh' => -1296,
    'pXh' => -1297, 'Xoj' => -1298, 'Xbk' => -1299,
    'Xaw' => -1300, 'ncX' => 1301, 'cnX' => -1302,
    'eXu' => -1303, 'eaX' => -1304, 'xoX' => -1305,
    ' kX' => -1306, 'hOX' => -1307, 'Xei' => -1308,
    'Xiw' => -1309, 'Xb ' => -1310, 'Xae' => -1311,
    'Xts' => -1312, 'Xk ' => -1313, 'Xho' => -1314,
    'Xw' => -1315, 'a halilagiX' => 1316,
    'en bregoviX' => -1317, '6 petroviX' => -1318,
    '  bregoviX' => -1319, 'e aXiyorum' => -1320,
    'menin aXis' => -1321, 'virdiGi Xa' => 1322,
    'edeki Xam' => 1323, 'i aXisini' => -1324,
    ' Xurumu ' => -1325, 'har korX' => 1326,
    'lise Xan' => 1327, 'nra Xam ' => 1328,
    'ha Xook ' => 1329, 'te tunXa' => 1330,
    'i aXiniz' => 1331, '1 ladiX' => -1332,
    'tanoviX' => -1333, 'jatoviX' => -1334,
    'ir Xim ' => 1335, ' Xildim' => -1336,
    'aXisiyd' => -1337, 'k aXimi' => -1338,
    ' sUrXu ' => 1339, 'ni aXar' => 1340,
    'Xiklet ' => -1341, '  aXisi' => -1342,
    '  Xad ' => 1343, 'UC Xim' => 1344,
    'akapaX' => 1345, ' daciX' => 1346, 'Xakimi' => -1347,
    'gazetX' => -1348, 'n Xilo' => -1349, 'Xaball' => -1350,
    'I Xad ' => 1351, 'raXk' => -1352, 'aXken ' => 1353,
    'za Xam' => 1354, ' sam X' => -1355, 'a in X' => -1356,
    'Ik Xa ' => 1357, 'ilitiX' => 1358, 'U aXar' => 1359,
    'r aXim' => -1360, 'tatUrX' => 1361, ' basiX' => -1362,
    ' Xilik' => -1363, ' kanXi' => 1364, 'Xitico' => -1365,
    'i Xipi' => -1366, 'Xoplu ' => -1367, ' Xinde' => 1368,
    'k aXar' => 1369, 'k Xile' => 1370, ' Xorle' => -1371,
    'olaXan' => 1372, 'o Xe ' => 1373, 'yra X' => 1374,
    ' laXm' => -1375, 'Xavas' => 1376, 'obuXi' => 1377,
    'demaX' => 1378, 'us X ' => 1379, 'sliXe' => -1380,
    'lXime' => 1381, 'Xubai' => 1382, 'ruXam' => 1383,
    ' anXu' => 1384, 'nCXi ' => -1385, 'aXely' => 1386,
    'Xusok' => 1387, 'Xorbu' => -1388, 'hoviX' => -1389,
    'Xigna' => -1390, 'Xenem' => 1391, 'ekliX' => -1392,
    'balIX' => -1393, 'gamiX' => -1394, ' Xis ' => 1395,
    'Xorbe' => -1396, 'Xeney' => 1397, 'Xibik' => 1398,
    'taXam' => -1399, ' Ximc' => -1400, 'Xungu' => 1401,
    'pereX' => -1402, 'iiriX' => 1403, 'iaXik' => 1404,
    'aoviX' => -1405, 'im X ' => 1406, 't Xe ' => 1407,
    'Xoper' => -1408, 'rXakl' => -1409, 'Xakas' => -1410,
    'ollaX' => -1411, 'garXo' => 1412, 'tUk X' => 1413,
    ' Xadl' => 1414, 'eXem ' => -1415, 'iCinX' => -1416,
    'Xurac' => 1417, 'e soX' => -1418, 'Xeng ' => 1419,
    'mariX' => -1420, 'ndiXt' => 1421, '06 aX' => -1422,
    'poGaX' => 1423, 'd Xil' => 1424, 'o Xol' => -1425,
    'Xarib' => -1426, 'utreX' => -1427, ' OzXe' => 1428,
    'the X' => -1429, 'Xara ' => -1430, ' Xert' => -1431,
    ' Xibi' => -1432, 'iXre ' => 1433, 'Xatal' => 1434,
    'Xeleb' => 1435, ' sanX' => -1436, 'ael X' => -1437,
    'CIkaX' => -1438, 'iXin ' => 1439, 'gorX' => 1440,
    'paiX' => -1441, 'Xhc ' => 1442, 'i Xb' => -1443,
    'njiX' => 1444, 'geXr' => 1445, 'kuXc' => 1446,
    'Xicl' => -1447, 'hXak' => -1448, ' uXo' => 1449,
    'SliX' => -1450, 'oXel' => -1451, 'Xkow' => -1452,
    'GaXd' => 1453, 'Xazo' => 1454, 'Xotu' => 1455,
    'nXka' => 1456, 'onUX' => 1457, 'komX' => 1458,
    'oboX' => -1459, ' yX ' => 1460, 'mUlX' => -1461,
    'UnXa' => 1462, 'Xorg' => -1463, 'turX' => -1464,
    'pheX' => -1465, 'naXy' => 1466, 'adaX' => -1467,
    'Xuli' => -1468, 'tXeb' => -1469, 'Ximo' => -1470,
    'iXla' => -1471, 'reXk' => -1472, 'relX' => -1473,
    'zorX' => -1474, 'Xif ' => -1475, 'corX' => -1476,
    '  Xb' => -1477, 'Xolm' => -1478, 'Xorv' => -1479,
    'meXk' => -1480, 'tXig' => 1481, 'Xaog' => -1482,
    'eduX' => -1483, 'n Xb' => -1484, 'Xinar' => 1485,
    'liXi' => -1486, 'nlIX' => -1487, 'UXet' => -1488,
    ' Xeg' => -1489, 'Xoda' => 1490, 'ny X' => -1491,
    'avaX' => -1492, 'lIXk' => 1493, ' baX' => -1494,
    'uXuz' => -1495, 'unuX' => -1496, 'luXu' => -1497,
    'IlaX' => -1498, 'yaXa' => -1499, 'sXv' => 1500,
    'plX' => -1501, 'Xoq' => -1502, 'Xgr' => -1503,
    'OhX' => -1504, 'Xaj' => -1505, 'CuX' => -1506,
    'coX' => -1507, 'Xyu' => 1508, 'In da Xab' => 1509,
    'e aXisini' => -1510, 'urhan OX' => 1511,
    ' Xisini ' => 1512, 'men Xe ' => 1513,
    ' Xatani' => -1514, ' Xinlig' => -1515, 'araXali' => 1516,
    'na Xop ' => -1517, 'r Xarsi' => 1518, 'aXiktik' => -1519,
    'Sin Xab' => -1520, 'na Xivi' => 1521, ' aXimi ' => -1522,
    'laXayi' => 1523, 'omoviX' => -1524, ' ganiX' => 1525,
    'Xarni ' => 1526, 'uloviX' => -1527, 'venXer' => 1528,
    ' abaXa' => 1529, 'k mesX' => -1530, 'Xankat' => -1531,
    'Xapra ' => -1532, 'p Xile' => 1533, 'genXog' => 1534,
    ' Xelie' => -1535, 's Xile' => 1536, ' Xemis' => 1537,
    'ureviX' => -1538, 'miraX ' => 1539, 'rteXin' => 1540,
    ' Xult' => -1541, 'zeviX' => -1542, 'tlaXi' => -1543,
    'Xarst' => -1544, 'ohaX ' => 1545, ' aXag' => 1546,
    'Xalme' => -1547, 'Xates' => 1548, ' uXul' => 1549,
    'ymiSX' => -1550, 'suXug' => -1551, 'geraX' => -1552,
    'opaXi' => -1553, 'Xiyak' => -1554, 'alXis' => -1555,
    'e Xa ' => 1556, 'Xore ' => -1557, 'Xobai' => -1558,
    'suriX' => 1559, 'haXam' => -1560, 'k Xi ' => 1561,
    'IlXay' => 1562, ' kuXu' => 1563, '4 aX ' => -1564,
    'Xenen' => 1565, 'Xciog' => 1566, 'Xopu ' => -1567,
    ' piX ' => 1568, 'aXill' => -1569, 'Xollo' => -1570,
    'rmuSX' => -1571, 'rpaXi' => -1572, 'Xuruf' => -1573,
    'aXars' => 1574, 'n Xeb' => -1575, 'Xizme' => 1576,
    'abanX' => -1577, 'yapaX' => -1578, 'dereX' => -1579,
    'IaX ' => -1580, 'iseX' => -1581, 'Xirr' => -1582,
    'CIrX' => 1583, 'Xtus' => -1584, 'Xopa' => -1585,
    'e Xb' => -1586, ' Xoy' => -1587, 'egiX' => -1588,
    'aXlu' => -1589, 'rUlX' => -1590, 'afuX' => -1591,
    'Xova' => 1592, 'Xopy' => -1593, ' auX' => -1594,
    'Xolt' => -1595, ' Xaa' => -1596, 'Xoch' => -1597,
    'kXad' => 1598, ' y X' => -1599, 'Xarv' => -1600,
    'mXav' => 1601, 'fusX' => 1602, 'Xecc' => -1603,
    'uXay' => 1604, 'rCaX' => -1605, 'Xach' => -1606,
    'uniX' => -1607, 'Xace' => -1608, 'Xabl' => -1609,
    'ksiX' => -1610, ' guX' => -1611, 'eXka' => -1612,
    'zeXi' => -1613, 'melX' => -1614, 'iXio' => -1615,
    'traX' => -1616, 'zXak' => 1617, 'rviX' => -1618,
    'Xopp' => -1619, 'dilX' => -1620, 'carX' => -1621,
    'ralX' => -1622, 'iXid' => -1623, 'risX' => -1624,
    'Xalo' => -1625, 'Xaye' => 1627, 'osiX' => -1628,
    'neXe' => -1629, 'resX' => -1630, 'niXk' => -1631,
    'Xage' => -1632, 'Xlau' => -1633, 'ruXu' => -1634,
    'dtX' => -1635, '0Xs' => -1636, 'cuX' => -1637,
    'Xee' => -1638, 'ioX' => -1639, 'Xiu' => -1640,
    'oeX' => -1641, 'Xkb' => -1642, 'hX ' => -1643,
    'Xgo' => 1644, 'cIX' => -1645, 'hoX' => -1646,
    'pIX' => -1647, 'Xh ' => -1648, ' nX' => -1649,
    'e aXisiy' => -1650, 'ptIGI aX' => 1651,
    'Xarlest' => 1652, 'u 5 geX' => 1653, ' Xarlar' => 1654,
    'eXekles' => 1655, 'tar taX' => -1656, 'rkaXin ' => -1657,
    'et aXar' => 1658, 'et Xile' => 1659, 'erda X' => 1660,
    ' cIk X' => -1661, 'eroviX' => -1662, 'Xelep ' => -1663,
    'Xhisar' => 1664, ' reXin' => 1665, ' aXlan' => -1666,
    'aXimiy' => -1667, 'eXeniz' => -1668, 'gUlXe ' => -1669,
    '  aXim' => -1670, 'nda X ' => 1671, ' aXiyd' => -1672,
    ' Xole ' => 1673, 'aXisiz' => -1674, 'i Xars' => 1675,
    'Xarnac' => 1676, 'Xoldi' => -1677, 'rIXak' => -1678,
    'Xopei' => -1679, 'Xamol' => 1680, 'araXc' => 1681,
    'kvenX' => -1682, 'indaX' => -1683, 'Xopen' => -1684,
    'Xarpe' => -1685, 'Xalai' => -1686, 'Xarso' => -1687,
    ' mXp ' => 1688, 'ikeXi' => -1689, 'Xasla' => 1690,
    'aXide' => -1691, 'hiXan' => 1692, 'palaX' => -1693,
    'Xomel' => 1694, ' Xeri' => -1695, ' eliX' => 1696,
    'zIl X' => 1697, 'IlIXh' => 1698, 'balXa' => -1699,
    'Xibal' => -1700, ' tUrX' => 1701, 'harXi' => -1702,
    'Xaris' => -1703, ' olaX' => -1704, 'Xobh' => -1705,
    'iXun' => 1706, ' iXd' => 1707, 'guXi' => 1708,
    'Xisu' => -1709, 'InXo' => 1710, 'phiX' => -1711,
    'naXl' => 1712, 'Xalp' => -1713, 'Xcam' => 1714,
    'Xevv' => -1715, 'IXer' => 1716, 'Xav ' => -1717,
    ' Xug' => 1718, 'gsiX' => -1719, 'yIrX' => -1720,
    'Xizz' => -1721, 'rdiX' => -1722, 'Xerp' => -1723,
    'mliX' => -1724, 'obiX' => -1725, 'miXi' => -1726,
    'Xat ' => -1727, 'lXy' => -1728, 'nXd' => 1729,
    'Xkn' => -1730, 'ouX' => -1731, 'iCX' => 1732,
    'tIX' => -1733, 'Xhe' => -1734, 'Xez' => -1735,
    'k aXisini' => -1736, 'aXikmist' => -1737,
    'cili biX' => -1738, 'lI haX ' => 1739,
    'r aXar ' => 1740, 'enoviX' => -1741, 'oger X' => -1742,
    ' anaX ' => 1743, 'haXin ' => 1744, ' erXeg' => -1745,
    'slamiX' => -1746, 'Xarli ' => -1747, '  Xars' => 1748,
    'eniXag' => 1749, 'a Xars' => 1750, 'Xerny' => -1751,
    'Xalte' => -1752, 'lerX ' => -1753, 'l Xim' => 1754,
    ' Ximi' => -1755, 'ukXev' => -1756, 'kolXa' => 1757,
    'doviX' => -1758, 'illaX' => -1759, ' aXas' => 1760,
    'Ximis' => -1761, 'CinXi' => 1762, ' aXiv' => 1763,
    'Xubur' => -1764, 'rnaXa' => 1765, ' Xaci' => -1766,
    'Xilde' => -1767, 'Xilas' => -1768, 'Xagiz' => -1769,
    ' CerX' => 1770, 'ukoX' => -1771, 'Xac ' => -1772,
    'eiXe' => -1773, 'Xoge' => -1774, 'Xirc' => -1775,
    'kleX' => 1776, 'Xapc' => -1777, 'dXi ' => 1778,
    'asXi' => 1779, 'Xorc' => -1780, 'Xtiv' => -1781,
    'bilX' => -1782, 'Xess' => -1783, 'Xaat' => -1784,
    'Xuko' => 1785, 'chIX' => -1786, ' OnX' => -1787,
    'UsX' => 1788, 'Xyp' => -1789, 'vIX' => -1790,
    'CoX' => -1791, 'zIX' => -1792, '1X' => -1793,
    ' Xeltik' => 1794, 'zIl haX' => 1795, 'pantiX' => 1796,
    'Xapaci' => -1797, 'arIXam' => 1798, ' Xeles' => -1799,
    ' Xanag' => 1800, ' Xire' => -1801, 'balXe' => -1802,
    'aliXo' => 1803, ' Xun ' => 1804, ' arX ' => -1805,
    'Xeka ' => -1806, ' iXac' => 1807, 'Xayim' => 1808,
    ' Xery' => -1809, 'jeviX' => -1810, ' Xise' => 1811,
    'naXiz' => 1812, 'oniXe' => 1813, 'Xatta' => -1814,
    'dansX' => 1815, 'Xivr' => 1816, 'Xeca' => -1817,
    'Xapp' => -1818, ' Xod' => -1819, ' iXn' => 1820,
    'skeX' => 1821, 'lXat' => -1822, 'Xori' => -1823,
    'Xabe' => -1824, 'kXam' => 1825, 'itaX' => -1826,
    'uCiX' => 1827, 'Xelt' => -1828, 'psaX' => 1829,
    'Xurp' => 1830, 'terX' => -1831, 'Xau' => -1832,
    'Xof' => -1833, ' Xf' => -1834, 'eeX' => -1835,
    'Xph' => -1836, 'alil Xi' => -1837, ' penXe ' => 1838,
    'adi Xam' => 1839, ' Xortu' => 1840, 'saraXh' => 1841,
    'isi X ' => 1842, 'haXan ' => 1843, ' kIlXi' => 1844,
    ' Xelin' => -1845, ' Xular' => -1846, 'eXeyim' => 1847,
    'nUlaX' => 1848, ' Xed ' => 1849, 'Xomez' => 1850,
    'f Xim' => 1851, 'buXug' => 1852, 'Xerez' => 1853,
    'ykelX' => -1854, 'Xaka ' => -1855, 'Xiviy' => 1856,
    ' Xabi' => -1857, 'aXiki' => -1858, 'Xikan' => 1859,
    'pXan' => -1860, 'UXur' => -1861, 'iiXi' => 1862,
    ' Xhd' => 1863, ' UXd' => 1864, 'atXu' => -1865,
    'aXte' => -1866, 'plaX' => -1867, 'Xcik' => 1868,
    'Xkov' => -1869, 'rpiX' => 1870, 'taXm' => -1871,
    'Xubi' => -1872, 'luXi' => -1873, 'yiX ' => 1874,
    'ogiX' => -1875, 'Xift' => 1876, ' viX' => -1877,
    ' pX' => -1878, ' rX' => -1879, 'wiX' => -1880,
    'oiX' => -1881, 'aeX' => -1882, 'Xj' => -1883,
    ' aXilarindan' => 1884, 'n aXisiy' => -1885,
    'dinXel ' => 1886, ' deneX ' => 1887, 't Xine' => 1888,
    ' Xenel' => 1889, ' Xaril' => -1890, ' Xado' => 1891,
    'parX ' => -1892, 'mXeli' => 1893, 'ick X' => -1894,
    'Xalde' => -1895, 'joviX' => -1896, ' aXir' => -1897,
    'gUXer' => 1898, 'Xengi' => -1899, 'Xepn' => 1900,
    'Xevz' => -1901, 'imiX' => -1902, 'IrXe' => -1903,
    'Xelp' => -1904, 'Xilk' => -1905, 'Xock' => -1906,
    'SXan' => -1907, 'Xiz ' => -1908, 'kiXe' => -1909,
    'nkaX' => -1910, 'galX' => -1911, ' iXe' => 1912,
    'hXr' => -1913, 'brX' => -1914, 'Xkm' => -1915,
    'huX' => -1916, 'loX' => -1917, 'jiX' => -1918,
    'Xalgar' => -1919, 'Xalabr' => -1920, ' Xigim' => -1921,
    'Xaptag' => -1922, ' Xinin' => -1923, ' Xecil' => -1924,
    'us Xar' => 1925, 'cem Xo' => 1926, 'Xamli ' => -1927,
    'holiX' => -1928, 'aXay ' => 1929, 'Xizir' => -1930,
    ' Xary' => -1931, 'ceviX' => -1932, ' troX' => 1933,
    'Xomle' => 1934, 'OnetX' => -1935, 'Xasod' => 1936,
    'nXev ' => 1937, 'larXa' => -1938, 'Xaly' => -1939,
    'SXen' => -1940, 'uXer' => -1941, 'Xlai' => -1942,
    'varX' => -1943, 'uXe ' => -1944, ' unX' => -1945,
    'veXi' => -1946, 'Xevr' => 1947, 'akaX' => -1948,
    'ebX' => -1949, 'Xkv' => -1950, ' tX' => -1951,
    'yuX' => -1952, ' de Xabas' => -1953, ' aXind' => -1954,
    'iXisi ' => -1955, 'u aXar' => 1956, 'jukiX' => -1957,
    'civiX' => -1958, 'erXam' => 1959, 'Xupi ' => 1960,
    'CariX' => 1961, 'deriX' => -1962, ' arXa' => -1963,
    ' orXu' => 1964, 'edriX' => -1965, 'elXa' => -1966,
    'imaX' => -1967, 'mpaX' => -1968, 'vilX' => -1969,
    'buXl' => 1970, 'yXan' => -1971, 'lnX' => -1972,
    'ieX' => -1973, 'iaX' => -1974, 'Xaf' => -1975,
    'Xoe' => -1976, ' uXla ' => -1977, 'GanXay' => 1978,
    'lIXasi' => -1979, 'Xarmin' => -1980, 'a aXar' => 1981,
    'e Xars' => 1982, 'bortaX' => 1983, 'Xollec' => -1984,
    'sIXan' => 1985, 'laXik' => -1986, 'ulusX' => 1987,
    ' Xak ' => -1988, 'selXi' => -1989, 'Xenko' => 1990,
    'Xapri' => -1991, ' peXe' => 1992, 'biliX' => -1993,
    'falX' => -1994, 'diX ' => 1995, 'siXi' => -1996,
    'speX' => -1997, ' Xt' => -1998, 'Xox' => -1999,
    'Xq' => -2000, 'Xandar ' => 2001, 'penXel' => 2002,
    'kemenX' => 2003, 'I Xars' => 2004, 'eXemiz' => -2005,
    'Xinto' => -2006, ' uXa ' => -2007, 'Xariy' => -2008,
    ' Xibo' => -2009, ' Xiga' => -2010, 't Xi ' => 2011,
    ' Xio' => -2012, ' Xea' => -2013, 'kaXc' => 2014,
    'Xlav' => -2015, 'Xepe' => 2016, 'Xkt' => -2017,
    'kX ' => -2018, ' IX' => -2019, 'Xyi' => 2020,
    ' oX' => -2021, 'xX' => -2022, 'ike Xan' => 2023,
    'Xingir' => 2024, 'Xiliye' => -2025, 'inalX' => -2026,
    'Xapon' => -2027, 'leiXi' => 2028, ' Xaym' => -2029,
    's Xop' => -2030, 'Xivis' => 2031, 'Ximbi' => -2032,
    ' CaXa' => 2033, ' sIrX' => 2034, 'inciX' => 2035,
    'IrmaX' => -2036, 'Xobr' => -2037, 'pOrX' => 2038,
    'nXko' => 2039, 'Xelb' => -2041, 'omiX' => -2042,
    ' Xorun' => -2043, ' aXina' => -2044, 'n Xars' => 2045,
    ' Xory' => -2046, 'Xocu ' => -2047, 'anrIX' => 2048,
    'metiX' => 2049, 'd inX' => -2050, ' alXi' => 2051,
    ' Xink' => 2052, 'IlhaX' => 2053, 'aXev' => 2054,
    'roXi' => 2055, 'lXio' => -2056, 'abiX' => -2057,
    'Xamk' => 2058, 'CXi ' => 2059, 'eXiv' => 2060,
    'Xarn' => -2061, 'I aXar' => 2062, ' Xene ' => 2063,
    'Xarey ' => -2064, 'felXi' => -2065, 'UteX ' => 2066,
    'ssiX' => -2067, 'Xort' => -2068, 'Xabb' => -2069,
    'niXo' => -2070, ' anX' => -2071, 'reXci' => 2072,
    'maXik' => -2073, ' uXuc' => 2074, 'Xince' => 2075,
    'Xips' => -2076, 'Xors' => -2077, 'Xur ' => 2078,
    'tliX' => -2079, 'mUX' => -2080, 'Xohacopu' => 2081,
    ' Xarda' => 2082, ' aXini' => -2083, 'SaXik ' => -2084,
    'Xayag' => -2085, ' Xara' => -2086, ' koXu' => 2087,
    'Xaret' => -2088, 'agiX' => -2089, ' Xeo' => -2090,
    'yaXo' => 2091, 'Xalv' => -2092, 'Xice' => 2093,
    'SiX' => -2094, 'noX' => -2095, ' da Xabas' => -2096,
    ' antiX ' => 2097, ' Xuruml' => -2098, 'Xitib' => -2099,
    'blanX' => -2100, 'Xerva' => -2101, 'bliX' => -2102,
    'CiXi' => -2103, 'aaX' => -2104, 'Xagalo' => -2105,
    ' paXi' => -2106, 'viXe ' => -2107, ' aliX' => -2108,
    'peX ' => -2109, 'Ximr' => -2110, 'Xeho' => 2111,
    'blaX' => -2112, 'SmaX' => -2113, 'oXi ' => 2114,
    'UtX' => 2115, 'Xakarta ' => -2116, ' Xiliv' => -2117,
    ' Xarli' => 2118, 'eXinde' => -2119, 'UrgeX' => -2120,
    ' Xot' => -2121, 'Xitr' => -2122, 'maXb' => -2123,
    'Xera' => -2124, 'alaXat' => 2125, ' reXel' => 2126,
    'Xabr' => -2127, 'Xid ' => -2128, 'Xoro' => -2129,
    ' Xiler' => -2130, ' parXa' => 2131, 'kalXa' => 2132,
    'iXan' => -2133, 'upaX' => -2134, 'nXb' => 2135,
    'wX' => -2136, 'Xv' => -2137, ' Xici' => -2138,
    'Xooo' => 2139, 'Xivile' => 2141, ' saXi ' => 2142,
    'siXe' => -2143, 'paXk' => -2144, 'fIrX' => 2145,
    'jX' => -2146, ' aXilarda' => 2147, 'ehiriX' => 2148,
    'kIrXa' => -2149, 'Xarg' => -2150, 'aXle' => -2151,
    'enXk' => 2152, 'ruXa' => -2153, 'isXi' => 2154,
    'zOlX' => 2155, ' Xp' => -2156, 'nXc' => 2157,
    'Xkl' => -2158, 'dX' => -2159, 'karaXi ' => 2160,
    ' Xildin' => -2161, ' serXe' => 2162,
    'Xolleg' => -2163, 'Xomak' => 2164, ' aXal' => 2165,
    'rusX' => 2166, 'aXib' => -2167, 'emiX' => -2168,
    'lXo' => -2169, 'Xyd' => 2170, ' Xy' => -2171,
    ' burXa' => 2172, ' maXo' => 2173, 'Xuha' => 2174,
    'ciX' => -2175, 'roX' => -2176, 'Xengel' => 2177,
    'musiX ' => -2178, 'Xanko' => -2179, ' Xini' => 2180,
    'suXuk' => -2181, 'Xoli' => -2182, ' aXiyl' => -2183,
    'gOXek' => -2184, 'Xals ' => -2185, 'Xubay' => 2186,
    'rayiX' => 2187, 'manXo' => 2188, 'irkaX' => 2189,
    'tehX' => -2190, 'feX' => -2191, 'foXa ' => 2192,
    'raXog' => -2193, 'sXis' => 2194, 'buX ' => 2195,
    'kXem' => 2196, 'Xebi ' => 2197, 'Xorr' => -2198,
    'Ximn' => -2199, 'triX' => -2200, ' Xurum ' => -2201,
    ' Xildi ' => -2202, 'salXa' => 2203, ' kaXi' => 2204,
    'eliX' => -2205, 'uXup' => 2206, 'fiX' => -2207,
    'koXan ' => 2208, 'Xolum' => -2209, 'Xonga' => 2210,
    'Xavc' => -2211, ' Xed' => -2212, 'skaXi' => -2213,
    ' ilX' => 2214, 'kiXi' => -2215, 'Xef' => -2216,
    'ceX' => -2217, 'IsaXi' => -2218, ' Xeyi' => 2219,
    'eXhiz' => 2220, 'sXe ' => 2221, 'fIX' => 2222,
    'yol aX' => 2223, 'Xelim' => 2224, 'Xenes' => 2225,
    'Xipla' => 2226, 'Xath' => -2227, 'nXoi' => 2228,
    'koXas ' => 2229, ' Xinle' => -2230, 'Xapit' => -2231,
    'UleX ' => 2232, 'IXd' => 2233, 'faX' => -2234,
    'zXeli' => 2235, ' luX' => -2236, 'Xikli ' => -2237,
    'iXeri' => 2238, 'Xolli' => -2240, 'Xels' => -2241,
    ' Xe ' => -2242, 'eseX' => -2243, 'naX' => -2244,
    ' aXidi' => -2245, 'Xorn' => -2246, 'Xorp' => -2247,
    ' Xayir ' => -2248, 'Xalla' => -2249, 'farsX' => 2250,
    'iniX' => -2251, 'Xehr' => 2252, ' eX' => -2253,
    'Xilarak' => 2254, 'tekelX' => -2255, 'eXilik' => -2256,
    'Xarme' => -2257, 'konXe' => 2258, ' Xis' => -2259,
    'nXm' => 2260, 'Xove' => -2261, ' kamX' => 2262,
    'Xirit' => -2263, ' saXa' => 2265, 'Xilv' => -2266,
    'Xava' => -2267, 'rUXh' => 2268, 'Xai' => -2269,
    ' yolaX' => 2270, 'Xole' => -2271, 'Xt ' => -2272,
    'caX' => -2273, 'aXiya' => -2274, 'bruX' => -2275,
    ' kIX' => 2276, ' Xs' => -2277, 'Xurcic' => 2278,
    'CurX' => 2279, 'Xky ' => -2280, 'Xiliz' => -2281,
    'Xanku' => -2282, 'Xamli' => 2283, 'Xalt' => 2284,
    ' Xivi ' => 2285, 'Xie' => -2286, 'penXes' => 2287,
    ' ayX' => 2288, 'aXiyi' => -2289, 'Xoca' => -2290,
    'koXak ' => 2291, ' Xul' => 2293, 'Xape' => -2294,
    'Xetv' => -2295, 'aXli' => 2296, ' Xelil' => -2297,
    'gaXi' => -2298, 'IoX' => -2299, 'S aXi' => 2300,
    'aXisi ' => -2301, 'anXer' => 2302, 'rXil ' => 2303,
    ' seXik' => 2304, 'piX' => -2305, 'aXit' => -2306,
    'gX' => -2307, ' paXa' => 2308, 'verX' => -2309,
    'rXin ' => 2310, ' Xila' => -2311, 'Xinge' => 2312,
    'Xubu' => 2313, 'Xity' => -2314, 'Xuyo' => 2315,
    'IXe' => -2316, 'koXar' => 2317, 'sXila' => 2318,
    ' liX' => -2320, 'eveX' => -2321, ' Xarsi ' => -2322,
    'Xeviz' => -2323, 'daXi' => -2324, 'Xolo' => -2325,
    'Xopla' => -2326, 'Xtio' => -2327, 'Xs ' => -2328,
    'Xalif' => -2329, 'akarX' => -2330, 'deX' => -2331,
    'aXili ' => -2332, 'Xola ' => -2333, 'telX' => -2334,
    'ortaXag' => 2335, 'lIXak' => -2336, 'Xizr' => -2337,
    'Xoa' => -2338, 'aXam' => 2339, 'IlX' => -2340,
    'niX ' => -2341, 'Xy ' => -2342, 'baliX' => -2343,
    'riXe' => -2344, ' Xari ' => -2345, ' aXiz' => -2346,
    'n aXisini' => -2347, 'Xart' => -2348, 'CeXi' => -2349,
    'fX ' => -2350, 'pX ' => -2351, ' aXar ' => -2352,
    ' Xevd' => -2353, 'sXi ' => 2354, 'beX' => -2355,
    'rXasina' => -2356, 'Xoh' => -2357, ' veX' => -2358,
    'elXe' => -2359, 'uXat' => -2360, ' uXa' => 2361,
    'Xaro' => -2362, 'aXile' => -2363, ' Xere' => -2364,
    ' biX' => 2365, 'Xarr' => -2367, 'luXa' => -2368,
    'alXil' => -2369, 'olX' => -2370, 'Xinsel' => -2371,
    ' Xard' => -2372, 'nXs' => 2373, 'fX' => 2374,
    'Xord' => -2375, 'baXi' => -2376, 'inXer' => 2377,
    ' seXe' => 2378, 'vaXi' => -2379, ' Xm' => -2380,
    'Xavi' => -2382, ' Xeket' => -2383, 'rXes' => -2384,
    'zliX' => -2385, 'Xell' => -2386, 'perinX' => 2387,
    'Xembe' => 2388, ' eriX' => -2389, 'ilaXi' => -2390,
    'bOX' => -2391, ' UXe' => 2392, 'barX' => -2393,
    'Xuv' => 2394, 'Xilt' => -2395, 'Xige' => -2396,
    'diX' => -2397, 'uXur' => 2398, ' Xaydi' => -2399,
    ' Xumr' => 2400, ' aXar' => 2401, 'eXete' => 2402,
    'Xkh' => -2403, 'Xate' => -2404, 'Xevh' => -2405,
    ' Xim ' => -2406, 'GaXi' => -2407, 'urtiX' => 2408,
    'Xn' => -2409, ' Xoo' => -2410, 'urXa' => -2411,
    'GXi' => -2412, 'iraX' => -2413, 'OXa' => -2414,
    'gOX' => 2415, 'bilanX' => 2416, 'hX' => 2417,
    ' Xadir' => 2418, 'Xou' => -2419, 'riXi' => -2420,
    'meXhu' => 2421, 'bX ' => -2422, 'Xke' => -2423,
    'yeX' => -2424, 'tXa ' => 2425, 'aziX' => 2427,
    'Xerr' => -2428, ' Xarl' => -2429, 'yX' => -2430,
    ' Xamas' => 2431, 'eXenin ' => -2432, 'eXem' => 2433,
    'IXal' => -2435, 'eXebi' => 2436, 'harXa' => -2437,
    ' reX' => -2438, 'kXan' => -2439, 'sIXr' => 2440,
    'Xinl' => 2441, 'Xamur' => 2442, 'aXima' => -2443,
    'Xo ' => -2444, 'ikaXi' => -2445, 'aXabi' => 2446,
    ' Xesu' => -2447, 'Xiro' => -2448, 'nXt' => 2449,
    'Xeh' => -2450, 'Xanta' => 2451, 'aXilig' => -2452,
    'Xeyr' => 2453, 'meXi' => -2454, 'jaX' => -2455,
    'Xto' => -2456, 'Xogr' => -2457, 'niXe' => -2458,
    ' maXa ' => 2459, ' gerX' => 2460, 'tXu' => 2461,
    ' merX' => -2462, ' Xela' => -2463, 'uXuk' => 2464,
    ' Xese' => -2465, 'taXi' => -2466, ' Xanak' => 2467,
    'Xon' => -2468, 'cX' => -2469, 'Xc' => -2471,
    ' uXus' => 2472, 'aXil ' => -2473, 'zaX' => -2474,
    'Xuku' => 2475, 'leX' => -2476, ' Xeb' => -2477,
    'parX' => 2478, 'kuXa' => -2479, 'Xuru' => 2481,
    'aXilik' => -2482, ' Xey' => -2483, 'iXa' => -2484,
    'yaXi' => -2485, ' Xas' => -2487, ' Xank' => 2488,
    'elX' => 2489, 'buX' => -2490, 'Xom' => -2491,
    'iXil' => -2492, 'aXac' => 2493, ' X ' => -2494,
    ' aXan' => 2495, 'pX' => 2496, 'haX' => -2498,
    'Xih' => -2499, 'Xia' => -2500, 'sviX' => 2501,
    'Xos' => -2502, 'Xaz' => -2503, 'eXik' => -2504,
    'eXec' => 2505, 'neX' => -2506, 'Xiv' => -2507,
    'suX' => 2508, 'Xz' => -2509, 'tX ' => -2510,
    'Xd' => -2511, ' maXi' => 2512, ' Xin ' => 2513,
    'sX' => -2514, 'UXe' => -2515, 'zX' => -2516,
    'Xep' => -2517, ' Xin' => -2518, 'reXi' => -2519,
    'sIX' => -2520, 'yiX' => -2521, 'aXe' => -2522,
    'SX' => 2523, 'aXilar' => -2524, ' Xeva' => -2525,
    ' Xl' => -2526, 'raXi' => -2527, 'geXer' => 2528,
    ' OlX' => 2529, ' kaX' => 2530, 'ilXi' => -2531,
    ' UXu' => 2532, ' Xen' => -2533, 'vX' => -2534,
    'IXi' => -2535, 'Xem' => -2536, 'rXe ' => -2538,
    'amaXi' => -2539, ' Xid' => -2540, 'tXi' => 2541,
    'teX' => -2542, 'Xam' => -2543, 'rXi' => -2544,
    'Xunk' => 2545, 'Xr' => -2546, 'nXl' => 2547,
    'Xad' => -2548, ' kUX' => 2549, 'kX' => 2550,
    'mX' => -2551, 'oXa' => -2552, ' meX' => -2553,
    'eXen' => 2554, 'Xan' => -2556, 'nX ' => 2557,
    'tiX' => -2558, 'Xa ' => -2559, 'Xh' => -2560,
    'Xi ' => -2561, 'Xu' => -2562, 'eXe' => -2563,
    'aXa' => -2564, 'nX' => -2565, 'X' => 2566,
    },
    's' => {
    'kut Xe' => 1, 'a nurX' => 2, 'lIXiyi' => -3, 'Xleriz' => -4, 'raXama' => 5,
    'de aX ' => -6, 'vard X' => 7, 'irmeX ' => 8, 'gin Xo' => -9, 'k miX ' => 10,
    're fiX' => 11, 'IXici ' => -12, 'oni Xe' => 13, 'y yaXt' => -14, 'r gUlX' => -15,
    'kafi X' => -16, 'UC Xu ' => -17, 'mum iX' => -18, 'i miX ' => 19, 'ce niX' => 20,
    'lal iX' => 21, ' diXid' => 22, 'Xalom ' => 23, 'a aXi ' => 24, 'a Xikl' => -25,
    'lIlIXi' => -26, 'ce koX' => 27, 'Xerket' => 28, 'Xayand' => 29, 'StU Xu' => -30,
    'GuXuyl' => -31, 'hide X' => -32, 'Xtemil' => 33, 'i Xovd' => 34, 'lIr Xu' => -35,
    ' naXit' => 36, 'e neXi' => -37, 'Xantar' => -38, 'at Xai' => -39, 'imi IX' => -40,
    'nI Xea' => 41, 'a k Xa' => -42, 'okum X' => -43, 'Sa Xok' => -44, 'bi IXi' => 45,
    'nU aXi' => 46, 'puk iX' => 47, 'tara X' => -48, ' hum X' => 49, 'oli Xa' => 50,
    '83 baX' => -51, 'nU Xer' => -52, 'je baX' => 53, 'nU eX ' => -54, 'k Xofo' => 55,
    'e Xak ' => 56, ' i Xad' => -57, 'e IXit' => -58, 'ir tuX' => 59, 'vgat X' => -60,
    'day Xu' => -61, 'nIXam ' => -62, 'IXitir' => 63, 'Uk Xut' => 64, 'or eXs' => 65,
    'yvan X' => -66, 's miX ' => 67, 'ul beX' => 68, 'yaXlad' => -69, 'rdu Xo' => -70,
    'a aXiy' => -71, 'zUn Xi' => -72, 'ila iX' => -73, 'canlIX' => -74, 'mlayIX' => 75,
    ' u yaX' => 76, 'a poXu' => 77, ' dilXe' => 78, 'hurdaX' => -79, 'le duX' => -80,
    'vefa X' => -81, 'r Xer ' => -82, 'n li X' => -83, 'e ma X' => 84, 'oluXa ' => -85,
    '55 yaX' => 86, 'rtarIX' => 87, 'lda Xu' => -88, 'Xkisis' => 89, 'a CalX' => 90,
    ' aGrIX' => -91, 'saGI X' => -92, 'yalleX' => 93, 'ma eXk' => -94, 'leXem ' => -95,
    'an leX' => 96, 'deXtil' => 97, 'kse Xu' => -98, 'saklaX' => -99, 'lafa X' => -100,
    'is Xad' => 101, 'ta a X' => 102, 'beci X' => -103, 'me ayX' => -104, 'a neXi' => -105,
    'git Xo' => -106, 'em eXk' => -107, 'Xahid ' => 108, 'tip aX' => -109, 'UktU X' => 110,
    'hak Xe' => 111, 'tfak X' => -112, 'zade X' => -113, 'It beX' => 114, 't Xims' => 115,
    'GI aXt' => -116, 'e aXi ' => 117, 'SlI aX' => -118, 'i Xeke' => 119, 'Gi paX' => -120,
    'nlIXid' => -121, 'In Xex' => 122, 'l beXl' => -123, 'ter Xo' => -124, 'e aXiy' => -125,
    'vi yaX' => -126, 'tu baX' => 127, ' Xive ' => 128, 'lina X' => -129, 'ruGu X' => -130,
    'a iXli' => 131, 'paXada' => 132, 'pek aX' => 133, 'sOz Xe' => -134, ' iXvel' => 135,
    'oklaXi' => -136, '  gUlX' => -137, 'ah Xim' => 138, 'ydar X' => -139, 'inan X' => -140,
    'kIXtan' => 141, 'sIk Xa' => -142, 'yanIXa' => -143, 'ak aXt' => -144, 'erdeX ' => 145,
    'hta aX' => -146, ' Xadir' => 147, 'ynir X' => -148, 'r aXa ' => -149, 'kCi Xu' => -150,
    'ir a X' => -151, 'Cim Xu' => -152, 'udum X' => -153, 'kana X' => -154, 'r kloX' => 155,
    'Sah Xe' => 156, 'ut aXk' => -157, 'k e Xo' => -158, 'um a X' => -159, 'mu muX' => 160,
    'boyXa ' => 161, 'z beXl' => -162, 'I 11 X' => 163, 'dur Xo' => -164, 'Xoveni' => 165,
    'z taXa' => -166, 'Xenger' => -167, 'o kurX' => -168, 'tken X' => -169, 'ulayIX' => 170,
    'onla X' => -171, 'o karX' => 172, 'Iz akX' => -173, 'Xlesti' => 174, ' tarXi' => 175,
    'Xismik' => -176, 'boyu X' => -177, 'Xemaya' => 178, 'cudu X' => -179, 'Sam Xu' => -180,
    ' ramuX' => 181, ' han X' => -182, 'cU kiX' => 183, 'ya aXt' => -184, 'nlUk X' => -185,
    'poda X' => -186, 'iC akX' => -187, 'Xohben' => 188, 'sI Xok' => -189, 'nlaXiy' => 190,
    'ski Xu' => -191, 'yaXlay' => -192, ' oy iX' => -193, 'ygulaX' => -194, 'Ximlik' => -195,
    ' al Xa' => -196, 'rISI X' => -197, 'IrI Xu' => -198, 'sOz Xo' => -199, 'r Xek ' => 200,
    'ce kIX' => -201, 'Xansan' => -202, 'itiriX' => 203, 'aXimiy' => 204, 'kIXta ' => 205,
    'niXant' => 206, 'SI Xey' => -207, 'tlayIX' => 208, 'yol Xe' => -209, 'tI paX' => -210,
    'CUk Xu' => -211, ' ay Xu' => -212, 'Cine X' => -213, 'se keX' => -214, 'a birX' => 215,
    'di Xis' => -216, 'akarIX' => 217, 'U manX' => 218, 'en Xeh' => 219, 'yuya X' => -220,
    'ltI Xu' => -221, 'e iXka' => -222, 'enek X' => -223, 'ldI Xu' => -224, 'eriXe ' => 225,
    ' Xinci' => 226, 'gI yaX' => -227, 'IXimiz' => 228, 'n Xand' => -229, 'keleX ' => 230,
    'yler X' => -231, 'garXi ' => 232, 'adaS X' => -233, 'ski Xo' => -234, 't i Xa' => -235,
    'zgeliX' => 236, 'rUXen ' => 237, 'rdI Xu' => -238, 'en Xat' => -239, ' eniXt' => 240,
    'eraniX' => 241, 'dUn Xa' => -242, 'e haXa' => -243, 'kanIXi' => -244, 'zeni X' => -245,
    'aSI Xi' => -246, ' yaXiy' => 247, 'dUXtu ' => 248, 'cu yaX' => 249, 'amaXti' => 250,
    ' kamaX' => 251, 'vfik X' => -252, 'Xutunl' => -253, 'olaXi ' => -254, 'rizi X' => -255,
    ' en eX' => -256, 'zyIl X' => -257, 'er Xin' => -258, ' aXiri' => 259, 'si Xin' => -260,
    'ne aXt' => -261, 'yaklaX' => 262, 'l keXi' => -263, 'Un kOX' => 264, 'sayIXi' => -265,
    'IXtir ' => 266, 'ISa Xu' => -267, 'laXtir' => 268, 'rtti X' => -269, 'gOrUX ' => 270,
    '  iXti' => -271, ' geliX' => 272, 'oXunu ' => -273, 'nma Xu' => -274, 'Xiddd' => 275,
    's eX ' => -276, 'Xisik' => 277, 'embaX' => -278, ' bieX' => 279, 'coXme' => -280,
    'CIrIX' => 281, 'mIXiy' => 282, 'yeXar' => -283, 'UrneX' => 284, 'alXim' => 285,
    'kUbaX' => -286, 'dataX' => -287, 'valIX' => -288, 'eXraf' => 289, 'Xovul' => -290,
    ' raX ' => -291, 'laSIX' => 292, ' eziX' => 293, 'aiXet' => 294, '6 taX' => -295,
    'Xanka' => 296, 'kuXac' => -297, 'taXna' => 298, 'oz eX' => -299, 'arkeX' => 300,
    'Xuyur' => 301, 'Xtenk' => -302, 'mUXev' => 303, 'imlaX' => -304, 'kavaX' => -305, 'niXim' => 306,
    ' Xapo' => 307, 'IpreX' => -308, 'aXisl' => 309, 'yuXan' => -310, 'Xerhl' => 311,
    'lIyIX' => 312, 'dUlXa' => 313, 'Xehay' => -314, 'Xengi' => -315, 'Xarkk' => 316,
    'leXle' => 317, 'aXrap' => 318, 'daXki' => 319, 'meXk ' => 320, 'bIXev' => 321,
    'ubeX ' => 322, ' Xakc' => 323, 'aXaro' => 324, 'raXer' => -325, 'aXere' => 326,
    'baXs ' => 327, 'nutuX' => 328, 'Xinta' => 329, 'Xeha ' => -330, ' iXev' => 331,
    'IG Xe' => -332, 'yaSIX' => 333, 'uriX ' => 334, 'kiyiX' => 335, ' OtUX' => 336,
    'taXab' => 337, 'Xalk ' => 338, 'Xefaf' => 339, 'uXin ' => 340, 'va aX' => -341,
    'mIyIX' => -342, 'bIrIX' => -343, 'td Xt' => 344, 'yh Xu' => -345, 'll Xo' => 346,
    'iriXk' => 347, 'ldaXa' => -348, 'inevX' => 349, 'n CaX' => 350, 'piXsi' => 351,
    'Xkodr' => 352, 'nIXal' => 353, 'koz X' => -354, 'luguX' => 355, 'Xansk' => -356,
    'S iXh' => 357, 'tiXa ' => 358, 'menaX' => 359, 'zabaX' => 360, 'SavaX' => 361,
    'Xakse' => 362, 'Ximov' => -363, 'aXmim' => 364, 'SIyIX' => 365, 'deXip' => 366,
    'olm X' => -367, 'hasaX' => -368, 'miXel' => 369, 'Xenno' => 370, ' kouX' => 371,
    'betiX' => -372, '4 Xim' => 373, 'z duX' => 374, 'arXor' => 375, 'tetiX' => -376,
    ' guX ' => 377, 'ornoX' => 378, 'uke X' => -379, 'Xivir' => 380, 'teXik' => 381,
    'Xlahl' => 382, 'I Xei' => 383, 'Xehi ' => -384, 'malaX' => -385, 'Xitta' => 386,
    'nnuXi' => 387, 'haXem' => 388, 't Xe ' => 389, 'calaX' => -390, 'dediX' => 391,
    'uXtar' => 392, 'nXlas' => 393, 'suruX' => 394, 'CiriX' => 395, ' tUXu' => 396,
    'andeX' => -397, 'caroX' => 398, 'p Xut' => -399, 'Xahim' => 400, 'Xirfi' => 401,
    'aXalo' => 402, 'OnleX' => -403, ' m aX' => -404, 'i c X' => 405, 'Xongo' => 406,
    'i U X' => -407, 'Xuhed' => 408, 'Xifay' => 409, 'ee aX' => -410, '39 Xu' => -411,
    'siniX' => -412, 'uXtuy' => 413, 'he iX' => -414, ' uXun' => 415, 'zeniX' => 416,
    'c taX' => -417, 'guk X' => -418, 'Xirte' => 419, 'ralIX' => -420, 'SetaX' => 421,
    'zkeX ' => 422, 'SaXan' => 423, 'lalIX' => -424, 'hic X' => -425, 'mOnUX' => -426, 'y iXs' => 427,
    'It aX' => -428, 'ainiX' => -429, 'kroXo' => 430, ' ibiX' => 431, 'hlu X' => -432,
    'sagaX' => -433, 'ciniX' => -434, 'Xuyuu' => 435, 'h koX' => -436, 'kiXe ' => 437,
    'Xahik' => 438, 'jikiX' => 439, 'GmIXi' => -440, 'mafiX' => 441, 'canIX' => -442,
    'luXti' => 443, 'Xugun' => 444, 'mUnXi' => 445, ' dUyX' => 446, ' unuX' => -447,
    ' niXi' => 448, 'adeXl' => 449, 'eXart' => -450, ' iriX' => -451, 'Xtimu' => 452,
    '911 X' => -453, 'Xangu' => -454, 'IltaX' => 455, 'Xenor' => -456, 'Xakak' => 457,
    ' iXey' => 458, 'C ayX' => 459, 'gUriX' => 460, 'Xoyel' => 461, '75 Xe' => -462,
    'ok X ' => 463, 'beyiX' => -464, 'bariX' => 465, 'CiliX' => 466, 'y muX' => -467, 'dIraX' => -468,
    'Gu IX' => -469, 'yna X' => -470, 'eXebu' => 471, 'tad X' => 472, 'koniX' => 473,
    'Xkoli' => 474, 'op Xo' => -475, 'ng Xu' => -476, 'IXins' => 477, 'oXma ' => -478,
    ' lt X' => -479, 'susuX' => 480, 'neXeb' => -481, 'geraX' => 482, 'yaXma' => 483,
    'm I X' => -484, 'Xinto' => 485, 'eviX ' => -486, 'ss iX' => -487, 'ngaXa' => 488,
    'gap X' => -489, 'tis X' => -490, 'OrkiX' => 491, 'Xinsi' => -492, ' Xaze' => 493,
    'peniX' => -494, 'InIXa' => -495, 'luXab' => 496, 'Ge Xu' => -497, 'reb X' => 498,
    'neXir' => 499, 'lediX' => 500, 'braXo' => 501, 'zelXe' => 502, 'eXtal' => 503,
    'Xetip' => 504, 'Xaray' => -505, 'raXar' => 506, 'ismiX' => -507, 'r ifX' => 508,
    'Xafii' => 509, 'r Xud' => -510, 'geciX' => 511, 'Xaki ' => -512, 'estiX' => -513,
    'eS IX' => 514, 'kUXe ' => 515, 'h Xak' => -516, 'kIbaX' => -517, 'msi X' => -518,
    'halIX' => -519, 'ataXe' => 520, 'CiXe ' => 521, '4 Xer' => -522, 'ro Xu' => -523,
    'Xenez' => 524, 'kIzIX' => 525, '6 Xan' => -526, 'Xtuml' => -527, 'Xismo' => -528,
    '000 X' => -529, 'Xihiz' => 530, 'f Xat' => -531, 'leXek' => -532, 'taraX' => -533,
    'rXahi' => 534, ' leXe' => 535, 'rUlUX' => 536, 'dreXt' => -537, ' anuX' => 538,
    'kIt X' => -539, 'preXm' => 540, 'chi X' => -541, 'hut X' => -542, 'peXen' => -543,
    't peX' => 544, ' ahXa' => 545, 'tleXt' => 546, 'n ifX' => 547, 'opluX' => 548,
    'eXeyd' => -549, 'fayaX' => -550, 'tIXiz' => -551, 'e oXi' => 552, 'faXis' => 553,
    'uXurk' => 554, 'Ximir' => -555, 'rdeXl' => 556, 'ntraX' => -557, ' koXm' => 558,
    'aXhan' => 559, 'oluXl' => 560, 'zuluX' => 561, 'UnUX ' => 562, 'uyuX ' => 563,
    'our X' => -564, 'naXa ' => -565, 'uXurm' => 566, 'miXil' => -567, 'e miX' => -568,
    'e iXb' => 569, 'Xbir ' => -570, 'iXinl' => 571, 'bUcUX' => 572, 'rC iX' => -573,
    'gIdIX' => -574, 'reyXi' => 575, 'taXak' => 576, 'eSleX' => 577, 'vgi X' => -578,
    'kUrXa' => 579, 'rmiXl' => 580, 'rio X' => -581, 'laraX' => -582, 'Xabim' => -583,
    'azmIX' => 584, 'akXan' => -585, 'sokuX' => 586, 'zeXin' => -587, 'emiXl' => 588,
    'Xtili' => -589, 'reXat' => 590, ' 43 X' => -591, 'kiXan' => 592, 'babaX' => -593,
    '8 kiX' => 594, 'ev Xu' => -595, 'fiXek' => 596, 'Xumul' => 597, 'UnleX' => 598,
    'p aXt' => -599, 'ritaX' => -600, 'omboX' => 601, 'laXiz' => -602, 'Xagil' => 603,
    'eve X' => -604, 'boXa ' => 605, 'Xiga ' => 606, 'rXiya' => 607, 'n Xih' => -608,
    'CabaX' => -609, ' anaX' => -610, 'leXil' => 611, 'varoX' => 612, 'liXed' => -613,
    'rkIXi' => -614, 'd Xen' => -615, 'S peX' => 616, 'CeXin' => -617, 'iXmey' => 618,
    'nayaX' => -619, 'alliX' => -620, 'IXan ' => 621, 'miXe ' => 622, 'arIXa' => 623,
    'kiXis' => 624, 'tesiX' => -625, 'Xkent' => 626, 'riXim' => 627, 'yeXim' => 628,
    'erleX' => 629, 'sIraX' => -630, 'CalIX' => 631, 'Xubw' => -632, 'piXp' => 633,
    'saaX' => -634, 'ziXa' => 635, 'Xsaa' => 636, 'iXre' => 637, 'aiXn' => 638,
    'Xbar' => -639, 'nXah' => -640, 'Xirv' => 641, 'Xnor' => 642, 'yXec' => 643,
    'hU X' => -644, 'yIXm' => 645, 'UraX' => -646, 'eyaX' => -647, 'I Xb' => 648,
    'jlIX' => -649, 'Xehl' => -650, 'hoXs' => 651, 'Xkez' => 652, 'Xomi' => 653,
    'abXe' => 654, 'Xdig' => 655, 'Xiru' => 656, 'diXu' => 657, 'GdOX' => 658,
    'UXir' => 659, 'Xico' => -660, 'fiSX' => 661, 'Xyin' => 662, 'IXii' => 663,
    'crIX' => -664, 'uXey' => -665, 'OXer' => -666, 'Xhay' => 667, 'yruX' => 668,
    'thaX' => -669, 'Xitc' => -670, 'utiX' => 671, 'kahX' => -672, 'Xash' => -673,
    'Xfan' => -674, 'Xenu' => 675, 'beXd' => -676, 'Xenz' => 677, 'uyX ' => 678,
    'rpuX' => 679, 'Xug ' => 680, 'gXor' => 681, 'CkuX' => 682, 'Xcii' => -683,
    'pl X' => 684, 'SikX' => 685, 'Xuhp' => 686, 'riXn' => 687, ' pIX' => -688,
    'nuXe' => 689, 'Xasu' => -690, 'ayuX' => -691, 'uXba' => 692, 'ukIX' => -693,
    'IniX' => 694, 'ryuX' => 695, ' ilX' => 696, 'Xiat' => -697, 'Xpay' => -698,
    'Xkav' => 699, 'keXo' => -700, 'yyuX' => -701, 'osaX' => -702, 'Xivg' => 703,
    'saXs' => -704, 'Xulg' => 705, 'meXv' => 706, 'Xeyc' => -707, 'omIX' => -708,
    'egiX' => -709, 'SIXl' => 710, 'ISIX' => 711, 'ipaX' => 712, 'GlIX' => -713,
    'pIXl' => -714, 'aXgi' => 715, 'rhaX' => -716, 'zUXu' => 717, 'Ximt' => 718,
    'SapX' => 719, 'Xvar' => -720, 'OXek' => -721, 'eXum' => 722, 'uXuc' => 723,
    'daXd' => 724, 'gUX ' => 725, 'nuXi' => -726, 'aXce' => 727, 'ilXu' => 728,
    'Xapc' => 729, 'tyaX' => -730, 'tyuX' => 731, 'oXay' => 732, 'guaX' => 733,
    'eXih' => -734, 'lnaX' => -735, 'zkuX' => 736, 'yXab' => 737, 'Xemu' => 738,
    'Xife' => -739, 'Xaso' => -740, 'iXoz' => 741, 'speX' => -742, 'Xupa' => -743,
    'bliX' => -744, 'Xyi ' => 745, 'Xney' => -746, 'Xfek' => -747, 'm3 X' => -748,
    'eguX' => -749, 'hXar' => -750, 'vlaX' => -751, 'SviX' => 752, 'eXam' => -753,
    'oe X' => -754, ' yUX' => 755, 'Xni ' => 756, 'SuXl' => 757, 'vXam' => -758,
    'braX' => -759, 'tuXi' => 760, ' uXm' => -761, 'Xsay' => 762, 'mpuX' => -763, 'nnoX' => 764,
    'tuXk' => -765, 'ttuX' => 766, 'oXus' => 767, 'oXup' => 768, 'leaX' => -769,
    'Xiku' => -770, 'Ximp' => -771, 'Xihl' => 772, 'rraX' => -773, 'fuhX' => 774,
    'Xetb' => -775, 'smIX' => 776, 'aXpo' => -777, 'aXuc' => 778, 'ndUX' => -779,
    'IsiX' => 780, 'Xuuu' => 781, 'uguX' => -782, 'vmiX' => 783, 'Xtia' => -784,
    'Xeyn' => -785, ' uSX' => 786, 'uXuk' => 787, 'IbaX' => 788, 'SuX ' => 789,
    '77 X' => -790, 'eXge' => 791, 'pmiX' => 792, 'Xsi ' => -793, 'Xeda' => -794,
    'nmuX' => 795, 'kiXo' => 796, 'kmiX' => 797, 'zmiX' => 798, 'eXk ' => -799,
    'ryaX' => -800, 'hXat' => -801, 'fliX' => -802, 'zbaX' => 803, 'lmUX' => 804,
    'Xubi' => -805, 'Xset' => -806, 'iXab' => -807, 'UlaX' => -808, 'UnaX' => -809,
    'UXva' => -810, 'iraX' => -811, 'Xai ' => -812, 'Xohr' => 813, 'apuX' => -814,
    'oXic' => -815, 'eXus' => -816, 'Xaac' => -817, 'iXay' => -818, 'aXem' => -819,
    'uliX' => -820, 'Xeld' => -821, 'Xtis' => -822, 'graX' => -823, 'e Xt' => -824,
    'binX' => -825, 'Xess' => -826, 'Xuro' => 827, 'olXa' => -828, 'oXki' => -829,
    'rUXv' => 830, 'Xru ' => 831, 'uriX' => -832, 'rmUX' => 833, 'tCIX' => -834,
    'SmIX' => 835, 'eXas' => -836, 'hiXa' => -837, 'inaX' => -838, 'SmUX' => 839,
    'rgiX' => -840, ' riX' => -841, 'umXu' => -842, 'cUXu' => -843, 'akiX' => -844,
    'Xuri' => -845, 'ntIX' => -846, 'kaXm' => 847, 'kXek' => -848, 'tlIX' => -849,
    'Xizl' => -850, 'toXu' => -851, 'Xuzl' => -852, 'Xeci' => -853, 'lmaX' => -854,
    'OXi' => -855, 'aX7' => 856, 'bXl' => 857, 'uX1' => 858, 'mXl' => -859,
    'axX' => 860, 'Xtv' => -861, 'Xzm' => 862, 'Xny' => -863, 'Xlm' => -864,
    'Xsr' => -865, 'flX' => -866, 'fX ' => -867, 'Xbt' => -868, 'Xtj' => -869,
    'uXn' => 870, 'Xss' => -871, 'oIX' => -872, 'chX' => -873, 'Xty' => -874,
    'Xuo' => -875, ' kX' => -876, 'lXh' => -877, 'Xkc' => -878, 'IzX' => -879,
    'Xae' => -880, 'efX' => -881, 'utX' => -882, 'feX' => -883, 'nXu' => -884,
    ' Xk' => -885, 'Xaj' => -886, 't in baXinda ' => -887, 'sunda baXina' => -888, 'in aXilmasiy' => -889,
    'n ve baXini ' => 890, 'nlar yaXama ' => -891, 'vleri baXi' => 892, 'eri baXinda ' => -893, 'Ginde yaXama' => -894,
    'k onlarI iX' => 895, 'kaldIrIXind' => 896, 'slerin baXi' => -897, 'i yeni baXi' => -898, 'Unden baXin' => 899,
    'i de baXina' => 900, 'Cindeki maX' => 901, 'ile baXinin' => 902, 'dan baXini ' => 903, 'masInda Xen' => 904,
    'seri nde iX' => 905, 'a Cin in Xi' => 906, 'u ve baXind' => 907, 'erlerini aX' => -908, 'duGunu baXi' => -909,
    ' nin baXini' => 910, 'e aXilmayac' => -911, 'iCtikleri X' => -912, 'Cin yaXama ' => -913, 'a dUnyada X' => -914,
    'kilinin yaX' => -915, 'Iyla yaXama' => -916, 'srIn yarIX' => 917, 'disini aXt' => -918, 'elecek yaX' => -919,
    'dsi nin Xu' => -920, 'man baXind' => -921, 'on yarIXin' => 922, 'Cin baXina' => -923, '  baXindak' => -924,
    'layI baXin' => -925, 'ra da baXi' => -926, 'n Once iXe' => -927, 'nlikle baX' => 928, 'e anlaXin ' => 929,
    'oyle bir X' => -930, '  besiktaX' => -931, 'n besiktaX' => -932, 'dar baXina' => -933, 'or yaXama ' => -934,
    ' Cok yaXa ' => 935, 'k iCin Xok' => -936, 'alarI baXi' => -937, 'icilere Xu' => -938, 'Inda miloX' => 939,
    'i baXindak' => -940, 'dil yanlIX' => 941, 'larIyla Xu' => -942, 'ak iCin Xu' => -943, 'e olan Xu ' => -944,
    'alarInI Xu' => -945, 'steriXin ' => 946, 'hal gUreX' => -947, 'ap yanlIX' => 948, 'Slerin aX' => -949,
    'an baXkim' => 950, 'na baXini' => -951, 'buraya Xu' => -952, 'ta baXina' => -953, 'eman kurX' => -954,
    'bi baXind' => -955, ' kaldIrIX' => 956, ' yeni iXe' => 957, 'nin aXild' => -958, 'ama baXin' => -959,
    'rI da Xu ' => -960, 'ra aXilac' => -961, 'el Xoyler' => 962, 'yan baXin' => -963, ' dava Xu ' => 964,
    'meyi baXi' => -965, 're baXind' => -966, 'ane baXin' => 967, 'stlar baX' => 968, 'aSInda Xu' => -969,
    'esiyle Xu' => -970, 'on yaXama' => -971, 'mer kaleX' => 972, 'li baXina' => -973, 'r o nun X' => 974,
    'in yaXini' => -975, 'Syerini X' => -976, 'dinci baX' => -977, 'kInda yaX' => -978, 'o da baXi' => -979,
    'a aXilmaz' => 980, 'zaman Xen' => -981, 'lan baXin' => -982, 'ulusu baX' => 983, 'lIGI baXi' => -984,
    'her ne iX' => -985, 'nlarI Xu ' => -986, 'Ik aXilm' => 987, 'en ekXi ' => 988, 'bile iXe' => 989,
    'sine Xut' => 990, 'man Xik ' => 991, 'ep gOGUX' => 992, 'eski taX' => -993, 'U baXina' => -994,
    's yaXin ' => -995, 'S yaXama' => -996, 'm ilk iX' => 997, 'onun iXe' => 998, ' aXmayal' => -999,
    'n de iXe' => 1000, 'et Xansa' => -1001, 'dece iXe' => 1002, 'ade Xik ' => 1003, 'len Xam ' => -1004,
    'elim yaX' => 1005, 'rIn yaXi' => -1006, 'I Xanlit' => -1007, '  bu Xu ' => -1008, 'Siyi iXe' => 1009,
    'pille iX' => 1010, 'rdi baXi' => -1011, 'ganI yaX' => -1012, 'an Xaka ' => -1013, 're aXilm' => -1014,
    'naya baX' => -1015, 'sene baX' => 1016, 'smini aX' => -1017, 'ki taXi ' => -1018, 'm yaXin ' => -1019,
    'a gene X' => -1020, 'ereden X' => -1021, 'kiden Xu' => -1022, 'rdeki Xu' => -1023, 'azin baX' => -1024,
    'mun Xeri' => 1025, 'cdet baX' => -1026, 'la aXilm' => 1027, 'man iXme' => 1028, 'hyol iXe' => 1029,
    'can Xen ' => -1030, 'sIr da X' => -1031, 'daha Xu ' => -1032, 'bun baXi' => -1033, 'I beXte ' => 1034,
    ' an kurX' => -1035, 'emen iXe' => 1036, 'zli Xirk' => 1037, 'ir eXin ' => 1038, 'p de baX' => 1039,
    'Once iXe' => 1040, 'ak miloX' => 1041, ' bin yaX' => 1042, 'um baXin' => -1043, 'SCiyi iX' => 1044,
    'nlar kaX' => -1045, 'u baXina' => -1046, 'ller aXi' => 1047, 'arasI Xu' => -1048, 'k beXte ' => 1049,
    'sIz baXi' => -1050, 'asan meX' => 1051, 'yasa taX' => -1052, 'u daki X' => -1053, 'le yarIX' => 1054,
    'zor kaX' => 1055, 'blo Xu ' => 1056, 'yaraXan' => 1057, 'Gur kOX' => -1058, ' Su iXe' => 1059,
    'met akX' => 1060, 'U faSiX' => 1061, 'anI iX ' => -1062, 'GI baX ' => -1063, 'vrim Xe' => 1064,
    'GI Xik ' => 1065, ' aXimiz' => 1066, ' Xanda ' => 1067, 'yetiXin' => 1068, ' aXinda' => -1069,
    'Xarkisy' => -1070, 'haXani ' => 1071, 'on baX ' => -1072, 'piXtim ' => 1073, ' ka laX' => 1074,
    'a Xoke ' => 1075, 'aha iXe' => 1076, 'tanIXin' => -1077, ' araXiy' => 1078, 'gemi Xu' => -1079,
    'Ska iXe' => 1080, 'san aX ' => 1081, 'masabaX' => 1082, 'e iXimd' => 1083, 'kle iXe' => 1084,
    'er baX ' => -1085, 'ap baXi' => -1086, ' aXimi ' => 1087, 'nli Xu ' => -1088, 'ti kurX' => -1089,
    'Gu yaXi' => -1090, 'ok yaX ' => -1091, 'y de Xu' => -1092, 'na aXar' => -1093, 'n baXal' => 1094,
    'eye Xut' => 1095, 'sse baX' => 1096, 'Uk yaX ' => -1097, 'alI aXa' => -1098, 'Xevketl' => 1099,
    'In taXl' => 1100, 'pa baXi' => -1101, 'Xahabet' => 1102, 'yikli X' => -1103, 'I Xoke ' => 1104,
    'al Xen ' => 1105, 'en Xis ' => 1106, 'amIn Xu' => -1107, 'InI peX' => 1108, 'lle baX' => 1109,
    'han kOX' => -1110, 'Unya Xu' => -1111, 'alI Xu ' => 1112, '  iXmen' => 1113, 'ir Xokt' => 1114,
    'nan Xu ' => -1115, 'azIlIXi' => -1116, 'Ina aXt' => -1117, 'na aXan' => -1118, 'Ik kurX' => -1119,
    'ek Xan ' => 1120, 'en ayXi' => -1121, 'nI bahX' => 1122, 'hane Xu' => 1123, 'riz aXi' => 1124,
    'trafI X' => -1125, 'Ci baX ' => -1126, 'SarIlIX' => -1127, 'ce yaXl' => 1128, 'de yaXl' => 1129,
    'gende X' => -1130, 'skin Xi' => -1131, 'taXarak' => 1132, 'r kayIX' => 1133, '  baXim' => 1134,
    'm baXim' => 1135, 'ste baX' => 1136, 'Xevkini' => 1137, 'baXindi' => -1138, 'are baX' => 1139,
    'e aXma ' => -1140, 'tme yaX' => -1141, 'ebe baX' => 1142, 'kten Xu' => -1143, 'aXlaniy' => 1144,
    'in kaXl' => 1145, ' tonkuX' => 1146, 'dur yaX' => -1147, 'llar Xu' => -1148, 'an Xisl' => -1149,
    'IndaXin' => -1150, 'I aXmal' => 1151, ' ne iXi' => 1152, ' Xutun ' => -1153, 'e Xutu' => -1154,
    'Xutunu ' => 1155, 'temiz X' => -1156, 'Xecerel' => 1157, 'en erXe' => 1158, 'aXarsa ' => 1159,
    'i Xeydi' => -1160, 'en beXt' => -1161, 'yaXamiz' => -1162, 'GU ne iX' => -1163, 'i ne iXe' => -1164,
    ' ne iXe' => 1165, 'na bahX' => 1166, 'aXiyan ' => 1167, 'd baXin' => -1168, 'aXardi ' => 1169,
    'rIn peX' => 1170, 'akXamis' => -1171, ' suXa ' => 1172, 'Xatiro' => 1173, 'dI niX' => 1174,
    'm miX ' => 1175, 'n Xom ' => 1176, ' ekXit' => 1177, 'hap aX' => -1178, 'iel Xa' => 1179,
    'di niX' => 1180, 'paXakl' => -1181, 'Xaril ' => 1182, 'cevanX' => 1183, 'y ariX' => 1184,
    'tiliXi' => 1185, 'piXik ' => 1186, 'taXis ' => -1187, ' iXkin' => 1188, ' aXmaz' => -1189,
    'e aXsa' => 1190, 'Xendir' => 1191, 'Xarhoy' => 1192, 'maviX ' => 1193, 'kaXeyl' => 1194,
    'Xarkik' => -1195, 'olibaX' => -1196, ' eliXi' => 1197, 't Xama' => 1198, 'k Xikt' => 1199,
    't Xik ' => 1200, 'nlaXa ' => -1201, 'e Xeyd' => -1202, 'Xoylek' => 1203, 'aXkul ' => -1204,
    'cuzlaX' => -1205, 'yaraXl' => 1206, 'Xovuyo' => -1207, 'Xiril ' => 1208, 'uzlu X' => -1209,
    'hteriX' => -1210, 'p miX ' => 1211, 'kiXin ' => -1212, ' kIXiy' => -1213, 'siliXl' => 1214,
    ' tIraX' => 1215, 'kalmiX' => -1216, 'kadreX' => 1217, ' Xeniz' => 1218, 'r Xal ' => 1219,
    'kkalIX' => -1220, 'igiriX' => -1221, 'a Xas ' => -1222, ' Xikid' => 1223, 'kXili ' => 1224,
    'l alaX' => 1225, 'ne aXm' => -1226, 'nI eX ' => -1227, 'mUze X' => -1228, 'Xkonse' => 1229,
    '  Xar ' => -1230, 'epreXe' => 1231, ' Xemis' => 1232, 'na fiX' => 1233, 'ur Xun' => 1234,
    ' deXiy' => 1235, ' deXer' => 1236, 'ki aX ' => -1237, ' biXi ' => 1238, 'el avX' => -1239,
    'Ga Xov' => -1240, 'azalIX' => 1241, 'beXler' => 1242, 'faltaX' => 1243, 'IXitan' => 1244,
    'fa Xut' => 1245, 'ikalIX' => -1246, 'ap Xap' => 1247, 'Xehven' => -1248, 'mla Xu' => -1249,
    ' traXi' => 1250, ' ayXin' => 1251, ' Xande' => 1252, 'l Xut ' => 1253, 'Xermin' => 1254,
    'ldaXin' => -1255, 'of Xen' => 1256, ' atlaX' => -1257, ' az Xu' => -1258, 'marXiy' => 1259,
    'damIXi' => -1260, ' kadIX' => -1261, 'iz aXm' => -1262, 'arakeX' => 1263, 'z Xas ' => -1264,
    'din Xo' => -1265, 'un arX' => -1266, ' Xenti' => 1267, ' Xort ' => 1268, ' paXay' => 1269,
    'l baX ' => -1270, 'aca aX' => -1271, ' maraX' => 1272, 'Xaplak' => 1273, 'leXele' => -1274,
    'yaXsa ' => 1275, 'taXliy' => -1276, 'ltIraX' => 1277, 'nti Xu' => -1278, 'mail X' => -1279,
    'baXars' => -1280, 'fiS aX' => -1281, ' iXti ' => 1282, 'IXilik' => 1283, 'saim X' => -1284,
    'deXilm' => 1285, ' nI aX' => 1286, 'l kurX' => -1287, 'rli Xu' => -1288, ' temiX' => -1289,
    'Une Xu' => -1290, 'leXemi' => 1291, 'I Xip ' => 1292, 'ddileX' => 1293, 'baXsa ' => -1294,
    'eltraX' => 1295, ' reXim' => -1296, 'rgun X' => -1297, ' aXin ' => -1298, '18 yaX' => 1299,
    'cer Xa' => -1300, 'Xenli ' => -1301, 'zelleX' => 1302, 'aSarIX' => -1303, 'e ulaX' => 1304,
    'SiktaX' => 1305, 'hIXir' => 1306, 'Xaftt' => 1307, 'Xuhut' => 1308, ' teXn' => 1309, 'Xuand' => 1310,
    ' yumX' => 1311, 'zereX' => 1312, 'COrUX' => 1313, 'Xbaya' => -1314, 'iapaX' => -1315,
    'a Xb ' => 1316, '  CiX' => 1317, 'Xamon' => 1318, ' adIX' => -1319, ' Xeca' => 1320,
    'monaX' => -1321, 'areXa' => 1322, 'Xutuy' => 1323, 'Xeydo' => -1324, 'gItIX' => -1325,
    'Xidet' => 1326, 'basIX' => 1327, 'menXa' => 1328, 'tasiX' => 1329, 'rdaXa' => -1330,
    'lahoX' => -1331, 'inkiX' => 1332, 'zanIX' => 1333, 'akmuX' => -1334, 'kaXes' => 1335,
    'inXat' => 1336, 'Xmany' => -1337, 'uyuXl' => 1338, 'Xendu' => 1339, 'UXrik' => 1340,
    ' iXer' => 1341, ' Xuh ' => 1342, 'Xeyy ' => 1343, 'tIXim' => 1344, '  Xoo' => 1345,
    'tutaX' => 1346, 'ahaX ' => -1347, 'yoveX' => 1348, 'yavuX' => 1349, 'kai X' => 1350,
    'palIX' => -1351, 'tepiX' => 1352, 'kaXer' => 1353, ' maXi' => -1354, 'rkaX ' => 1355,
    'haXa ' => 1356, 'Xbiht' => 1357, 'hlaXl' => -1358, 'libeX' => 1359, 'tIrIX' => -1360,
    'lunuX' => 1361, 'Xivey' => 1362, 'Xikec' => -1363, 'bediX' => 1364, 'Xambr' => 1365,
    'baSaX' => 1366, 'tt IX' => -1367, 'tibaX' => -1368, 'Xtahb' => -1369, 'umpaX' => -1370,
    'o aXi' => -1371, 'Xovup' => -1372, 'beXa ' => -1373, 'ekkaX' => 1374, ' Xad ' => 1375,
    ' Xaas' => 1376, 'la tuX' => 1377, ' tuX ' => -1378, ' aXas' => -1379, 'celeX' => -1380,
    'SUnUX' => 1381, 'kuXti' => -1382, ' aiXe' => 1383, 'Xakad' => 1384, ' i X ' => 1385,
    'OXem ' => -1386, ' ho X' => 1387, 'oka X' => -1388, 'Xeyyy' => 1389, ' CakX' => 1390,
    'kUp X' => -1391, 'Xippa' => 1392, ' Xipi' => 1393, 'Xahiz' => 1394, 'tleXs' => 1395,
    'okleX' => -1396, 'ronXi' => 1397, 'lIXim' => 1398, ' laXm' => 1399, 'Xaffa' => 1400,
    ' mIXt' => -1401, ' taXy' => 1402, 'zeXe ' => -1403, 'Oy aX' => -1404, ' iXen' => -1405,
    ' Xih ' => 1406, ' Xukl' => -1407, 'p duX' => -1408, 'm 7 X' => -1409, 'UzleX' => 1410,
    'paXab' => 1411, 'iXlin' => -1412, 'amboX' => -1413, 'aliXm' => -1414, 'koluX' => 1415,
    'hXete' => 1416, 'oplaX' => -1417, 'diXis' => 1418, 'Xikag' => 1419, '6 niX' => -1420, 'Xafag' => 1421,
    'Xaziy' => 1422, ' fiXl' => 1423, 'kamIX' => 1424, 'Xivel' => 1425, 'ynarX' => -1426,
    'ynaXt' => 1427, 'IndiX' => 1428, ' eciX' => 1429, 'haXla' => 1430, 'nXall' => 1431,
    'aXdir' => 1432, 'tiXma' => -1433, 'p kaX' => -1434, 'urmuX' => 1435, 'daXim' => 1436,
    'ehveX' => 1437, 'Xela ' => -1438, 'naXin' => -1439, ' leXi' => 1440, 'iXmin' => -1441,
    'gIzIX' => -1442, 'aXtam' => -1443, 'r peX' => 1444, 'chriX' => -1445, ' pekX' => 1446,
    'iXmi ' => -1447, ' olaX' => -1448, 'Xatis' => -1449, 'kleXt' => 1450, 'iXile' => 1451,
    'Xubl' => -1452, 'vyaX' => -1453, 'nvaX' => -1454, 'C X ' => 1455, 'Xnag' => 1456,
    'OXur' => 1457, 'goaX' => -1458, 'Xesh' => 1459, 'iXov' => -1460, 'Xezl' => 1461,
    'iaXe' => 1462, 'zeXt' => 1463, 'haaX' => -1464, 'aluX' => -1465, 'Xass' => -1466,
    'IXaf' => -1467, 'Xakr' => 1468, 'IXiv' => 1469, 'UXat' => 1470, 'Xviy' => -1471,
    'Xarc' => -1472, 'zbeX' => 1473, 'Xamc' => 1474, 'Xeto' => -1475, 'iXai' => -1476,
    'iCXa' => 1477, 'Xcie' => -1478, 'Xubs' => -1479, ' nIX' => -1480, 'SXaf' => 1481,
    'ruXs' => -1482, 'tUSX' => 1483, 'ytUX' => 1484, 'SkOX' => 1485, 'pUXo' => 1486,
    'SipX' => 1487, 'buXa' => -1488, 'aXe ' => -1489, 'laXh' => -1490, 'daXa' => 1491,
    'khoX' => -1492, 'Xnin' => 1493, 'oXin' => -1494, 'Xcev' => 1495, 'ilaX' => -1496,
    'SaaX' => -1497, 'mlIX' => -1498, 'erXt' => -1499, 'lhaX' => -1500, 'fXaa' => 1501,
    'obUX' => -1502, 'Xehr' => 1503, 'mevX' => -1504, 'Xing' => -1505, 'Xeyu' => -1506,
    'aXap' => -1507, 'Xay ' => -1508, ' uXt' => -1509, 'boXi' => -1510, 'apiX' => -1511,
    'Xlim' => -1512, ' siX' => -1513, 'asiX' => -1514, 'breX' => -1515, 'mXen' => -1516,
    'SbaX' => 1517, 'umuX' => 1518, 'aXhi' => -1519, 'kraX' => -1520, 'Xica' => -1521,
    'q X' => -1522, 'kXc' => -1523, 'Xbc' => -1524, 'Xmd' => 1525, 'dyX' => -1526,
    'jaX' => -1527, 'Xf ' => -1528, 'Xie' => -1529, 'Xo ' => -1530, 'Xea' => -1531,
    ' oX' => -1532, 'cIX' => -1533, 'seX' => -1534, 'Xep' => -1535, 'hyol un baXi' => -1536,
    ' nIn yaXama' => -1537, 'ken baXini ' => -1538, 'yanlar Xu ' => -1539, 'deri fatoX' => -1540, 'indeki Xu ' => -1541,
    ' metin kOX' => -1542, 'lerden Xu ' => -1543, 'lar baXina' => -1544, 'Xemalarini' => 1545, 'ndi baXini' => 1546,
    'ler baXina' => -1547, 'may I baX' => -1548, 'ettin Xen' => 1549, 'a koXtum ' => 1550, 'Gil baXin' => -1551,
    'eti baXin' => -1552, 'eden baXi' => -1553, 'zIlI baXi' => -1554, 'un baXini' => 1555, 'rInI baXi' => -1556,
    'nelin iXe' => 1557, 'para baXi' => -1558, 'in yanlIX' => 1559, 'e ve Xu ' => -1560, 'met Xen ' => 1561,
    'resim iX' => 1562, 'a haXim ' => -1563, 'e haXim ' => -1564, 'st sUrUX' => 1565, 'la iXim ' => 1566,
    'SIma Xu ' => -1567, 'Ci yarIX' => 1568, 'z ve Xu ' => -1569, 'zca baXi' => -1570, ' aXiliyo' => 1571,
    'In ekXi ' => 1572, 'n Xeydi ' => -1573, 'Gi yarIX' => 1574, 'a haXir ' => 1575, 'hem baXi' => -1576,
    'a yaXlan' => -1577, 'te yarIX' => 1578, '  Xundan' => 1579, 'mi baXin' => -1580, 'ayak baX' => -1581,
    'ece Xik ' => 1582, 'ar aXilm' => 1583, '  yanlIX' => 1584, 'Xisinda ' => 1585, 'uldan iX' => 1586,
    'e yanlIX' => 1587, 'por baXi' => -1588, 'gut Xen' => 1589, 'u Xutu ' => 1590, 'Xoyleym' => 1591,
    'beXten ' => 1592, 'bah iXe' => 1593, ' IXini ' => 1594, 'ne yaX ' => -1595, 'el Xik ' => 1596,
    ' kantaX' => 1597, 'sI Xam ' => -1598, 'tle iXe' => 1599, 'nI aXka' => -1600, ' Xiklig' => 1601,
    '6 yarIX' => 1602, 'ice Xu ' => -1603, 'l ve Xu' => -1604, 'm Xuna ' => 1605, ' iXteki' => 1606,
    'Ika baX' => -1607, 'k Xuna ' => 1608, 'el Xeke' => -1609, 'ne Xis ' => 1610, 'ziz ekX' => 1611,
    'ayandaX' => -1612, 'Xirkeli' => -1613, 'Xarkis ' => -1614, 'ele iXe' => 1615, 'k Xirk ' => 1616,
    'saG baX' => -1617, 'Ur baXi' => -1618, 'us baXi' => -1619, 'z Xuna ' => 1620, 'an beXt' => -1621,
    'cU kurX' => -1622, 'Xekvac' => 1623, ' akkaX' => 1624, 'Xarkoy' => 1625, 'da aX ' => -1626,
    'Xeyder' => -1627, 'Xoktan' => 1628, 'pis Xu' => -1629, ' yumaX' => 1630, ' uXsal' => -1631,
    'Xantan' => -1632, 'kandiX' => 1633, ' tebaX' => -1634, 'GuXuna' => -1635, 'jak eX' => -1636,
    'yarImX' => 1637, 'I Xas ' => -1638, 'yanarX' => -1639, 'kuXuyo' => -1640, 'Xavata' => 1641,
    'Xasinl' => 1642, 'laXam ' => -1643, ' diXiy' => 1644, 'ndiriX' => 1645, 'yboluX' => 1646,
    'l Xam ' => -1647, 'baltaX' => -1648, ' rOveX' => 1649, 'niXanc' => 1650, 'atIk X' => -1651,
    'Xokta ' => 1652, 'niXtik' => -1653, 'va Xu ' => -1654, ' parIX' => -1655, 'a aXip' => -1656,
    'e Xove' => -1657, ' seyiX' => -1658, ' meXin' => 1659, 'tre Xu' => -1660, ' eXten' => 1661,
    'in aXt' => -1662, ' kIXik' => -1663, 'Ip iXe' => 1664, 'IXalim' => 1665, 'taXlad' => 1666,
    ' kIXip' => -1667, 'let Xu' => -1668, 'k taXlam' => -1669, 'taXlam' => 1670, ' aXmal' => -1671,
    'Il aXi' => 1672, ' Xisti' => 1673, 'sen Xo' => -1674, 'Xaksak' => 1675, 'leXel ' => -1676,
    'miXlig' => 1677, 'Xiklar' => 1678, 'iyileX' => 1679, 'apIXin' => -1680, 'araXti' => 1681,
    'konuXt' => 1682, 'aXkisi' => -1683, 'mayIX ' => -1684, 'yaXli ' => 1685, 'Xlika' => -1686,
    'Xenca' => 1687, 'araXs' => 1688, 'traXa' => 1689, 'lpaX ' => -1690, 'Xavki' => 1691,
    'Xteru' => -1692, 'i bXk' => 1693, 'rusiX' => 1694, ' IXim' => 1695, 'paXak' => 1696,
    ' neX ' => 1697, 'iXist' => 1698, 'aXane' => 1699, 'C aX ' => -1700, '0 aXa' => -1701,
    'ataXt' => 1702, 'Xvili' => 1703, 'Xovda' => 1704, 'UveyX' => 1705, 'Xuhub' => 1706, 'gedaX' => -1707,
    'fatoX' => 1708, 'lagaX' => 1709, 'elkIX' => -1710, 'SikaX' => 1711, ' Xenb' => 1712,
    'yiyiX' => 1713, 'Xindu' => -1714, 'aXado' => 1715, 'iXmer' => 1716, 'yuXul' => 1717,
    'aXsor' => 1718, 'uXamb' => 1719, 'Xarkl' => 1720, 'kaXli' => 1721, 'Xifai' => 1722,
    'abiXe' => 1723, ' Xuym' => 1724, 'dUnUX' => 1725, ' Xuka' => -1726, 'Xubab' => -1727,
    '4 Xu ' => -1728, 'Xambl' => -1729, ' uXa ' => -1730, 'Xeyis' => -1731, 'efriX' => 1732,
    'rtUXe' => 1733, 'CamaX' => 1734, 'Xelik' => 1735, 'OktaX' => 1736, 'kuXar' => -1737,
    'nXira' => 1738, 'kIrIX' => 1739, 'U kaX' => -1740, 'yaSXa' => 1741, 'keSiX' => 1742,
    'diXer' => 1743, 'mibeX' => 1744, 'yeXe ' => -1745, 'sabiX' => 1746, 'Xkill' => 1747,
    'Xerh ' => 1748, 'fataX' => -1749, 'maaaX' => 1750, 'Xiven' => 1751, 'tIXir' => 1752,
    'varIX' => 1753, 'yanuX' => -1754, 'Xele ' => -1755, ' aXev' => 1756, 'rataX' => 1757,
    'yeXte' => -1758, 'ahriX' => 1759, ' alIX' => 1760, 'Xerri' => 1761, 'Xayia' => 1762,
    'aXant' => 1763, 'e peX' => 1764, 'iriXt' => 1765, ' Ximd' => 1766, 'oXman' => -1767,
    ' maXt' => -1768, 'aXlat' => 1769, 'Xikta' => 1770, ' yUkX' => -1771, 'iXtan' => -1772,
    ' IXd' => -1773, 'mXor' => 1774, 'GdiX' => 1775, 'thoX' => -1776, ' SeX' => 1777,
    ' rIX' => -1778, 'duXl' => 1779, 'kaXc' => -1780, 'lapX' => 1781, 'Xish' => 1782,
    'SbeX' => 1783, 'Xpem' => -1784, 'cmiX' => -1785, 'iXl ' => 1786, 'ahXo' => 1787,
    'zaX ' => -1788, 'taXg' => 1789, 'oooX' => 1790, 'llIX' => -1791, 'aXle' => -1792,
    'yaXm' => -1793, 'Xkad' => 1794, 'Xehe' => -1795, 'Xmat' => -1796, 'cUX ' => 1797,
    'Xsut' => -1798, 'triX' => -1799, 'neXc' => -1800, 'uvaX' => -1801, 'puXt' => 1802,
    'phoX' => -1803, 'SmuX' => 1804, 'efiX' => -1805, 'aGuX' => -1806, 'iCiX' => 1807,
    'Xmit' => -1808, 'iteX' => -1809, 'uXut' => 1810, 'oliX' => -1811, 'baXl' => 1812,
    'IXf' => -1813, 'OX ' => -1814, 'IaX' => -1815, 'iXz' => -1816, 'ouX' => -1817,
    'auX' => -1818, 'cuX' => -1819, 'ceX' => -1820, 'Xon' => -1821, 'Xq' => -1822,
    'ile yaXama ' => -1823, 'iden yaXam' => 1824, 'ci yaXama' => -1825, ' diye iXe' => 1826, 'ili yaXin' => -1827,
    'aile baXi' => 1828, 'ri baXina' => -1829, 'i duruXu' => -1830, 'aki Xik ' => 1831, 'Se baXin' => 1832,
    'I ve Xu ' => -1833, 'n yaXin ' => -1834, 'prak Xu ' => -1835, 'e yaXlan' => -1836, 'Gine Xu ' => -1837,
    'GIr baXa' => -1838, 'sI baXin' => -1839, 'ik yarIX' => 1840, ' sarIXin' => 1841, 'ek yarIX' => 1842,
    'eklere X' => -1843, 'ki baXin' => -1844, 'u yanlIX' => 1845, 'ara aXt' => -1846, 've baX ' => -1847,
    'hiC Xu ' => -1848, 'raGI aX' => -1849, 'en Xane' => 1850, 'e Xerai' => 1851, 'le haXi' => 1852,
    'stekleX' => -1853, 'n Xara ' => 1854, 'ail a X' => -1855, 'n Xahi ' => 1856, 'mek Xu ' => -1857,
    'kart aX' => -1858, 'baXarla' => -1859, 'tanIXir' => 1860, 'rIna aX' => -1861, 'Xarkiyo' => -1862,
    'baXiniz' => 1863, 'SlaXa ' => -1864, 'eXirin' => -1865, 'kaXkar' => 1866, ' moXe ' => 1867,
    'na aXm' => -1868, 'taXlas' => 1869, 'taXark' => 1870, 'IXtirt' => 1871, 'haXin ' => 1872,
    ' Xurda' => 1873, 'Xirke ' => -1874, 'faXit ' => -1875, 'haXiml' => -1876, 'd yaXi' => -1877,
    'rt Xut' => 1878, 'CIklaX' => -1879, 'ce aXt' => -1880, 'Xahase' => 1881, ' dalIX' => 1882,
    'rtemiX' => -1883, 'iveriX' => -1884, 'ulya X' => 1885, 'Xenses' => 1886, ' finiX' => 1887,
    'le i X' => 1888, 'OrtUXt' => 1889, ' Xemil' => 1890, 'kalleX' => 1891, ' a Xev' => -1892,
    ' tInIX' => -1893, 'zanlIX' => -1894, 'rlaXa ' => -1895, 'rundaX' => -1896, 'okuyuX' => 1897,
    'takIXi' => -1898, 'Is a X' => -1899, ' kaXi ' => 1900, 'ydoGuX' => -1901, 'Xeydis' => -1902,
    'laXiyo' => 1903, ' mIXs' => -1904, 'nureX' => 1905, 'Xmeka' => 1906, 'eXarp' => 1907,
    'aXtun' => 1908, 'OnkaX' => 1909, 'Xakag' => 1910, 'Xarto' => -1911, ' iXyu' => 1912,
    ' piXe' => 1913, 'talIX' => -1914, 'mahXe' => 1915, 'UrUX ' => 1916, 'aXime' => -1917,
    'huXu ' => 1918, 'Xubap' => -1919, ' Xeny' => 1920, 'Xifah' => 1921, ' taXp' => 1922,
    'Xahha' => 1923, 'paXas' => 1924, 'kotaX' => -1925, 'Xefir' => -1926, 'Xevka' => 1927,
    ' eSXe' => 1928, 'Xerht' => 1929, 'nayIX' => 1930, 'Xmann' => -1931, 'priXt' => 1932,
    'beXir' => 1933, 'mabaX' => 1934, 'Xtirh' => -1935, 'Xubar' => -1936, ' Xuks' => -1937,
    'SayIX' => 1938, 'Xikir' => 1939, 'nanIX' => 1940, ' aloX' => 1941, 'iXeme' => 1942,
    ' Xato' => 1943, 'aXsiz' => 1944, 'GlaXa' => -1945, 'paXan' => 1946, 'Xsek ' => 1947,
    ' okuX' => -1948, 'Xabiy' => -1949, 'leXip' => 1950, ' SakX' => 1951, 'emXir' => 1952,
    'variX' => -1953, 'tutuX' => 1954, 'Xira ' => -1955, ' Xtl' => 1956, 'oXav' => -1957,
    'Xsie' => -1958, 'oboX' => -1959, 'SlIX' => -1960, 'SaXe' => -1961, 'vXal' => -1962,
    'mtaX' => -1963, 'ruXo' => -1964, 'shaX' => -1965, 'nuXk' => 1966, 'mUXk' => 1967,
    'fyaX' => -1968, 'ruXi' => -1969, 'taXb' => 1970, 'kiXn' => 1971, 'IyaX' => -1972,
    'eXic' => -1973, 'pXir' => 1974, 'UXer' => 1975, 'rSIX' => -1976, 'Xgal' => 1977,
    'iXtl' => -1978, 'kkuX' => 1979, 'lbaX' => 1980, 'rasInda baXin' => -1981, 'n Xutunda ' => 1982,
    'yarak iXe' => 1983, '  yaXama ' => -1984, 'u yaXama ' => -1985, 're yarIX' => 1986, 'can Xunu' => -1987,
    'Cin yaX ' => -1988, 'dIS baXi' => -1989, 'dUn baXi' => -1990, 'el baXin' => -1991, 'a ve Xu ' => -1992,
    'ne baXa ' => -1993, ' marXa ' => 1994, 'her iXe' => 1995, ' Xilan ' => 1996, 'hir kOX' => -1997,
    'kaXigi ' => 1998, 'onuXun ' => 1999, ' aXilar' => -2000, ' Xanlim' => -2001, 'Cok iXe' => 2002,
    ' aXilip' => 2003, 'yaXadir' => -2004, 'taXland' => 2005, 'rt baX ' => -2006, '  ekXi ' => 2007,
    '  yaXin' => -2008, 'Xullah' => 2009, 'oleyiX' => -2010, 'rtabaX' => -2011, '  koX ' => -2012,
    ' baXis' => -2013, 'Xereme' => 2014, 'Xikki ' => 2015, 've iXe' => 2016, 'ait aX' => -2017,
    'Xamari' => 2018, ' kIXi ' => 2019, 'amanaX' => 2020, 'lI Xu ' => -2021, 'mUXtek' => 2022,
    'i Xarv' => 2023, 'armaX ' => 2024, 'eman X' => -2025, 'Xterih' => -2026, 'hIXiml' => 2027,
    ' aXarl' => -2028, 'zveriX' => -2029, 'keXis ' => 2030, 'skolaX' => -2031, 'baXim ' => -2032,
    'im Xab' => -2033, 'Xokuyl' => 2034, 'miXsin' => 2035, 'uyanIX' => 2036, 'rik Xu' => -2037,
    'lk beX' => 2038, 'Xurani' => 2039, 'birXey' => 2040, 'y Xere' => 2041, 'Xakar ' => 2042,
    'betleX' => 2043, 'paylaX' => 2044, 'rUkUX' => 2045, 'Xenku' => 2046, 'lakuX' => 2047,
    ' eXip' => -2048, 'cayiX' => 2049, ' CiXi' => 2050, 'oulaX' => -2051, 'ekeX ' => 2052,
    'o Xun' => -2053, ' Xilt' => 2054, 'dOGUX' => 2055, 'Xkire' => -2056, 'kiriX' => 2057,
    'rciX ' => 2058, 'Xakka' => 2059, 'timaX' => -2060, 'mIXi ' => 2061, 'hi Xu' => -2062,
    ' raXm' => -2063, 'Xeyeh' => -2064, 'kaXar' => 2065, 'atraX' => -2066, 'IXinc' => 2067,
    ' Xti ' => 2068, 'Xalte' => 2069, 'Xahap' => 2070, 'ediX ' => 2071, 'Xives' => 2072,
    's kIX' => -2073, 'nuluX' => 2074, 'Xataf' => 2075, 'gUlUX' => 2076, 'ontaX' => -2077,
    ' eleX' => 2078, 'Xevkl' => 2079, 'hadiX' => -2080, 'kiXd' => 2081, 'guXe' => 2082,
    'OniX' => 2083, 'Xmav' => -2084, 'Xahb' => 2085, 'Xuke' => -2086, 'aXao' => 2087,
    ' daX' => -2088, 'zoXi' => 2089, 'ddaX' => -2090, 'Xofb' => 2091, 'Xevl' => 2092,
    'yeX ' => -2093, 'eyXa' => 2094, ' iXc' => 2095, ' dIX' => 2096, 'Xdu' => 2097,
    'orX' => -2098, 'en yaXama ' => -2099, '  baXinda ' => -2100, 'ken Xen ' => -2101, ' Xiklik ' => 2102,
    'vara aX' => -2103, 'kdire X' => 2104, 'n Xutu ' => 2105, 'n aXild' => 2106, 'sel ekX' => 2107,
    'yol Xu ' => -2108, 'len Xu ' => -2109, 'm yarIX' => 2110, 'Xuheyd' => 2111, 'Xakaci' => 2112,
    'boXta ' => 2113, 'ton Xu' => -2114, 'Xilili' => 2115, 'Ximsir' => 2116, ' aXtir' => -2117,
    'Xarkit' => -2118, 'rdak X' => -2119, 'tlaXa ' => -2120, 's baX ' => -2121, ' arXin' => 2122,
    'rdaXin' => -2123, 'mUXur' => 2124, 'ukaXe' => 2125, 'aXalt' => 2126, 'ifXa ' => 2127,
    'nakaX' => 2128, 'InmeX' => 2129, 'labaX' => 2130, 'vebaX' => -2131, 'keXan' => 2132,
    'onXim' => 2133, 'Xirre' => 2134, ' Xahp' => 2135, 'Xkemb' => 2136, ' elaX' => -2137,
    'didiX' => 2138, 'trbaX' => -2139, 'sOGUX' => 2140, 'hu Xa' => 2141, 'Xabi ' => -2142,
    'Xevar' => 2143, 'Guk X' => -2144, 'Xehir' => 2145, 'aXiye' => -2146, ' irX' => 2147,
    'hXa ' => -2148, 'hrIX' => -2149, 'ferX' => -2150, 'oXid' => -2151, 'IsaX' => -2152,
    'aXoc' => 2153, 'kteX' => -2154, 'Xuay' => 2155, 'Xyen' => -2156, 'laXy' => -2157,
    'draX' => -2158, 'Xs ' => -2159, 'eiX' => -2160, 'suX' => -2161, ' yeni yaXama' => -2162,
    'In baXinin ' => 2163, 'ya baXini' => -2164, 'basa baX' => -2165, 'tI baXin' => -2166, 'cI baXin' => -2167,
    ' Xisler' => -2168, 'Xendil' => 2169, 'eklanX' => 2170, 'mastaX' => 2171, 'etiXin' => -2172,
    ' beleX' => 2173, 'menekX' => 2174, 'taXtik' => -2175, 'g Xam ' => -2176, 'tabaXi' => 2177,
    ' diXim' => 2178, ' neXte' => 2179, ' kalIX' => 2180, 'SanlIX' => -2181, 'iXter ' => -2182,
    ' meXe ' => 2183, 'rkadaX' => 2184, 'almuX' => -2185, 'Xampu' => 2186, 'baXkl' => -2187,
    ' inX ' => 2188, ' eXe ' => 2189, 'Xiris' => 2190, 'Xsoy ' => 2191, 'dalaX' => 2192,
    'mUXuz' => 2193, 'fadoX' => 2194, 'yabaX' => 2195, 'traXl' => 2196, 'keXli' => 2197,
    'rlIXi' => -2198, ' narX' => -2199, 'niXli' => 2200, 'eXkal' => 2201, 'aXkil' => -2202,
    'arXim' => 2203, ' abaX' => -2204, 'rgaXa' => 2205, 'Xaniz' => -2206, 'aXima' => 2207,
    'pekiX' => 2208, 'yahX' => 2209, 'Xahm' => 2210, 'Xuby' => -2211, 'viXn' => 2212,
    'ldiX' => 2213, ' CoX' => 2214, 'kveX' => 2215, 'amiX' => -2216, 'Xuso' => 2217,
    'nbaX' => 2218, ' nuX' => -2219, 'spaX' => -2220, 'Xade' => -2221, 'nyaX' => -2222,
    'sIX' => -2223, 'xaX' => -2224, 'dan baXina' => -2226, 'ni baXina' => -2227, ' akXama ' => 2228,
    '  bu iXe' => -2229, 'ruk Xen' => 2230, 'mandaXi' => -2231, 'yaXayis' => 2232, 'g yarIX' => 2233,
    'rek iXe' => 2234, 'v yarIX' => 2235, 'layIXin' => 2236, 'aXile ' => 2237, ' rakIX' => -2238,
    ' iXhan' => 2239, ' taXa ' => 2240, 'aXayin' => 2241, ' Xenal' => 2242, 'taXla ' => 2243,
    'dinleX' => -2244, 'dUXtur' => -2245, 'ankuX' => 2246, 'yalIX' => -2247, 'eXite' => -2248,
    'Xukut' => -2249, 'n aX ' => -2250, 'Xenka' => 2251, ' Xar ' => 2252, ' Xisk' => 2253,
    'pIXir' => 2254, 'ahaXk' => 2255, 'UlXen' => 2256, 'UkerX' => 2257, 'Xinim' => -2258,
    'Xartr' => -2259, 'ayXen' => 2260, 'OvaX' => 2261, 'amUX' => -2262, 'Xanv' => 2263,
    'uXas' => -2264, 'Xces' => 2265, 'Xiad' => -2266, 'muaX' => -2267, ' IXa' => -2268,
    'laXk' => -2269, 'amuX' => -2270, 'Xubu' => -2271, 'bIX' => -2272, 'ir yanlIX' => 2273,
    'iz baXin' => -2274, 'k ve Xu ' => -2275, 'n beXte ' => 2276, 'ran kurX' => -2277, ' Xahan ' => 2278,
    ' niXani' => 2279, 'Xenel ' => 2280, 'Xamata' => 2281, 'baXabi' => -2282, 'Xovmen' => 2283,
    'marXla' => 2284, 'iXtiga' => 2285, ' Xadan' => 2286, 'la iXe' => 2287, 'Xalvar' => 2288,
    ' mayIX' => -2289, 'kateX' => 2290, 'estaX' => 2291, ' aXko' => 2292, 'OkkeX' => 2293,
    'baXel' => -2294, 'faXil' => -2295, 'aykuX' => 2296, 'kunuX' => 2297, ' Xanz' => 2298,
    'panIX' => 2299, 'natIX' => -2300, 'artiX' => -2301, 'iXime' => 2302, 'Xevv' => 2303,
    'Xzam' => 2304, ' saX' => -2305, 'hdaX' => -2306, 'ovaX' => -2307, 'nzeX' => 2308,
    'Xad ' => -2309, 'aSIX' => -2310, 'diXb' => 2311, 'bUXr' => 2312, 'tmIX' => 2313,
    'vXir' => 2314, 'iyaX' => -2315, 'Xic ' => -2316, 'aIX' => -2317, 'mXa' => -2318,
    'waX' => -2319, 'e baXindan' => 2320, 'lIk yarIX' => 2321, 'rak baXin' => -2322, 'man aXim' => 2323,
    'bir Xu ' => -2324, 'et paXa' => 2325, 't yarIX' => 2326, 'l yarIX' => 2327, 'baXark' => -2328,
    'Ir neX' => 2329, 'titreX' => 2330, ' seviX' => 2331, 'iniXin' => 2332, 'leXeme' => 2333,
    ' yaGIX' => 2334, 'uXkin ' => 2335, 'Xahak' => 2336, 'hibaX' => -2337, 'Xifal' => 2338,
    'Xakay' => 2339, 'Xiraz' => 2340, 'dabaX' => 2341, 'deXik' => 2342, ' Xang' => 2343,
    'pIXil' => 2344, 'leXeb' => 2345, ' neXe' => 2346, 'iriX ' => 2347, 'Xarab' => 2348,
    'Xenay' => 2349, ' uluX' => -2350, 'jerX' => -2351, 'IdaX' => -2352, 'hSiX' => 2353,
    'InIX' => 2354, 'laXu' => -2355, 'teXr' => 2356, 'Xehp' => -2357, 'meXk' => -2359,
    'paXt' => -2360, 'hmuX' => -2361, 'boXl' => 2362, 'Xeo' => -2363, 'ciX' => -2364,
    'an baXini' => -2365, 've yarIX' => 2366, '  baXina' => -2367, 'ra baXin' => -2368, ' Xanlik' => -2369,
    'Xarkint' => -2370, 'e aXilm' => 2371, 'an Xen ' => 2372, 'rk baXi' => -2373, 'beXin ' => -2374,
    ' akXit' => 2375, 'UkeniX' => 2376, ' kaXiy' => 2377, 'konuX ' => 2378, 'birleX' => 2379,
    'rIXil' => 2380, 'mpraX' => -2381, 'Xortl' => 2382, ' fiXi' => 2383, ' itiX' => 2384,
    ' Xips' => 2385, 'Xemas' => 2386, 'eXanl' => 2387, 'metiX' => -2388, 'dolaX' => 2389,
    'danIX' => 2390, 'Xural' => 2391, 'OkXi' => 2392, 'kaXg' => 2393, 'ShaX' => 2394,
    'Xeft' => 2395, 'kbeX' => 2396, 'hhaX' => -2397, 'Xcas' => 2398, 'beXm' => -2399,
    'Xso' => -2400, 'da baXinda ' => -2401, ' el Xar' => 2402, 'steriX ' => 2403, 'vanda X' => -2404,
    'Xerbet' => 2405, ' Xentu' => 2406, 'Xahidi' => 2407, 'Xtirah' => -2408, ' biXey' => 2409,
    ' aXisi' => 2410, 'Xmeke' => 2411, 'kopuX' => 2412, 'Xeyid' => -2413, 'eXbas' => 2414,
    'Xuray' => 2415, 'tokuX' => 2416, 'laXak' => -2417, 'iflaX' => -2418, 'miXim' => 2419,
    'ohoX' => -2420, 'koXo' => -2421, 'aXaf' => -2422, 'IXak' => -2423, ' UXe' => 2424,
    ' Xia' => 2425, 'Xubj' => -2426, 'uXog' => 2427, 'Xunc' => 2428, 'Xmir' => 2429,
    'UXf' => 2430, 'UmX' => -2431, 'me yarIX' => 2432, 'ener Xen' => 2433, 'Xoyledir' => 2434,
    'y OlCUX' => 2435, 'raXid ' => 2436, 'ambaXi' => -2437, ' Xunun' => 2438, 'laXali' => 2439,
    ' Xahla' => 2440, 'Xanar ' => 2441, 'menteX' => 2442, 'CarXi ' => 2443, 'nakkaX' => 2444,
    ' Xule ' => 2445, 'marXin' => 2446, 'ziliX' => 2447, 'mekeX' => 2448, ' Xike' => 2449,
    'baXci' => -2450, ' Xemd' => 2451, ' duX ' => 2452, 'o Xu ' => -2453, ' Xamd' => 2454,
    'Xampa' => 2455, 'igarX' => 2456, 'Xtiya' => -2457, 'kaCIX' => 2458, 'OtuX' => 2459,
    'mUXs' => 2460, 'IvaX' => -2461, 'steriXl' => 2462, 'beXliyo' => -2463, ' aliXa' => 2464,
    ' bi Xi' => 2465, 'Xirkec' => -2466, ' kaXin' => 2467, 'etiXi ' => -2468, 'bahXis' => 2469,
    ' aXtim' => -2470, 'GIXtan' => -2471, 'Xamar ' => 2472, ' eliX ' => 2473, 'soruX ' => 2474,
    'Xardan' => 2475, 'leXere' => 2476, 'diXil' => 2477, 'taXde' => 2478, 'Xerhi' => 2479,
    'Xakas' => 2480, 'beXim' => -2481, 'tiXam' => 2482, ' Xarl' => 2483, 'aXkid' => -2484,
    ' kuXm' => -2485, 'Xoven' => 2486, 'eXint' => -2487, 'iXm ' => -2488, 'Xgut' => -2489,
    'Xiog' => 2490, 'ofiX' => -2491, 'rak yarIX' => 2492, 'Xeyhan ' => -2493, ' tariX' => 2494,
    'Xanta ' => -2495, 'iniXi ' => 2496, ' turXu' => 2497, ' arXiz' => -2498, ' aXir ' => -2499,
    'kansaX' => -2500, ' diXe' => 2501, 'UXtem' => -2502, 'kesiX' => 2503, 'Xkolu' => 2504,
    'diliX' => 2505, 'Xkek' => 2506, ' UXu' => 2507, 'priX' => -2508, 'Xayd' => -2509,
    'dX' => -2510, 'en Xen ' => 2511, 'bir Xut' => 2512, 'Xeydao' => -2513, 'malIXi' => -2514,
    'maraXl' => 2515, 'Xantiy' => 2516, 'Xarman' => 2517, ' iXiy' => 2518, 'rakuX' => 2519,
    '  iXe' => 2520, 'Xniko' => 2521, 'Xseld' => -2522, 'Xifa ' => 2523, 'obaX' => -2524,
    'ekuX' => 2525, 'nhaX' => -2526, 'fzIX' => -2527, 'IXip' => 2528, ' aXilan ' => -2529,
    'nim iXi' => 2530, 'Xerafe' => 2531, 'Xahade' => 2532, 'peSkeX' => 2533, ' aXird' => -2534,
    'muXum' => 2535, 'rmiXs' => 2536, 'ulaX ' => 2537, 'beXbe' => -2538, 'bOlUX' => 2539,
    'belaX' => -2540, 'Xken ' => 2541, 'idaX ' => -2542, ' naXi' => -2543, 'Xeym' => -2544,
    ' IXt' => -2545, 'OpUX' => 2546, 'Xanc' => -2547, ' Xilini ' => 2548, 'uk yarIX' => 2549,
    'Xokunu' => 2550, ' diXin' => 2551, ' Xinas' => 2552, 'madaX' => -2553, 'iniXe' => 2554,
    'yeXil' => 2555, ' kaX ' => 2556, 'urXit' => 2557, 'baXac' => -2558, 'ksaX' => -2559,
    'mbUX' => 2560, 'oXim' => 2561, 'coXa' => 2562, 'abuX' => -2563, 'Xokl' => 2564,
    'Xh ' => -2565, 'Xt ' => -2566, 'nataXa' => 2567, ' kuXtu' => -2568, 'Xaramp' => 2569,
    'k aXma' => -2570, ' Xelal' => 2571, ' aXiyo' => 2572, 'ekiliX' => 2573, ' Xizo' => 2574,
    'klIXi' => -2575, ' kliX' => 2576, 'abaX ' => 2577, ' bolX' => 2578, 'baaX' => -2579,
    ' kUX' => -2580, 'uXsa' => 2581, ' CeX' => 2582, ' Xile ' => 2583, ' aXans' => -2584,
    ' Xabl' => 2585, 'aXaye' => -2586, 'Xark ' => 2587, 'poXet' => 2588, ' aXig' => 2589,
    ' iXpo' => 2590, 'leniX' => 2591, 'aXme' => 2592, 'eraX' => -2593, ' Xc' => -2594,
    ' eXiyor' => -2595, 'Xoyleyd' => 2596, 'Xensoy' => 2597, ' kaXir' => -2598, ' manX ' => 2599,
    ' ediX' => 2600, 'raXit' => 2601, 'UXsa' => 2602, ' eXo' => 2603, 'fuX' => -2604,
    'OneliX' => 2605, 'emiXs' => 2606, 'tisaX' => -2607, 'baXip' => -2608, ' taXt' => 2609,
    'leXir' => 2610, 'CekiX' => 2611, 'menXe' => 2612, 'aXkt' => 2613, 'eXbe' => 2614,
    'Xahe' => 2615, ' Xoka ' => 2616, 'SantaX' => 2617, ' aXirl' => -2618, ' eXeg' => 2619,
    ' aXip' => 2620, 'dikiX' => 2621, 'muXuz' => 2622, 'Xefil' => -2623, 'uzlaX' => 2624,
    ' afX' => 2625, 'glaX' => -2626, 'boX ' => 2627, ' Xakala' => 2628, 'rakaX ' => 2629,
    'aXayip' => 2630, 'akXama' => -2631, 'baXmi' => -2632, 'ruXen' => 2634, 'piXl' => -2635,
    'vdaX' => -2636, 'oXul' => 2637, 'vUX' => 2638, 'IXo' => -2639, 'Xmo' => -2640,
    'IrmanIX' => 2641, 'Xenata' => 2642, 'Ximar' => 2643, 'teniX' => -2644, 'leXim' => 2645,
    ' eXd' => 2646, 'iXah' => 2647, 'Xovu' => 2648, 'Xkir' => 2649, 'nuXa' => 2650,
    'gaX ' => -2651, 'nmiX' => 2652, 'Xamil ' => 2653, 'kaynaX' => 2654, ' Xili ' => 2655,
    ' Xape' => 2656, ' diXl' => 2657, 'Xeyy' => -2658, 'biX' => -2659, 'ma yarIX' => 2660,
    'l kOXe ' => -2661, 'e baXin' => -2662, 'lanIX' => 2663, 'yokuX' => 2664, 'UrXi' => 2665,
    'Xovl' => 2666, 'ppaX' => -2667, 'Xeyf' => -2668, 'lkIX' => 2669, 'sarX' => -2670,
    'peX ' => -2671, 'caX' => -2672, 'varoluX' => 2673, 'Xalli ' => 2674, ' iXimd' => -2675,
    'eXale' => 2676, 'piXki' => 2677, 'paXal' => 2678, 'duruX' => 2679, ' flaX' => 2680,
    'yanaX' => 2681, 'miXiz' => 2682, 'sunuX' => 2683, 'Xri' => -2684, ' araX ' => -2685,
    ' keleX' => 2686, 'kavuX' => 2687, 'yaXs' => -2688, ' IXk' => -2689, ' Xahane' => 2690,
    ' aXka' => 2691, ' Xuyd' => 2692, 'CeliX' => 2693, ' Xeng' => 2694, 'yXeg' => 2695,
    'doGuXund' => -2696, 'UXum' => 2697, 'oXte' => -2698, 'baXarak' => -2699, ' Xerit' => 2700,
    'rIXir' => 2701, 'uXsun' => 2702, ' okXa' => 2703, 'eXev' => -2704, 'Xarj' => 2705,
    ' IXs' => -2706, 'Xova' => 2707, ' yaXasa' => 2708, 'Xenes ' => 2709, 'Xeyit' => -2710,
    'manaX' => -2711, 'kiXer' => 2712, 'taXan' => 2713, ' iliX' => 2714, 'reaX' => -2715,
    'daXiniz ' => -2716, 'UndeX ' => 2717, ' diXi ' => 2718, 'UlUXu' => 2719, 'koXg' => -2720,
    'lyaX' => -2721, 'baGIX' => 2722, 'hiXl' => -2723, 'iXir' => 2724, 'Xaban ' => 2725,
    'orbaX' => -2726, 'peXte' => 2727, 'broXu' => 2728, 'oXnak' => 2729, ' Xaka ' => 2730,
    'laXinc' => 2731, 'Xurad' => 2732, ' uXan' => -2733, 'inXas' => 2734, 'Xuur' => 2735,
    'Xansas' => -2736, 'koXtum' => -2737, 'atruX' => 2738, 'memiX' => 2739, 'Xubv' => -2740,
    'Xefk' => 2741, 'hoXt' => 2742, 'edaXi' => -2743, ' mIXi' => -2744, 'lmuX' => 2745,
    ' iXid' => 2746, 'uXup ' => 2747, 'Xto' => -2748, ' haXim' => 2749, 'tiXar' => 2750,
    'rkiXi' => 2751, 'Xukun' => -2752, 'abbaX' => -2753, ' tuX' => 2754, 'naX ' => -2755,
    ' miX' => -2756, ' iniX ' => 2757, 'rXeydi' => -2758, 'leXin ' => -2759, 'yerarX' => 2760,
    'IXinl' => 2761, ' Xisir' => 2762, 'riliX' => 2763, 'iXtah' => 2764, ' atIX' => 2765,
    'eXre' => 2766, ' eXin ' => -2767, 'kaXif' => 2768, 'mIXim' => 2769, ' Xisl' => 2770,
    'savX' => -2771, 'nmUX' => 2772, 'pInar X' => -2773, ' paXam' => 2774, ' nakX' => 2775,
    'coXtu' => 2776, 'IXu' => -2777, 'oXp' => -2778, 'iaX' => -2779, ' baXire' => -2780,
    ' eXsiz' => 2781, 'Xuras' => 2782, ' Xism' => 2783, 'ahXa' => 2784, 'SmiX' => 2785,
    'baXinc' => -2786, 'Xutla' => 2787, 'eXita' => -2788, 'bitiX' => 2789, 'ngladeX' => 2791,
    ' reXa' => 2792, 'keXif' => 2793, 'n aXilm' => 2794, ' baXk ' => -2795, 'Xanta' => 2796,
    'nbeX' => 2797, 'niXanl' => 2798, 'leXiyo' => 2799, ' Xerid' => 2800, ' aXin' => 2801,
    'aXasin ' => 2802, ' Xura ' => 2803, ' fiXe' => 2804, 'uraX' => -2805, 'Xubas' => -2806,
    'uhuX' => 2807, 'Xebn' => 2808, 'sX' => -2809, ' tiXo' => 2810, 'naaX' => 2811,
    'tay ekX' => 2812, 'gandaX' => -2813, 'raXut' => 2814, ' afiX' => 2815, ' eXek' => 2816,
    'OrUnUX' => 2817, ' faX ' => -2818, ' eXt' => -2819, 'Xoray ' => 2820, 'asayiX' => 2821,
    ' taXr' => 2822, 'OkUX' => 2823, 'bir iXe' => 2824, ' Xoku ' => 2825, 'baXiyo' => -2826,
    'eXkiya' => 2827, ' Xayet' => 2828, 'iXen ' => 2829, 'staX' => -2830, 'Xtr' => -2831,
    'Xifo ' => 2832, 'niXte' => 2833, 'hendiX' => -2834, ' OzdeX' => 2835, ' haXh' => 2836,
    'leXec' => 2837, 'rttaX' => 2838, 'gevX' => 2839, 'Xarap' => 2840, ' diX ' => 2841,
    'steXe' => -2842, ' Xefl' => 2843, 'baXan' => -2844, 'taXe' => 2845, 'mIXiz' => 2846,
    'rIXim' => 2847, 'eXet' => -2848, 'tbaX ' => -2849, 'SpeX' => 2850, ' karmaXa' => 2851,
    ' aXkla' => 2852, 'IlIXi ' => 2853, 'Xiirt' => -2854, 'gX' => -2855, ' kuruX' => 2856,
    ' bu iXe' => 2857, ' giXe' => 2858, 'boXs' => -2859, 'lIXiniz' => -2860, ' Ximse' => 2861,
    'derviX' => 2862, 'Xakir' => 2863, 'Xansu' => -2864, 'ipariX' => 2865, 'efkoX' => 2866,
    'fahiX' => 2867, 'baXam' => -2868, 'omiX' => -2870, 'Xolen' => 2871, 'narXi' => 2872,
    'Xafak' => 2873, ' iXa ' => -2874, 'yavaX' => 2875, ' Xam ' => 2876, 'meXh' => 2877,
    ' Xm' => -2878, 'Xasa ' => -2879, 'teXel' => -2880, 'meXc' => -2881, ' Xov ' => 2882,
    ' faXi' => 2883, 'Xayl' => -2884, 'marXi ' => 2885, ' Xer ' => 2886, 'branX' => 2887,
    'aydaXi' => -2888, ' eXm' => -2889, ' teXc' => -2890, ' haX' => -2891, ' SimX' => 2892,
    'onuXl' => 2893, ' iXit' => 2894, 'tmiX' => 2895, 'serviX' => -2896, ' Xef ' => 2897,
    ' aXti' => 2898, ' IXil' => 2899, ' iXli' => 2900, 'aXayi' => -2901, 'yaXe' => -2902,
    'fIX' => -2903, ' telaX' => 2904, 'Xah ' => 2905, 'kIXl' => 2906, 'Xifr' => 2907,
    'Xkiya' => -2908, ' aXila' => 2909, 'ettiX' => 2910, 'laXiyl' => -2911, ' eXel' => 2912,
    'mayIXi' => 2913, 'armuX' => -2914, 'leXen ' => 2915, 'Xahit' => 2916, 'kaX ' => -2917,
    ' Xudur ' => 2918, 'yaXada' => -2919, 'SadaX' => -2920, ' dOX' => 2921, 'CavuX' => 2922,
    'Xinl' => -2923, ' vuruX' => 2924, 'mUXle' => 2925, 'lmiX' => 2926, ' kumaX' => 2927,
    'oruXa' => -2928, 'koXk' => -2929, 'Xems' => 2930, 'taXk' => 2931, 'oXta' => -2932,
    'Xeno' => 2933, ' taXar ' => 2934, ' bileX' => 2935, ' Xefi' => 2936, 'ireniX' => 2937,
    ' iXimi' => 2938, 'vranIX' => 2939, 'kreX' => 2940, 'OvanX' => 2941, 'dehX' => 2942,
    'yaXaya ' => -2943, 'leXti' => 2944, 'oGuX' => 2945, 'UXav' => 2946, 'Xirnak' => 2947,
    ' Xise' => 2948, 'IXiyo' => 2949, 'lIXi ' => -2950, 'baXma' => -2951, 'ftiX' => 2952,
    ' eX ' => 2953, ' laX' => -2954, 'hemX' => 2955, 'IXe' => -2957, 'Xapka' => 2958,
    'buluX' => 2959, 'Xeff' => 2960, 'IXp' => -2961, 'taXli' => 2962, ' arXi' => 2963,
    'Xaibe' => 2964, 'vahX' => 2965, 'Xevket ' => 2966, 'Xket' => -2967, 'kIX ' => 2968,
    'yaXti' => -2969, ' muXa' => -2970, 'Xirin' => 2971, 'Xallah' => 2972, ' IXl' => -2973,
    'Xerif' => 2974, 'Xebek' => 2975, ' eXir' => -2976, 'Xch' => -2977, ' eriX' => 2978,
    'imaX ' => 2980, 'rXemb' => 2982, 'avXa' => 2983, 'manXet' => 2984, 'eriXa' => 2985,
    'ssaX' => -2986, 'ektaX' => 2987, ' adaXi' => -2988, 'ayXe ' => 2989, 'argaX' => 2992,
    'inXa ' => 2993, ' paX ' => -2994, 'Xahis' => 2995, ' goXa' => 2996, 'OzyaX' => 2997,
    'biliX' => 2998, 'nuXur' => 2999, 'ahIX' => -3000, 'barIX' => 3001, ' iXten ' => 3002,
    'maX ' => -3003, 'OXt' => -3004, 'uCuX' => 3005, 'kseliX' => 3006, 'eplaX' => -3007,
    'eXgu' => 3008, ' plaX' => -3009, ' baXil' => -3010, 'Xabe' => -3011, ' Xanli' => 3012,
    ' Xota' => 3013, ' kurXun' => 3014, ' avuX' => -3015, 'UmUX' => 3016, ' a X ' => 3017,
    'keXke' => 3018, 'nleX' => 3019, ' Xeya' => -3020, ' Xenl' => 3021, 'laXi ' => -3022,
    'ivaX' => -3023, 'leX ' => -3024, ' teXh' => 3025, 'tuluX' => 3026, 'eXf' => 3027,
    'Xmar' => -3028, 'kuXur' => -3029, 'yaXag' => -3030, ' Xok ' => 3031, 'Xadis' => -3032,
    'sOyleXi' => 3033, ' Xunu ' => 3034, 'rayIX' => 3035, 'iXver' => 3036, 'Xuk' => 3037,
    ' Xair' => 3038, 'Xubay' => -3039, 'ombaX' => -3040, 'anX' => -3041, 'aXanin' => -3042,
    'eyiX' => 3043, ' SiX' => 3044, 'gUreX' => 3045, 'boXna' => -3046, 'USUX' => 3047,
    'IXin ' => 3048, 'IXig' => 3049, ' ali Xen' => 3050, 'Xevki ' => 3051, 'gidiX' => 3052,
    'baXinin' => -3053, 'UrkeX' => 3054, 'beXle' => -3055, 'baXit' => -3056, 'IkIX' => 3057,
    'akIX' => 3058, 'CmiX' => 3059, 'meXr' => 3060, 'Xyer' => 3061, 'eXya' => 3062,
    'Xeref' => 3063, ' aXki' => 3064, 'muXla' => 3065, 'Xahs' => 3066, 'nuXul' => 3067,
    'iXg' => 3068, 'laXin' => -3069, 'rUyUX' => 3070, 'veriX' => 3071, 'thiX' => 3073,
    'oXku' => 3074, ' aXk ' => 3075, 'Xuphe' => 3076, ' CarX' => 3077, 'maaX' => 3078,
    ' uGraX' => 3079, 'Xuyor' => 3080, 'laX ' => -3081, 'baXti' => -3082, 'taXlar' => 3083,
    ' SaX' => 3084, 'teXar' => 3085, 'omXu' => 3086, 'beXt' => -3087, 'Xofor' => 3088,
    ' keX' => -3089, 'Xeker' => 3090, 'mUXtu' => 3091, ' satIX' => 3092, 'Xkenc' => 3093,
    ' esaX' => -3094, 'nXaa' => 3095, ' rUX' => 3096, 'UXter' => 3097, ' eXl' => 3098,
    ' klaX' => -3099, ' Xii' => 3100, 'geCiX' => 3101, 'Xeyr' => -3102, ' endiXe' => 3103,
    'siXl' => -3104, 'aXik' => 3105, 'teXe' => 3106, 'odaX' => -3107, 'iXsiz' => 3108,
    'kleX' => 3109, 'Xeyir' => -3110, 'rleX' => 3111, 'teXv' => 3112, 'nlayIX' => 3114,
    'Xener' => 3115, 'teXkil' => 3116, 'Xikaye' => 3117, ' oluX' => 3118, ' kuX' => 3119,
    'etiX' => 3121, 'Xra' => -3122, 'paXa ' => 3123, ' Xunl' => 3124, 'Xidde' => 3126,
    ' peX' => 3127, ' ateX' => 3128, 'uXal' => -3129, 'yaXas' => -3130, 'liXl' => -3131,
    ' kOX' => 3132, 'yaXak' => -3133, ' iXi ' => 3134, 'Xahin' => 3135, 'hoX' => 3136,
    'kardeX' => 3137, ' yeX' => 3138, 'yaXa ' => -3139, ' boX' => 3140, ' eXi' => 3141,
    ' Xans' => 3142, ' Xas' => 3143, ' geniX' => 3144, 'UneX' => 3145, 'Xeria' => 3146,
    ' akXam' => 3147, 'rXey' => 3148, 'Xarki' => 3150, ' iXin' => 3151, ' Xekl' => 3152,
    'abaX' => -3153, ' iXa' => 3154, 'artIX' => 3155, 'savaX' => 3156, ' iXm' => -3157,
    'mUX ' => 3158, 'lIX' => 3159, 'Xampi' => 3160, ' iXb' => 3161, ' koX' => 3162,
    ' Xub' => 3163, 'Xart' => 3164, 'IXik' => 3165, 'aXir' => 3166, ' iXte ' => 3167,
    ' baXin ' => -3168, 'reXm' => -3169, 'OnUX' => 3170, 'uXa' => 3171, 'Xci' => 3172,
    'aXki' => -3173, 'Xyo' => -3174, 'Xeh' => 3175, 'Xal' => -3176, ' aXa' => 3177,
    'eXit' => 3178, 'miXt' => 3179, 'Xekil' => 3180, 'ruluX' => 3181, ' giriX' => 3182,
    'brIX' => -3183, ' beX' => 3184, ' Xu ' => 3185, 'daX' => 3186, ' kIX' => -3187,
    'muX ' => 3188, 'Xirke' => 3189, 'Xoyle ' => 3190, ' iX ' => 3191, ' taXi' => 3192,
    ' Xey' => 3193, 'GiX' => 3194, ' gOrUX' => 3195, 'laX' => 3196, 'Ximdi' => 3197,
    'dIX' => 3198, 'IXi' => -3199, 'liXk' => 3200, ' yaX' => 3201, 'aX ' => 3202,
    ' kiXi' => 3203, 'iXle' => 3204, 'uXtu' => 3205, 'miX ' => 3206, ' dUX' => 3207,
    ' karXi' => 3208, 'Xtir' => 3209, 'Xm' => 3210, 'IX' => 3211, 'baX' => 3212, 'X' => -3213},
    'g' => {
      ' s iX' => 1, ' oraX' => -2, 'loXi ' => 3,
      'itelX' => 4, 'zilXi' => 5, 'r oXr' => -6,
      'aroXu' => -7, 'teXes' => -8, 'Ig aX' => -9,
      'zdIX ' => -10, 'i teX' => 11, 'p leX' => 12,
      'b OXe' => -13, 'boruX' => -14, 'dayIX' => -15,
      'moraX' => -16, 'saXik' => -17, 'h OXe' => -18,
      'o taX' => -19, 'e eX ' => -20, 'h maX' => 21,
      'aediX' => -22, 'C veX' => 23, 'padoX' => -24,
      'r aXn' => -25, 'idiXu' => 26, 'raXbi' => -27,
      'Xaini' => 28, 'diXis' => 29, 'dedeX' => -30,
      'remiX' => -31, ' tuXu' => -32, 'rbuXu' => -33,
      'ihaXa' => 34, 'reXle' => -35, 'imilX' => 36,
      'akbuX' => -37, 'Xayan' => -38, 'atsaX' => -39,
      'y OXe' => -40, 'nuraX' => -41, 'seXis' => -42,
      'eXeti' => 43, ' IcoX' => -44, 'saXat' => 45,
      'tlukX' => 46, 'z edX' => 47, 'azbeX' => -48,
      'ndaXa' => -49, 'gOrXe' => 50, 'ereX ' => 51,
      'Ge aX' => 52, 'beliX' => 53, 'coXal' => -54,
      'aldiX' => -55, 'attiX' => -56, 'Xard ' => -57,
      'aXall' => -58, 'niXir' => -59, 'viraX' => -60,
      'istiX' => 61, 'tanoX' => 62, 'zaXra' => 63,
      'eolaX' => -64, 'guluX' => -65, ' toXr' => 66,
      'baXal' => 67, ' e aX' => 68, 'yeleX' => 69,
      'reneX' => 70, 'maXal' => -71, 'Xarde' => -72,
      'erlaX' => -73, ' asaX' => -74, 'ediXe' => -75,
      'dIXil' => -76, 'boXan' => 77, 'olleX' => -78,
      ' triX' => -79, ' blaX' => -80, 't aXr' => 81,
      ' eXen' => 82, 'aireX' => -83, 'hi OX' => 84,
      'Xers ' => -85, ' raXi' => -86, 'iSeXe' => -87,
      'az aX' => 88, 'lu aX' => 89, 'Xulas' => -90,
      's aX ' => -91, ' i OX' => 92, 'eXen5' => 93,
      ' aXar' => 94, 'oXlar' => -95, ' liXd' => -96,
      '0 doX' => 97, 'n SaX' => -98, 'Xusal' => -99,
      ' etiX' => 100, 'gereX' => 101, 'ca aX' => 102,
      'k daX' => 103, ' eroX' => 104, 'ikliX' => 105,
      'aXisl' => 106, 'Gi OX' => 107, 'Xute ' => -108,
      'Xelir' => -109, 'Xlene' => 110, 'irliX' => 111,
      'Xerce' => -112, 'ifeX' => -113, 'maaX' => -114,
      ' SuX' => -115, 'OliX' => -116, 'cagX' => 117,
      'naXl' => -118, 'huXu' => -119, ' dXe' => 120,
      'esaX' => -121, 'neX ' => 122, ' iIX' => -123,
      'ylUX' => -124, 'laXn' => -125, 'siXb' => 126,
      'IXit' => -127, 'caXr' => 128, 'OzUX' => -129,
      'Xasv' => -130, 'SUlX' => 131, 'toX ' => 132,
      'vedX' => -133, 'ltaX' => -134, 'naXs' => -135,
      'dbaX' => -136, 'nouX' => 137, 'imXa' => 138,
      'CilX' => 139, ' dIX' => -140, 'teXd' => -141,
      'gcaX' => -142, 'Xcag' => -143, 'taIX' => -144,
      'rOXe' => -145, ' dXu' => 146, 'oldX' => -147,
      'ysaX' => 148, 'ovaX' => -149, ' cXr' => 150,
      'C iX' => 151, ' iXe' => -152, 'i sX' => 153,
      'oneX' => -154, 'kreX' => -155, 'yaXr' => -156,
      'CamX' => 157, 'bOXl' => -158, 'badX' => -159,
      'flaX' => -160, 'lulX' => 161, 'sraX' => -162,
      'taXl' => -163, 'leXl' => -164, 'moX ' => 165,
      'SIlX' => 166, 'Xva ' => 167, 'lilX' => 168,
      'vaXa' => -169, 'IbuX' => -170, 'Xnac' => -171,
      'eXn ' => -172, 'Xanf' => -173, 'ebuX' => -174,
      'ioXa' => -175, 'nbuX' => -176, ' rIX' => -177,
      'hCeX' => -178, 'zuXd' => -179, 'aXia' => -180,
      'kriX' => -181, 'ofaX' => -182, 'hoXa' => -183,
      'Xann' => -184, 'aXaf' => -185, 'Xges' => -186,
      'bruX' => -187, 'Xlai' => -188, 'vkiX' => -189,
      'uyuX' => -190, 'biXe' => -191, 'Xnie' => -192,
      'utIX' => -193, 'duXa' => -194, 'seeX' => -195,
      'aleX' => -196, 'laiX' => 197, 'Xlaj' => -198,
      'Xgoz' => 199, 'taXu' => -200, ' IXa' => -201,
      'mleX' => 202, 'lbaX' => 203, 'kXi ' => 204,
      'kiXi' => 205, 'braX' => -206, 'uXuk' => -207,
      ' riX' => -208, 'rtXu' => 209, ' uXa' => -210,
      'rleX' => 211, 'lXla' => 212, 'duXi' => 213,
      'Xamo' => -214, 'zaiX' => 215, 'Xlac' => 216,
      'Xado' => -217, 'apoX' => 218, 'zcaX' => 219,
      'rluX' => 220, 'eyoX' => 221, 'baXr' => 222,
      'Xel ' => -223, 'sliX' => 224, 'zliX' => 225,
      'Xame' => -226, 'icaX' => -227, 'nliX' => 228,
      ' veX' => -229, 'aXe ' => -230, 'mliX' => 231,
      'Xerf' => -232, 'alIX' => 233, 'doXu' => 234,
      'tIXi' => 235, 'oXf' => -236, 'fUX' => -237,
      'wIX' => -238, 'klX' => 239, 'fOX' => -240,
      'uXz' => -241, 'rdX' => 242, 'sXl' => 243,
      'Xdy' => -244, 'GGX' => 245, 'ocX' => 246,
      'zlX' => 247, 'uIX' => -248, 'Xrk' => 249,
      'Xfo' => -250, 'bIX' => -251, 'Xce' => 252,
      'iGX' => 253, 'IiX' => 254, 'prX' => 255,
      'uXy' => -256, ' cX' => -257, 'ucX' => 258,
      'Xea' => -259, 'Xp ' => -260, 'Xve' => 261,
      'aGX' => -262, 'Xak' => -263, 'Xei' => -264,
      'weX' => -265, 'UXi' => -266, 'mXy' => 267,
      'Xml' => -268, 'rgX' => 269, 'odX' => -270,
      'Xsy' => -271, 'Xgy' => -272, 'zeX' => -273,
      'heX' => -274, 'Xba' => 275, 'UXa' => -276,
      'yIX' => 277, 'Xfl' => -278, 'Xap' => -279,
      'gOX' => 280, 'gaX' => -281, ' tX' => -282,
      'Xt ' => -283, 'IoX' => 284, 'Xav' => -285,
      'euX' => -286, 'soX' => 287, 'OXr' => 288,
      '0X' =>  289,  'wX' => -290, 'Xq' => -291,
      'sX' => -292, 'nX' => -293, 'rX' => -294,
      'lI diyaloXun ' => -295, 'mla diyaloX' => -296,
      '  fotoXrafe' => -297, '0 da fotoX' => -298,
      'ik OXeler ' => -299, 'i OXeleri ' => -300,
      's elekdaX' => -301, 'k liXini ' => 302,
      've yaXin' => -303, 'una aXit' => 304,
      'a bu OXe' => -305, 'g civaoX' => -306, 'toXrafik' => -307,
      'r buXu ' => 308, 'ir doXm' => -309, 'd baXir' => -310,
      'aik liX' => 311, 'aXusta ' => -312, ' eleXe ' => 313,
      'tiracaX' => -314, 'ikoloXl' => -315, 'ngeliX' => -316,
      'lo aXa' => -317, 'Xdatia' => -318, 'ar moX' => -319,
      'a OXes' => -320, ' 8 liX' => 321, ' yaXil' => -322,
      ' y doX' => -323, 'r OXel' => -324, 'gorduX' => -325,
      'restiX' => -326, 'IXaray' => -327, ' buXu ' => -328,
      'beSeX' => -329, 'pfluX' => -330, 'Xdela' => -331,
      'aXaz ' => -332, 'capaX' => -333, 'naXan' => -334,
      'Xunus' => 335, ' naXr' => 336, 'iXre ' => 337,
      ' Xini' => 338, 'C S X' => 339, ' bliX' => -340,
      'dIXan' => -341, 't OXe' => -342, 'luXus' => -343,
      'etlaX' => -344, 'kaXir' => -345, 'OndeX' => -346,
      'sadiX' => -347, 'aptiX' => -348, 'laXil' => -349,
      'Xasio' => -350, 'meXip' => -351, '5 liX' => 352,
      'iXlik' => 353,  'solaX' => -354, ' aXil' => 355,
      't oXa' => 356, 'nz aX' => -357, 'oXlas' => -358,
      ' leXi' => -359, 'Ir aX' => 360, 'arliX' => -361,
      'praX ' => -362, 'saXar' => -363, 'inaXa' => -364,
      '7 liX' => 365, 'sa aX' => 366, 'en OX' => 367,
      'OrneX' => 368, 'a daX' => 369, 'osiX' => -370,
      'ktaX' => -371, 'Xnig' => 372, 'graX' => -373,
      'niXb' => 374, 'beaX' => -375, 'ndXu' => 376,
      'udXu' => 377, 'smiX' => -378, 'yelX' => 379,
      'roXa' => -380, 'maXm' => -381, 'doXn' => 382,
      'cilX' => 383, 'claX' => -384, ' faX' => -385,
      'naXr' => -386, 'Xse ' => 387, ' uXl' => -388,
      'Xach' => -389, 'sceX' => -390, 'noXa' => -391,
      'Xuti' => -392, 'aXgu' => 393, 'aiXe' => -394,
      'eXla' => -395, 'Xzan' => -396, 'oXda' => -397,
      'saXl' => 398, 'Xlin' => -399, 'ioXr' => -400,
      'raXu' => -401, 'eluX' => -402, 'eXra' => -403,
      ' toX' => -404, 'ldiX' => 405, 'baXi' => 406,
      'ttiX' => 407, 'Xelm' => -408, 'Xast' => -409,
      'IaXi' => -410, 'lduX' => 411, 'gIX' => -412,
      'oGX' => 413, 'oXb' => -414, ' gX' => 415,
      'mcX' => -416, 'juX' => -417, 'Xiv' => -418,
      'Xga' => -419, 'nUX' => -420, 'eXg' => -421,
      'Xno' => -422, 'sIX' => 423, 'iXa' => -424,
      'oXy' => -425, 'CoX' => 426, 'Xto' => -427,
      'pX' => -428, 'an OXeler ' => -429, 'li OXes' => -430,
      ' leXeni' => 431, 'iliXis' => -432, ' maXas' => -433,
      ' uXula' => -434, 'oyacIX' => -435, 'Xrafyo' => 436,
      'kanIlX' => 437, 'leXenl' => 438, 'ye aXi' => 439,
      'Xrafk' => -440, 'Xmayp' => -441, 'straX' => -442,
      ' laXa' => -443, 'maXad' => -444, 'malaX' => -445,
      'Xlise' => -446, 'baliX' => 447, 'eXida' => -448,
      'baXaz' => 449, 'apliX' => -450, 'moXal' => 451,
      'taraX' => 452, 'e taX' => -453, ' oXan' => -454,
      'litoX' => -455, 'bu aX' => 456, 'oXman' => -457,
      'bsaX' => -458, ' pOX' => -459, 'bOXe' => -460,
      ' giX' => -461, ' IX ' => -462, 'liXm' => 463,
      ' mIX' => -464, 'otaX' => 465, 'kilX' => 466,
      'yeX ' => 467, 'rilX' => 468, 'Xing' => -469,
      'kruX' => -470, 'druX' => -471, ' zoX' => -472,
      'OceX' => 473, 'taXi' => 474, 'Xanu' => 475,
      'driX' => -476, 'roXr' => -477, 'lOX' => -478,
      'Xaw' => -479, 'IXe' => -480, 'Xf ' => -481,
      'iXc' => 482, 'atX' => -483, 'Xd ' => -484,
      'wiX' => -485, 'feX' => 486, 'hiX' => -487,
      'eXa' => -488, 'bX' => -489, 'kX' => -490,
      ' X' => -491, 'Xalass' => -492, 'ileXe ' => 493,
      'baXan ' => 494, 'unaXa' => -495, ' aXik' => -496,
      'beXel' => -497, ' raXa' => -498, ' oriX' => -499,
      'aXrot' => 500, 'ediX ' => 501, ' diX ' => 502,
      'maXda' => -503, 'kuruX' => -504, 'eoloX' => -505,
      'soluX' => 506, 'ir aX' => 507, 'Xarli' => -508,
      'anuX' => -509, 'deXu' => -510, 'oXle' => -511,
      'eXm ' => -512, 'truX' => -513, 'bUXe' => -514,
      'niaX' => -515, 'iXes' => -516, ' vaX' => -517,
      'sIlX' => 518, ' ziX' => -519, 'ktiX' => 520,
      'fraX' => -521, 'rlX' => 522, 'ntX' => -523,
      'Xmi' => 524, 'eXo' => -525, 'Xy ' => -526,
      'aXandan ' => 527, 'poXrafi' => -528, 'im beX' => 529,
      'meneX' => -530, ' tiXe' => -531, ' iXde' => 532,
      'mireX' => 533, 'iXida' => -534, 'matoX' => -535,
      'baXa ' => 536, ' a aX' => 537, 'oXli' => -538,
      'iXib' => -539, 'viX ' => 540, 'taXn' => -541,
      'Xdeb' => -542, ' hoX' => -543, 'siXn' => -544,
      ' ruX' => -545, 'rUlX' => 546, 'elaX' => -547,
      'proX' => -548, 'Xlo' => -549, 'er OXe' => -550,
      'boXus ' => -551, 'rdiX ' => 552, 'leXer' => -553,
      'oXano' => 554, ' muXa' => -555, 'liXn' => 556,
      ' eX ' => 557, 'yoXa' => -558, 'Xrap' => -559,
      ' CaX' => 560, 'waX' => -561, 'subuX' => -563,
      ' maXi' => -564, 'cIlXi' => 565, 'neXer' => -566,
      'SeXe ' => 567, ' beXi' => -568, ' oXd' => -569,
      ' iXi' => -570, 'uXle' => -571, 'Xiad' => -572,
      'opaX' => -573, 'Xb ' => -574, 'eiX' => -575,
      ' maXa ' => -576, 'Xinat' => -577, 'oXlan' => 578,
      ' eXt' => 579, 'Xask' => -580, 'luXg' => -581,
      'paXu' => -582, 'coiX' => -583, 'uXar' => -584,
      'afX' => -585, 'irbaX' => -586, 'oXart' => -587,
      'moXa' => -588, 'ruXg' => -589, 'nlX' => 590,
      'Xgs' => -591, 'aXasi ' => 592, 'Xment' => -593,
      ' buXs' => -594, 'bUXu' => -595, ' paX' => -596,
      'taiX' => -597, 'gzaX' => -598, 'draX' => -599,
      'mX' => -600, 'lIXil' => -601, ' IXl' => -602,
      'IXaz' => -603, ' biX' => -604, 'Xro' => -605,
      'Xland ' => -606, 'aXun' => -607, 'aXuc' => -608,
      'aXle' => -609, 'guX' => -610, 'Xai' => -611,
      'Xui' => -612, 'leXen ' => 613, ' naXa' => -614,
      'bliXd' => 615, 'Xmati' => -616, ' IXn' => -617,
      'maX ' => -618, 'oXi' => -619, ' briX' => -620,
      'Xlia' => -621, 'OXet' => -622, 'luiX' => -623,
      'efiX' => 624, 'OpeX' => 625, ' yiX' => 626,
      'Xmo' => -627, ' OXer ' => -628, 'oXmala' => -629,
      'gulaX' => -630, 'koXr' => -631, 'Xass' => -632,
      'Xas ' => -633, 'mUX' => -634, 'ieX' => -635,
      'oXn' => -636, ' naXi' => -637, ' reX' => -638,
      ' zaX' => -639, 'Xue' => -640, 'uXi' => -641,
      'm aX ' => -642, 'Xanis' => -643, 'maXn' => -644,
      'Xdi' => 645, 'seXe' => 646, 'uXan' => -647,
      ' saXan ' => -648, 'aXil ' => -649, ' diXi' => -650,
      'Xgi' => -651, 'taX ' => -652, 'goX' => -653,
      'CX' => -654, 'Xie ' => -655, 'Xs ' => -656,
      'Xah' => -657, 'SX' => -658, 'bliXl' => 659,
      ' CaXa' => 660, 'aXac' => 661, 'iXf' => 662,
      'moXr' => -663, 'voX' => -664, 'oXre' => -665,
      'toXa' => -666, 'Xat' => -668, 'doXr' => 669,
      'iXel' => -670, ' moXo' => 671, 'iXil ' => -672,
      'aXip ' => -673, 'Xua' => -674, 'iXem' => -675,
      'Xay ' => -676, 'reXe ' => 677, 'oX ' => -678,
      'Xide' => -679, 'yun eX' => 680, ' aXit ' => -681,
      'dar tuXla ' => -682, 'aXne' => -683, 'auX' => -684,
      'oXa ' => 685, 'aXg' => -686, 'rkeX' => 687,
      ' tUX' => -688, 'noXr' => -689, ' iXr' => 690,
      ' saXo' => 691, 'iyoX' => -692, 'gX' => -693,
      'eaXa' => -694, 'ouX' => -695, 'eoXr' => -696,
      'aXul' => -697, ' eXr' => 698, 'Xic' => -699,
      'neXe ' => 700, 'iXen' => -701, 'CeXe' => 702,
      'deX' => 703, 'aXre' => -704, 'uXe' => -705,
      'sleX' => 706, 'aXra' => -707, 'Xano' => -708,
      'oXe' => -709, 'bliX ' => 710, ' baXa' => -711,
      'CiX' => 712, 'aXazi' => -713, 'IXu' => -714,
      'teXe' => 715, 'yeXe' => 716, ' vuX' => -717,
      'meXe ' => 718, 'niXd' => 719, 'eXm' => 720,
      ' oXun' => -721, ' praX' => -722, 'Xio' => -723,
      'eceX' => 724, 'oXu' => 725, 'aXe' => -726,
      'oXlu' => 727, 'Xanda' => -728, 'araX' => -729,
      'uXay' => -730, 'Xne' => 731, 'Xh' => -732,
      'haX' => -733, 'beXe' => 734, 'eXl' => 735,
      'Xo' => -736, 'loX' => -737, ' liX' => -738,
      'zX' => -739, 'vX' => -740, 'IX' => 741,
      'yX' => -742, 'oXram' => -743, 'iXe' => 744,
      'eXer' => 745, ' buXun' => -746, 'UX' => 747,
      'OX' => 750, 'lX' => -751, 'oX' => 752,
      'uX' => 753, 'aX' => 754, 'Xi' => 756, 'X' => -757},
    'u' => {
      'Xctugu' => -1, ' ay sX' => 2, 'fXtur ' => 3, 'Xysuz ' => -4,
      'Gi rXs' => -5, 'an Xnv' => 6, 'Xldeni' => -7, 'dIn Xn' => 8, 'i kXcu' => 9,
      '43 sX ' => 10, 'Xcunda' => -11, 'g kulX' => 12, ' aC kX' => -13, 'Xrunda' => -14,
      'lI tXt' => -15, 'pon bX' => 16, 'ok bXy' => 17, 'z abdX' => -18, 'mamXlu' => -19,
      'Xllah ' => -20, 'im sXl' => 21, 'ne gXn' => 22, 'Xstafa' => -23, ' en bX' => 24,
      'bXlet' => 25, 'CustX' => 26, 'ep lX' => 27, 'Us yX' => 28, 'as X ' => 29,
      'tXtki' => 30, ' fXru' => 31, 'hU sX' => 32, 'c itX' => -33, 'i tX ' => 34,
      'n lXx' => 35, 'u Xs ' => 36, 'SundX' => 37, 'lm X ' => 38, 'm Xs ' => 39,
      'okXme' => -40, 'gOzXc' => -41, 'zangX' => 42, 'Xrtum' => 43, 'S sXz' => 44,
      'Xtaas' => -45, 'Xktio' => -46, ' nXsu' => -47, ' mayX' => -48, 'atarX' => 49,
      'CayXs' => 50, ' ayhX' => 51, 'bu X ' => 52, 'Ors X' => 53, 'orsXd' => 54,
      'ergXt' => -55, 'Xrats' => -56, 'cXmul' => 57, 'Xslam' => 58, 'Xnsan' => -59,
      'mXstt' => -60, 'Xltad' => -61, 'Xnett' => 62, 'm sXk' => -63, 'Xlmut' => -64,
      'CXku ' => 65, 'evarX' => 66, 'lXylu' => 67, 'thXri' => 68, 'rakX ' => -69,
      'jektX' => 70, 'Xrsad' => 71, 'alXle' => -72, 'u iCX' => -73, 'ttXrs' => 74,
      '65 X ' => 75, ' bXsl' => -76, 'Ul Xs' => -77, 'kXsuf' => 78, 'Xteki' => -79,
      '4 tX ' => 80, 'a kX ' => 81, 'rUkX ' => -82, 'kUslX' => 83, 'rtikX' => 84,
      'mXsub' => 85, ' Xndu' => 86, 'uritX' => -87, 'OnktX' => 88, 'sXlie' => 89,
      'cUdXn' => -90, 'm Xsk' => 91, 'ec tX' => 92, 'mnibX' => -93, 'Ol dX' => 94,
      'nU X ' => 95, 'Xnita' => -96, 'Oy lX' => 97, 'bXkat' => 98, 'yad X' => 99,
      ' itsX' => 100, 'kU zX' => 101, 'ogXnk' => -102, ' cUlX' => -103, 'cXlus' => 104,
      'lXkun' => -105, 'iyedX' => 106, 'GangX' => 107, 'pXste' => 108, 'sXste' => 109,
      ' Xrin' => 110, 'yergX' => -111, 'i Xld' => 112, 'zz tX' => 113, 'lXsse' => 114,
      'adOlX' => -115, '3 nX ' => 116, 'bambX' => -117, 'kXsey' => -118, 'i Xr ' => -119,
      ' trXk' => 120, 'yIndX' => 121, 'mad X' => -122, 'GardX' => 123, 'Ok mX' => 124,
      ' rUcX' => -125, 'tXsev' => 126, 'hXku ' => 127, ' Xsst' => 128, 'hulXl' => 129,
      'S dXo' => 130, 'gge X' => 131, ' Xyke' => 132, 'U alX' => 133, 'et X ' => 134,
      'rC gX' => -135, 'Xrgem' => 136, 'ingXs' => -137, 'Xvero' => -138, 'ei kX' => -139,
      'mXkur' => -140, 'Xiti ' => 141, 'Xrba ' => -142, 'sXtre' => 143, '13 tX' => 144,
      'hXmet' => 145, 'Xhlef' => -146, 'UC X ' => 147, 'sfatX' => 148, 'Xmmuh' => 149,
      'enusX' => 150, 'mXhr ' => 151, 'UslXk' => 152, ' Xsur' => 153, 'sXmel' => -154,
      'ok zX' => 155, '0 yX ' => 156, 'gXlli' => -157, 'lXzun' => -158, 'Xtlur' => 159,
      ' 0 tX' => -160, 'k Xz ' => 161, 'Xctuu' => 162, 'huldX' => 163, 'Xrkua' => -164, 'ak X ' => 165,
      'sXkay' => -166, 'sXled' => -167, 'tUpsX' => 168, 'h dXr' => -169, 'Xhuri' => -170, 'tol X' => 171,
      'erkXn' => 172, 'bXrud' => 173, 'jOr X' => 174, 'hXsev' => 175, ' bXrl' => -176,
      'ue cX' => -177, 'StXyu' => 178, 'n lXm' => 179, 'Xliyy' => 180, ' yXne' => -181,
      'Xresh' => -182, ' cXr ' => 183, 'vandX' => -184, 'Xnifi' => -185, 'Xngus' => 186,
      'Xrita' => -187, 'mi X ' => -188, 'Xvete' => 189, 'tXsib' => 190, 'Ci Xs' => 191,
      'wandX' => -192, 'g Xnv' => 193, 'hi Xn' => -194, ' klXm' => -195, ' tUSX' => -196,
      'tXyug' => -197, 'n tX ' => 198, ' g mX' => -199, ' nXss' => 200, 'bi X ' => -201,
      'rm X ' => 202, 'mkXm ' => 203, ' I lX' => -204, ' bXta' => -205, 'olXme' => -206,
      'Xtlaa' => 207, 'uIp X' => -208, 'dis X' => -209, ' orgX' => -210, 'dr zX' => 211,
      ' fXnu' => 212, 'mI Xb' => 213, 'n yX ' => 214, 'yrXl ' => 215, 'irk X' => 216,
      'Uks X' => 217, 'duh X' => 218, 'zallX' => 219, ' yXve' => 220, 'abakX' => 221,
      ' bXto' => -222, 'ptXn ' => 223, 'l Xsu' => -224, 'Uh mX' => 225, 'm Xnc' => 226,
      'g yXn' => -227, 'i hX ' => 228, 'Xveti' => 229, 'satXk' => -230, 'tXnet' => 231,
      'bXruy' => 232, 'tXzud' => -233, '3 nXn' => 234, 'kXrts' => 235, 'Xrruo' => 236,
      'a mX ' => 237, 'kap X' => -238, '  Xlt' => -239, 'hXkul' => 240, '3 yX ' => 241,
      'Off X' => 242, 'Xrtee' => -243, 'Ut kX' => -244, 'gXlug' => -245, 's mX ' => 246,
      'Ul lX' => 247, 'mXski' => -248, 'r dXl' => -249, 'Xtsu ' => 250, 'dXkas' => -251,
      'tUk X' => 252, 'mXsar' => -253, 'Xsusm' => 254, ' m X ' => 255, 'tf nX' => 256,
      'r Xlt' => -257, 'rks X' => 258, 'prokX' => 259, 'mXril' => -260, 'Xruks' => 261,
      'jI Xc' => -262, 'be lX' => 263, 'kabbX' => -264, ' gXye' => -265, 'Xroyo' => -266,
      'hi CX' => -267, ' Xzga' => -268, 'kXrtt' => 269, 'rikXt' => 270, 'UC lX' => 271,
      ' kXp ' => 272, 'zut X' => -273, 'Xyluo' => -274, 'a dX ' => 275, 'Ulk X' => 276, 'izamX' => 277,
      'zeynX' => 278, 'yUz X' => 279, 'rU jX' => -280, 'ko tX' => 281, 'ns Xs' => 282,
      'Ukm X' => 283, 'Xruri' => -284, 'di X ' => -285, 'g sXp' => -286, ' mXk ' => -287,
      'asXre' => -288, 'Sun X' => 289, 'erg X' => 290, 'sellX' => 291, 'Xtayi' => -292,
      's Xsl' => -293, 'Xlali' => -294, 'e rXb' => 295, 'r Xss' => 296, 'rls X' => 297,
      'bdulX' => 298, 'k h X' => 299, 'y tXb' => -300, 'ytemX' => 301, 'e tX ' => 302,
      'gXrba' => -303, 'IS pX' => -304, 'sXylu' => 305, 'mXcu ' => -306, '  mX ' => 307,
      'lorX ' => 308, ' Xcuy' => -309, 'yekXm' => -310, 'ee tX' => 311, 'gs bX' => -312,
      ' indX' => -313, 'gXri ' => -314, 'm Xnv' => 315, 'Ip Xn' => 316, 'UrtCX' => 317,
      'Xzari' => 318, 'rU hX' => 319, ' hXru' => -320, 'polXn' => 321, 'Xlca ' => -322,
      'ot lX' => -323, 'kXlel' => -324, 'mXbil' => -325, 'gXrke' => -326, 'Xterl' => 327,
      'mXzlu' => -328, 'Se Xm' => 329, ' hUdX' => -330, 'n rX ' => -331, 'r pXt' => -332,
      'e Xhu' => 333, 'lt lX' => 334, 'ova X' => 335, 'kXrse' => 336, 'to sX' => -337,
      'Xskum' => -338, 'Xdusu' => 339, 'Xretc' => 340, 'r izX' => 341, 'kXlut' => 342,
      'laykX' => 343, 'rmatX' => 344, 'rIS X' => -345, 'ze Xs' => 346, 'kXmbe' => 347,
      'rXtuk' => 348, ' Xcus' => -349, 'Xbeyr' => -350, 'ef Xn' => 351, 'rkXte' => 352,
      'Xlas ' => -353, 'tI X ' => 354, 'aribX' => -355, 'kulXy' => -356, 'orlXr' => 357,
      'n sXk' => 358, 'Xturo' => 359, 'kXbiz' => 360, 'l p X' => 361, 'Xzaya' => -362,
      'm alX' => 363, 'ul gX' => 364, 'rn mX' => 365, 'Xnsel' => 366, 'mXsab' => 367,
      'tU tX' => 368, 'C kXs' => -369, 'n Xf ' => 370, 'm Xct' => 371, 'fi Xc' => -372,
      'eS Xn' => 373, 'Xdus ' => -374, 'sormX' => -375, 's dXr' => -376, 'Xrge ' => 377,
      'Xrkos' => -378, 'ore X' => -379, 'k dX ' => 380, 'sla X' => -381, '00 fX' => 382,
      'mXral' => -383, 'age X' => -384, 'tXnc ' => -385, 'mXhen' => 386, 'SaygX' => 387,
      'elXcl' => 388, 'Xmdan' => -389, 'l alX' => 390, 'lektX' => 391, 'ur rX' => -392, 'mXrsa' => -393,
      'UtXne' => 394, 'nSUmX' => -395, 'Xlak ' => -396, 'su gX' => 397, 'Xrsan' => -398,
      'mXrid' => 399, 'igXre' => -400, 'a gX ' => -401, ' kXre' => 402, ' kXnu' => 403,
      'mXssa' => -404, 'Xvenc' => 405, 'Xrmes' => 406, 'Xrgul' => -407, 'epeX' => 408,
      'acXm' => 409, 'Xrrt' => 410, 'Xmob' => -411, 'anXp' => 412, 'Xfam' => 413,
      'pXrn' => 414, 'nnXk' => -415, 'rtdX' => -416, 'fUcX' => -417, 'fXcu' => 418,
      'rtcX' => 419, 'mXmm' => -420, 'mXif' => 421, 'cz X' => 422, 'lXrr' => 423,
      'nyXa' => 424, 'ulCX' => -425, 'Xsef' => 426, 'Xsfa' => -427, 'lksX' => 428,
      'cXpe' => -429, 'OSrX' => 430, ' Xnt' => 431, 'gXbb' => -432, 'hinX' => -433,
      'OStX' => 434, 'kekX' => -435, 'mehX' => 436, 'agX ' => -437, 'mXlh' => -438,
      'Xdev' => 439, ' aSX' => -440, 'grXt' => 441, 'UclX' => 442, 'rIsX' => 443,
      'OlgX' => 444, 'Xbid' => 445, 'rbrX' => 446, 'hXnk' => 447, 'Xhek' => -448,
      'Xssy' => -449, 'Xtc ' => -450, 'sn X' => 451, 'rlyX' => 452, 'nXkr' => -453,
      'Xlot' => 454, 'Xleg' => -455, ' sXo' => 456, 'tesX' => 457, 'Xkyu' => 458,
      'Xloj' => 459, 'dXed' => -460, 'rcXb' => 461, 'UysX' => 462, 'zdeX' => 463,
      'jXpo' => 464, 'OtlX' => 465, 'q sX' => 466, 'lXkk' => -467, 'aXsk' => 468,
      'tXpe' => -469, 'holX' => 470, 'tXem' => 471, 'etdX' => 472, 'Xndk' => -473,
      'tXbt' => 474, 'OrnX' => 475, 'Xsir' => 476, 'kalX' => 477, 'Xrds' => -478,
      'ybXk' => 479, 'fXzz' => -480, 'Xrua' => -481, 'genX' => -482, 'UrpX' => 483,
      'Xzbu' => -484, 'x lX' => -485, 'jXre' => -486, 'tCXo' => -487, 'bb X ' => -488,
      'b X ' => 489, 'ehXz' => 490, 'UnnX' => 491, ' Xnn' => 492, 'Xbut' => 493,
      'alXc' => -494, 'dUfX' => 495, 'hXkt' => 496, 'zbXk' => 497, ' Xsd' => -498,
      ' Xrb' => -499, 'tezX' => 500, 'Xsei' => -501, 'lptX' => -502, 'UhtX' => 503,
      ' Xcd' => 504, 'Xfad' => 505, 'hXla' => 506, 'UmmX' => 507, 'tXnr' => 508,
      'dXbe' => 509, 'nesX' => 510, 'yXrr' => 511, 'Xhih' => 512, 'Xlts' => -513,
      'hXup' => 514, 'fagX' => -515, 'pidX' => -516, 'inXs' => 517, 'dekX' => -518,
      'Xnia' => -519, 'UtcX' => 520, ' g X' => 521, 'SidX' => 522, 'kClX' => 523,
      'otCX' => -524, 'irXl' => 525, 'yerX' => 526, 'Xmld' => -527, ' Xry' => -528,
      ' yXu' => 529, 'o pX' => -530, 'UzkX' => 531, ' Xzk' => 532, 'Xkbu' => -533,
      'Xzo ' => -534, 'kkXn' => 535, 'ddXp' => 536, ' sgX' => -537, 'vXke' => 538,
      'visX' => -539, 'ItrX' => 540, 'rXto' => -541, 'Co X' => -542, 'gelX' => 543,
      'cekX' => -544, 'adgX' => -545, 'CXrc' => -546, 'Xnug' => 547, 'OrcX' => 548,
      'eygX' => 549, 'cinX' => -550, 'Om X' => 551, 'sb X' => 552, 'anCX' => -553,
      'irXp' => 554, ' CXt' => -555, 'nbXk' => 556, 'U2nX' => 557, 'Xrro' => -558,
      'fXme' => 559, 'eptX' => -560, 'gXth' => -561, 'Xgar' => -562, 'Ub X' => 563,
      'dXvu' => 564, 'yadX' => 565, 'OtsX' => 566, 'ww X' => -567, 'Xtei' => -568,
      'ntXk' => 569, 'Xlek' => 570, 'd pX' => -571, 'Xttn' => -572, 'Xrru' => -573,
      'd Xp' => -574, ' Xin' => 575, 'erXf' => 576, 'Xnsi' => 577, 'Od X' => 578,
      'lXke' => -579, 'cXrm' => 580, 'I Xk' => -581, 'akXd' => 582, 'Xrio' => -583,
      'vrXb' => 584, 'kubX' => -585, ' Xke' => 586, 'rXdd' => 587, 'IldX' => -588,
      'Xyme' => 589, 'eobX' => -590, 'eatX' => -591, 'Xset' => -592, 'ebXz' => 593,
      'rcXr' => -594, 'Xldi' => -595, 'nXmb' => -596, 'Xrop' => -597, 'mXhn' => 598,
      'Xckn' => -599, ' CXp' => -600, '  Xh' => 601, '9 rX' => 602, 'cetX' => 603,
      'sXpa' => -604, 'Xkev' => -605, 'aCbX' => 606, 'bylX' => -607, 'Xyap' => 608,
      ' ngX' => -609, 'OC X' => 610, 'Xnim' => -611, 'Xkak' => -612, 'vvXl' => 613,
      'tUfX' => 614, 'sXts' => 615, 'ecXm' => 616, 'gerX' => 617, 'I Xu' => 618,
      'Xsol' => -619, '3 jX' => -620, 'grgX' => -621, 'Xnny' => -622, '3 Xs' => -623,
      'pXsr' => 624, ' imX' => 625, 'ayXl' => 626, ' agX' => -627, ' pXn' => -628,
      'nefX' => 629, 'cidX' => 630, 'yazX' => 631, ' Xrm' => -632, 'etXb' => 633, 'mXnk' => -634,
      'Xebe' => -635, 'Xsi ' => -636, 'anXd' => 637, 'eGXz' => 638, 'imiX' => 639,
      'ksXb' => -640, ' rX ' => 641, 'CXyd' => -642, 'Xcar' => -643, 'pXso' => 644,
      'Xpus' => 645, 'fasX' => -646, 'anhX' => 647, 'lalX' => 648, 'yhXl' => 649,
      'bXy ' => 650, 'sikX' => 651, 'radX' => -652, 'bXlf' => -653, 'mXve' => 654,
      'fsXr' => 655, 'gayX' => 656, 'edXc' => -657, 'nkXt' => -658, 'Xths' => -659,
      'anzX' => -660, 'Xlp ' => -661, 'Xrkh' => -662, 'urkX' => -663, 'umbX' => -664,
      'accX' => 665, 'aldX' => -666, ' Xlg' => 667, 'OCsX' => 668, 'Xlna' => 669,
      'iltX' => 670, ' hyX' => -671, 'ebbX' => 672, 'ddXt' => 673, 'aulX' => 674,
      'Xkti' => 675, 'batX' => -676, 'lXbr' => -677, 'Xrpr' => 678, 'Xsee' => -679,
      ' tXv' => -680, 'jXl ' => 681, 'Xgi ' => -682, 'rXc ' => -683, 'dakX' => -684,
      'OmmX' => 685, 'Xth ' => -686, 'CXle' => 687, 'Xbuz' => -688, 'fXz ' => -689,
      'ozXn' => -690, 'Xnam' => -691, 'udXk' => -692, 'sokX' => -693, 'Xbai' => -694,
      'kXmk' => -695, 'prXn' => 696, 'obXs' => 697, 'adXf' => 698, 'Xgay' => -699,
      'nSXm' => 700, 'orXl' => -701, 'ekXl' => 702, 'Xfla' => -703, 'lXla' => -704,
      'Xra ' => -705, 'eyXp' => 706, 'prXl' => 707, 'Xsto' => -708, 'urmX' => -709,
      'sXtu' => 710, 'OrpX' => 711, 'ozXk' => -712, 'Xsem' => 713, 'orXc' => -714,
      'vXc ' => -715, 'cXrr' => -716, 'Xrri' => 717, 'Xkru' => 718, 'mXt ' => -719,
      'Xzan' => -720, 'unsX' => -721, 'oksX' => -722, 'udXr' => -723, 'bXtc' => 724,
      ' Xfa' => -725, 'Xhar' => -726, ' UzX' => 727, 'uStX' => -728, 'ulXs' => -729,
      'sXna' => -730, 'Xtma' => -731, 'oplX' => -732, 'Xkla' => -733, 'hUkX' => 734,
      ' olX' => -735, 'Xzig' => 736, 'kurX' => -737, 'lkXr' => -738, 'fdX' => 739,
      'mCX' => -740, 'iGX' => 741, 'zGX' => 742, 'CCX' => -743, 'OfX' => 744,
      'Xfh' => -745, 'dnX' => 746, 'Xbc' => 747, 'CcX' => 748, 'Xkv' => -749,
      'Xua' => -750, 'Xyn' => 751, 'Xhh' => 752, 'Xtw' => -753, 'UvX' => 754,
      'GaX' => 755, 'lSX' => 756, 'Xmn' => -757, 'Xhy' => 758, 'aiX' => -759,
      'byX' => 760, 'Xyz' => 761, 'hXv' => 762, 'Xgh' => -763, 'aXy' => 764,
      'jXn' => -765, ' Xo' => 766, 'acX' => -767, 'pXz' => -768, 'CyX' => 769,
      'mnX' => -770, 'icX' => -771, 'eaX' => -772, 'fXa' => -773, 'laX' => -774,
      'Xtk' => -775, 'ocX' => -776, 'sXc' => -777, 'UyX' => 778, 'OnX' => 779,
      'n konsensXs ' => -780, 'Uk bir tXr ' => -781, 'sI Xnvanini' => 782, 'yla tXrban ' => -783, 'k tXrkcell' => -784,
      'ler tXrunu' => -785, '  diGer Xc' => 786, 'ara bir dX' => -787, 'i iki tXr ' => -788, 'minde tXrb' => -789,
      'nI Xslup ' => -790, 'taki tXrb' => -791, 'Xmenalti ' => -792, 'i tXrkcel' => -793, 'I Xnvani ' => 794,
      'cak kXrsu' => 795, 'mada tXr ' => 796, 'yi enstrX' => -797, 'inin tXrb' => -798, ' nIn tXrb' => -799,
      'lan tXtun' => 800, 'Xniversia' => -801, 'ine kXrt ' => -802, 'terini dX' => 803, '  bir kXr' => -804,
      'Xverturle' => 805, 'usion tX' => 806, 'In usXl ' => 807, 'go tribX' => -808, 'Xkuneti ' => -809,
      'n bXyum ' => -810, 'z sXrat ' => -811, 'ttin kXr' => -812, 'yen ambX' => 813, 'i bXyur ' => -814,
      'aya tXrb' => -815, 'li tXrba' => -816, 'han abdX' => -817, 'eki Xcu ' => -818, 'Cok tXr ' => 819,
      'ir ergXn' => -820, 'tU bir X' => 821, 'kan kXrt' => -822, 'de ergXn' => -823, 'I tXtun ' => 824,
      'r Xnvani' => 825, 'uran kXr' => -826, 'lu ergX' => -827, 'Xrator ' => 828, 'en kXl ' => 829,
      'si Xzuy' => -830, 'bu tXmu' => -831, 'mir mXr' => -832, 'Xvette ' => 833, 'et ergX' => -834,
      'ar kXl ' => 835, 'bin kXs' => 836, 'l tXtun' => 837, 'hXkumuz' => -838, ' tXruyo' => 839,
      'in ambX' => 840, 'iyet Xn' => 841, 'a Xcunu' => 842, 'Ik tXru' => 843, 'a kXsuy' => 844,
      'zi kXrt' => -845, 'adet sX' => 846, '  bXyur' => -847, 'mle kXs' => 848, 've mXz ' => -849,
      'Um dXr ' => 850, 'di tXru' => 851, 'bXrundi' => -852, 'e ve Xs' => 853, 'Inar kX' => 854,
      'I tXru ' => 855, 'Ur mXsu' => 856, 'tmen mX' => -857, 'rdIm kX' => -858, 'lk tXru' => -859,
      'a mXsah' => -860, 'an Xcun' => 861, 'cU kXrs' => -862, 'r kondX' => -863, 'at tXru' => 864,
      'Xsunuyo' => 865, 'mon sX' => -866, 'Xkutuy' => -867, ' o mXd' => -868, 'oole X' => -869,
      'n sXr ' => -870, 'cla pX' => -871, 're kXb' => 872, '0 tXr ' => 873, 'k sXs ' => 874,
      'birt X' => 875, 'makulX' => 876, 'bXzul ' => -877, 'UC mX ' => 878, 'Xrtcuk' => -879,
      'dXslu ' => -880, 'eux nX' => 881, 'e 3 sX' => 882, '  sUtX' => 883, 'bXyurl' => 884,
      'gXlham' => -885, 'r sXs ' => 886, 'lkolsX' => 887, 'Xverno' => -888, ' akXsu' => 889,
      'sI Xn ' => 890, ' mXzu ' => -891, 'Xtena ' => -892, ' dXrda' => 893, 'gUl sX' => -894,
      'ney Xc' => -895, 'Xbapla' => 896, 'koldXr' => 897, ' gXru ' => -898, ' Xctur' => 899,
      'as mXs' => -900, 'Xrtceb' => -901, 'oyle X' => -902, 'Ur dX ' => 903, 'ara X ' => 904,
      'Xrdish' => -905, 'e Xzuy' => -906, 'Xrtaji' => -907, 's ta X' => 908, 'suC mX' => -909,
      'kXstuk' => 910, 'sul mX' => 911, 'sus nX' => 912, 'i ambX' => 913, ' kXcu ' => -914,
      't mXz ' => -915, 'bXrunl' => -916, 'kbuldX' => 917, 'i mXz ' => -918, 'g tXru' => -919,
      'ton tX' => 920, 'Xtture' => 921, 'Xralis' => 922, 'Xcunde' => 923, 'zor mX' => -924,
      'n Xnva' => -925, 'sIk sX' => -926, 'Xtunde' => 927, 'ir Xr ' => -928, 'troldX' => 929,
      'Xruyle' => 930, 'lkollX' => 931, 'tXrce ' => 932, 'ari Xn' => -933, 'kOr mX' => 934,
      'r Xzuy' => -935, 'Si yXn' => -936, 'na tXt' => -937, 'I tXtu' => -938, 'st sXp' => -939,
      '5 lXk ' => -940, ' Xnet' => 941, 'naryX' => 942, ' gXsa' => 943, ' ad X' => 944,
      'insXl' => 945, 'a sXh' => -946, 'i Xz ' => 947, 'Xgsas' => 948, 'gercX' => 949,
      'me zX' => 950, ' cXss' => 951, 'rXnin' => 952, 'ktXtu' => 953, '  alX' => 954,
      'bXlem' => -955, 'eur X' => 956, 'i alX' => 957, 'tasX ' => 958, 'U Xct' => 959,
      ' gXts' => -960, 'I Xs ' => 961, 'algXl' => 962, '3 tX ' => 963, ' tXrt' => 964,
      'UrzX ' => 965, 'UnCtX' => 966, 'UltmX' => 967, 'UrCtX' => 968, 't Xs ' => 969,
      'dOksX' => 970, 'UnC X' => 971, ' kXld' => 972, 'Xluku' => -973, 'fXlut' => 974,
      'k alX' => 975, '4 nXn' => 976, ' sXzd' => 977, ' Xzur' => 978, ' takX' => -979,
      'mXhel' => -980, ' Xsud' => 981, 'oollX' => -982, ' akXl' => 983, 'z Xnv' => 984,
      'atXm ' => -985, 'by tX' => 986, ' Xmmi' => 987, 'itXtu' => 988, 'Xlaym' => 989,
      ' erXs' => 990, 'r alX' => 991, 'otokX' => 992, 'Xtupa' => 993, 'pXrde' => 994,
      'alXll' => 995, 'hXmas' => 996, 'rgXme' => -997, 'Oln X' => 998, 'i zXh' => 999,
      ' yXn ' => 1000, 'bXzus' => 1001, 'sXder' => 1002, 'aytXl' => 1003, 'emXrt' => 1004,
      'mXlar' => -1005, 'Up Xn' => 1006, 'Xsri ' => -1007, 'mXst ' => -1008, 'lXgan' => -1009,
      'tekX ' => -1010, 'izanX' => 1011, 'rollX' => 1012, 'gulsX' => 1013, 'tXrud' => 1014,
      'ulXz ' => 1015, 'ssXsu' => 1016, 'gXyan' => -1017, 'GUslX' => 1018, 'rdesX' => 1019,
      'io tX' => 1020, 'Xtter' => -1021, ' pXr ' => 1022, ' Xnir' => -1023, 'stXtu' => 1024,
      'Xklum' => 1025, 'mXtas' => -1026, 'd e X' => 1027, 'bXcus' => 1028, 'Xlien' => 1029,
      'evalX' => 1030, 'dUstX' => -1031, 'Xtune' => 1032, 'Xcune' => 1033, 'Xstal' => -1034,
      ' mXth' => 1035, 'Xstas' => -1036, ' Xrka' => -1037, 'Xsta ' => -1038, 'amulX' => 1039,
      ' yXmu' => -1040, 'mXste' => 1041, 'Xrula' => -1042, 'Xnda ' => -1043, 'rXze' => -1044,
      'mXpa' => -1045, 'eilX' => 1046, 'upCX' => -1047, 'hXrt' => -1048, 'stXv' => 1049,
      'inXn' => 1050, 'renX' => -1051, 'Xchn' => 1052, 'anXz' => 1053, 'tXpo' => -1054,
      'kbXk' => 1055, 'kXsp' => 1056, 'orfX' => -1057, 'tXh ' => 1058, 'enXr' => -1059,
      'udXz' => -1060, ' Xrc' => -1061, 'trX ' => 1062, 'OydX' => 1063, 'gogX' => -1064,
      'hr X' => 1065, 'OzXg' => -1066, 'lCtX' => 1067, 'uXzu' => 1068, 'OGrX' => 1069,
      'UfrX' => 1070, 'O yX' => 1071, 'dXrz' => 1072, ' Xvi' => -1073, 'ogXz' => -1074,
      'rl X' => 1075, 'lliX' => -1076, 'litX' => -1077, 'sXlo' => 1078, 'zXc ' => 1079,
      'bXlk' => -1080, 'Xkku' => 1081, 'yXpo' => 1082, 'UptX' => 1083, ' lXy' => -1084,
      'Xchi' => -1085, 'IndX' => -1086, 'Xns ' => -1087, ' lXp' => -1088, 'UrrX' => 1089,
      'CXlh' => -1090, 'UySX' => 1091, 'lpsX' => 1092, 'dXnb' => -1093, 'ugXl' => -1094,
      'mXto' => -1095, 'bXko' => -1096, 'utgX' => -1097, 'Xlia' => -1098, 'mXpp' => -1099,
      'mXnt' => -1100, 'Xfte' => 1101, 'kXsy' => 1102, 'dXng' => -1103, 'apCX' => -1104,
      'hU X' => 1105, 'Xnth' => -1106, 'egXs' => -1107, 'OlsX' => 1108, 'evvX' => 1109,
      'dXld' => 1110, 'rCmX' => 1111, 'yXde' => 1112, 'bXht' => 1113, ' Xrs' => -1114,
      'skXl' => 1115, ' lXm' => -1116, 'Xli ' => -1117, 'Xgab' => -1118, 'bXge' => 1119,
      'zsXy' => -1120, ' Xsa' => -1121, 'Xtat' => -1122, 'Xlre' => 1123, 'untX' => -1124,
      'vizX' => 1125, 'dXls' => 1126, 'cXlc' => 1127, 'CXgi' => -1128, 'togX' => -1129,
      'ombX' => -1130, 'ailX' => 1131, ' Xgd' => 1132, 'akXp' => -1133, 'virX' => 1134,
      'Xfli' => 1135, 'hXum' => 1136, 'OpmX' => 1137, 'Xch ' => -1138, ' Xri' => -1139,
      'nXsr' => -1140, 'Xsli' => 1141, 'dizX' => 1142, 'sXns' => -1143, 'kkXr' => 1144,
      'dXlg' => 1145, 'ekkX' => 1146, ' Xyd' => -1147, 'nXll' => 1148, ' yXc' => 1149,
      'Xcci' => -1150, 'CXrp' => -1151, 'Xla ' => -1152, 'fXne' => -1153, 'GanX' => 1154,
      'Xtme' => 1155, 'rubX' => -1156, 'kutX' => -1157, 'yUzX' => 1158, 'uygX' => -1159,
      'okXy' => -1160, ' uzX' => -1161, ' Xca' => -1162, ' Xza' => -1163, ' dXy' => -1164,
      'Xsle' => 1165, 'mXu' => -1166, 'gXo' => -1167, 'lXi' => -1168, 'lXh' => -1169,
      'kdX' => 1170, 'Xrj' => -1171, 'mXb' => 1172, ' Xh' => -1173, 'Xg ' => -1174,
      'thX' => -1175, 'OkX' => 1176, 'Xen' => -1177, 'apX' => -1178, ' Xg' => -1179,
      'rXf' => -1180, 'OX' => 1181, 'de konsensX' => -1182, ' la tribX' => -1183, 'nIn Xcu ' => -1184,
      'i bXyuka' => -1185, 'r kXrsu ' => 1186, 'iki tXr ' => 1187, 'i tXtun ' => 1188, 'in dXrul' => 1189,
      'ra sXres' => -1190, 'anI kXrt' => -1191, 'da ergXn' => -1192, 'bir tXru' => 1193, 'Xrsunun ' => 1194,
      'z tXtun' => 1195, 'rk tXr ' => 1196, 't usXlu' => 1197, 'm tXru ' => 1198, 'ka tXr ' => 1199,
      'Urk tX ' => 1200, 've sUtX' => 1201, 'de kXl ' => 1202, 'nkulXnu' => 1203, 'ya tXru' => -1204,
      'ali kXr' => -1205, 'ne ergX' => -1206, 'un Xcu ' => -1207, 'ir mXz ' => -1208, 'mXltes' => -1209,
      'Oz mX ' => 1210, 'emerrX' => 1211, 'rgXner' => -1212, ' sXlun' => 1213, 'nI Xnv' => 1214,
      'mXhada' => 1215, 'ci Xnv' => 1216, 'Ur mXy' => 1217, 'S tXru' => 1218, 'Une Xn' => 1219,
      ' hXlle' => 1220, 'or mXz' => 1221, 'ir Xs ' => 1222, 'Un dX ' => 1223, 'a Xnu ' => 1224,
      'US mXy' => 1225, 'r sXty' => 1226, 'm kXm ' => 1227, ' blXm ' => 1228, 'letlX ' => 1229,
      ' tXrsa' => 1230, ' tXne ' => -1231, 'r Xcu ' => -1232, 'mer Xn' => 1233, 'n sXzu' => 1234,
      'Xnder ' => 1235, 'Xlucul' => -1236, 'Xnden ' => 1237, 'i mamX' => -1238, 'Xlunk' => 1239,
      'Xrks ' => -1240, ' atXt' => 1241, 'tXfey' => -1242, ' nXne' => -1243, 'lXsiy' => 1244,
      'mlubX' => 1245, 'bXcur' => 1246, 'gUrsX' => -1247, 'sXspa' => 1248, 'sXvey' => 1249,
      'n alX' => 1250, 'halXs' => 1251, 'klubX' => 1252, 'luldX' => 1253, 'cXre ' => -1254,
      'U mXy' => 1255, ' Xcum' => 1256, ' zUlX' => -1257, ' onXc' => 1258, 'CXkun' => 1259,
      'kXsup' => 1260, 'dXdul' => -1261, ' volX' => 1262, ' dXke' => -1263, 'sargX' => -1264,
      ' Xsus' => 1265, 'tXrik' => 1266, 'Xtedi' => -1267, 'z Xs ' => 1268, 'gUm X' => 1269,
      'Xsamb' => -1270, 'Xlka ' => -1271, ' fXr ' => 1272, ' mXlu' => -1273, 'ndXsu' => -1274,
      'f mX ' => 1275, ' tXll' => 1276, 'guldX' => 1277, 'tXkiy' => 1278, '4 sX ' => 1279,
      'bXgru' => 1280, 'zXhre' => 1281, 'merkX' => 1282, ' matX' => -1283, 'rtsXn' => 1284,
      ' dXmd' => 1285, 'Xnker' => -1286, 'U lX ' => 1287, 'revX ' => 1288, ' Xnay' => 1289, 'tXman' => -1290,
      ' bXs ' => -1291, 'ChulX' => 1292, 'Urt X' => 1293, 'Xllat' => 1294, 'ude X' => 1295,
      'mXre ' => -1296, 'kulXm' => -1297, 'Xndek' => 1298, 'Xslum' => 1299, 'OGdX' => 1300,
      'aSXs' => 1301, ' Xci' => 1302, 'lXzz' => -1303, 'fXhr' => 1304, 'Of X' => 1305,
      'Xcko' => 1306, 'ikXr' => 1307, 'sXny' => -1308, 'tCXk' => -1309, 'nitX' => -1310,
      'OstX' => 1311, 'Xseu' => -1312, 'nsXt' => 1313, 'dXnn' => -1314, 'yyXl' => 1315,
      'trXb' => 1316, 'ifXs' => 1317, 'Ov X' => 1318, 'mXnz' => -1319, 'II X' => 1320,
      'Xmuk' => 1321, 'alyX' => -1322, ' CXc' => -1323, 'kadX' => 1324, 'lsXy' => -1325,
      'tXbl' => 1326, 'Xbab' => 1327, 'UGrX' => 1328, 'tXko' => 1329, 'rOtX' => -1330,
      'Xloz' => 1331, 'eytX' => 1332, 'Xmtu' => -1333, 'ufCX' => -1334, 'ogX ' => -1335,
      'ebXl' => 1336, 'dXlv' => 1337, 'uzsX' => -1338, 'Xdle' => 1339, 'rXks' => 1340,
      'timX' => -1341, 'Xks ' => 1342, 'effX' => 1343, 'dXna' => -1344, 'Xbat' => -1345,
      ' Xyu' => -1346, 'ultX' => -1347, 'hXri' => -1348, 'Xnle' => 1349, 'pdX' => 1350,
      'jmX' => 1351, 'lGX' => 1352, 'Xju' => -1353, 'Xev' => -1354, 'Xbj' => 1355,
      'UdX' => 1356, 'z kXsur ' => 1357, ' kXsmus ' => 1358, ' mXslug' => -1359, 'Xratind' => -1360,
      'ngi tXr' => 1361, 'irmi kX' => 1362, 'Xzelge' => 1363, 'den wX' => 1364, 'Xstafi' => 1365,
      'tXzunu' => -1366, 'ak Xn ' => 1367, ' yXnu ' => 1368, ' kXpun' => 1369, 'bXker ' => -1370,
      'im gXl' => 1371, 'o mXz ' => -1372, 'esulX ' => 1373, 'Um mX ' => 1374, 'da Xs ' => 1375,
      'Xpert ' => -1376, 'bXyur ' => 1377, 'sUrXr ' => -1378, 'vertXr' => 1379, 'n fXtu' => 1380,
      'Xnvanl' => 1381, 'nlU Xn' => 1382, ' Xzun ' => -1383, 'Xstura' => -1384, ' Xys ' => 1385,
      'Urn X' => 1386, 'gXreb' => -1387, 'karbX' => 1388, 'fUnXn' => -1389, 'nXmay' => 1390,
      'gh X ' => 1391, '3 lX ' => 1392, 'nurgX' => 1393, ' mXsh' => -1394, 'prosX' => 1395,
      'lm Xn' => 1396, 'dXvel' => 1397, ' pXri' => 1398, ' akXn' => 1399, 'tXriy' => 1400,
      'masaX' => 1401, ' zXbu' => 1402, ' tXbe' => 1403, 'lahX ' => 1404, 'aullX' => 1405,
      ' Xnuv' => 1406, 'ilatX' => 1407, 'i Xs ' => 1408, ' Xnsu' => -1409, 'Ust X' => 1410,
      'Xnune' => 1411, 'Xnv ' => 1412, 'zXht' => 1413, 'Xkaf' => 1414, 'sXke' => -1415,
      'dXmt' => 1416, 'dXnh' => -1417, 'jXpi' => 1418, 'jXly' => 1419, 'UflX' => 1420,
      'absX' => 1421, 'Xnna' => -1422, 'dXmb' => 1423, ' lXa' => -1424, 'zsX ' => -1425,
      'inXb' => 1426, 'sXpp' => -1427, ' nXv' => 1428, 'usCX' => -1429, 'Xhle' => 1430,
      ' msX' => 1431, 'bXka' => -1432, 'UsmX' => 1433, 'yamX' => -1434, 'Xkem' => 1435,
      'Xsa ' => -1436, 'ulXc' => -1437, 'urXy' => -1438, 'Xmku' => 1439, 'Xzle' => 1440,
      'yavX' => -1441, 'Xnca' => -1442, 'cXt ' => -1443, 'Xfuk' => -1444, 'Xnla' => -1445,
      'Xnut' => -1446, ' Xya' => -1447, 'Xtan' => -1448, 'sorX' => -1449, 'OSX' => 1450,
      'aCX' => -1451, 'jdX' => -1452, 'kvX' => 1453, 'gXg' => -1454, 'lXj' => -1455,
      'Xmp' => -1456, 'gXj' => -1457, 'ChX' => -1458, 'Xgd' => -1459, 'kbX' => -1460,
      'mXj' => 1461, 'fkX' => -1462, 'Xe ' => -1463, 'GdX' => -1464, 'osX' => -1465,
      'unX' => -1466, 'Xltimato' => 1467, 'nbir sX' => -1468, 'alip yX' => 1469, 'ni tXr ' => 1470,
      'Xratina' => -1471, '  tXtun' => 1472, 'kulubX' => 1473, 'd dXst' => -1474, 'Ul mX ' => 1475,
      'Xllen ' => -1476, 'tXzumu' => -1477, ' mXnis' => -1478, 'Xcuzle' => 1479, ' idolX' => 1480,
      'Xratim' => -1481, 'him kX' => -1482, 'paul X' => 1483, 'ropolX' => 1484, ' pXlum' => 1485,
      'fUtXr ' => -1486, 'U ergX' => -1487, 'Xnunde' => 1488, 'mXsur' => 1489, 'le X ' => 1490,
      ' kakX' => 1491, 'UsttX' => 1492, 'eessX' => 1493, 'urolX' => -1494, 'ngXcu' => 1495,
      'UrkmX' => 1496, 'UstsX' => 1497, 'elamX' => 1498, 'kXner' => -1499, 'n Xs ' => 1500,
      'n Xck' => 1501, 'lXkar' => 1502, 'on lX' => -1503, 'OrtmX' => 1504, ' Xsut' => 1505,
      ' sXme' => 1506, ' tXl ' => 1507, 'mettX' => 1508, 'U ydX' => 1509, 'kulXn' => -1510,
      ' kXse' => 1511, 'nnXm' => 1512, 'efXj' => 1513, 'Xpsa' => -1514, 'Xlaz' => 1515,
      ' ytX' => 1516, 'bXsc' => -1517, 'Xlf ' => -1518, 'ksXr' => 1519, 'mXrp' => -1520,
      'dXsa' => -1521, 'kakX' => -1522, 'UzcX' => 1523, 'OttX' => 1524, 'Xnga' => -1525,
      'Xdde' => 1526, 'mXrn' => -1527, 'ssXl' => 1528, 'pXtu' => 1529, 'Xrie' => -1530,
      'Xtta' => -1531, 'Xnid' => -1532, 'Xrga' => -1533, 'Xysa' => -1534, 'Xcur' => -1535,
      'Xyle' => 1536, 'fXzu' => -1537, ' lXs' => -1538, 'angX' => -1539, 'kanX' => -1540,
      'uSkX' => -1541, 'Xma ' => -1542, ' iX' => 1543, 'zrX' => 1544, 'Xx' => -1545,
      ' celik gX' => -1546, 'e tXrunu' => -1547, 'ne sUtX' => 1548, 'trol Xn' => 1549, 'ynI tXr' => 1550,
      ' Xnunu ' => 1551, 'iki Xcu' => -1552, 'z kXrt ' => -1553, 'an tXru' => 1554, 'da kXl ' => 1555,
      'ban sX' => 1556, 'Xbyanc' => 1557, ' Xsume' => 1558, 'natXre' => -1559, 'tXtunu' => 1560,
      'Uk mX ' => 1561, 'lu mXd' => 1562, 'or mXd' => 1563, 'Xrley ' => -1564, 'US mX ' => 1565,
      'Xpero' => -1566, 'Xccac' => 1567, 'rpolX' => 1568, 'dXrbu' => 1569, 'utXre' => -1570,
      'atXru' => 1571, 'irolX' => -1572, 't bXb' => 1573, 'gravX' => 1574, 'altXn' => -1575,
      'hrolX' => -1576, 'UnClX' => 1577, 'Xngor' => 1578, 'Xlasy' => 1579, 'mXrta' => -1580,
      ' mXca' => 1581, 'Xlar ' => -1582, 'ptXs' => 1583, 'Xkoi' => -1584, 'rXku' => 1585,
      'yfXs' => 1586, 'evXk' => 1587, 'lugX' => -1588, 'eykX' => 1589, 'Xkto' => 1590,
      'UplX' => 1591, 'bogX' => -1592, 'arbX' => -1593, 'OCmX' => 1594, 'Xzir' => -1595,
      'Xnio' => -1596, 'UymX' => 1597, 'idXg' => 1598, 'ezzX' => 1599, 'Ut X' => 1600,
      'Xphi' => -1601, 'easX' => 1602, 'rrXr' => 1603, 'bXtl' => -1604, 'adXr' => -1605,
      'UlgX' => 1606, 'silX' => 1607, 'Xban' => -1608, 'OllX' => 1609, ' Xla' => -1610,
      'gXk' => -1611, 'OsX' => 1612, 'nXh' => -1613, 'Xty' => -1614, 'UnX' => 1615,
      'e tXtun ' => 1616, 'okol Xn' => 1617, 'jale kX' => -1618, 'ik tXru' => 1619, 'Xsturi' => -1620,
      ' kXstur' => 1621, 'CUksX ' => -1622, 'jXrgen' => 1623, 'hakkXm' => 1624, 'Guk dX' => -1625,
      ' mXska' => -1626, ' orXs' => 1627, 'erolX' => -1628, ' bXse' => -1629, ' yXnl' => 1630,
      'kXbra' => 1631, 'rXsta' => -1632, 'm s X' => 1633, 'alkXt' => 1634, 'y tXn' => -1635,
      'OrttX' => 1636, 'babXr' => 1637, 'a Xn ' => 1638, 'mXdan' => -1639, 'enstX' => 1640,
      ' zXbe' => 1641, 'Xrmen' => 1642, 'tXzu ' => -1643, ' Xzmu' => 1644, 'UyUkX' => -1645,
      'UlttX' => 1646, ' mXnd' => -1647, 'mXzda' => -1648, 'pagX' => -1649, 'ezXk' => 1650,
      'plXt' => 1651, 'Xdir' => 1652, ' ktX' => 1653, 'uylX' => -1654, 'Xrmo' => 1655,
      'Xzma' => -1656, ' ulX' => -1657, 'OvmX' => 1658, 'UCX' => 1659, 'onX' => -1660,
      'eX' => -1661, ' o tXr ' => 1662, 'I ergXn' => -1663, ' bXrum' => 1664, 'Xtuncu' => 1665,
      'uS mXd' => -1666, ' kollX' => -1667, 'ontXr' => 1668, 'Xzull' => -1669, 'ekabX' => 1670,
      ' Xsuy' => 1671, 'sXtop' => -1672, 'amdXl' => 1673, 'Xles ' => -1674, ' a X ' => 1675,
      'Xruc ' => -1676, 'pol X' => 1677, 'ntXre' => -1678, ' Xsen' => 1679, 'mXck' => -1680,
      'etXs' => 1681, 'ardX' => -1682, ' gXs' => -1683, 'mXh ' => 1684, ' Xrh' => -1685,
      'Xlse' => 1686, 'zXpp' => 1687, 'orCX' => -1688, 'sXrv' => -1689, 'gOzX' => 1690,
      'Xzr' => 1691, 'fXl' => -1692, 'Xgg' => -1693, 'Xtr' => -1694, 'rXh' => -1695,
      'oGX' => -1696, ' ileri Xc' => -1697, 'konsensX' => 1698, 'ld tribX' => -1699, ' en Xc ' => -1700,
      'oportX' => 1701, 'I kXl ' => 1702, 'songXl' => 1703, 'e Xcun' => 1704, 'Xlale' => 1705,
      'Xngu ' => 1706, 'lkolX' => 1707, 'hsulX' => 1708, 'sollX' => -1709, 'zXraf' => 1710,
      'rk Xy' => -1711, 'mXden' => 1712, 'yollX' => -1713, ' hXda' => 1714, 'Xther' => -1715,
      'ehlX' => 1716, 'gXy ' => -1717, 'zXre' => -1718, 'Xcok' => 1719, 'dXlm' => 1720,
      'Xhru' => 1721, 'UlcX' => 1722, 'OnsX' => 1723, 'O nX' => 1724, 'Xlko' => -1725,
      'Xtad' => -1726, ' Xrn' => -1727, 'kodX' => -1728, ' Xsu' => -1729, 'oX' => -1730,
      'kXskunu' => -1731, 'sXratla' => -1732, 'le Xn ' => 1733, 'kXrsul' => 1734, 'tXlay' => 1735,
      ' akXz' => 1736, 'Xmeyr' => 1737, '4 lX ' => 1738, 'sXha ' => 1739, ' mXmu' => -1740,
      'mUhrX' => 1741, 'UrktX' => 1742, 'mXrah' => -1743, 'Xben ' => -1744, ' endX' => 1745,
      'Xzuyo' => 1746, 'sXnge' => 1747, 'mXteb' => -1748, 'elXl' => 1749, 'SamX' => 1750,
      'vekX' => -1751, 'Xpu ' => 1752, 'Xtaf' => -1753, 'OCtX' => 1754, ' ncX' => 1755,
      'dXlc' => 1756, 'hXlk' => -1757, ' nXz' => 1758, 'UlbX' => 1759, 'Xrak' => -1760,
      'Xlag' => -1761, 'Xrnb' => 1762, 'Xna ' => -1763, 'cXh' => 1764, 'Xniversity' => -1765,
      'gun mXd' => -1766, 'dXrumlu' => 1767, 'mXkted' => -1768, 'morgXl' => 1769, 'tXrbo' => -1770,
      'golsX' => 1771, 'argXn' => -1772, 'amXrs' => 1773, 'rns X' => 1774, 'bXket' => -1775,
      'Xckag' => 1776, 'nyatX' => 1777, 'Ur X ' => 1778, 'dugX' => -1779, 'aahX' => 1780,
      'OynX' => 1781, 'kXkr' => 1782, 'kstX' => 1783, 'ortX' => -1784, 'Xpi' => -1785,
      'Xdi' => -1786, 'u mXdur ' => -1787, 'konsXl' => 1788, 'Ogh Xn' => 1789, 'hXmani' => 1790,
      'gUC Xn' => 1791, 'zelgX' => 1792, ' encX' => 1793, ' Xrug' => -1794, ' sXva' => 1795,
      'Uk X ' => 1796, ' kupX' => 1797, 'Xyuk ' => 1798, 'Xlet' => -1799, 'ktXs' => 1800,
      'IrtX' => 1801, 'CXr ' => -1802, 'tXli' => 1803, 'uzgX' => -1804, 'mXrr' => -1805,
      'latX' => -1806, 'UzmX' => 1807, 'Xzyo' => 1808, 'Xsal' => -1809, 'zXg' => 1810,
      'an ergXn' => -1811, 'in Xcu ' => -1812, ' sXsu ' => 1813, 'mXteme' => -1814, 'U dXr ' => 1815,
      'drolX' => -1816, 'UslX ' => 1817, 'sXluk' => 1818, 'Xrtaj' => 1819, ' rXst' => 1820,
      'Xson' => -1821, 'darX' => 1822, 'tXgi' => 1823, 'Ol X' => 1824, 'oSXr' => 1825,
      ' nXa' => 1826, 'Xta ' => -1827, 'Xlam' => -1828, 'Xsme' => 1829, 'Xcp' => 1830,
      'bXrun ' => -1831, 'kapsX' => 1832, 'Xlema' => -1833, 'kXlah' => 1834, ' mXrd' => -1835,
      'USsXn' => 1836, 'kulX ' => -1837, 'mXdo' => -1838, 'lXft' => -1839, 'logX' => -1840,
      'zatX' => 1841, 'Xsso' => -1842, '9 lX' => -1843, 'Xng ' => -1844, 'Xcle' => 1845,
      'Xoz' => 1846, 'OcX' => 1847, 'fUtXrs' => -1848, ' kXtug' => 1849, 'trolsX' => 1850,
      ' satX' => 1851, 'US X' => 1852, 'bXsh' => -1853, ' Xtu' => 1854, 'UtXf' => -1855,
      'evlX' => 1856, 'brXn' => -1857, 'Xzme' => 1858, 'ssXm' => 1859, 'Xple' => 1860,
      ' SXl' => -1861, 'Xcb' => 1862, 'rfX' => 1863, 'fXturs' => 1864, 'Ur mX ' => 1865,
      'bXyurk' => 1866, '0 kXsu' => 1867, ' Xrum' => -1868, ' tXte' => 1869, ' tXba' => 1870,
      'OSk X' => 1871, 'hXrma' => -1872, 'kXkur' => 1873, 'etXl ' => 1874, 'Uz X ' => 1875,
      'hXrsi' => -1876, 'bXsi' => -1877, 'ebXs' => -1878, ' lXd' => -1879, 'itXt' => -1880,
      'pXf ' => 1881, 'cXmbus' => 1883, 'Xrkish' => -1884, 'Um Xn ' => 1885, 'ayakX' => 1886,
      '00 lX' => 1887, ' mXsu' => -1888, 'mXmy' => -1889, 'uSCX' => -1890, 'esgX' => -1891,
      'UtCX' => 1892, 'Xvi' => 1893, 'Xlb' => 1894, 'otX' => -1895, 'UX' => 1896,
      'ok mXd' => -1897, 'tXrunu' => 1898, 'gollX' => 1899, 'UrttX' => 1900, ' Xnis' => -1901,
      'hXngu' => 1902, 'kXlc' => 1903, ' jXb' => 1904, 'nXsh' => 1905, 'UtmX' => 1906,
      'Xsti' => -1907, 'usX' => -1908, ' Xcunun' => 1909, 'a bXrun' => 1910, 'Or Xn' => 1911,
      ' aygX' => 1912, 'Xnver' => 1914, 'edavX' => 1915, 'Xraf' => -1916, 'Xzip' => -1917,
      'modXl' => 1918, 'Xrkey' => -1919, 'suCXs' => 1920, ' bulX' => -1921, 'cXpp' => 1922,
      ' Xl ' => 1923, 'UksX' => 1924, 'uydX' => -1925, ' Xcy' => 1926, 'rkXk' => 1927,
      'Xpac' => -1928, 'Xyg' => -1929, ' gXlu ' => -1930, ' zulmX' => 1931, 'bdXl ' => 1932,
      ' tXk ' => 1933, 'Xrsat' => 1934, 'ozgX' => -1935, 'Oy X' => 1936, 'bXku' => 1937,
      'OmdX' => 1938, 'sUkX' => -1939, 'Xlti' => -1940, 'flXt' => 1941, 'Xsla' => -1942,
      'oyX' => -1943, 'onktX' => 1944, 'Un X ' => 1945, ' kXrk' => 1946, ' Xcu ' => 1947,
      'rbXs' => -1948, 'i X ' => 1949, 'OptX' => 1950, ' rXc' => 1951, 'sXe' => 1952,
      ' mXsluk' => -1953, 'kXskun' => 1954, 'ekolX' => 1955, 'COzmX' => 1956, 'baSmX' => 1957,
      'UtsX' => 1958, 'Xfur' => 1959, ' kXz' => -1960, 'SdX' => 1961, ' Xnusa' => 1962,
      ' Xnic' => -1963, 'settX' => 1964, ' dXrt' => 1965, 'ampXs' => 1966, 'kostX' => 1967,
      'zXmr' => 1968, 'UCtX' => 1969, ' fXsu' => 1970, 'UmbX' => 1971, 'lfX' => 1972,
      'UrklX' => 1973, 'nipX' => 1974, 'bXre' => -1975, 'gXst' => -1976, 'Xleb' => 1977,
      'okCX' => -1978, 'dokXman' => 1979, 'hUznX' => 1980, 'gXrup' => -1981, 'zarX' => -1982,
      ' dXe' => 1983, 'Xbv' => 1984, 'oCX' => -1985, ' kXlle' => 1986, ' mXml' => -1987,
      ' tXku' => 1988, ' kXle' => -1989, 'fUsX' => -1990, 'brXt' => 1991, 'Xzz' => 1992,
      ' ne tXr' => 1993, '223 X' => -1994, 'rolcX' => 1995, 'Xrune' => 1996, ' zXri' => 1997,
      'herkX' => 1998, 'Xstah' => 1999, 'UsXr' => -2000, 'UnsX' => 2001, 'iki tXrl' => -2002,
      ' sXslu' => 2003, ' Xmme' => 2004, ' cXru' => 2005, ' Xfl' => 2006, 'OlcX' => 2007,
      'mXsia' => 2008, 'mXsi' => -2009, 'Xrub' => -2010, 'hCX' => -2011, ' Xvey' => 2012,
      ' lXn' => -2013, ' klX' => 2014, ' CXl' => -2015, ' lXc' => -2016, 'UhX' => 2017,
      'Xpe' => 2018, ' Xzel' => -2019, 'sUslX' => 2020, 'hXzn' => 2021, 'UmdX' => 2022,
      ' mXstak' => 2024, 'Uz mX ' => 2025, ' Xnes' => -2026, 'raSX' => 2027, 'cXbb' => 2028,
      ' etX' => 2029, 'UmcX' => 2030, 'OplX' => 2031, 'Xrad' => -2032, ' mXhur' => 2033,
      'ohl X' => 2034, 'Xstad' => 2035, ' rXsd' => 2036, ' kamX' => -2037, 'sedX' => 2038,
      'OkmX' => 2039, '0 lXk ' => -2040, 'zulXm' => 2041, ' sXk' => 2042, 'arolX' => -2043,
      ' Xto' => 2044, 'Xsev' => -2045, 'kXme' => 2046, ' Xzum' => 2047, 'USkX' => 2048,
      'Xlisla' => 2049, ' kXlli' => 2050, 'mbolX' => 2051, 'kXrtc' => 2052, 'OksXz' => 2053,
      'lUzX' => -2054, ' nXk' => 2055, 'Xlge' => 2056, 'Xlha' => 2057, 'UCsX' => 2058,
      'Uk Xn ' => 2059, 'OGsX' => 2060, 'Us X' => 2061, 'Xlf' => 2062, 'hXrda' => -2063,
      'edXk' => 2064, 'aX' => -2065, ' pXru' => 2066, 'skXd' => 2067, 'dXme' => 2068,
      'Xti' => -2069, 'e bXrun' => 2070, 'Ur Xn' => 2071, ' Xzd' => 2072, 'UrbX' => 2073,
      'OzdX' => 2074, 'U yX ' => 2075, 'kXrdi' => 2076, 'Xrle' => 2077, ' Xmra' => 2078,
      'Oz X' => 2079, ' Xslu ' => -2080, 'Un mX ' => 2081, 'Xzaf' => -2082, 'Xnet' => -2084,
      'UsbX' => 2085, 'kXrsus' => 2086, ' tXcc' => 2087, ' mXy' => -2088, 'rXce' => -2089,
      'zahX' => 2090, 'Xrsude' => 2091, 'altXs' => 2092, ' mXs ' => -2093, 'Xtle' => 2094,
      'Uz Xn ' => 2095, 'kXle ' => -2096, 'OrsX' => 2097, ' Xlu' => -2098, 'hXsr' => 2099,
      'sulsX' => 2100, 'bXyur' => -2101, ' tXy' => 2102, 'CXb' => -2103, 'Xrunde' => 2104,
      'ilgX' => 2105, 'strXma' => 2106, ' Xsku' => 2107, 'kXrsuy' => 2108, 'rodXk' => 2109,
      'OvdX' => 2110, 'U mX ' => 2111, 'pXsku' => 2112, 'ongX' => -2113, 'Xndee' => -2114,
      'Xsn' => 2115, 'zsX' => 2116, 'kXtuk' => 2117, 'rabXk' => 2118, 'tfX ' => 2119,
      'Xrf' => -2120, 'r tXr ' => 2121, 'UsnX' => 2122, 'rtXk' => 2123, ' usulX' => 2124,
      ' Xzuc' => 2125, 'CXmr' => -2126, 'Xnsa' => 2128, 'UlsX' => 2129, 'Xdurn' => -2130,
      'sUtXn' => -2131, 'UstlX' => 2132, 'hXsam' => 2133, 'UrtX' => 2134, 'rXtb' => 2135,
      'UpX' => 2136, 'UzdX' => 2137, 'kabulX' => 2138, 'mXts' => -2139, ' nXr' => -2140,
      'Ok X' => 2141, 'bXlt' => 2142, 'iyanX' => 2143, 'esadX' => 2144, 'UftX' => 2145,
      'Un Xn ' => 2146, 'UrcX' => 2147, 'sansX' => 2148, ' cXz' => 2149, 'oylX' => -2150,
      'OlmX' => 2151, ' eymXr ' => 2152, 'Xhim' => 2153, 'Xrbet' => -2154, 'UkrX' => 2155,
      'tokolX' => 2156, ' dXdu' => 2157, 'ollX ' => 2158, 'tXrde' => 2159, 'ktXe' => 2160,
      'Xnited' => -2161, 'cavX' => 2162, 'OpX' => 2163, 'hXzun' => 2164, 'erXv' => 2165,
      'OrtlX' => 2166, 'Xtuph' => 2167, 'katX' => 2168, 'UssX' => 2169, 'tXdy' => 2170,
      'Xrtle' => 2171, ' tXbi' => 2172, 'kXtah' => 2173, 'bXtt' => -2174, 'UydX' => 2175,
      'Xva' => -2178, 'UtlX' => 2179, 'OnmX' => 2180, '00 X' => 2181, 'OktX' => 2182,
      'sXmer' => 2183, ' jXri' => 2184, 'Xi' => -2185, ' Xssu' => 2186, 'Xcg' => 2187,
      'Xrla' => -2188, 'OSkX' => 2189, 'OklX' => 2190, ' bXk' => 2191, 'SgXl' => -2192,
      ' lX ' => -2193, 'uyX' => -2194, 'hXly' => 2195, 'UttX' => 2196, 'OmrX' => 2197,
      'umX' => -2198, ' mXm ' => -2199, 'sXnn' => 2200, 'reddX' => 2201, 'sXret' => -2202,
      'popXl' => 2203, 'golcX' => 2204, 'UkmX' => 2205, 'gXe' => -2206, ' Xra' => -2207,
      'UktX' => 2208, 'UmsX' => 2209, 'abX' => -2210, 'Xcte' => 2211, '3 X' => 2212,
      'dtX' => 2213, 'kXba' => 2214, 'hhX' => 2215, 'USlX' => 2216, 'ungX' => -2217,
      'Xgl' => -2218, 'tXzu' => 2219, 'UlkX' => 2220, 'UskX' => 2221, 'elCX' => -2222, 'mXtf' => -2223,
      'Xkse' => 2224, 'Xzunt' => 2225, 'UrmX' => 2226, 'ercXm' => 2227, 'opCX' => -2228,
      ' CXk' => -2229, ' Xslu' => 2230, ' dXk' => 2231, ' rXy' => 2232, 'ObX' => 2233,
      'akX ' => 2234, 'tCX ' => -2235, 'rXstu ' => 2236, 'ekX' => 2237, '4 X' => 2238,
      'UrkX' => 2239, ' Xclu' => 2240, 'Xci' => -2241, 'Xsv' => 2242, 'Xtab' => -2243,
      'UfXr' => 2244, 'UrsX' => 2245, 'Xkm' => 2246, 'OGX' => 2247, 'ukCX' => -2248, 'Xne' => 2249,
      'kXf' => 2250, ' eyX' => 2251, 'tXrb' => 2252, 'Ul X' => 2253, 'OncX' => 2254,
      'dXz' => 2255, 'gXrur' => -2256, 'Xzul' => 2257, ' kXrt ' => 2258, 'ribX' => 2259,
      'hUcX' => -2260, 'UzlX' => 2261, 'OnlX' => 2262, 'OvX' => 2263, 'yor mX' => -2264,
      'OrlX' => 2265, ' Xnal' => 2266, 'USmX' => 2267, ' vUcX' => -2268, ' vXc' => 2269,
      'Xmle' => 2270, ' SXk' => 2271, ' hXc' => 2272, 'tatX' => 2273, 'itX' => 2274,
      'UllX' => 2275, ' sXt' => 2277, ' tXp' => 2278, 'OrmX' => 2279, ' golX' => 2280,
      'ecrX' => 2281, 'Xph' => 2282, 'tXsia' => 2283, 'Xkle' => 2284, 'lgX' => -2285,
      'komXn' => 2286, 'UmlX' => 2287, 'UnmX' => 2288, ' hXr' => 2289, 'OykX' => 2290,
      'ordX' => -2291, 'UlmX' => 2293, ' dXg' => 2294, 'dXrust' => 2295, 'Xhe' => 2296,
      'Xfe' => 2297, 'mXrat' => -2298, 'mXl' => 2299, 'Xny' => 2300, 'OzlX' => 2301,
      ' sXp' => 2302, 'anXstu' => 2303, 'hXsey' => 2304, 'OrtX' => 2305, 'Urk X' => 2306,
      'OprX' => 2307, 'UklX' => 2308, 'UmrX' => 2309, 'mXmc' => -2310, 'Xzg' => 2311,
      'OmX' => 2312, 'Xka' => -2313, 'sXri' => -2314, ' lX' => 2315, 'UbX' => 2316,
      ' Xmi' => 2317, 'OyX' => 2318, 'bXs' => 2320, 'OzcX' => 2321, 'lbXm' => 2322,
      'orgX' => -2323, 'UntX' => 2324, 'Xce' => 2325, 'u tXr ' => 2326, 'Xa' => -2327,
      'rolX' => 2328, 'kXz' => -2329, 'U nX' => 2330, 'bXro' => 2331, ' Xcunc' => 2332,
      'OndX' => 2333, ' mX ' => -2334, 'mUslX' => 2335, ' nXf' => 2336, ' tXke' => 2337,
      'enX' => 2338, ' Xr' => 2339, 'UldX' => 2340, 'tXrl' => 2341, 'uCX' => -2342,
      'UmkX' => 2343, ' kulX' => 2344, 'UClX' => 2345, 'UncX' => 2346, 'Xcr' => 2347,
      'UndX' => 2348, 'OdX' => 2349, 'Xsta' => -2350, 'mXtl' => -2351, ' Xnl' => 2352,
      'OldX' => 2353, 'UcX' => 2354, 'urgX' => -2356, 'OrdX' => 2357, 'UStX' => 2358,
      'UltX' => 2359, ' Xni' => 2360, 'ylX' => 2361, 'UstX' => 2362, 'OtX' => 2363,
      ' yXru' => 2364, 'kXcu' => 2365, 'UsX' => 2366, 'kXlt' => 2367, 'UzX' => 2368,
      'UnlX' => 2369, 'Xle' => 2370, 'Xh' => -2372, ' Xc ' => 2373, 'UrdX' => 2374,
      'UlX' => 2375, ' mX' => 2376, ' Xst' => 2377, 'UrlX' => 2378, ' tXm' => 2379,
      'UnkX' => 2380, 'Xye' => 2381, ' bXt' => 2382, 'CX' => 2383, 'OlX' => 2384,
      ' sXr' => 2385, 'UtX' => 2386, 'OzX' => 2387, 'ygX' => -2388, 'UmX' => 2389,
      'UGX' => 2390, 'USX' => 2392, 'hXkum' => 2393, 'yXz' => 2394, 'dXs' => 2395,
      ' dXn' => 2396, 'UkX' => 2397, 'Xlk' => 2398, 'bXyu' => 2399, 'UrX' => 2400,
      'Xre' => 2402, 'OrX' => 2403, 'yXk' => 2404, 'Xze' => 2405, 'tXrk' => 2406,
      'gX' => 2407, 'X' => -2408
    },
    'i' => {
      'n kXsan' => -1, ' nin Xn' => -2, 'tIyor X' => -3, ' armanX' => -4,
      'Xstirab' => 5, 'aktXgim' => 6, 'eci Xsi' => 7, 'er de X' => -8, 'ere Xsi' => 9,
      'ne takX' => 10, 'ratan X' => -11, 'Uyen Xn' => 12, ' alanXs' => -13, 'sI sXr ' => -14,
      'ivan mX' => 15, 'iran Xn' => 16, 'na katX' => 17, 'ka katX' => 18, 'fsIz lX' => 19,
      'ble X ' => 20, 'fbI lX' => 21, 'Cal mX' => 22, ' otanX' => -23, 'yevarX' => -24,
      'c k nX' => -25, 'when X' => 26, 'ortarX' => -27, 'an e X' => 28, '  mXnc' => 29,
      'le rXo' => 30, 'und Xn' => 31, 'zaks X' => 32, 'lain X' => 33, 'daS lX' => 34,
      ' laklX' => -35, 'endarX' => -36, 'rsal X' => -37, 'nat sX' => 38, 'tovanX' => -39,
      'Iz mXy' => 40, 'ilallX' => -41, 'yanC X' => 42, 'solarX' => -43, 'scu sX' => -44,
      'gi acX' => -45, 'glantX' => -46, 'inlarX' => -47, 'glarXz' => -48, 'ikmamX' => -49,
      'asadXg' => -50, 'k acXs' => -51, 'l acXk' => -52, 'c sayX' => -53, '6 yasX' => -54,
      'asamX ' => -55, 'asamXs' => -56, 'stayXm' => -57, 'sXnanm' => 58, 'r cXne' => 59,
      'in deX' => 60, 'ears X' => -61, 'nanayX' => -62, ' farX ' => 63, 'az Xnk' => 64,
      ' aSXya' => -65, 'k stXn' => 66, 're dX ' => 67, 'ayaldX' => -68, ' arXs ' => -69, 'r mXri' => 70,
      'mhk nX' => -71, 'Ul clX' => 72, 'ssa kX' => 73, 'itladX' => -74, 'islarX' => -75,
      ' sXniz' => -76, 'tXciyd' => -77, 'Im dXr' => 78, 'yara X' => -79, 'ol Xn ' => 80,
      'ru bX ' => 81, 'al Xni' => 82, 'bI Xnt' => -83, 'nike X' => 84, 'sisasX' => -85,
      'rodalX' => -86, 'InC Xn' => 87, 'kuk tX' => 88, 'rI rXf' => 89, 'Ip mX ' => 90,
      'sXzinc' => 91, 'am akX' => -92, 'fXtild' => 93, 'ur anX' => 94, 'ah mX ' => 95,
      'n hXr ' => -96, 'kal lX' => 97, '  kXng' => 98, 'I aczX' => 99, 't mXst' => 100,
      'mu Xnt' => 101, ' day X' => -102, ' nasX ' => -103, 'ars sX' => -104, '4 takX' => 105,
      'miranX' => -106, 'mir nX' => 107, 'an cX ' => 108, 'fayI X' => 109, 'ayrXni' => 110,
      '  Xnfo' => 111, 'cu Xl ' => 112, ' alXmd' => -113, 'partlX' => -114, 'non bX' => 115, 'k pXli' => 116,
      'ISI lX' => 117, 'trt lX' => -118, 'zans X' => 119, 'oru tX' => -120, 'a asXa' => 121,
      'iGe Xs' => 122, ' u adX' => -123, 'tman X' => -124, 'alXmid' => -125, 'morasX' => -126,
      'rXlin ' => -127, 'rter X' => 128, ' taS X' => 129, 'gIt gX' => 130, 'zal sX' => 131,
      'nI mXz' => 132, ' I alX' => -133, 'gast X' => 134, 'pper X' => 135, 'UlmalX' => -136,
      'inaldX' => -137, 'Uk Xnt' => 138, 'ra fXs' => 139, 'ften X' => 140, 'ot Xsi' => 141,
      ' Xsinc' => 142, 'an Xmi' => 143, 'side X' => 144, 'zlU Xs' => 145, 'el asX' => -146,
      'i akXb' => -147, 'bIk mX' => 148, 'nne sX' => -149, 'yder X' => 150, 'imsalX' => -151,
      ' gandX' => -152, 'r lXkt' => 153, 'st zXn' => 154, 'amXss' => 155, 'slamXs' => -156,
      'virXl ' => 157, 'ar dX ' => 158, 'si rXf' => -159, 'rke X ' => 160, 'sXzliy' => 161,
      ' cazX ' => 162, 'urantX' => 163, 'Cta tX' => -164, 'sarfX ' => 165, 'ark X ' => 166,
      'urlerX' => 167, 'e falX' => -168, '  dXsn' => 169, ' may X' => -170, 'ni joX' => 171,
      'in Xso' => 172, 'umentX' => 173, 'CuvalX' => 174, 'a Xce ' => 175, 'etallX' => -176,
      'marXyi' => -177, 've fXf' => -178, 'rhallX' => 179, 'Gi anX' => 180, ' hatXm' => -181,
      'sXnana' => 182, 'ght X ' => 183, ' ceksX' => 184, 'halabX' => -185, 'an cXv' => -186,
      'ne Xmi' => 187, 'dden X' => 188, 'nap nX' => 189, 'Xrayim' => 190, 'ncam X' => 191,
      'saattX' => -192, 'n u sX' => -193, 'krem X' => 194, 'rake X' => 195, 'agI sX' => -196,
      'gi sXz' => 197, 'ns e X' => 198, 'z tamX' => -199, 'rane X' => 200, 'miS tX' => -201,
      'dent X' => 202, 'n pXrv' => 203, 'tasXya' => -204, 'sinasX' => -205, 'lu asX' => -206,
      'eans X' => -207, 'vaS mX' => 208, 'justXn' => 209, 'iryakX' => -210, 'nba nX' => -211,
      'ay mXy' => 212, 'eday X' => -213, 'dame X' => 214, 'Ir tXp' => 215, ' dandX' => -216,
      'sUratX' => -217, 'Xsilip' => 218, 'eraatX' => -219, ' fazX ' => -220, 'elallX' => -221,
      'htarXk' => -222, 'tIn tX' => 223, 'ras nX' => 224, 'kIr kX' => -225, 'ast mX' => 226,
      'h lXdi' => 227, 'habatX' => -228, 'a yXm ' => 229, 'Si kXr' => -230, 'sXllik' => -231,
      'tehabX' => 232, 'Xsinir' => 233, 'mmer X' => 234, 'sinanX' => -235, 'aC mX ' => 236,
      'ksXyor' => 237, 'e cXne' => 238, ' mayX ' => -239, 'O parX' => -240, 'istamX' => -241,
      'n Xnki' => 242, 'or Xnk' => 243, 'tahrX ' => 244, 'Xnirse' => -245, 'icanX ' => -246,
      'ark mX' => 247, 's aslX' => -248, 'hire X' => 249, 'al nXn' => 250, 'entarX' => -251,
      'iyov X' => 252, 'ova Xn' => 253, 'wn dX ' => 254, 'hhatsX' => -255, 'ha Xli' => 256,
      'Ca adX' => -257, 'lUz tX' => 258, 'inyanX' => -259, 'garatX' => -260, 'zce Xn' => 261,
      'kite X' => 262, '2 hafX' => -263, 'zca dX' => -264, 'evallX' => -265, 'tan aX' => -266,
      'hak kX' => 267, 'markXm' => -268, 'a cXne' => 269, 'ep tXr' => -270, 'ikat X' => -271,
      'et asX' => -272, 'ersanX' => -273, 'b I kX' => -274, 'nda X ' => 275, 'ek CXg' => -276,
      'kIr fX' => 277, 'ehatXn' => -278, 'UlmasX' => -279, 'at tXp' => 280, 'Xspana' => 281,
      'raz dX' => -282, 'e CXli' => 283, 'lu ayX' => -284, 'unarX ' => -285, 'lplarX' => -286,
      'rI tXp' => 287, 'Ik mXy' => 288, 'i astX' => -289, 'efrayX' => -290, ' tadXl' => -291,
      'alI yX' => 292, 'lI anX' => 293, 'kers X' => -294, 'ce anX' => 295, ' opalX' => -296,
      'elds X' => -297, 'Xmpexb' => 298, 'taktXm' => 299, 'GI Xsr' => -300, 'bahatX' => -301,
      'tar Xi' => -302, 'takXs ' => -303, 'i 8 kX' => -304, 'itarXs' => -305, 'n Xvan' => 306,
      'rksXz ' => 307, 'du sXn' => -308, 'eim Xn' => 309, 'ap sXz' => 310, 'garXn ' => 311,
      'sen mX' => -312, 'e patX' => -313, 'hatXn ' => -314, 'GI anX' => 315, 'Xlinik' => -316,
      'b adlX' => -317, 'nar Xn' => 318, 'sra sX' => -319, 'kXral ' => 320, 'ika lX' => -321,
      'za anX' => 322, 'tUn dX' => -323, 'tCXsiy' => -324, 'mli tX' => 325, 'mak tX' => 326,
      'tiyarX' => -327, ' alXya' => -328, 'de agX' => 329, 'rda Xn' => -330, 'lk kXz' => -331,
      'IkantX' => 332, 'h sXz ' => 333, 'jinalX' => -334, ' ki CX' => -335, 'Si akX' => -336,
      'ak asX' => 337, 'ubranX' => -338, ' ne tX' => -339, 'ki sXr' => 340, '  aStX' => -341,
      'afXni ' => 342, 'Xsiliy' => 343, 'alk mX' => 344, 'zde dX' => -345, 'Xriste' => -346,
      'arsa X' => -347, 'arIm X' => -348, 'Gar Xn' => 349, 'C malX' => 350, 'dere X' => -351,
      '0 bayX' => -352, 'yuz yX' => -353, 'n dXsl' => 354, 'k artX' => 355, 'zun Xs' => 356,
      'iladX ' => -357, 'beralX' => -358, 'zi Xsi' => 359, 'min Xn' => -360, 'enal X' => -361,
      'havarX' => -362, 'ntartX' => -363, 'sakXnl' => -364, 'apardX' => 365, 'klasXk' => -366,
      'aens X' => 367, 'eve X' => -368, 've Xn ' => 369, 'aten X' => -370, 'gat Xn' => 371,
      'ri tXp' => 372, 'z katX' => 373, 'SeranX' => -374, 'osmanX' => -375, 'talXma' => -376,
      'l basX' => 377, 'in dXr' => -378, 'falsXz' => 379, 'la anX' => 380, 'rat tX' => 381,
      'U takX' => 382, 'dIk dX' => -383, 'Ip Xn ' => 384, 'Xninda' => 385, 'zanovX' => 386,
      ' yattX' => 387, 'az kXl' => 388, 'kanadX' => 389, 'zalp X' => -390, 'gUn Xn' => -391,
      'alXnda' => 392, ' danX ' => -393, 'kesi X' => -394, 'otansX' => -395, 'larXna' => 396,
      'l da X' => -397, 'zel tX' => -398, 'l tXp ' => 399, ' ardXn' => 400, ' olabX' => -401,
      'okrasX' => -402, 'ler Xn' => -403, ' Xndir' => -404, 'nasrX' => -405, 'lakXr' => 406,
      'Xkkim' => -407, 'chapX' => -408, ' If X' => 409, 'ims X' => -410, 'Xrkpa' => -411,
      'er Xg' => 412, 'ddahX' => 413, 'i Xso' => 414, 'sassX' => 415, 'urtXd' => 416,
      'lXmis' => 417, 'ach X' => 418, ' Xnke' => 419, 'a CX ' => 420, 'baf X' => 421,
      'eig X' => 422, ' asXd' => -423, ' fXr ' => 424, 'mIs X' => 425, 'liyXl' => 426,
      'sralX' => -427, 'slatX' => -428, 'lasXr' => -429, 'ogalX' => -430, 'imacX' => -431,
      'payrX' => 432, 'amm X' => 433, '  sX ' => 434, 'Xsrae' => 435, 'klazX' => -436,
      'atf X' => 437, ' Xstr' => 438, 'rammX' => -439, ' Xsto' => 440, 'sXvan' => 441,
      'gayXr' => -442, 'wh nX' => 443, 'hk yX' => -444, 'saskX' => -445, 'eu nX' => -446,
      'natXg' => -447, 'w fbX' => -448, 'Xmano' => 449, 'CXtas' => 450, ' nXm ' => 451,
      'acs X' => 452, 'ih nX' => 453, 'lfasX' => -454, 'lk X ' => 455, 'sXzis' => 456,
      ' Xtar' => 457, 'rhIrX' => -458, 'tInXk' => -459, 'I tXn' => -460, 'nharX' => -461,
      'Xminy' => -462, 'dm lX' => 463, 'ys X ' => 464, 'nty X' => 465, 'ai tX' => 466,
      'rSadX' => -467, 'dXgma' => -468, 'rty X' => 469, 'canXp' => -470, 't Xro' => 471,
      'vhamX' => 472, 'idalX' => -473, 'I fXf' => -474, 'dXraj' => 475, 'cebXt' => 476,
      'nahXr' => 477, 'uff X' => 478, 'bIs X' => 479, '40 nX' => 480, 'daSdX' => 481,
      'damsX' => 482, ' Xrib' => 483, 'ndt X' => 484, 'ilavX' => 485, 'ecalX' => -486,
      ' Xllo' => 487, 'I vXp' => 488, 'l dX ' => 489, 'Xrati' => 490, 'a c X' => 491,
      'h ayX' => -492, 'akslX' => 493, 'S Xm ' => 494, 'tarXt' => 495, 'II sX' => -496,
      'bst X' => 497, 'ogarX' => 498, 'nalmX' => -499, 'c mX ' => 500, 'kXvi ' => -501,
      'd sXz' => 502, 'dXmim' => 503, ' arXo' => 504, 'dasXl' => -505, 'msahX' => 506,
      'n lXy' => 507, 'SamdX' => 508, ' SasX' => -509, '6 dXr' => 510, 'pe fX' => 511,
      ' fXtt' => 512, 'a Xgl' => 513, 'Xyars' => -514, 'rfatX' => -515, ' mXkl' => 516,
      'isatX' => -517, 'esamX' => -518, ' nXku' => 519, 'zangX' => 520, 'unasX' => -521,
      'canXd' => -522, 'h mXt' => 523, 'hs mX' => 524, 'azatX' => -525, 'nXsiz' => -526,
      ' naSX' => -527, 'CaprX' => 528, 'rXmad' => 529, 'fhamX' => 530, ' mXci' => 531,
      'UpabX' => 532, 'ccarX' => 533, 'k Xmi' => -534, ' epX ' => 535, '0 ydX' => 536,
      'ack X' => 537, 'aco X' => 538, 'ib nX' => 539, 'Xrizi' => -540, 'phy X' => 541, 'bakrX' => 542,
      'ie tX' => 543, 'U fXa' => 544, 'ic Xc' => 545, ' IrXs' => -546, 'ie Xr' => 547,
      'th Xc' => 548, ' dXc ' => 549, 'S Xnk' => 550, 'GartX' => 551, 'ss kX' => 552,
      'CXtam' => 553, 'i igX' => 554, 'uus X' => 555, 'icatX' => -556, 'antCX' => 557,
      'is tX' => -558, ' fXka' => 559, 'unk X' => 560, 'he X ' => 561, ' Xndy' => 562,
      'dangX' => 563, 'badX ' => -564, 'm cXz' => 565, 'ony X' => 566, ' Xss ' => 567,
      'I Xni' => 568, '  hXh' => 569, 'usIkX' => -570, 'n tasX' => 571, ' tasX' => -572,
      'a zXm' => -573, 'i fXf' => -574, ' avXn' => 575, ' adXa' => 576, 'Xvarl' => 577,
      'itadX' => -578, 'lhadX' => -579, 'arXer' => 580, ' lXym' => 581, 'yahXm' => 582,
      ' ag X' => 583, 'hXzma' => 584, 'lannX' => 585, 'hasXp' => -586, '6 ydX' => 587,
      ' vXz ' => 588, 'CXtla' => 589, 'IC mX' => 590, 't yXd' => 591, 'pXlis' => 592,
      'th tX' => -593, 'aafXm' => 594, 'hparX' => -595, 'cazXn' => 596, 'lXnok' => 597,
      'CXtak' => 598, 'IkatX' => 599, 'Xsilk' => 600, 'baksX' => 601, 'UganX' => -602, '  vXp' => 603,
      'CakSX' => 604, 'C tXn' => 605, 'adXlm' => 606, 'halXk' => 607, 'ein X' => 608,
      'y rXf' => -609, 'gallX' => -610, 'hi tX' => 611, 'us kX' => -612, 'nXsab' => 613,
      'safXz' => 614, 'rtXma' => 615, 'fadX ' => -616, 'bXkir' => -617, 'iaf X' => 618,
      'Xncio' => -619, 'y CXt' => 620, 'ryanX' => -621, 'Xrilo' => -622, 'uvafX' => 623,
      'f anX' => 624, 'Sk Xn' => 625, 'IC Xn' => 626, 'rmXn ' => -627, 'oy nX' => 628,
      'rkXs ' => -629, ' nanX' => -630, 'r ahX' => 631, '  mbX' => 632, 'h yX ' => 633,
      'mazXm' => -634, 'cIlkX' => 635, ' k rX' => 636, 'igarX' => 637, 'eyamX' => -638,
      'utafX' => 639, 'Xmper' => 640, 'granX' => -641, 'ealdX' => -642, 't lX ' => 643,
      'tXnaz' => 644, 'SahX ' => 645, '2 asX' => -646, 'Xlivi' => 647, 'kulXk' => 648,
      'a dXm' => 649, 'bazXd' => -650, 'tacXm' => 651, 'd Xll' => 652, 'rdXnl' => -653,
      'lamdX' => 654, 'haldX' => -655, 'um fX' => -656, 'fcanX' => -657, 'abXk ' => 658,
      'rz Xn' => 659, 'e Xzg' => 660, 'rnXzl' => -661, 'ad lX' => 662, 'urakX' => 663,
      'htacX' => 664, 'tip X' => -665, 'aIn X' => -666, ' Xlid' => 667, 'yantX' => 668,
      'd Xnc' => 669, 'saysX' => 670, 'ap tX' => 671, 'ku X ' => 672, ' SafX' => -673,
      'C anX' => 674, 'taatX' => -675, 'SahlX' => 676, 'ke fX' => 677, 'aph X' => 678,
      '85 kX' => -679, 'kc nX' => 680, 'CXlav' => -681, 'uzXr ' => 682, 'Xsida' => 683,
      ' Xsis' => 684, 'tXlic' => 685, 'u nXn' => 686, 'hazmX' => 687, 'stCX ' => 688,
      'vrakX' => 689, 'nS X ' => 690, 'stIkX' => -691, 'ocamX' => 692, 'asmXn' => -693,
      'uay X' => -694, ' Xsic' => 695, '94 dX' => 696, 'Xvasl' => 697, 'Xragi' => 698,
      'dalsX' => 699, 'sakXl' => -700, 'naazX' => -701, 'raasX' => -702, 'r Xii' => -703,
      'or Xi' => -704, 'hassX' => 705, 'Is Xs' => -706, ' acXr' => 707, 'v dXn' => 708,
      'si Xv' => 709, ' SXkl' => 710, 'ele X' => -711, 'dk sX' => 712, 'nI lX' => -713,
      'I lX ' => 714, 'hardX' => -715, 'ead X' => -716, 'Xllio' => 717, ' tosX' => 718,
      'Xray ' => -719, 'lXnay' => -720, 'I ahX' => 721, 'ibXl ' => 722, 'g h X' => 723,
      'CtakX' => -724, 'bXtas' => 725, ' 6 lX' => 726, 'tikXl' => 727, 'sekXn' => 728,
      ' Xcar' => 729, 'tu tX' => 730, 'e Xt ' => 731, 'Ik rX' => -732, 's tXp' => 733,
      'usk X' => 734, 'd asX' => -735, 'serbX' => 736, 'n jXn' => 737, 'masrX' => -738,
      'aadXg' => 739, ' Xnt ' => 740, 'Xtari' => -741, 'acIbX' => -742, 'UlayX' => -743,
      'rstX ' => 744, 'r zXv' => 745, ' nafX' => -746, 'afalX' => 747, 'mazmX' => 748,
      'arlXd' => 749, ' kXsl' => 750, 'kXtan' => 751, 'Xtma ' => 752, 'allXt' => -753,
      'Xspon' => -754, 'tl sX' => 755, 'oparX' => 756, 'kXrti' => 757, 'tchkX' => 758,
      'gXyim' => -759, 'mon X' => 760, 'p anX' => 761, 'lasXn' => 762, 'tardX' => 763,
      'akXz ' => 764, ' gs X' => 765, 'Xbler' => -766, 'garSX' => -767, 'ramdX' => 768,
      'var X' => -769, 'r hXy' => -770, 'etamX' => -771, 'Il X ' => 772, 'nalXk' => 773,
      'Xnine' => -774, 'ng Xt' => 775, 'ory X' => 776, 'a Xsa' => -777, 'nsarX' => -778,
      'S atX' => 779, 'rz X ' => 780, 'nrakX' => -781, 'Xtmas' => 782, 'alkXs' => 783,
      'yI Xn' => -784, 'bim X' => -785, 'Xrla ' => 786, 'kXnde' => -787, 'vallX' => 788,
      'rXsik' => 789, 'ktXla' => 790, ' avX ' => 791, 'kXyam' => 792, 'ma Xn' => -793,
      'nla X' => -794, 's sXl' => -795, 'samXm' => -796, 'lp Xn' => 797, 'ttakX' => -798,
      'nazXl' => -799, ' CXna' => 800, 'acXni' => 801, 'un Xn' => -802, 'rescX' => 803,
      'lahtX' => -804, 'lkXng' => -805, 'yku X' => 806, 'ues X' => 807, ' rasX' => -808,
      'Xkla ' => 809, 'sXnli' => -810, 'Xmall' => -811, 'Xrtak' => -812, 'rattX' => 813,
      'sastX' => 814, 'tlarX' => 815, 'stakX' => -816, 'arXm ' => 817, 'lattX' => 818,
      'tIrdX' => 819, ' fatX' => -820, 'aldXk' => 821, 'rkasX' => 822, 'yrXmi' => 823,
      'i Xns' => -824, 'Xtart' => 825, 'Xktan' => 826, 'yXlma' => 827, 'ln X' => 828,
      'kkXo' => 829, 'kXsr' => 830, 'w X ' => 831, 'nIkX' => -832, ' bsX' => 833,
      'IytX' => 834, 'Xdg ' => 835, 'cXls' => -836, ' Xnv' => 837, 'fXgl' => 838,
      'vajX' => -839, 'IomX' => 840, 'uzyX' => -841, 'Xcef' => 842, 'Xonc' => 843,
      'oabX' => 844, 'Xngv' => 845, 'idlX' => -846, ' Xol' => 847, ' SyX' => -848,
      'nr X' => 849, 'lunX' => 850, 'mXg ' => 851, 'iGXi' => 852, ' Xrn' => -853,
      'cXss' => 854, 'IdaX' => 855, ' npX' => 856, 'gp X' => 857, 'rzdX' => 858,
      'hXzd' => 859, 'rsXf' => 860, 'yflX' => 861, 'ngXt' => 862, ' Xrs' => -863,
      'fXch' => -864, 'rrrX' => 865, 'yavX' => 866, 'ryXm' => 867, ' Xmc' => 868,
      'adXc' => -869, 'kXlk' => 870, 'yXta' => -871, 'mXhi' => 872, 'gXbt' => 873,
      'Xspu' => -874, ' Xyd' => 875, 'uStX' => 876, 'Xyig' => 877, 'kl X' => 878,
      'kXpa' => 879, 'tXnn' => 880, 'epmX' => -881, 'ipmX' => -882, 'Xdoo' => -883,
      'Xgam' => 884, 'avmX' => 885, 'kXlt' => 886, 'Xmmu' => 887, 'Xmia' => 888,
      'rSmX' => 889, 'Xgab' => 890, 'yXgs' => 891, ' lmX' => 892, ' zmX' => 893,
      ' tmX' => 894, 'nCXl' => -895, 'IteX' => 896, 'rseX' => 897, 'ozlX' => 898,
      'ysrX' => 899, 'SktX' => 900, 'hs X' => 901, 'ibaX' => 902, 'IsrX' => 903,
      'llXu' => 904, 'jalX' => -905, 'ytXm' => 906, 'mXzr' => 907, 'Id X' => 908,
      'IsaX' => 909, 'rSlX' => 910, 'Xkut' => -911, 'lmCX' => 912, 'ySrX' => 913,
      'ytXk' => 914, 'htsX' => 915, 'InCX' => 916, 'cXag' => 917, ' zX ' => 918,
      'yfXf' => 919, ' Xga' => 920, 'Xle' => -921, 'wn X' => 922, 'cXlg' => 923,
      'tXy ' => -924, 'rngX' => 925, 'ckCX' => 926, 'IGnX' => 927, 'fXtr' => 928,
      'aXrc' => 929, ' Xfm' => 930, 'ahaX' => 931, 'Xcbm' => 932, 'Xvoj' => -933,
      'anXf' => -934, 'hXzv' => 935, 'rhaX' => 936, 'aySX' => 937, 'cXzz' => 938,
      'IGmX' => 939, 'Xyav' => -940, 'nXny' => -941, 'asXv' => -942, ' Xaf' => 943,
      'IrvX' => 944, ' Xrv' => 945, ' hnX' => 946, 'sXsy' => 947, 'sr X' => 948,
      'yIhX' => -949, 'srXl' => -950, ' kXk' => 951, 'GunX' => 952, 'cXlk' => 953,
      'watX' => -954, 'bXc ' => -955, 'lXvi' => -956, 'xarX' => -957, 'Gr X' => 958,
      'pXtr' => 959, 'ndoX' => 960, 'ctXr' => 961, 'Xnch' => -962, 'anCX' => 963,
      'SndX' => 964, 'myoX' => 965, 'rzCX' => 966, 'iIGX' => -967, 'aCcX' => 968,
      'GahX' => 969, 'aSrX' => 970, 'zafX' => -971, 'dzaX' => 972, 'jakX' => -973,
      'zXlg' => 974, ' lXp' => -975, ' vXi' => 976, 'ktlX' => 977, 'stdX' => -978,
      'nSXr' => -979, 'nrXl' => 980, 'mmXs' => 981, 'SXps' => -982, 'Xdio' => -983,
      'IzkX' => 984, 'rXzk' => 985, 'ynXk' => -986, 'sfX ' => 987, 'yCX ' => 988,
      'llXm' => -989, 'aCCX' => 990, 'Xyop' => -991, 'dIkX' => -992, 'isCX' => -993,
      'sXku' => -994, 'IhlX' => 995, 'Xppo' => -996, 'kXvo' => -997, 'fXba' => 998,
      'cXlo' => -999, 'yeCX' => 1000, 'eatX' => -1001, 'Xgd ' => 1002, 'Xbal' => -1003,
      'yXgm' => 1004, 'rIzX' => -1005, 'Xpaz' => -1006, ' yaX' => 1007, 'InoX' => 1008,
      'tlmX' => 1009, 'olkX' => 1010, 'CmXn' => -1011, ' aXp' => 1012, 'kXca' => 1013,
      'tlXn' => -1014, 'eIGX' => -1015, 'nXar' => 1016, 'fSaX' => 1017, 'nclX' => 1018,
      'cXga' => 1019, 'Xrij' => -1020, 'd cX' => 1021, 'Xrst' => -1022, ' Xh ' => 1023,
      ' Xsu' => 1024, 'cXbi' => 1025, 'Xros' => -1026, 's Xh' => 1027, 'zXmn' => 1028,
      ' Xvi' => 1029, 'klXe' => 1030, 'G CX' => -1031, 'we X' => 1032, 'ftXg' => -1033,
      'gXlg' => 1034, 'aGmX' => 1035, 'cXs ' => -1036, 'Xkem' => -1037, 'nXdo' => 1038,
      'Xper' => -1039, 'lkXb' => -1040, 'shXg' => 1041, 'SXrg' => 1042, ' nXd' => -1043,
      'Xad ' => -1044, 'Xpik' => -1045, 'jXye' => -1046, 'ncXc' => -1047, 'mXkn' => 1048,
      ' abX' => -1049, 'rXc ' => -1050, 'mXnk' => -1051, 'sXng' => -1052, 'ts X' => -1053,
      'Xpek' => -1054, 'lXll' => -1055, ' Xlc' => -1056, 'zXme' => -1057, 'gXct' => 1058,
      'cXld' => -1059, 'nXka' => -1060, 'Xch ' => -1061, 'ecXc' => 1062, ' mXm' => -1063,
      'bXti' => -1064, 'nXko' => -1065, 'Xstl' => -1066, 'zylX' => 1067, 'Xset' => -1068,
      'r Xo' => 1069, 'fXbr' => 1070, ' stX' => -1071, 'ttXs' => -1072, 'lXan' => -1073,
      'Xyaz' => -1074, 'lImX' => 1075, 'Xst ' => -1076, ' Xzl' => -1077, 'Xzde' => -1078,
      'Xkit' => -1079, 'Xkel' => -1080, 'ldXz' => 1081, 'tXca' => -1082, 'intX' => -1083,
      'engX' => -1084, 'IzXl' => 1085, 'eskX' => -1086, 'CatX' => 1087, 'o cX' => -1088,
      'surX' => -1089, 'aynX' => 1090, 'Xdde' => -1091, 'yXll' => 1092, ' lXr' => -1093,
      ' Xbr' => -1094, 'Xagi' => 1095, 'nlXs' => 1096, 'Xfak' => -1097, 'kaCX' => 1098,
      'yarX' => 1099, 'sXc ' => -1100, 'Xkri' => -1101, 'cgX' => 1102, ' uX' => 1103,
      'GIX' => -1104, 'CdX' => 1105, 'pnX' => 1106, 'Xsf' => -1107, 'Xih' => -1108,
      'fmX' => 1109, 'Xlz' => -1110, 'Xae' => 1111, 'jjX' => 1112, 'Xom' => 1113,
      'jXv' => -1114, 'kGX' => 1115, 'cXf' => -1116, 'pcX' => 1117, 'Xho' => -1118,
      'fcX' => 1119, 'dSX' => 1120, 'SyX' => 1121, 'uGX' => 1122, 'Xbc' => 1123,
      'jmX' => 1124, 'Xdc' => 1125, 'pfX' => 1126, 'Xzo' => -1127, 'cXw' => 1128,
      'kXi' => -1129, 'egX' => -1130, 'cvX' => 1131, 'Xmu' => -1132, 'Xdr' => -1133,
      'cXp' => -1134, 'Xzn' => -1135, 'Xyy' => -1136, 'SXf' => -1137, 'zXb' => -1138,
      'guX' => -1139, 'iXl' => -1140, 'idX' => -1141, 'Xze' => -1142, 'IkX' => 1143,
      'edX' => -1144, 'ekX' => -1145, 'iSX' => -1146, ' Xc' => -1147, '2X' => 1148,
      'slarIn yanX ' => -1149, 've vicdanX ' => -1150, 'letin malX ' => 1151,
      'ok hayatX ' => -1152, 'den sXnirl' => -1153, 'de insanX ' => -1154, 'da hayatX ' => -1155,
      'ak hayatX ' => -1156, 'Xnformatio' => 1157, 'an insanX ' => -1158, ' inin yanX' => -1159,
      'kInda adlX' => -1160, 'i sOmestrX' => 1161, 'lecek yanX' => 1162, 'nacak yanX' => 1163,
      'senin malX' => 1164, 'gini artX' => -1165, 're ahlakX' => 1166, 'an moralX' => 1167,
      'Cin yanX ' => -1168, 'Il hayatX' => -1169, 'er ahlakX' => 1170, 'sin yanX ' => -1171,
      'ban taktX' => 1172, 'ni sXnir ' => -1173, 'ri sXnir ' => -1174, 'si Xntern' => 1175,
      've insanX' => -1176, 'In firarX' => 1177, 'gar washX' => 1178, 'ukada nX' => -1179,
      'Xsrafin ' => 1180, 'ordan kX' => -1181, 'In kIsmX' => -1182, 'in kastX' => -1183,
      'ise Xsin' => 1184, 'U Xntern' => 1185, 'se martX' => -1186, 'ili Xsin' => 1187, 'r mimarX' => -1188,
      'ya malX ' => 1189, 'mli yanX' => 1190, 'niz sakX' => 1191, 'rek adlX' => -1192, ' alXmin ' => -1193,
      'na taktX' => 1194, 'unu adlX' => -1195, 'i sakXn ' => 1196, 'neden dX' => -1197, 'ksek Xsi' => 1198,
      'nIn adlX' => -1199, 'itesi tX' => 1200, 'tak yanX' => 1201, ' In adlX' => -1202, 'I ve Xnt' => 1203,
      'men gazX' => 1204, 'hal malX' => 1205, 'zel yanX' => 1206, 'detli kX' => 1207, 'Xsitilme' => -1208,
      'after X' => 1209, 'basladX' => -1210, 'am dXr ' => 1211, 'ren Xnt' => 1212, 'ark Xn ' => 1213,
      'aktXrdi' => 1214, 'p camX ' => 1215, ' Xlgin ' => 1216, 'gemi dX' => -1217, 'ter dX ' => 1218, 'se sakX' => 1219,
      'binalX ' => -1220, 'Xversia' => 1221, 'tihbarX' => -1222, 'sIz mXs' => 1223, 'CXnliyo' => 1224,
      'ormandX' => -1225, ' artXn ' => -1226, 'din alX' => -1227, 'Uk alXm' => -1228, ' rahatX' => 1229,
      ' cumalX' => -1230, 'dI sakX' => 1231, 'si kXt ' => 1232, 'te Xnte' => 1233, 'adikalX' => -1234,
      'nin fXs' => 1235, 'dXsimi ' => -1236, ' kXlli ' => 1237, 'stner X' => 1238, 'uness X' => 1239,
      ' Xnter ' => 1240, 'a dXsli' => -1241, 'r dXsli' => -1242, '  dXsi ' => -1243, 'aksXniz' => 1244,
      'teryalX' => -1245, 'dXsiyle' => -1246, 'ne katX' => 1247, 'sanlXt' => -1248, ' ast X' => 1249,
      'camlXk' => -1250, 'askanX' => -1251, 'ilmasX' => -1252, 'iklarX' => -1253, ' karsX' => -1254,
      'rlanX ' => -1255, 'eSkalX' => -1256, 'ikadX ' => -1257, 'dU ayX' => -1258, 'Xrtill' => 1259,
      'ans mX' => 1260, 're Xnt' => 1261, 'baldX ' => -1262, 'r fXs ' => 1263, 'at mXs' => 1264,
      'Xnayac' => 1265, 'maatsX' => -1266, 'a lXk ' => 1267, 'eftalX' => -1268, 'Xrmanl' => -1269,
      'cadXll' => -1270, 'or Xn ' => 1271, 'asalsX' => 1272, ' arXti' => 1273, 'eles X' => 1274,
      'kahXr ' => 1275, 'vuz fX' => -1276, 'sea nX' => -1277, 'daG dX' => 1278, 'atXlid' => -1279,
      'imallX' => -1280, 'ak tX ' => 1281, 'dya mX' => 1282, 'dakatX' => -1283, 'epal X' => -1284,
      'rhatlX' => 1285, 'hamasX' => -1286, 'rI CXg' => -1287, 'martX ' => 1288, 'asch X' => 1289,
      'si hXd' => 1290, 'as lX ' => 1291, 'lganX ' => -1292, 'play X' => -1293, 'taktXn' => 1294,
      'ay dX ' => 1295, 'Ca sXn' => -1296, 'prkacX' => -1297, 'alamXz' => -1298, 'ylanX ' => -1299,
      ' aC mX' => 1300, ' sabX ' => -1301, ' set X' => 1302, 'sarXn ' => -1303, ' SarX ' => -1304,
      'n mXyd' => 1305, 'mbanX ' => -1306, ' dalXd' => -1307, 'uranX ' => -1308, ' zatX ' => -1309,
      'a mXzi' => 1310, 'ak Xnt' => -1311, 'Xmages' => 1312, ' nakXs' => 1313, 'Xsisi ' => -1314,
      'diyadX' => -1315, 'I Xnte' => 1316, 't tXr ' => -1317, 'ille X' => -1318, 'ismanX' => -1319,
      'of Xsl' => 1320, 'tXlin ' => -1321, ' varX ' => -1322, 'camXna' => 1323, 'e calX' => -1324,
      'ar lX ' => 1325, 'kralcX' => 1326, 'ar mXd' => 1327, ' kamX ' => -1328, 'alXya ' => 1329,
      'Xsabel' => 1330, ' platX' => -1331, ' garX ' => 1332, ' Xnkin' => 1333, 'arabX ' => -1334,
      ' rakX ' => 1335, '  dXsl' => -1336, 'bXckin' => 1337, 'rtre X' => 1338, 'azXni ' => 1339,
      'r Xni ' => 1340, 'In sXl' => -1341, 'erganX' => -1342, 'ikal X' => -1343, 'Iz Xn ' => 1344,
      'en Xnt' => -1345, 'arabXn' => 1346, 'Xsinde' => -1347, 'nvarX' => -1348, 'tahXl' => 1349,
      'zalcX' => 1350, 'ralXo' => 1351, ' sasX' => -1352, 'UnafX' => 1353, 'g Xnt' => 1354,
      'o nXn' => 1355, 'menkX' => 1356, '60 nX' => 1357, 'Xvila' => 1358, 'Xhlam' => 1359,
      'krazX' => -1360, ' Xnaf' => 1361, 'larkX' => -1362, 'ItkX ' => 1363, 'm Xss' => 1364,
      'Xmil ' => 1365, 'edayX' => -1366, 'rb X ' => 1367, 'fXtig' => 1368, 'aS tX' => 1369,
      'salcX' => 1370, 'cXlal' => -1371, 'lI Xg' => 1372, 'icalX' => -1373, 'Xfki ' => 1374,
      'drXa ' => 1375, 'zaa X' => 1376, 'kXrad' => -1377, 'nd X ' => 1378, 'iSarX' => -1379,
      'Xzint' => 1380, 'Iz X ' => 1381, 'oi nX' => 1382, 'ozayX' => -1383, 'k Xso' => 1384,
      'erptX' => -1385, 'ck CX' => 1386, 'yvalX' => 1387, 'cXmbi' => 1388, 'fasXt' => -1389,
      'hhatX' => -1390, 'zIkkX' => -1391, 'balXr' => -1392, 'zXlka' => 1393, 'Xnise' => -1394,
      'h kXl' => 1395, 'a cX ' => 1396, 'sualX' => -1397, 'SisXr' => 1398, 'bayXs' => -1399,
      'cXla ' => -1400, 'saytX' => 1401, 'asbXh' => 1402, 'ac X ' => 1403, ' zXri' => 1404,
      'adXga' => -1405, 'Xslav' => -1406, ' gIlX' => -1407, ' gXli' => 1408, 'fkatX' => -1409,
      'k lXy' => 1410, 'dXski' => 1411, 'a nXz' => 1412, 'gUnXs' => 1413, 'I Xme' => 1414,
      'dXsit' => -1415, 'Is lX' => 1416, 'thamX' => 1417, ' cXng' => 1418, 'galdX' => -1419,
      'shabX' => 1420, ' Xna ' => 1421, 'brIsX' => -1422, 'kaatX' => -1423, 'bbXni' => 1424,
      'Xndia' => 1425, 'keltX' => 1426, 'caalX' => -1427, 'kahrX' => 1428, 'sXviy' => 1429,
      'falcX' => 1430, 'I rXf' => -1431, 'cardX' => -1432, ' ramX' => -1433, 'kXyos' => -1434,
      'fiG X' => -1435, 'mp Xn' => 1436, 'Xvank' => 1437, 'gXdim' => 1438, 'Xrcal' => 1439,
      'atXls' => 1440, ' Xdor' => 1441, 'vaklX' => 1442, 'CXnga' => 1443, 'CakcX' => 1444,
      'h sXn' => 1445, 'aC X ' => 1446, ' nXza' => -1447, 'kXsip' => 1448, ' al X' => -1449,
      'Xbetl' => -1450, 'Xnson' => -1451, ' laCX' => -1452, 'fXsir' => 1453, 'azXnd' => 1454,
      'saksX' => 1455, 'kahlX' => 1456, 't Xmi' => 1457, 'tch X' => 1458, 'kabX ' => 1459,
      'Xcki ' => -1460, 'ndXs ' => -1461, 'fahlX' => 1462, 'zXvan' => 1463, 'IC X ' => 1464,
      'Xpta ' => 1465, 'kXsik' => 1466, 'fXtik' => 1467, 'kXrla' => 1468, 'Xrpik' => -1469,
      'IS X ' => 1470, 'dattX' => 1471, 'Xsila' => 1472, 'lazgX' => -1473, 'kbalX' => -1474,
      'klalX' => -1475, 'labX ' => 1476, 'ipatX' => -1477, 'irasX' => 1478, 'k avX' => 1479,
      'akanX' => 1480, 'Xnar ' => 1481, 'InalX' => 1482, 'dIrdX' => 1483, 'allXs' => -1484,
      'asXna' => 1485, 'eratX' => -1486, 'karlX' => 1487, 'rklX ' => 1488, 'rtCX' => 1489,
      'Xsbc' => 1490, 'rXg ' => -1491, 'amaX' => 1492, 'kdXs' => 1493, 'ilGX' => -1494,
      ' aXm' => 1495, 'mXsr' => 1496, 'msXe' => 1497, 'Xnaj' => -1498, ' Xf ' => 1499,
      ' 0 X' => 1500, 'Xstt' => -1501, 'rktX' => 1502, ' pXh' => 1503, 'hsXm' => 1504,
      'sXft' => -1505, 'InnX' => 1506, 'j X ' => 1507, 'mr X' => 1508, ' zXd' => 1509,
      'akXd' => -1510, 'v Xv' => 1511, 'zXrt' => 1512, 'nXsk' => -1513, 'xanX' => -1514,
      'apSX' => 1515, 'tGX ' => 1516, 'ICtX' => 1517, 'l Xv' => 1518, 'uadX' => -1519,
      'acXr' => -1520, 'pXro' => -1521, 'sXpr' => 1522, 'dXi ' => -1523, '5 tX' => -1524,
      'd Xs' => -1525, 'rnX ' => 1526, 'tXg ' => 1527, 'dabX' => 1528, 'Xsch' => -1529,
      '0 Xs' => -1530, 'rx X' => 1531, 'nXnn' => -1532, 'spX ' => 1533, 'lXki' => -1534,
      'ipCX' => -1535, '0 Xl' => -1536, 'tXkk' => -1537, 'Xmat' => -1538, 'rXbi' => -1539,
      ' zXp' => 1540,  'brXk' => -1541, 'Xpak' => -1542, 'gXla' => 1543, 'papX' => -1544, 'IrtX' => 1545,
      'plXn' => -1546, 'iddX' => -1547, 'Xns ' => -1548, 'Xgit' => -1549, 'rXa ' => -1550,
      'Xnce' => -1551, 'layX' => 1552, 'kdXr' => -1554, 'IsXy' => 1555, 'amrX' => 1556,
      'alXg' => 1557, 'Xyle' => -1558, 'rXll' => -1559, 'faSX' => -1560, 'Xyet' => -1561,
      ' Xmz' => -1562, 'Xtat' => -1563, ' Xki' => -1564, 'Xkte' => -1565, 'Xbg' => 1566,
      'hcX' => 1567, ' Xq' => 1568, 'vvX' => 1569, 'GtX' => 1570, 'iXg' => 1571,
      'tXj' => -1572, 'IXc' => -1573, 'Xkv' => -1574, 'CrX' => 1575, 'jXt' => -1576,
      'Xol' => -1577, 'Xd ' => -1578, 'Xdz' => -1579, 'eyX' => -1580, 'aGX' => 1581,
      'etX' => -1582, 'irX' => -1583, 'Xw' => -1584, ' alan arkX' => -1585, 'e mUtevazX' => -1586,
      'san hayatX' => 1587, 'an hayatX ' => -1588, 'Sbakan lX' => -1589, 'u hayatX ' => -1590, 'el insanX' => -1591,
      'arker Xn' => 1592, 'r SakXr ' => 1593, 'un aslX ' => -1594, 'apital X' => -1595, ' ve adlX' => -1596,
      'u SahsX ' => 1597, ' dXstan ' => 1598, 'ter katX' => -1599, 'in aslX ' => -1600, 'e kIsmX ' => -1601,
      'tralXnin' => -1602, 'In sabrX' => 1603, 'an Xsin ' => 1604, 'yasamXn' => -1605, 'sakXni ' => -1606,
      'Xstirap' => 1607, 'kXstin ' => -1608, 'di hacX' => -1609, 'akikatX' => -1610, 'a mXsin' => 1611,
      'i aslX ' => -1612, 'rice Xn' => 1613, 'az dXr ' => 1614, 'a ayXni' => -1615, ' bayXni' => -1616,
      ' Xsiniy' => 1617, 'n mXsin' => 1618, 'ek tXr ' => -1619, 'varXsi ' => -1620, 'irladX' => -1621,
      'tylamX' => -1622, 'c parX' => 1623, 'ap mX ' => 1624, ' azXm ' => -1625, 'n Xlia' => 1626,
      ' bay X' => -1627, 'gray X' => -1628, 'batXst' => -1629, 'ivaldX' => -1630, 'orallX' => -1631,
      ' alXca' => -1632, 'ytanX ' => -1633, 'Xsitma' => 1634, ' kXlip' => 1635, 'rner X' => 1636,
      'hael X' => 1637, 'taktX ' => 1638, 'ant Xn' => 1639, 'n asrX' => -1640, 'art mX' => 1641,
      'ruhanX' => -1642, 'elvarX' => -1643, 'i sXr ' => -1644, 'somalX' => -1645, 'anklXk' => 1646,
      'senarX' => -1647, 'fkatlX' => -1648, ' arSXn' => 1649, ' attXn' => 1650, 'stXnde' => -1651,
      ' takXs' => 1652, 'ikmalX' => -1653, 'ir anX' => 1654, 'etrafX' => 1655, ' Xlik ' => 1656,
      ' natXk' => 1657, 'inalXn' => -1658, 'af mX ' => 1659, 'yalXst' => -1660, 'dallX ' => -1661,
      'coachX' => 1662, ' hazXn' => -1663, 'rangX' => 1664, ' cXc ' => 1665, 'kXtas' => 1666,
      'trahX' => 1667, ' kXlc' => 1668, ' Xndo' => 1669, 'att X' => 1670, 'CXngi' => 1671,
      'val X' => -1672, 'aj mX' => 1673, 'way X' => -1674, 'anstX' => 1675, 'Xslat' => 1676,
      'Xsita' => 1677, 'takXr' => 1678, 'namX ' => -1679, 'ds Xn' => 1680, 'antlX' => 1681,
      'sXvis' => 1682, 'zdarX' => -1683, 'I ymX' => 1684, 'laldX' => -1685, 'uk X ' => 1686,
      'yahXn' => 1687, 'Xkac ' => -1688, ' SXki' => 1689, 'ihabX' => 1690, 'IbbXn' => 1691,
      'ndXl ' => -1692, ' Xone' => 1693, 'ktXra' => 1694, 'IlttX' => 1695, 'gXdik' => 1696,
      'IsXsi' => 1697, 'Xnas ' => -1698, 'ac Xn' => 1699, 'ass X' => 1700, ' matX' => -1701,
      'SkadX' => 1702, '96 Xn' => 1703, 'yurdX' => 1704, 'or X ' => 1705, 'Se dX' => -1706,
      'ibalX' => -1707, 'CXtay' => 1708, ' alXo' => -1709, ' aXs ' => 1710, 'dXnal' => -1711,
      'kalsX' => 1712, 'ntatX' => 1713, 'av Xn' => 1714, 'sXdik' => 1715, 'iyamX' => -1716,
      'laatX' => -1717, 'GdadX' => -1718, 'camX ' => -1719, ' savX' => 1720, 'haSXn' => -1721,
      'sXgan' => 1722, 'rmXna' => -1723, 'rabX ' => 1724, ' SXkt' => 1725, 'tankX' => 1726,
      'k lXn' => 1727, 'navlX' => 1728, 'Xnist' => -1729, 'azXna' => 1730, 'I sX ' => 1731,
      'amXzi' => 1732, 'stXya' => -1733, 'barXz' => -1734, ' vXdi' => 1735, 'barCX' => 1736,
      'sXnai' => 1737, 'Xliye' => -1738, 'Xsine' => -1739, '6 ncX' => 1740, 'argXn' => 1741,
      'lasX ' => 1742, 'Xlisk' => -1743, ' Xnsa' => -1744, 'urXu' => 1745, 'IfkX' => 1746,
      'ykX ' => -1747, 'naCX' => -1748, 'OzXs' => 1749, 'ajcX' => 1750, ' usX' => 1751,
      'amCX' => 1752, 'IngX' => 1753, 'amXc' => -1754, 'IfsX' => 1755, 'InzX' => 1756,
      'IftX' => 1757, 'mdXs' => 1758, 'I Xv' => 1759, 'rsmX' => 1760, 'atcX' => 1761,
      'ItCX' => 1762, 'hajX' => -1763, 'IssX' => 1764, 'IpsX' => 1765, ' SXp' => 1766,
      'tXc ' => -1767, 'Xmit' => -1768, 'rClX' => 1769, 'lXme' => -1770, 'Xkec' => -1771,
      'Xvas' => -1772, 'kafX' => -1773, 'aglX' => 1774, 'Xsaa' => 1775, 'taCX' => -1776,
      'rlXn' => -1777, ' lXn' => -1778, ' mXt' => -1779, 'IklX' => 1780, 'Xyel' => -1781,
      'fXye' => -1782, 'Xyat' => -1783, 'Xdar' => -1784, ' dXl' => -1785, 'Xne ' => -1786,
      'csX' => 1787, 'IXs' => -1788, 'IhX' => 1789, 'kXo' => -1790, 'bXh' => -1791,
      'lGX' => 1792, ' Xh' => -1793, 'dXt' => -1794, 'IdX' => 1795, 'erX' => -1797,
      'SImIz Xsin ' => 1798, 'sal islamX' => 1799, 've sXnir ' => -1800, 'I taktXg' => 1801, 'center X' => 1802,
      ' rantX ' => 1803, 'I sXni ' => 1804, 'a sXnin' => 1805, 'cilarX' => -1806, ' tXras' => 1807,
      'kXrani' => -1808, 'si fXs' => 1809, 'gazalX' => -1810, 'bangXr' => 1811, 'basXl ' => -1812,
      'tal mX' => 1813, 'hatlX ' => -1814, 'ak mXd' => 1815, 'ar tXp' => 1816, 'an dX ' => 1817,
      'camXn ' => 1818, 'sXmsiy' => -1819, 'I akXs' => -1820, ' kXli ' => 1821, 'galatX' => -1822,
      'ock Xn' => 1823, 'gle X ' => 1824, 'at tX ' => 1825, 'rahlX' => 1826, 'sXmil' => -1827,
      'sXrac' => -1828, 'rd X ' => 1829, ' SXma' => 1830, 'It Xn' => 1831, 'darXk' => -1832,
      'ardXs' => -1833, ' macX' => -1834, 'anCsX' => 1835, ' tXna' => 1836, 'etalX' => -1837,
      ' azXl' => -1838, 'anCtX' => 1839, 'UvarX' => -1840, 'basrX' => -1841, ' tXks' => -1842,
      'uassX' => 1843, 'sXvi ' => 1844, 'InClX' => 1845, 'damdX' => 1846, 'etabX' => 1847,
      'hazzX' => 1848, 'nallX' => -1849, 'Xckir' => 1850, 'varkX' => -1851, 'adXs ' => -1852,
      'kXray' => -1853, ' asX ' => -1854, 'matXg' => -1855, 'y Xnn' => 1856, 'ihamX' => 1857,
      'Xraye' => -1858, 'hal X' => -1859, 'Xzasy' => -1860, ' tXni' => 1861, 'ann X' => 1862,
      'on X ' => 1863, 'gal X' => -1864, 'Xsiti' => 1865, ' mXsa' => -1866, 'bbanX' => -1867,
      'arslX' => 1868, ' aydX' => 1869, 'Xrej' => 1870, 'Xpca' => 1871, ' Xas' => 1872,
      'aCsX' => 1873, ' Xvo' => 1874, 'IplX' => 1875, 'G X ' => 1876, 'CXgr' => 1877,
      'sXdd' => 1878, 'w Xn' => 1879, 'CXba' => 1880, 'iGlX' => -1881, 'akXa' => 1882,
      'lksX' => 1883, 'sXtk' => 1884, ' Xrm' => 1885, 'ayXh' => 1886, 'pXrl' => 1887,
      'ozkX' => 1888, 'Xsbu' => -1889, 'aGnX' => 1890, ' vXl' => -1891, 'USkX' => 1892,
      'tXme' => -1893, 'SXka' => -1894, 'askX' => 1895, 'jdX' => 1896, 'GsX' => 1897,
      'nXj' => -1898, 'Xfo' => -1899, 'Xsh' => -1900, 'Xgh' => -1901, 'sXs' => -1902,
      'itX' => -1903, 'a Xnternet ' => 1904, 'n Xnternet ' => 1905, 'lbert X' => 1906, 'an mXdi' => 1907,
      'ar dXr ' => 1908, 'halanX' => -1909, 'myk nX' => -1910, 'l Xnte' => 1911, 'I mXdi' => 1912,
      ' azlX ' => -1913, 'edralX' => -1914, 'Il Xn ' => 1915, ' miamX' => -1916, 'mUbaSX' => -1917,
      'rmallX' => -1918, ' sadX ' => -1919, 'Il mX ' => 1920, 'dXsini' => -1921, 'lvadX' => -1922,
      'lasXm' => -1923, 'canXk' => -1924, 'dartX' => 1925, 'Xrkil' => -1926, 'kXras' => -1927,
      'bus X' => 1928, 'h lX ' => 1929, 'ms Xn' => 1930, 'a lX ' => 1931, ' hXsi' => 1932,
      'zarfX' => 1933, 'arjX ' => 1934, ' kXbl' => 1935, 'CXnla' => 1936, 'ap X ' => 1937,
      ' Xnst' => 1938, 'erazX' => -1939, 'a ydX' => 1940, 'yassX' => 1941, 'sXvay' => 1942,
      'ldXnc' => -1943, 'nnadX' => -1944, 'artXc' => 1945, 'itamX' => -1946, 'abrXm' => 1947,
      'SanzX' => 1948, ' kakX' => 1949, 'Xsint' => 1950, 'navsX' => 1951, 'Xslak' => 1952,
      'yakXt' => 1953, 'azmXs' => 1954, 'zXrva' => 1955, 'Xskir' => 1956, 'slahX' => 1957,
      'tacX ' => 1958, 'Xrgir' => 1959, 'kXlla' => 1960, 'ak nX' => -1961, 'kXtay' => 1962,
      'akXyo' => 1963, 'lIC X' => 1964, 'Xgna' => 1965, ' mfX' => 1966, ' fXf' => 1967,
      'inCX' => -1968, 'Xpis' => 1969, ' fXc' => 1970, 'sfXn' => 1971, 'rapX' => -1972,
      'Xmir' => -1973, 'adsX' => 1974, 'pXrt' => 1975, 'yXc ' => -1976, 'tXng' => -1977,
      ' zXn' => -1978, 'Xzir' => 1979, 'bXs ' => -1980, 'Xbre' => -1981, 'Xraz' => -1982,
      'Xtle' => -1983, 'htXy' => -1984, ' SXl' => -1985, ' nXl' => -1986, 'ercX' => -1987,
      'Xlp' => 1989, 'dXb' => -1990, 'Xkb' => -1991, 'Xos' => -1992, 'Cin hayatX ' => -1993,
      'inC yanX' => 1994, 'Gal gazX' => 1995, 'a mXdir' => 1996, 'I mXsin' => 1997, 'a sXni ' => 1998,
      'ak mXy' => 1999, ' aSXna' => -2000, 'atia X' => 2001, 'ovladX' => -2002, 'uralsX' => 2003,
      'art X ' => 2004, 'yahatX' => -2005, 'rs Xn ' => 2006, ' kXri ' => -2007, 'IS mX ' => 2008,
      'lkantX' => 2009, 'ance X' => 2010, 'bXkin' => -2011, ' Xra ' => 2012, 'cXliz' => 2013,
      'jam X' => -2014, 'garcX' => -2015, ' CXri' => 2016, 'bantX' => 2017, 'kadXm' => -2018,
      ' fanX' => -2019, 'ubalX' => -2020, ' sXyi' => 2021, ' zatX' => 2022, 'nalXs' => -2023,
      'hmalX' => -2024, 'Xmild' => 2025, 'aat X' => -2026, 'StahX' => 2027, 'tl lX' => 2028,
      'daktX' => -2029, '6 lXk' => 2030, 'bXcag' => 2031, 'ntazX' => -2032, 'yamXn' => -2033,
      'enalX' => -2034, 'Xspan' => -2035, ' rXh' => 2036, ' Xai' => 2037, ' epX' => -2038,
      'fXrk' => 2039, 'IhtX' => 2040, 'azXh' => 2041, 'hXnz' => 2042, 'IvrX' => 2043,
      'jamX' => -2044, 'raCX' => -2045, ' pXl' => -2046, 'Xnma' => 2047, ' Xst' => -2048,
      ' mXl' => -2049, 'jsX' => 2050, 'IaX' => 2051, 'SXh' => 2052, 'zXf' => -2053,
      'Xmo' => -2054, 'isX' => -2055, 'eSX' => -2056, 'eX' => -2058, 'n taktXr' => 2059,
      'rli malX' => 2060, 'selamX ' => -2061, 'Xnterna' => 2062, 'sXgar ' => 2063, ' SadX ' => -2064,
      ' mXril' => 2065, 'ay mX ' => 2066, 'transX' => -2067, 'a dXr ' => 2068, 'art Xn' => 2069,
      'a tXp ' => 2070, 'n tXp ' => 2071, ' stalX' => -2072, 'kXsilm' => 2073, 'irarX ' => -2074,
      'le Xn ' => 2075, ' sXmsi' => 2076, 'farksX' => 2077, 'tles X' => 2078, 'tXlini' => -2079,
      'an nXn' => 2080, ' kXta ' => 2081, ' hattX' => 2082, 'asXni ' => 2083, 'Xndex' => 2084,
      'I Xn ' => 2085, ' Xlic' => 2086, 'balXs' => -2087, 'iSahX' => 2088, 'zXmli' => -2089,
      'izahX' => 2090, ' tXpa' => 2091, 'nalXt' => -2092, 'CXtir' => 2093, 'nk X ' => 2094,
      ' hXya' => 2095, 'ikadX' => 2096, 'ylXn ' => -2097, 'abXna' => 2098, 'a mXy' => 2099,
      'sXro' => -2100, 'Xph ' => 2101, 'IClX' => 2102, 'fdXs' => 2103, 'hXsm' => 2104,
      'Xstr' => -2105, 'napX' => -2106, 'aadX' => -2107, 'tXz' => -2108, 'aftX' => 2109,
      'IzsX' => 2110, 'ICkX' => 2111, 'Xnge' => -2112, 'gXrt' => 2113, ' amX' => -2114,
      'fXrc' => 2115, 'aygX' => 2116, 'kXye' => -2117, 'Xbk' => 2118, ' gazXn ' => 2119,
      ' 0 lXk' => 2120, 'saatCX' => -2121, ' Xzdir' => 2122, 'UratlX' => -2123, 'tikalX' => -2124,
      'kXsin ' => 2125, ' 40 lX' => 2126, ' atXk ' => 2127, 'merakX' => 2128, 'eid X' => 2129,
      'faksX' => 2130, 'IrtmX' => 2131, 'Ir X ' => 2132, 'Xsinm' => 2133, ' cXvi' => 2134,
      'kXssa' => 2135, 'ab X ' => 2136, 'nakXl' => -2137, 'zantX' => 2138, 'narSX' => -2139,
      'as X ' => 2140, ' zalX' => -2141, 'Xrpan' => 2142, 'I mXy' => 2143, 'pXnar' => 2144,
      'afXmi' => 2145, 'Xzda ' => 2146, ' tXr ' => 2147, 'aratX' => 2148, ' gXc' => 2149,
      'nsXf' => -2150, 'IddX' => 2151, 'atXb' => -2152, 'optX' => -2153, 'fXdi' => 2154,
      '6 sX' => 2155, 'etCX' => -2156, 'batX' => 2157, 'sCX' => 2158, 'barXs' => 2159,
      'Xsn' => -2160, 'kanI salX' => -2161, ' bayX ' => -2162, 'mutabX' => 2163, 'Im mX ' => 2164,
      'ar mXy' => 2165, 'zikalX' => -2166, ' banlX' => -2167, 'ekanXk' => -2168, ' Xsmar' => 2169,
      'opartX' => 2170, 'e tXp ' => 2171, 'varXl ' => -2172, ' anXma' => -2173, 'Xring' => 2174,
      'hapsX' => -2175, 'ovalX' => 2176, 'okalX' => -2177, ' apX ' => -2178, 'faatX' => -2179,
      'I ydX' => 2180, 'Xndep' => 2181, 'dXnav' => -2182, 'zabX ' => 2183, 'alXci' => 2184,
      'ianX ' => -2185, 'safX ' => -2186, 'kXya ' => 2187, ' satX' => 2188, 'rkXr' => 2189,
      'kdXv' => 2190, 'nkXr' => 2191, 'farX' => -2192, 'Xkih' => 2193, 'IrzX' => 2194,
      'dXda' => -2195, 'aSsX' => 2196, 'azcX' => 2197, 'lkCX' => 2198, 'sXg ' => 2199,
      'CXra' => 2200, 'babX' => 2201, 'sXtm' => 2202, 'eGdX' => -2203, 'Xsto' => -2204,
      ' nXk' => -2205, 'kImX' => 2206, ' nXc' => -2207, 'Xko ' => -2208, 'Xse ' => -2209,
      'Xrme' => -2210, 'en yanX ' => -2211, 'ak mXsin' => 2212, 'baybaSX' => -2213, ' Xslik' => 2214,
      'ab Xn ' => 2215, '  adlX' => -2216, 'aC Xn ' => 2217, 'Xsinla' => 2218, 'hatabX' => 2219,
      ' Xskal' => 2220, 'ang X' => 2221, 'maldX' => -2222, 'ebalX' => -2223, ' gXa ' => 2224,
      ' astX' => 2225, 'accX' => 2226, 'aScX' => 2227, 'capX' => -2228, ' Xky' => 2229,
      'hXra' => -2230, 'IflX' => 2231, 'cXt ' => -2232, 'nXc ' => -2233, 'lXla' => 2234,
      'istX' => -2235, 'dakX' => -2236, ' rantXn' => 2237, ' yasXn ' => -2238, 'eral Xn' => -2239,
      'at mX ' => 2240, ' sarXo' => 2241, 'ahatX ' => -2242, ' enayX' => -2243, '60 lX ' => 2244,
      'niyazX' => -2245, 'kXtal' => 2246, ' narX' => -2247, 'uvazX' => -2248, ' CXt ' => 2249,
      'kXtli' =>  2250, 'adXya' => 2251, 'astlX' => 2252, 'gharX' => -2253, 'hallX' => -2254, 'rdarX' => -2255,
      'hacXm' => -2256, ' CXld' => 2257, ' vadX' => -2258, 'asmX' => 2259, ' hXv' => 2260,
      'IalX' => -2261, ' kXc' => 2262, 'apCX' => 2263, 'yrXl' => 2264, 'ICX' => 2265,
      'Xly' => -2266, 'an yanX ' => -2267, 'an dXr ' => 2268, 'sancanX' => -2269, 'coGrafX' => -2270,
      'an lX ' => 2271, 'tilalX' => -2272, 'finalX' => -2273, 'I sXva' => 2274, ' pratX' => -2275,
      'Ik X ' => 2276, 'efahX' => 2277, 'ganXk' => -2278, ' anX ' => -2279, 'Xslan' => 2280,
      ' Xsir' => 2281, 'Xslah' => 2282, '  tXp' => 2283, 'haSXm' => -2284, 'ndXc' => -2285,
      'IrkX' => 2286, 'IzmX' => 2287, ' Xlo' => 2288, 'IkkX' => 2289, 'tXvi' => -2290,
      'IrhX' => 2291, ' nXs' => -2292, 'asrX' => 2293, 'Xlai' => -2294, 'eptX' => -2295,
      'CXj' => 2296, 'Xfz' => 2297, 'atlantX' => -2298, 'neralX' => -2299, 'dli tX' => 2300,
      'ans Xn' => 2301, 'misalX' => -2302, 'ar mXs' => 2303, ' kXliy' => 2304, 'orantX' => 2305,
      'sakXn ' => -2306, 'kk lX ' => 2307, 'In X ' => 2308, 'fXsil' => 2309, '90 lX' => 2310,
      'UnahX' => 2311, 'darkX' => -2312, 'abXni' => 2313, 'laysX' => 2314, 'Xnlar' => 2315,
      'Xsina' => 2316, 'abXz' => 2317, 'ItmX' => 2318, 'arXt' => -2319, '6 yX' => 2320, 'katXli ' => -2321,
      'inyalX' => -2322, 'and Xn' => 2323, ' sXni ' => -2324, 'rd Xn' => 2325, 'hXzip' => -2326,
      ' ganX' => -2327, 'nazXk' => -2328, 'bakX ' => -2329, 'acayX' => -2330, ' rXfa' => 2331,
      'kXsla' => 2332, 'Xlif' => 2333, 'anSX' => 2334, 'InsX' => 2335, 'Xvao' => 2336,
      ' Xan' => 2337, 'iStX' => -2338, 'cXu' => -2339, 'Xbm' => 2340, 'gXp' => 2341,
      'bzX' => 2342, 'sXb' => -2343, 'S ahlakX' => 2345, ' Xsiya' => 2346, ' azamX' => -2347,
      ' akXt ' => -2348, 'avalX ' => 2349, 'Ik mX ' => 2350, ' stasX' => -2351, ' kXlig' => 2352,
      'sabrXn' => 2353, 'sXnam' => 2354, 'Xraci' => -2355, 'katXp' => -2356, 'arstX' => 2357,
      'artlX' => 2358, 'rarSX' => -2359, 'arXl ' => 2360, 'hXnc' => 2361, ' Xrz' => 2362,
      'Xoc' => 2363, 'Xo ' => -2364, 'nXh' => -2365, 'am mX ' => 2366, 'lgazX' => 2367,
      ' fXst' => 2368, 'naatX' => -2369, 'rartX' => 2370, ' kXst' => 2371, 'balXn' => -2372,
      'astXr' => 2373, ' raSX' => -2374, 'IrCX' => 2375, 'IfzX' => 2376, 'ikmX' => -2377,
      'nsXk' => -2378, 'Xkdp' => 2379, 'omXr' => 2380, 'ealX' => -2381, 'sXza' => 2382,
      'hXl' => -2383, 'yonal X' => -2384, ' SinasX' => -2385, 'SahsXn' => 2386, 'mIS mX' => 2387,
      'aksXn ' => 2388, 'kkatsX' => -2389, ' trajX' => -2390, 'ramXk' => -2391, 'k yX ' => 2392,
      'elafX' => -2393, 'karnX' => 2394, '40 X' => 2395, 'sXyr' => 2396, '60 X' => 2397,
      'Xgdi' => 2398, 'ahtX' => 2399, ' kXz' => 2400, 'bir yanX' => 2402, 'ografX' => -2403,
      ' malXk' => -2404, 'Im X ' => 2405, 'balcX' => 2406, ' kXl ' => 2407, 'yXgil' => 2408, 'italX' => -2409,
      ' SXk ' => 2410, ' tacX' => -2411, 'ifakX' => 2412, '90 X' => 2413, 'IbX' => 2414,
      ' barX ' => -2415, 'In mX ' => 2416, ' sXnin' => -2417, 'IS Xn ' => 2418, 'iramX' => -2419,
      'hlalX' => -2420, 'icabX' => 2421, 'kXra ' => -2422, 'nasXp' => -2423, 'SgalX' => -2424,
      ' zXt' => 2425, 'tXbb' => 2426, 'dafX' => -2427, 'yayX' => 2428, 'fbX' => 2429,
      'eCX' => -2430, ' anadX' => -2431, 'artsX' => 2432, 'alttX' => 2433, 'altmX' => 2434,
      'alXc ' => -2435, 'yankX' => 2436, 'uhtX' => 2437, 'dXsc' => -2438, 'CXg ' => 2439,
      'Xyik' => 2440, 'yXrt' => 2441, 'eGlX' => -2442, 'yXi' => -2443, ' var mX' => 2444,
      'ahsXz' => 2445, 'apkXn' => 2446, 'Xlgaz' => 2447, 'a sX ' => 2448, ' sXzm' => 2449,
      'azgX' => 2450, 'hasX' => 2451, 'francX' => -2452, ' nazX ' => -2453, 'selanX' => -2454,
      'zgarX' => 2455, 'atmXy' => 2456, 'IstX' => 2457, 'sXrk' => -2458, 'Xgma' => 2459,
      'dolabX' => 2460, 'moralX' => -2461, ' asXt' => -2462, 'Xspar' => 2463, ' nadX' => -2464,
      ' akXf' => -2465, 'gXya' => 2466, ' sXzdi' => 2467, 'I dXr ' => 2468, 'IrslX' => 2469,
      'yastX' => 2470, 'kastX' => 2471, 'am X ' => 2472, ' lakX' => -2473, 'IltX' => 2475,
      'CXgi' => 2476, 'emal X' => -2477, 'tXla ' => -2478, ' atXf' => 2479, ' kXde' => 2480,
      'aycX' => 2481, 'bektaSX' => -2482, 'ht Xn' => 2483, 'ahCX' => 2484, 'IzgX' => 2485,
      'Xkol' => -2486, 'Xmle' => -2487, ' kXlik' => 2488, ' asayX' => -2489, ' fXnd' => 2490,
      'IrhlX' => 2491, 'Xye ' => -2492, ' allahX' => 2493, 'maatX' => -2494, 'IymX' => 2495,
      ' bXk' => 2496, 'Xlle' => -2497, 'rXo' => -2498, ' arXk' => 2499, 'vaktX' => -2500,
      'amsXz' => 2501, ' sXca' => 2502, 'Xhh' => 2503, 'Xge' => -2504, ' arSX' => -2505,
      'haydX' => -2506, 'marXs' => -2507, 'aymX' => 2508, 'alXf' => -2509, 'aslX' => 2510,
      'IvX' => 2511, ' naklX' => -2512, ' kXniy' => 2513, ' nacX' => -2514, ' latX' => -2515,
      'amdXr' => 2516, 'anClX' => 2517, 'artmX' => 2518, 'Xsah' => -2519, 'ISsX' => 2520,
      'IrgX' => 2521, 'dXya' => -2522, 'iyX' => -2523, ' yXgin' => 2524, 'saatlX' => -2525,
      'aSantX' => 2526, 'alsXn' => 2527, ' affX' => 2528, 'Ir mX' => 2529, 'IslX' => 2530,
      'aSCX' => 2531, 'G Xn' => 2532, 'Xzla' => 2533, ' taklX' => -2534, 'anslX' => 2535,
      'IrttX' => 2536, 'abahX' => 2537, 'omasX' => -2538, 'Xgli' => 2539, 'enazX' => -2540,
      'rdXn ' => -2541, 'mpatX' => -2542, 'kXsir' => 2543, 'isadX' => -2544, 'IrcX' => 2545,
      'iptX' => -2546, ' akX' => 2547, 'Ir Xn ' => 2548, 'ah X ' => 2549, ' zabX' => 2550,
      'al X ' => 2551, 'IrrX' => 2552, 'barzanX' => -2553, ' salXm' => -2554, ' kXld' => 2555,
      ' Xsil' => 2556, 'Xrmak' => 2557, 'Xltu' => -2558, 'Xfe' => -2559, 'Xde' => -2560,
      'bXt' => -2561, 'katXl ' => -2562, ' fXri' => 2563, 'GazXc' => -2564, 'ajlX' => 2565,
      ' pXril' => 2566, 'hafXz' => 2567, 'Xnle' => -2568, 'Xce' => -2569, ' kXlin' => 2570,
      ' sakX' => 2571, 'algX' => 2572, 'fXne' => -2573, 'cazX' => -2574, 'Xyla' => 2575,
      'Xklar' => 2576, 'nssX' => 2577, 'CXlg' => 2578, 'afX ' => 2579, ' Xi' => 2580,
      'Iz mX ' => 2581, ' hXz ' => 2582, 'zXrh' => 2583, 'IrpX' => 2584, 'sXcr' => 2585,
      'cXz' => -2586, 'tXt' => -2587, 'ak X ' => 2588, 'pasXf' => -2589, '6 nXn' => 2590,
      'afCX' => 2591, ' dXn' => -2592, 'ISCX' => 2593, 'az mX' => 2594, 'aXds' => 2595,
      'as Xn ' => 2596, 'kkarX' => -2597, ' marX' => -2598, 'kXv' => 2599, 'timalX' => -2600,
      'IlcX' => 2601, 'CXrp' => 2602, 'ItsX' => 2603, 'Ik Xn ' => 2604, 'az X ' => 2605, 'Xpir' => 2606,
      'ImdX' => 2607, 'In Xn ' => 2608, 'azXz' => -2609, 'lktX' => 2610, 'labanX' => -2611,
      'an mX ' => 2612, 'kXral' => -2613, 'IpX' => 2614, 'iptalX' => -2615, 'on Xn ' => 2616,
      'saatX' => -2617, 'altXn' => 2618, 'apsX' => 2619, 'IkCX' => 2620, ' Xlimli' => 2621,
      'hatXp' => -2622, 'at X ' => 2623, 'fXrat' => 2624, 'aGcX' => 2626, 'dIz X' => 2627,
      'ay X ' => 2628, 'elCX' => -2629, 'ihtX' => -2630, 'Xon' => -2631, ' hXzi' => 2632,
      'Xrna' => 2633, 'anrX' => 2634, 'GzX' => 2635, 'Xdv' => 2636, ' idarX' => -2637,
      'ap lX ' => 2638, 'bartX' => 2639, 'rkCX' => 2640, 'Xpla' => 2641, 'hIrXs' => -2642,
      ' kXlm' => 2643, ' tayX' => -2644, 'gahX' => 2645, 'afsX' => 2646, ' hanX ' => -2647,
      'kadXr' => -2648, 'azsX' => 2649, 'kkabX' => 2650, 'bXl' => -2651, ' kXs ' => 2652,
      'kaybX' => 2653, 'arcX' => 2654, 'IlgX' => 2655, 'fXl' => -2656, 'bank Xn' => 2657,
      ' bastX' => 2658, ' makX' => -2659, ' fXrl' => 2660, ' cXa ' => 2661, 'aS X ' => 2662,
      'cXv' => -2663, 'inamX' => -2664, 'IX' => 2665, 'fXkra' => 2666, ' kXna' => 2667,
      'aflX' => 2668, 'Xnirle' => -2669, 'ekmX' => -2670, 'arazX' => -2671, 'GdX' => 2672,
      'Xno' => -2673, 'afXn' => 2675, 'kXsim' => 2676, ' sXgi' => 2677, 'aCmX' => 2678,
      'CakX' => 2679, 'SartX' => 2680, 'ISkX' => 2681, ' yXp' => 2682, 'ak mX ' => 2683,
      ' panX' => -2684, ' acXl ' => -2685, 'kXlic' => 2686, 'kkatlX' => -2687, ' Xlie' => 2688,
      'Xnde' => -2689, 'pahalX' => 2690, 'ap Xn ' => 2691, 'ammX' => 2692, 'Im Xn ' => 2693,
      ' harX' => -2694, 'Xng ' => -2695, 'ecatX' => -2696, 'SahXs' => 2697, 'fXrt' => 2698,
      'IncX' => 2699, 'IttX' => 2700, 'alCX' => 2701, 'yazmX' => 2702, ' gXda' => 2703,
      'IksX' => 2704, 'vabX' => 2705, 'sXyon' => -2706, ' Xrk' => 2707, 'rXza' => 2708,
      'ar X ' => 2709, 'aplX' => 2710, ' klasX' => -2711, 'alXz' => -2712, 'IlsX' => 2713,
      'arzX' => 2714, 'kXsit' => 2715, ' Xsig' => 2716, ' hadX' => -2717, 'kapXt' => -2718,
      'Xmar' => -2719, 'hayalX' => -2720, 'sarX' => 2721, 'rIs X' => 2722, 'zXran' => -2723,
      'gazX' => -2724, 'silahX' => 2725, 'basXt' => -2726, 'kXsk' => 2727, 'I yX ' => 2729,
      'mXsir' => 2730, 'mXl' => -2731, 'kampX' => 2732, 'ikkatX' => -2733, ' adXl' => -2734,
      'ikahX' => 2735, 'am Xn ' => 2736, 'tXll' => -2737, 'InavX' => 2738, 'dXsk' => -2739,
      'Xpk' => 2740, ' malX ' => -2741, 'ad X' => 2742, 'IpkX' => 2743, ' sXf' => 2744,
      'mXr' => -2745, 'ItlX' => 2746, 'attXn ' => -2747, 'hamX' => -2748, 'rasXm' => -2749,
      'caksX' => 2750, 'allX' => 2751, 'n yanX' => 2752, ' samX' => -2753, 'haylX' => -2754,
      ' tXk' => 2755, 'ISmX' => 2756, ' taktX' => -2757, 'langX' => 2758, 'at Xn ' => 2759,
      ' islamX ' => -2760, 'dXka' => -2761, 'ImlX' => 2762, 'garX' => -2763, 'carX' => -2764,
      ' mantX' => 2765, 'Xre' => -2766, 'ganXz' => -2767, 'IrakX' => 2768, 'amcX' => 2769,
      'IzdX' => 2770, 'IsmX' => 2771, 'vakXf' => 2772, 'a mX ' => 2773, ' manX' => -2774,
      'zayXr' => -2775, 'k nXn ' => 2776, ' yakX' => 2777, 'IfX' => 2778, 'Xe' => -2779,
      'ISlX' => 2780, 'parXs' => -2781, 'sabX' => 2782, 'aSmX' => 2783, 'Xlat' => -2784,
      'aClX' => 2785, 'avrX' => 2786, 'dXsip' => -2787, 'InmX' => 2788, 'kartX' => 2789,
      ' atXna' => -2790, 'CarpX' => 2791, ' kXrl' => -2792, 'tXrma' => 2793, 'rXf' => -2794,
      ' yangX' => 2795, 'Xmf' => 2796, 'al Xn ' => 2797, 'tdX' => 2798, 'I mX ' => 2799,
      'az Xn ' => 2800, 'akCX' => 2801, 'Xsrar' => 2802, 'tavXr' => 2803, 'ahlX ' => 2804,
      'IrsX' => 2805, 'ak Xn ' => 2806, 'atkX' => 2807, 'valX' => -2809, 'ah Xn ' => 2810,
      'Xcak' => 2811, 'ar mX ' => 2812, 'itabX' => 2813, 'akfX' => 2815, 'hapXs' => -2816,
      'aS Xn ' => 2817, 'ykX' => 2818, 'armX' => 2819, 'ay Xn ' => 2820, 'atmXs' => 2821,
      ' talX' => -2822, ' kXsm' => 2823, 'fadXm' => -2824, 'arafX' => 2825, 'kmX' => 2826,
      'Xrsa' => 2827, 'Xsyo' => -2828, 'tatX' => -2829, ' galX' => -2830, ' yXk' => 2831,
      'Xzm' => -2832, 'aSkX' => 2833, ' hXr' => 2834, 'arsX' => 2835, 'hakX' => -2836,
      'arttX' => 2837, 'aylX' => 2838, 'Xa' => -2839, 'pmX' => 2840, 'IrmX' => 2841,
      'IydX' => 2842, 'amlX' => 2843, 'aksXz' => 2844, 'Xte' => -2845, 'an X ' => 2846,
      'sanayX' => -2847, ' Xsik' => 2848, 'ImsX' => 2849, 'aCtX' => 2850, 'azlX' => 2851,
      'azdX' => 2852, 'tXf' => -2853, 'ajX' => 2854, ' kXy' => 2855, 'ar Xn ' => 2856,
      'a yX ' => 2857, 'IntX' => 2858, 'akXl' => 2860, 'aGrX' => 2861, ' Xl' => -2862,
      'Xnav' => 2863, 'atCX' => 2864, 'Xne' => -2865, 'almX' => 2866, ' hakkX' => 2867,
      'Xsle' => -2868, 'hXzl' => 2869, 'atsX' => 2870, 'InlX' => 2871, 'alnX' => 2872,
      ' yanX ' => -2873, 'arkX' => 2874, 'IllX' => 2876, ' kXr' => 2877, 'IndX' => 2878,
      'dlX ' => 2879, 'arklX' => 2880, 'Xstan' => -2881, 'rXka' => -2882, 'bakX' => 2883,
      'kXsa' => 2884, 'avcX' => 2885, 'atlX' => 2886, ' camX' => -2887, 'ImcX' => 2888,
      ' alX ' => -2889, ' sXni' => 2890, 'aktX' => 2891, 'yargX' => 2892, 'ItX' => 2893,
      'IyX' => 2894, 'aSlX' => 2895, 'IlmX' => 2896, 'anmX' => 2898, 'IktX' => 2899,
      'IrdX' => 2900, 'IrlX' => 2901, 'Xye' => -2902, 'atXk' => -2904, 'IbrX' => 2905,
      'IzlX' => 2906, 'kXbris' => 2907, 'aklX' => 2908, 'Xlah' => -2910, 'tartX' => 2911,
      'akXm' => 2912, 'ansX' => 2913, 'IcX' => 2914, 'siyasX' => -2915, 'lantX' => 2916,
      'alkX' => 2917, 'akXn' => 2918, 'GlX' => 2919, 'attX' => 2920, 'halX' => -2921,
      ' sXk' => 2922, 'Xrak' => 2924, 'altX' => 2926, 'ancX' => 2927, 'I nX' => 2928,
      'aydX' => 2929, 'Xh' => -2932, ' artX' => 2933, 'arlX' => 2934, 'an Xn ' => 2935,
      ' ayrX' => 2936, 'a nXn' => 2937, ' sXr' => 2938, 'aStX' => 2939, 'IzX' => 2940,
      'acX' => 2941, 'IStX' => 2942, ' dXs' => 2943, 'aCX' => 2944, 'ptX' => 2945,
      'amX' => 2946, 'IldX' => 2947, 'ISX' => 2948, 'ardX' => 2949, 'apX' => 2950,
      'aldX' => 2951, 'andX' => 2952, 'arSX' => 2953, 'azX' => 2954, 'IrX' => 2955,
      'anlX' => 2956, 'IsX' => 2957, 'ImX' => 2958, 'atX' => 2959, 'aSX' => 2960,
      'IlX' => 2961, 'alX' => 2962, 'yXl' => 2963, 'Xla' => 2964, 'CXk' => 2966,
      'ayX' => 2967, 'adX' => 2968, 'Xnda' => 2969, 'IGX' => 2970, 'anX' => 2971,
      'asX' => 2972, 'InX' => 2973, 'arX' => 2974, 'X' => -2975
    },
    'o' => {
      'uz kXr' => -1, 'ni kXt' => 2, 'dir gX' => 3,
      'Il dXn' => 4, 'an Xng' => 5, 'lum kX' => -6,
      'niS kX' => -7, 'Giz gX' => 8, ' te gX' => 9,
      'rka kX' => -10, 'jIn Xn' => 11, 'ki rXp' => 12,
      'Xnerek' => 13, 'ezik X' => -14, 'dIn CX' => -15,
      'ni Xng' => 16, 'klI gX' => 17, 'ran CX' => -18,
      'ok Xn ' => 19, 'I Xrt ' => 20, 'm gXcu' => -21,
      ' efe X' => 22, 'OSe dX' => 23, 'Ge dXn' => 24,
      'ik dXk' => -25, 'bu sXn' => -26, 'ma sXk' => -27,
      'e Xlup' => -28, ' Xlusm' => -29, 'Xyune ' => 30,
      ' la gX' => 31, 'an rXm' => -32, 'ka yXn' => 33,
      'tacI X' => -34, 'nuGu X' => -35, 'Xnusma' => -36,
      'nde rX' => -37, 'bin Xg' => 38, 'zle gX' => 39,
      'dXnsa' => -40, 'Xktel' => -41, 'Xnalm' => 42,
      'kXley' => 43, 'lXwen' => 44, 'gakgX' => -45,
      'sXnus' => 46, 'ektX ' => 47, 'uyanX' => -48,
      '0 Xz ' => -49, 'rI Xo' => 50, ' Xnut' => 51,
      ' us X' => 52, 'jon X' => -53, ' lXm ' => 54,
      'Xykoy' => 55, 'l bXb' => -56, 'kXsog' => 57,
      ' tXko' => 58, 'bOksX' => 59, 'a sX ' => 60,
      'e Xt ' => 61, '76 yX' => 62, 'e Xmn' => 63,
      'CXpul' => -64, ' kXrv' => 65, 'd gX ' => -66,
      ' dXb ' => 67, 'Xtana' => 68, 'I Xes' => 69,
      'lerXn' => 70, 'he Xp' => -71, 'Xrtuc' => -72,
      '  Xdi' => 73, 'i gyX' => -74, 'i lXw' => 75,
      'gXdek' => 76, 'hektX' => -77, 'Xlumy' => 78,
      'th jX' => 79, 'pt yX' => 80, 'Xkse ' => 81,
      'i i X' => 82, ' setX' => 83, 'arelX' => 84,
      'GI Xb' => -85, 'Xpece' => 86, 'r m X' => -87,
      'Xglel' => 88, 'Xrfun' => 89, 'b Xde' => -90,
      'e I X' => -91, 'Xgdur' => 92, 'rXdov' => 93,
      'Xseri' => -94, 'Xunen' => 95, 'Xksen' => -96,
      'ortrX' => 97, ' Xrce' => 98, 'Sii X' => 99,
      'mXren' => 100, ' kXtl' => 101, 'fritX' => 102,
      'yl X ' => 103, 'dirX ' => 104, 'lemX ' => 105,
      ' rXba' => 106, 'b dXl' => -107, 'k sX ' => 108,
      'k bX ' => 109, 'guadX' => 110, 'Xrdev' => 111,
      'en aX' => 112, 'rXney' => -113, 'dr Xv' => 114,
      'n gaX' => 115, 'orn X' => 116, 'Xrdio' => -117,
      'ou gX' => -118, 'Ol gX' => 119, 'i bX ' => 120,
      'ny bX' => -121, '11 kX' => -122, 'n dXt' => 123,
      'ga gX' => 124, 'nCo X' => 125, 'CXkes' => -126,
      'Xsler' => 127, 'Xyuym' => 128, 'Xnlus' => -129,
      'Xders' => 130, 'Xryo ' => 131, 'Xnkin' => -132,
      'os gX' => 133, 'ilo X' => -134, '5 yXk' => 135,
      'zu kX' => -136, ' sXv ' => 137, 'm Xr ' => 138,
      'ukkXy' => 139, 's Xlv' => 140, ' me X' => -141,
      'bXles' => -142, 'CUn X' => 143, ' hXrg' => 144,
      '  UrX' => 145, '4 kXy' => 146, ' Xve ' => 147,
      ' d gX' => 148, 'Xnkay' => 149, 'Xnbaz' => -150,
      'mis X' => -151, 'kXreb' => 152, 'dXkse' => 153,
      ' tetX' => 154, ' b dX' => -155, '7 yXk' => 156,
      ' 1 Xz' => 157, 'morkX' => 158, 'romXr' => 159,
      'Xleme' => 160, 'm y X' => 161, 'Xtsun' => 162,
      'shi X' => -163, '2 kXy' => 164, 'C rXm' => -165,
      'li Xe' => 166, 'lu Xb' => -167, 'go gX' => 168,
      'bi lX' => -169, ' lXp ' => 170, 'tXral' => -171,
      'gerlX' => 172, 'inagX' => -173, 'i n X' => -174,
      'cal X' => -175, 's Xzi' => -176, 'nC Xd' => -177,
      'ry kX' => -178, 'fe gX' => 179, 'dI mX' => -180,
      ' dXle' => 181, 'kUp X' => -182, 'ka Xg' => 183,
      'Or SX' => 184, 'ad yX' => 185, 'Xvern' => -186,
      ' bXnl' => 187, 'dr gX' => 188, 'Xluye' => 189,
      'ketXr' => 190, 'r gX ' => -191, 'st gX' => 192,
      'tu dX' => -193, 'foklX' => 194, 'Xvell' => -195,
      'oza X' => 196, 'Xzale' => -197, 'tXrki' => 198,
      'rk gX' => 199, 'o dX ' => 200, 'Xrsen' => 201,
      ' CXsu' => 202, 'e yXe' => 203, 'c bXl' => 204,
      ' dXl ' => 205, 'tXron' => -206, 'Xrtla' => -207,
      'ku kX' => -208, '8 kXy' => 209, 'ndXrt' => 210,
      'ov Xn' => 211, 'S Xna' => -212, 'rXlov' => 213,
      'dyatX' => 214, 'Xmuyo' => 215, 'gI kX' => -216,
      'Xzar ' => -217, 'Xlcen' => 218, 'u Xst' => 219,
      'Xyser' => -220, 'Xlulu' => -221, 'na lX' => -222,
      'Xvula' => -223, 'd Xlm' => -224, 'Um gX' => 225,
      ' Xrun' => 226, 'z rXm' => -227, 'ngXl ' => 228,
      'CerdX' => 229, ' Xnyu' => 230, 'r Xme' => 231,
      '  tXb' => -232, 'bXluk' => 233, 'Xsym ' => 234,
      'Xzbek' => 235, 'm Xtu' => -236, 'Xtusu' => 237,
      'n lXk' => -238, 'gXrsu' => 239, '13 kX' => -240,
      'Xnely' => -241, 'Xlmun' => 242, 'gXrta' => -243,
      'ha Xb' => -244, 'gXnul' => 245, 'Xrebi' => 246,
      'Xyuna' => -247, 'Xzenl' => 248, 'n Xve' => -249,
      'dXkuz' => -250, 'Xrcul' => -251, 'rXc' => -252,
      ' terX' => 253, 'Xzgur' => 254, 'Xzerk' => 255,
      '  Xrm' => -256, 'a Xne' => 257, ' Xkta' => -258,
      'Xkenl' => 259, 'Xnlar' => -260, 'Xsunu' => -261,
      'dXviz' => 262, 'gXbl' => -263, 'n iX' => 264,
      'bXcu' => 265, 'Xooy' => 266, 'Xnm ' => 267,
      'Xpem' => 268, 'CdXk' => 269, 'uerX' => -270,
      ' Xsu' => 271, ' Xsr' => 272, 'Xgge' => 273,
      'Xyre' => 274, 'elXz' => 275, 'Xzia' => -276,
      'Xrrk' => 277, 'objX' => 278, 'ho X' => 279,
      'kyXd' => 280, ' Xen' => 281, 'lIdX' => 282,
      'rkXp' => 283, 'eanX' => -284, 'Xk1 ' => 285,
      'pXku' => 286, 'gXlm' => 287, 'Xnex' => -288,
      'gXmo' => -289, 'dXlk' => 290, 'tXve' => 291,
      'Xccu' => 292, ' jXl' => 293, 'InXz' => 294,
      'bXhl' => 295, 'Xgrt' => 296, 'kXkk' => -297,
      'Xnnb' => 298, 'ykXy' => 299, 'nbXn' => 300,
      'bXnb' => 301, 'ntXz' => 302, 'Xzie' => -303,
      'taXv' => 304, ' G X' => 305, 'Xlea' => -306,
      'segX' => -307, 'uc X' => -308, 'rIXn' => 309,
      'nh X' => 310, 'Xrso' => -311, 'esiX' => 312,
      ' O X' => 313, 'Xkuv' => 314, 'lldX' => -315,
      'Xnss' => -316, 'sXro' => -317, 'dXny' => -318,
      'kyXn' => 319, 'SXru' => 320, ' Xkd' => 321,
      'Xzni' => -322, 'fosX' => 323, 'yorX' => 324,
      'irXr' => 325, ' IgX' => -326, 'Xrki' => -327,
      'Xnra' => -328, ' Xnr' => 329, 'Xyok' => 330,
      'zXme' => 331, ' pnX' => 332, 'Xnea' => -333,
      'rOlX' => 334, 'Xtov' => -335, 'Xvma' => -336,
      'kXye' => 337, 'Xbeg' => 338, 'Xyti' => -339,
      'gXvi' => -340, 'barX' => -341, 'Xpti' => -342,
      'Xneb' => 343, 'Xryu' => -344, '9 Xc' => -345,
      'Xket' => -346, 'Xrsi' => -347, 'Xril' => -348,
      'vagX' => -349, 'tXbu' => -350, 'Xrph' => -351,
      'hXni' => -352, 'Xnan' => -353, 'Xzit' => -354,
      'stXk' => -355, 'gUnX' => 356, '1 Xc' => -357,
      'viyX' => -358, ' kXg' => -359, 'Xro ' => -360,
      'gXrm' => 361, 'Xnda' => -362, 'Xrie' => -363,
      'Xhn ' => -364, 'Xndr' => -365, 'Xrsk' => -366,
      'Xndo' => -367, 'lkXg' => 368, 'bXlg' => 369,
      'siyX' => -370, 'Xmis' => -371, 'Xrku' => -372,
      'sXym' => -373, ' gXz' => 374, 'Xnom' => -375,
      'Xpla' => -376, 'Xmh' => -377, 'Xdv' => -378,
      'Xmg' => 379, 'Xem' => 380, 'lXh' => 381,
      'GdX' => 382, 'Xvt' => -383, 'jyX' => -384,
      'Xtc' => -385, 'vXa' => 386, 'Xez' => 387,
      'SXp' => 388, 'Xdm' => -389, 'OyX' => 390,
      'Xdd' => -391, 'tuX' => -392, 'Xa ' => -393,
      'Xgo' => -394, 'Xtl' => -395, 'cXb' => -396,
      'maX' => -397, 'nXy' => -398, 'nyX' => -399,
      'Xpa' => -400, 'Xma' => -401, 'plX' => -402,
      'Xth' => -403, 'vXn' => -404, 'llX' => -405,
      'Xga' => -406, 'ncX' => -407, 'Xmb' => -408,
      've taksi SXforleri ' => 409, 'nesinin Xldugun' => 410,
      'n babasI Xldu ' => 411, 'elerinin yXk ' => 412,
      'aSlarInIn Xld' => 413, 'tan sonra Xlu' => 414,
      'er zaman Xn ' => 415, 'mlarInIn yXk' => 416,
      'i sonucu Xld' => 417, 'i olarak Xld' => -418,
      'Once de yXk ' => 419, 'Xlumsuzluge' => 420,
      'ilileri yXk' => 421, ' ya da Xlur' => 422,
      '  kaptan Xl' => 423, 'SIsInda dXn' => -424,
      'vinde Xldu ' => 425, 'y iCinde yX' => 426,
      'atUrk Xldu ' => 427, 'tOrleri yX' => 428,
      'lInda Xlmu' => 429, 'urduGu yXk' => 430,
      'lamIStI Xl' => 431, 'mak ve yXk' => -432,
      'GInda Xlum' => 433, 'ilde Xldu ' => 434,
      'saat sektX' => -435, 'kkInda yXk' => 436,
      'nra dXndur' => -437, 'irinin Xld' => 438,
      'nlerde Xn ' => -439, 'Once Xldu ' => 440,
      'rinin Xnu ' => -441, 'ylerim Xl' => 442,
      'amIS Xlmu' => 443, 'ede dXgus' => 444,
      'l bir Xn ' => -445, 'tedi yXk ' => 446,
      'rden yXk ' => 447, 'arada yXk' => 448,
      ' nun yXk ' => 449, 'lUyor Xlu' => 450,
      'ses Xldu ' => 451, 'lde Xlurk' => 452,
      'vil Xldu ' => 453, 'a dXkuman' => 454,
      'In Xlusun' => 455, 'rIn Xdun ' => -456,
      'a her kXy' => 457, 'meden Xld' => 458,
      'ker Xldu ' => 459, 'rde gXlle' => 460,
      'ncak yXk ' => 461, ' ndan yXk' => 462,
      'cuk Xldu ' => 463, 'tadan Xn ' => 464,
      'hir ve gX' => 465, 'in dXnmus' => -466,
      'dolayI Xl' => 467, 'ra da Xn ' => -468,
      'lerle Xn ' => 469, 'Coktan Xl' => 470,
      'rerek yXk' => 471, 'i de dXnd' => -472,
      'GInI yXk ' => 473, 'gUnde Xn ' => -474,
      'arken yXk' => -475, 'rada Xlmu' => 476,
      'Xlduguyle' => 477, ' in Xlums' => 478,
      ' hem yXk ' => 479, 'amInda Xn' => 480,
      'Serek Xld' => 481, 'asInI Xn ' => -482,
      'nan Xdun ' => -483, 'nsana Xl' => 484,
      'yni Xlmu' => 485, ' sIk gXl' => -486,
      'Cin Xluy' => 487, 'aya kXy ' => -488,
      'lona Xnc' => -489, 'smen Xld' => 490,
      'r jXrgen' => 491, 'ten Xluy' => 492,
      'niz gXl ' => 493, 'eye gXlu' => 494,
      'n 35 Xld' => 495, 'In kXyu ' => -496,
      'tiGi Xn ' => -497, 'san Xluy' => 498,
      '14 Xldu ' => 499, 'Iz kXyu ' => -500,
      'gUnU Xn ' => 501, 'ki dXkum' => 502,
      'saG Xluy' => 503, 'e ali Xl' => 504,
      'nciyi yX' => 505, 'stUn kXr' => 506,
      'rIn Xluy' => 507, 'ble ve X' => -508,
      'InI Xlmu' => 509, 'n in Xn ' => 510,
      'duvar Xr' => 511, 'kanIm dX' => -512,
      'erde Xn ' => 513, '3 te Xld' => 514,
      'rI sXkul' => 515, 'e sol Xl' => 516,
      'rtInI dX' => 517, 'IyI Xlum' => 518, 'nun Xnu ' => 519,
      'eGin yXk' => 520, 'yata dXn' => 521, 'ileri gX' => 522,
      'lUler kX' => -523, 'yine Xn ' => 524, 'Ilan Xn ' => 525,
      'gibi Xn ' => -526, 'u dXnmu' => -527, 'kaCta X' => 528,
      'lIn Xn ' => -529, 'ul dXnd' => -530, 'nyol Xl' => 531,
      'ir dXnm' => -532, 'ldU Xld' => 533, 'ldi yXk' => 534,
      'Xkulunm' => -535, 'mlili X' => 536, 'Xlesiyu' => -537,
      'ok kXyu' => 538, 'isi Xn ' => -539, 'gUn Xn ' => -540,
      ' Xlunu ' => 541, 'zse Xlu' => 542, 'mde Xn ' => 543,
      'lim kXy' => -544, 'eli kXy' => 545, 'n dXnu ' => -546,
      'ater gX' => 547, 'en kXk ' => -548, 'anat Xn' => 549,
      'ray Xn ' => 550, 'Xktukle' => 551, 'Iz sXne' => 552,
      'iseye X' => 553, 'fa majX' => 554, 're majX' => 555,
      '  Xluru' => 556, 'r flXre' => 557, 'kuGu gX' => 558,
      'kle Xgu' => 559, 'k kXru ' => 560, '86 da X' => 561,
      'na Xdun' => -562, 'rIlI Xn' => -563, 'llar gX' => 564,
      ' Xludur' => 565, 'halI dX' => -566, 'sen Xld' => 567,
      'dam Xn ' => 568, 'sIn dXn' => 569, 'ortun X' => 570,
      'diki yX' => 571, 'eli Xn ' => -572, '  Xlmus' => 573,
      'nasI Xn' => 574, 'n dXnuk' => -575, ' in Xn ' => -576,
      'ttin gX' => 577, '  dXnmu' => -578, 'ir Xgun' => 579,
      'mIz Xna' => 580, 'et kXyu' => -581, 'Xlduyse' => 582,
      'gton gX' => 583, 'en kXr ' => -584, 'zeki CX' => -585,
      'da dXn ' => -586, 'nIna kX' => -587, ' Un Xld' => 588,
      'dI kXyu' => 589, 'sXkturu' => 590, 'GIda dX' => 591,
      'aha CXk' => -592, 'eye sXk' => -593, 'stXriz' => 594,
      'xel kX' => 595, 'rXlans' => 596, 'lkU yX' => 597, 'n Xle ' => -598,
      '24 Xld' => 599, 'aim gX' => 600, 'mI Xn ' => -601, 'n cXnk' => 602,
      'ya Xl ' => 603, 'dXnuna' => -604, 'fa kXz' => 605, 'buk Xp' => 606,
      ' Xgunc' => 607, 'n Xkce' => 608, 'dXnele' => -609, 'co Xld' => 610,
      'ncar X' => 611, 'Xmurla' => -612, 'uni gX' => 613, 'Xyuyle' => 614,
      'm a Xr' => -615, 'yp kXk' => 616, '9 bXlu' => 617, 'sXkum ' => 618,
      'sXksa ' => 619, 'gXttin' => 620, 'n Xpen' => 621, 'I an X' => 622,
      'nt CXk' => -623, 'Xcunuy' => -624, ' Xlune' => 625, 'fre kX' => -626,
      'sol Xn' => 627, 'CI Xn ' => -628, 'l Xpen' => 629, 'rk kXt' => -630,
      'Xktule' => 631, 'furt X' => -632, 'Iz Xn ' => 633, 'liz yX' => 634,
      'nis kX' => -635, 'ha Xn ' => 636, 'l Xdun' => -637, 'ste Xn' => 638,
      'i sXle' => 639, 'eva gX' => 640, 'Xlurle' => 641, ' Xper ' => 642,
      'e Xrt ' => 643, 'Xnunce' => 644, 'sXkun ' => 645, ' dXnel' => 646,
      'i Xrt ' => 647, 'Xrdule' => 648, 'Ip Xn ' => 649, 't kXru' => -650,
      'i o X ' => -651, 'Xnukla' => -652, 'oca gX' => 653, 'na sXk' => -654,
      ' I gX ' => -655, 'CIk gX' => 656, 'pek CX' => -657, ' Xteb' => 658,
      'tXdef' => 659, 'Xlyat' => 660, 'rXlev' => 661, 'Xoooy' => 662,
      ' Xsko' => 663, '3 Xre' => -664, 'benkX' => 665, 'Xlena' => -666, 'sOktX' => 667,
      ' Xgdu' => 668, 'ntenX' => 669, 'kXri ' => 670, 'fkI X' => 671, 'lokXr' => 672,
      'ekXyu' => 673, 'Xrelm' => 674, 'Xrulc' => 675, 'eb yX' => 676, 'sXnup' => 677,
      'eny X' => -678, 'r dX ' => 679, 'bXld ' => -680, '34 yX' => -681, ' kXne' => -682,
      'cXrdu' => 683, '5 kXy' => 684, 'sXmer' => -685, 'On bX' => 686, '174 X' => -687,
      'Xser ' => -688, 'bXll ' => 689, 'cok X' => -690, 'tXnuk' => 691, 'veryX' => -692,
      'Xcund' => -693, 'Xrtur' => -694, 'Xgune' => 695, 'oy kX' => -696, ' dXk ' => 697,
      'bXren' => -698, 'ym yX' => 699, 'Xpeli' => 700, 'sU SX' => 701, 'likXr' => 702,
      'Xrsut' => 703, 'Xkucu' => 704, 'Xtuna' => -705, 'Xnurm' => 706, ' lXk ' => 707,
      'kXleo' => 708, ' Xtec' => 709, 'Xkluy' => -710, 'nXtr ' => 711, 'nXtra' => 712,
      ' Xrik' => 713, 'rXne ' => -714, 'Xrnel' => -715, 'esXrl' => 716, '7 kXy' => 717,
      'sXter' => 718, 'Xkume' => 719, 'Xrdeg' => 720, 'Xnerk' => 721, 'fXyun' => 722,
      'dXrtl' => 723, 'Xrece' => 724, 'Xdeme' => 725, 'SutX' => 726, 'Xror' => 727,
      'dXvr' => 728, 'tOrX' => 729, 'sXvs' => 730, 'Xsec' => -731, 'IrdX' => 732,
      'hlXg' => 733, 'gasX' => -734, 'ekXz' => 735, 'mb X' => 736, 'yOzX' => 737,
      'O dX' => 738, ' Xnv' => 739, 'mXbl' => 740, 'UtXz' => 741, ' gyX' => 742,
      'bXge' => 743, 'Xyl ' => 744, 'ntgX' => -745, 'SXmi' => 746, 'Xtet' => 747,
      ' uCX' => 748, 'Xntt' => -749, 'bagX' => -750, 'nkXn' => -751, 'rbjX' => 752,
      'Xyey' => 753, ' jXt' => 754, 'gXlp' => 755, 'Xrea' => -756, 'bXlt' => -757,
      'IkXz' => 758, ' Xnp' => 759, 'Xyel' => 760, 'catX' => -761, 'dXpy' => 762,
      'Xfek' => 763, 'eXrn' => 764, ' f X' => 765, 'mkXy' => 766, ' iX ' => 767,
      'CXrd' => 768, 'Xyse' => 769, 'hXcs' => 770, 'aSXr' => 771, 'ublX' => 772,
      'nXrl' => 773, 'zXle' => -774, 'Xnt ' => -775, 'lkXn' => -776, 'hXdu' => 777,
      'Xcha' => -778, 'dXr ' => -779, 'gOzX' => 780, 'Xri ' => -781, 'Xvi ' => -782,
      'Xztu' => 783, 'ekXr' => -784, 'Xlsu' => -785, 'argX' => -786, ' Xyu' => -787,
      'Xrga' => -788, ' ekX' => -789, 'wXb' => 790, 'eeX' => 791, 'Xvc' => -792,
      'Xds' => -793, 'cXr' => -794, 'liX' => -795, 'crX' => -796, 'iXn' => -797,
      ' zX' => -798, 'Xya' => -799, 'nXm' => -800, 'X2' => -801, 'kanlIGI yXk ' => 802,
      'zalarInda Xl' => 803, ' in Xldugun' => 804, 'itelerin yX' => 805,
      'y yerinde X' => 806, 'Sinin de Xl' => 807, 'kerinin Xl' => 808,
      'e dXkumanl' => 809, ' olan yXk ' => 810, 'da dXndurd' => -811,
      'li dXkuman' => 812, 'zasInda Xl' => 813, 'en kiSi Xl' => -814,
      'rdan yXk ' => 815, 'i kXyunun' => -816, 'sunda Xn ' => 817,
      'lis Xldu ' => 818, 'leri sXku' => 819, 'san Xldu ' => 820,
      'n kim Xlm' => 821, 'nce Xlmus' => 822, 'ada Xldu ' => 823,
      'SInda yXk' => 824, ' ve gXre ' => -825, 'sitesi yX' => 826,
      'rlanan Xn' => 827, 'cuGun Xld' => 828, 'arak Xlur' => 829,
      'le kXyu ' => -830, 'ken kXyu' => 831, 'san Xlmu' => 832,
      'kan kXk ' => -833, 'ri kXyun' => 834, 'amlIk kX' => -835,
      'Glarla X' => 836, 'un kXyu ' => -837, 'in kXruk' => 838,
      'ilen Xn ' => 839, 'kXpuklu ' => 840, 'yIlI yXk' => 841,
      'mada Xn ' => 842, 'Cilen yX' => 843, 'elere Xn' => 844,
      'dUn Xlmu' => 845, 'Olen Xlu' => 846, 'rtIk Xld' => 847,
      'imin Xnu' => 848, 'edi Xn ' => -849, 'iyi Xn ' => -850,
      'kle Xn ' => 851, 'uzla gX' => 852, 'rda Xn ' => 853,
      'sun Xn ' => -854, 'eve dXn' => 855, 'anca gX' => 856,
      'Ica yXk' => 857, 'bXldur ' => -858, 'hep Xn ' => 859, '  Xldug' => 860,
      '  kXku ' => 861, ' gXlov' => 862, '7 Xrt ' => 863, 'sXylus' => -864,
      'gut Xn' => 865, 'Xkumuz' => -866, 'l bXn ' => 867, 'i Xpen' => 868,
      'Xpukte' => 869, ' tXban' => 870, 'kXruk ' => 871, 'l o bX' => -872,
      'nal Xn' => 873, 'dur gX' => 874, 'tek Xp' => 875, ' kXran' => 876,
      ' Xgunl' => 877, 'r CXkt' => -878, 'm tXr ' => 879, 'uat Xr' => -880,
      'fer gX' => 881, 'kXni ' => 882, 'Xnig ' => 883, 'sXmek' => 884,
      ' Xrgo' => 885, 'sXkto' => 886, 'lstrX' => -887, 'brXta' => 888,
      'G Xn ' => 889, ' dXsu' => 890, 'CXrus' => 891, '3 Xgu' => 892,
      'kXrug' => 893, 'Xrteg' => -894, 'gXlko' => 895, 'Xgusk' => 896,
      ' Xsta' => 897, 'Xreck' => -898, 'sistX' => 899, ' Xter' => 900,
      'Xrdog' => 901, 'ntrXn' => 902, 'Xlsen' => -903, 'i gaX' => 904,
      'rkXz ' => 905, ' tXs ' => 906, 'duktX' => 907, 'Xrmuk' => 908,
      'Xpus ' => -909, 'tUnkX' => 910, 'Xnsez' => 911, 'dikdX' => 912,
      ' akXz' => 913, 'Xlero' => -914, ' tXb ' => 915, 'Xmurt' => -916,
      'onglX' => 917, 'ardXn' => -918, ' o X ' => 919, 'Xrtec' => 920,
      'Xcuna' => -921, ' Xvus' => -922, ' inXn' => 923, 'gesX' => 924,
      'ndXv' => 925, 'gXss' => -926, 'zakX' => -927, 'hXh ' => 928,
      ' mfX' => 929, ' Xhc' => 930, ' asX' => -931, ' Xgm' => 932,
      'Xge ' => 933, 'hXch' => 934, 'yyXr' => 935, 'dXro' => -936,
      'anXn' => -937, 'dXrf' => -938, 'Xng ' => -939, 'Xrla' => -940,
      'Xgra' => -941, 'Xsev' => -942, 'gXf' => -943, 'gXh' => -944,
      'Xiz' => -945, 'hXz' => -946, 'zgX' => 947, 'Xyg' => -948,
      'ana nIn Xl' => 949, 'e sonra Xl' => 950, 'rken Xldu ' => 951,
      'elerin Xnu' => 952, 'izinden X' => 953, 'ler kXyun' => 954,
      'ana Xldu ' => 955, ' In Xldug' => 956, 'Xndurulme' => 957, 'kIn dXgus' => 958,
      'nserden X' => 959, 'Once yXk ' => 960, 'celiyle X' => 961, 'metin yXk' => 962,
      've Xldu ' => 963, 'zla kXyu' => -964, 'a da Xn ' => 965, 'yla yXk ' => 966,
      'nra yXk ' => 967, 'Ore yXk ' => 968, 'ken yXk ' => 969, ' In yXk ' => 970,
      'bir yXk ' => 971, 'nIn Xnu ' => 972, ' kXrle ' => -973, 'ani Xn ' => -974,
      'zar gXl' => 975, 'eman gX' => 976, 'nde Xn ' => 977, 'Ina Xn ' => 978,
      'k dXkuy' => -979, 'yas gX' => 980, 'sli Xn' => -981, ' dXksu' => 982,
      'nI sX ' => 983, 'Xpukle' => 984, 'kXrdov' => 985, '  kXk ' => -986,
      'bat gX' => 987, 'pardXs' => 988, 'Xdunda' => -989, 'Un Xn ' => 990,
      'a dXn ' => 991, 'vinatX' => -992, 'ora kX' => -993, ' Xver ' => -994,
      'bXrtu' => 995, 'Xkumc' => 996, 'razXz' => 997, 'ertXr' => 998, 'gXmes' => -999,
      'eryXn' => 1000, ' d Xr' => -1001, 'Xruko' => 1002, 'kXstl' => 1003,
      'the X' => -1004, ' sXbu' => 1005, 'Xnces' => 1006, 'kXyk' => 1007,
      ' lX ' => 1008, 'nrIX' => 1009, 'IgXl' => 1010, 'Xyli' => 1011,
      ' Xdi' => -1012, 'CXle' => 1013, 'Xcek' => 1014, 'Xnoz' => 1015,
      'Xnuv' => 1016, 'dXnl' => -1017, 'dXra' => -1019, 'Xnez' => -1020,
      'Xrto' => -1021, ' tXy' => -1022, 'Xzla' => -1023, 'Xnta' => -1024,
      ' fXr' => -1025, 'Xley' => -1026, 'Xzle' => 1027, 'gXst' => 1028,
      'Xyy' => 1029, 'SXo' => 1030, 'cXz' => -1031, 'gkX' => -1032,
      'CgX' => 1033, 'Xt ' => -1034, 'dXy' => -1035, 'Xbb' => -1036,
      'fXn' => -1037, ' jX' => -1038, 'babasI Xlmu' => 1039,
      'Xlumsuzdur' => 1040, 'arak Xlmus' => 1041, 'asInda Xn ' => 1042,
      'sunda yXk ' => 1043, 'Cin Xldu ' => 1044, 'ucu Xldu ' => 1045,
      ' ile yXk ' => 1046, ' iCin yXk' => 1047, 'ClIktan X' => 1048,
      'de kXyu ' => -1049, 'iden Xn ' => 1050, 'Glam yXk' => 1051,
      'I sXkup' => 1052, ' Xrdugu' => 1053, 'rI dXnd' => -1054,
      ' en kXy' => -1055, ' hXrt ' => 1056, 'pXrsum' => 1057,
      ' Xluyu' => 1058, ' Xker' => 1059, 'presX' => 1060,
      ' Xnis' => 1061, 'Xlusp' => -1062, 'gXne ' => -1063, 'dXnay' => 1064,
      ' Xkse' => 1065, 'nkXr ' => 1066, 'tXzum' => 1067, ' sXre' => -1068,
      'Xmuri' => -1069, 'rXves' => 1070, 'gXmez' => -1071, 'rkXru' => 1072,
      'etXr ' => 1073, ' Xtem' => 1074, 'dXven' => 1075, 'CXpur' => -1076,
      'bu Xn ' => -1077, 'u Xn ' => 1078, ' Xtuy' => 1079, 'dXrdu' => 1080,
      'Xngor' => 1081, 'iXk ' => 1082, 'kXko' => -1083, 'Xrod' => -1084,
      'Xmea' => 1085, 'bXgr' => 1086, 'dXri' => -1087, 'gXp ' => -1088,
      ' ihX' => 1089, 'Xdeo' => -1090, 'Xdug' => -1091, 'izXr' => 1092,
      'rXme' => -1093, 'Xska' => -1094, 'fiX' => -1095, 'Xti' => -1096,
      'Xbi' => -1097, 'Xrr' => -1098, 'yaSInda Xldu ' => 1099,
      'an kXyun ' => 1100, ' kiSi Xlu' => 1101, ' yeni yXk' => 1102,
      'ogan gX' => 1103, 'ler Xn ' => 1104, 'laya gX' => 1105, 'dnan CX' => -1106,
      '  kXyu ' => -1107, 'Xlmusle' => 1108, 'orce X' => -1109,
      'un Xn ' => 1110, 'Xburlu' => -1111, 'baS dX' => 1112,
      'ye dXn' => 1113, 'Xverm' => -1114, 'Xdesa' => -1115,
      ' ikX ' => 1116, 'kXktu' => -1117, ' Xpey' => 1118, 'Xngoz' => 1119,
      'IlXte' => 1120, 'rXmiy' => 1121, ' Xrer' => 1122, 'Xkunt' => 1123,
      'Xcunm' => -1124, 'sXvd' => 1125, 'hXne' => -1126, 'Xvid' => -1127,
      'jisX' => 1128, ' SXy' => 1129, 'gXru' => 1130, 'Xlv' => -1131,
      'OnX' => 1132, 'sanlar Xlu' => 1133, ' Xlunun ' => 1134, ' kale Xn' => 1135,
      'Xldugune' => 1136, 'Xldukle' => 1137, 'geri dX' => 1138, 'Xtariu' => -1139,
      'rXlant' => 1140, ' Xngur' => 1141, 'Xldule' => 1142, 'Xteler' => 1143,
      'Xrdub' => 1144, 'schXn' => 1145, 'l Xke' => 1146, 'kXrpe' => 1147,
      'Xguto' => 1148, 'Xgutl' => 1149, 'Xlcek' => 1150, 'Xntma' => -1151,
      'Xkule' => 1152, 'jXrk ' => 1153, 'lpXz' => 1154, 'OrdX' => 1155,
      'Xldo' => -1156, ' Xrl' => -1157, 'gXv ' => -1158, 'rkXy' => 1159,
      'Xvgu' => 1160, 'agXl' => 1161, 'fXz' => -1162, 'Xsp' => -1163,
      'Xlk' => -1164, 'Xdy' => -1165, 'a dXnduruld' => 1166,
      ' van gXl' => 1167, 'nin Xnu ' => 1168, '  gXre ' => -1169,
      'sXnuyo' => 1170, 'k pXrc' => 1171, 'la Xn ' => 1172,
      'Xkulup' => 1173, 'Xrunde' => 1174, ' Xgud' => 1175, 'kXpug' => 1176,
      ' m X ' => 1177, 'minXr' => 1178, 'Xnesc' => -1179, 'Xtary' => -1180,
      ' Xnge' => 1181, 'Xleg' => -1182, 'yXng' => -1183, 'CXml' => 1184,
      'dXrn' => -1185, 'sXvu' => 1186, 'Xniz' => -1187, 'erXn' => -1188,
      'larak Xn ' => 1189, 'l kXyun ' => 1190, ' ile Xn ' => 1191,
      'n sXkup' => 1192, 'lIk Xn ' => 1193, '  kXyun' => 1194,
      'Xnkolo' => -1195, ' Xnsec' => 1196, '173 X' => -1197,
      'kXper' => -1198, 'mirXz' => 1199, ' Xsyo' => -1200,
      'Xksur' => 1201, 'Xkpit' => -1202, 'Xregi' => 1203,
      ' Xnel' => 1204, 'kXyis' => 1205, 'sXgus' => 1206,
      'kkeX' => 1207, 'CXme' => 1208, 'Xnev' => -1209,
      'SbXl' => 1210, ' fkX' => 1211, 'efkX' => -1212,
      'dXnk' => -1213, 'Xnec' => 1214, 'IrXz' => 1215,
      'Xdo' => -1216, ' pX' => -1217, 'Xno' => -1218, 'r kXyunu' => -1219,
      'dXndurur' => 1220, 'n kXku ' => 1221, 'ik Xn ' => 1222, 'kOk sX' => 1223,
      'isarX' => 1224, 'Xlude' => 1225, 'schrX' => 1226, ' Xruy' => 1227,
      ' Xrfi' => 1228, ' mXsy' => 1229, 'bXgur' => 1230, ' Xkke' => 1231,
      'CXlun' => 1232, ' yXne' => 1233, 'nSXr' => 1234, 'fXtr' => 1235,
      'syXd' => 1236, 'OsyX' => 1237, 'Xnma' => -1238, 'odX' => -1239, 'gXk' => 1240,
      'arak Xld' => 1241, ' van gX' => -1242, 'tuz gX' => 1243, ' Xperi' => 1244,
      'Xkulu ' => -1245, 'opXrta' => 1246, 'trXst' => 1247, ' Xylu' => -1248,
      'dXkup' => 1249, ' Xlse' => 1250, 'Xdess' => -1251, 'Xkuml' => 1252,
      'rXmor' => 1253, 'gXve' => -1254, 'Xkuk' => 1255, 'Xbek' => 1256, 'Xndurule' => 1257,
      'Si Xlmu' => 1258, 'wim kX' => -1259, 'Xnlis' => 1260, '1 Xz ' => -1261, 'CkinX' => 1262,
      'valXr' => 1263, 'trXm ' => 1264, 'CXlu ' => 1265, ' Xttu' => 1266, 'dXkmu' => 1267,
      ' Xtv' => 1268, 'bXke' => 1269, 'Xntu' => -1270, 'bUtX' => 1271, 'dXnm' => 1272,
      'Xbp' => 1273, 'rXner ' => -1274, 'nXtro' => 1275, 'Xrelt' => 1276, 'gXng' => -1277,
      'rjXr' => 1278, 'Xkug' => 1279, 'kXkm' => -1280, 'kXlesi' => 1281, 'Xktur ' => -1282,
      '179 X' => -1283, ' Xrtt' => 1284, 'grXnl' => 1285, 'Xsemi' => 1286, 'vatX' => -1287,
      'Xtme' => 1288, 'Xkez' => 1289, 'gXlb' => 1290, 'oX' => -1291, ' Xkmey' => -1292,
      'sXylul' => -1293, ' SXval' => 1294, 'rolXr' => 1295, 'CXld' => 1296,
      'UzXl' => 1297, 'tXri' => -1298, 'Xlumsuzle' => 1299, 'lar Xn ' => 1300,
      'Xndura' => -1301, '178 X' => -1302, ' Xnsa' => 1303, 'CXlas' => 1304, 'sXmes' => 1305,
      'Xkula' => -1306, 'tXmu' => -1307, 'tXhm' => 1308, 'Xlo' => -1309, 'prX' => -1310,
      've yXk ' => 1311, 'Xperek' => 1312, ' Xkuz' => 1313, '77 X' => -1314, 'negX' => 1315,
      'rlXr' => 1316, ' SXr' => -1317, 'j gX' => 1318, 'Xyku' => 1319, 'eXn' => -1320,
      'OX' => 1321, 'Xu' => -1322, 'Xlurse' => 1323, ' kXrdu' => 1324, 'Xkulme' => 1325,
      ' dXlle' => 1326, 'sIk gX' => 1327, ' nXro' => 1328, 'mos X' => -1329, 'dXrt ' => 1330,
      ' Xcu' => 1331, 'UmXr' => 1332, 'kXly' => -1333, 'cXn' => -1334, 'Xldukte' => 1335,
      ' en Xn ' => 1336, 'sXgut ' => 1337, ' Xten' => 1338, 'rXtus' => 1339, 'kXyum' => 1340,
      'dXge' => 1341, ' kXyunu' => 1342, 'kXyc' => 1343, 'Xldugunde' => 1344, 'bir Xn ' => 1345,
      'ber Xn' => -1346, 'kXrlu' => 1347, 'anXr ' => 1348, ' Xrum' => 1349, 'lmX ' => 1350,
      '76 X' => -1351, 'sXve' => 1352, 'sXvm' => 1353, 'oyXz' => 1354, 'Xnayak' => 1355,
      'portXr' => 1356, 'ki Xn ' => -1357, 'Xzon ' => -1358, 'dXrs' => -1359,
      'aXf' => 1360, 'SdX' => 1361, 'Xtt' => -1362, 'sXnuk' => 1363, ' Xc ' => 1364,
      'Xrpu' => 1365, ' stX ' => 1366, ' Xrme' => 1367, 'Xyuk' => 1368, 'syXv' => 1369,
      'njXr' => 1370, 'dXrm' => -1371, 'Xpur' => 1372, 'uafX' => 1373, ' Xgr' => 1374,
      ' gXlet' => 1375, ' Xmurg' => -1376, 'sXnen' => 1377, ' dXn ' => -1378,
      'Xnel ' => -1379, 'Xkla' => -1380, 'gXd' => -1381, 'Xe' => -1382, ' al gXr' => -1383,
      'dXkuyo' => 1384, '2 Xz ' => -1385, 'mXlle' => 1386, ' mXnu' => 1387, 'CXlle' => 1388,
      ' Xrf ' => 1389, 'i Xn ' => 1390, ' tXre' => 1391, 'sXrf' => 1392, 'tXvb' => 1393,
      'blXf' => 1394, ' kXle ' => 1395, ' Xpm' => 1396, 'Xrdek' => 1397, 'kXhne' => 1398,
      'angX' => -1399, 'bXls' => -1400, 'anliyX' => 1401, '  yXk ' => 1402, 'asXr' => 1403,
      'ntXr' => 1404, 'CXkm' => 1405, 'tfXy' => 1406, 'OfX' => 1407, 'CXkul' => -1408,
      'Xlumc' => 1409, 'hXm' => -1410, ' kan gX' => 1411, ' Xptu' => 1412, 'ntrX ' => 1413,
      'rXtar' => 1414, 'lXrt' => 1415, ' Xib' => 1416, ' Xnso' => 1417, 'Xnn' => -1418,
      'Xlunce' => 1419, 'ansXr' => 1420, 'UktX' => 1421, ' yXruk' => 1422, 'sXylu ' => -1423,
      'kXrog' => 1424, 'kXlel' => 1425, 'Xran' => -1426, 'zXz' => -1427, 'Xksuz' => 1428,
      'itXr' => 1429, 'kgX' => 1430, ' Xrul' => 1431, 'dXkumu' => 1432, 'yXner' => -1433,
      'oksXr' => 1434, 'CXl ' => 1435, ' Xlul' => 1436, 'Xrdo' => -1437, 'iSinin Xld' => 1438,
      ' Xkte' => 1439, 'bXll' => -1440, 'In Xn ' => 1441, 'in Xn ' => 1442, 'Xpen' => -1443,
      'Xla' => -1444, 'myX' => -1445, ' Xrte' => 1446, ' rXnt' => 1447, 'CXktu' => 1448,
      ' Xlumd' => 1449, 'Xry' => -1450, 'Xlcuk' => 1451, 'I Xn ' => 1452, ' Xnk' => 1453,
      'dXnduru' => -1454, 'kXkun' => 1455, 'Xnz' => -1456, 'kXfte' => 1457, 'sXke' => 1458,
      'Xrek' => 1459, 'yXrun' => 1460, 'Xdur' => -1461, 'dXktu' => 1462, 'lXrl' => 1463,
      'bXce' => 1464, ' Xnlu' => 1465, ' Xnal ' => 1466, 'Xrb' => -1467, 'bXylu' => -1468,
      ' Xle' => 1469, 'Xrtb' => 1470, 'Xtey' => 1472, 'Xndurma' => -1473, 'Xlca' => -1474,
      ' Xrs' => 1475, 'lX' => -1476, 'Xrune' => 1477, 'tXn' => -1478, 'kXrukl' => 1479,
      ' Xlume' => 1480, ' kXr ' => 1481, 'kiSi Xldu ' => 1482, 'Xye ' => 1483,
      ' Xgut' => 1484, ' Xcal' => 1485, 'kXln' => 1486, 'Xted' => 1487, 'nXz' => -1488,
      'dXkum ' => 1489, ' Xys ' => 1490, 'Xktey' => -1491, 'kXka' => -1492, ' Xpu' => 1493,
      'kXk ' => 1494, 'bXbre' => 1495, 'bingX' => 1496, ' Xrtm' => 1497, ' Xss ' => 1498,
      'dXz' => -1499, 'gXlt' => 1500, ' Xturu ' => 1501, 'dXke' => 1502, 'kXyt' => 1503,
      ' Xva' => -1504, 'Xnes ' => -1505, 'Xsko' => -1506, 'kXyde' => 1507, 'Xbet' => 1508,
      'Xnca' => -1509, ' dXse' => 1510, 'kXku' => -1511, 'hXgh ' => 1512, ' Xlumle' => 1513,
      'SXhr' => 1514, 'Xo' => -1515, ' rXva' => 1516, ' Xsy' => 1517, 'gXlh' => 1518,
      ' Xnyar' => 1519, 'ndX' => -1520, 'kXsk' => 1521, 'CXke' => 1522, ' Xge' => 1523,
      'Xlye' => 1524, 'dXru' => -1525, 'kXpe' => 1526, 'tXz' => -1527, 'Xdt' => -1528,
      'Xyunde' => 1529, 'Xporta' => 1530, 'Xrtm' => -1531, 'kXste' => 1532,
      'Xzan' => -1533, ' Xmr' => 1534, ' kXyu ' => 1535, 'mX' => -1536, 'bXn' => -1537,
      'bXlu ' => -1538, 'Xsa' => -1539, ' CXp' => 1540, 'Xkme' => 1541, 'SXrt' => 1542,
      'CXku' => 1543, 'yXz' => -1544, 'Xbur' => 1545, ' Xnunu' => 1546, 'trenX' => 1547,
      'aratX' => -1548, 'Xktas ' => 1549, ' Xte ' => 1550, 'dXkul' => 1551,
      'Xyme' => 1552, ' Xlme' => 1553, ' Xlu ' => 1554, 'rXz' => -1555, 'atXr' => 1556,
      'SgX' => 1557, 'kXrfe' => 1558, ' Xgle' => 1559, 'IldX' => 1560, 'aktXr' => 1561,
      ' Xrtu' => 1562, 'yXre' => 1563, 'SofX' => 1564, 'gXlg' => 1565, ' Xv' => 1566,
      'sXnm' => 1567, 'Xli' => -1568, 'ofesX' => 1569, 'Xl ' => -1570, 'Xfk' => 1571,
      'Xlen' => 1572, 'nXnu ' => 1573, ' sXndu' => 1574, 'Xmer' => 1575,
      'Xtesi' => 1576, 'Xna' => -1577, 'syX' => -1578, ' Xtek' => 1579,
      'kXz' => -1580, ' Xlum ' => 1581, 'Xmur' => 1582, ' kXk' => 1583,
      ' Xyk' => 1584, ' Xnde' => 1585, 'kXpr' => 1586, 'Xrle' => 1587,
      ' Xlumu' => 1588, 'sXn' => -1589, 'ngXr' => 1590, 'Xnune' => 1591,
      'kXse' => 1592, ' Xngo' => 1593, ' dXv' => 1594, ' Xnum' => 1595,
      'pX' => -1596, 'kXy ' => 1597, 'Xylu' => 1598, ' Xlc' => 1599,
      ' Xldur' => 1600, 'Xnunde' => 1601, 'Xda' => -1602, ' kXtu' => 1603,
      'bXz' => -1604, 'erXr' => 1605, 'Xren' => 1606, ' dXr' => 1607,
      'gXl' => -1608, 'Xnle' => 1609, 'ektXr' => 1610, ' Xrgu' => 1611,
      ' Xrne' => 1612, ' yXn' => 1613, ' Xd' => 1614, ' dXn' => 1615,
      ' bXl' => 1616, 'Xgre' => 1617, ' Xnc' => 1618, 'Xyle' => 1619,
      'Xne' => 1620, 'Xz' => 1621, ' gX' => 1622, 'X' => -1623
    }
  }
;

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Lingua::TR::ASCII::Data

=head1 VERSION

version 0.16

=head1 SYNOPSIS

None.

=head1 DESCRIPTION

Private module.

=head1 NAME

Lingua::TR::ASCII::Data - Data sets for Lingua::TR::ASCII

=head1 SEE ALSO

L<Lingua::TR::ASCII>.

=head1 AUTHOR

Burak Gursoy <burak@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2011 by Burak Gursoy.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
