use 5.008001;
use strict;
use warnings;

package Dist::Zilla::Plugin::CopyFilesFromBuild::Filtered;
# ABSTRACT: Copy files from build directory, but filter out lines

our $VERSION = '0.001';

use Moose;
use Path::Tiny;
use Try::Tiny;

with qw/ Dist::Zilla::Role::AfterBuild /;

#pod =attr copy
#pod
#pod Specifies files to be copied from the build directory back to the
#pod distribution source.  May be given multiple times.
#pod
#pod =cut

has copy => (
    is      => 'ro',
    isa     => 'ArrayRef[Str]',
    default => sub { [] },
);

#pod =attr filter
#pod
#pod A regular expression given as string.  Any matching lines in copied files
#pod will be filtered out of the copy.  It will be applied as follows:
#pod
#pod     s{^$filter\n}{}mg
#pod
#pod Thus, it is anchored at the start of a line to the newline at the end.
#pod
#pod The default matches a comment with "generated by" as found in typical
#pod generated *.PL files:  C<#.*generated by.*>
#pod
#pod =cut

has filter => (
    is      => 'ro',
    isa     => 'Str',
    default => '#.*generated by.*',
);

sub mvp_multivalue_args { 'copy' }

sub after_build {
    my ( $self, $stash ) = @_;

    my $build_root = $stash->{build_root};

    for my $file ( @{ $self->copy } ) {
        next unless length $file;
        my $from = $build_root->file($file);
        my $to   = $self->zilla->root->file($file);
        $self->_copy_filtered( $from, $to ) if -e $from;
    }
}

sub _copy_filtered {
    my ( $self, $from, $to ) = @_;

    try {
        my $filter = $self->filter;
        my $guts   = path($from)->slurp_raw;
        $guts =~ s{^$filter\n}{}mg;
        path($to)->spew_raw($guts);
        $self->log("Copied $from to $to");
    }
    catch {
        $self->log_fatal("Unable to copy $from to $to: $_");
    }
}

__PACKAGE__->meta->make_immutable;

1;


# vim: ts=4 sts=4 sw=4 et tw=75:

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::CopyFilesFromBuild::Filtered - Copy files from build directory, but filter out lines

=head1 VERSION

version 0.001

=head1 SYNOPSIS

    [CopyFilesFromBuild::Filtered]
    copy = Makefile.PL

    [CopyFilesFromBuild::Filtered]
    copy = whatever.txt
    filter = #.*whatever.*

=head1 DESCRIPTION

This module copies files from the build directory to the source directory,
but filters out lines from the copied file.  It is designed in particular
for generated *.PL files, which often have unwanted "generated by" comments
that cause unwanted file churn under source control.

=head1 ATTRIBUTES

=head2 copy

Specifies files to be copied from the build directory back to the
distribution source.  May be given multiple times.

=head2 filter

A regular expression given as string.  Any matching lines in copied files
will be filtered out of the copy.  It will be applied as follows:

    s{^$filter\n}{}mg

Thus, it is anchored at the start of a line to the newline at the end.

The default matches a comment with "generated by" as found in typical
generated *.PL files:  C<#.*generated by.*>

=for Pod::Coverage mvp_multivalue_args after_build

=head1 SEE ALSO

=over 4

=item *

L<Dist::Zilla::Plugin::CopyFilesFromBuild>

=item *

L<Dist::Zilla::Plugin::CopyReadmeFromBuild>

=item *

L<Dist::Zilla::Plugin::CopyFilesFromRelease>

=item *

L<Dist::Zilla::Plugin::CopyMakefilePLFromBuild>

=back

=for :stopwords cpan testmatrix url annocpan anno bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the issue tracker
at L<https://github.com/dagolden/Dist-Zilla-Plugin-CopyFilesFromBuild-Filtered/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/dagolden/Dist-Zilla-Plugin-CopyFilesFromBuild-Filtered>

  git clone https://github.com/dagolden/Dist-Zilla-Plugin-CopyFilesFromBuild-Filtered.git

=head1 AUTHOR

David Golden <dagolden@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by David Golden.

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut
