package Locale::Country::EU;

use 5.016003;
use strict;
use warnings;
use Carp;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
    is_eu_country
    list_eu_countries
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw();

our $VERSION = '0.001';
# ABSTRACT: Module to check if a country is in the EU ( Europe )

use Readonly;
Readonly my $EU_COUNTRY_MAP => [
    {
        'ISO-name'   => 'Bulgaria',
        'ISO-m49'    => '100',
        'ISO-alpha3' => 'BGR',
        'ISO-alpha2' => 'BG',
    },
    {
        'ISO-name'   => 'Hungary',
        'ISO-m49'    => '348',
        'ISO-alpha3' => 'HUN',
        'ISO-alpha2' => 'HU',
    },
    {
        'ISO-name'   => 'Poland',
        'ISO-m49'    => '616',
        'ISO-alpha3' => 'POL',
        'ISO-alpha2' => 'PL',
    },
    {
        'ISO-name'   => 'Romania',
        'ISO-m49'    => '642',
        'ISO-alpha3' => 'ROU',
        'ISO-alpha2' => 'RO',
    },
    {
        'ISO-name'   => 'Slovakia',
        'ISO-m49'    => '703',
        'ISO-alpha3' => 'SVK',
        'ISO-alpha2' => 'SK',
    },
    {
        'ISO-name'   => 'Denmark',
        'ISO-m49'    => '208',
        'ISO-alpha3' => 'DNK',
        'ISO-alpha2' => 'DK',
    },
    {
        'ISO-name'   => 'Estonia',
        'ISO-m49'    => '233',
        'ISO-alpha3' => 'EST',
        'ISO-alpha2' => 'EE',
    },
    {
        'ISO-name'   => 'Finland',
        'ISO-m49'    => '246',
        'ISO-alpha3' => 'FIN',
        'ISO-alpha2' => 'FI',
    },
    {
        'ISO-name'    => 'Iceland',
        'ISO-m49'     => '352',
        'ISO-alpha3'  => 'ISL',
        'ISO-alpha2'  => 'IS',
        'EFTA-member' => 'true',
    },
    {
        'ISO-name'   => 'Ireland',
        'ISO-m49'    => '372',
        'ISO-alpha3' => 'IRL',
        'ISO-alpha2' => 'IE',
    },
    {
        'ISO-name'   => 'Lithuania',
        'ISO-m49'    => '440',
        'ISO-alpha3' => 'LTU',
        'ISO-alpha2' => 'LT',
    },
    {
        'ISO-name'    => 'Norway',
        'ISO-m49'     => '578',
        'ISO-alpha3'  => 'NOR',
        'ISO-alpha2'  => 'NO',
        'EFTA-member' => 'true',
    },
    {
        'ISO-name'   => 'Sweden',
        'ISO-m49'    => '752',
        'ISO-alpha3' => 'SWE',
        'ISO-alpha2' => 'SE',
    },
    {
        'ISO-name'   => 'United Kingdom',
        'ISO-m49'    => '826',
        'ISO-alpha3' => 'GBR',
        'ISO-alpha2' => 'GB',
    },
    {
        'ISO-name'   => 'Croatia',
        'ISO-m49'    => '191',
        'ISO-alpha3' => 'HRV',
        'ISO-alpha2' => 'HR',
    },
    {
        'ISO-name'   => 'Greece',
        'ISO-m49'    => '300',
        'ISO-alpha3' => 'GRC',
        'ISO-alpha2' => 'GR',
    },
    {
        'ISO-name'   => 'Italy',
        'ISO-m49'    => '380',
        'ISO-alpha3' => 'ITA',
        'ISO-alpha2' => 'IT',
    },
    {
        'ISO-name'   => 'Malta',
        'ISO-m49'    => '470',
        'ISO-alpha3' => 'MLT',
        'ISO-alpha2' => 'MT',
    },
    {
        'ISO-name'   => 'Portugal',
        'ISO-m49'    => '620',
        'ISO-alpha3' => 'PRT',
        'ISO-alpha2' => 'PT',

    },
    {
        'ISO-name'   => 'Slovenia',
        'ISO-m49'    => '705',
        'ISO-alpha3' => 'SVN',
        'ISO-alpha2' => 'SI',
    },
    {
        'ISO-name'   => 'Spain',
        'ISO-m49'    => '724',
        'ISO-alpha3' => 'ESP',
        'ISO-alpha2' => 'ES',
    },
    {
        'ISO-name'   => 'Austria',
        'ISO-m49'    => '40',
        'ISO-alpha3' => 'AUT',
        'ISO-alpha2' => 'AT',
    },
    {
        'ISO-name'   => 'Belgium',
        'ISO-m49'    => '56',
        'ISO-alpha3' => 'BEL',
        'ISO-alpha2' => 'BE',
    },
    {
        'ISO-name'   => 'France',
        'ISO-m49'    => '250',
        'ISO-alpha3' => 'FRA',
        'ISO-alpha2' => 'FR',
    },
    {
        'ISO-name'   => 'Germany',
        'ISO-m49'    => '276',
        'ISO-alpha3' => 'DEU',
        'ISO-alpha2' => 'DE',
    },
    {
        'ISO-name'    => 'Liechtenstein',
        'ISO-m49'     => '438',
        'ISO-alpha3'  => 'LIE',
        'ISO-alpha2'  => 'LI',
        'EFTA-member' => 'true',
    },
    {
        'ISO-name'   => 'Luxembourg',
        'ISO-m49'    => '442',
        'ISO-alpha3' => 'LUX',
        'ISO-alpha2' => 'LU',
    },
    {
        'ISO-name'   => 'Netherlands',
        'ISO-m49'    => '528',
        'ISO-alpha3' => 'NLD',
        'ISO-alpha2' => 'NL',
    },
    {
        'ISO-name'    => 'Switzerland',
        'ISO-m49'     => '756',
        'ISO-alpha3'  => 'CHE',
        'ISO-alpha2'  => 'CH',
        'EFTA-member' => 'true',
    },
    {
        'ISO-name'   => 'Cyprus',
        'ISO-m49'    => '196',
        'ISO-alpha3' => 'CYP',
        'ISO-alpha2' => 'CY',
    },
    {
        'ISO-name'   => 'Czech Republic',
        'ISO-m49'    => '203',
        'ISO-alpha3' => 'CZE',
        'ISO-alpha2' => 'CZ',
    },
    {
        'ISO-name' => 'Latvia',
        'ISO-m49'        => '428',
        'ISO-alpha3' => 'LVA',
        'ISO-alpha2' => 'LV',
    }
];

sub is_eu_country {
    my $arg = shift || croak "is_eu_country requires a country";

    foreach my $country ( @{$EU_COUNTRY_MAP} )
    {
        foreach my $value ( values $country )
        {
            if ( $value eq $arg ) {
                return 1;
            }
        }
    }

    return 0;
}

sub list_eu_countries {
    return $EU_COUNTRY_MAP;
}

1;

__END__

=head1 NAME

Locale::Country::EU - Perl extension for determining if a country is within the EU using various ISO values

=head1 SYNOPSIS

  use Locale::Country::EU;

  my $array = list_eu_countries;
  my $boolean = is_eu_country($array->[0]->{'ISO-name'});

=head1 DESCRIPTION

Provides a list of EU countries with various ISO data and allows for checking if a country is in the EU.

=head2 EXPORT

    is_eu_country
    list_eu_countries

=head1 AUTHOR

mgreen, E<lt>matt@mattsizzle.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2018 by mgreen

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.16.3 or,
at your option, any later version of Perl 5 you may have available.

=cut