use strict;
use warnings;

use Test::More;
use Data::Dumper;

my @tests = (
    [
        number => [
            [ -1,        1 ],
            [ 3.1415926, 1 ],
            [ '100',     1 ],
            [ '0E0',     1 ],
            [ 1_000,     1 ],
            [ '1_000',   0 ],
            [ "text",    0 ],
            [ undef,     0 ],
            [ '-10',     1 ],
            [ '+10',     1 ],
            [ '-',       0 ],
            [ '+',       0 ],
            [ '1+1',     0 ],
            [ [],        0 ],
            [ bless({}, 'Foo::Bar'), 0 ],
        ],
    ],
    [
        positive => [
            [ -1,        0 ],
            [ 3.1416926, 1 ],
            [ 0,         0 ],
            [ 'text',    0 ],
            [ undef,     0 ],
        ],
    ],
    [
        non_negative => [
            [ -1,        0 ],
            [ 3.1416926, 1 ],
            [ 0,         1 ],
            [ 'text',    0 ],
            [ undef,     0 ],
        ],
    ],
    [
        negative => [
            [ -1,        1 ],
            [ 3.1416926, 0 ],
            [ 0,         0 ],
            [ 'text',    0 ],
            [ undef,     0 ],
        ],
    ],
    [
        non_positive => [
            [ -1,        1 ],
            [ 3.1416926, 0 ],
            [ 0,         1 ],
            [ 'text',    0 ],
            [ undef,     0 ],
        ],
    ],
    [
        integer => [
            [ -1,        1 ],
            [ 100,       1 ],
            [ 3.1416926, 0 ],
            [ 0,         1 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ '10',      1 ],
            [ '-10',     1 ],
            [ {},        0 ],
        ]
    ],
    [
        positive_int => [
            [ -1,        0 ],
            [ 100,       1 ],
            [ 3.1416926, 0 ],
            [ 0,         0 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ '10',      1 ],
            [ '-10',     0 ],
            [ {},        0 ],
        ],
    ],
    [
        non_negative_int => [
            [ -1,        0 ],
            [ 100,       1 ],
            [ 3.1416926, 0 ],
            [ 0,         1 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ '10',      1 ],
            [ '-10',     0 ],
            [ {},        0 ],
        ],
    ],
    [
        negative_int => [
            [ -1,        1 ],
            [ 100,       0 ],
            [ 3.1416926, 0 ],
            [ 0,         0 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ '10',      0 ],
            [ '-10',     1 ],
            [ {},        0 ],
        ],
    ],
    [
        non_positive_int => [
            [ -1,        1 ],
            [ 100,       0 ],
            [ 3.1416926, 0 ],
            [ 0,         1 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ '10',      0 ],
            [ '-10',     1 ],
            [ {},        0 ],
        ],
    ],
    [
        string => [
            [ -1,        1 ],
            [ '',        1 ],
            [ 100,       1 ],
            [ 3.1416926, 1 ],
            [ 0,         1 ],
            [ 'text',    1 ],
            [ undef,     0 ],
            [ '10',      1 ],
            [ '-10',     1 ],
            [ {},        0 ],
        ],
    ],
    [
        array => [
            [ -1,        0 ],
            [ 100,       0 ],
            [ 3.1416926, 0 ],
            [ 0,         0 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ '10',      0 ],
            [ '-10',     0 ],
            [ {},        0 ],
            [ [],        1 ],
            [ [1,2,'q'], 1 ],
            [ [{}],      1 ],
        ],
    ],
    [
        hash => [
            [ -1,        0 ],
            [ 100,       0 ],
            [ 3.1416926, 0 ],
            [ 0,         0 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ '10',      0 ],
            [ '-10',     0 ],
            [ {},        1 ],
            [ [],        0 ],
            [ {1 =>2},   1 ],
            [ {1=>{}},   1 ],
        ],
    ],
    [
        code => [
            [ -1,        0 ],
            [ 3.1416926, 0 ],
            [ 'text',    0 ],
            [ undef,     0 ],
            [ {},        0 ],
            [ [],        0 ],
            [ sub { 1;}, 1 ],
        ],
    ],
);

# Tests for enum
my @enum_tests = (
    [ -1,     [1..5],                 0 ],
    [ 100,    [95..105],              1 ],
    [ 'text', [ qw/word text book/ ], 1 ],
    [ undef,  [ qw/a b c d e f/ ],    0 ],
    [ {},     [ 1 ],                  0 ],
    [ [],     [ 1 ],                  0 ],
);

for my $et ( @enum_tests ) {
    $et->[1] = { map { $_ => undef } @{ $et->[1]} };
}

push @tests, [ enum => \@enum_tests ];

# Tests for spec
my @spec_tests = ();


my $test_count = 0;
for my $f ( @tests ) {
    for my $c ( @{ $f->[1] } ) {
        $test_count++;
    }
}

plan tests =>
    1              # Use Data::Types
    + 1            # Use the class
    + 1            # Create an object
    + $test_count  # Tests all, but spec
    + $test_count  # Test spec
    ;

use_ok('Data::Types');
use_ok('Validate::Simple');
my $validate = new_ok( 'Validate::Simple' );


for my $test ( @tests ) {
    my ( $meth, @cases ) = ( $test->[0], @{ $test->[1] } );
    for my $c ( @cases ) {
        my $expected_true = pop @$c;
        my $printable = join(',', map {
            defined( $_ )
                ? (
                    ref( $_ )
                    ? Dumper( $_ )
                    : $_ )
                : "[undef]";
        } @$c );
        my $validate_spec = Validate::Simple->new(
            {
                var => {
                    type => $meth,
                    # Arrays and hashes require "of"
                    ( $meth =~ /^(array|hash)$/
                      ? ( of => { type => 'any' }, empty => 1 )
                      : () ),
                    # Enums require "values"
                    ( $meth eq 'enum'
                      ? ( values => [ keys %{ $c->[1] } ] )
                      : () ),
                }
            }
        );
        my $message = "'$meth' with <$printable> returns";
        if ( $expected_true ) {
            ok( $validate->$meth( @$c ), "$message true" );
            ok( $validate->spec( { var => $c->[0] }, $validate_spec ), "Subspec: $message true" );
        }
        else {
            ok( !$validate->$meth( @$c ), "$message false" );
            ok( !$validate->spec( $c, $validate_spec ), "Subspec: $message false" );
        }
    }
}

1;
