# Verify serialization

use strict;
use warnings;

use Test::More 'tests' => 2004;
use Config;

my @WARN;
BEGIN {
    # Warning signal handler
    $SIG{__WARN__} = sub { push(@WARN, @_); };
}

use_ok('Math::Random::MT::Auto', ':!auto');
use_ok('Math::Random::MT::Auto::Range');

# Known test values for irand()
my @base_rint = ($Config{'uvsize'} == 8) ?
  (  # 64-bit randoms
     7266447313870364031,  4946485549665804864, 16945909448695747420, 16394063075524226720,  4873882236456199058,
    14877448043947020171,  6740343660852211943, 13857871200353263164,  5249110015610582907, 10205081126064480383,
     1235879089597390050, 17320312680810499042, 16489141110565194782,  8942268601720066061, 13520575722002588570,
    14226945236717732373,  9383926873555417063, 15690281668532552105, 11510704754157191257, 15864264574919463609,
     6489677788245343319,  5112602299894754389, 10828930062652518694, 15942305434158995996, 15445717675088218264,
     4764500002345775851, 14673753115101942098,   236502320419669032, 13670483975188204088, 14931360615268175698,
     8904234204977263924, 12836915408046564963, 12120302420213647524, 15755110976537356441,  5405758943702519480,
    10951858968426898805, 17251681303478610375,  4144140664012008120, 18286145806977825275, 13075804672185204371,
    10831805955733617705,  6172975950399619139, 12837097014497293886, 12903857913610213846,   560691676108914154,
     1074659097419704618, 14266121283820281686, 11696403736022963346, 13383246710985227247,  7132746073714321322,
    10608108217231874211,  9027884570906061560, 12893913769120703138, 15675160838921962454,  2511068401785704737,
    14483183001716371453,  3774730664208216065,  5083371700846102796,  9583498264570933637, 17119870085051257224,
     5217910858257235075, 10612176809475689857,  1924700483125896976,  7171619684536160599, 10949279256701751503,
    15596196964072664893, 14097948002655599357,   615821766635933047,  5636498760852923045, 17618792803942051220,
      580805356741162327,   425267967796817241,  8381470634608387938, 13212228678420887626, 16993060308636741960,
      957923366004347591,  6210242862396777185,  1012818702180800310, 15299383925974515757, 17501832009465945633,
    17453794942891241229, 15807805462076484491,  8407189590930420827,   974125122787311712,  1861591264068118966,
      997568339582634050, 18046771844467391493, 17981867688435687790,  3809841506498447207,  9460108917638135678,
    16172980638639374310,   958022432077424298,  4393365126459778813, 13408683141069553686, 13900005529547645957,
    15773550354402817866, 16475327524349230602,  6260298154874769264, 12224576659776460914,  6405294864092763507,
     7585484664713203306,  5187641382818981381, 12435998400285353380, 13554353441017344755,   646091557254529188,
    11393747116974949255, 16797249248413342857, 15713519023537495495, 12823504709579858843,  4738086532119935073,
     4429068783387643752,   585582692562183870,  1048280754023674130,  6788940719869959076, 11670856244972073775,
     2488756775360218862,  2061695363573180185,  6884655301895085032,  3566345954323888697, 12784319933059041817,
     4772468691551857254,  6864898938209826895,  7198730565322227090,  2452224231472687253, 13424792606032445807,
    10827695224855383989, 11016608897122070904, 14683280565151378358,  7077866519618824360, 17487079941198422333,
     3956319990205097495,  5804870313319323478,  8017203611194497730,  3310931575584983808,  5009341981771541845,
     6930001938490791874, 14415278059151389495, 11001114762641844083,  6715939435439735925,   411419160297131328,
     4522402260441335284,  3381955501804126859, 15935778656111987797,  4345051260540166684, 13978444093099579683,
     9219789505504949817,  9245142924137529075, 11628184459157386459,  7242398879359936370,  8511401943157540109,
    11948130810477009827,  6865450671488705049, 13965005347172621081, 15956599226522058336,  7737868921014130584,
     2107342503741411693, 15818996300425101108, 16399939197527488760, 13971145494081508107,  3910681448359868691,
     4249175367970221090,  9735751321242454020, 12418107929362160460,   241792245481991138,  5806488997649497146,
    10724207982663648949,  1121862814449214435,  1326996977123564236,  4902706567834759475, 12782714623891689967,
     7306216312942796257, 15681656478863766664,   957364844878149318,  5651946387216554503,  8197027112357634782,
     6302075516351125977, 13454588464089597862, 15638309200463515550, 10116604639722073476, 12052913535387714920,
     2889379661594013754, 15383926144832314187,  7841953313015471731, 17310575136995821873,  9820021961316981626,
    15319619724109527290, 15349724127275899898, 10511508162402504492,  6289553862380300393, 15046218882019267110,
    11772020174577005930,  3537640779967351792,  6801855569284252424, 17687268231192623388, 12968358613633237218,
     1429775571144180123, 10427377732172208413, 12155566091986788996, 16465954421598296115, 12710429690464359999,
     9547226351541565595, 12156624891403410342,  2985938688676214686, 18066917785985010959,  5975570403614438776,
    11541343163022500560, 11115388652389704592,  9499328389494710074,  9247163036769651820,  3688303938005101774,
     2210483654336887556, 15458161910089693228,  6558785204455557683,  1288373156735958118, 18433986059948829624,
     3435082195390932486, 16822351800343061990,  3120532877336962310, 16681785111062885568,  7835551710041302304,
     2612798015018627203, 15083279177152657491,  6591467229462292195, 10592706450534565444,  7438147750787157163,
      323186165595851698,  7444710627467609883,  8473714411329896576,  2782675857700189492,  3383567662400128329,
     3200233909833521327, 12897601280285604448,  3612068790453735040,  8324209243736219497, 15789570356497723463,
     1083312926512215996,  4797349136059339390,  5556729349871544986, 18266943104929747076,  1620389818516182276,
      172225355691600141,  3034352936522087096,  1266779576738385285,  3906668377244742888,  6961783143042492788,
    17159706887321247572,  4676208075243319061, 10315634697142985816, 13435140047933251189,   716076639492622016,
    13847954035438697558,  7195811275139178570, 10815312636510328870,  6214164734784158515, 16412194511839921544,
     3862249798930641332,  1005482699535576005,  4644542796609371301, 17600091057367987283,  4209958422564632034,
     5419285945389823940, 11453701547564354601,  9951588026679380114,  7425168333159839689,  8436306210125134906,
    11216615872596820107,  3681345096403933680,  5770016989916553752, 11102855936150871733, 11187980892339693935,
      396336430216428875,  6384853777489155236,  7551613839184151117, 16527062023276943109, 13429850429024956898,
     9901753960477271766,  9731501992702612259,  5217575797614661659, 10311708346636548706, 15111747519735330483,
     4353415295139137513,  1845293119018433391, 11952006873430493561,  3531972641585683893, 16852246477648409827,
    15956854822143321380, 12314609993579474774, 16763911684844598963, 16392145690385382634,  1545507136970403756,
    17771199061862790062, 12121348462972638971, 12613068545148305776,   954203144844315208,  1257976447679270605,
     3664184785462160180,  2747964788443845091, 15895917007470512307, 15552935765724302120, 16366915862261682626,
     8385468783684865323, 10745343827145102946,  2485742734157099909,   916246281077683950, 15214206653637466707,
    12895483149474345798,  1079510114301747843, 10718876134480663664,  1259990987526807294,  8326303777037206221,
    14104661172014248293, 15531278677382192198,  3874303698666230242,  3611366553819264523,  1358753803061653874,
     1552102816982246938, 14492630642488100979, 15001394966632908727,  2273140352787320862, 17843678642369606172,
     2903980458593894032, 16971437123015263604, 12969653681729206264,  3593636458822318001,  9719758956915223015,
     7437601263394568346,  3327758049015164431, 17851524109089292731, 14769614194455139039,  8017093497335662337,
    12026985381690317404,   739616144640253634, 15535375191850690266,  2418267053891303448, 15314073759564095878,
    10333316143274529509, 16565481511572123421, 16317667579273275294, 13991958187675987741,  3753596784796798785,
     9078249094693663275,  8459506356724650587, 12579909555010529099,  7827737296967050903,  5489801927693999341,
    10995988997350541459, 14721747867313883304,  7915884580303296560,  4105766302083365910, 12455549072515054554,
    13602111324515032467,  5205971628932290989,  5034622965420036444,  9134927878875794005, 11319873529597990213,
    14815445109496752058,  2266601052460299470,  5696993487088103383,  6540200741841280242,  6631495948031875490,
     5328340585170897740, 17897267040961463930,  9030000260502624168, 14285709137129830926, 12854071997824681544,
    15408328651008978682,  1063314403033437073, 13765209628446252802,   242013711116865605,  4772374239432528212,
     2515855479965038648,  5872624715703151235, 14237704570091006662,   678604024776645862, 12329607334079533339,
    17570877682732917020,  2695443415284373666,  4312672841405514468,  6454343485137106900,  8425658828390111343,
    16335501385875554899,  5551095603809016713, 11781094401885925035,  9395557946368382509,  9765123360948816956,
    18107191819981188154, 16049267500594757404, 16349966108299794199,  1040405303135858246,  2366386386131378192,
      223761048139910454, 15375217587047847934, 15231693398695187454, 12916726640254571028,  8878036960829635584,
     1626201782473074365,  5758998126998248293, 18077917959300292758, 10585588923088536745, 15072345664541731497,
     3559348759319842667, 12744591691872202375,  2388494115860283059,  6414691845696331748,  3069528498807764495,
     8737958486926519702, 18059264986425101074,  3139684427605102737, 12378931902986734693,   410666675039477949,
    12139894855769838924,  5780722552400398675,  7039346665375142557,  3020733445712569008,  2612305843503943561,
    13651771214166527665, 16478681918975800939,   566088527565499576,  4715785502295754870,  6957318344287196220,
    11645756868405128885, 13139951104358618000, 17650948583490040612, 18168787973649736637,  5486282999836125542,
     6122201977153895166, 17324241605502052782, 10063523107521105867, 17537430712468011382, 10828407533637104262,
    10294139354198325113, 12557151830240236401, 16673044307512640231, 10918020421896090419, 11077531235278014145,
     5499571814940871256,  2334252435740638702, 18177461912527387031,  2000007376901262542,  7968425560071444214,
     1472650787501520648,  3115849849651526279,  7980970700139577536, 12153253535907642097,  8109716914843248719,
     3154976533165008908,  5553369513523832559, 10345792701798576501,  3677445364544507875, 10637177623943913351,
     7380255087060498096, 14479400372337014801, 15381362583330700960,   204531043189704802, 13699106540959723942,
     3817903465872254783, 10972364467110284934,  2701394334530963810,  2931625600749229147, 16428252083632828910,
    11873166501966812913,  5566810080537233762,  7840617383807795056, 10699413880206684652, 18259119259617231436,
    10332714341486317526, 10137911902863059694,   669146221352346842,  8373571610024623455, 10620002450820868661,
    12220730820779815970,  5902974968095412898,  7931010481705150841, 16413777368097063650, 11273457888324769727,
    13719113891065284171,  8327795098009702553, 10333342364827584837,  6202832891413866653,  9137034567886143162,
    14514450826524340059,   473610156015331016,   813689571029117640, 13776316799690285717, 10429708855338427756,
     8995290140880620858,  2320123852041754384,  8082864073645003641,  6961777411740398590, 10008644283003991179,
     3239064015890722333, 16762634970725218787, 16467281536733948427, 10563290046315192938,  5108560603794851559,
    15121667220761532906, 14155440077372845941, 10050536352394623377, 15474881667376037792,  3448088038819200619,
     3692020001240358871,  6444847992258394902,  8687650838094264665,  3028124591188972359, 16945232313401161629,
    15547830510283682816,  3982930188609442149, 14270781928849894661, 13768475593433447867, 13815150225221307677,
     8502397232429564693,   718377350715476994,  7459266877697905475,  8353375565171101521,  7807281661994435472,
    16924127046922196149, 10157812396471387805,  2519858716882670232,  7384148884750265792,  8077153156180046901,
     3499231286164597752,  2700106282881469611, 14679824700835879737, 14188324938219126828,  3016120398601032793,
    10858152824243889420,  9412371965669250534,  4857522662584941069,   984331743838900386,  4094160040294753142,
     2368635764350388458, 15101240511397838657, 15584415763303953578,  7831857200208015446,  1952643641639729063,
     4184323302594028609, 16795120381104846695,  3541559381538365280, 15408472870896842474,  5628362450757896366,
    16277348886873708846, 12437047172652330846, 10172715019035948149,  1999700669649752791,  6217957085626135027,
    11220551167830336823, 16478747645632411810,  5437280487207382147, 11382378739613087836, 15866932785489521505,
     5502694314775516684, 16440179278067648435, 15510104554374162846, 15722061259110909195, 10760687291786964354,
    10736868329920212671,  4166148127664495614, 14303518358120527892,  9122250801678898571, 10028508179936801946,
      216630713752669403, 10655207865433859491,  4041437116174699233,  6280982262534375348,   297501356638818866,
    13976146806363377485, 13752396481560145603, 11472199956603637419, 16393728429143900496, 14752844047515986640,
     1524477318846038424,  6596889774254235440,  1591982099532234960,  8065146456116391065,  3964696017750868345,
    17040425970526664920, 11511165586176539991,  3443401252003315103, 16314977947073778249, 16860120454903458341,
     5370503221561340846, 15362920279125264094,  2822458124714999779, 14575378304387898337,  9689406052675046032,
     2872149351415175149, 13019620945255883050, 14929026760148695825,  8503417349692327218,  9677798905341573754,
      828949921821462483, 16110482368362750196, 15794218816553655671, 14942910774764855088, 12026350906243760195,
    13610867176871462505, 18324536557697872582,  2658962269666727629,   327225403251576027,  9207535177029277544,
     8744129291351887858,  6129603385168921503, 18385497655031085907, 13024478718952333892, 14547683159720717167,
     5932119629366981711,   325385464632594563,  3559879386019806291,  6629264948665231298, 14358245326238118181,
    15662449672706340765, 13975503159145803297,  3609534220891499022,  4224273587485638227,  9274084767162416370,
    13156843921244091998, 18284750575626858789, 14664767920489118779, 11292057742031803221, 13919998707305829132,
    14473305049457001422,  9696877879685767807,  1406758246007973837,  2429517644459056881, 14361215588101587430,
    11386164476149757528, 10474116023593331839,  2921165656527786564, 15604610369733358953, 12955027028676000544,
    10314281035410779907,  3167047178514709947,  1088721329408346700, 17930425515478182741,  7466411836095405617,
    15534027454610690575, 10879629128927506091, 11502219301371200635, 13915106894453889418,  4226784327815861027,
    12335222183627106346,  3648499746356007767, 18441388887898023393, 18117929843327093625,  4237736098094830438,
    14229123019768296655,  3930112058127932690, 12663879236019645778,  9281161952002617309,  4978473890680876319,
      845759387067546611,  1386164484606776333,  8008554770639925512, 11159581016793288971, 18065390393740782906,
    17647985458967631018,  9092379465737744314,  2914678236848656327,  4376066698447630270, 16057186499919087528,
     3031333261848790078,  2926746602873431597,  7931945763526885287,   147649915388326849, 15801792398814946230,
     5265900391686545347, 16173686275871890830,  7562781050481886043,  5853506575839330404, 14957980734704564792,
    10944286556353523404,  1783009880614150597,  9529762028588888983,   822992871011696119,  2130074274744257510,
     8000279549284809219,  3514744284158856431,   128770032569293263,  3737367602618100572, 16364836605077998543,
      783266423471782696,  4569418252658970391, 11093950688157406886, 14888808512267628166,  4217786261273670948,
    17047486076688645713, 14133826721458860485, 17539744882220127106, 12394675039129853905,  5757634999463277090,
     9621947619435861331,  1182210208559436772, 14603391040490913939, 17481976703660945893, 14063388816234683976,
     2046622692581829572,  8294969799792017441,  5293778434844788058, 17976364049306763808,   399482430848083948,
    16495545010129798933, 15241340958282367519,   989828753826900814, 17616558773874893537,  2471817920909589004,
    11764082277667899978,  9618755269550400950,  1240014743757147125,  1887649378641563002,  1842982574728131416,
    13243531042427194002,  7688268125537013927,  3080422097287486736,  2562894809975407783, 12428984115620094788,
     1355581933694478148,  9895969242586224966,  8628445623963160889,  4298916726468199239, 12773165416305557280,
     5240726258301567487,  4975412836403427561,  1842172398579595303,  7812151462958058676, 17974510987263071769,
    14980707022065991200, 18294903201142729875, 12911672684850242753,  8979482998667235743, 16808468362384462073,
     5981317232108359798, 12373702800369335100, 16119707581920094765,  2782738549717633602, 15454155188515389391,
    16495638000603654629, 16348757069342790497,  7769562861984504567, 17504300515449231559,  5557710032938318996,
    11846125204788401203, 13957316349928882624,  2738350683717432043, 15738068448047700954,  6224714837294524999,
     6081930777706411111, 11366312928059597928,  4355315799925031482, 12393324728734964015, 15277140291994338591,
     1406052433297386355, 15859448364509213398,  1672805458341158435,  2926095111610982994, 11056431822276774455,
    12083767323511977430,  3296968762229741153, 12312076899982286460, 17769284994682227273, 15349428916826953443,
     1056147296359223910, 18305757538706977431,  6214378374180465222, 14279648441175008454, 17791306410319136644,
      956593013486324072,  2921235772936241950, 10002890515925652606, 10399654693663712506,  6446247931049971441,
     6380465770144534958, 11439178472613251620, 10131486500045494660,  3692642123868351947, 10972816599561388940,
     4931112976348785580,  8213967169213816566, 15336469859637867841, 15026830342847689383,  7524668622380765825,
    17309937346758783807,   372780684412666438,  5642417144539399955, 18303842993081194577, 11085303253831702827,
    15658163165983586950,  8517521928922081563, 16091186344159989860, 17614656488010863910,  4736067146481515156,
    13449945221374241354, 17755469346196579408, 13300502638545717375,  6611828134763118043, 14177591906740276597,
     9340430243077460347,  7499765399826404087,  3409518087967832469,  9013253864026602045,  4444307427984430192,
     3729283608700519712, 13642048880719588383, 16486557958022946240,  2996465014991157904, 10020049344596426576,
    12302485648009883778,  8492591321344423126, 17407986443716172520, 10530482934957373052, 15740662350540828750,
     1790629986901049436,  6305948377669917188, 15092985352503125323,   928505047232899787, 14404651977039851607,
     7564177565277805597,  3411236815351677870,  7752718145953236134, 12315979971311483798, 12477729506691004724,
    14654956300924793305,  6689803038918974388,  1540738812233000153, 13508351811701989957, 15864432023192136053,
     7990997967273843917,  7424300239290765161,    39585249496300263,  3877436595063283319, 10710642254398044448,
     4653804418844456375,  1232267496410380283,  3690525514009038824, 15459770765077428485, 13240346522153894145,
     5674964360688390624, 16973644653010587289, 15924280764204855206, 15196708627253442662, 17596174821341373274,
    16196745023027393691,  6980050627399795351, 17582264380857746637, 18170372407506856324, 12108126025631005514,
    15687749089493373169,  5814107289258228434,  9381977959648494876, 15895601183088112734, 16267869075651604263,
    15228381979765852785, 11949618678312581999,  4545324791131029438,   582725409406225185, 15282520250746126790,
    14758446535973412711,  7605613563088071833,  1111140641057375915,  5364843095234852245,   218335432181198977,
     4891472444796201742,  4564628942836375772, 15500501278323817088,  4913946328556108657,  2684786251736694229,
    12090498456116310122,  5310885782157038567,  5032788439854011923, 12627401038822728242, 11869662610126430929,
    17650156853043540226, 12126672500118808436, 10437658933435653256, 13133995470637873311,  4601324715591152820,
     1874350460376708372,  5808688626286061164, 13777088437302430376,  5018451954762213522,  2588296738534474754,
     5503414509154170711,  5230497186769951796, 13261090710400573914,  8515217303152165705, 11074538219737365303,
    15481562385740613213, 12705484409881007350, 14221931471178549498, 12905633420087112297, 17337759164357146506,
    14081997515778175224, 17384320185513122939,  7131793076779216692, 17483217190312403109,   900692047897995877,
    14723287313048560400,  6132094372965340305,  7572797575350925726, 12725160700431903514,   380860122911632449,
     1900504978569024571,  8423729759529914138,  7305587201606052334, 12446871355267313320,  4615812356515386206,
     3361817115406652303, 17690418922000878428, 14632214537567910559,  2709702289926174775,  3459675155951086144,
     7788364399926538150, 16043992474431955950, 15830963823784930267,  4216893617835797954,   538159724689093771,
    16029152738918251363, 14444848757576686696, 12941757045272633696, 10900480525147953314, 12547307449905859302,
    16001571796892398181,   407942194622690676, 13873235372903944444, 18071603799493008777,  1015646077646778622,
     9387605808959554815, 11566702442022019410,  7061722181092883183,  2629032108249254109,  5271820053177594520,
    12640880742139693547, 10098688629735675775,  5716304472850923064,  3312674502353063071,  7295926377425759633,
      833281439103466115, 16316743519466861667,  9912050326606348167, 11651133878100804242, 18026798122431692459,
     6157758321723692663,  4856021830695749349,  7074321707293278978, 10748097797809573561,  2949954440753264783,
     9813922580940661152,  9949237950172138336, 15643982711269455885, 16078663425810239127, 12508044395364228880,
    12920301578340189344, 15368071871011048915,  1610400750626363239, 11994736084146033126,  6042574085746186088,
     4154587549267685807, 15915752367312946034,  1191196620621769193,   467437822242538360,  2836463788873877488,
    10476401302029164984,  1716169985450737419,  5327734953288310341,  3994170067185955262,   884431883768190063,
    11019001754831208284, 14322807384384895215,   161011537360955545,  1466223959660131656,  5227048585229497539,
    12410731857504225031,  2142243279080761103, 17682826799106851430,  1792612570704179953, 14727410295243056025,
     1459567192481221274,  5669760721687603135, 17507918443756456845, 10354471145847018200, 10362475129248202288,
    13143844410150939443,  6861184673150072028, 18396524361124732580,   543906666394301875, 12476817828199026728,
    11853496871128122868, 12747674713108891748,  7986179867749890282,  9158195177777627533,  2217320706811118570,
     8631389005200569973,  5538133061362648855,  3369942850878700758,  7813559982698427184,   509051590411815948,
    10197035660403006684, 13004818533162292132,  9831652587047067687,  7619315254749630976,   994412663058993407
  ) :
  (  # 32-bit randoms
    1067595299,  955945823,  477289528, 4107218783, 4228976476,
    3344332714, 3355579695,  227628506,  810200273, 2591290167,
    2560260675, 3242736208,  646746669, 1479517882, 4245472273,
    1143372638, 3863670494, 3221021970, 1773610557, 1138697238,
    1421897700, 1269916527, 2859934041, 1764463362, 3874892047,
    3965319921,   72549643, 2383988930, 2600218693, 3237492380,
    2792901476,  725331109,  605841842,  271258942,  715137098,
    3297999536, 1322965544, 4229579109, 1395091102, 3735697720,
    2101727825, 3730287744, 2950434330, 1661921839, 2895579582,
    2370511479, 1004092106, 2247096681, 2111242379, 3237345263,
    4082424759,  219785033, 2454039889, 3709582971,  835606218,
    2411949883, 2735205030,  756421180, 2175209704, 1873865952,
    2762534237, 4161807854, 3351099340,  181129879, 3269891896,
     776029799, 2218161979, 3001745796, 1866825872, 2133627728,
      34862734, 1191934573, 3102311354, 2916517763, 1012402762,
    2184831317, 4257399449, 2899497138, 3818095062, 3030756734,
    1282161629,  420003642, 2326421477, 2741455717, 1278020671,
    3744179621,  271777016, 2626330018, 2560563991, 3055977700,
    4233527566, 1228397661, 3595579322, 1077915006, 2395931898,
    1851927286, 3013683506, 1999971931, 3006888962, 1049781534,
    1488758959, 3491776230,  104418065, 2448267297, 3075614115,
    3872332600,  891912190, 3936547759, 2269180963, 2633455084,
    1047636807, 2604612377, 2709305729, 1952216715,  207593580,
    2849898034,  670771757, 2210471108,  467711165,  263046873,
    3569667915, 1042291111, 3863517079, 1464270005, 2758321352,
    3790799816, 2301278724, 3106281430,    7974801, 2792461636,
     555991332,  621766759, 1322453093,  853629228,  686962251,
    1455120532,  957753161, 1802033300, 1021534190, 3486047311,
    1902128914, 3701138056, 4176424663, 1795608698,  560858864,
    3737752754, 3141170998, 1553553385, 3367807274,  711546358,
    2475125503,  262969859,  251416325, 2980076994, 1806565895,
     969527843, 3529327173, 2736343040, 2987196734, 1649016367,
    2206175811, 3048174801, 3662503553, 3138851612, 2660143804,
    1663017612, 1816683231,  411916003, 3887461314, 2347044079,
    1015311755, 1203592432, 2170947766, 2569420716,  813872093,
    1105387678, 1431142475,  220570551, 4243632715, 4179591855,
    2607469131, 3090613241,  282341803, 1734241730, 1391822177,
    1001254810,  827927915, 1886687171, 3935097347, 2631788714,
    3905163266,  110554195, 2447955646, 3717202975, 3304793075,
    3739614479, 3059127468,  953919171, 2590123714, 1132511021,
    3795593679, 2788030429,  982155079, 3472349556,  859942552,
    2681007391, 2299624053,  647443547,  233600422,  608168955,
    3689327453, 1849778220, 1608438222, 3968158357, 2692977776,
    2851872572,  246750393, 3582818628, 3329652309, 4036366910,
    1012970930,  950780808, 3959768744, 2538550045,  191422718,
    2658142375, 3276369011, 2927737484, 1234200027, 1920815603,
    3536074689, 1535612501, 2184142071, 3276955054,  428488088,
    2378411984, 4059769550, 3913744741, 2732139246,   64369859,
    3755670074,  842839565, 2819894466, 2414718973, 1010060670,
    1839715346, 2410311136,  152774329, 3485009480, 4102101512,
    2852724304,  879944024, 1785007662, 2748284463, 1354768064,
    3267784736, 2269127717, 3001240761, 3179796763,  895723219,
     865924942, 4291570937,   89355264, 1471026971, 4114180745,
    3201939751, 2867476999, 2460866060, 3603874571, 2238880432,
    3308416168, 2072246611, 2755653839, 3773737248, 1709066580,
    4282731467, 2746170170, 2832568330,  433439009, 3175778732,
      26248366, 2551382801,  183214346, 3893339516, 1928168445,
    1337157619, 3429096554, 3275170900, 1782047316, 4264403756,
    1876594403, 4289659572, 3223834894, 1728705513, 4068244734,
    2867840287, 1147798696,  302879820, 1730407747, 1923824407,
    1180597908, 1569786639,  198796327,  560793173, 2107345620,
    2705990316, 3448772106, 3678374155,  758635715,  884524671,
     486356516, 1774865603, 3881226226, 2635213607, 1181121587,
    1508809820, 3178988241, 1594193633, 1235154121,  326117244,
    2304031425,  937054774, 2687415945, 3192389340, 2003740439,
    1823766188, 2759543402,   10067710, 1533252662, 4132494984,
      82378136,  420615890, 3467563163,  541562091, 3535949864,
    2277319197, 3330822853, 3215654174, 4113831979, 4204996991,
    2162248333, 3255093522, 2219088909, 2978279037,  255818579,
    2859348628, 3097280311, 2569721123, 1861951120, 2907080079,
    2719467166,  998319094, 2521935127, 2404125338,  259456032,
    2086860995, 1839848496, 1893547357, 2527997525, 1489393124,
    2860855349,   76448234, 2264934035,  744914583, 2586791259,
    1385380501,   66529922, 1819103258, 1899300332, 2098173828,
    1793831094,  276463159,  360132945, 4178212058,  595015228,
     177071838, 2800080290, 1573557746, 1548998935,  378454223,
    1460534296, 1116274283, 3112385063, 3709761796,  827999348,
    3580042847, 1913901014,  614021289, 4278528023, 1905177404,
      45407939, 3298183234, 1184848810, 3644926330, 3923635459,
    1627046213, 3677876759,  969772772, 1160524753, 1522441192,
     452369933, 1527502551,  832490847, 1003299676, 1071381111,
    2891255476,  973747308, 4086897108, 1847554542, 3895651598,
    2227820339, 1621250941, 2881344691, 3583565821, 3510404498,
     849362119,  862871471,  797858058, 2867774932, 2821282612,
    3272403146, 3997979905,  209178708, 1805135652,    6783381,
    2823361423,  792580494, 4263749770,  776439581, 3798193823,
    2853444094, 2729507474, 1071873341, 1329010206, 1289336450,
    3327680758, 2011491779,   80157208,  922428856, 1158943220,
    1667230961, 2461022820, 2608845159,  387516115, 3345351910,
    1495629111, 4098154157, 3156649613, 3525698599, 4134908037,
     446713264, 2137537399, 3617403512,  813966752, 1157943946,
    3734692965, 1680301658, 3180398473, 3509854711, 2228114612,
    1008102291,  486805123,  863791847, 3189125290, 1050308116,
    3777341526, 4291726501,  844061465, 1347461791, 2826481581,
     745465012, 2055805750, 4260209475, 2386693097, 2980646741,
     447229436, 2077782664, 1232942813, 4023002732, 1399011509,
    3140569849, 2579909222, 3794857471,  900758066, 2887199683,
    1720257997, 3367494931, 2668921229,  955539029, 3818726432,
    1105704962, 3889207255, 2277369307, 2746484505, 1761846513,
    2413916784, 2685127085, 4240257943, 1166726899, 4215215715,
    3082092067, 3960461946, 1663304043, 2087473241, 4162589986,
    2507310778, 1579665506,  767234210,  970676017,  492207530,
    1441679602, 1314785090, 3262202570, 3417091742, 1561989210,
    3011406780, 1146609202, 3262321040, 1374872171, 1634688712,
    1280458888, 2230023982,  419323804, 3262899800,   39783310,
    1641619040, 1700368658, 2207946628, 2571300939, 2424079766,
     780290914, 2715195096, 3390957695,  163151474, 2309534542,
    1860018424,  555755123,  280320104, 1604831083, 2713022383,
    1728987441, 3639955502,  623065489, 3828630947, 4275479050,
    3516347383, 2343951195, 2430677756,  635534992, 3868699749,
     808442435, 3070644069, 4282166003, 2093181383, 2023555632,
    1568662086, 3422372620, 4134522350, 3016979543, 3259320234,
    2888030729, 3185253876, 4258779643, 1267304371, 1022517473,
     815943045,  929020012, 2995251018, 3371283296, 3608029049,
    2018485115,  122123397, 2810669150, 1411365618, 1238391329,
    1186786476, 3155969091, 2242941310, 1765554882,  279121160,
    4279838515, 1641578514, 3796324015,   13351065,  103516986,
    1609694427,  551411743, 2493771609, 1316337047, 3932650856,
    4189700203,  463397996, 2937735066, 1855616529, 2626847990,
      55091862, 3823351211,  753448970, 4045045500, 1274127772,
    1124182256,   92039808, 2126345552,  425973257,  386287896,
    2589870191, 1987762798, 4084826973, 2172456685, 3366583455,
    3602966653, 2378803535, 2901764433, 3716929006, 3710159000,
    2653449155, 3469742630, 3096444476, 3932564653, 2595257433,
     318974657, 3146202484,  853571438,  144400272, 3768408841,
     782634401, 2161109003,  570039522, 1886241521,   14249488,
    2230804228, 1604941699, 3928713335, 3921942509, 2155806892,
     134366254,  430507376, 1924011722,  276713377,  196481886,
    3614810992, 1610021185, 1785757066,  851346168, 3761148643,
    2918835642, 3364422385, 3012284466, 3735958851, 2643153892,
    3778608231, 1164289832,  205853021, 2876112231, 3503398282,
    3078397001, 3472037921, 1748894853, 2740861475,  316056182,
    1660426908,  168885906,  956005527, 3984354789,  566521563,
    1001109523, 1216710575, 2952284757, 3834433081, 3842608301,
    2467352408, 3974441264, 3256601745, 1409353924, 1329904859,
    2307560293, 3125217879, 3622920184, 3832785684, 3882365951,
    2308537115, 2659155028, 1450441945, 3532257603, 3186324194,
    1225603425, 1124246549,  175808705, 3009142319, 2796710159,
    3651990107,  160762750, 1902254979, 1698648476, 1134980669,
     497144426, 3302689335, 4057485630, 3603530763, 4087252587,
     427812652,  286876201,  823134128, 1627554964, 3745564327,
    2589226092, 4202024494,   62878473, 3275585894, 3987124064,
    2791777159, 1916869511, 2585861905, 1375038919, 1403421920,
      60249114, 3811870450, 3021498009, 2612993202,  528933105,
    2757361321, 3341402964, 2621861700,  273128190, 4015252178,
    3094781002, 1621621288, 2337611177, 1796718448, 1258965619,
    4241913140, 2138560392, 3022190223, 4174180924,  450094611,
    3274724580,  617150026, 2704660665, 1469700689, 1341616587,
     356715071, 1188789960, 2278869135, 1766569160, 2795896635,
      57824704, 2893496380, 1235723989, 1630694347, 3927960522,
     428891364, 1814070806, 2287999787, 4125941184, 3968103889,
    3548724050, 1025597707, 1404281500, 2002212197,   92429143,
    2313943944, 2403086080, 3006180634, 3561981764, 1671860914,
    1768520622, 1803542985,  844848113, 3006139921, 1410888995,
    1157749833, 2125704913, 1789979528, 1799263423,  741157179,
    2405862309,  767040434, 2655241390, 3663420179, 2172009096,
    2511931187, 1680542666,  231857466, 1154981000,  157168255,
    1454112128, 3505872099, 1929775046, 2309422350, 2143329496,
    2960716902,  407610648, 2938108129, 2581749599,  538837155,
    2342628867,  430543915,  740188568, 1937713272, 3315215132,
    2085587024, 4030765687,  766054429, 3517641839,  689721775,
    1294158986, 1753287754, 4202601348, 1974852792,   33459103,
    3568087535, 3144677435, 1686130825, 4134943013, 3005738435,
    3599293386,  426570142,  754104406, 3660892564, 1964545167,
     829466833,  821587464, 1746693036, 1006492428, 1595312919,
    1256599985, 1024482560, 1897312280, 2902903201,  691790057,
    1037515867, 3176831208, 1968401055, 2173506824, 1089055278,
    1748401123, 2941380082,  968412354, 1818753861, 2973200866,
    3875951774, 1119354008, 3988604139, 1647155589, 2232450826,
    3486058011, 3655784043, 3759258462,  847163678, 1082052057,
     989516446, 2871541755, 3196311070, 3929963078,  658187585,
    3664944641, 2175149170, 2203709147, 2756014689, 2456473919,
    3890267390, 1293787864, 2830347984, 3059280931, 4158802520,
    1561677400, 2586570938,  783570352, 1355506163,   31495586,
    3789437343, 3340549429, 2092501630,  896419368,  671715824,
    3530450081, 3603554138, 1055991716, 3442308219, 1499434728,
    3130288473, 3639507000,   17769680, 2259741420,  487032199,
    4227143402, 3693771256, 1880482820, 3924810796,  381462353,
    4017855991, 2452034943, 2736680833, 2209866385, 2128986379,
     437874044,  595759426,  641721026, 1636065708, 3899136933,
     629879088, 3591174506,  351984326, 2638783544, 2348444281,
    2341604660, 2123933692,  143443325, 1525942256,  364660499,
     599149312,  939093251, 1523003209,  106601097,  376589484,
    1346282236, 1297387043,  764598052, 3741218111,  933457002,
    1886424424, 3219631016,  525405256, 3014235619,  323149677,
    2038881721, 4100129043, 2851715101, 2984028078, 1888574695,
    2014194741, 3515193880, 4180573530, 3461824363, 2641995497,
    3179230245, 2902294983, 2217320456, 4040852155, 1784656905,
    3311906931,   87498458, 2752971818, 2635474297, 2831215366,
    3682231106, 2920043893, 3772929704, 2816374944,  309949752,
    2383758854,  154870719,  385111597, 1191604312, 1840700563,
     872191186, 2925548701, 1310412747, 2102066999, 1504727249,
    3574298750, 1191230036, 3330575266, 3180292097, 3539347721,
     681369118, 3305125752, 3648233597,  950049240, 4173257693,
    1760124957,  512151405,  681175196,  580563018, 1169662867,
    4015033554, 2687781101,  699691603, 2673494188, 1137221356,
     123599888,  472658308, 1053598179, 1012713758, 3481064843,
    3759461013, 3981457956, 3830587662, 1877191791, 3650996736,
     988064871, 3515461600, 4089077232, 2225147448, 1249609188,
    2643151863, 3896204135, 2416995901, 1397735321, 3460025646);


# Set predetermined seed for verification test
my @base_seed = ($Config{'uvsize'} == 8)
                    ? (0x12345, 0x23456, 0x34567, 0x45678)
                    : (0x123, 0x234, 0x345, 0x456);

my $obj = Math::Random::MT::Auto->new('SEED' => \@base_seed);
if (! ok(! $@, 'Create object')) {
    diag('Failure creating object: ' . $@);
}

my $dump = $obj->dump();
my $obj2 = Object::InsideOut->pump($dump);

foreach my $rand (@base_rint) {
    is($obj2->irand, $rand => 'irand');
}


my ($LO, $HI) = (1000, 9999);
sub range
{
    return (($_[0] % (($HI + 1) - $LO)) + $LO);
}

my $robj = Math::Random::MT::Auto::Range->new('SEED' => \@base_seed,
                                              'LOW'  => $LO,
                                              'HIGH' => $HI,
                                              'TYPE' => 'INTEGER');
if (! ok(! $@, 'Create object')) {
    diag('Failure creating object: ' . $@);
}

my $dump;
if ($] > 5.006) {
    $dump = $robj->dump(1);
} else {
    $dump = $robj->dump();
}
my $robj2 = Object::InsideOut->pump($dump);

foreach my $rand (@base_rint) {
    is($robj2->rrand, range($rand) => 'rrand');
}

exit(0);

# EOF
