/*    UConverter.xs
 *
 *    $Id: UConverter.xs,v 1.8 2002/03/24 20:57:06 bstell Exp $
 *
 *    Copyright (c) 2000 Brian Stell
 *
 *    This package is free software and is provided ``as is'' without
 *    express or implied warranty. It may be used, redistributed and/or
 *    modified under the terms of the Perl Artistic License
 *    (see http://www.perl.com/perl/misc/Artistic.html)
 *
 */

extern "C" {
#include <stdio.h>
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
}

#include  "picu/picu_debug.h"
#include  "picu/picu_utf8.h"
#include  "picu/picu_util.h"
#include  "unicode/locid.h"
#include  "unicode/ucnv.h"
#include  "picu/UConverter.h"

char *UConverter_class_string = "ICU::UConverter";
static int debugLevel = 0;

MODULE = ICU::UConverter    PACKAGE = ICU::UConverter

# This requires xsubpp version 1.925 or greater
REQUIRE: 1.925

void
UConverter::DESTROY()
    CODE:
        ucnv_close(THIS);

char *
UConverter::ucnv_fromUChars(u8chars, ...)
        char *u8chars;
    PREINIT:
        UnicodeString unistring(u8chars, "UTF-8");
        const UChar *unichars = unistring.getBuffer();
        int32_t unichar_len = unistring.length();
        int32_t char_len;
        UErrorCode status = U_ZERO_ERROR;
        SV *sv_status = NULL;
        U8 *buf;
        U8 buf1[10000];
    CODE:
        if (items > 3) {
            Perl_croak(aTHX_ "Usage: UConverter::ucnv_fromUChars"
                             "(u8chars [, \\$status])");
        }
        if (items == 3) {
            if (!SVIsIntRef(ST(2))) {
                Perl_croak(aTHX_ "ICU::UConverter::ucnv_fromUChars: "
                                 "param 2 must be an integer reference");
            }
            sv_status = SvRV(ST(2));
            status = (UErrorCode)SvIV(sv_status);
            // if an error already occurred then don't do the
            // preflighting just return now
            if (U_FAILURE(status)) {
                XSRETURN_UNDEF;
            }
        }

        DEBUG100("UConverter::ucnv_fromUChars");
        IF_DEBUG100 dump_U8String((U8*)u8chars);
        DEBUG100("unichars (%d)", unichar_len); 
        IF_DEBUG100 dump_UCharString(unichars, unistring.length());

        // preflight to get the output size
        char_len = ucnv_fromUChars(THIS, NULL, 0, unichars, unichar_len, 
                                                          &status);
        if (status != U_BUFFER_OVERFLOW_ERROR) {
            DEBUG100("status = %s", u_errorName(status));
            if (sv_status) {
                sv_setiv(sv_status, (IV)status);
            }
            // if an error occurred then don't do the malloc
            // just return now
            XSRETURN_UNDEF;
        }
        // clear the preflighting error status
        status = U_ZERO_ERROR;
        DEBUG100("char_len = %d", char_len);

        // alloc space for the new string
        Newz(UCONVERTER_MEM_COOKIE, buf, char_len+1, U8);
        // now actually convert
        ucnv_fromUChars(THIS, (char*)buf, char_len+1, unichars, 
                                                  unichar_len, &status);
        if (sv_status) {
            sv_setiv(sv_status, (IV)status);
        }
        if (U_FAILURE(status)) {
            DEBUG100("status = %s", u_errorName(status));
            Safefree(buf);
            XSRETURN_UNDEF;
        }
        DEBUG100("buf"); 
        IF_DEBUG100 dump_CharStringAsHex((const char *)buf);

        ST(0) = sv_newmortal();
        sv_usepvn(ST(0), (char*)buf, char_len);
        RETVAL = (char *)buf;
    OUTPUT:
        RETVAL

## try to use a stack based buffer 
## to avoid malloc'ing a temporary buffer
#define LOCAL_U16BUF_SIZE 8192

char *
UConverter::ucnv_toUChars(lchars, ...)
        char *lchars;
    PREINIT:
        int32_t lchar_len = strlen(lchars);
        UErrorCode status = U_ZERO_ERROR;
        SV *sv_status = NULL;
        UChar local_u16buf[LOCAL_U16BUF_SIZE];
        UChar *u16buf = local_u16buf;
        int32_t u16buf_len;
        const UChar *uchars_left;
        U8 *u8str;
        int32_t u8str_len;
    CODE:
        if (items > 3) {
            Perl_croak(aTHX_ "Usage: UConverter::ucnv_fromUChars"
                             "(u8chars [, \\$status])");
        }
        if (items == 3) {
            if (!SVIsIntRef(ST(2))) {
                Perl_croak(aTHX_ "ICU::UConverter::ucnv_fromUChars: "
                                 "param 2 must be an integer reference");
            }
            sv_status = SvRV(ST(2));
            status = (UErrorCode)SvIV(sv_status);
            // if an error already occurred the just return now
            if (U_FAILURE(status)) {
                XSRETURN_UNDEF;
            }
        }

        DEBUG100("lchars"); 
        IF_DEBUG100 dump_CharStringAsHex((const char *)lchars);

        // preflight to get the UTF16 size
        u16buf_len = ucnv_toUChars(THIS, NULL, 0, lchars, lchar_len, 
                                                               &status);
        if (status != U_BUFFER_OVERFLOW_ERROR) {
            DEBUG100("status = %s", u_errorName(status));
            if (sv_status) {
                sv_setiv(sv_status, (IV)status);
            }
            XSRETURN_UNDEF;
        }
        status = U_ZERO_ERROR;

        // alloc UTF16 space
        // we could probably make this a loop to avoid any malloc of a 
        // temp buffer
        if (u16buf_len > LOCAL_U16BUF_SIZE) { // try to avoid malloc'ing
            Newz(UCONVERTER_MEM_COOKIE, u16buf, u16buf_len+1, UChar);
        }
        // now actually convert
        ucnv_toUChars(THIS, u16buf, u16buf_len+1, lchars, lchar_len, 
                                                              &status);
        if (sv_status) {
            sv_setiv(sv_status, (IV)status);
        }
        if (U_FAILURE(status)) {
            DEBUG100("status = %s", u_errorName(status));
            XSRETURN_UNDEF;
            if (u16buf_len > LOCAL_U16BUF_SIZE) {
                Safefree(u16buf);
            }
        }
        u16buf[u16buf_len] = 0;
        // now convert to UTF8 for output
        u8str_len = UCharStringToUTF8strlen(u16buf, u16buf_len);
        // alloc space for the new string
        Newz(UCONVERTER_MEM_COOKIE, u8str, u8str_len+1, U8);
        uchars_left = UCharStringToUTF8String((const UChar *)u16buf, u16buf_len,
                                              (char*)u8str, u8str_len+1);
        if (u16buf_len > LOCAL_U16BUF_SIZE) {
            Safefree(u16buf);
        }
        ST(0) = sv_newmortal();
        sv_usepvn(ST(0), (char*)u8str, u8str_len);
        RETVAL = (char *)u8str;
    OUTPUT:
        RETVAL

UConverter *
UConverter::new(convertName, ...)
        const char *convertName;
    PREINIT:
        UErrorCode status = U_ZERO_ERROR;
        SV *sv_status = NULL;
    CODE:
        if (items > 3) {
            Perl_croak(aTHX_ "Usage: new UConverter(converter_name "
                             " [, \\$status])");
        }
        if (items == 3) {
            if (!SVIsIntRef(ST(2))) {
                Perl_croak(aTHX_ "ICU::UConverter::new: param 2 must "
                                 "be an integer reference");
            }
            sv_status = SvRV(ST(2));
            status = (UErrorCode)SvIV(sv_status);
            if (U_FAILURE(status)) {
                XSRETURN_UNDEF;
            }
        }
        DEBUG100("convertName = \"%s\"", convertName);
        RETVAL = ucnv_open(convertName, &status);
        if (sv_status) {
            sv_setiv(sv_status, (IV)status);
        }
        if (U_FAILURE(status)) {
            DEBUG100("status = %s", u_errorName(status));
            XSRETURN_UNDEF;
        }
    OUTPUT:
        RETVAL

void
UConverter::reset()
    CODE:
        ucnv_reset(THIS);

int
setDebugLevel(...)
    CODE:
        RETVAL = debugLevel; // return old value
        if (items > 1) {
            Perl_croak(aTHX_ "Usage: ICU::UConverter::setDebugLevel"
                             "([new_debug_level])");
        }
        else if (items == 1) {
            debugLevel = SvIV(ST(0));
        }
        DEBUG100("debugLevel = %d", debugLevel);
    OUTPUT:
        RETVAL

