#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util qw/powersum/;

my $usexs = Math::Prime::Util::prime_get_config->{'xs'};
my $extra = defined $ENV{EXTENDED_TESTING} && $ENV{EXTENDED_TESTING};

my @btests = (
  [32,
    [qw/1624 2 1426381124 1429018500 1431659125/],
    [qw/67 4 260049713 280200834 301582210/],
    [qw/44 5 1128489076 1293405300 1477933425/],
    [qw/19 6 105409929 152455810 216455810/],
    [qw/17 7 680856256 1091194929 1703414961/],
    [qw/9 8 24684612 67731333 167731333/],
  ],
  [64,
    [qw/2642245 2 6148904570708739070 6148911552167379095 6148918533631303611/],
    [qw/5724 4 1228396075811803854 1229469567040762830 1230543808634903455/],
    [qw/1824 5 6127498198691333376 6147687633902880000 6167932473678270625/],
    [qw/482 6 857181367123038321 869720925148346545 882417389116663114/],
    [qw/288 7 5834458368653531904 5998799932115786496 6167177758675187425/],
    [qw/115 8 375757354506665017 406347583132055642 439131732051867738/],
  ],
);

plan tests => 3 + 3 + 1 + scalar(@btests) + 2;

is_deeply([map { powersum(0,$_) } 0..9], [map { 0 } 0..9], "powersum(0,n) = 0");
is_deeply([map { powersum(1,$_) } 0..9], [map { 1 } 0..9], "powersum(1,n) = 1");
is_deeply([map { powersum($_,0) } 1..9], [map { $_ } 1..9], "powersum(n,0) = 1 for n>0");

is_deeply([map { powersum(10,$_) } 0..5], [qw/10 55 385 3025 25333 220825/], "powersum(10,0..5)");
is_deeply([map { "".powersum(16,$_) } 0..15], [qw/16 136 1496 18496 243848 3347776 47260136 680856256 9961449608 147520415296 2206044295976 33254351828416 504635269460168 7701103353860416 118091609375661416 1818399978159990976/], "powersum(16,0..15)");
is_deeply([map { "".powersum(1711,$_) } 0..9], [qw/1711 1464616 1671126856 2145100027456 2937070623367384 4188996380716653376 6145256641557655809016 9202904066587083191010496 14000681533539207660658782424 21565944278446437945896721933376/], "powersum(1711,0..9)");

subtest 'Tests used by Math::AnyNum', sub {
  is("".powersum(97, 20),   '27930470253682554320726764539206479400753', "powersum(97,20)");
  is("".powersum(1234, 13), '1363782530586069716227147685797600627310545', "powersum(1234,13)");
  is("".powersum(30, 80),   '15824906698911682552620450221533100599157410235977820404994404262610329210567189683421455768203096083923986638110352399', "powersum(30,80)");
  is("".powersum('36893488147419103232', 6), '13290765244262525999877070971093849105865118528347431876799549931828154109852970889789225381341531108777505296823405714971493113182289920', "powersum(36893488147419103232,6)");
};

for my $btest (@btests) {
  my($bits,@tarr) = @$btest;
  subtest "Window around $bits-bit analytic boundaries", sub {
    for my $test (@tarr) {
      my($n,$k,@exp) = @$test;
      my @got = (powersum($n-1,$k), powersum($n,$k), powersum($n+1,$k));
      is_deeply(\@got, \@exp, "powersum($n,$k) = $exp[1]");
    }
  };
}

is_deeply( ["".powersum(115,4), "".powersum(116,4), "".powersum(117,4)],
           [qw/4110671642 4291735578 4479124299/],
           "powersum({115,116,117},4) correct [test 32-bit overflow]" );
is_deeply( ["".powersum(9838,4), "".powersum(9839,4), "".powersum(9840,4)],
           [qw/18436328665468109431 18445700051808767672 18455075248624127672/],
           "powersum({9838,9839,9840},4) correct [test 64-bit overflow]" );
