#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util qw/lshiftint rshiftint rashiftint/;

# Negative right shifts:
#   ">>"    MPU, Pari/GP, Mathematica   = -rshiftint(-n,k]
#   ">>a"   Math::BigInt, Python, Java
my @negshifts = (
  # n, k,  <<,   >>,  >>arith
  [ 0, 1,  0,    0,   0],
  [-1, 1,  -2,   0,  -1],
  [-5, 1,  -10, -2,  -3],
  [-8, 2,  -32, -2,  -2],

  [qw/-65535 15 -2147450880 -1 -2/],
  [qw/-65536 15 -2147483648 -2 -2/],
  [qw/-65535 16 -4294901760 0 -1/],
  [qw/-65536 16 -4294967296 -1 -1/],

  [qw/-65535 47 -9223231299366420480 0 -1/],   #  8
  [qw/-65536 47 -9223372036854775808 0 -1/],   #  9
  [qw/-65535 48 -18446462598732840960 0 -1/],  # 10
  [qw/-65536 48 -18446744073709551616 0 -1/],  # 11
  [qw/-65536 80 -79228162514264337593543950336 0 -1/],  # 12

  [qw/-307385513 6 -19672672832 -4802898 -4802899/],
  [qw/-637526413 6 -40801690432 -9961350 -9961351/],
  [qw/-2045651239 6 -130921679296 -31963300 -31963301/],
  [qw/-3675663743 6 -235242479552 -57432245 -57432246/],
  [qw/-2332267979728172537 6 -149265150702603042368 -36441687183252695 -36441687183252696/],
  [qw/-8408654401686460807 6 -538153881707933491648 -131385225026350950 -131385225026350951/],
  [qw/-17640827963513397449 6 -1129012989664857436736 -275637936929896835 -275637936929896836/],
  [qw/-32659506018295865747 6 -2090208385170935407808 -510304781535872902 -510304781535872903/],
  [qw/-79231600218559026832557301750107210001 6 -5070822413987777717283667312006861440064 -1237993753414984794258707839845425156 -1237993753414984794258707839845425157/],
  [qw/-131954888069700539887213633881194728277 6 -8445112836460834552781672568396462609728 -2061795126089070935737713029393667629 -2061795126089070935737713029393667630/],
  [qw/-254262665582332530470619504253273698569 6 -16272810597269281950119648272209516708416 -3972854149723945788603429753957401540 -3972854149723945788603429753957401541/],
  [qw/-416649423645764932216789232242651032187 6 -26665563113328955661874510863529666059968 -6510147244465077065887331753791422377 -6510147244465077065887331753791422378/],
);

plan tests => 4 + 3 + 2;

###### lshiftint
is_deeply([map { lshiftint($_) } 0..50], [map { $_ << 1 } 0..50], "lshiftint(0..50)");
is_deeply([map { rshiftint($_) } 0..50], [map { $_ >> 1 } 0..50], "rshiftint(0..50)");
is_deeply([map { rashiftint($_) } 0..50], [map { $_ >> 1 } 0..50], "rashiftint(0..50)");
is_deeply([map { lshiftint($_,5) } -65 .. 65], [map { $_ * 32 } -65 .. 65], "lshiftint(-65 .. 65, 5)");

# lshiftint for native size k is:  mulint($n, 1 << $k)
# but for testing we want to avoid using our other functions.

is_deeply( [map { "".lshiftint($_->[0], $_->[1]) } @negshifts],
           [map { $_->[2] } @negshifts],
           "left shift negative inputs" );
is_deeply( [map { "".rshiftint($_->[0], $_->[1]) } @negshifts],
           [map { $_->[3] } @negshifts],
           "right shift negative inputs" );
is_deeply( [map { "".rashiftint($_->[0], $_->[1]) } @negshifts],
           [map { $_->[4] } @negshifts],
           "signed arithmetic right shift negative inputs" );

is("".lshiftint("2147483648"),"4294967296","left shift of 2^31 with implied 1 bit");
is("".lshiftint("9223372036854775808"),"18446744073709551616","left shift of 2^63 with implied 1 bit");
