/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.spark.k8s.operator.utils;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Field;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

import org.apache.spark.k8s.operator.config.ConfigOption;
import org.apache.spark.k8s.operator.config.SparkOperatorConf;

@Slf4j
public class ConfOptionDocGenerator {
  public static final String CONF_FILE_NAME = "config_properties.md";
  public static final String DEFAULT_DOCS_PATH = "docs";
  public static final String GENERATED_FILE_HEADER =
      "This doc is automatically generated by gradle task, manual updates would be overridden.";

  public void generate(String docsPath) throws IOException, IllegalAccessException {
    Field[] fields = SparkOperatorConf.class.getDeclaredFields();
    File docsDir = new File(docsPath);
    if (!docsDir.exists() && docsDir.mkdir()) {
      log.info("Creating docs directory at {}", docsPath);
    }
    File generated = new File(docsPath, CONF_FILE_NAME);
    if (generated.createNewFile()) {
      log.info("Creating props at {}/{}", docsPath, CONF_FILE_NAME);
    }
    PrintWriter printWriter = new PrintWriter(generated, "UTF-8");
    printWriter.println(String.format("[//]: # (%s)", GENERATED_FILE_HEADER));
    printWriter.println("# Spark Operator Config Properties");
    DocTable table =
        DocTable.builder()
            .headers(List.of("Key", "Type", "Default Value", "Allow Hot Reloading", "Description"))
            .columns(5)
            .build();
    for (Field f : fields) {
      if (ConfigOption.class.isAssignableFrom(f.getType())) {
        ConfigOption<?> conf = (ConfigOption<?>) f.get(this);
        table.addRow(
            List.of(
                conf.getKey(),
                conf.getTypeParameterClass().getSimpleName(),
                conf.getDefaultValue().toString(),
                String.valueOf(conf.isEnableDynamicOverride()),
                conf.getDescription()));
      }
    }
    table.flush(printWriter);
    printWriter.close();
  }

  public static void main(String[] args) throws IOException, IllegalAccessException {
    ConfOptionDocGenerator generator = new ConfOptionDocGenerator();
    String docsPath = DEFAULT_DOCS_PATH;
    if (args.length > 0) {
      docsPath = args[0];
    }
    try {
      generator.generate(docsPath);
    } catch (IOException | IllegalAccessException e) {
      log.error("Failed to generate docs for config props.", e);
      throw e;
    }
  }
}
