/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.servicecomb.transport.rest.servlet;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.servicecomb.common.rest.RestConst;
import org.apache.servicecomb.foundation.common.LegacyPropertyFactory;
import org.apache.servicecomb.foundation.common.exceptions.ServiceCombException;
import org.apache.servicecomb.foundation.common.utils.ClassLoaderScopeContext;
import org.apache.servicecomb.registry.definition.DefinitionConst;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Test;
import org.junit.jupiter.api.Assertions;
import org.mockito.Mockito;
import org.springframework.core.env.Environment;

import jakarta.servlet.MultipartConfigElement;
import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletRegistration;
import jakarta.servlet.ServletRegistration.Dynamic;
import jakarta.servlet.http.HttpServlet;
import mockit.Expectations;
import mockit.Mock;
import mockit.MockUp;
import mockit.Mocked;

public class TestServletUtils {
  Environment environment = Mockito.mock(Environment.class);

  @Before
  public void setUp() {
    Mockito.when(environment.getProperty(RestConst.UPLOAD_DIR, RestConst.UPLOAD_DEFAULT_DIR))
        .thenReturn(RestConst.UPLOAD_DEFAULT_DIR);
    Mockito.when(environment.getProperty(RestConst.UPLOAD_MAX_FILE_SIZE, long.class, -1L))
        .thenReturn(-1L);
    Mockito.when(environment.getProperty(RestConst.UPLOAD_MAX_SIZE, long.class, -1L))
        .thenReturn(-1L);
    Mockito.when(environment.getProperty(RestConst.UPLOAD_FILE_SIZE_THRESHOLD, int.class, 0))
        .thenReturn(0);
    LegacyPropertyFactory.setEnvironment(environment);
  }

  @Test
  public void testCheckUrlPatternNormal() {
    ServletUtils.checkUrlPattern("/*");
    ServletUtils.checkUrlPattern("/abc/*");
    ServletUtils.checkUrlPattern("/abc/def/*");

    // normal, must not throw exception, no need to check
  }

  @Test
  public void testCheckUrlPatternMiddleWideChar() {
    try {
      ServletUtils.checkUrlPattern("/abc/*def");
      Assertions.fail("must throw exception");
    } catch (ServiceCombException e) {
      Assertions.assertEquals("only support rule like /* or /path/* or /path1/path2/* and so on.", e.getMessage());
    }
  }

  @Test
  public void testCheckUrlPatternNoWideChar() {
    try {
      ServletUtils.checkUrlPattern("/abcdef");
      Assertions.fail("must throw exception");
    } catch (ServiceCombException e) {
      Assertions.assertEquals("only support rule like /* or /path/* or /path1/path2/* and so on.", e.getMessage());
    }
  }

  @Test
  public void testCheckUrlPatternNotStartWithSlash() {
    try {
      ServletUtils.checkUrlPattern("abcdef/*");
      Assertions.fail("must throw exception");
    } catch (ServiceCombException e) {
      Assertions.assertEquals("only support rule like /* or /path/* or /path1/path2/* and so on.", e.getMessage());
    }
  }

  @Test
  public void testFilterUrlPatternsNormal() {
    String urlPattern = "/r1/*";

    Collection<String> urlPatterns = Arrays.asList(urlPattern);
    String[] result = ServletUtils.filterUrlPatterns(urlPatterns);
    MatcherAssert.assertThat(result, Matchers.arrayContaining("/r1/*"));

    result = ServletUtils.filterUrlPatterns(urlPattern);
    MatcherAssert.assertThat(result, Matchers.arrayContaining("/r1/*"));
  }

  @Test
  public void testFilterUrlPatternsEmpty() {
    Collection<String> urlPatterns = Arrays.asList(" ", "\t");
    String[] result = ServletUtils.filterUrlPatterns(urlPatterns);
    MatcherAssert.assertThat(result, Matchers.emptyArray());
  }

  @Test
  public void testFilterUrlPatternsInvalid() {
    Collection<String> urlPatterns = Arrays.asList("/abc");
    try {
      ServletUtils.filterUrlPatterns(urlPatterns);
      Assertions.fail("must throw exception");
    } catch (ServiceCombException e) {
      Assertions.assertEquals("only support rule like /* or /path/* or /path1/path2/* and so on.", e.getMessage());
    }
  }

  @Test
  public void testcollectUrlPatternsNoRestServlet(@Mocked ServletContext servletContext,
      @Mocked ServletRegistration servletRegistration) {
    new Expectations() {
      {
        servletRegistration.getClassName();
        result = "test";
        servletContext.getServletRegistrations();
        result = Collections.singletonMap("test", servletRegistration);
      }
    };

    String[] result = ServletUtils.collectUrlPatterns(servletContext, RestServlet.class);
    MatcherAssert.assertThat(result, Matchers.emptyArray());
  }

  @Test
  public void testcollectUrlPatternsNormalMapping(@Mocked ServletContext servletContext,
      @Mocked ServletRegistration r1, @Mocked ServletRegistration r2) {
    Map<String, ServletRegistration> servletRegistrationMap = new LinkedHashMap<>();
    servletRegistrationMap.put("r1", r1);
    servletRegistrationMap.put("r2", r2);

    new Expectations() {
      {
        r1.getClassName();
        result = RestServlet.class.getName();
        r1.getMappings();
        result = Arrays.asList("/r1/*", "/r1/1/*");

        r2.getClassName();
        result = RestServlet.class.getName();

        servletContext.getServletRegistrations();
        result = servletRegistrationMap;
      }
    };

    String[] result = ServletUtils.collectUrlPatterns(servletContext, RestServlet.class);
    MatcherAssert.assertThat(result, Matchers.arrayContaining("/r1/*", "/r1/1/*"));
  }

  @Test
  public void testSaveUrlPrefixNull(@Mocked ServletContext servletContext) {
    ClassLoaderScopeContext.clearClassLoaderScopeProperty();

    ServletUtils.saveUrlPrefix(servletContext);

    Assertions.assertNull(ClassLoaderScopeContext.getClassLoaderScopeProperty(DefinitionConst.URL_PREFIX));
    ClassLoaderScopeContext.clearClassLoaderScopeProperty();
  }

  @Test
  public void testSaveUrlPrefixNormal(@Mocked ServletContext servletContext,
      @Mocked ServletRegistration servletRegistration) {
    ClassLoaderScopeContext.clearClassLoaderScopeProperty();
    new Expectations() {
      {
        servletContext.getContextPath();
        result = "/root";
        servletRegistration.getClassName();
        result = RestServlet.class.getName();
        servletRegistration.getMappings();
        result = Arrays.asList("/rest/*");
        servletContext.getServletRegistrations();
        result = Collections.singletonMap("test", servletRegistration);
      }
    };

    ServletUtils.saveUrlPrefix(servletContext);

    MatcherAssert.assertThat(ClassLoaderScopeContext.getClassLoaderScopeProperty(DefinitionConst.URL_PREFIX),
        Matchers.is("/root/rest"));
    ClassLoaderScopeContext.clearClassLoaderScopeProperty();
  }

  @Test
  public void createUploadDir_relative(@Mocked ServletContext servletContext) throws IOException {
    File tempDir = Files.createTempDirectory("temp").toFile();
    new Expectations() {
      {
        servletContext.getAttribute(ServletContext.TEMPDIR);
        result = tempDir;
      }
    };

    File expectDir = new File(tempDir, "upload");
    Assertions.assertFalse(expectDir.exists());

    File dir = ServletUtils.createUploadDir(servletContext, "upload");
    Assertions.assertTrue(expectDir.exists());
    Assertions.assertEquals(expectDir.getAbsolutePath(), dir.getAbsolutePath());

    dir.delete();
    Assertions.assertFalse(expectDir.exists());

    tempDir.delete();
    Assertions.assertFalse(tempDir.exists());
  }

  @Test
  public void createUploadDir_absolute(@Mocked ServletContext servletContext) throws IOException {
    File tempDir = Files.createTempDirectory("temp").toFile();

    File expectDir = new File(tempDir, "upload");
    Assertions.assertFalse(expectDir.exists());

    File dir = ServletUtils.createUploadDir(servletContext, expectDir.getAbsolutePath());
    Assertions.assertTrue(expectDir.exists());
    Assertions.assertEquals(expectDir.getAbsolutePath(), dir.getAbsolutePath());

    dir.delete();
    Assertions.assertFalse(expectDir.exists());

    tempDir.delete();
    Assertions.assertFalse(tempDir.exists());
  }

  @Test
  public void setServletParameters_notSupportUpload() {
    // not support upload will not set parameters to servlet, so servletContext is null will not throw exception
    ServletUtils.setServletParameters(null, environment);
  }

  @Test
  public void setServletParameters_supportUpload(@Mocked ServletContext servletContext, @Mocked Dynamic d1,
      @Mocked ServletRegistration d2) throws IOException {
    Map<String, ServletRegistration> servletRegistrations = new HashMap<>();
    servletRegistrations.put("d1", d1);
    servletRegistrations.put("d2", d2);
    new Expectations() {
      {
        servletContext.getServletRegistrations();
        result = servletRegistrations;
        d1.getClassName();
        result = RestServlet.class.getName();
        d2.getClassName();
        result = HttpServlet.class.getName();
      }
    };

    List<MultipartConfigElement> multipartConfigs = new ArrayList<>();
    new MockUp<Dynamic>(d1) {
      @Mock
      void setMultipartConfig(MultipartConfigElement multipartConfig) {
        multipartConfigs.add(multipartConfig);
      }
    };

    File tempDir = Files.createTempDirectory("temp").toFile();
    File uploadDir = new File(tempDir, "upload");
    Mockito.when(environment.getProperty(RestConst.UPLOAD_DIR, RestConst.UPLOAD_DEFAULT_DIR))
        .thenReturn(uploadDir.getAbsolutePath());

    ServletUtils.setServletParameters(servletContext, environment);

    Assertions.assertEquals(1, multipartConfigs.size());

    MultipartConfigElement multipartConfigElement = multipartConfigs.get(0);
    Assertions.assertEquals(uploadDir.getAbsolutePath(), multipartConfigElement.getLocation());
    Assertions.assertEquals(-1, multipartConfigElement.getMaxFileSize());
    Assertions.assertEquals(-1, multipartConfigElement.getMaxRequestSize());
    Assertions.assertEquals(0, multipartConfigElement.getFileSizeThreshold());

    uploadDir.delete();
    tempDir.delete();
    Assertions.assertFalse(tempDir.exists());
  }
}
