/****************************************************************************
 * boards/arm/stm32f7/steval-eth001v1/include/board.h
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

#ifndef __BOARDS_ARM_STM32F7_STEVAL_ETH001V1_INCLUDE_BOARD_H
#define __BOARDS_ARM_STM32F7_STEVAL_ETH001V1_INCLUDE_BOARD_H

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#ifndef __ASSEMBLY__
#  include <stdint.h>
#endif

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

/* Clocking *****************************************************************/

/* HSI: 16 MHz RC factory-trimmed
 * LSI: 32 KHz RC
 * HSE: On-board crystal frequency is 26MHz
 * LSE: 32.768 kHz
 */

#define STM32_BOARD_XTAL        26000000ul

#define STM32_HSI_FREQUENCY     16000000ul
#define STM32_LSI_FREQUENCY     32000
#define STM32_HSE_FREQUENCY     STM32_BOARD_XTAL
#define STM32_LSE_FREQUENCY     32768

/* Main PLL Configuration.
 *
 * PLL source is HSE = 26,000,000
 *
 * PLL_VCO = (STM32_HSE_FREQUENCY / PLLM) * PLLN
 * Subject to:
 *
 *     2 <= PLLM <= 63
 *   192 <= PLLN <= 432
 *   192 MHz <= PLL_VCO <= 432MHz
 *
 * SYSCLK  = PLL_VCO / PLLP
 * Subject to
 *
 *   PLLP = {2, 4, 6, 8}
 *   SYSCLK <= 216 MHz
 *
 * USB OTG FS, SDMMC and RNG Clock = PLL_VCO / PLLQ
 * Subject to
 *   The USB OTG FS requires a 48 MHz clock to work correctly. The SDMMC
 *   and the random number generator need a frequency lower than or equal
 *   to 48 MHz to work correctly.
 *
 * 2 <= PLLQ <= 15
 */

/* Highest SYSCLK
 *
 * PLL_VCO = (26,000,000 / 26) * 432 = 432 MHz
 * SYSCLK  = 432 MHz / 2 = 216 MHz
 */

#define STM32_PLLCFG_PLLM       RCC_PLLCFG_PLLM(26)
#define STM32_PLLCFG_PLLN       RCC_PLLCFG_PLLN(432)
#define STM32_PLLCFG_PLLP       RCC_PLLCFG_PLLP_2
#define STM32_PLLCFG_PLLQ       RCC_PLLCFG_PLLQ(10)

#define STM32_VCO_FREQUENCY     ((STM32_HSE_FREQUENCY / 26) * 432)
#define STM32_SYSCLK_FREQUENCY  (STM32_VCO_FREQUENCY / 2)

/* Configure Dedicated Clock Configuration Register */

#define STM32_RCC_DCKCFGR1_PLLI2SDIVQ  RCC_DCKCFGR1_PLLI2SDIVQ(0)
#define STM32_RCC_DCKCFGR1_PLLSAIDIVQ  RCC_DCKCFGR1_PLLSAIDIVQ(0)
#define STM32_RCC_DCKCFGR1_PLLSAIDIVR  RCC_DCKCFGR1_PLLSAIDIVR(1)
#define STM32_RCC_DCKCFGR1_SAI1SRC     0
#define STM32_RCC_DCKCFGR1_SAI2SRC     0
#define STM32_RCC_DCKCFGR1_TIMPRESRC   0
#define STM32_RCC_DCKCFGR1_DFSDM1SRC   0
#define STM32_RCC_DCKCFGR1_ADFSDM1SRC  0

/* Configure Dedicated Clock Configuration Register 2 */

#define STM32_RCC_DCKCFGR2_USART1SRC  RCC_DCKCFGR2_USART1SEL_APB
#define STM32_RCC_DCKCFGR2_USART2SRC  RCC_DCKCFGR2_USART2SEL_APB
#define STM32_RCC_DCKCFGR2_UART4SRC   RCC_DCKCFGR2_UART4SEL_APB
#define STM32_RCC_DCKCFGR2_UART5SRC   RCC_DCKCFGR2_UART5SEL_APB
#define STM32_RCC_DCKCFGR2_USART6SRC  RCC_DCKCFGR2_USART6SEL_APB
#define STM32_RCC_DCKCFGR2_UART7SRC   RCC_DCKCFGR2_UART7SEL_APB
#define STM32_RCC_DCKCFGR2_UART8SRC   RCC_DCKCFGR2_UART8SEL_APB
#define STM32_RCC_DCKCFGR2_I2C1SRC    RCC_DCKCFGR2_I2C1SEL_HSI
#define STM32_RCC_DCKCFGR2_I2C2SRC    RCC_DCKCFGR2_I2C2SEL_HSI
#define STM32_RCC_DCKCFGR2_I2C3SRC    RCC_DCKCFGR2_I2C3SEL_HSI
#define STM32_RCC_DCKCFGR2_I2C4SRC    RCC_DCKCFGR2_I2C4SEL_HSI
#define STM32_RCC_DCKCFGR2_LPTIM1SRC  RCC_DCKCFGR2_LPTIM1SEL_APB
#define STM32_RCC_DCKCFGR2_CECSRC     RCC_DCKCFGR2_CECSEL_HSI
#define STM32_RCC_DCKCFGR2_CK48MSRC   RCC_DCKCFGR2_CK48MSEL_PLLSAI
#define STM32_RCC_DCKCFGR2_SDMMCSRC   RCC_DCKCFGR2_SDMMCSEL_48MHZ
#define STM32_RCC_DCKCFGR2_SDMMC2SRC  RCC_DCKCFGR2_SDMMC2SEL_48MHZ

/* Several prescalers allow the configuration of the two AHB buses, the
 * high-speed APB (APB2) and the low-speed APB (APB1) domains. The maximum
 * frequency of the two AHB buses is 216 MHz while the maximum frequency of
 * the high-speed APB domains is 108 MHz. The maximum allowed frequency of
 * the low-speed APB domain is 54 MHz.
 */

/* AHB clock (HCLK) is SYSCLK (216 MHz) */

#define STM32_RCC_CFGR_HPRE     RCC_CFGR_HPRE_SYSCLK  /* HCLK  = SYSCLK / 1 */
#define STM32_HCLK_FREQUENCY    STM32_SYSCLK_FREQUENCY

/* APB1 clock (PCLK1) is HCLK/4 (54 MHz) */

#define STM32_RCC_CFGR_PPRE1    RCC_CFGR_PPRE1_HCLKd4     /* PCLK1 = HCLK / 4 */
#define STM32_PCLK1_FREQUENCY   (STM32_HCLK_FREQUENCY/4)

/* Timers driven from APB1 will be twice PCLK1 */

#define STM32_APB1_TIM2_CLKIN   (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM3_CLKIN   (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM4_CLKIN   (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM5_CLKIN   (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM6_CLKIN   (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM7_CLKIN   (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM12_CLKIN  (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM13_CLKIN  (2*STM32_PCLK1_FREQUENCY)
#define STM32_APB1_TIM14_CLKIN  (2*STM32_PCLK1_FREQUENCY)

/* APB2 clock (PCLK2) is HCLK/2 (108MHz) */

#define STM32_RCC_CFGR_PPRE2    RCC_CFGR_PPRE2_HCLKd2     /* PCLK2 = HCLK / 2 */
#define STM32_PCLK2_FREQUENCY   (STM32_HCLK_FREQUENCY/2)

/* Timers driven from APB2 will be twice PCLK2 */

#define STM32_APB2_TIM1_CLKIN   (2*STM32_PCLK2_FREQUENCY)
#define STM32_APB2_TIM8_CLKIN   (2*STM32_PCLK2_FREQUENCY)
#define STM32_APB2_TIM9_CLKIN   (2*STM32_PCLK2_FREQUENCY)
#define STM32_APB2_TIM10_CLKIN  (2*STM32_PCLK2_FREQUENCY)
#define STM32_APB2_TIM11_CLKIN  (2*STM32_PCLK2_FREQUENCY)

/* FLASH wait states
 *
 *  --------- ---------- -----------
 *  VDD       MAX SYSCLK WAIT STATES
 *  --------- ---------- -----------
 *  1.7-2.1 V   180 MHz    8
 *  2.1-2.4 V   216 MHz    9
 *  2.4-2.7 V   216 MHz    8
 *  2.7-3.6 V   216 MHz    7
 *  --------- ---------- -----------
 */

#define BOARD_FLASH_WAITSTATES 7

/* DMA Channel/Stream Selections ********************************************/

/* ADC 1 */

#define ADC1_DMA_CHAN           DMAMAP_ADC1_1

/* Alternate function pin selections ****************************************/

/* ADC1 */

#define GPIO_ADC1_IN0   GPIO_ADC1_IN0_0   /* PA0 */
#define GPIO_ADC1_IN1   GPIO_ADC1_IN1_0   /* PA1 */
#define GPIO_ADC1_IN2   GPIO_ADC1_IN2_0   /* PA2 */
#define GPIO_ADC1_IN3   GPIO_ADC1_IN3_0   /* PA3 */
#define GPIO_ADC1_IN4   GPIO_ADC1_IN4_0   /* PA4 */
#define GPIO_ADC1_IN5   GPIO_ADC1_IN5_0   /* PA5 */
#define GPIO_ADC1_IN6   GPIO_ADC1_IN6_0   /* PA6 */
#define GPIO_ADC1_IN7   GPIO_ADC1_IN7_0   /* PA7 */
#define GPIO_ADC1_IN8   GPIO_ADC1_IN8_0   /* PB0 */
#define GPIO_ADC1_IN9   GPIO_ADC1_IN9_0   /* PB1 */
#define GPIO_ADC1_IN10  GPIO_ADC1_IN10_0  /* PC0 */
#define GPIO_ADC1_IN11  GPIO_ADC1_IN11_0  /* PC1 */
#define GPIO_ADC1_IN12  GPIO_ADC1_IN12_0  /* PC2 */
#define GPIO_ADC1_IN13  GPIO_ADC1_IN13_0  /* PC3 */
#define GPIO_ADC1_IN14  GPIO_ADC1_IN14_0  /* PC4 */
#define GPIO_ADC1_IN15  GPIO_ADC1_IN15_0  /* PC5 */

/* USART3
 * TX - PB10
 * RX - PB11
 */

#define GPIO_USART3_RX (GPIO_USART3_RX_1|GPIO_SPEED_100MHz) /* PB11 */
#define GPIO_USART3_TX (GPIO_USART3_TX_1|GPIO_SPEED_100MHz) /* PB10 */

/* USART6 (RS485)
 * TX  - PG14
 * RX  - PG9
 * RTS - PG12
 * CK  - PC8
 */

#define GPIO_USART6_TX  (GPIO_USART6_TX_2|GPIO_SPEED_100MHz)  /* PG14 */
#define GPIO_USART6_RX  (GPIO_USART6_RX_2|GPIO_SPEED_100MHz)  /* PG9 */
#define GPIO_USART6_RTS GPIO_USART6_RTS_1                     /* PG12 */
#define GPIO_USART6_CK  GPIO_USART6_CK_1                      /* PC8 */

/* PWM1 - FOC */

#define GPIO_TIM1_CH1OUT  (GPIO_TIM1_CH1OUT_1|GPIO_SPEED_50MHz)   /* PA8 */
#define GPIO_TIM1_CH1NOUT (GPIO_TIM1_CH1NOUT_2|GPIO_SPEED_50MHz)  /* PB13 */
#define GPIO_TIM1_CH2OUT  (GPIO_TIM1_CH2OUT_1|GPIO_SPEED_50MHz)   /* PA9 */
#define GPIO_TIM1_CH2NOUT (GPIO_TIM1_CH2NOUT_1|GPIO_SPEED_50MHz)  /* PB0 */
#define GPIO_TIM1_CH3OUT  (GPIO_TIM1_CH3OUT_1|GPIO_SPEED_50MHz)   /* PA10 */
#define GPIO_TIM1_CH3NOUT (GPIO_TIM1_CH3NOUT_1|GPIO_SPEED_50MHz)  /* PB1 */
#define GPIO_TIM1_CH4OUT  0                                       /* not used as output */

/* TIM2 - QENCO */

#define GPIO_TIM2_CH1IN (GPIO_TIM2_CH1IN_1|GPIO_SPEED_50MHz) /* PA0 */
#define GPIO_TIM2_CH2IN (GPIO_TIM2_CH2IN_1|GPIO_SPEED_50MHz) /* PA1 */
#define GPIO_TIM2_CH3IN (GPIO_TIM2_CH3IN_1|GPIO_SPEED_50MHz) /* PA2 */

#endif /* __BOARDS_ARM_STM32F7_STEVAL_ETH001V1_INCLUDE_BOARD_H */
