/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shiro.crypto;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.shiro.crypto.CipherService;
import org.apache.shiro.crypto.CryptoException;
import org.apache.shiro.util.ByteSource;
import org.apache.shiro.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class JcaCipherService
implements CipherService {
    private static final Logger log = LoggerFactory.getLogger(JcaCipherService.class);
    private static final int DEFAULT_KEY_SIZE = 128;
    private static final int DEFAULT_STREAMING_BUFFER_SIZE = 512;
    private static final int BITS_PER_BYTE = 8;
    private static final String RANDOM_NUM_GENERATOR_ALGORITHM_NAME = "SHA1PRNG";
    private String algorithmName;
    private int keySize;
    private int streamingBufferSize;
    private boolean generateInitializationVectors;
    private int initializationVectorSize;
    private SecureRandom secureRandom;

    protected JcaCipherService(String algorithmName) {
        if (!StringUtils.hasText((String)algorithmName)) {
            throw new IllegalArgumentException("algorithmName argument cannot be null or empty.");
        }
        this.algorithmName = algorithmName;
        this.keySize = 128;
        this.initializationVectorSize = 128;
        this.streamingBufferSize = 512;
        this.generateInitializationVectors = true;
    }

    public String getAlgorithmName() {
        return this.algorithmName;
    }

    public int getKeySize() {
        return this.keySize;
    }

    public void setKeySize(int keySize) {
        this.keySize = keySize;
    }

    public boolean isGenerateInitializationVectors() {
        return this.generateInitializationVectors;
    }

    public void setGenerateInitializationVectors(boolean generateInitializationVectors) {
        this.generateInitializationVectors = generateInitializationVectors;
    }

    public int getInitializationVectorSize() {
        return this.initializationVectorSize;
    }

    public void setInitializationVectorSize(int initializationVectorSize) throws IllegalArgumentException {
        if (initializationVectorSize % 8 != 0) {
            String msg = "Initialization vector sizes are specified in bits, but must be a multiple of 8 so they can be easily represented as a byte array.";
            throw new IllegalArgumentException(msg);
        }
        this.initializationVectorSize = initializationVectorSize;
    }

    protected boolean isGenerateInitializationVectors(boolean streaming) {
        return this.isGenerateInitializationVectors();
    }

    public int getStreamingBufferSize() {
        return this.streamingBufferSize;
    }

    public void setStreamingBufferSize(int streamingBufferSize) {
        this.streamingBufferSize = streamingBufferSize;
    }

    public SecureRandom getSecureRandom() {
        return this.secureRandom;
    }

    public void setSecureRandom(SecureRandom secureRandom) {
        this.secureRandom = secureRandom;
    }

    protected static SecureRandom getDefaultSecureRandom() {
        try {
            return SecureRandom.getInstance(RANDOM_NUM_GENERATOR_ALGORITHM_NAME);
        }
        catch (NoSuchAlgorithmException e) {
            log.debug("The SecureRandom SHA1PRNG algorithm is not available on the current platform.  Using the platform's default SecureRandom algorithm.", (Throwable)e);
            return new SecureRandom();
        }
    }

    protected SecureRandom ensureSecureRandom() {
        SecureRandom random = this.getSecureRandom();
        if (random == null) {
            random = JcaCipherService.getDefaultSecureRandom();
        }
        return random;
    }

    protected String getTransformationString(boolean streaming) {
        return this.getAlgorithmName();
    }

    protected byte[] generateInitializationVector(boolean streaming) {
        int size = this.getInitializationVectorSize();
        if (size <= 0) {
            String msg = "initializationVectorSize property must be greater than zero.  This number is typically set in the " + CipherService.class.getSimpleName() + " subclass constructor.  " + "Also check your configuration to ensure that if you are setting a value, it is positive.";
            throw new IllegalStateException(msg);
        }
        if (size % 8 != 0) {
            String msg = "initializationVectorSize property must be a multiple of 8 to represent as a byte array.";
            throw new IllegalStateException(msg);
        }
        int sizeInBytes = size / 8;
        byte[] ivBytes = new byte[sizeInBytes];
        SecureRandom random = this.ensureSecureRandom();
        random.nextBytes(ivBytes);
        return ivBytes;
    }

    @Override
    public ByteSource encrypt(byte[] plaintext, byte[] key) {
        byte[] ivBytes = null;
        boolean generate = this.isGenerateInitializationVectors(false);
        if (generate && ((ivBytes = this.generateInitializationVector(false)) == null || ivBytes.length == 0)) {
            throw new IllegalStateException("Initialization vector generation is enabled - generated vectorcannot be null or empty.");
        }
        return this.encrypt(plaintext, key, ivBytes, generate);
    }

    private ByteSource encrypt(byte[] plaintext, byte[] key, byte[] iv, boolean prependIv) throws CryptoException {
        byte[] output;
        boolean MODE = true;
        if (prependIv && iv != null && iv.length > 0) {
            byte[] encrypted = this.crypt(plaintext, key, iv, 1);
            output = new byte[iv.length + encrypted.length];
            System.arraycopy(iv, 0, output, 0, iv.length);
            System.arraycopy(encrypted, 0, output, iv.length, encrypted.length);
        } else {
            output = this.crypt(plaintext, key, iv, 1);
        }
        if (log.isTraceEnabled()) {
            log.trace("Incoming plaintext of size " + (plaintext != null ? plaintext.length : 0) + ".  Ciphertext " + "byte array is size " + (output != null ? output.length : 0));
        }
        return ByteSource.Util.bytes((byte[])output);
    }

    @Override
    public ByteSource decrypt(byte[] ciphertext, byte[] key) throws CryptoException {
        byte[] encrypted = ciphertext;
        byte[] iv = null;
        if (this.isGenerateInitializationVectors(false)) {
            try {
                int ivSize = this.getInitializationVectorSize();
                int ivByteSize = ivSize / 8;
                iv = new byte[ivByteSize];
                System.arraycopy(ciphertext, 0, iv, 0, ivByteSize);
                int encryptedSize = ciphertext.length - ivByteSize;
                encrypted = new byte[encryptedSize];
                System.arraycopy(ciphertext, ivByteSize, encrypted, 0, encryptedSize);
            }
            catch (Exception e) {
                String msg = "Unable to correctly extract the Initialization Vector or ciphertext.";
                throw new CryptoException(msg, e);
            }
        }
        return this.decrypt(encrypted, key, iv);
    }

    private ByteSource decrypt(byte[] ciphertext, byte[] key, byte[] iv) throws CryptoException {
        byte[] decrypted;
        if (log.isTraceEnabled()) {
            log.trace("Attempting to decrypt incoming byte array of length " + (ciphertext != null ? ciphertext.length : 0));
        }
        return (decrypted = this.crypt(ciphertext, key, iv, 2)) == null ? null : ByteSource.Util.bytes((byte[])decrypted);
    }

    private Cipher newCipherInstance(boolean streaming) throws CryptoException {
        String transformationString = this.getTransformationString(streaming);
        try {
            return Cipher.getInstance(transformationString);
        }
        catch (Exception e) {
            String msg = "Unable to acquire a Java JCA Cipher instance using " + Cipher.class.getName() + ".getInstance( \"" + transformationString + "\" ). " + this.getAlgorithmName() + " under this configuration is required for the " + this.getClass().getName() + " instance to function.";
            throw new CryptoException(msg, e);
        }
    }

    private byte[] crypt(byte[] bytes, byte[] key, byte[] iv, int mode) throws IllegalArgumentException, CryptoException {
        if (key == null || key.length == 0) {
            throw new IllegalArgumentException("key argument cannot be null or empty.");
        }
        Cipher cipher = this.initNewCipher(mode, key, iv, false);
        return this.crypt(cipher, bytes);
    }

    private byte[] crypt(Cipher cipher, byte[] bytes) throws CryptoException {
        try {
            return cipher.doFinal(bytes);
        }
        catch (Exception e) {
            String msg = "Unable to execute 'doFinal' with cipher instance [" + cipher + "].";
            throw new CryptoException(msg, e);
        }
    }

    private void init(Cipher cipher, int mode, Key key, AlgorithmParameterSpec spec, SecureRandom random) throws CryptoException {
        try {
            if (random != null) {
                if (spec != null) {
                    cipher.init(mode, key, spec, random);
                } else {
                    cipher.init(mode, key, random);
                }
            } else if (spec != null) {
                cipher.init(mode, key, spec);
            } else {
                cipher.init(mode, key);
            }
        }
        catch (Exception e) {
            String msg = "Unable to init cipher instance.";
            throw new CryptoException(msg, e);
        }
    }

    @Override
    public void encrypt(InputStream in, OutputStream out, byte[] key) throws CryptoException {
        byte[] iv = null;
        boolean generate = this.isGenerateInitializationVectors(true);
        if (generate && ((iv = this.generateInitializationVector(true)) == null || iv.length == 0)) {
            throw new IllegalStateException("Initialization vector generation is enabled - generated vectorcannot be null or empty.");
        }
        this.encrypt(in, out, key, iv, generate);
    }

    private void encrypt(InputStream in, OutputStream out, byte[] key, byte[] iv, boolean prependIv) throws CryptoException {
        if (prependIv && iv != null && iv.length > 0) {
            try {
                out.write(iv);
            }
            catch (IOException e) {
                throw new CryptoException(e);
            }
        }
        this.crypt(in, out, key, iv, 1);
    }

    @Override
    public void decrypt(InputStream in, OutputStream out, byte[] key) throws CryptoException {
        this.decrypt(in, out, key, this.isGenerateInitializationVectors(true));
    }

    private void decrypt(InputStream in, OutputStream out, byte[] key, boolean ivPrepended) throws CryptoException {
        byte[] iv = null;
        if (ivPrepended) {
            int read;
            int ivSize = this.getInitializationVectorSize();
            int ivByteSize = ivSize / 8;
            iv = new byte[ivByteSize];
            try {
                read = in.read(iv);
            }
            catch (IOException e) {
                String msg = "Unable to correctly read the Initialization Vector from the input stream.";
                throw new CryptoException(msg, e);
            }
            if (read != ivByteSize) {
                throw new CryptoException("Unable to read initialization vector bytes from the InputStream.  This is required when initialization vectors are autogenerated during an encryption operation.");
            }
        }
        this.decrypt(in, out, key, iv);
    }

    private void decrypt(InputStream in, OutputStream out, byte[] decryptionKey, byte[] iv) throws CryptoException {
        this.crypt(in, out, decryptionKey, iv, 2);
    }

    private void crypt(InputStream in, OutputStream out, byte[] keyBytes, byte[] iv, int cryptMode) throws CryptoException {
        if (in == null) {
            throw new NullPointerException("InputStream argument cannot be null.");
        }
        if (out == null) {
            throw new NullPointerException("OutputStream argument cannot be null.");
        }
        Cipher cipher = this.initNewCipher(cryptMode, keyBytes, iv, true);
        CipherInputStream cis = new CipherInputStream(in, cipher);
        int bufSize = this.getStreamingBufferSize();
        byte[] buffer = new byte[bufSize];
        try {
            int bytesRead;
            while ((bytesRead = cis.read(buffer)) != -1) {
                out.write(buffer, 0, bytesRead);
            }
        }
        catch (IOException e) {
            throw new CryptoException(e);
        }
    }

    private Cipher initNewCipher(int jcaCipherMode, byte[] key, byte[] iv, boolean streaming) throws CryptoException {
        Cipher cipher = this.newCipherInstance(streaming);
        SecretKeySpec jdkKey = new SecretKeySpec(key, this.getAlgorithmName());
        IvParameterSpec ivSpec = null;
        if (iv != null && iv.length > 0) {
            ivSpec = new IvParameterSpec(iv);
        }
        this.init(cipher, jcaCipherMode, jdkKey, ivSpec, this.getSecureRandom());
        return cipher;
    }
}

