/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.catalog;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Scheduler;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Streams;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.gravitino.Catalog;
import org.apache.gravitino.CatalogChange;
import org.apache.gravitino.CatalogProvider;
import org.apache.gravitino.Config;
import org.apache.gravitino.Configs;
import org.apache.gravitino.Entity;
import org.apache.gravitino.EntityAlreadyExistsException;
import org.apache.gravitino.EntityStore;
import org.apache.gravitino.NameIdentifier;
import org.apache.gravitino.Namespace;
import org.apache.gravitino.StringIdentifier;
import org.apache.gravitino.catalog.CatalogDispatcher;
import org.apache.gravitino.catalog.PropertiesMetadataHelpers;
import org.apache.gravitino.connector.BaseCatalog;
import org.apache.gravitino.connector.BaseCatalogPropertiesMetadata;
import org.apache.gravitino.connector.CatalogOperations;
import org.apache.gravitino.connector.HasPropertyMetadata;
import org.apache.gravitino.connector.PropertyEntry;
import org.apache.gravitino.connector.SupportsSchemas;
import org.apache.gravitino.connector.authorization.BaseAuthorization;
import org.apache.gravitino.connector.capability.Capability;
import org.apache.gravitino.exceptions.CatalogAlreadyExistsException;
import org.apache.gravitino.exceptions.CatalogInUseException;
import org.apache.gravitino.exceptions.CatalogNotInUseException;
import org.apache.gravitino.exceptions.GravitinoRuntimeException;
import org.apache.gravitino.exceptions.MetalakeNotInUseException;
import org.apache.gravitino.exceptions.NoSuchCatalogException;
import org.apache.gravitino.exceptions.NoSuchEntityException;
import org.apache.gravitino.exceptions.NoSuchMetalakeException;
import org.apache.gravitino.exceptions.NonEmptyCatalogException;
import org.apache.gravitino.exceptions.NonEmptyEntityException;
import org.apache.gravitino.file.FilesetCatalog;
import org.apache.gravitino.messaging.TopicCatalog;
import org.apache.gravitino.meta.AuditInfo;
import org.apache.gravitino.meta.CatalogEntity;
import org.apache.gravitino.meta.SchemaEntity;
import org.apache.gravitino.metalake.MetalakeManager;
import org.apache.gravitino.model.ModelCatalog;
import org.apache.gravitino.rel.SupportsPartitions;
import org.apache.gravitino.rel.Table;
import org.apache.gravitino.rel.TableCatalog;
import org.apache.gravitino.storage.IdGenerator;
import org.apache.gravitino.utils.IsolatedClassLoader;
import org.apache.gravitino.utils.PrincipalUtils;
import org.apache.gravitino.utils.ThrowableFunction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CatalogManager
implements CatalogDispatcher,
Closeable {
    private static final String CATALOG_DOES_NOT_EXIST_MSG = "Catalog %s does not exist";
    private static final Logger LOG = LoggerFactory.getLogger(CatalogManager.class);
    private final Config config;
    @VisibleForTesting
    final Cache<NameIdentifier, CatalogWrapper> catalogCache;
    private final EntityStore store;
    private final IdGenerator idGenerator;

    public static void checkCatalogInUse(EntityStore store, NameIdentifier ident) throws NoSuchMetalakeException, NoSuchCatalogException, CatalogNotInUseException, MetalakeNotInUseException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        MetalakeManager.checkMetalake(metalakeIdent, store);
        if (!CatalogManager.getCatalogInUseValue(store, ident)) {
            throw new CatalogNotInUseException("Catalog %s is not in use, please enable it first", new Object[]{ident});
        }
    }

    public CatalogManager(Config config, EntityStore store, IdGenerator idGenerator) {
        this.config = config;
        this.store = store;
        this.idGenerator = idGenerator;
        long cacheEvictionIntervalInMs = config.get(Configs.CATALOG_CACHE_EVICTION_INTERVAL_MS);
        this.catalogCache = Caffeine.newBuilder().expireAfterAccess(cacheEvictionIntervalInMs, TimeUnit.MILLISECONDS).removalListener((k, v, c) -> {
            LOG.info("Closing catalog {}.", k);
            ((CatalogWrapper)v).close();
        }).scheduler(Scheduler.forScheduledExecutorService((ScheduledExecutorService)new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("catalog-cleaner-%d").build()))).build();
    }

    @Override
    public void close() {
        this.catalogCache.invalidateAll();
    }

    @Override
    public NameIdentifier[] listCatalogs(Namespace namespace) throws NoSuchMetalakeException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])namespace.levels());
        MetalakeManager.checkMetalake(NameIdentifier.of((String[])new String[]{namespace.level(0)}), this.store);
        try {
            return (NameIdentifier[])this.store.list(namespace, CatalogEntity.class, Entity.EntityType.CATALOG).stream().map(entity -> NameIdentifier.of((Namespace)namespace, (String)entity.name())).toArray(NameIdentifier[]::new);
        }
        catch (IOException ioe) {
            LOG.error("Failed to list catalogs in metalake {}", (Object)metalakeIdent, (Object)ioe);
            throw new RuntimeException(ioe);
        }
    }

    @Override
    public Catalog[] listCatalogsInfo(Namespace namespace) throws NoSuchMetalakeException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])namespace.levels());
        MetalakeManager.checkMetalake(metalakeIdent, this.store);
        try {
            List<CatalogEntity> catalogEntities = this.store.list(namespace, CatalogEntity.class, Entity.EntityType.CATALOG);
            return (Catalog[])catalogEntities.stream().map(e -> e.toCatalogInfoWithResolvedProps(this.getResolvedProperties((CatalogEntity)e))).toArray(Catalog[]::new);
        }
        catch (IOException ioe) {
            LOG.error("Failed to list catalogs in metalake {}", (Object)metalakeIdent, (Object)ioe);
            throw new RuntimeException(ioe);
        }
    }

    @Override
    public Catalog loadCatalog(NameIdentifier ident) throws NoSuchCatalogException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        MetalakeManager.checkMetalake(metalakeIdent, this.store);
        return this.loadCatalogAndWrap(ident).catalog;
    }

    @Override
    public Catalog createCatalog(NameIdentifier ident, Catalog.Type type, String provider, String comment, Map<String, String> properties) throws NoSuchMetalakeException, CatalogAlreadyExistsException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        MetalakeManager.checkMetalake(metalakeIdent, this.store);
        Map<String, String> mergedConfig = this.buildCatalogConf(provider, properties);
        long uid = this.idGenerator.nextId();
        StringIdentifier stringId = StringIdentifier.fromId(uid);
        Instant now = Instant.now();
        String creator = PrincipalUtils.getCurrentPrincipal().getName();
        CatalogEntity e = CatalogEntity.builder().withId(uid).withName(ident.name()).withNamespace(ident.namespace()).withType(type).withProvider(provider).withComment(comment).withProperties(StringIdentifier.newPropertiesWithId(stringId, mergedConfig)).withAuditInfo(AuditInfo.builder().withCreator(creator).withCreateTime(now).withLastModifier(creator).withLastModifiedTime(now).build()).build();
        boolean needClean = true;
        try {
            this.store.put(e, false);
            CatalogWrapper wrapper = (CatalogWrapper)this.catalogCache.get((Object)ident, id -> this.createCatalogWrapper(e, mergedConfig));
            needClean = false;
            BaseCatalog baseCatalog = wrapper.catalog;
            return baseCatalog;
        }
        catch (EntityAlreadyExistsException e1) {
            needClean = false;
            LOG.warn("Catalog {} already exists", (Object)ident, (Object)e1);
            throw new CatalogAlreadyExistsException("Catalog %s already exists", new Object[]{ident});
        }
        catch (IllegalArgumentException | NoSuchMetalakeException e2) {
            throw e2;
        }
        catch (Exception e3) {
            this.catalogCache.invalidate((Object)ident);
            LOG.error("Failed to create catalog {}", (Object)ident, (Object)e3);
            if (e3 instanceof RuntimeException) {
                throw (RuntimeException)e3;
            }
            throw new RuntimeException(e3);
        }
        finally {
            if (needClean) {
                try {
                    this.store.delete(ident, Entity.EntityType.CATALOG, true);
                }
                catch (IOException e4) {
                    LOG.error("Failed to clean up catalog {}", (Object)ident, (Object)e4);
                }
            }
        }
    }

    @Override
    public void testConnection(NameIdentifier ident, Catalog.Type type, String provider, String comment, Map<String, String> properties) {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        MetalakeManager.checkMetalake(metalakeIdent, this.store);
        try {
            if (this.store.exists(ident, Entity.EntityType.CATALOG)) {
                throw new CatalogAlreadyExistsException("Catalog %s already exists", new Object[]{ident});
            }
            Map<String, String> mergedConfig = this.buildCatalogConf(provider, properties);
            Instant now = Instant.now();
            String creator = PrincipalUtils.getCurrentPrincipal().getName();
            CatalogEntity dummyEntity = CatalogEntity.builder().withId(StringIdentifier.DUMMY_ID.id()).withName(ident.name()).withNamespace(ident.namespace()).withType(type).withProvider(provider).withComment(comment).withProperties(StringIdentifier.newPropertiesWithId(StringIdentifier.DUMMY_ID, mergedConfig)).withAuditInfo(AuditInfo.builder().withCreator(creator).withCreateTime(now).withLastModifier(creator).withLastModifiedTime(now).build()).build();
            CatalogWrapper wrapper = this.createCatalogWrapper(dummyEntity, mergedConfig);
            wrapper.doWithCatalogOps(c -> {
                c.testConnection(ident, type, provider, comment, mergedConfig);
                return null;
            });
        }
        catch (GravitinoRuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            LOG.warn("Failed to test catalog creation {}", (Object)ident, (Object)e);
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new RuntimeException(e);
        }
    }

    @Override
    public void enableCatalog(NameIdentifier ident) throws NoSuchCatalogException, CatalogNotInUseException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        MetalakeManager.checkMetalake(metalakeIdent, this.store);
        try {
            if (CatalogManager.catalogInUse(this.store, ident)) {
                return;
            }
            this.store.update(ident, CatalogEntity.class, Entity.EntityType.CATALOG, catalog -> {
                CatalogEntity.Builder newCatalogBuilder = this.newCatalogBuilder(ident.namespace(), (CatalogEntity)catalog);
                HashMap<String, String> newProps = catalog.getProperties() == null ? new HashMap<String, String>() : new HashMap<String, String>(catalog.getProperties());
                newProps.put("in-use", "true");
                newCatalogBuilder.withProperties(newProps);
                return newCatalogBuilder.build();
            });
            this.catalogCache.invalidate((Object)ident);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void disableCatalog(NameIdentifier ident) throws NoSuchCatalogException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        MetalakeManager.checkMetalake(metalakeIdent, this.store);
        try {
            if (!CatalogManager.catalogInUse(this.store, ident)) {
                return;
            }
            this.store.update(ident, CatalogEntity.class, Entity.EntityType.CATALOG, catalog -> {
                CatalogEntity.Builder newCatalogBuilder = this.newCatalogBuilder(ident.namespace(), (CatalogEntity)catalog);
                HashMap<String, String> newProps = catalog.getProperties() == null ? new HashMap<String, String>() : new HashMap<String, String>(catalog.getProperties());
                newProps.put("in-use", "false");
                newCatalogBuilder.withProperties(newProps);
                return newCatalogBuilder.build();
            });
            this.catalogCache.invalidate((Object)ident);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Catalog alterCatalog(NameIdentifier ident, CatalogChange ... changes) throws NoSuchCatalogException, IllegalArgumentException {
        CatalogManager.checkCatalogInUse(this.store, ident);
        CatalogWrapper catalogWrapper = this.loadCatalogAndWrap(ident);
        if (catalogWrapper == null) {
            throw new NoSuchCatalogException(CATALOG_DOES_NOT_EXIST_MSG, new Object[]{ident});
        }
        try {
            catalogWrapper.doWithPropertiesMeta(f -> {
                Pair<Map<String, String>, Map<String, String>> alterProperty = this.getCatalogAlterProperty(changes);
                PropertiesMetadataHelpers.validatePropertyForAlter(f.catalogPropertiesMetadata(), (Map)alterProperty.getLeft(), (Map)alterProperty.getRight());
                return null;
            });
        }
        catch (IllegalArgumentException e1) {
            throw e1;
        }
        catch (Exception e) {
            LOG.error("Failed to alter catalog {}", (Object)ident, (Object)e);
            throw new RuntimeException(e);
        }
        this.catalogCache.invalidate((Object)ident);
        try {
            CatalogEntity updatedCatalog = this.store.update(ident, CatalogEntity.class, Entity.EntityType.CATALOG, catalog -> {
                CatalogEntity.Builder newCatalogBuilder = this.newCatalogBuilder(ident.namespace(), (CatalogEntity)catalog);
                HashMap<String, String> newProps = catalog.getProperties() == null ? new HashMap<String, String>() : new HashMap<String, String>(catalog.getProperties());
                newCatalogBuilder = this.updateEntity(newCatalogBuilder, newProps, changes);
                return newCatalogBuilder.build();
            });
            return Objects.requireNonNull((CatalogWrapper)this.catalogCache.get((Object)updatedCatalog.nameIdentifier(), id -> this.createCatalogWrapper(updatedCatalog, null))).catalog;
        }
        catch (NoSuchEntityException ne) {
            LOG.warn("Catalog {} does not exist", (Object)ident, (Object)ne);
            throw new NoSuchCatalogException(CATALOG_DOES_NOT_EXIST_MSG, new Object[]{ident});
        }
        catch (IllegalArgumentException iae) {
            LOG.warn("Failed to alter catalog {} with unknown change", (Object)ident, (Object)iae);
            throw iae;
        }
        catch (IOException ioe) {
            LOG.error("Failed to alter catalog {}", (Object)ident, (Object)ioe);
            throw new RuntimeException(ioe);
        }
    }

    @Override
    public boolean dropCatalog(NameIdentifier ident, boolean force) throws NonEmptyEntityException, CatalogInUseException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        MetalakeManager.checkMetalake(metalakeIdent, this.store);
        try {
            CatalogEntity catalogEntity;
            boolean catalogInUse = CatalogManager.catalogInUse(this.store, ident);
            if (catalogInUse && !force) {
                throw new CatalogInUseException("Catalog %s is in use, please disable it first or use force option", new Object[]{ident});
            }
            Namespace schemaNamespace = Namespace.of((String[])new String[]{ident.namespace().level(0), ident.name()});
            CatalogWrapper catalogWrapper = this.loadCatalogAndWrap(ident);
            List<SchemaEntity> schemaEntities = this.store.list(schemaNamespace, SchemaEntity.class, Entity.EntityType.SCHEMA);
            if (this.containsUserCreatedSchemas(schemaEntities, catalogEntity = this.store.get(ident, Entity.EntityType.CATALOG, CatalogEntity.class), catalogWrapper) && !force) {
                throw new NonEmptyCatalogException("Catalog %s has schemas, please drop them first or use force option", new Object[]{ident});
            }
            if (this.includeManagedEntities(catalogEntity)) {
                schemaEntities.forEach(schema -> {
                    try {
                        catalogWrapper.doWithSchemaOps(schemaOps -> schemaOps.dropSchema(schema.nameIdentifier(), true));
                    }
                    catch (Exception e) {
                        LOG.warn("Failed to drop schema {}", (Object)schema.nameIdentifier());
                        throw new RuntimeException("Failed to drop schema " + schema.nameIdentifier(), e);
                    }
                });
            }
            this.catalogCache.invalidate((Object)ident);
            return this.store.delete(ident, Entity.EntityType.CATALOG, true);
        }
        catch (NoSuchCatalogException | NoSuchMetalakeException ignored) {
            return false;
        }
        catch (GravitinoRuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private boolean containsUserCreatedSchemas(List<SchemaEntity> schemaEntities, CatalogEntity catalogEntity, CatalogWrapper catalogWrapper) throws Exception {
        NameIdentifier[] allSchemas;
        if (schemaEntities.isEmpty()) {
            return false;
        }
        if (schemaEntities.size() == 1) {
            if ("kafka".equals(catalogEntity.getProvider())) {
                return false;
            }
            if ("jdbc-postgresql".equals(catalogEntity.getProvider())) {
                return !schemaEntities.get(0).name().equals("public");
            }
            if ("hive".equals(catalogEntity.getProvider())) {
                return !schemaEntities.get(0).name().equals("default");
            }
        }
        if ((allSchemas = catalogWrapper.doWithSchemaOps(schemaOps -> schemaOps.listSchemas(Namespace.of((String[])new String[]{catalogEntity.namespace().level(0), catalogEntity.name()})))).length == 0) {
            return false;
        }
        Set availableSchemaNames = Arrays.stream(allSchemas).map(NameIdentifier::name).collect(Collectors.toSet());
        return schemaEntities.stream().map(SchemaEntity::name).anyMatch(availableSchemaNames::contains);
    }

    private boolean includeManagedEntities(CatalogEntity catalogEntity) {
        return catalogEntity.getType().equals((Object)Catalog.Type.FILESET);
    }

    public CatalogWrapper loadCatalogAndWrap(NameIdentifier ident) throws NoSuchCatalogException {
        return (CatalogWrapper)this.catalogCache.get((Object)ident, this::loadCatalogInternal);
    }

    private static boolean catalogInUse(EntityStore store, NameIdentifier ident) throws NoSuchMetalakeException, NoSuchCatalogException {
        NameIdentifier metalakeIdent = NameIdentifier.of((String[])ident.namespace().levels());
        return MetalakeManager.metalakeInUse(store, metalakeIdent) && CatalogManager.getCatalogInUseValue(store, ident);
    }

    private static boolean getCatalogInUseValue(EntityStore store, NameIdentifier catalogIdent) {
        try {
            CatalogEntity catalogEntity = store.get(catalogIdent, Entity.EntityType.CATALOG, CatalogEntity.class);
            return (Boolean)BaseCatalogPropertiesMetadata.BASIC_CATALOG_PROPERTIES_METADATA.getOrDefault(catalogEntity.getProperties(), "in-use");
        }
        catch (NoSuchEntityException e) {
            LOG.warn("Catalog {} does not exist", (Object)catalogIdent, (Object)e);
            throw new NoSuchCatalogException(CATALOG_DOES_NOT_EXIST_MSG, new Object[]{catalogIdent});
        }
        catch (IOException e) {
            LOG.error("Failed to do store operation", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    private CatalogEntity.Builder newCatalogBuilder(Namespace namespace, CatalogEntity catalog) {
        CatalogEntity.Builder builder = CatalogEntity.builder().withId(catalog.id()).withName(catalog.name()).withNamespace(namespace).withType(catalog.getType()).withProvider(catalog.getProvider()).withComment(catalog.getComment());
        AuditInfo newInfo = AuditInfo.builder().withCreator(catalog.auditInfo().creator()).withCreateTime(catalog.auditInfo().createTime()).withLastModifier(PrincipalUtils.getCurrentPrincipal().getName()).withLastModifiedTime(Instant.now()).build();
        return builder.withAuditInfo(newInfo);
    }

    private Map<String, String> buildCatalogConf(String provider, Map<String, String> properties) {
        Map<String, String> newProperties = Optional.ofNullable(properties).orElse(Maps.newHashMap());
        Map<String, String> catalogSpecificConfig = this.loadCatalogSpecificConfig(newProperties, provider);
        return CatalogManager.mergeConf(newProperties, catalogSpecificConfig);
    }

    private Pair<Map<String, String>, Map<String, String>> getCatalogAlterProperty(CatalogChange ... catalogChanges) {
        HashMap upserts = Maps.newHashMap();
        HashMap deletes = Maps.newHashMap();
        Arrays.stream(catalogChanges).forEach(catalogChange -> {
            if (catalogChange instanceof CatalogChange.SetProperty) {
                CatalogChange.SetProperty setProperty = (CatalogChange.SetProperty)catalogChange;
                upserts.put(setProperty.getProperty(), setProperty.getValue());
            } else if (catalogChange instanceof CatalogChange.RemoveProperty) {
                CatalogChange.RemoveProperty removeProperty = (CatalogChange.RemoveProperty)catalogChange;
                deletes.put(removeProperty.getProperty(), removeProperty.getProperty());
            }
        });
        return Pair.of((Object)upserts, (Object)deletes);
    }

    private CatalogWrapper loadCatalogInternal(NameIdentifier ident) throws NoSuchCatalogException {
        try {
            CatalogEntity entity = this.store.get(ident, Entity.EntityType.CATALOG, CatalogEntity.class);
            return this.createCatalogWrapper(entity, null);
        }
        catch (NoSuchEntityException ne) {
            LOG.warn("Catalog {} does not exist", (Object)ident, (Object)ne);
            throw new NoSuchCatalogException(CATALOG_DOES_NOT_EXIST_MSG, new Object[]{ident});
        }
        catch (IOException ioe) {
            LOG.error("Failed to load catalog {}", (Object)ident, (Object)ioe);
            throw new RuntimeException(ioe);
        }
    }

    private CatalogWrapper createCatalogWrapper(CatalogEntity entity, @Nullable Map<String, String> propsToValidate) {
        Map<String, String> conf = entity.getProperties();
        String provider = entity.getProvider();
        IsolatedClassLoader classLoader = this.createClassLoader(provider, conf);
        BaseCatalog<?> catalog = this.createBaseCatalog(classLoader, entity);
        CatalogWrapper wrapper = new CatalogWrapper(catalog, classLoader);
        classLoader.withClassLoader(cl -> {
            PropertiesMetadataHelpers.validatePropertyForCreate(catalog.catalogPropertiesMetadata(), propsToValidate);
            wrapper.catalog.properties();
            wrapper.catalog.capability();
            return null;
        }, IllegalArgumentException.class);
        return wrapper;
    }

    private Map<String, String> getResolvedProperties(CatalogEntity entity) {
        Map<String, String> conf = entity.getProperties();
        String provider = entity.getProvider();
        try (IsolatedClassLoader classLoader = this.createClassLoader(provider, conf);){
            BaseCatalog<?> catalog = this.createBaseCatalog(classLoader, entity);
            Map map = classLoader.withClassLoader(cl -> catalog.properties(), RuntimeException.class);
            return map;
        }
    }

    private Set<String> getHiddenPropertyNames(CatalogEntity entity) {
        Map<String, String> conf = entity.getProperties();
        String provider = entity.getProvider();
        try (IsolatedClassLoader classLoader = this.createClassLoader(provider, conf);){
            BaseCatalog<?> catalog = this.createBaseCatalog(classLoader, entity);
            Set set = classLoader.withClassLoader(cl -> catalog.catalogPropertiesMetadata().propertyEntries().values().stream().filter(PropertyEntry::isHidden).map(PropertyEntry::getName).collect(Collectors.toSet()), RuntimeException.class);
            return set;
        }
    }

    private BaseCatalog<?> createBaseCatalog(IsolatedClassLoader classLoader, CatalogEntity entity) {
        BaseCatalog<?> catalog = this.createCatalogInstance(classLoader, entity.getProvider());
        ((BaseCatalog)catalog.withCatalogConf(entity.getProperties())).withCatalogEntity(entity);
        catalog.initAuthorizationPluginInstance(classLoader);
        return catalog;
    }

    private IsolatedClassLoader createClassLoader(String provider, Map<String, String> conf) {
        if (this.config.get(Configs.CATALOG_LOAD_ISOLATED).booleanValue()) {
            String catalogPkgPath = this.buildPkgPath(conf, provider);
            String catalogConfPath = this.buildConfPath(conf, provider);
            ArrayList libAndResourcesPaths = Lists.newArrayList((Object[])new String[]{catalogPkgPath, catalogConfPath});
            BaseAuthorization.buildAuthorizationPkgPath(conf).ifPresent(libAndResourcesPaths::add);
            return IsolatedClassLoader.buildClassLoader(libAndResourcesPaths);
        }
        return new IsolatedClassLoader(Collections.emptyList(), Collections.emptyList(), Collections.emptyList());
    }

    private BaseCatalog<?> createCatalogInstance(IsolatedClassLoader classLoader, String provider) {
        BaseCatalog catalog;
        try {
            catalog = classLoader.withClassLoader(cl -> {
                try {
                    Class<? extends CatalogProvider> providerClz = this.lookupCatalogProvider(provider, (ClassLoader)cl);
                    return (BaseCatalog)providerClz.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                }
                catch (Exception e) {
                    LOG.error("Failed to load catalog with provider: {}", (Object)provider, (Object)e);
                    throw new RuntimeException(e);
                }
            });
        }
        catch (Exception e) {
            LOG.error("Failed to load catalog with class loader", (Throwable)e);
            throw new RuntimeException(e);
        }
        if (catalog == null) {
            throw new RuntimeException("Failed to load catalog with provider: " + provider);
        }
        return catalog;
    }

    private Map<String, String> loadCatalogSpecificConfig(Map<String, String> properties, String provider) {
        if ("test".equals(provider)) {
            return Maps.newHashMap();
        }
        String catalogSpecificConfigFile = provider + ".conf";
        HashMap catalogSpecificConfig = Maps.newHashMap();
        String fullPath = this.buildConfPath(properties, provider) + File.separator + catalogSpecificConfigFile;
        try (FileInputStream inputStream = FileUtils.openInputStream((File)new File(fullPath));){
            Properties loadProperties = new Properties();
            loadProperties.load(inputStream);
            loadProperties.forEach((BiConsumer<? super Object, ? super Object>)((BiConsumer<Object, Object>)(key, value) -> catalogSpecificConfig.put(key.toString(), value.toString())));
        }
        catch (Exception e) {
            LOG.warn("Failed to load catalog specific configurations, file name: '{}'", (Object)catalogSpecificConfigFile, (Object)e);
        }
        return catalogSpecificConfig;
    }

    static Map<String, String> mergeConf(Map<String, String> properties, Map<String, String> conf) {
        HashMap mergedConf = conf != null ? Maps.newHashMap(conf) : Maps.newHashMap();
        Optional.ofNullable(properties).ifPresent(mergedConf::putAll);
        return Collections.unmodifiableMap(mergedConf);
    }

    private String buildConfPath(Map<String, String> properties, String provider) {
        String gravitinoHome = System.getenv("GRAVITINO_HOME");
        Preconditions.checkArgument((gravitinoHome != null ? 1 : 0) != 0, (Object)"GRAVITINO_HOME not set");
        boolean testEnv = System.getenv("GRAVITINO_TEST") != null;
        String pkg = properties.get("package");
        String confPath = pkg != null ? String.join((CharSequence)File.separator, pkg, "conf") : (testEnv ? String.join((CharSequence)File.separator, gravitinoHome, "catalogs", "catalog-" + provider, "build", "resources", "main") : String.join((CharSequence)File.separator, gravitinoHome, "catalogs", provider, "conf"));
        return confPath;
    }

    private String buildPkgPath(Map<String, String> conf, String provider) {
        String gravitinoHome = System.getenv("GRAVITINO_HOME");
        Preconditions.checkArgument((gravitinoHome != null ? 1 : 0) != 0, (Object)"GRAVITINO_HOME not set");
        boolean testEnv = System.getenv("GRAVITINO_TEST") != null;
        String pkg = conf.get("package");
        String pkgPath = pkg != null ? String.join((CharSequence)File.separator, pkg, "libs") : (testEnv ? String.join((CharSequence)File.separator, gravitinoHome, "catalogs", "catalog-" + provider, "build", "libs") : String.join((CharSequence)File.separator, gravitinoHome, "catalogs", provider, "libs"));
        return pkgPath;
    }

    private Class<? extends CatalogProvider> lookupCatalogProvider(String provider, ClassLoader cl) {
        ServiceLoader<CatalogProvider> loader = ServiceLoader.load(CatalogProvider.class, cl);
        List providers = Streams.stream(loader.iterator()).filter(p -> p.shortName().equalsIgnoreCase(provider)).map(Object::getClass).collect(Collectors.toList());
        if (providers.isEmpty()) {
            throw new IllegalArgumentException("No catalog provider found for: " + provider);
        }
        if (providers.size() > 1) {
            throw new IllegalArgumentException("Multiple catalog providers found for: " + provider);
        }
        return (Class)Iterables.getOnlyElement(providers);
    }

    private CatalogEntity.Builder updateEntity(CatalogEntity.Builder builder, Map<String, String> newProps, CatalogChange ... changes) {
        for (CatalogChange change : changes) {
            if (change instanceof CatalogChange.RenameCatalog) {
                CatalogChange.RenameCatalog rename = (CatalogChange.RenameCatalog)change;
                if ("system".equals(((CatalogChange.RenameCatalog)change).getNewName())) {
                    throw new IllegalArgumentException("Can't rename a catalog with with reserved name `system`");
                }
                builder.withName(rename.getNewName());
                continue;
            }
            if (change instanceof CatalogChange.UpdateCatalogComment) {
                CatalogChange.UpdateCatalogComment updateComment = (CatalogChange.UpdateCatalogComment)change;
                builder.withComment(updateComment.getNewComment());
                continue;
            }
            if (change instanceof CatalogChange.SetProperty) {
                CatalogChange.SetProperty setProperty = (CatalogChange.SetProperty)change;
                newProps.put(setProperty.getProperty(), setProperty.getValue());
                continue;
            }
            if (change instanceof CatalogChange.RemoveProperty) {
                CatalogChange.RemoveProperty removeProperty = (CatalogChange.RemoveProperty)change;
                newProps.remove(removeProperty.getProperty());
                continue;
            }
            throw new IllegalArgumentException("Unsupported catalog change: " + change.getClass().getSimpleName());
        }
        return builder.withProperties(newProps);
    }

    public static class CatalogWrapper {
        private BaseCatalog catalog;
        private IsolatedClassLoader classLoader;

        public CatalogWrapper(BaseCatalog catalog, IsolatedClassLoader classLoader) {
            this.catalog = catalog;
            this.classLoader = classLoader;
        }

        public BaseCatalog catalog() {
            return this.catalog;
        }

        public <R> R doWithSchemaOps(ThrowableFunction<SupportsSchemas, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> {
                if (this.asSchemas() == null) {
                    throw new UnsupportedOperationException("Catalog does not support schema operations");
                }
                return fn.apply(this.asSchemas());
            });
        }

        public <R> R doWithTableOps(ThrowableFunction<TableCatalog, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> {
                if (this.asTables() == null) {
                    throw new UnsupportedOperationException("Catalog does not support table operations");
                }
                return fn.apply(this.asTables());
            });
        }

        public <R> R doWithFilesetOps(ThrowableFunction<FilesetCatalog, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> {
                if (this.asFilesets() == null) {
                    throw new UnsupportedOperationException("Catalog does not support fileset operations");
                }
                return fn.apply(this.asFilesets());
            });
        }

        public <R> R doWithCredentialOps(ThrowableFunction<BaseCatalog, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> fn.apply(this.catalog));
        }

        public <R> R doWithTopicOps(ThrowableFunction<TopicCatalog, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> {
                if (this.asTopics() == null) {
                    throw new UnsupportedOperationException("Catalog does not support topic operations");
                }
                return fn.apply(this.asTopics());
            });
        }

        public <R> R doWithModelOps(ThrowableFunction<ModelCatalog, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> {
                if (this.asModels() == null) {
                    throw new UnsupportedOperationException("Catalog does not support model operations");
                }
                return fn.apply(this.asModels());
            });
        }

        public <R> R doWithCatalogOps(ThrowableFunction<CatalogOperations, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> fn.apply(this.catalog.ops()));
        }

        public <R> R doWithPartitionOps(NameIdentifier tableIdent, ThrowableFunction<SupportsPartitions, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> {
                Preconditions.checkArgument((this.asTables() != null ? 1 : 0) != 0, (Object)"Catalog does not support table operations");
                Table table = this.asTables().loadTable(tableIdent);
                Preconditions.checkArgument((table.supportPartitions() != null ? 1 : 0) != 0, (Object)"Table does not support partition operations");
                return fn.apply(table.supportPartitions());
            });
        }

        public <R> R doWithPropertiesMeta(ThrowableFunction<HasPropertyMetadata, R> fn) throws Exception {
            return (R)this.classLoader.withClassLoader(cl -> fn.apply(this.catalog));
        }

        public Capability capabilities() throws Exception {
            return this.classLoader.withClassLoader(cl -> this.catalog.capability());
        }

        public void close() {
            try {
                this.classLoader.withClassLoader(cl -> {
                    if (this.catalog != null) {
                        this.catalog.close();
                    }
                    this.catalog = null;
                    return null;
                });
            }
            catch (Exception e) {
                LOG.warn("Failed to close catalog", (Throwable)e);
            }
            this.classLoader.close();
        }

        private SupportsSchemas asSchemas() {
            return this.catalog.ops() instanceof SupportsSchemas ? (SupportsSchemas)((Object)this.catalog.ops()) : null;
        }

        private TableCatalog asTables() {
            return this.catalog.ops() instanceof TableCatalog ? (TableCatalog)this.catalog.ops() : null;
        }

        private FilesetCatalog asFilesets() {
            return this.catalog.ops() instanceof FilesetCatalog ? (FilesetCatalog)this.catalog.ops() : null;
        }

        private TopicCatalog asTopics() {
            return this.catalog.ops() instanceof TopicCatalog ? (TopicCatalog)this.catalog.ops() : null;
        }

        private ModelCatalog asModels() {
            return this.catalog.ops() instanceof ModelCatalog ? (ModelCatalog)this.catalog.ops() : null;
        }
    }
}

