/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hugegraph.store;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.hugegraph.pd.common.PDException;
import org.apache.hugegraph.pd.grpc.Metapb;
import org.apache.hugegraph.rocksdb.access.RocksDBFactory;
import org.apache.hugegraph.store.business.BusinessHandler;
import org.apache.hugegraph.store.business.BusinessHandlerImpl;
import org.apache.hugegraph.store.business.DataMover;
import org.apache.hugegraph.store.cmd.HgCmdClient;
import org.apache.hugegraph.store.cmd.HgCmdProcessor;
import org.apache.hugegraph.store.cmd.UpdatePartitionRequest;
import org.apache.hugegraph.store.cmd.UpdatePartitionResponse;
import org.apache.hugegraph.store.meta.Partition;
import org.apache.hugegraph.store.meta.PartitionManager;
import org.apache.hugegraph.store.meta.ShardGroup;
import org.apache.hugegraph.store.meta.Store;
import org.apache.hugegraph.store.metric.HgMetricService;
import org.apache.hugegraph.store.options.HgStoreEngineOptions;
import org.apache.hugegraph.store.options.PartitionEngineOptions;
import org.apache.hugegraph.store.pd.DefaultPdProvider;
import org.apache.hugegraph.store.pd.FakePdServiceProvider;
import org.apache.hugegraph.store.pd.PdProvider;
import org.apache.hugegraph.store.raft.RaftClosure;
import org.apache.hugegraph.store.raft.RaftOperation;
import org.apache.hugegraph.store.util.HgRaftError;
import org.apache.hugegraph.store.util.Lifecycle;

import com.alipay.sofa.jraft.JRaftUtils;
import com.alipay.sofa.jraft.Status;
import com.alipay.sofa.jraft.conf.Configuration;
import com.alipay.sofa.jraft.core.NodeMetrics;
import com.alipay.sofa.jraft.entity.PeerId;
import com.alipay.sofa.jraft.option.RpcOptions;
import com.alipay.sofa.jraft.rpc.RaftRpcServerFactory;
import com.alipay.sofa.jraft.rpc.RpcServer;
import com.alipay.sofa.jraft.util.Endpoint;

import lombok.extern.slf4j.Slf4j;

/**
 * The core class of the storage engine, initializing PD client and raft client
 */
@Slf4j
public class HgStoreEngine implements Lifecycle<HgStoreEngineOptions>, HgStoreStateListener {

    private final static HgStoreEngine instance = new HgStoreEngine();
    private static ConcurrentHashMap<Integer, Object> engineLocks = new ConcurrentHashMap<>();
    // Partition raft engine, key is GraphName_PartitionID
    private final Map<Integer, PartitionEngine> partitionEngines = new ConcurrentHashMap<>();
    private RpcServer rpcServer;
    private HgStoreEngineOptions options;
    private PdProvider pdProvider;
    private HgCmdClient hgCmdClient;
    private PartitionManager partitionManager;
    private HeartbeatService heartbeatService;
    private BusinessHandler businessHandler;
    private HgMetricService metricService;
    private DataMover dataMover;

    public static HgStoreEngine getInstance() {
        return instance;
    }

    /**
     * 1. Read StoreId, register with PD, the StoreId is generated by PD for the first registration and stored locally.
     * 2. Registration successful, start the raft service.
     * 3. Timely send Store heartbeats and Partition heartbeats to maintain contact with PD.
     *
     * @param opts
     * @return
     */
    @Override
    public synchronized boolean init(final HgStoreEngineOptions opts) {
        if (rpcServer != null) {
            log.info("HgStoreEngine already started.");
            return true;
        }

        this.options = opts;

        BusinessHandlerImpl.initRocksdb(opts.getRocksdbConfig(), getRocksdbListener());

        if (opts.isFakePD()) {
            pdProvider = new FakePdServiceProvider(opts.getFakePdOptions());
        } else {
            pdProvider = new DefaultPdProvider(opts.getPdAddress());
            pdProvider.addPartitionInstructionListener(new PartitionInstructionProcessor(this));
        }
        options.setPdProvider(pdProvider);

        partitionManager = new PartitionManager(pdProvider, opts);

        partitionManager.addPartitionChangedListener(new PartitionChangedListener());

        businessHandler = new BusinessHandlerImpl(partitionManager);
        // Need businessHandler initialization afterwards
        partitionManager.load();

        rpcServer = createRaftRpcServer(opts.getRaftAddress());

        hgCmdClient = new HgCmdClient();
        hgCmdClient.init(new RpcOptions(), (graphName, ptId) -> {
            // When splitting, it has not been reported to pd in time.
            if (getPartitionEngine(ptId) != null) {
                return getPartitionEngine(ptId).waitForLeader(
                        options.getWaitLeaderTimeout() * 1000);
            } else {
                // May occur cross-partition migration
                Metapb.Shard shard = pdProvider.getPartitionLeader(graphName, ptId);
                return JRaftUtils.getEndPoint(
                        pdProvider.getStoreByID(shard.getStoreId()).getRaftAddress());
            }
        });

        heartbeatService = new HeartbeatService(this);
        heartbeatService.setStoreMetadata(partitionManager.getStoreMetadata());
        heartbeatService.addStateListener(this).init(options);

        metricService = HgMetricService.getInstance();
        metricService.setHgStoreEngine(this).init(null);

        dataMover = opts.getDataTransfer();
        if (dataMover != null) {
            this.dataMover.setBusinessHandler(this.businessHandler);
            this.dataMover.setCmdClient(hgCmdClient);
        }
        return true;
    }

    /**
     * Create raft rpc server, for communication between stores
     */
    private RpcServer createRaftRpcServer(String raftAddr) {
        Endpoint endpoint = JRaftUtils.getEndPoint(raftAddr);
        RpcServer rpcServer = RaftRpcServerFactory.createRaftRpcServer(endpoint,
                                                                       JRaftUtils.createExecutor(
                                                                               "RAFT-RPC-",
                                                                               options.getRaftRpcThreadPoolSize()),
                                                                       null);
        HgCmdProcessor.registerProcessor(rpcServer, this);
        rpcServer.init(null);
        return rpcServer;
    }

    @Override
    public void shutdown() {
        if (rpcServer == null) {
            return;
        }
        partitionEngines.forEach((k, v) -> {
            v.shutdown();
        });
        partitionEngines.clear();
        rpcServer.shutdown();
        // HgStoreEngine.init function check rpcServer whether is null, skipped if the instance
        // exists even shut down.
        rpcServer = null;
        heartbeatService.shutdown();
        metricService.shutdown();
        // close all db session
        RocksDBFactory.getInstance().releaseAllGraphDB();
    }

    public void snapshotForTest() {
        partitionEngines.forEach((k, v) -> {
            v.snapshot();
        });
    }

    /**
     * Store registration status has changed
     */
    @Override
    public void stateChanged(Store store, Metapb.StoreState oldState, Metapb.StoreState newState) {
        log.info("stateChanged, oldState {}, newState {}", oldState, newState);
        if (newState == Metapb.StoreState.Up) {
            // Status changes to online, record store information
            partitionManager.setStore(store);
            partitionManager.loadPartition();
            restoreLocalPartitionEngine();
        }
    }

    /**
     * Restore the local PartitionEngine, restore the partition information returned by PD
     * 1. Need to check the partition saved this time, delete the invalid partitions.
     */
    public void restoreLocalPartitionEngine() {
        try {
            if (!options.isFakePD()) {  // FakePD mode does not require synchronization
                partitionManager.syncPartitionsFromPD(partition -> {
                    log.warn(
                            "The local partition information is inconsistent with the PD server. " +
                            "Please delete the redundant data manually,  {}", partition);
                });
            }
            partitionManager.getPartitions().forEach((k, g) -> {
                g.forEach((id, p) -> {
                    try {
                        createPartitionEngine(p);
                    } catch (Exception e) {
                        log.error("Partition {}-{} restore  exception {}", p.getGraphName(),
                                  p.getId(), e);
                    }
                });
            });
        } catch (PDException e) {
            log.error("HgStoreEngine restoreLocalPartitionEngine error {}", e);
        }
    }

    /**
     * Received store raft addr change, need to recreate raft group
     *
     * @param storeId Changed store id
     */
    public void rebuildRaftGroup(long storeId) {
        partitionEngines.forEach((partId, engine) -> {
            try {
                var partitions = pdProvider.getPartitionsByStore(storeId);
                if (partitions.size() > 0) {
                    var shards = pdProvider.getShardGroup(partId).getShardsList();
                    if (shards.stream().anyMatch(s -> s.getStoreId() == storeId)) {
                        var peers = partitionManager.shards2Peers(shards);
                        Configuration initConf = engine.getOptions().getConf();
                        if (initConf == null) {
                            engine.getOptions().setPeerList(peers);
                        } else {
                            peers.stream()
                                 .forEach(peer -> initConf.addPeer(JRaftUtils.getPeerId(peer)));
                        }

                        // engine.getOptions().getConf().setPeers();
                        engine.restartRaftNode();
                    }
                }
            } catch (PDException e) {
                log.error("rebuild raft group error: {}", e.getMessage());
            }
        });
    }

    /**
     * Create raft Node
     *
     * @param partition
     * @return
     */
    public PartitionEngine createPartitionEngine(Partition partition) {
        return createPartitionEngine(partition, null);
    }

    public PartitionEngine createPartitionEngine(Partition partition, Configuration conf) {
        partitionManager.updatePartition(partition, false);

        var shardGroup = partitionManager.getShardGroup(partition.getId());
        return createPartitionEngine(partition.getId(), shardGroup, conf);
    }

    private PartitionEngine createPartitionEngine(int groupId, ShardGroup shardGroup,
                                                  Configuration conf) {
        PartitionEngine engine;
        if ((engine = partitionEngines.get(groupId)) == null) {
            engineLocks.computeIfAbsent(groupId, k -> new Object());
            synchronized (engineLocks.get(groupId)) {
                // Special cases during partition splitting (different number of graph partitions in the cluster) can cause the splitting partition not to be on this machine.
                if (conf != null) {
                    var list = conf.listPeers();
                    list.addAll(conf.listLearners());
                    if (!list.stream().anyMatch(
                            p -> p.getEndpoint().toString().equals(options.getRaftAddress()))) {
                        log.info(
                                "raft {}, conf {} does not contains raft address:{}, skipped " +
                                "create partition engine",
                                groupId, conf, options.getRaftAddress());
                        return null;
                    }
                } else {
                    var storeId = partitionManager.getStore().getId();
                    if (!shardGroup.getShards().stream().anyMatch(s -> s.getStoreId() == storeId)) {
                        log.info("raft {}, shard group {} does not contains current storeId {}, " +
                                 "skipped create partition engine", groupId, shardGroup, storeId);
                        return null;
                    }
                }

                if ((engine = partitionEngines.get(groupId)) == null) {
                    log.info("createPartitionEngine {}, with shards: {}", groupId, shardGroup);

                    engine = new PartitionEngine(this, shardGroup);
                    PartitionEngineOptions ptOpts = new PartitionEngineOptions();
                    if (conf != null) {
                        ptOpts.setConf(conf);
                    } else {
                        ptOpts.setPeerList(partitionManager.getPartitionPeers(shardGroup));
                    }
                    ptOpts.setGroupId(groupId);

                    ptOpts.setRaftAddress(options.getRaftAddress());
                    ptOpts.setRaftDataPath(partitionManager.getRaftDataPath(groupId));
                    ptOpts.setRaftSnapShotPath(partitionManager.getRaftSnapShotPath(groupId));
                    ptOpts.setRaftOptions(options.getRaftOptions());
                    // raft task processor
                    ptOpts.setTaskHandler(options.getTaskHandler());

                    // Partition status listening
                    engine.addStateListener(this.heartbeatService);
                    engine.init(ptOpts);
                    partitionEngines.put(ptOpts.getGroupId(), engine);
                }
            }
        }
        // Check if active, if not active, recreate
        engine.checkActivity();
        return engine;
    }

    /**
     * Create raft group, in addition to creating the local raft node, also need to notify other peers to create raft nodes.
     * 1. Traverse partition.shards
     * 2. Retrieve Store information based on storeId
     * 3. Establish Raft RPC to other stores, send StartRaft messages.
     *
     * @param partition
     * @return
     */
    public PartitionEngine createPartitionGroups(Partition partition) {
        PartitionEngine engine = partitionEngines.get(partition.getId());
        if (engine == null) {
            engine = createPartitionEngine(partition);
            if (engine == null) {
                return null;
            }

            var shardGroup = partitionManager.getShardGroup(partition.getId());
            if (shardGroup != null) {
                // raft not exist, notify follower to create raft
                shardGroup.getShards().forEach((shard) -> {
                    Store store = partitionManager.getStore(shard.getStoreId());
                    if (store == null || partitionManager.isLocalStore(store)) {
                        return;
                    }
                    // Send messages to other peers, create raft groups. This is an asynchronous send.
                    hgCmdClient.createRaftNode(store.getRaftAddress(), List.of(partition),
                                               status -> {
                                                   log.info(
                                                           "send to {} createRaftNode rpc call " +
                                                           "result {} partitionId {}",
                                                           store.getRaftAddress(), status,
                                                           partition.getId());
                                               });
                });
            }
        } else {
            // raft exists, modify partition list, synchronize to follower through raft
            engine = createPartitionEngine(partition);
        }
        return engine;
    }

    public void destroyPartitionGroups(Partition partition) {
        var shardGroup = partitionManager.getShardGroup(partition.getId());
        if (shardGroup != null) {
            shardGroup.getShards().forEach((shard) -> {
                Store store = partitionManager.getStore(shard.getStoreId());
                if (store == null) {
                    return;
                }
                // Send messages to other peers, create raft groups. This is an asynchronous send.
                hgCmdClient.destroyRaftNode(store.getRaftAddress(),
                                            Arrays.asList(new Partition[]{partition}),
                                            status -> {
                                                log.info(
                                                        "send to {} - {} DestroyRaftNode rpc call" +
                                                        " result {}",
                                                        store.getRaftAddress(), partition.getId(),
                                                        status);
                                            });
            });
        }
    }

    /**
     * Stop partitioning, and destroy data
     */
    public synchronized void destroyPartitionEngine(Integer groupId, List<String> graphNames) {
        log.info("Partition {} start to be destroyed", groupId);
        if (!partitionEngines.containsKey(groupId)) {
            return;
        }
        PartitionEngine ptEngine = partitionEngines.get(groupId);
        graphNames.forEach(graphName -> {
            ptEngine.removePartition(graphName);
            // Delete data
            businessHandler.deletePartition(graphName, groupId);
        });

        if (ptEngine.getPartitions().size() == 0) {
            ptEngine.destroy();
            partitionEngines.remove(groupId);
            // Delete the corresponding db folder
            businessHandler.destroyGraphDB(graphNames.get(0), groupId);
        } else {
            graphNames.forEach(graphName -> {
                businessHandler.dbCompaction(graphName, groupId);
            });
        }
        log.info("Partition {} has been destroyed", groupId);
    }

    /**
     * Delete graph data, delete local data, and delete partition information on PD.
     */
    public void deletePartition(Integer groupId, String graphName) {
        log.info("Partition {}-{} deletePartition", graphName, groupId);
        if (!partitionEngines.containsKey(groupId)) {
            return;
        }
        PartitionEngine ptEngine = partitionEngines.get(groupId);
        ptEngine.removePartition(graphName);
        // Delete data
        businessHandler.deletePartition(graphName, groupId);
        // Notify PD to delete partition data
        if (ptEngine.isLeader()) {
            synchronized (this) {
                partitionManager.deletePartition(graphName, groupId);
            }
        }
    }

    /**
     * Get all leader partitions
     *
     * @return
     */
    public List<PartitionEngine> getLeaderPartition() {
        List<PartitionEngine> partitions = new ArrayList<>();
        this.partitionEngines.forEach((k, v) -> {
            if (v.isLeader()) {
                partitions.add(v);
            }
        });
        return partitions;
    }

    /**
     * Get all active peers in the partition
     *
     * @return
     */
    public Map<Long, PeerId> getAlivePeers(int groupId) {
        PartitionEngine engine = this.partitionEngines.get(groupId);
        try {
            if (engine != null) {
                return engine.getAlivePeers();
            }
        } catch (Exception e) {
            log.error("getAlivePeers {}", e);
        }
        return new HashMap<>();
    }

    /**
     * Get the last committed log id of the partition
     *
     * @param groupId
     * @return
     */
    public long getLeaderTerm(int groupId) {
        PartitionEngine engine = this.partitionEngines.get(groupId);
        return engine.getLeaderTerm();
    }

    public long getCommittedIndex(int groupId) {
        PartitionEngine engine = this.partitionEngines.get(groupId);
        if (engine != null) {
            return engine.getCommittedIndex();
        }
        return 0;
    }

    public RpcServer getRaftRpcServer() {
        return rpcServer;
    }

    public PartitionManager getPartitionManager() {
        return partitionManager;
    }

    // For test
    public void setPartitionManager(PartitionManager ptm) {
        this.partitionManager = ptm;
    }

    public DataMover getDataMover() {
        return dataMover;
    }

    public PdProvider getPdProvider() {
        return pdProvider;
    }

    public BusinessHandler getBusinessHandler() {
        return businessHandler;
    }

    public HgCmdClient getHgCmdClient() {
        return hgCmdClient;
    }

    public HeartbeatService getHeartbeatService() {
        return heartbeatService;
    }

    public boolean isClusterReady() {
        return heartbeatService.isClusterReady();
    }

    public List<String> getDataLocations() {
        return partitionManager.getStoreMetadata().getDataLocations();
    }

    /**
     * Add raft task
     * 1. Check if the partition exists
     * 1.1 If not exist, query PD to determine if the partition is local.
     * 1.1.1 If the partition is local, create a raft group and notify other Stores.
     * 1.1.2 If the partition is not local, throw an exception.
     * 1.2 Check if Partition is the leader
     * 1.2.1 If it is the leader, then submit the task.
     * 1.2.2 Otherwise, return an error.
     *
     * @param partId
     * @param operation
     */
    public void addRaftTask(String graphName, Integer partId, RaftOperation operation,
                            RaftClosure closure) {
        PartitionEngine engine = getPartitionEngine(graphName, partId);
        if (engine == null) {
            engineLocks.computeIfAbsent(partId, k -> new Object());
            synchronized (engineLocks.get(partId)) {
                engine = getPartitionEngine(graphName, partId);
                if (engine == null) {
                    Partition partition = partitionManager.findPartition(graphName, partId);
                    if (partition != null) {
                        engine = this.createPartitionGroups(partition);
                        // May migrate, should not create, put in synchronize block, avoid subsequent ones.
                        if (engine != null) {
                            engine.waitForLeader(options.getWaitLeaderTimeout() * 1000);
                        }
                    }
                }
            }
        }

        if (engine != null) {
            // Waiting for Leader
            Endpoint leader = engine.waitForLeader(options.getWaitLeaderTimeout() * 1000);
            if (engine.isLeader()) {
                engine.addRaftTask(operation, closure);
            } else if (leader != null) {
                // Currently not the leader, return the storeId where the leader is located.
                Store store = partitionManager.getStoreByRaftEndpoint(engine.getShardGroup(),
                                                                      leader.toString());
                if (store.getId() == 0) {
                    // Local store information for the Leader was not found, possibly the Partition has not been synchronized yet, reacquire from the Leader.
                    Store leaderStore = hgCmdClient.getStoreInfo(leader.toString());
                    store = leaderStore != null ? leaderStore : store;
                    log.error("getStoreByRaftEndpoint error store:{}, shard: {}, leader is {}",
                              store, engine.getShardGroup().toString(), leader);
                }
                // Leader is not local, notify client
                closure.onLeaderChanged(partId, store.getId());
                closure.run(new Status(HgRaftError.NOT_LEADER.getNumber(),
                                       String.format("Partition %s-%d leader changed to %x",
                                                     graphName, partId, store.getId())));
                log.error("Raft Partition {}-{} not leader, redirectTo leader {}.", graphName,
                          partId, leader);
            } else {
                closure.run(new Status(HgRaftError.WAIT_LEADER_TIMEOUT.getNumber(),
                                       HgRaftError.WAIT_LEADER_TIMEOUT.getMsg()));
                log.error("Partition {}-{} waiting for leader timeout.", graphName, partId);
            }
        } else {
            closure.run(
                    new Status(HgRaftError.NOT_LOCAL.getNumber(), HgRaftError.NOT_LOCAL.getMsg()));
            log.error("Partition {}-{} does not belong to local store.", graphName, partId);
        }
    }

    public PartitionEngine getPartitionEngine(Integer partitionId) {
        PartitionEngine engine = partitionEngines.get(partitionId);
        return engine;
    }

    public PartitionEngine getPartitionEngine(String graphName, Integer partitionId) {
        PartitionEngine engine = partitionEngines.get(partitionId);
        if (engine != null && engine.hasPartition(graphName)) {
            return engine;
        }
        return null;
    }

    public Map<Integer, PartitionEngine> getPartitionEngines() {
        return partitionEngines;
    }

    public Map<String, NodeMetrics> getNodeMetrics() {
        Map<String, NodeMetrics> metrics = new HashMap();
        partitionEngines.forEach((k, v) -> {
            metrics.put(Integer.toString(k), v.getNodeMetrics());
        });
        return metrics;
    }

    /**
     * Number of raft-group.
     *
     * @return
     */
    public int getRaftGroupCount() {
        return partitionEngines.size();
    }

    /**
     * Listen to rocksdb events
     *
     * @return
     */
    private RocksDBFactory.RocksdbChangedListener getRocksdbListener() {
        return new RocksDBFactory.RocksdbChangedListener() {
            @Override
            public void onCompacted(String dbName) {
                String sid = dbName.substring(dbName.lastIndexOf("/") + 1);
                try {
                    Integer groupId = Integer.parseInt(sid);
                    PartitionEngine engine = getPartitionEngine(groupId);
                    if (engine != null) {
                        engine.addBlankRaftTask();
                    }
                } catch (Exception e) {

                }
            }
        };
    }

    class PartitionChangedListener implements PartitionManager.PartitionChangedListener {

        /**
         * Partition object changes, leader notifies other followers.
         */
        @Override
        public void onChanged(Partition partition) {
            PartitionEngine engine = getPartitionEngine(partition.getId());

            if (engine != null && engine.isLeader()) {
                try {
                    engine.addRaftTask(RaftOperation.create(RaftOperation.SYNC_PARTITION,
                                                            partition.getProtoObj()),
                                       new RaftClosure() {
                                           @Override
                                           public void run(Status status) {
                                               log.info(
                                                       "Partition {}-{}-{} sync partition status " +
                                                       "is {}",
                                                       partition.getGraphName(), partition.getId(),
                                                       partition.getWorkState(),
                                                       status);
                                           }
                                       });
                } catch (IOException e) {
                    log.error("Partition {}-{} sync partition exception {}",
                              partition.getGraphName(), partition.getId(), e);
                }
            }
        }

        /**
         * Partition object key range, status changes, notify other followers by actively finding the leader.
         */
        @Override
        public UpdatePartitionResponse rangeOrStateChanged(UpdatePartitionRequest request) {
            UpdatePartitionResponse response = null;
            try {
                response = hgCmdClient.raftUpdatePartition(request);

                log.info("not leader request threadId:{} pId:{} range:{}-{} state:{} response:{}",
                         Thread.currentThread().getId(), request.getPartitionId(),
                         request.getStartKey(),
                         request.getEndKey(), request.getWorkState(), response.getStatus());

            } catch (Exception e) {
                e.printStackTrace();
            }

            return response;
        }

    }
}
