/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.utils;

import java.time.Duration;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.catalog.IntervalFreshness;

@Internal
public class IntervalFreshnessUtils {
    private static final String SECOND_CRON_EXPRESSION_TEMPLATE = "0/%s * * * * ? *";
    private static final String MINUTE_CRON_EXPRESSION_TEMPLATE = "0 0/%s * * * ? *";
    private static final String HOUR_CRON_EXPRESSION_TEMPLATE = "0 0 0/%s * * ? *";
    private static final String ONE_DAY_CRON_EXPRESSION_TEMPLATE = "0 0 0 * * ? *";
    private static final long SECOND_CRON_UPPER_BOUND = 60L;
    private static final long MINUTE_CRON_UPPER_BOUND = 60L;
    private static final long HOUR_CRON_UPPER_BOUND = 24L;

    private IntervalFreshnessUtils() {
    }

    @VisibleForTesting
    static void validateIntervalFreshness(IntervalFreshness intervalFreshness) {
        if (!NumberUtils.isParsable((String)intervalFreshness.getInterval())) {
            throw new ValidationException(String.format("The interval freshness value '%s' is an illegal integer type value.", intervalFreshness.getInterval()));
        }
        if (!NumberUtils.isDigits((String)intervalFreshness.getInterval())) {
            throw new ValidationException("The freshness interval currently only supports integer type values.");
        }
    }

    public static Duration convertFreshnessToDuration(IntervalFreshness intervalFreshness) {
        IntervalFreshnessUtils.validateIntervalFreshness(intervalFreshness);
        long interval = Long.parseLong(intervalFreshness.getInterval());
        switch (intervalFreshness.getTimeUnit()) {
            case DAY: {
                return Duration.ofDays(interval);
            }
            case HOUR: {
                return Duration.ofHours(interval);
            }
            case MINUTE: {
                return Duration.ofMinutes(interval);
            }
            case SECOND: {
                return Duration.ofSeconds(interval);
            }
        }
        throw new ValidationException(String.format("Unknown freshness time unit: %s.", new Object[]{intervalFreshness.getTimeUnit()}));
    }

    public static String convertFreshnessToCron(IntervalFreshness intervalFreshness) {
        switch (intervalFreshness.getTimeUnit()) {
            case SECOND: {
                return IntervalFreshnessUtils.validateAndConvertCron(intervalFreshness, 60L, SECOND_CRON_EXPRESSION_TEMPLATE);
            }
            case MINUTE: {
                return IntervalFreshnessUtils.validateAndConvertCron(intervalFreshness, 60L, MINUTE_CRON_EXPRESSION_TEMPLATE);
            }
            case HOUR: {
                return IntervalFreshnessUtils.validateAndConvertCron(intervalFreshness, 24L, HOUR_CRON_EXPRESSION_TEMPLATE);
            }
            case DAY: {
                return IntervalFreshnessUtils.validateAndConvertDayCron(intervalFreshness);
            }
        }
        throw new ValidationException(String.format("Unknown freshness time unit: %s.", new Object[]{intervalFreshness.getTimeUnit()}));
    }

    private static String validateAndConvertCron(IntervalFreshness intervalFreshness, long cronUpperBound, String cronTemplate) {
        long interval = Long.parseLong(intervalFreshness.getInterval());
        IntervalFreshness.TimeUnit timeUnit = intervalFreshness.getTimeUnit();
        if (interval >= cronUpperBound) {
            throw new ValidationException(String.format("In full refresh mode, freshness must be less than %s when the time unit is %s.", new Object[]{cronUpperBound, timeUnit}));
        }
        if (cronUpperBound % interval != 0L) {
            throw new ValidationException(String.format("In full refresh mode, only freshness that are factors of %s are currently supported when the time unit is %s.", new Object[]{cronUpperBound, timeUnit}));
        }
        return String.format(cronTemplate, interval);
    }

    private static String validateAndConvertDayCron(IntervalFreshness intervalFreshness) {
        long interval = Long.parseLong(intervalFreshness.getInterval());
        if (interval > 1L) {
            throw new ValidationException("In full refresh mode, freshness must be 1 when the time unit is DAY.");
        }
        return ONE_DAY_CRON_EXPRESSION_TEMPLATE;
    }
}

